/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2025  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QtXml/QDomNodeList>
#include <QtXml/QDomText>
#include <QtXml/QDomNamedNodeMap>
#include <QFile>
#include <QTextStream>
#include <QFileInfo>
#include <assert.h>

#include "xmlfile.h"

TXmlFileException::TXmlFileException(
	const QString 						&error,
	const TXmlFileException::TExceptionType	 type)
:d_error_text(error),
 d_error_type(type)
{

}

TXmlFileException::~TXmlFileException(void)
{

}

TXmlFile::TXmlFile(void)
{
	QDomImplementation::setInvalidDataPolicy(QDomImplementation::ReturnNullNode);
}

TXmlFile::~TXmlFile(void)
{

}

void TXmlFile::Open(
	const QString						&file_identification,
	int 								*version_number,
	const bool							process_namespace)
{
	QFile								file;
	QString								error;
	int									error_line;
	int									error_column;
	QString								text;
	QDomDocumentType					document_type;
	QFileInfo							file_info(d_file_name);

	if(!file_info.isReadable())
	{
		text = QString("ERR:  Cannot open file '%1' for reading.").arg(d_file_name);
		throw TXmlFileException(text,TXmlFileException::FILE_OPEN_ERROR);
	}

	file.setFileName(d_file_name);

	if(!file.open(QFile::ReadOnly | QFile::Text))
	{
		text = QString("ERR:  Cannot open file: '%1'.  Reason: %2").arg(d_file_name).arg(file.errorString());
		throw TXmlFileException(text,TXmlFileException::FILE_OPEN_ERROR);
	}

	if(!d_dom_document.setContent(
					&file,
					process_namespace,
					&error,
					&error_line,
					&error_column))
	{
		file.close();

		text = QString("ERR:  Reference file XML parse error at line %1 column %2: %3")
					.arg(error)
					.arg(error_line)
					.arg(error_column);
		
		throw TXmlFileException(text);
	}

	file.close();

	d_root_node = d_dom_document.documentElement();

	if(d_root_node.tagName() != file_identification)
	{
		text = QString("ERR:  Identification '%1' does not match '%2'")
									.arg(file_identification)
									.arg(d_root_node.tagName());

		throw TXmlFileException(text);
	}

	document_type = d_dom_document.doctype();
	d_document_type = document_type.name();

	d_file_identification = file_identification;

	if(version_number)
	{
		if(d_root_node.hasAttribute("File_Version"))
		{
			*version_number = d_root_node.attribute("File_Version").toInt();
		}
		else
		{
			text = QString("ERR:  The 'File_Version' attribute is missing.");
			throw TXmlFileException(text);
		}
	}
}

void TXmlFile::Open_Data(
	const QByteArray					&data,
	const QString						&file_identification,
	int 								*version_number,
	const bool							process_namespace)
{
	QString								error;
	int									error_line;
	int									error_column;
	QString								text;
	QDomDocumentType					document_type;
	
	if(!d_dom_document.setContent(
							 data,
							 process_namespace,
							 &error,
							 &error_line,
							 &error_column))
	{
		text = QString("ERR:  Reference data XML parse error at line %1 column %2: %3")
							.arg(error)
							.arg(error_line)
							.arg(error_column);
		
		throw TXmlFileException(text);
	}
	
	d_root_node = d_dom_document.documentElement();
	
	if(d_root_node.tagName() != file_identification)
	{
		text = QString("ERR:  Identification '%1' does not match '%2'")
							.arg(file_identification)
							.arg(d_root_node.tagName());
		
		throw TXmlFileException(text);
	}
	
	document_type = d_dom_document.doctype();
	d_document_type = document_type.name();
	
	d_file_identification = file_identification;
	
	if(version_number)
	{
		if(d_root_node.hasAttribute("File_Version"))
		{
			*version_number = d_root_node.attribute("File_Version").toInt();
		}
		else
		{
			text = QString("ERR:  The 'File_Version' attribute is missing.");
			throw TXmlFileException(text);
		}
	}
}

QString TXmlFile::Read_Text_Node(
	const QDomNode 						*node,
	const bool 							optional) const
{
	QString								text;
	QDomNode							text_node;

	assert(node);

	if(node->isNull())
	{
		throw TXmlFileException(QString("ERR:  Cannot read data from a NULL node"));
	}

	text_node = node->firstChild();

	if(text_node.isNull() == false && text_node.isText() == true)
	{
		text = text_node.nodeValue();
	}
	else
	{
		if(!optional)
		{
			throw TXmlFileException(QString("ERR:  Node Text for '%1' is not optional.").arg(node->localName()));
		}

		text.clear();
	}

	return text;
}

QString TXmlFile::Read_Text_Node(
	const QDomElement 					*node,
	const bool 							optional) const
{
	QString								text;
	QDomNode							text_node;

	assert(node);

	if(node->isNull())
	{
		throw TXmlFileException(QString("ERR:  Cannot read data from a NULL node"));
	}

	text_node = node->firstChild();

	if(text_node.isNull() == false && text_node.isText() == true)
	{
		text = text_node.nodeValue();
	}
	else
	{
		if(!optional)
		{
			throw TXmlFileException(QString("ERR:  Node Text for '%1' is not optional.").arg(node->tagName()));
		}

		text.clear();
	}

	return text;
}

QString TXmlFile::Read_Attribute(
	const QDomElement 					*node,
	const QString 						&name,
	const bool 							optional) const
{
	QString								text;

	assert(node);

	if(node->isNull())
	{
		throw TXmlFileException(QString("ERR:  Cannot read attribute from a NULL node"));
	}

	if(node->hasAttribute(name))
	{
		text = node->attribute(name);
	}
	else
	{
		if(!optional)
		{
			throw TXmlFileException(QString("ERR:  Node attribute '%1' missing and not optional.").arg(name));
		}

		text.clear();
	}

	return text;
}

QString TXmlFile::Read_Attribute(
	const QDomNode 						*node,
	const QString 						&name,
	const bool 							optional) const
{
	QDomNamedNodeMap					NamedNodeMap;
	QDomNode							AttributeNode;
	QString								text;
	bool								FoundAttribute(false);

	assert(node);

	if(node->isNull())
	{
		throw TXmlFileException(QString("ERR:  Cannot read attribute from a NULL node"));
	}

	if(node->hasAttributes())
	{
		NamedNodeMap = node->attributes();

		if(NamedNodeMap.contains(name))
		{
			AttributeNode = NamedNodeMap.namedItem(name);
			text = AttributeNode.nodeValue();
			FoundAttribute = true;
		}
	}

	if(!FoundAttribute)
	{
		if(!optional)
		{
			throw TXmlFileException(QString("ERR:  Node attribute '%1' missing and not optional.").arg(name));
		}

		text.clear();
	}

	return text;
}

QDomElement TXmlFile::Get_Node(
	const QDomElement 					*node,
	const QString 						&name,
	const bool 							optional) const
{
	QDomElement							child_node;

	assert(node);

	if(node->isNull())
	{
		throw TXmlFileException(QString("ERR:  Cannot get child from a NULL node"));
	}

	child_node = node->firstChildElement(name);

	if(child_node.isNull())
	{
		if(!optional)
		{
			throw TXmlFileException(QString("ERR:  Node '%1' is missing from node '%2' and not optional.")
												.arg(name)
												.arg(node->tagName()));
		}
	}

	return child_node;
}

QDomElement TXmlFile::Get_Node(
	const QDomNode 						*node,
	const QString 						&name,
	const bool 							optional) const
{
	QDomElement							child_node;

	assert(node);

	if(node->isNull())
	{
		throw TXmlFileException(QString("ERR:  Cannot get child from a NULL node"));
	}

	child_node = node->firstChildElement(name);

	if(child_node.isNull())
	{
		if(!optional)
		{
			throw TXmlFileException(QString("ERR:  Node '%1' is missing from node '%2' and not optional.")
												.arg(name)
												.arg(node->nodeName()));
		}
	}

	return child_node;
}

QDomElement TXmlFile::Get_Sibling_Node(
	const QDomElement 					*current_node,
	const QString 						&name,
	const bool 							optional) const
{
	QDomElement							NextNode;

	assert(current_node);

	if(current_node->isNull())
	{
		throw TXmlFileException(QString("ERR:  Cannot get sibling from a NULL node"));
	}

	NextNode = current_node->nextSiblingElement(name);

	if(NextNode.isNull())
	{
		if(!optional)
		{
			throw TXmlFileException(QString("ERR:  Node '%1' is missing from node '%2' and not optional.")
												.arg(name)
												.arg(current_node->tagName()));
		}
	}

	return NextNode;
}

QDomNode TXmlFile::Get_First_Node(
	const QDomElement 					*parent,
	const bool 							optional) const
{
	QDomNode							node;

	assert(parent);

	node = parent->firstChild();

	if(node.isNull())
	{
		if(!optional)
		{
			throw TXmlFileException(QString("ERR:  Parent '%1' does not have nodes and they are not optional.")
												.arg(parent->tagName()));
		}
	}

	return node;
}

QDomNode TXmlFile::Get_First_Node(
	const QDomNode	 					*parent,
	const bool 							optional) const
{
	QDomNode							node;

	assert(parent);

	node = parent->firstChild();

	if(node.isNull())
	{
		if(!optional)
		{
			throw TXmlFileException(QString("ERR:  Parent '%1' does not have nodes and they are not optional.")
												.arg(parent->localName()));
		}
	}

	return node;
}

QDomNode TXmlFile::Get_Sibling_Node(
	const QDomNode	 					*current_node,
	const bool 							optional) const
{
	QDomNode							node;

	assert(current_node);

	node = current_node->nextSibling();

	if(node.isNull())
	{
		if(!optional)
		{
			throw TXmlFileException(QString("ERR:  Node '%1' does not have siblings and they are not optional.")
												.arg(current_node->localName()));
		}
	}

	return node;
}

QDomElement TXmlFile::Create_Node(
	QDomElement 						*parent,
	const QString 						&name)
{
	QDomElement							node;

	assert(parent);

	if(parent->isNull())
	{
		throw TXmlFileException(QString("ERR:  Cannot create node from a NULL node"));
	}

	if(name.length() == 0)
	{
		throw TXmlFileException(QString("ERR:  Cannot create an unnamed node"));
	}

	node = d_dom_document.createElement(name);
	parent->appendChild(node);

	return node;
}

void TXmlFile::Clear(void)
{
	d_dom_document.clear();
}

void TXmlFile::Reset(
	const QString 						&file_identification,
	const int 							version_number)
{
	QDomProcessingInstruction 			process_instruction;

	d_dom_document.clear();

    process_instruction = d_dom_document.createProcessingInstruction("xml", "version=\"1.0\" encoding=\"utf-8\"");
    d_dom_document.appendChild(process_instruction);

    d_root_node = d_dom_document.createElement(file_identification);
    d_root_node.setAttribute("File_Version",QString("%1").arg(version_number));
    d_dom_document.appendChild(d_root_node);
}

void TXmlFile::Reset(
	const QString 						&doc_type,
	const QString 						&file_identification,
	const int 							version_number)
{

	d_dom_document = QDomDocument(doc_type);

    d_root_node = d_dom_document.createElement(file_identification);
    d_root_node.setAttribute("File_Version",QString("%1").arg(version_number));
    d_dom_document.appendChild(d_root_node);
}

void TXmlFile::Write_Text_Node(
	QDomElement 						*node,
	const QString 						&data)
{
	QDomText							text_node;

	assert(node);

	if(node->isNull())
	{
		throw TXmlFileException(QString("ERR:  Cannot append data to a NULL node"));
	}

	text_node = d_dom_document.createTextNode(data);

	if(text_node.isNull())
	{
		throw TXmlFileException(QString("ERR:  Cannot write node '%1' with data '%2'")
														.arg(node->tagName())
														.arg(data));
	}
	node->appendChild(text_node);
}

void TXmlFile::Write_Text_Node(
	QDomElement 						*node,
	const double 						&value,
	const int 							precision)
{
	this->Write_Text_Node(node,QString("%1").arg(value,0,'f',precision));
}

void TXmlFile::Write_Text_Node(
	QDomElement 						*node,
	const int 							value)
{
	this->Write_Text_Node(node,QString("%1").arg(value));
}

void TXmlFile::Write_Attribute(
	QDomElement 						*node,
	const QString						&name,
	const QString 						&attribute)
{
	assert(node);

	if(name.length() == 0)
	{
		throw TXmlFileException(QString("ERR:  Cannot create an unnamed attribute"));
	}

	node->setAttribute(name,attribute);
}

void TXmlFile::Write_Attribute(
	QDomElement 						*node,
	const QString 						&name,
	const double 						&attribute,
	const int							precision)
{
	this->Write_Attribute(node,name,QString("%1").arg(attribute,0,'f',precision));
}

void TXmlFile::Write_Attribute(
	QDomElement 						*node,
	const QString 						&name,
	const int 							attribute)
{
	this->Write_Attribute(node,name,QString("%1").arg(attribute));
}


void TXmlFile::Write_File(
	const bool							inline_xml)
{
	QFile								file;
	const int							indent_size = 4;
	QString 							text;
	QFileInfo							file_info(d_file_name);

	if(file_info.exists() == true && file_info.isWritable() == false)
	{
		text = QString("ERR:  Cannot open file '%1' for writing").arg(d_file_name);
		throw TXmlFileException(text,TXmlFileException::FILE_OPEN_ERROR);
	}

	file.setFileName(d_file_name);

    if (!file.open(QIODevice::WriteOnly | QIODevice::Text))
    {
		text = QString("ERR:  Cannot write file '%1'.  Reason: %2").arg(d_file_name).arg(file.errorString());
		throw TXmlFileException(text);
    }

	if(inline_xml)
	{
		QByteArray						byte_array;
		QByteArray						write_byte_array;
		int								cntr;
		QTextStream 					text_stream(&byte_array);
		
		d_dom_document.save(text_stream, 0);
		
		for(cntr = 0;cntr < byte_array.size();++cntr)
		{
			if(byte_array[cntr] != '\n' && byte_array[cntr] != '\r')
			{
				write_byte_array.push_back(byte_array[cntr]);
			}
		}
		
		file.write(write_byte_array);
	}
	else
	{
		QTextStream 					text_stream(&file);

		d_dom_document.save(text_stream, indent_size);
	}
	
	file.close();
}

void TXmlFile::Write_Data(
	QByteArray							* const data,
	const bool							inline_xml)
{
	QString 							text;
	const int							indent_size = 4;
	
	assert(data);
	
	data->clear();
	
	if(inline_xml)
	{
		QByteArray						byte_array;
		int								cntr;
		QTextStream 					text_stream(&byte_array);
		
		d_dom_document.save(text_stream, 0);
		
		for(cntr = 0;cntr < byte_array.size();++cntr)
		{
			if(byte_array[cntr] != '\n' && byte_array[cntr] != '\r')
			{
				data->push_back(byte_array[cntr]);
			}
		}
	}
	else
	{
		QTextStream 					text_stream(data,QIODevice::WriteOnly);
		
		d_dom_document.save(text_stream, indent_size);
	}
}


