/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2025  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QScreen>
#include <QToolBar>
#include <QFrame>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QPushButton>
#include <QToolButton>
#include <QComboBox>
#include <QSlider>
#include <QDockWidget>
#include <QListWidget>
#include <QListWidgetItem>
#include <QAction>
#include <QSizePolicy>
#include <QSpacerItem>
#include <QSettings>
#include <QCloseEvent>
#include <QMenuBar>
#include <QMenu>
#include <QIcon>
#include <QDir>
#include <QRect>
#include <QSize>
#include <QFont>
#include <vector>
#include <assert.h>

#include "../../core/messagebox.h"

#include "displaywidget.h"
#include "measurementeditwidget.h"
#include "deflectionsimulator.h"

TDeflectionSimulator::TDeflectionSimulator(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QMainWindow(const_cast<QWidget*>(parent),flags)
{
	QWidget								*central_widget;
	QWidget								*dock_contents_widget;
	QDockWidget							*dock_widget;
	QToolBar							*tool_bar;
	QFrame								*lower_hline;
	QFrame								*upper_hline;
	QFrame								*separator_hline;
	QGridLayout							*central_widget_layout;
	QVBoxLayout							*dock_contents_vlayout;
	QLabel								*deflection_model_label;
	QLabel								*exaggeration_label;
	QLabel								*range_y_label;
	QLabel								*range_y_units_label;
	QLabel								*range_z_label;
	QLabel								*range_z_units_label;
	QLabel								*copyright_label;
	QSpacerItem							*control_hspacer;
	QHBoxLayout							*button_hlayout;
	QSpacerItem							*button_hspacer;
	QListWidgetItem						*list_item;
	QMenuBar							*main_menu;
	QMenu								*file_menu;
	QMenu								*help_menu;
	QMenu								*view_menu;
	std::vector<int>					ids;
	std::vector<int>::const_iterator	id_iter;
	TMeasurementData::TMeasurement		measurement;
	QSizePolicy							sizepolicy_preferred_preferred(QSizePolicy::Preferred,QSizePolicy::Preferred);
	QSizePolicy							sizepolicy_mexpanding_preferred(QSizePolicy::MinimumExpanding,QSizePolicy::Preferred);
	QSizePolicy							sizepolicy_mexpanding_mexpanding(QSizePolicy::MinimumExpanding,QSizePolicy::MinimumExpanding);
	QDir								data_dir;
	QRect								desktop_rect;
	QRect								window_rect;
	QPoint								position;
	QSize								size;
	QFont								copyright_font;
	double								dval;

	sizepolicy_preferred_preferred.setHorizontalStretch(0);
	sizepolicy_preferred_preferred.setVerticalStretch(0);

	sizepolicy_mexpanding_preferred.setHorizontalStretch(0);
	sizepolicy_mexpanding_preferred.setVerticalStretch(0);
	
	sizepolicy_mexpanding_mexpanding.setHorizontalStretch(0);
	sizepolicy_mexpanding_mexpanding.setVerticalStretch(0);

	copyright_font.setFamily(QStringLiteral("Verdana"));
	copyright_font.setPointSize(8);
	copyright_font.setItalic(true);
	
	data_dir = QDir::home();
	
	if(!data_dir.exists(QStringLiteral(".deflectionsimulator")))
	{
		data_dir.mkdir(QStringLiteral(".deflectionsimulator"));
	}
	
	data_dir.cd(QStringLiteral(".deflectionsimulator"));
	d_data_path = data_dir.absolutePath();
	
	d_settings = new QSettings(d_data_path + QStringLiteral("/settings.ini"),QSettings::IniFormat,this);
	
	position = d_settings->value("Mainwindow_Position", QPoint(30,30)).toPoint();
	size = d_settings->value("Mainwindow_Size", QSize(801, 601)).toSize();
	
	desktop_rect = QGuiApplication::primaryScreen()->availableGeometry();
	
	window_rect.moveTopLeft(position);
	window_rect.setSize(size);
	
	window_rect = window_rect & desktop_rect;
	
	if(window_rect.isValid() && window_rect.width() > 100 && window_rect.height() > 100)
	{
		this->resize(window_rect.size());
		this->move(window_rect.topLeft());
	}
	else
	{
		this->resize(901,901);
		this->move(QPoint(30,30));
	}
	
	d_msg_box = new TMessageBox(this);

	d_action_new = new QAction(this);
	d_action_new->setShortcut(QKeySequence::New);
	d_action_new->setIcon(QIcon(QStringLiteral(":/menu/file_new_x32.png")));
	
	d_action_init = new QAction(this);
	d_action_init->setIcon(QIcon(QStringLiteral(":/menu/file_initialize_x32.png")));
	
	d_action_quit = new QAction(this);
	d_action_quit->setShortcut(QKeySequence::Quit);

	d_action_about = new QAction(this);
	
	main_menu = new QMenuBar(this);
	this->setMenuBar(main_menu);
	
	file_menu = new QMenu(main_menu);
	main_menu->addMenu(file_menu);
	
	file_menu->addAction(d_action_new);
	file_menu->addAction(d_action_init);
	file_menu->addSeparator();
	file_menu->addAction(d_action_quit);

	view_menu = new QMenu(main_menu);
	main_menu->addMenu(view_menu);

	help_menu = new QMenu(main_menu);
	main_menu->addMenu(help_menu);
	
	help_menu->addAction(d_action_about);

	main_menu->resize(800,22);

	main_menu->addAction(file_menu->menuAction());
	main_menu->addAction(view_menu->menuAction());
	main_menu->addAction(help_menu->menuAction());

	tool_bar = new QToolBar(this);
	this->addToolBar(Qt::TopToolBarArea, tool_bar);
	
	tool_bar->addAction(d_action_new);
	tool_bar->addAction(d_action_init);
	tool_bar->addSeparator();
	
	dock_widget = new QDockWidget(this);
	dock_widget->setAllowedAreas(Qt::LeftDockWidgetArea|Qt::RightDockWidgetArea);
	this->addDockWidget(static_cast<Qt::DockWidgetArea>(1),dock_widget);
	
	view_menu->addAction(dock_widget->toggleViewAction());

	dock_contents_widget = new QWidget();
	dock_widget->setWidget(dock_contents_widget);
	
	dock_contents_vlayout = new QVBoxLayout(dock_contents_widget);
	
	d_measurement_list = new QListWidget(dock_contents_widget);
	d_measurement_list->setEditTriggers(QAbstractItemView::NoEditTriggers);
	d_measurement_list->setSelectionMode(QAbstractItemView::ExtendedSelection);
	dock_contents_vlayout->addWidget(d_measurement_list);
	
	button_hlayout = new QHBoxLayout();
	
	d_add_measurement_button = new QToolButton(dock_contents_widget);
	d_add_measurement_button->setMinimumSize(20,20);
	d_add_measurement_button->setMaximumSize(20,20);
	button_hlayout->addWidget(d_add_measurement_button);
	
	d_remove_measurement_button = new QToolButton(dock_contents_widget);
	d_remove_measurement_button->setMinimumSize(20,20);
	d_remove_measurement_button->setMaximumSize(20,20);
	button_hlayout->addWidget(d_remove_measurement_button);
	
	button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_hlayout->addItem(button_hspacer);
	dock_contents_vlayout->addLayout(button_hlayout);
	
	separator_hline = new QFrame(dock_contents_widget);
	separator_hline->setFrameShape(QFrame::HLine);
	separator_hline->setFrameShadow(QFrame::Sunken);
	dock_contents_vlayout->addWidget(separator_hline);

	d_measurement_edit_widget = new TMeasurementEditWidget(dock_contents_widget);
	dock_contents_vlayout->addWidget(d_measurement_edit_widget);

	central_widget = new QWidget(this);
	this->setCentralWidget(central_widget);

	central_widget_layout = new QGridLayout(central_widget);
	
	d_display_widget = new TDisplayWidget(central_widget);
	d_display_widget->setSizePolicy(sizepolicy_mexpanding_mexpanding);
	central_widget_layout->addWidget(d_display_widget,0,0,1,7);
	
	upper_hline = new QFrame(central_widget);
	upper_hline->setFrameShape(QFrame::HLine);
	upper_hline->setFrameShadow(QFrame::Sunken);
	central_widget_layout->addWidget(upper_hline,1,0,1,10);
	
	deflection_model_label = new QLabel(central_widget);
	central_widget_layout->addWidget(deflection_model_label,2,0,1,1);
	
	d_deflection_model_type_combo = new QComboBox(central_widget);
	central_widget_layout->addWidget(d_deflection_model_type_combo,2,1,1,1);
	
	range_y_label = new QLabel(central_widget);
	central_widget_layout->addWidget(range_y_label,3,0,1,1);
	
	d_range_y_edit = new QLineEdit(central_widget);
	d_range_y_edit->setAlignment(Qt::AlignCenter);
	central_widget_layout->addWidget(d_range_y_edit,3,1,1,1);
	
	range_y_units_label = new QLabel(central_widget);
	central_widget_layout->addWidget(range_y_units_label,3,2,1,1);
	
	range_z_label = new QLabel(central_widget);
	central_widget_layout->addWidget(range_z_label,4,0,1,1);
	
	d_range_z_edit = new QLineEdit(central_widget);
	d_range_z_edit->setAlignment(Qt::AlignCenter);
	central_widget_layout->addWidget(d_range_z_edit,4,1,1,1);
	
	range_z_units_label = new QLabel(central_widget);
	central_widget_layout->addWidget(range_z_units_label,4,2,1,1);
	
	d_parameter1_label = new QLabel(central_widget);
	d_parameter1_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	central_widget_layout->addWidget(d_parameter1_label,2,3,1,1);
	
	d_parameter1_edit = new QLineEdit(central_widget);
	d_parameter1_edit->setAlignment(Qt::AlignCenter);
	central_widget_layout->addWidget(d_parameter1_edit,2,4,1,1);

	d_parameter1_units_label = new QLabel(central_widget);
	central_widget_layout->addWidget(d_parameter1_units_label,2,5,1,1);

	d_parameter2_label = new QLabel(central_widget);
	d_parameter2_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	central_widget_layout->addWidget(d_parameter2_label,3,3,1,1);
	
	d_parameter2_edit = new QLineEdit(central_widget);
	d_parameter2_edit->setAlignment(Qt::AlignCenter);
	central_widget_layout->addWidget(d_parameter2_edit,3,4,1,1);
	
	d_parameter2_units_label = new QLabel(central_widget);
	central_widget_layout->addWidget(d_parameter2_units_label,3,5,1,1);

	d_parameter3_label = new QLabel(central_widget);
	d_parameter3_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	central_widget_layout->addWidget(d_parameter3_label,4,3,1,1);

	d_parameter3_edit = new QLineEdit(central_widget);
	d_parameter3_edit->setAlignment(Qt::AlignCenter);
	central_widget_layout->addWidget(d_parameter3_edit,4,4,1,1);
	
	d_parameter3_units_label = new QLabel(central_widget);
	central_widget_layout->addWidget(d_parameter3_units_label,4,5,1,1);
	
	d_parameter4_label = new QLabel(central_widget);
	d_parameter4_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	central_widget_layout->addWidget(d_parameter4_label,5,3,1,1);
	
	d_parameter4_edit = new QLineEdit(central_widget);
	d_parameter4_edit->setAlignment(Qt::AlignCenter);
	central_widget_layout->addWidget(d_parameter4_edit,5,4,1,1);
	
	d_parameter4_units_label = new QLabel(central_widget);
	central_widget_layout->addWidget(d_parameter4_units_label,5,5,1,1);

	control_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	central_widget_layout->addItem(control_hspacer,2,6,4,1);

	exaggeration_label = new QLabel();
	exaggeration_label->setSizePolicy(sizepolicy_mexpanding_preferred);
	exaggeration_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	tool_bar->addWidget(exaggeration_label);

	d_exaggeration_slider = new QSlider();
	d_exaggeration_slider->setMinimum(1);
	d_exaggeration_slider->setMaximum(1000);
	d_exaggeration_slider->setSingleStep(10);
	d_exaggeration_slider->setPageStep(100);
	d_exaggeration_slider->setOrientation(Qt::Horizontal);
	d_exaggeration_slider->setTickPosition(QSlider::TicksBelow);
	d_exaggeration_slider->setSizePolicy(sizepolicy_preferred_preferred);
	d_exaggeration_slider->setMinimumWidth(200);
	tool_bar->addWidget(d_exaggeration_slider);
	
	lower_hline = new QFrame(central_widget);
	lower_hline->setFrameShape(QFrame::HLine);
	lower_hline->setFrameShadow(QFrame::Sunken);
	central_widget_layout->addWidget(lower_hline,6,0,1,7);
	
	copyright_label = new QLabel(this);
	copyright_label->setFont(copyright_font);
	central_widget_layout->addWidget(copyright_label,7,0,1,7);

	QWidget::setTabOrder(d_deflection_model_type_combo,d_range_y_edit);
	QWidget::setTabOrder(d_range_y_edit,d_range_z_edit);
	QWidget::setTabOrder(d_range_z_edit,d_parameter1_edit);
	QWidget::setTabOrder(d_parameter1_edit,d_parameter2_edit);
	QWidget::setTabOrder(d_parameter2_edit,d_parameter3_edit);
	QWidget::setTabOrder(d_parameter3_edit,d_parameter4_edit);
	QWidget::setTabOrder(d_parameter4_edit,d_exaggeration_slider);

	this->setWindowTitle(QStringLiteral("Deflection Simulator"));	// version set in Menu_About
	dock_widget->setWindowTitle(QStringLiteral("Measurement Editor"));

	d_deflection_model_type_combo->addItem(QStringLiteral("Generic Deflection"),TDisplayWidget::GENERIC_MODEL);
	d_deflection_model_type_combo->addItem(QStringLiteral("Renishaw Deflection"),TDisplayWidget::RENISHAW_MODEL);
	d_deflection_model_type_combo->addItem(QStringLiteral("LK Deflection"),TDisplayWidget::LK_MODEL);

	deflection_model_label->setText(QStringLiteral("Deflection Model:"));
	range_y_label->setText(QStringLiteral("Y Range:"));
	range_y_units_label->setText(QStringLiteral("mm"));
	range_z_label->setText(QStringLiteral("Z Range:"));
	range_z_units_label->setText(QStringLiteral("mm"));
	exaggeration_label->setText(QStringLiteral("Exaggeration:"));
	d_add_measurement_button->setText(QStringLiteral("+"));
	d_remove_measurement_button->setText(QStringLiteral("-"));
	copyright_label->setText(QStringLiteral("Copyright (C) 2021 Select Calibration Incorporated.  www.selectcalibration.ca"));

	d_action_new->setText(QStringLiteral("New"));
	d_action_init->setText(QStringLiteral("Init"));
	d_action_quit->setText(QStringLiteral("Quit"));
	d_action_about->setText(QStringLiteral("About"));
	
	file_menu->setTitle(QStringLiteral("File"));
	view_menu->setTitle(QStringLiteral("View"));
	help_menu->setTitle(QStringLiteral("Help"));
	

	// default values
	d_parameter1_edit->setText(QString("%1").arg(d_settings->value(QStringLiteral("Model_Parameter1"),0.0).toDouble(),0,'f',4));
	d_parameter2_edit->setText(QString("%1").arg(d_settings->value(QStringLiteral("Model_Parameter2"),0.0).toDouble(),0,'f',4));
	d_parameter3_edit->setText(QString("%1").arg(d_settings->value(QStringLiteral("Model_Parameter3"),0.0).toDouble(),0,'f',4));
	d_parameter4_edit->setText(QString("%1").arg(d_settings->value(QStringLiteral("Model_Parameter4"),0.0).toDouble(),0,'f',4));
	d_range_y_edit->setText(QString("%1").arg(d_settings->value(QStringLiteral("Range_Y"),1000.0).toDouble(),0,'f',3));
	d_range_z_edit->setText(QString("%1").arg(d_settings->value(QStringLiteral("Range_Z"),1000.0).toDouble(),0,'f',3));
	
	d_measurement_data.Load_Data(d_data_path + QStringLiteral("/measurements.xml"));
	
	ids = d_measurement_data.Get_Measurement_Ids();
	
	for(id_iter = ids.begin();id_iter != ids.end();++id_iter)
	{
		list_item = new QListWidgetItem();
		
		list_item->setText(QString("Measurement %1").arg(*id_iter));
		list_item->setData(Qt::UserRole,(*id_iter));
		
		d_measurement_list->addItem(list_item);
		
		if(d_measurement_data.Get_Measurement((*id_iter),&measurement))
		{
			dval = measurement.nominal_length / 2.0;
			d_display_widget->Add_Length_Measurement(measurement.id,measurement.pos + measurement.vec * dval,measurement.pos - measurement.vec * dval);
		}
	}
	
	d_remove_measurement_button->setEnabled(false);
	d_measurement_edit_widget->setEnabled(false);
	
	// make sure range is non-zero
	dval = d_range_y_edit->text().toDouble();
	if(dval < 1.0) d_range_y_edit->setText(QStringLiteral("1000.000"));
	
	dval = d_range_z_edit->text().toDouble();
	if(dval < 1.0) d_range_z_edit->setText(QStringLiteral("1000.000"));
	
	d_deflection_model_type_combo->setCurrentIndex(d_settings->value(QStringLiteral("Deflection_Model"),0).toInt());
	this->Deflection_Model_Changed(d_deflection_model_type_combo->currentIndex());
	
	connect(d_deflection_model_type_combo,static_cast<void (QComboBox::*)(int)>(&QComboBox::currentIndexChanged),this,&TDeflectionSimulator::Deflection_Model_Changed);
	connect(d_parameter1_edit,&QLineEdit::textEdited,this,&TDeflectionSimulator::Parameter1_Changed);
	connect(d_parameter2_edit,&QLineEdit::textEdited,this,&TDeflectionSimulator::Parameter2_Changed);
	connect(d_parameter3_edit,&QLineEdit::textEdited,this,&TDeflectionSimulator::Parameter3_Changed);
	connect(d_parameter4_edit,&QLineEdit::textEdited,this,&TDeflectionSimulator::Parameter4_Changed);
	connect(d_range_y_edit,&QLineEdit::textEdited,this,&TDeflectionSimulator::Range_Y_Changed);
	connect(d_range_z_edit,&QLineEdit::textEdited,this,&TDeflectionSimulator::Range_Z_Changed);		
	connect(d_exaggeration_slider,&QSlider::valueChanged,this,&TDeflectionSimulator::Exaggeration_Changed);

	connect(d_add_measurement_button,&QToolButton::clicked,this,&TDeflectionSimulator::Add_Measurement);
	connect(d_remove_measurement_button,&QToolButton::clicked,this,&TDeflectionSimulator::Remove_Measurements);
	connect(d_measurement_list,&QListWidget::itemSelectionChanged,this,&TDeflectionSimulator::Measurement_Selection_Changed);
	connect(d_measurement_edit_widget,&TMeasurementEditWidget::Data_Changed,this,&TDeflectionSimulator::Measurement_Data_Changed);
	
	connect(d_action_new,&QAction::triggered,this,&TDeflectionSimulator::Menu_New);
	connect(d_action_init,&QAction::triggered,this,&TDeflectionSimulator::Menu_Init);
	connect(d_action_quit,SIGNAL(triggered(void)),this,SLOT(close(void)));
	connect(d_action_about,&QAction::triggered,this,&TDeflectionSimulator::Menu_About);
}

TDeflectionSimulator::~TDeflectionSimulator(void)
{
}

void TDeflectionSimulator::Deflection_Model_Changed(
	int									index)
{
	TDisplayWidget::TDeflectionModelType model;
	double								param1;
	double								param2;
	double								param3;
	double								param4;
	double								range_y;
	double								range_z;
	bool								state;
	
	model = static_cast<TDisplayWidget::TDeflectionModelType>(d_deflection_model_type_combo->itemData(index).toInt());
	
	switch(model)
	{
		case TDisplayWidget::NO_MODEL:
			d_parameter1_label->setVisible(false);
			d_parameter1_edit->setVisible(false);
			d_parameter1_units_label->setVisible(false);
			
			d_parameter2_label->setVisible(false);
			d_parameter2_edit->setVisible(false);
			d_parameter2_units_label->setVisible(false);
			
			d_parameter3_label->setVisible(false);
			d_parameter3_edit->setVisible(false);
			d_parameter3_units_label->setVisible(false);
			
			d_parameter4_label->setVisible(false);
			d_parameter4_edit->setVisible(false);
			d_parameter4_units_label->setVisible(false);
			break;
			
		case TDisplayWidget::GENERIC_MODEL:
			d_parameter1_label->setText(QStringLiteral("Tower Rotation:"));
			d_parameter1_units_label->setText(QStringLiteral("mm/m"));
			
			d_parameter2_label->setText(QStringLiteral("Correction Y:"));
			d_parameter2_units_label->setText(QStringLiteral("mm"));
			
			d_parameter3_label->setText(QStringLiteral("Correction Z:"));
			d_parameter3_units_label->setText(QStringLiteral("mm"));
			
			d_parameter1_label->setVisible(true);
			d_parameter1_edit->setVisible(true);
			d_parameter1_units_label->setVisible(true);
			
			d_parameter2_label->setVisible(true);
			d_parameter2_edit->setVisible(true);
			d_parameter2_units_label->setVisible(true);
			
			d_parameter3_label->setVisible(true);
			d_parameter3_edit->setVisible(true);
			d_parameter3_units_label->setVisible(true);
			
			d_parameter4_label->setVisible(false);
			d_parameter4_edit->setVisible(false);
			d_parameter4_units_label->setVisible(false);
			break;
			
		case TDisplayWidget::RENISHAW_MODEL:
			d_parameter1_label->setText(QStringLiteral("Tilt:"));
			d_parameter1_units_label->setText(QStringLiteral("mRad/m"));
			
			d_parameter2_label->setText(QStringLiteral("Bend2:"));
			d_parameter2_units_label->setText(QStringLiteral("mm/m2/m"));
			
			d_parameter3_label->setText(QStringLiteral("Bend3:"));
			d_parameter3_units_label->setText(QStringLiteral("mm/m3/m"));
			
			d_parameter1_label->setVisible(true);
			d_parameter1_edit->setVisible(true);
			d_parameter1_units_label->setVisible(true);
			
			d_parameter2_label->setVisible(true);
			d_parameter2_edit->setVisible(true);
			d_parameter2_units_label->setVisible(true);
			
			d_parameter3_label->setVisible(true);
			d_parameter3_edit->setVisible(true);
			d_parameter3_units_label->setVisible(true);
			
			d_parameter4_label->setVisible(false);
			d_parameter4_edit->setVisible(false);
			d_parameter4_units_label->setVisible(false);
			break;
			
		case TDisplayWidget::LK_MODEL:
			d_parameter1_label->setText(QStringLiteral("ZCRS.R1.C1:"));
			d_parameter1_units_label->setText(QStringLiteral("mm/mm"));
			
			d_parameter2_label->setText(QStringLiteral("ZCRS.R2.C1:"));
			d_parameter2_units_label->setText(QStringLiteral("mm/mm"));
			
			d_parameter3_label->setText(QStringLiteral("ZCRS.R3.C1:"));
			d_parameter3_units_label->setText(QStringLiteral("arcsec"));
			
			d_parameter4_label->setText(QStringLiteral("ZCRS.R4.C1:"));
			d_parameter4_units_label->setText(QStringLiteral("mm/mm"));
		
			d_parameter1_label->setVisible(true);
			d_parameter1_edit->setVisible(true);
			d_parameter1_units_label->setVisible(true);

			d_parameter2_label->setVisible(true);
			d_parameter2_edit->setVisible(true);
			d_parameter2_units_label->setVisible(true);
			
			d_parameter3_label->setVisible(true);
			d_parameter3_edit->setVisible(true);
			d_parameter3_units_label->setVisible(true);
			
			d_parameter4_label->setVisible(true);
			d_parameter4_edit->setVisible(true);
			d_parameter4_units_label->setVisible(true);
			break;
	}
	
	d_display_widget->Set_Deflection_Model(model);
	
	param1 = d_parameter1_edit->text().toDouble(&state);
	if(!state) param1 = 0.0;
	
	param2 = d_parameter2_edit->text().toDouble(&state);
	if(!state) param2 = 0.0;
	
	param3 = d_parameter3_edit->text().toDouble(&state);
	if(!state) param3 = 0.0;
	
	param4 = d_parameter4_edit->text().toDouble(&state);
	if(!state) param4 = 0.0;
	
	range_y = d_range_y_edit->text().toDouble(&state);
	if(!state) range_y = 1000.0;
	
	range_z = d_range_z_edit->text().toDouble(&state);
	if(!state) range_z = 1000.0;
	
	d_display_widget->Set_Range(range_y,range_z);
	d_display_widget->Set_Model_Parameters(param1,param2,param3,param4);
	
	this->Update_Actual_Length();
}

void TDeflectionSimulator::Parameter1_Changed(
	const QString						&text)
{
	double								dval;
	bool								state;
	
	dval = text.toDouble(&state);
	
	if(state)
	{
		d_display_widget->Set_Model_Parameter1(dval);
		this->Update_Actual_Length();
	}
}

void TDeflectionSimulator::Parameter2_Changed(
	const QString						&text)
{
	double								dval;
	bool								state;
	
	dval = text.toDouble(&state);
	
	if(state)
	{
		d_display_widget->Set_Model_Parameter2(dval);
		this->Update_Actual_Length();
	}
}

void TDeflectionSimulator::Parameter3_Changed(
	const QString						&text)
{
	double								dval;
	bool								state;
	
	dval = text.toDouble(&state);
	
	if(state)
	{
		d_display_widget->Set_Model_Parameter3(dval);
		this->Update_Actual_Length();
	}
}

void TDeflectionSimulator::Parameter4_Changed(
	const QString						&text)
{
	double								dval;
	bool								state;
	
	dval = text.toDouble(&state);
	
	if(state)
	{
		d_display_widget->Set_Model_Parameter4(dval);
		this->Update_Actual_Length();
	}
}

void TDeflectionSimulator::Range_Y_Changed(
	const QString						&text)
{
	double								dval;
	bool								state;
	
	dval = text.toDouble(&state);
	
	if(state)
	{
		d_display_widget->Set_Range_Y(dval);
		this->Update_Actual_Length();
	}
}

void TDeflectionSimulator::Range_Z_Changed(
	const QString						&text)
{
	double								dval;
	bool								state;
	
	dval = text.toDouble(&state);
	
	if(state)
	{
		d_display_widget->Set_Range_Z(dval);
		this->Update_Actual_Length();
	}
}

void TDeflectionSimulator::Exaggeration_Changed(
	int									value)
{
	d_display_widget->Set_Exaggeration(value);
}

void TDeflectionSimulator::Add_Measurement(void)
{
	int									id;
	TMeasurementData::TMeasurement		measurement;
	double								dval;
	QListWidgetItem						*item;
	
	id = d_measurement_data.Add_Measurement();
	
	if(d_measurement_edit_widget->isEnabled())
	{
		measurement.id = id;
		
		measurement.pos = d_measurement_edit_widget->Position();
		measurement.vec = d_measurement_edit_widget->Axis();
		
		measurement.nominal_length = d_measurement_edit_widget->Nominal_Length();
		
		d_measurement_data.Update_Measurement(measurement);
	}

	
	item = new QListWidgetItem();
	
	item->setText(QString("Measurement %1").arg(id));
	item->setData(Qt::UserRole,id);
	
	d_measurement_list->addItem(item);
	
	if(d_measurement_data.Get_Measurement(id,&measurement))
	{
		dval = measurement.nominal_length / 2.0;
		d_display_widget->Add_Length_Measurement(measurement.id,measurement.pos + measurement.vec * dval,measurement.pos - measurement.vec * dval);
	}
	
	d_measurement_list->setCurrentItem(item,QItemSelectionModel::ClearAndSelect);
}

void TDeflectionSimulator::Remove_Measurements(void)
{
	QList<QListWidgetItem*>				items;
	QList<QListWidgetItem*>::iterator	iter;
	QListWidgetItem*					item;
	int									row;
	int									id;

	items = d_measurement_list->selectedItems();
	
	for(iter = items.begin();iter != items.end();++iter)
	{
		row = d_measurement_list->row(*iter);
		
		item = d_measurement_list->takeItem(row);
		id = item->data(Qt::UserRole).toInt();
		
		delete item;
		
		d_measurement_data.Remove_Measurement(id);
		d_display_widget->Remove_Length_Measurement(id);
	}
	
	d_measurement_edit_widget->setEnabled(false);
	d_measurement_edit_widget->Clear_Display();

}

void TDeflectionSimulator::Measurement_Selection_Changed(void)
{
	QList<QListWidgetItem*>				items;
	QListWidgetItem						*item;
	int									id;
	TMeasurementData::TMeasurement		measurement;
	
	items = d_measurement_list->selectedItems();
	
	d_remove_measurement_button->setEnabled(items.size() > 0);
	d_measurement_edit_widget->setEnabled(false);

	if(items.size() == 1)
	{
		item = items[0];
		
		id = item->data(Qt::UserRole).toInt();
		
		if(d_measurement_data.Get_Measurement(id,&measurement))
		{
			d_measurement_edit_widget->Set_Id(measurement.id);
			d_measurement_edit_widget->Set_Nominal_Length(measurement.nominal_length);

			d_measurement_edit_widget->Set_Position(measurement.pos);
			d_measurement_edit_widget->Set_Axis(measurement.vec);
			
			d_measurement_edit_widget->setEnabled(true);
			
			d_display_widget->Select_Length_Measurement(id);
			
			this->Update_Actual_Length();
			
		}
		else
		{
			d_measurement_edit_widget->Clear_Display();
			d_display_widget->Select_Length_Measurement(-1);
		}
	}
	else
	{
		d_measurement_edit_widget->Clear_Display();
		d_display_widget->Select_Length_Measurement(-1);
	}
}

void TDeflectionSimulator::Measurement_Data_Changed(void)
{
	TMeasurementData::TMeasurement		measurement;
	double								dval;
	
	measurement.id = d_measurement_edit_widget->Id();
	measurement.nominal_length = d_measurement_edit_widget->Nominal_Length();
	measurement.pos = d_measurement_edit_widget->Position();
	measurement.vec = d_measurement_edit_widget->Axis();
	
	d_measurement_data.Update_Measurement(measurement);
	
	dval = measurement.nominal_length / 2.0;
	d_display_widget->Update_Length_Measurement(measurement.id,measurement.pos + measurement.vec * dval,measurement.pos - measurement.vec * dval);
	
	this->Update_Actual_Length();
}
void TDeflectionSimulator::Menu_New(void)
{
	int									result;
	bool								prev_state;

	d_msg_box->setText("Delete Measurements");
	d_msg_box->setInformativeText("Do you want to delete all measurements and reset settings?");
	d_msg_box->setDetailedText(QString());
	d_msg_box->setIcon(QMessageBox::Warning);
	d_msg_box->setStandardButtons(QMessageBox::Yes | QMessageBox::No);
	d_msg_box->setDefaultButton(QMessageBox::Yes);
	
	result = d_msg_box->exec();
	
	switch(result)
	{
		case QMessageBox::Yes:
			break;
			
		case QMessageBox::No:
		default:
			return;
	}
	
	prev_state = d_measurement_list->blockSignals(true);
	d_measurement_list->clear();
	d_measurement_list->blockSignals(prev_state);
	
	d_measurement_data.Clear_Measurements();
	d_display_widget->Clear_Length_Measurements();
	
	d_measurement_edit_widget->Clear_Display();
	d_measurement_edit_widget->setEnabled(false);
	
	this->Menu_Init();
}

void TDeflectionSimulator::Menu_Init(void)
{
	double								param1;
	double								param2;
	double								param3;
	double								param4;
	TDisplayWidget::TDeflectionModelType model;
	
	model = static_cast<TDisplayWidget::TDeflectionModelType>(d_deflection_model_type_combo->currentData().toInt());
	
	switch(model)
	{
		case TDisplayWidget::NO_MODEL:
			param1 = 0.0000;
			param2 = 0.0000;
			param3 = 0.0000;
			param4 = 0.0000;
			break;
			
		case TDisplayWidget::GENERIC_MODEL:
		case TDisplayWidget::RENISHAW_MODEL:
			d_parameter1_edit->setText("-0.0500");
			d_parameter2_edit->setText("-0.0100");
			d_parameter3_edit->setText("-0.0100");
			d_parameter4_edit->setText("0.0000");
			
			param1 = -0.0500;
			param2 = -0.0100;
			param3 = -0.0100;
			param4 = 0.0000;
			break;
			
		case TDisplayWidget::LK_MODEL:
			d_parameter1_edit->setText("0.00001");
			d_parameter2_edit->setText("0.0010");
			d_parameter3_edit->setText("-0.0103");
			d_parameter4_edit->setText("0.0010");
			
			param1 = 0.00001;
			param2 = 0.0010;
			param3 = -0.0103;
			param4 = 0.0010;
			break;
	}
	
	d_display_widget->Set_Model_Parameters(param1,param2,param3,param4);
	
	this->Update_Actual_Length();
}

void TDeflectionSimulator::Menu_About(void)
{
	d_msg_box->setText("Help - About");
	d_msg_box->setInformativeText("Deflection Simulator - Version 2.0");
	d_msg_box->setDetailedText(QString());
	d_msg_box->setIcon(QMessageBox::Information);
	d_msg_box->setStandardButtons(QMessageBox::Ok);
	d_msg_box->setDefaultButton(QMessageBox::Ok);
	
	d_msg_box->exec();
}

void TDeflectionSimulator::closeEvent(
	QCloseEvent							*event)
{
	d_settings->setValue("Mainwindow_Position", this->pos());
	d_settings->setValue("Mainwindow_Size", this->size());
	
	d_settings->setValue(QStringLiteral("Model_Parameter1"),d_display_widget->Model_Parameter1());
	d_settings->setValue(QStringLiteral("Model_Parameter2"),d_display_widget->Model_Parameter2());
	d_settings->setValue(QStringLiteral("Model_Parameter3"),d_display_widget->Model_Parameter3());
	d_settings->setValue(QStringLiteral("Model_Parameter4"),d_display_widget->Model_Parameter4());
	d_settings->setValue(QStringLiteral("Range_Y"),d_range_y_edit->text().toDouble());
	d_settings->setValue(QStringLiteral("Range_Z"),d_range_z_edit->text().toDouble());
	d_settings->setValue(QStringLiteral("Deflection_Model"),d_deflection_model_type_combo->currentIndex());
	
	d_measurement_data.Save_Data(d_data_path + QStringLiteral("/measurements.xml"));
	
	event->accept();
}

void TDeflectionSimulator::Update_Actual_Length(void)
{
	TVector3							pos,vec;
	TVector3							pn1,pn2;
	TVector3							pa1,pa2;
	double								radius;
	double								length;
	
	if(d_measurement_edit_widget->isEnabled())
	{
		pos = d_measurement_edit_widget->Position();
		vec = d_measurement_edit_widget->Axis();
		
		radius = d_measurement_edit_widget->Nominal_Length() / 2.0;
		
		pn1 = pos + vec * radius;
		pn2 = pos - vec * radius;
		
		pa1 = d_display_widget->Get_Gauge_Point(pn1);
		pa2 = d_display_widget->Get_Gauge_Point(pn2);
		
		pa1 -= pa2;
		length = pa1.Length();
		
		d_measurement_edit_widget->Set_Actual_Length(length);
	}
}



