/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2025  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#ifndef DISPLAYWIDGETHEADERFILE
#define DISPLAYWIDGETHEADERFILE

#include <QWidget>
#include <QFont>
#include <QSize>
#include <QRect>
#include <QColor>
#include <vector>

#include "../../core/vector2.h"
#include "../../core/vector3.h"

class QPaintEvent;
class QResizeEvent;
class TDeflectionModel;

class TDisplayWidget : public QWidget
{
	Q_OBJECT
	
public:
	
enum TDeflectionModelType
{
	NO_MODEL = 0,
	GENERIC_MODEL,
	RENISHAW_MODEL,
	LK_MODEL
};
	
	// CREATORS
	TDisplayWidget(const QWidget *parent=0, Qt::WindowFlags flags=Qt::WindowFlags());
	virtual ~TDisplayWidget(void);
	
	// ACCESSORS
	virtual QSize minimumSizeHint(void) const{return d_minimum_size;}
	
	double Model_Parameter1(void) const;
	double Model_Parameter2(void) const;
	double Model_Parameter3(void) const;
	double Model_Parameter4(void) const;
	
	TVector3 Get_Gauge_Point(const TVector3 &pnt) const;

	// MANIPULATORS
	void Set_Range_Y(const double &d);
	void Set_Range_Z(const double &d);
	void Set_Range(const double &range_y,const double &range_z);
	
	void Set_Model_Parameter1(const double &param1);
	void Set_Model_Parameter2(const double &param2);
	void Set_Model_Parameter3(const double &param3);
	void Set_Model_Parameter4(const double &param3);
	void Set_Model_Parameters(const double &param1,const double &param2, const double &param3,const double &param4);
	
	void Set_Exaggeration(const int i);
	void Set_Deflection_Model(const TDeflectionModelType &model);
	
	void Clear_Length_Measurements(void);
	void Add_Length_Measurement(const int id, const TVector3 &p1,const TVector3 &p2);
	void Update_Length_Measurement(const int id, const TVector3 &p1,const TVector3 &p2);
	void Select_Length_Measurement(const int id);
	void Remove_Length_Measurement(const int id);
	
signals:
	
private slots:
	
protected:
	virtual void paintEvent(QPaintEvent *event);
	virtual void resizeEvent(QResizeEvent *event);
	
private:
	
struct TLengthMeasurement
{
	TVector3							p1;
	TVector3							p2;
	int									id;
	bool								selected;
};

struct TViewportMat
{
	double								tx[2];
	double								ty[2];
	double								to[2];
	
	TViewportMat& operator*=(const TViewportMat &rhs)
	{
		double							x,y,o[2];
		
		x = tx[0] * rhs.tx[0];
		y = ty[1] * rhs.ty[1];
		o[0] = to[0] * rhs.tx[0] + rhs.to[0];
		o[1] = to[1] * rhs.ty[1] + rhs.to[1];
		
		tx[0] = x;
		ty[1] = y;
		to[0] = o[0];
		to[1] = o[1];
		
		return *this;
	}
	
	void Position_To_Viewport(const double Xd, const double Yd, int *X, int *Y) const
	{
		*X = static_cast<int>(Xd * tx[0] + to[0]);
		*Y = static_cast<int>(Yd * ty[1] + to[1]);
	}
	
	void Viewport_To_Position(const int X, const int Y, double *Xd, double *Yd) const
	{
		*Xd = (static_cast<double>(X) - to[0]) / tx[0];
		*Yd = (static_cast<double>(Y) - to[1]) / ty[1];
	}
};
	
	TViewportMat						d_viewport_mat;
	QSize								d_minimum_size;
	QRect								d_grid_rect;
	bool								d_update_viewport;
	TDeflectionModel					*d_deflection_model;
	double								d_error_multiplier;
	std::vector<TLengthMeasurement>		d_length_measurements;

	TVector3 Length_Volume_Minimum(void) const;
	TVector3 Length_Volume_Maximum(void) const;
	
	void Draw_Arrow(QPainter * const painter,const QPoint &p1,const QPoint &p2,int arrow_length,int arrow_width,const bool start_only = false) const;
	void Draw_Gauge(QPainter * const painter,const QPoint &p1,const QPoint &p2,const QString &title,int gauge_width) const;
	
	TVector3 Calculate_Correction(const TVector3 &pos) const;
	inline TVector3 Calculate_Correction(const double &ypos, const double &zpos) const {return this->Calculate_Correction(TVector3(0.0,ypos,zpos));}

	void Update_Viewport(const TVector3 &min,const TVector3 &max);
	
	// NOT IMPLEMENTED
	TDisplayWidget(const TDisplayWidget&);
	TDisplayWidget& operator=(const TDisplayWidget&);
};

#endif
