/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2025  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <assert.h>

#include "../../core/xmlfile.h"

#include "measurementdata.h"

TMeasurementData::TMeasurementData(void)
{
}

TMeasurementData::~TMeasurementData(void)
{
}

bool TMeasurementData::Get_Measurement(
	const int							id,
	TMeasurement						* const measurement) const
{
	std::vector<TMeasurementData::TMeasurement>::const_iterator iter;

	assert(measurement);
	
	for(iter = d_data.begin();iter != d_data.end();++iter)
	{
		if((*iter).id == id)
		{
			*measurement = (*iter);
			return true;
		}
	}

	return false;
}

std::vector<int> TMeasurementData::Get_Measurement_Ids(void) const
{
	std::vector<int>					ids;
	std::vector<TMeasurementData::TMeasurement>::const_iterator iter;
	
	for(iter = d_data.begin();iter != d_data.end();++iter)
	{
		ids.push_back((*iter).id);
	}
	
	return ids;
}

bool TMeasurementData::Save_Data(
	const QString						&file_name) const
{
	TXmlFile							xml_file;
	QDomElement							e0;
	QDomElement							e1;
	QString								text;
	std::vector<TMeasurementData::TMeasurement>::const_iterator iter;
	
	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Reset("Deflection_Simulator_Measurement_Data",1);
	
		for(iter = d_data.begin();iter != d_data.end();++iter)
		{
			e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Measurement"));
			
			e1 = xml_file.Create_Node(&e0,QStringLiteral("Id"));
			xml_file.Write_Text_Node(&e1,(*iter).id);
			
			e1 = xml_file.Create_Node(&e0,QStringLiteral("Nominal_Length"));
			xml_file.Write_Text_Node(&e1,QString("%1").arg((*iter).nominal_length,0,'f',6));
			
			e1 = xml_file.Create_Node(&e0,QStringLiteral("Position"));
			text = QString("%1,%2,%3").arg((*iter).pos.x,0,'f',6).arg((*iter).pos.y,0,'f',6).arg((*iter).pos.z,0,'f',6);
			xml_file.Write_Text_Node(&e1,text);
			
			e1 = xml_file.Create_Node(&e0,QStringLiteral("Axis"));
			text = QString("%1,%2,%3").arg((*iter).vec.x,0,'f',12).arg((*iter).vec.y,0,'f',12).arg((*iter).vec.z,0,'f',12);
			xml_file.Write_Text_Node(&e1,text);
		}
		
		xml_file.Write_File();
	}
	catch(TXmlFileException exception)
	{
		d_last_error = exception.ErrorText();
		return false;
	}
 
	return true;
}

bool TMeasurementData::Load_Data(
	const QString						&file_name)
{
	TXmlFile							xml_file;
	int									version;

	d_data.clear();
	
	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Open("Deflection_Simulator_Measurement_Data",&version);
		
		switch(version)
		{
			case 1:
				this->Load_Data_V1(&xml_file);
				break;
				
			default:
				throw TXmlFileException(QStringLiteral("ERR:  File version not recognized"));
				break;
		}
	}
	catch(TXmlFileException exception)
	{
		d_last_error = exception.ErrorText();
		return false;
	}
	
	return true;
}

int TMeasurementData::Add_Measurement(void)
{
	int									id(1);
	TMeasurementData::TMeasurement		measurement;
	
	while(Measurement_Id_Exists(id))
	{
		++id;
	};
	
	measurement.id = id;
	measurement.vec.z = 1.0;
	measurement.nominal_length = 1000.0;
	
	d_data.push_back(measurement);
	
	return id;
}

bool TMeasurementData::Update_Measurement(
	const TMeasurementData::TMeasurement &measurement)
{
	std::vector<TMeasurementData::TMeasurement>::iterator iter;
	
	for(iter = d_data.begin();iter != d_data.end();++iter)
	{
		if((*iter).id == measurement.id)
		{
			(*iter) = measurement;
			return true;
		}
	}
	
	return false;
}

bool TMeasurementData::Remove_Measurement(
	const int							id)
{
	std::vector<TMeasurementData::TMeasurement>::iterator iter;
	
	for(iter = d_data.begin();iter != d_data.end();++iter)
	{
		if((*iter).id == id)
		{
			d_data.erase(iter);
			return true;
		}
	}

	return false;
}

void TMeasurementData::Clear_Measurements(void)
{
	d_data.clear();
}

void TMeasurementData::Load_Data_V1(
	TXmlFile							* const xml_file)
{
	QDomElement							e0;
	QDomElement							e1;
	QString								text;
	QStringList							list;
	TMeasurementData::TMeasurement		measurement;
	static const double					ZERO_EPSILON(0.000001);

	assert(xml_file);
	
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Measurement"),true);
	
	while(!e0.isNull())
	{
		e1 = xml_file->Get_Node(&e0,QStringLiteral("Id"));
		text = xml_file->Read_Text_Node(&e1);
		measurement.id = text.toInt();
		
		e1 = xml_file->Get_Node(&e0,QStringLiteral("Nominal_Length"));
		text = xml_file->Read_Text_Node(&e1);
		measurement.nominal_length = text.toDouble();
			
		e1 = xml_file->Get_Node(&e0,QStringLiteral("Position"));
		text = xml_file->Read_Text_Node(&e1);
		list = text.split(',');
		
		if(list.size() == 3)
		{
			measurement.pos.x = list[0].toDouble();
			measurement.pos.y = list[1].toDouble();
			measurement.pos.z = list[2].toDouble();
		}
		else
		{
			throw TXmlFileException(QStringLiteral("ERR:  Measurement position must have three values"));
		}
		
		e1 = xml_file->Get_Node(&e0,QStringLiteral("Axis"));
		text = xml_file->Read_Text_Node(&e1);
		list = text.split(',');
		
		if(list.size() == 3)
		{
			measurement.vec.x = 0.0;
			measurement.vec.y = list[1].toDouble();
			measurement.vec.z = list[2].toDouble();
		}
		else
		{
			throw TXmlFileException(QStringLiteral("ERR:  Measurement axis must have three values"));
		}
		
		if(measurement.vec.Length() < ZERO_EPSILON)
		{
			throw TXmlFileException(QStringLiteral("ERR:  Measurement axis not specified"));
		}
		
		measurement.vec.Normal();

		
		this->Add_Measurement(measurement);
		
		e0 = xml_file->Get_Sibling_Node(&e0,QStringLiteral("Measurement"),true);
	}
}

void TMeasurementData::Add_Measurement(
	const TMeasurementData::TMeasurement &measurement)
{
	// only add if data is valid.
	// id must be greater than zero and unique
	
	if(measurement.id < 1)
	{
		return;
	}
	
	if(!Measurement_Id_Exists(measurement.id))
	{
		d_data.push_back(measurement);
	}
}

bool TMeasurementData::Measurement_Id_Exists(
	const int							id) const
{
	std::vector<TMeasurementData::TMeasurement>::const_iterator iter;

	for(iter = d_data.begin();iter != d_data.end();++iter)
	{
		if((*iter).id == id)
		{
			return true;
		}
	}

	return false;
}
