/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2025  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QFrame>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QSpacerItem>
#include <QToolButton>

#include "measurementeditwidget.h"

TMeasurementEditWidget::TMeasurementEditWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QFrame								*separator_hline;
	QGridLayout							*widget_layout;
	QLabel								*actual_label;
	QLabel								*actual_units_label;
	QLabel								*i_label;
	QLabel								*j_label;
	QLabel								*k_label;
	QLabel								*nominal_label;
	QLabel								*nominal_units_label;
	QLabel								*x_label;
	QLabel								*x_units_label;
	QLabel								*y_label;
	QLabel								*y_units_label;
	QLabel								*z_label;
	QLabel								*z_units_label;

	widget_layout = new QGridLayout(this);
	
	nominal_label = new QLabel(this);
	nominal_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	widget_layout->addWidget(nominal_label,0,0,1,1);

	d_nominal_edit = new QLineEdit(this);
	d_nominal_edit->setAlignment(Qt::AlignCenter);
	widget_layout->addWidget(d_nominal_edit,0,1,1,1);

	nominal_units_label = new QLabel(this);
	widget_layout->addWidget(nominal_units_label,0,2,1,1);

	actual_label = new QLabel(this);
	actual_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	widget_layout->addWidget(actual_label,1,0,1,1);

	d_actual_edit = new QLineEdit(this);
	d_actual_edit->setReadOnly(true);
	d_actual_edit->setAlignment(Qt::AlignCenter);
	widget_layout->addWidget(d_actual_edit,1,1,1,1);

	actual_units_label = new QLabel(this);
	widget_layout->addWidget(actual_units_label,1,2,1,1);

	separator_hline = new QFrame(this);
	separator_hline->setFrameShape(QFrame::HLine);
	separator_hline->setFrameShadow(QFrame::Sunken);
	widget_layout->addWidget(separator_hline,2,0,1,3);

	x_label = new QLabel(this);
	x_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	widget_layout->addWidget(x_label,3,0,1,1);

	d_x_edit = new QLineEdit(this);
	d_x_edit->setReadOnly(true);
	d_x_edit->setAlignment(Qt::AlignCenter);
	widget_layout->addWidget(d_x_edit,3,1,1,1);

	x_units_label = new QLabel(this);
	widget_layout->addWidget(x_units_label,3,2,1,1);

	y_label = new QLabel(this);
	y_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	widget_layout->addWidget(y_label,4,0,1,1);

	d_y_edit = new QLineEdit(this);
	d_y_edit->setAlignment(Qt::AlignCenter);
	widget_layout->addWidget(d_y_edit,4,1,1,1);

	y_units_label = new QLabel(this);
	widget_layout->addWidget(y_units_label,4,2,1,1);

	z_label = new QLabel(this);
	z_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	widget_layout->addWidget(z_label,5,0,1,1);

	d_z_edit = new QLineEdit(this);
	d_z_edit->setAlignment(Qt::AlignCenter);
	widget_layout->addWidget(d_z_edit,5,1,1,1);

	z_units_label = new QLabel(this);
	widget_layout->addWidget(z_units_label,5,2,1,1);
	
	i_label = new QLabel(this);
	i_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	widget_layout->addWidget(i_label,6,0,1,1);

	d_i_edit = new QLineEdit(this);
	d_i_edit->setReadOnly(true);
	d_i_edit->setAlignment(Qt::AlignCenter);
	widget_layout->addWidget(d_i_edit,6,1,1,1);

	j_label = new QLabel(this);
	j_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	widget_layout->addWidget(j_label,7,0,1,1);

	d_j_edit = new QLineEdit(this);
	d_j_edit->setAlignment(Qt::AlignCenter);
	widget_layout->addWidget(d_j_edit,7,1,1,1);

	k_label = new QLabel(this);
	k_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	widget_layout->addWidget(k_label,8,0,1,1);

	d_k_edit = new QLineEdit(this);
	d_k_edit->setAlignment(Qt::AlignCenter);
	widget_layout->addWidget(d_k_edit,8,1,1,1);
	
	x_label->setText(QStringLiteral("X:"));
	y_label->setText(QStringLiteral("Y:"));
	z_label->setText(QStringLiteral("Z:"));
	i_label->setText(QStringLiteral("I:"));
	j_label->setText(QStringLiteral("J:"));
	k_label->setText(QStringLiteral("k:"));
	nominal_label->setText(QStringLiteral("Nominal:"));
	actual_label->setText(QStringLiteral("Actual:"));
	x_units_label->setText(QStringLiteral("mm"));
	y_units_label->setText(QStringLiteral("mm"));
	z_units_label->setText(QStringLiteral("mm"));
	nominal_units_label->setText(QStringLiteral("mm"));
	actual_units_label->setText(QStringLiteral("mm"));
	
	QWidget::setTabOrder(d_nominal_edit,d_actual_edit);
	QWidget::setTabOrder(d_actual_edit,d_x_edit);
	QWidget::setTabOrder(d_x_edit,d_y_edit);
	QWidget::setTabOrder(d_y_edit,d_z_edit);
	QWidget::setTabOrder(d_z_edit,d_i_edit);
	QWidget::setTabOrder(d_i_edit,d_j_edit);
	QWidget::setTabOrder(d_j_edit,d_k_edit);

	// defaults
	d_id = 0;
	
	connect(d_y_edit,&QLineEdit::textEdited,this,&TMeasurementEditWidget::Text_Edited);
	connect(d_z_edit,&QLineEdit::textEdited,this,&TMeasurementEditWidget::Text_Edited);
	connect(d_j_edit,&QLineEdit::textEdited,this,&TMeasurementEditWidget::Text_Edited);
	connect(d_k_edit,&QLineEdit::textEdited,this,&TMeasurementEditWidget::Text_Edited);
	connect(d_nominal_edit,&QLineEdit::textEdited,this,&TMeasurementEditWidget::Text_Edited);
}

TMeasurementEditWidget::~TMeasurementEditWidget(void)
{
}

double TMeasurementEditWidget::Nominal_Length(void) const
{
	double								dval;
	QString								text;
	bool								state;
	
	text = d_nominal_edit->text();
	
	dval = text.toDouble(&state);
	
	if(!state) dval = 1000.0;
	if(dval < 1.0) dval = 1.0;
	
	return dval;
}

double TMeasurementEditWidget::Actual_Length(void) const
{
	double								dval;
	QString								text;
	bool								state;
	
	text = d_actual_edit->text();
	
	dval = text.toDouble(&state);
	
	if(!state) dval = 1000.0;
	
	return dval;
}

TVector3 TMeasurementEditWidget::Position(void) const
{
	TVector3							v;
	
	v.x = d_x_edit->text().toDouble();
	v.y = d_y_edit->text().toDouble();
	v.z = d_z_edit->text().toDouble();
	
	return v;
}

TVector3 TMeasurementEditWidget::Axis(void) const
{
	TVector3							v;
	static const double					ZERO_EPSILON(0.000001);
	
	v.x = d_i_edit->text().toDouble();
	v.y = d_j_edit->text().toDouble();
	v.z = d_k_edit->text().toDouble();
	
	if(v.Length() < ZERO_EPSILON)
	{
		v.Set(0,0,1);
	}
	
	v.Normal();
	
	return v;
}

void TMeasurementEditWidget::Set_Nominal_Length(
	const double						&length)
{
	d_nominal_edit->setText(QString("%1").arg(length,0,'f',4));
}

void TMeasurementEditWidget::Set_Actual_Length(
	const double						&length)
{
	d_actual_edit->setText(QString("%1").arg(length,0,'f',4));
}

void TMeasurementEditWidget::Set_Position(
	const TVector3						&pos)
{
	d_x_edit->setText(QString("%1").arg(pos.x,0,'f',4));
	d_y_edit->setText(QString("%1").arg(pos.y,0,'f',4));
	d_z_edit->setText(QString("%1").arg(pos.z,0,'f',4));
}

void TMeasurementEditWidget::Set_Axis(
	const TVector3						&vec)
{
	d_i_edit->setText(QString("%1").arg(vec.x,0,'f',6));
	d_j_edit->setText(QString("%1").arg(vec.y,0,'f',6));
	d_k_edit->setText(QString("%1").arg(vec.z,0,'f',6));
}

void TMeasurementEditWidget::Clear_Display(void)
{
	d_nominal_edit->setText(QString());
	d_actual_edit->setText(QString());

	d_x_edit->setText(QString());
	d_y_edit->setText(QString());
	d_z_edit->setText(QString());

	d_i_edit->setText(QString());
	d_j_edit->setText(QString());
	d_k_edit->setText(QString());
}

void TMeasurementEditWidget::Text_Edited(void)
{
	emit Data_Changed();
}


