/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2025  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <assert.h>

#include "interpolation.h"

static const double						ZERO_EPSILON(0.000001);

static bool Solve_Mat(int rows,int columns,double *mat,double *results);
static double Sum_Of_X(const std::vector<TVector2> &pnts);
static double Sum_Of_Y(const std::vector<TVector2> &pnts);
static double Sum_Of_XY(const std::vector<TVector2> &pnts);
static double Sum_Of_Xx4(const std::vector<TVector2> &pnts);
static double Sum_Of_Xx3(const std::vector<TVector2> &pnts);
static double Sum_Of_Xx2(const std::vector<TVector2> &pnts);
static double Sum_Of_Xx2Y(const std::vector<TVector2> &pnts);
static double A_Term(const std::vector<TVector2> &pnts);
static double B_Term(const std::vector<TVector2> &pnts);
static double C_Term(const std::vector<TVector2> &pnts);


bool Regression::Get_Slope_Through_Zero(
	const std::vector<TVector2>			&pnts,
	double								*slope)
{
	double								x,y;
	std::vector<TVector2>::const_iterator pnt_iter;
	static const double					ZERO_EPSILON(0.000001);
	
	assert(slope);
	
	*slope = 0;
	
	if(pnts.empty())
	{
		return false;
	}
		
	x=y=0;
	
	for(pnt_iter = pnts.begin();pnt_iter != pnts.end();++pnt_iter)
	{
		if((*pnt_iter).x > 0)
		{
			x += (*pnt_iter).x;
			y += (*pnt_iter).y;
		}
		else
		{
			x -= (*pnt_iter).x;
			y -= (*pnt_iter).y;
		}
	}
	
	x /= static_cast<double>(pnts.size());
	y /= static_cast<double>(pnts.size());
	
	if(fabs(x) < fabs(y) || fabs(x) < ZERO_EPSILON)
	{
		return Fit_Line(pnts,slope);
	}
	
	*slope = y / x;
	
	return true;
}

bool Regression::Get_Endfit_Slope(
	const std::vector<TVector2>			&pnts,
	double								*slope)
{
	TVector2							pnt_start;
	TVector2							pnt_end;
	static const double					ZERO_EPSILON(0.000001);
	
	assert(slope);
	
	*slope = 0;
	
	if(pnts.size() < 2)
	{
		return false;
	}
	
	pnt_start = pnts[0];
	pnt_end = pnts[pnts.size() - 1];
	
	if(fabs(pnt_end.x - pnt_start.x) > ZERO_EPSILON)
	{
		*slope = (pnt_end.y - pnt_start.y) / (pnt_end.x - pnt_start.x);
		
		return true;
	}
	
	return false;
}

bool Regression::Fit_Line(
	const std::vector<TVector2>			&pnts,
	double 								*slope,
	double 								*yintercept)
{
	double								XY,X2,Y2,Xs,Ys;
	int									point_count;
	std::vector<TVector2>::const_iterator 	pnt_iter;
	
	point_count = static_cast<int>(pnts.size());
	
	if(point_count < 2)
	{
		return false;
	}
	
	assert(slope);
	
	XY = X2 = Y2 = Xs = Ys =0;
	
	for(pnt_iter = pnts.begin();pnt_iter != pnts.end();++pnt_iter)
	{
		XY += ((*pnt_iter).x * (*pnt_iter).y);
		X2 += ((*pnt_iter).x * (*pnt_iter).x);
		Y2 += ((*pnt_iter).y * (*pnt_iter).y);
		Xs += (*pnt_iter).x;
		Ys += (*pnt_iter).y;
	}
	
	*slope = static_cast<double>(point_count) * XY - Xs * Ys;
	*slope /= (static_cast<double>(point_count) * (X2) - Xs * Xs);
	
	if(yintercept)
	{
		*yintercept = Ys - (*slope) * Xs;
		*yintercept /= static_cast<double>(point_count);
	}
	
	return true;
}

bool LeastSquares::Fit_Curve(
	const std::vector<TVector2>			&pnts,
	double								*a,
	double								*b,
	double								*c)
{
	int									size;
	
	assert(a);
	assert(b);
	assert(c);
	
	size = static_cast<int>(pnts.size());
	
	if(size < 3)
	{
		return false;
	}
	
	*a = A_Term(pnts);
	*b = B_Term(pnts);
	*c = C_Term(pnts);
		
	return true;
}

bool LeastSquares::Fit_Curve(
	const std::vector<TVector2>			&pnts,
	double								*a,
	double								*b,
	double								*c,
	double								*d)
{
	std::vector<TVector2>::const_iterator iter;
	int									size;
	int									cntr;
	int									sub_cntr;
	double								mat[4][5];
	double								results[4];
	
	assert(a);
	assert(b);
	assert(c);
	assert(d);
	
	*a = 0.0;
	*b = 0.0;
	*c = 0.0;
	*d = 0.0;

	size = static_cast<int>(pnts.size());
	
	if(size < 4)
	{
		return false;
	}
	
	for(cntr = 0;cntr < 4;++cntr)
	{
		mat[cntr][4] = 0;
		for(iter = pnts.begin();iter != pnts.end();++iter)
		{
			mat[cntr][4] -= pow((*iter).x,static_cast<double>(cntr)) * (*iter).y;
		}
		
		for(sub_cntr = 0;sub_cntr < 4;++sub_cntr)
		{
			mat[cntr][sub_cntr] = 0;
			for(iter = pnts.begin();iter != pnts.end();++iter)
			{
				mat[cntr][sub_cntr] -= pow((*iter).x,static_cast<double>(cntr + sub_cntr));
			}
		}
	}
	
	if(Solve_Mat(4,5,mat[0],results))
	{
		*a = results[3];
		*b = results[2];
		*c = results[1];
		*d = results[0];
		
		return true;
	}
	
	return false;
}

bool LeastSquares::Fit_Curve(
	const std::vector<TVector2>			&pnts,
	double								*a,
	double								*b,
	double								*c,
	double								*d,
	double								*e)
{
	std::vector<TVector2>::const_iterator iter;
	int									size;
	int									cntr;
	int									sub_cntr;
	double								mat[5][6];
	double								results[5];
	
	assert(a);
	assert(b);
	assert(c);
	assert(d);
	assert(e);
	
	*a = 0.0;
	*b = 0.0;
	*c = 0.0;
	*d = 0.0;
	*e = 0.0;
	
	size = static_cast<int>(pnts.size());
	
	if(size < 5)
	{
		return false;
	}
	
	for(cntr = 0;cntr < 5;++cntr)
	{
		mat[cntr][5] = 0;
		for(iter = pnts.begin();iter != pnts.end();++iter)
		{
			mat[cntr][5] -= pow((*iter).x,static_cast<double>(cntr)) * (*iter).y;
		}
		
		for(sub_cntr = 0;sub_cntr < 5;++sub_cntr)
		{
			mat[cntr][sub_cntr] = 0;
			for(iter = pnts.begin();iter != pnts.end();++iter)
			{
				mat[cntr][sub_cntr] -= pow((*iter).x,static_cast<double>(cntr + sub_cntr));
			}
		}
	}
	
	if(Solve_Mat(5,6,mat[0],results))
	{
		*a = results[4];
		*b = results[3];
		*c = results[2];
		*d = results[1];
		*e = results[0];
		
		return true;
	}
	
	return false;
}

bool Solve_Mat(
	int									rows,
	int									columns,
	double								*mat,
	double								*results)
{
	int									i,j,k;
	int									cntr;
	double								dval;
	static const double					ZERO_EPSILON(0.000000000001);
	
	assert(mat);
	assert(results);
	
	for(i = 0;i < rows;++i)
	{
		if(fabs(*(mat + i * columns + i)) < ZERO_EPSILON)
		{
			for(j = i + 1; j < rows; ++j)
			{
				if(fabs(*(mat + j * columns + i)) > ZERO_EPSILON)
				{
					for(k = i; k < columns; ++k)
					{
						dval = *(mat + i * columns + k);
						*(mat + i * columns + k) = *(mat + j * columns + k);
						*(mat + j * columns + k) = dval;
					}
					break;
				}
			}
		}
		
		dval = *(mat + i * columns + i);
		if (fabs(dval) < ZERO_EPSILON)
		{
			// no unique solution
			return false;
		}
		
		// normalize
		for(j = i; j < columns; ++j)
		{
			*(mat + i * columns + j) /= dval;
		}
		
		for(j = 0; j < rows; ++j)
		{
			// Skip the ith equation.
			if (j != i)
			{
				dval = *(mat + j * columns + i);
				for(cntr = 0; cntr < columns; ++cntr)
				{
					*(mat + j * columns + cntr) -= *(mat + i * columns + cntr) * dval;
				}
			}
		}
	}
	
	for(cntr = 0; cntr < rows; ++cntr)
	{
		results[cntr] = *(mat + cntr * columns + rows);
	}
	
	return true;
}

double Sum_Of_X(
	const std::vector<TVector2>			&pnts)
{
	std::vector<TVector2>::const_iterator iter;
	double								res(0);
	
	for(iter = pnts.begin();iter != pnts.end();++iter)
	{
		res += (*iter).x;
	}
	
	return res;
}

double Sum_Of_Y(
	const std::vector<TVector2>			&pnts)
{
	std::vector<TVector2>::const_iterator iter;
	double								res(0);
	
	for(iter = pnts.begin();iter != pnts.end();++iter)
	{
		res += (*iter).y;
	}
	
	return res;

}

double Sum_Of_XY(
	const std::vector<TVector2>			&pnts)
{
	std::vector<TVector2>::const_iterator iter;
	double								res(0);
	
	for(iter = pnts.begin();iter != pnts.end();++iter)
	{
		res += ((*iter).x * (*iter).y);
	}
	
	return res;
}

double Sum_Of_Xx4(
	const std::vector<TVector2>			&pnts)
{
	std::vector<TVector2>::const_iterator iter;
	double								res(0);
	
	for(iter = pnts.begin();iter != pnts.end();++iter)
	{
		res += pow((*iter).x,4);
	}
	
	return res;
}

double Sum_Of_Xx3(
	const std::vector<TVector2>			&pnts)
{
	std::vector<TVector2>::const_iterator iter;
	double								res(0);
	
	for(iter = pnts.begin();iter != pnts.end();++iter)
	{
		res += pow((*iter).x,3);
	}
	
	return res;

}

double Sum_Of_Xx2(
	const std::vector<TVector2>			&pnts)
{
	std::vector<TVector2>::const_iterator iter;
	double								res(0);
	
	for(iter = pnts.begin();iter != pnts.end();++iter)
	{
		res += pow((*iter).x,2);
	}
	
	return res;

}

double Sum_Of_Xx2Y(
	const std::vector<TVector2>			&pnts)
{
	std::vector<TVector2>::const_iterator iter;
	double								res(0);
	
	for(iter = pnts.begin();iter != pnts.end();++iter)
	{
		res += ((*iter).x * (*iter).x) * (*iter).y;
	}
	
	return res;
}

double A_Term(
	const std::vector<TVector2>			&pnts)
{
	double s40 = Sum_Of_Xx4(pnts);
	double s30 = Sum_Of_Xx3(pnts);
	double s20 = Sum_Of_Xx2(pnts);
	double s10 = Sum_Of_X(pnts);
	double s00 = static_cast<double>(pnts.size());
	
	double s21 = Sum_Of_Xx2Y(pnts);
	double s11 = Sum_Of_XY(pnts);
	double s01 = Sum_Of_Y(pnts);
	
	return (s21*(s20 * s00 - s10 * s10) -
			s11*(s30 * s00 - s10 * s20) +
			s01*(s30 * s10 - s20 * s20))
			/
			(s40*(s20 * s00 - s10 * s10) -
			 s30*(s30 * s00 - s10 * s20) +
			 s20*(s30 * s10 - s20 * s20));

}

double B_Term(
	const std::vector<TVector2>			&pnts)
{
	double s40 = Sum_Of_Xx4(pnts);
	double s30 = Sum_Of_Xx3(pnts);
	double s20 = Sum_Of_Xx2(pnts);
	double s10 = Sum_Of_X(pnts);
	double s00 = static_cast<double>(pnts.size());
	
	double s21 = Sum_Of_Xx2Y(pnts);
	double s11 = Sum_Of_XY(pnts);
	double s01 = Sum_Of_Y(pnts);
	
	return (s40*(s11 * s00 - s01 * s10) -
			s30*(s21 * s00 - s01 * s20) +
			s20*(s21 * s10 - s11 * s20))
			/
			(s40 * (s20 * s00 - s10 * s10) -
			 s30 * (s30 * s00 - s10 * s20) +
			 s20 * (s30 * s10 - s20 * s20));
}

double C_Term(
	const std::vector<TVector2>			&pnts)
{
	double s40 = Sum_Of_Xx4(pnts);
	double s30 = Sum_Of_Xx3(pnts);
	double s20 = Sum_Of_Xx2(pnts);
	double s10 = Sum_Of_X(pnts);
	double s00 = static_cast<double>(pnts.size());
	
	double s21 = Sum_Of_Xx2Y(pnts);
	double s11 = Sum_Of_XY(pnts);
	double s01 = Sum_Of_Y(pnts);
	
	return (s40*(s20 * s01 - s10 * s11) -
			s30*(s30 * s01 - s10 * s21) +
			s20*(s30 * s11 - s20 * s21))
			/
			(s40 * (s20 * s00 - s10 * s10) -
			 s30 * (s30 * s00 - s10 * s20) +
			 s20 * (s30 * s10 - s20 * s20));
}

bool Extrapolation::Get_Value(
	const std::vector<TVector2>			&pnts,
    const double						&position,
	double								*value)
{
	TVector2							start_pnt;
	TVector2							end_pnt;
	TVector2							low_pnt;
	TVector2							high_pnt;
	TVector2							pnt;
	TVector2							vec;
	std::vector<TVector2>::const_iterator pnt_iter;
	double								point_range;
	double								u;
	
	assert(value);
	
	*value = 0;
	
	if(pnts.size() < 2)
	{
		return false;
	}
	
	start_pnt = pnts[0];
	end_pnt = pnts[pnts.size()-1];
	
	point_range = end_pnt.x - start_pnt.x;
	
	if(fabs(point_range) < ZERO_EPSILON)
	{
		return false;
	}
	
	// estimate 'u'
	u = position - start_pnt.x;
	u /= point_range;
	
	if(!(u < 0 || u > 1))	// taget position inside point range
	{
		// find points adjacent to target position
		for(pnt_iter = pnts.begin();pnt_iter != (pnts.end() - 1);++pnt_iter)
		{
			low_pnt = *(pnt_iter);
			high_pnt = *(pnt_iter+1);
			
			point_range = high_pnt.x - low_pnt.x;
			
			if(fabs(point_range) > ZERO_EPSILON)
			{
				u = position - low_pnt.x;
				u /= point_range;
				
				if(!(u < 0 || u > 1))	// local taget position inside point range
				{
					vec = high_pnt - low_pnt;
					vec *= (position - low_pnt.x) / vec.x;
					pnt = low_pnt + vec;
					
					*value = pnt.y;

					return true;
				}
			}
			
		}
		
		return false;	
	}
	else if(u < 0)						// target position below start point
	{
		high_pnt = pnts[1];
		low_pnt = pnts[0];
		
		vec = high_pnt - low_pnt;
		vec *= (position - low_pnt.x) / vec.x;
		pnt = low_pnt + vec;
		
		*value = pnt.y;				

	}
	else								// target position above end point
	{
		high_pnt = pnts[pnts.size() - 1];
		low_pnt = pnts[pnts.size() - 2];
		
		vec = low_pnt - high_pnt;
		vec *= (position - high_pnt.x) / vec.x;
		pnt = high_pnt + vec;
		
		*value = pnt.y;

		return true;
	}
	return true;
}

bool Extrapolation::Get_Value_XY(
	 const std::vector<TVector3>		&pnts,
	 const double						&z_position,
	 double								*x_value,
	 double								*y_value)
{
	TVector3							start_pnt;
	TVector3							end_pnt;
	TVector3							low_pnt;
	TVector3							high_pnt;
	TVector3							pnt;
	TVector3							vec;
	std::vector<TVector3>::const_iterator pnt_iter;
	double								point_range;
	double								u;
	
	assert(x_value);
	assert(y_value);
	
	*x_value = 0;
	*y_value = 0;
	
	if(pnts.size() < 2)
	{
		return false;
	}
	
	start_pnt = pnts[0];
	end_pnt = pnts[pnts.size()-1];
	
	point_range = end_pnt.z - start_pnt.z;
	
	if(fabs(point_range) < ZERO_EPSILON)
	{
		return false;
	}
	
	// estimate 'u'
	u = z_position - start_pnt.z;
	u /= point_range;
		
	if(!(u < 0 || u > 1))	// taget position inside point range
	{
		// find points adjacent to target position
		for(pnt_iter = pnts.begin();pnt_iter != (pnts.end() - 1);++pnt_iter)
		{
			low_pnt = *(pnt_iter);
			high_pnt = *(pnt_iter+1);
			
			point_range = high_pnt.z - low_pnt.z;
			
			if(fabs(point_range) > ZERO_EPSILON)
			{
				u = z_position - low_pnt.z;
				u /= point_range;
				
				if(!(u < 0 || u > 1))	// local taget position inside point range
				{
					vec = high_pnt - low_pnt;
					vec *= (z_position - low_pnt.z) / vec.z;
					pnt = low_pnt + vec;
					
					*x_value = pnt.x;
					*y_value = pnt.y;
					
					return true;
				}
			}
			
		}
		
		return false;	
	}
	else if(u < 0)						// target position below start point
	{
		high_pnt = pnts[1];
		low_pnt = pnts[0];
		
		vec = high_pnt - low_pnt;
		vec *= (z_position - low_pnt.z) / vec.z;
		pnt = low_pnt + vec;
		
		*x_value = pnt.x;
		*y_value = pnt.y;
		
	}
	else												// target position above end point
	{
		high_pnt = pnts[pnts.size() - 1];
		low_pnt = pnts[pnts.size() - 2];
		
		vec = low_pnt - high_pnt;
		vec *= (z_position - high_pnt.z) / vec.z;
		pnt = high_pnt + vec;
		
		*x_value = pnt.x;
		*y_value = pnt.y;
		
		return true;
	}
	return true;
}

bool Extrapolation::Get_Value_YZ(
	const std::vector<TVector3>			&pnts,
	const double						&x_position,
	double								*y_value,
	double								*z_value)
{
	TVector3							start_pnt;
	TVector3							end_pnt;
	TVector3							low_pnt;
	TVector3							high_pnt;
	TVector3							pnt;
	TVector3							vec;
	std::vector<TVector3>::const_iterator pnt_iter;
	double								point_range;
	double								u;
	
	assert(y_value);
	assert(z_value);
	
	*y_value = 0;
	*z_value = 0;
	
	if(pnts.size() < 2)
	{
		return false;
	}
	
	start_pnt = pnts[0];
	end_pnt = pnts[pnts.size()-1];
	
	point_range = end_pnt.x - start_pnt.x;
	
	if(fabs(point_range) < ZERO_EPSILON)
	{
		return false;
	}
	
	// estimate 'u'
	u = x_position - start_pnt.x;
	u /= point_range;
	
	if(!(u < 0 || u > 1))	// taget position inside point range
	{
		// find points adjacent to target position
		for(pnt_iter = pnts.begin();pnt_iter != (pnts.end() - 1);++pnt_iter)
		{
			low_pnt = *(pnt_iter);
			high_pnt = *(pnt_iter+1);
			
			point_range = high_pnt.x - low_pnt.x;
			
			if(fabs(point_range) > ZERO_EPSILON)
			{
				u = x_position - low_pnt.x;
				u /= point_range;
				
				if(!(u < 0 || u > 1))	// local taget position inside point range
				{
					vec = high_pnt - low_pnt;
					vec *= (x_position - low_pnt.x) / vec.x;
					pnt = low_pnt + vec;
					
					*y_value = pnt.y;
					*z_value = pnt.z;
					
					return true;
				}
			}
			
		}
		
		return false;	
	}
	else if(u < 0)						// target position below start point
	{
		high_pnt = pnts[1];
		low_pnt = pnts[0];
		
		vec = high_pnt - low_pnt;
		vec *= (x_position - low_pnt.x) / vec.x;
		pnt = low_pnt + vec;
		
		*y_value = pnt.y;
		*z_value = pnt.z;
		
	}
	else												// target position above end point
	{
		high_pnt = pnts[pnts.size() - 1];
		low_pnt = pnts[pnts.size() - 2];
		
		vec = low_pnt - high_pnt;
		vec *= (x_position - high_pnt.x) / vec.x;
		pnt = high_pnt + vec;
		
		*y_value = pnt.y;
		*z_value = pnt.z;
		
		return true;
	}
	return true;
}

bool Extrapolation::Get_Value_ZX(
	const std::vector<TVector3>			&pnts,
	const double						&y_position,
	double								*z_value,
	double								*x_value)
{
	TVector3							start_pnt;
	TVector3							end_pnt;
	TVector3							low_pnt;
	TVector3							high_pnt;
	TVector3							pnt;
	TVector3							vec;
	std::vector<TVector3>::const_iterator pnt_iter;
	double								point_range;
	double								u;
	
	assert(x_value);
	assert(z_value);
	
	*x_value = 0;
	*z_value = 0;
	
	if(pnts.size() < 2)
	{
		return false;
	}
	
	start_pnt = pnts[0];
	end_pnt = pnts[pnts.size()-1];
	
	point_range = end_pnt.y - start_pnt.y;
	
	if(fabs(point_range) < ZERO_EPSILON)
	{
		return false;
	}
	
	// estimate 'u'
	u = y_position - start_pnt.y;
	u /= point_range;
	
	if(!(u < 0 || u > 1))	// taget position inside point range
	{
		// find points adjacent to target position
		for(pnt_iter = pnts.begin();pnt_iter != (pnts.end() - 1);++pnt_iter)
		{
			low_pnt = *(pnt_iter);
			high_pnt = *(pnt_iter+1);
			
			point_range = high_pnt.y - low_pnt.y;
			
			if(fabs(point_range) > ZERO_EPSILON)
			{
				u = y_position - low_pnt.y;
				u /= point_range;
				
				if(!(u < 0 || u > 1))	// local taget position inside point range
				{
					vec = high_pnt - low_pnt;
					vec *= (y_position - low_pnt.y) / vec.y;
					pnt = low_pnt + vec;
					
					*x_value = pnt.x;
					*z_value = pnt.z;
					
					return true;
				}
			}
		}
		
		return false;	
	}
	else if(u < 0)						// target position below start point
	{
		high_pnt = pnts[1];
		low_pnt = pnts[0];
		
		vec = high_pnt - low_pnt;
		vec *= (y_position - low_pnt.y) / vec.y;
		pnt = low_pnt + vec;
		
		*x_value = pnt.x;
		*z_value = pnt.z;
		
	}
	else												// target position above end point
	{
		high_pnt = pnts[pnts.size() - 1];
		low_pnt = pnts[pnts.size() - 2];
		
		vec = low_pnt - high_pnt;
		vec *= (y_position - high_pnt.y) / vec.y;
		pnt = high_pnt + vec;
		
		*x_value = pnt.x;
		*z_value = pnt.z;
		
		return true;
	}
	return true;
}



