/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2025  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#ifndef VECTOR3HEADERFILE
#define VECTOR3HEADERFILE

#include <limits>
#include <math.h>
#include <assert.h>

class TVector3
{
public:
// CREATORS
	TVector3(void);
	TVector3(const TVector3&);
	TVector3(const double &X,const double &Y,const double &Z);
	~TVector3(void);

// ACCESSORS
	union
	{
										double x;
										double i;
	};
	union
	{
										double y;
										double j;
	};
	union
	{
										double z;
										double k;
	};

	double 			Length(void) const {return sqrt(x*x + y*y + z*z);}
	double 			SquaredLength(void) const {return x*x + y*y + z*z;}

	// indexed value.  0 = x, 1 = y, 2 = z
	double 			operator()(const int Offset) const;

// MANIPULATORS
	double& 		operator()(const int Offset);

	void 			Set(const double &X,const double &Y,const double &Z);
	void 			Zero(void) {x=0;y=0;z=0;};
	TVector3& 		Normal(void);

	TVector3& 		operator=(const TVector3 &lhs);
	TVector3& 		operator*=(const TVector3 &lhs); // Cross product
	TVector3& 		operator+=(const TVector3 &lhs);
	TVector3& 		operator-=(const TVector3 &lhs);

	TVector3& 		operator*=(const double &d);
	TVector3& 		operator/=(const double &d);
	TVector3& 		operator+=(const double &d);
	TVector3& 		operator-=(const double &d);

private:
// NOT IMPLEMENTED
};

// Inline member functions

inline void TVector3::Set(
    const double                  		&X,
    const double                  		&Y,
    const double                  		&Z)
{
	x=X;
	y=Y;
	z=Z;
}

inline TVector3& TVector3::operator=(
    const TVector3                  	&lhs)
{
	if(&lhs != this)
	{
		x = lhs.x;
		y = lhs.y;
		z = lhs.z;
	}

	return *this;
}


inline TVector3& TVector3::operator+=(
    const TVector3                  	&lhs)
{
	if(&lhs != this)
	{
		x += lhs.x;
		y += lhs.y;
		z += lhs.z;
	}

	return *this;
}

inline TVector3& TVector3::operator-=(
    const TVector3                  	&lhs)
{
	if(&lhs != this)
	{
		x -= lhs.x;
		y -= lhs.y;
		z -= lhs.z;
	}

	return *this;
}

inline TVector3& TVector3::operator*=(
    const double                  		&d)
{
	x *= d;
	y *= d;
	z *= d;

	return *this;
}
inline TVector3& TVector3::operator/=(
    const double                  		&d)
{
    assert(d > std::numeric_limits<double>::epsilon());

	x /= d;
	y /= d;
	z /= d;

	return *this;
}
inline TVector3& TVector3::operator+=(
    const double                  		&d)
{
	x += d;
	y += d;
	z += d;

	return *this;
}
inline TVector3& TVector3::operator-=(
    const double                  		&d)
{
	x -= d;
	y -= d;
	z -= d;

	return *this;
}

// Operators
double 			operator^(const TVector3&,const TVector3&);
TVector3 		operator*(const TVector3&,const TVector3&);// Cross Product
TVector3 		operator*(const TVector3&,const double&);
TVector3 		operator*(const double&,const TVector3&);
TVector3 		operator+(const TVector3&,const TVector3&);
TVector3 		operator+(const TVector3&,const double&);
TVector3 		operator+(const double&,const TVector3&);
TVector3 		operator-(const TVector3&,const TVector3&);
TVector3 		operator-(const TVector3&,const double&);
TVector3 		operator-(const double&,const TVector3&);
TVector3 		operator/(const TVector3&,const double&);


#endif
