/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2025  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QListWidget>
#include <QListWidgetItem>
#include <QSpacerItem>
#include <QToolButton>
#include <QVariant>
#include <QList>
#include <algorithm>
#include <assert.h>

#include "entrywidget.h"

// sort entry by length
struct TEntryItem_Sort
{
	bool operator() (Type::TEntryItem a,Type::TEntryItem b)
	{
		return (a.actual < b.actual);
	}
};

TEntryWidget::TEntryWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QGridLayout							*widget_layout;
	QHBoxLayout							*button_hlayout;
	QLabel								*actual_label;
	QLabel								*actual_units_label;
	QLabel								*nominal_label;
	QLabel								*nominal_units_label;
	QLabel								*uc_label;
	QLabel								*uc_units_label;
	QLabel								*offset_units_label;
	QSpacerItem							*button_hspacer;

	this->resize(280,343);
	
	d_entry_font.setFamily("Courier New");
	d_entry_font.setFixedPitch(true);

	widget_layout = new QGridLayout(this);
	widget_layout->setContentsMargins(0,0,0,0);

	d_entry_list = new QListWidget(this);
	d_entry_list->setSelectionMode(QAbstractItemView::ExtendedSelection);
	widget_layout->addWidget(d_entry_list,0,0,1,3);

	button_hlayout = new QHBoxLayout();

	d_add_button = new QToolButton(this);
	d_add_button->setMinimumSize(20,20);
	d_add_button->setMaximumSize(20,20);
	button_hlayout->addWidget(d_add_button);

	d_remove_button = new QToolButton(this);
	d_remove_button->setMinimumSize(20,20);
	d_remove_button->setMaximumSize(20,20);
	button_hlayout->addWidget(d_remove_button);

	button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_hlayout->addItem(button_hspacer);

	d_offset_label = new QLabel(this);
	button_hlayout->addWidget(d_offset_label);

	d_offset_edit = new QLineEdit(this);
	d_offset_edit->setAlignment(Qt::AlignCenter);
	button_hlayout->addWidget(d_offset_edit);

	offset_units_label = new QLabel(this);
	button_hlayout->addWidget(offset_units_label);

	widget_layout->addLayout(button_hlayout,1,0,1,3);

	nominal_label = new QLabel(this);
	widget_layout->addWidget(nominal_label,2,0,1,1);

	d_nominal_edit = new QLineEdit(this);
	d_nominal_edit->setAlignment(Qt::AlignCenter);
	widget_layout->addWidget(d_nominal_edit,2,1,1,1);

	nominal_units_label = new QLabel(this);
	widget_layout->addWidget(nominal_units_label,2,2,1,1);

	actual_label = new QLabel(this);
	widget_layout->addWidget(actual_label,3,0,1,1);

	d_actual_edit = new QLineEdit(this);
	d_actual_edit->setAlignment(Qt::AlignCenter);
	widget_layout->addWidget(d_actual_edit,3,1,1,1);

	actual_units_label = new QLabel(this);
	widget_layout->addWidget(actual_units_label,3,2,1,1);

	uc_label = new QLabel(this);
	widget_layout->addWidget(uc_label,4,0,1,1);

	d_uc_edit = new QLineEdit(this);
	d_uc_edit->setAlignment(Qt::AlignCenter);
	widget_layout->addWidget(d_uc_edit,4,1,1,1);

	uc_units_label = new QLabel(this);
	widget_layout->addWidget(uc_units_label,4,2,1,1);
	
	this->setWindowTitle(QStringLiteral("Entry Widget"));

	d_add_button->setText(QStringLiteral("+"));
	d_remove_button->setText(QStringLiteral("-"));
	nominal_label->setText(QStringLiteral("Nominal:"));
	nominal_units_label->setText(QStringLiteral("mm"));
	actual_label->setText(QStringLiteral("Actual:"));
	actual_units_label->setText(QStringLiteral("mm"));
	uc_label->setText(QStringLiteral("Uc (k=2)"));
	uc_units_label->setText(QStringLiteral("mm"));
	d_offset_label->setText(QStringLiteral("Entry Offset:"));
	d_offset_edit->setText(QStringLiteral("0.0000"));
	offset_units_label->setText(QStringLiteral("mm"));

	// defaults
	d_remove_button->setEnabled(false);
	
	d_actual_edit->setEnabled(false);
	d_nominal_edit->setEnabled(false);
	d_uc_edit->setEnabled(false);

	connect(d_actual_edit,&QLineEdit::editingFinished,this,&TEntryWidget::Entry_Changed);
	connect(d_nominal_edit,&QLineEdit::editingFinished,this,&TEntryWidget::Entry_Changed);
	connect(d_uc_edit,&QLineEdit::editingFinished,this,&TEntryWidget::Entry_Changed);
	connect(d_offset_edit,&QLineEdit::editingFinished,this,&TEntryWidget::Data_Changed);

	connect(d_entry_list,&QListWidget::itemSelectionChanged,this,&TEntryWidget::Selection_Changed);
	
	connect(d_add_button,&QToolButton::clicked,this,&TEntryWidget::Add_Entry);
	connect(d_remove_button,&QToolButton::clicked,this,&TEntryWidget::Remove_Entry);
}

TEntryWidget::~TEntryWidget(void)
{
}

std::vector<Type::TEntryItem> TEntryWidget::Entry_Items(void) const
{
	std::vector<Type::TEntryItem>		entry_items;
	Type::TEntryItem					entry_item;
	TEntryItem_Sort						entry_item_sort;
	QListWidgetItem						*item;
	QVariant							variant;
	double								offset;
	int									count;
	int									cntr;
	
	d_entry_list->setFocus(Qt::ActiveWindowFocusReason);	// forces completion of line edit if needed.
	
	count = d_entry_list->count();
	offset = d_offset_edit->text().toDouble();
	
	for(cntr = 0;cntr < count;++cntr)
	{
		item = d_entry_list->item(cntr);
		
		variant = item->data(Qt::UserRole);
		entry_item = qvariant_cast<Type::TEntryItem>(variant);
		
		entry_item.nominal += offset;
		entry_item.actual += offset;

		entry_items.push_back(entry_item);
	}
	
	// sort data from min to max
	std::sort(entry_items.begin(), entry_items.end(), entry_item_sort);
	
	return entry_items;
}

double TEntryWidget::Entry_Offset(void) const
{
	return d_offset_edit->text().toDouble();
}

void TEntryWidget::Clear_Entry_Items(void)
{
	d_entry_list->clear();
}

void TEntryWidget::Add_Entry_Item(
	const Type::TEntryItem				&entry_item)
{
	QListWidgetItem						*item;
	QVariant							variant;
	
	item = new QListWidgetItem(d_entry_list);
	item->setFont(d_entry_font);
	
	item->setText(QString("%1, %2, %3").arg(entry_item.nominal,10,'f',4).arg(entry_item.actual,10,'f',4).arg(entry_item.uc,8,'f',4));
	
	variant.setValue(entry_item);
	item->setData(Qt::UserRole,variant);
}

void TEntryWidget::Add_Entry_Items(
	const std::vector<Type::TEntryItem> &items)
{
	std::vector<Type::TEntryItem>::const_iterator iter;
	
	for(iter = items.begin();iter != items.end();++iter)
	{
		this->Add_Entry_Item(*iter);
	}
}

void TEntryWidget::Enable_Entry_Offset(
	const bool 							state)
{
	d_offset_label->setEnabled(state);
	d_offset_edit->setEnabled(state);
}

void TEntryWidget::Set_Entry_Offset(
	const double 						&offset)
{
	d_offset_edit->setText(QString::number(offset,'f',4));
	
	emit Data_Changed();
}

void TEntryWidget::Selection_Changed(void)
{
	QList<QListWidgetItem*>				sel_items;
	QVariant							variant;
	Type::TEntryItem					entry_item;

	sel_items = d_entry_list->selectedItems();
	
	d_remove_button->setEnabled(sel_items.size() > 0);

	if(sel_items.size() == 1)
	{
		variant = sel_items[0]->data(Qt::UserRole);
		entry_item = qvariant_cast<Type::TEntryItem>(variant);
		
		d_actual_edit->setText(QString::number(entry_item.actual,'f',4));
		d_nominal_edit->setText(QString::number(entry_item.nominal,'f',4));
		d_uc_edit->setText(QString::number(entry_item.uc,'f',4));
		
		d_actual_edit->setEnabled(true);
		d_nominal_edit->setEnabled(true);
		d_uc_edit->setEnabled(true);
	}
	else
	{
		d_actual_edit->setText(QString());
		d_nominal_edit->setText(QString());
		d_uc_edit->setText(QString());
		
		d_actual_edit->setEnabled(false);
		d_nominal_edit->setEnabled(false);
		d_uc_edit->setEnabled(false);
	}
}

void TEntryWidget::Entry_Changed(void)
{
	QList<QListWidgetItem*>				sel_items;
	QVariant							variant;
	Type::TEntryItem					entry_item;
	QObject								*object;
	
	sel_items = d_entry_list->selectedItems();

	if(sel_items.size())
	{
		entry_item.nominal = d_nominal_edit->text().toDouble();
		entry_item.actual = d_actual_edit->text().toDouble();
		entry_item.uc = d_uc_edit->text().toDouble();
		
		variant.setValue(entry_item);
		
		sel_items[0]->setData(Qt::UserRole,variant);
		sel_items[0]->setText(QString("%1, %2, %3").arg(entry_item.nominal,10,'f',4).arg(entry_item.actual,10,'f',4).arg(entry_item.uc,8,'f',4));
		
		emit Data_Changed();
		
		// move focus to next input item
		object = this->sender();
		if(object == d_offset_edit)
		{
			d_nominal_edit->setFocus(Qt::OtherFocusReason);
		}
		else if(object == d_nominal_edit)
		{
			d_actual_edit->setFocus(Qt::OtherFocusReason);
		}
		else if(object == d_actual_edit)
		{
			d_uc_edit->setFocus(Qt::OtherFocusReason);
		}
		else if(object == d_uc_edit)
		{
			d_entry_list->setFocus(Qt::OtherFocusReason);
		}
	}
}

void TEntryWidget::Add_Entry(void)
{
	QListWidgetItem						*item;
	QVariant							variant;
	Type::TEntryItem					entry_item;
	double								dval;

	dval = static_cast<double>(d_entry_list->count() + 1);
	
	item = new QListWidgetItem(d_entry_list);
	item->setFont(d_entry_font);
	
	entry_item.nominal = dval * 100;
	entry_item.actual = dval * 100;
	entry_item.uc = 0.001 + dval / 1000;
	
	item->setText(QString("%1, %2, %3").arg(entry_item.nominal,10,'f',4).arg(entry_item.actual,10,'f',4).arg(entry_item.uc,8,'f',4));
	
	variant.setValue(entry_item);
	item->setData(Qt::UserRole,variant);
	
	emit Data_Changed();
}

void TEntryWidget::Remove_Entry(void)
{
	QList<QListWidgetItem*>				sel_items;
	int									row;
	int									cntr;
	
	sel_items = d_entry_list->selectedItems();
	
	for(cntr = 0;cntr != sel_items.size();++cntr)
	{
		row = d_entry_list->row(sel_items[cntr]);
		
		d_entry_list->takeItem(row);
		delete sel_items[cntr];
	}
	
	emit Data_Changed();
}

