/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2025  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QFrame>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QPushButton>
#include <QSpacerItem>
#include <QWidget>
#include <QToolBar>
#include <QMenuBar>
#include <QMenu>
#include <QCloseEvent>
#include <QSettings>
#include <QDragEnterEvent>
#include <QDropEvent>
#include <QMimeData>
#include <QUrl>
#include <QDir>
#include <QColor>
#include <QFile>
#include <QFileDialog>
#include <QScreen>
#include <QFont>
#include <QPoint>
#include <QSize>
#include <QRect>
#include <cmath>
#include <assert.h>

#include "../../core/graphwidget.h"
#include "../../core/vector2.h"
#include "../../core/interpolation.h"
#include "../../core/messagebox.h"
#include "../../../core/xmlfile.h"

#include "entrywidget.h"
#include "optionsdialog.h"

#include "lengthcompare.h"

// sort pnt by position
struct Position_Sort
{
	bool operator() (double a,double b)
	{
		return (a < b);
	}
};

TLengthCompare::TLengthCompare(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QMainWindow(const_cast<QWidget*>(parent),flags)
{
	QWidget								*central_widget;
	QToolBar							*toolbar;
	QMenuBar							*main_menu;
	QMenu								*file_menu;
	QMenu								*edit_menu;
	QMenu								*help_menu;
	QFrame								*separator1_hline;
	QFrame								*separator2_hline;
	QFrame								*separator_vline;
	QGridLayout							*widget_layout;
	QHBoxLayout							*copyright_hlayout;
	QLabel								*copyright_label;
	QLabel								*logo_label;
	QSpacerItem							*copyright_hspacer;
	QPoint								position;
	QSize								size;
	QRect								desktop_rect;
	QRect								window_rect;
	QFont								copyright_font;
	QDir								data_path;

	this->resize(901,801);
	this->setAcceptDrops(true);
	
	copyright_font.setFamily(QStringLiteral("Verdana"));
	copyright_font.setPointSize(8);
	copyright_font.setItalic(true);

#ifdef Q_OS_WIN
	d_file_path = qApp->applicationDirPath();
#else
	d_file_path = QDir::homePath();
#endif
	
	data_path.setPath(d_file_path);
	
	if(!data_path.exists(QStringLiteral(".lengthcompare")))
	{
		data_path.mkdir(QStringLiteral(".lengthcompare"));
	}
	
	data_path.cd(QStringLiteral(".lengthcompare"));
	
	d_settings = new QSettings(data_path.absolutePath() + QStringLiteral("/lengthcompare_settings.ini"),QSettings::IniFormat,this);

	position = d_settings->value("Mainwindow_Position", QPoint(30,30)).toPoint();
	size = d_settings->value("Mainwindow_Size", QSize(901,801)).toSize();
	
	desktop_rect = QGuiApplication::primaryScreen()->availableGeometry();
	
	window_rect.moveTopLeft(position);
	window_rect.setSize(size);
	
	window_rect = window_rect & desktop_rect;
	
	if(window_rect.isValid() && window_rect.width() > 500 && window_rect.height() > 500)
	{
		this->resize(window_rect.size());
		this->move(window_rect.topLeft());
	}
	else
	{
		this->resize(901,801);
		this->move(QPoint(30,30));
	}

	d_file_path= d_settings->value(QStringLiteral("File_Path"), d_file_path).toString();
	d_options.remove_deviations = d_settings->value(QStringLiteral("Remove_Deviations"), true).toBool();
	
	central_widget = new QWidget(this);
	this->setCentralWidget(central_widget);
	
	d_action_open = new QAction(this);
	d_action_open->setToolTip(QStringLiteral("Open an existing length compare file."));
	d_action_open->setShortcut(QStringLiteral("Ctrl+O"));
	d_action_open->setIcon(QIcon(QStringLiteral(":/menu/file_open_x32.png")));
	
	d_action_save = new QAction(this);
	d_action_save->setToolTip(QStringLiteral("File_Save the current length compare file."));
	d_action_save->setShortcut(QStringLiteral("Ctrl+S"));
	d_action_save->setIcon(QIcon(QStringLiteral(":/menu/file_save_x32.png")));
	
	d_action_export = new QAction(this);
	d_action_export->setToolTip(QStringLiteral("File_Export data from the length compare file."));
	d_action_export->setShortcut(QStringLiteral("Ctrl+E"));
	d_action_export->setIcon(QIcon(QStringLiteral(":/menu/file_export_x32.png")));
	
	d_action_about = new QAction(this);
	d_action_about->setToolTip(QStringLiteral("About the Length Compare utility."));
	d_action_about->setIcon(QIcon(QStringLiteral(":/menu/help_about_x32.png")));
	
	d_action_options = new QAction(this);
	d_action_options->setToolTip(QStringLiteral("Options for the Length Compare utility."));
	d_action_options->setIcon(QIcon(QStringLiteral(":/menu/edit_options_x32.png")));

	d_action_quit = new QAction(this);
	d_action_quit->setToolTip(QStringLiteral("Close the length compare utility."));
	d_action_quit->setShortcut(QStringLiteral("Ctrl+Q"));

	widget_layout = new QGridLayout(central_widget);

	d_entry_a_widget = new TEntryWidget(central_widget);
	d_entry_a_widget->Enable_Entry_Offset(false);
	widget_layout->addWidget(d_entry_a_widget,0,0,1,1);
	
	separator_vline = new QFrame(central_widget);
	separator_vline->setFrameShape(QFrame::VLine);
	separator_vline->setFrameShadow(QFrame::Sunken);
	widget_layout->addWidget(separator_vline,0,1,1,1);

	d_entry_b_widget = new TEntryWidget(central_widget);
	d_entry_b_widget->Enable_Entry_Offset(true);
	widget_layout->addWidget(d_entry_b_widget,0,2,1,1);

	separator1_hline = new QFrame(central_widget);
	separator1_hline->setFrameShape(QFrame::HLine);
	separator1_hline->setFrameShadow(QFrame::Sunken);
	widget_layout->addWidget(separator1_hline,1,0,1,4);

	d_graph_data_widget = new TGraphWidget(central_widget);
	d_graph_data_widget->setMinimumSize(QSize(100, 200));
	widget_layout->addWidget(d_graph_data_widget,2,0,1,3);
	
	d_graph_en_widget = new TGraphWidget(central_widget);
	d_graph_en_widget->setMinimumSize(QSize(100, 200));
	widget_layout->addWidget(d_graph_en_widget,3,0,1,3);

	separator2_hline = new QFrame(central_widget);
	separator2_hline->setFrameShape(QFrame::HLine);
	separator2_hline->setFrameShadow(QFrame::Sunken);
	widget_layout->addWidget(separator2_hline,4,0,1,3);
	
	copyright_hlayout = new QHBoxLayout();
	
	copyright_label = new QLabel(central_widget);
	copyright_label->setFont(copyright_font);
	copyright_label->setAlignment(Qt::AlignBottom|Qt::AlignLeading|Qt::AlignLeft);
	copyright_hlayout->addWidget(copyright_label);

	copyright_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	copyright_hlayout->addItem(copyright_hspacer);
	
	logo_label = new QLabel(central_widget);
	logo_label->setMinimumSize(100,40);
	logo_label->setMaximumSize(100,40);
	copyright_hlayout->addWidget(logo_label);

	widget_layout->addLayout(copyright_hlayout,5,0,1,3);

	main_menu = new QMenuBar(this);
	this->setMenuBar(main_menu);
	
	file_menu = new QMenu(main_menu);
	main_menu->addMenu(file_menu);
	file_menu->addAction(d_action_open);
	file_menu->addSeparator();
	file_menu->addAction(d_action_save);
	file_menu->addAction(d_action_export);
	file_menu->addSeparator();
	file_menu->addAction(d_action_quit);
	
	edit_menu = new QMenu(main_menu);
	main_menu->addMenu(edit_menu);
	edit_menu->addAction(d_action_options);

	help_menu = new QMenu(main_menu);
	main_menu->addMenu(help_menu);
	help_menu->addAction(d_action_about);

	toolbar = new QToolBar(this);
	toolbar->setIconSize(QSize(32,32));
	this->addToolBar(Qt::TopToolBarArea,toolbar);
	
	toolbar->addAction(d_action_open);
	toolbar->addSeparator();
	toolbar->addAction(d_action_save);
	toolbar->addAction(d_action_export);
	toolbar->addSeparator();
	toolbar->addAction(d_action_options);
	toolbar->addSeparator();
	toolbar->addAction(d_action_about);
	
	// defaults
	d_options_dialog = 0;
	
	logo_label->setPixmap(QPixmap(QStringLiteral(":/images/logo.png")));

	this->setWindowTitle(QStringLiteral("Length Compare"));
	
	d_action_open->setText(QStringLiteral("Open"));
	d_action_save->setText(QStringLiteral("File_Save"));
	d_action_export->setText(QStringLiteral("File_Export"));
	d_action_options->setText(QStringLiteral("Options"));
	d_action_about->setText(QStringLiteral("About"));
	d_action_quit->setText(QStringLiteral("Quit"));

	copyright_label->setText(QStringLiteral("Copyright (C) 2025 Select Calibration Incorporated.  www.selectcalibration.ca"));
	file_menu->setTitle(QStringLiteral("&File"));
	edit_menu->setTitle(QStringLiteral("&Edit"));
	help_menu->setTitle(QStringLiteral("&Help"));

	d_graph_data_widget->Title(QStringLiteral("Length Comparison"));
	d_graph_data_widget->VerticalLegend(QStringLiteral("Deviation"));
	d_graph_data_widget->HorizontalLegend(QStringLiteral("Length"));
	
	d_graph_en_widget->Title(QStringLiteral("En Score"));
	d_graph_en_widget->VerticalLegend(QStringLiteral("En Value"));
	d_graph_en_widget->HorizontalLegend(QStringLiteral("Length"));
	
	d_action_save->setEnabled(false);
	d_action_export->setEnabled(false);

	d_msg_box = new TMessageBox(this);
	
	connect(d_action_open,&QAction::triggered,this,&TLengthCompare::File_Load);
	connect(d_action_save,&QAction::triggered,this,&TLengthCompare::File_Save);
	connect(d_action_export,&QAction::triggered,this,&TLengthCompare::File_Export);
	connect(d_action_options,&QAction::triggered,this,&TLengthCompare::Edit_Options);
	connect(d_action_about,&QAction::triggered,this,&TLengthCompare::Help_About);
	connect(d_action_quit,&QAction::triggered,this,&TLengthCompare::close);

	connect(d_entry_a_widget,&TEntryWidget::Data_Changed,this,&TLengthCompare::Update);
	connect(d_entry_b_widget,&TEntryWidget::Data_Changed,this,&TLengthCompare::Update);
	
	this->Update();
	d_data_changed = false;
}

TLengthCompare::~TLengthCompare(void)
{
}

bool TLengthCompare::Save_Data(
	const QString						&file_name) const
{
	QString								text;
	std::vector<Type::TEntryItem>		input_a_data;
	std::vector<Type::TEntryItem>		input_b_data;
	std::vector<Type::TEntryItem>::const_iterator entry_iter;
	TXmlFile							xml_file;
	double								entry_a_offset;
	double								entry_b_offset;
	QDomElement							e0;
	QDomElement							e1;
	QDomElement							e2;

	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Reset(QStringLiteral("Length_Compare_Data"),2);
		
		entry_a_offset = d_entry_a_widget->Entry_Offset();
		entry_b_offset = d_entry_b_widget->Entry_Offset();
		
		entry_b_offset -= entry_a_offset;
		entry_a_offset = 0.0;

		e1 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Data_Set_A_Offset"));
		xml_file.Write_Text_Node(&e1,QString("%1").arg(entry_a_offset,0,'f',5));

		e1 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Data_Set_B_Offset"));
		xml_file.Write_Text_Node(&e1,QString("%1").arg(entry_b_offset,0,'f',5));
		
		input_a_data = d_entry_a_widget->Entry_Items();
		
		// data set a
		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Data_Set_A"));
		
		for(entry_iter = input_a_data.begin();entry_iter != input_a_data.end();++entry_iter)
		{
			e1 = xml_file.Create_Node(&e0,QStringLiteral("Entry"));
			
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Nominal"));
			xml_file.Write_Text_Node(&e2,QString("%1").arg((*entry_iter).nominal - entry_a_offset,0,'f',5));
			
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Actual"));
			xml_file.Write_Text_Node(&e2,QString("%1").arg((*entry_iter).actual - entry_a_offset,0,'f',5));
			
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Uc"));
			xml_file.Write_Text_Node(&e2,QString("%1").arg((*entry_iter).uc,0,'f',5));
			
		}
		
		input_b_data = d_entry_b_widget->Entry_Items();
		
		// data set b
		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Data_Set_B"));
		
		for(entry_iter = input_b_data.begin();entry_iter != input_b_data.end();++entry_iter)
		{
			e1 = xml_file.Create_Node(&e0,QStringLiteral("Entry"));
			
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Nominal"));
			xml_file.Write_Text_Node(&e2,QString("%1").arg((*entry_iter).nominal - entry_b_offset,0,'f',5));
			
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Actual"));
			xml_file.Write_Text_Node(&e2,QString("%1").arg((*entry_iter).actual - entry_b_offset,0,'f',5));
			
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Uc"));
			xml_file.Write_Text_Node(&e2,QString("%1").arg((*entry_iter).uc,0,'f',5));
		}
		
		xml_file.Write_File();
	}
	catch(TXmlFileException exception)
	{
		d_last_error = QString("ERR  Save_Data: %1").arg(exception.ErrorText()).toLatin1();
		return false;
	}

	return true;
}

bool TLengthCompare::Export_Data(
	const QString						&file_name)
{
	QFile								file;
	std::vector<TLengthCompare::TExportData>::const_iterator iter;
	TLengthCompare::TExportData			export_data;
	QString								text;
	double								en_mean;
	double								en_max;
	int									en_count;

	file.setFileName(file_name);
	
	if(!file.open(QIODevice::WriteOnly | QIODevice::Text))
	{
		d_last_error = QStringLiteral("ERR  Cannot open file for writing.");
		return false;
	}
	
	en_count = 0;
	en_mean = 0.0;
	en_max = 0.0;
	
	file.write("Comparison Data\n\n");
	
	//          012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789
	file.write("Position       Dev. A     Dev. B      Uc A      Uc A Adj    Uc B        En\n");
	file.write("----------------------------------------------------------------------------------\n");
	
	for(iter = d_export_data.begin();iter != d_export_data.end();++iter)
	{
		export_data = (*iter);
		
		text = QString("%1 ").arg(export_data.position,10,'f',4);
		
		if(export_data.dev_a_valid)
		{
			text += QString(" %1").arg(export_data.dev_a,10,'f',4);
		}
		else
		{
			text += QStringLiteral("     ------");
		}
		
		if(export_data.dev_b_valid)
		{
			text += QString(" %1").arg(export_data.dev_b,10,'f',4);
		}
		else
		{
			text += QStringLiteral("     ------");
		}

		if(export_data.uc_a_valid)
		{
			text += QString(" %1").arg(export_data.uc_a_raw,10,'f',4);
			text += QString(" %1").arg(export_data.uc_a,10,'f',4);
		}
		else
		{
			text += QStringLiteral("     ------");
			text += QStringLiteral("     ------");
		}

		if(export_data.uc_b_valid)
		{
			text += QString(" %1").arg(export_data.uc_b,10,'f',4);
		}
		else
		{
			text += QStringLiteral("     ------");
		}
		
		if(export_data.uc_a_valid && export_data.uc_b_valid)
		{
			text += QString(" %1").arg(export_data.uc_en,10,'f',4);
			
			++en_count;
			en_mean += export_data.uc_en;
			
			if(export_data.uc_en > en_max) en_max = export_data.uc_en;
		}
		else
		{
			text += QStringLiteral("     ------");
		}
		
		text += QString('\n');
		file.write(text.toLatin1());

	}
	
	if(en_count)
	{
		en_mean /= static_cast<double>(en_count);
		
		text = QString("\n\nData B Pos Offset: %1\n").arg(d_entry_b_widget->Entry_Offset(),9,'f',4);
		file.write(text.toLatin1());

		text = QString("Data B Dev Offset: %1\n").arg(d_data_b_deviation_offset,9,'f',4);
		file.write(text.toLatin1());

		text = QString("En Avg:            %1\n").arg(en_mean,9,'f',4);
		file.write(text.toLatin1());
		
		text = QString("En Max:            %1\n").arg(en_max,9,'f',4);
		file.write(text.toLatin1());
	}
	
	file.close();
	
	return true;
}

bool TLengthCompare::Load_Data(
	const QString						&file_name)
{
	TXmlFile							xml_file;
	int									version;
	
	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Open("Length_Compare_Data",&version);
		
		switch(version)
		{
			case 1:
				this->Load_Data_V1(&xml_file);
				break;
				
			case 2:
				this->Load_Data_V2(&xml_file);
				break;

			default:
				throw TXmlFileException(QStringLiteral("ERR:  File version not recognized"));
				break;
		}
	}
	catch(TXmlFileException exception)
	{
		d_last_error = QString("ERR  Load_Data: %1").arg(exception.ErrorText()).toLatin1();
		return false;
	}
	
	return true;
}

void TLengthCompare::File_Load(void)
{
	QString								file_name;
	QString								file_name_filter(QStringLiteral("Length Comparison Files (*.lencomp)"));
	int									index;
	
	if(d_data_changed)
	{
		d_msg_box->setText("Data Not Saved");
		d_msg_box->setInformativeText("Do you want to continue without saving?");
		d_msg_box->setDetailedText(QString());
		d_msg_box->setStandardButtons(QMessageBox::Yes | QMessageBox::No);
		d_msg_box->setDefaultButton(QMessageBox::No);
		d_msg_box->setIcon(QMessageBox::Warning);
		
		if(QMessageBox::No == d_msg_box->exec())
		{
			return;
		}
	}
	
	file_name = QFileDialog::getOpenFileName(this,
											 QStringLiteral("Select length comparison file to open"),
											 d_file_path,
											 file_name_filter);
	
	if(file_name.length())
	{
		d_file_path = file_name;
		index = d_file_path.lastIndexOf('/');
		if(!(index < 0))
		{
			d_file_path.truncate(index);
		}

		if(!this->Load_Data(file_name))
		{
			d_msg_box->setText("Error Loading File");
			d_msg_box->setInformativeText("The selected length comparison file cannot be loaded.");
			d_msg_box->setDetailedText(d_last_error);
			d_msg_box->setStandardButtons(QMessageBox::Ok);
			d_msg_box->setDefaultButton(QMessageBox::Ok);
			d_msg_box->setIcon(QMessageBox::Critical);
			
			d_msg_box->exec();
		}
		
		this->Update();
		d_data_changed = false;
	}
}

void TLengthCompare::File_Save(void)
{
	QString								file_name;
	QString								file_name_filter(QStringLiteral("Length Comparison Files (*.lencomp)"));
	int									index;
	
	file_name = QFileDialog::getSaveFileName(this,
											 QStringLiteral("Enter name of length comparison file to create"),
											 d_file_path,
											 file_name_filter);
	
	if(file_name.length())
	{
		if(!file_name.endsWith(QStringLiteral(".lencomp"),Qt::CaseInsensitive))
		{
			file_name += QStringLiteral(".lencomp");
		}
		
		d_file_path = file_name;
		index = d_file_path.lastIndexOf('/');
		if(!(index < 0))
		{
			d_file_path.truncate(index);
		}
		
		if(!this->Save_Data(file_name))
		{
			d_msg_box->setText("Error Saving File");
			d_msg_box->setInformativeText("The length comparison file cannot be written.");
			d_msg_box->setDetailedText(d_last_error);
			d_msg_box->setStandardButtons(QMessageBox::Ok);
			d_msg_box->setDefaultButton(QMessageBox::Ok);
			d_msg_box->setIcon(QMessageBox::Critical);
			
			d_msg_box->exec();
		}
		
		d_data_changed = false;
	}
}

void TLengthCompare::File_Export(void)
{
	QString								file_name;
	QString								file_name_filter(QStringLiteral("Text files (*.txt)"));
	
	
	file_name = QFileDialog::getSaveFileName(this,
											 QStringLiteral("Enter name of text file to create"),
											 d_file_path,
											 file_name_filter);
	
	if(file_name.length())
	{
		if(!this->Export_Data(file_name))
		{
			d_msg_box->setText("Error Exporting File");
			d_msg_box->setInformativeText("The length comparison file cannot be exported.");
			d_msg_box->setDetailedText(d_last_error);
			d_msg_box->setStandardButtons(QMessageBox::Ok);
			d_msg_box->setDefaultButton(QMessageBox::Ok);
			d_msg_box->setIcon(QMessageBox::Critical);
			
			d_msg_box->exec();
		}
	}
}

void TLengthCompare::Edit_Options(void)
{
	if(!d_options_dialog)
	{
		d_options_dialog = new TOptionsDialog(this);
	}
	
	d_options_dialog->Reset(d_options);
	
	if(d_options_dialog->exec() == QDialog::Accepted)
	{
		d_options = d_options_dialog->Options();
		this->Update();
	}
}

void TLengthCompare::Help_About(void)
{
	d_msg_box->setText(QStringLiteral("Length Compare"));
	d_msg_box->setInformativeText(QStringLiteral("Length Compare - Version 2.0"));
	d_msg_box->setDetailedText(QString());
	d_msg_box->setStandardButtons(QMessageBox::Ok);
	d_msg_box->setDefaultButton(QMessageBox::Ok);
	d_msg_box->setIcon(QMessageBox::Information);
	
	d_msg_box->exec();
}

void TLengthCompare::Update(void)
{
	TLengthCompare::TExportData			export_data;
	std::vector<Type::TEntryItem>		input_a_data;
	std::vector<Type::TEntryItem>		input_b_data;
	std::vector<TVector2>				dev_a_data;
	std::vector<TVector2>				dev_b_data;
	std::vector<TVector2>				uc_a_data;
	std::vector<TVector2>				uc_b_data;
	std::vector<Type::TEntryItem>::const_iterator entry_iter;
	std::vector<double>::const_iterator	pos_iter;
	TVector2							pnt;
	TVector2							p1,p2,p3,p4;
	TVector2							p5,p6,p7,p8;
	Position_Sort						position_sort;
	double								min_a_position(0);
	double								max_a_position(0);
	double								min_b_position(0);
	double								max_b_position(0);
	double								min_en_position(0);
	double								max_en_position(0);
	double								min_position(0);
	double								max_position(0);
	double								dev_a,dev_b;
	double								dev_b_position_offset;
	double								uc_a_offset;
	double								uc_a,uc_b;
	double								difference;
	double								uc_ab;
	double								en;
	double								uc_radius;
	double								dval;
	int									set_a_id;
	int									set_b_id;
	int									set_en_id;
	int									set_id;
	bool								init;
	static const double					MAX_EXTRAPOLATION_LENGTH(10.0);
	static const double					ZERO_EPSILON(0.00001);
	static const double					UC_RADIUS_MULTIPLIER(0.005);
			
	input_a_data = d_entry_a_widget->Entry_Items();
	input_b_data = d_entry_b_widget->Entry_Items();
	dev_b_position_offset = d_entry_b_widget->Entry_Offset();
	
	d_graph_data_widget->DeleteDataSets();
	d_graph_en_widget->DeleteDataSets();
	d_sample_positions.clear();
	d_export_data.clear();
	
	// must have at least two entries
	if(input_a_data.size() < 2 ||
	   input_b_data.size() < 2)
	{
		d_action_save->setEnabled(false);
		d_action_export->setEnabled(false);

		return;
	}
	
	d_data_changed = true;

	d_action_save->setEnabled(true);
	d_action_export->setEnabled(true);

	// data is sorted by length from entry widgets
	min_a_position = input_a_data[0].actual;
	max_a_position = input_a_data[input_a_data.size()-1].actual;
	
	min_b_position = input_b_data[0].actual;
	max_b_position = input_b_data[input_b_data.size()-1].actual;
	
	min_position = min_a_position;
	if(min_b_position < min_position) min_position = min_b_position;
	
	max_position = max_a_position;
	if(max_b_position > max_position) max_position = max_b_position;
	
	for(entry_iter = input_a_data.begin();entry_iter != input_a_data.end();++entry_iter)
	{
		pnt.x = (*entry_iter).nominal;
		
		pnt.y = (*entry_iter).actual - (*entry_iter).nominal;
		dev_a_data.push_back(pnt);
		
		pnt.y = (*entry_iter).uc;
		uc_a_data.push_back(pnt);
		
		this->Add_Graph_Position(pnt.x);
	}
	
	for(entry_iter = input_b_data.begin();entry_iter != input_b_data.end();++entry_iter)
	{
		pnt.x = (*entry_iter).nominal;
		
		pnt.y = (*entry_iter).actual - (*entry_iter).nominal;
		dev_b_data.push_back(pnt);
		
		pnt.y = (*entry_iter).uc;
		uc_b_data.push_back(pnt);
		
		this->Add_Graph_Position(pnt.x);

	}
	
	if(d_options.remove_deviations)
	{
		// get deviation from data a at dev_b_position_offset (entry offset)
		if(!Extrapolation::Get_Value(dev_a_data,dev_b_position_offset,&d_data_b_deviation_offset))
		{
			d_data_b_deviation_offset = 0.0;
		}

		if(!Extrapolation::Get_Value(uc_a_data,dev_b_position_offset,&uc_a_offset))
		{
			uc_a_offset = 0.0;
		}
		
		Extrapolation::Get_Value(uc_a_data,0.0,&dval);
		
		uc_a_offset -= dval;
	}
	else
	{
		uc_a_offset = 0.0;
		d_data_b_deviation_offset = 0.0;
	}
	
	std::sort(d_sample_positions.begin(), d_sample_positions.end(), position_sort);
	
	// should never fail but just in case
	assert(dev_a_data.size() == uc_a_data.size());
	assert(dev_b_data.size() == uc_b_data.size());

	// calculate en score for all a entries if length is common to both
	set_a_id = d_graph_data_widget->CreateDataSet(QColor(128,40,40));
	set_b_id = d_graph_data_widget->CreateDataSet(QColor(20,96,20));
	set_en_id = d_graph_en_widget->CreateDataSet(QColor(40,40,128));
	
	uc_radius = (max_position - min_position) * UC_RADIUS_MULTIPLIER;

	init = true;
	for(pos_iter = d_sample_positions.begin();pos_iter != d_sample_positions.end();++pos_iter)
	{
		export_data.dev_a_valid = false;
		export_data.dev_b_valid = false;
		export_data.uc_a_valid = false;
		export_data.uc_b_valid = false;
		export_data.position = (*pos_iter);
		
		pnt.x = (*pos_iter);
		
		Extrapolation::Get_Value(dev_a_data,pnt.x,&dev_a);
		Extrapolation::Get_Value(dev_b_data,pnt.x,&dev_b);
		
		dev_b += d_data_b_deviation_offset;
		
		export_data.dev_a = dev_a;
		export_data.dev_b = dev_b;
		
		if(pnt.x > (min_a_position - MAX_EXTRAPOLATION_LENGTH) &&
		   pnt.x < (max_a_position + MAX_EXTRAPOLATION_LENGTH))
		{
			export_data.dev_a_valid = true;
		}
		
		if(pnt.x > (min_b_position - MAX_EXTRAPOLATION_LENGTH) &&
		   pnt.x < (max_b_position + MAX_EXTRAPOLATION_LENGTH))
		{
			export_data.dev_b_valid = true;
		}

		Extrapolation::Get_Value(uc_a_data,pnt.x,&uc_a);
		Extrapolation::Get_Value(uc_b_data,pnt.x,&uc_b);
		
		export_data.uc_a_raw = uc_a;

		// only calculate score is data point is inside range of both data sets
		if(export_data.dev_a_valid && export_data.dev_b_valid)
		{
			uc_a -= uc_a_offset;

			difference = fabs(dev_a - dev_b);
			
			uc_ab = sqrt(uc_a * uc_a + uc_b * uc_b);

			if(uc_ab > ZERO_EPSILON)
			{
				en = difference / uc_ab;
			}
			else
			{
				en = 1000.0;
			}
						
			d_graph_en_widget->AddPoint(set_en_id,TVector2(pnt.x,en));
			
			if(init)
			{
				min_en_position = pnt.x;
				max_en_position = pnt.x;
				
				init = false;
			}
			else
			{
				if(pnt.x < min_en_position) min_en_position = pnt.x;
				if(pnt.x > max_en_position) max_en_position = pnt.x;
			}
		}
		
		if(export_data.dev_a_valid)
		{
			export_data.uc_a = uc_a;
			export_data.uc_a_valid = true;

			pnt.y = dev_a;
			d_graph_data_widget->AddPoint(set_a_id,pnt);
			
			p1 = pnt;
			p1.x += (uc_radius/2.0);
			p4 = p1;

			p1.y += uc_a;
			p4.y -= uc_a;
			
			p2 = p1;
			p2.x -= uc_radius;
			
			p3 = p4;
			p3.x = p2.x;
			
			p5 = p1;
			p6 = p2;
			p7 = p3;
			p8 = p4;
			
			p5.y += uc_a_offset;
			p6.y += uc_a_offset;
			p7.y -= uc_a_offset;
			p8.y -= uc_a_offset;
			
			if(pnt.x > (min_b_position - MAX_EXTRAPOLATION_LENGTH) &&
			   pnt.x < (max_b_position + MAX_EXTRAPOLATION_LENGTH))
			{
				set_id = d_graph_data_widget->CreateDataSet(QColor(212,212,212));
				d_graph_data_widget->AddPoint(set_id,p5);
				d_graph_data_widget->AddPoint(set_id,p6);
				d_graph_data_widget->AddPoint(set_id,p7);
				d_graph_data_widget->AddPoint(set_id,p8);
			}
			
			set_id = d_graph_data_widget->CreateDataSet(QColor(212,96,96));
			d_graph_data_widget->AddPoint(set_id,p1);
			d_graph_data_widget->AddPoint(set_id,p2);
			d_graph_data_widget->AddPoint(set_id,p3);
			d_graph_data_widget->AddPoint(set_id,p4);
		}

		if(export_data.dev_b_valid)
		{
			export_data.uc_b = uc_b;
			export_data.uc_b_valid = true;
			
			pnt.y = dev_b;
			d_graph_data_widget->AddPoint(set_b_id,pnt);
			
			p1 = pnt;
			p1.x -= (uc_radius/2.0);
			p4 = p1;
			
			p1.y += uc_b;
			p4.y -= uc_b;
			
			p2 = p1;
			p2.x += uc_radius;
			
			p3 = p4;
			p3.x = p2.x;
			
			set_id = d_graph_data_widget->CreateDataSet(QColor(20,96,20));
			d_graph_data_widget->AddPoint(set_id,p1);
			d_graph_data_widget->AddPoint(set_id,p2);
			d_graph_data_widget->AddPoint(set_id,p3);
			d_graph_data_widget->AddPoint(set_id,p4);
		}
		
		if(export_data.uc_a_valid && export_data.uc_b_valid)
		{
			difference = fabs(export_data.dev_a - export_data.dev_b);
			dval = sqrt(export_data.uc_a * export_data.uc_a + export_data.uc_b * export_data.uc_b);
			
			if(dval > ZERO_EPSILON)
			{
				export_data.uc_en = difference / dval;
			}
			else
			{
				export_data.uc_en = 1000.0;
			}
		}
		
		d_export_data.push_back(export_data);
	}
	
	// draw tolerance line for en
	set_en_id = d_graph_en_widget->CreateDataSet(QColor(196,196,196));

	d_graph_en_widget->AddPoint(set_en_id,TVector2(min_position,1.0));
	d_graph_en_widget->AddPoint(set_en_id,TVector2(max_position,1.0));
	
	d_graph_data_widget->UpdateGraph();
	d_graph_en_widget->UpdateGraph();
}

void TLengthCompare::closeEvent(
	QCloseEvent							*event)
{
	if(d_data_changed)
	{
		d_msg_box->setText("Data Not Saved");
		d_msg_box->setInformativeText("Do you want to close without saving?");
		d_msg_box->setDetailedText(QString());
		d_msg_box->setStandardButtons(QMessageBox::Yes | QMessageBox::No);
		d_msg_box->setDefaultButton(QMessageBox::No);
		d_msg_box->setIcon(QMessageBox::Warning);
		
		if(QMessageBox::No == d_msg_box->exec())
		{
			event->ignore();
			return;
		}
	}
	
	d_settings->setValue("Mainwindow_Position", this->pos());
	d_settings->setValue("Mainwindow_Size", this->size());

	d_settings->setValue(QStringLiteral("File_Path"), d_file_path);
	d_settings->setValue(QStringLiteral("Remove_Deviations"), d_options.remove_deviations);

	event->accept();
}


void TLengthCompare::dragEnterEvent(
	QDragEnterEvent 					*drag_enter_event)
{
	if(drag_enter_event->mimeData()->hasFormat("text/uri-list"))
	{
		drag_enter_event->acceptProposedAction();
	}
}

void TLengthCompare::dropEvent(
	QDropEvent 							*drop_event)
{
	QList<QUrl>							urls;
	QString								file_name;
	int									result;
	
	urls = drop_event->mimeData()->urls();
	
	drop_event->acceptProposedAction();
	
	if(urls.size() == 1)
	{
		file_name = (*urls.begin()).toLocalFile();
		
		if(file_name.endsWith(QStringLiteral(".lencomp"),Qt::CaseInsensitive) == true)
		{
			if(d_data_changed)
			{
				d_msg_box->setText("Data Not Saved");
				d_msg_box->setInformativeText("The length comparison data has been modified.  Do you want to save the changes?");
				d_msg_box->setDetailedText(QString());
				d_msg_box->setIcon(QMessageBox::Warning);
				d_msg_box->setStandardButtons(QMessageBox::Yes | QMessageBox::No | QMessageBox::Cancel);
				d_msg_box->setDefaultButton(QMessageBox::Yes);
				
				result = d_msg_box->exec();
				
				switch(result)
				{
					case QMessageBox::Yes:
						this->File_Save();
						break;
						
					case QMessageBox::No:
						break;
						
					case QMessageBox::Cancel:
					default:
						return;
				}
			}
			
			this->Load_Data(file_name);
			
			this->Update();
			d_data_changed = false;
		}
	}
}


void TLengthCompare::Add_Graph_Position(
	const double						&position)
{
	std::vector<double>::const_iterator iter;
	double								separation;
	const double						MIN_SEPARATION(1.0);
	
	for(iter = d_sample_positions.begin();iter != d_sample_positions.end();++iter)
	{
		separation = fabs((*iter) - position);
		
		if(separation < MIN_SEPARATION)
		{
			return;
		}
	}
	
	d_sample_positions.push_back(position);
}

void TLengthCompare::Load_Data_V1(
	TXmlFile							* const xml_file)
{
	QDomElement							e0;
	QDomElement							e1;
	QDomElement							e2;
	std::vector<Type::TEntryItem>		entry_items_a;
	std::vector<Type::TEntryItem>		entry_items_b;
	Type::TEntryItem					entry_item;
	
	assert(xml_file);
	
	d_entry_a_widget->Clear_Entry_Items();
	d_entry_b_widget->Clear_Entry_Items();
	
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Data_Set_A"));
	e1 = xml_file->Get_Node(&e0,QStringLiteral("Entry"),true);
	
	while(!e1.isNull())
	{
		e2 = xml_file->Get_Node(&e1,QStringLiteral("Nominal"));
		entry_item.nominal = xml_file->Read_Text_Node(&e2).toDouble();
		
		e2 = xml_file->Get_Node(&e1,QStringLiteral("Actual"));
		entry_item.actual = xml_file->Read_Text_Node(&e2).toDouble();
		
		e2 = xml_file->Get_Node(&e1,QStringLiteral("Uc"));
		entry_item.uc = xml_file->Read_Text_Node(&e2).toDouble();
		
		entry_items_a.push_back(entry_item);
		
		e1 = xml_file->Get_Sibling_Node(&e1,QStringLiteral("Entry"),true);
	}
	
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Data_Set_B"));
	e1 = xml_file->Get_Node(&e0,QStringLiteral("Entry"),true);
	
	while(!e1.isNull())
	{
		e2 = xml_file->Get_Node(&e1,QStringLiteral("Nominal"));
		entry_item.nominal = xml_file->Read_Text_Node(&e2).toDouble();
		
		e2 = xml_file->Get_Node(&e1,QStringLiteral("Actual"));
		entry_item.actual = xml_file->Read_Text_Node(&e2).toDouble();
		
		e2 = xml_file->Get_Node(&e1,QStringLiteral("Uc"));
		entry_item.uc = xml_file->Read_Text_Node(&e2).toDouble();
		
		entry_items_b.push_back(entry_item);
		
		e1 = xml_file->Get_Sibling_Node(&e1,QStringLiteral("Entry"),true);
	}
	
	d_entry_a_widget->Add_Entry_Items(entry_items_a);
	d_entry_b_widget->Add_Entry_Items(entry_items_b);
	
	d_entry_a_widget->Set_Entry_Offset(0.0);
	d_entry_b_widget->Set_Entry_Offset(0.0);
}


void TLengthCompare::Load_Data_V2(
	TXmlFile							* const xml_file)
{
	QDomElement							e0;
	QDomElement							e1;
	QDomElement							e2;
	std::vector<Type::TEntryItem>		entry_items_a;
	std::vector<Type::TEntryItem>		entry_items_b;
	Type::TEntryItem					entry_item;
	double								entry_offset_a;
	double								entry_offset_b;
	
	assert(xml_file);
	
	d_entry_a_widget->Clear_Entry_Items();
	d_entry_b_widget->Clear_Entry_Items();
		
	e1 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Data_Set_A_Offset"));
	entry_offset_a = xml_file->Read_Text_Node(&e1).toDouble();

	e1 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Data_Set_B_Offset"));
	entry_offset_b = xml_file->Read_Text_Node(&e1).toDouble();
	
	entry_offset_b -= entry_offset_a;
	entry_offset_a = 0.0;

	d_entry_a_widget->Set_Entry_Offset(entry_offset_a);
	d_entry_b_widget->Set_Entry_Offset(entry_offset_b);

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Data_Set_A"));
	e1 = xml_file->Get_Node(&e0,QStringLiteral("Entry"),true);
	
	while(!e1.isNull())
	{
		e2 = xml_file->Get_Node(&e1,QStringLiteral("Nominal"));
		entry_item.nominal = xml_file->Read_Text_Node(&e2).toDouble();
		
		e2 = xml_file->Get_Node(&e1,QStringLiteral("Actual"));
		entry_item.actual = xml_file->Read_Text_Node(&e2).toDouble();
		
		e2 = xml_file->Get_Node(&e1,QStringLiteral("Uc"));
		entry_item.uc = xml_file->Read_Text_Node(&e2).toDouble();
		
		entry_items_a.push_back(entry_item);
		
		e1 = xml_file->Get_Sibling_Node(&e1,QStringLiteral("Entry"),true);
	}
	
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Data_Set_B"));
	e1 = xml_file->Get_Node(&e0,QStringLiteral("Entry"),true);
	
	while(!e1.isNull())
	{
		e2 = xml_file->Get_Node(&e1,QStringLiteral("Nominal"));
		entry_item.nominal = xml_file->Read_Text_Node(&e2).toDouble();
		
		e2 = xml_file->Get_Node(&e1,QStringLiteral("Actual"));
		entry_item.actual = xml_file->Read_Text_Node(&e2).toDouble();
		
		e2 = xml_file->Get_Node(&e1,QStringLiteral("Uc"));
		entry_item.uc = xml_file->Read_Text_Node(&e2).toDouble();
		
		entry_items_b.push_back(entry_item);
		
		e1 = xml_file->Get_Sibling_Node(&e1,QStringLiteral("Entry"),true);
	}
	
	d_entry_a_widget->Add_Entry_Items(entry_items_a);
	d_entry_b_widget->Add_Entry_Items(entry_items_b);
}
