/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <vector>
#include <assert.h>
#include <float.h>
#include <string.h>

#include "referencesystem.h"
#include "vector3.h"
#include "bestfitplane.h"

TBestFitPlane::TBestFitPlane(void)
:d_position(0,0,0),
 d_axis(0,0,1),
 d_form_error(0)
{
}

TBestFitPlane::~TBestFitPlane(void)
{
}

bool TBestFitPlane::Fit(
	const int 							number_of_points,
	const TVector3 						*points)
{
	TReferenceSystem					refsys;
	TVector3							*plane_points;
	int									cntr;
	int									pnt_index;
	double								pnt_length;
	double								inverse_quantity;
	double								sum[6];
	TVector3							pnt;
	TVector3							v1,v2,v3;
	TVector3							difference;
	double								eigen_values[3];
	double								eigen_vectors[9];
	static const double					ZERO_EPSILON(0.000001);

	d_position = TVector3(0, 0, 0);
	d_axis = TVector3(0,0,1);
	d_form_error = 0;

	if(number_of_points < 3)
	{
		return false;
	}
	else if(number_of_points == 3)
	{
		// find the center
		for (cntr = 0; cntr < number_of_points; cntr++)
		{
			d_position += points[cntr];
		}

		inverse_quantity = 1.0 / static_cast<double> (number_of_points);
		d_position *= inverse_quantity;
		
		// find the vector
		v1 = points[1] - points[0];
		v2 = points[2] - points[1];
		v3 = v1 * v2;
		
		if(v3.Length() > ZERO_EPSILON)
		{
			v3.Normal();
			
			d_axis = v3;
			return true;
		}
		
		return false;
	}
	
	// more than 3 points
	v1 = points[number_of_points - 1] - points[0];
	
	if(v1.Length() > ZERO_EPSILON)
	{
		v1.Normal();
	}
	else
	{
		return false;
	}

	pnt_index = -1;
	for(cntr = 1;cntr < (number_of_points - 1);++cntr)
	{
		v2 = points[cntr] - points[0];
		
		if(v2.Length() > ZERO_EPSILON)
		{
			v3 = v1 * v2;
			
			if(pnt_index < 0)
			{
				pnt_index = cntr;
				pnt_length = v3.Length();
			}
			else if(v3.Length() > pnt_length)
			{
				pnt_index = cntr;
				pnt_length = v3.Length();
			}
		}
	}
	
	if(pnt_index < 0)
	{
		return false;
	}
	
	// estimate plane axis
	v2 = points[pnt_index] - points[0];
	v3 = v1 * v2;
	
	if(v3.Length() > ZERO_EPSILON)
	{
		v3.Normal();
		refsys.Skew1(v3,TReferenceSystem::ZPLUS);
	}
	else
	{
		return false;
	}
	
	plane_points = new TVector3[number_of_points];

	for(cntr = 0;cntr < number_of_points;++cntr)
	{
		plane_points[cntr] = refsys.FromWorld(points[cntr]);
	}

	// Find the center
	d_position.Set(0,0,0);
	for (cntr = 0; cntr < number_of_points; cntr++)
	{
		d_position += plane_points[cntr];
	}
	
	inverse_quantity = 1.0 / static_cast<double> (number_of_points);
	d_position *= inverse_quantity;


	// Create error matrix
	memset(sum, 0, sizeof(sum));

	for (cntr = 0; cntr < number_of_points; cntr++)
	{
		difference = plane_points[cntr] - d_position;

		sum[0] += difference.x * difference.x;
		sum[1] += difference.x * difference.y;
		sum[2] += difference.y * difference.y;
		sum[3] += difference.x * difference.z;
		sum[4] += difference.y * difference.z;
		sum[5] += difference.z * difference.z;
	}
	
	// Find Eigen values from data
	Get_Eigen_Values(sum, 3, eigen_vectors, eigen_values);
	
	// isotropic case (infinite number of directions)
	if (eigen_values[0] == eigen_values[1] && eigen_values[0]
	        == eigen_values[2])
	{
		delete[] plane_points;
		return false;
	}

	// regular case
	// The first three (0,1,2) values represent the best fit line
	// The last three values (6,7,8) represent the best fit plane

	d_axis.x = eigen_vectors[6];
	d_axis.y = eigen_vectors[7];
	d_axis.z = eigen_vectors[8];
	
	this->Calculate_Form_Error(number_of_points,plane_points);
	
	d_position = refsys.ToWorld(d_position);
	d_axis = refsys.AxisToWorld(d_axis);
	
	delete[] plane_points;
	return true;
}

void TBestFitPlane::Calculate_Form_Error(
	int 								number_of_points,
	const TVector3 						*points)
{
	double								MaxError;
	double								MinError;
	double								Error;
	TVector3							pnt;
	int									cntr;

	if(number_of_points < 4)
	{
		d_form_error = 0;
		return;
	}

	assert(points);

	// Find farthest points from center.  This represents LS form

	for (cntr = 0; cntr < number_of_points; ++cntr)
	{
		pnt = points[cntr] - d_position;

		Error = pnt.x * d_axis.i + pnt.y * d_axis.j + pnt.z * d_axis.k;

		if(cntr == 0)
		{
			MaxError = Error;
			MinError = Error;
		}
		else
		{
			if(Error > MaxError)
			{
				MaxError = Error;
			}
			else if(Error < MinError)
			{
				MinError = Error;
			}
		}
	}

	d_form_error = MaxError - MinError;
}

void TBestFitPlane::Get_Eigen_Values(
	const double 						*mat,
	const int 							n,
	double 								*eigen_vectors,
	double 								*eigen_values)
{
	static const int 					MAX_ITER(100);	
	static const double 				ZERO_EPSILON(0.000000001);

	// number of entries in mat
	int nn = (n * (n + 1)) / 2;

	// copy matrix
	double *a = new double[nn];
	int ij;
	for (ij = 0; ij < nn; ij++)
	{
		a[ij] = mat[ij];
	}
	// Fortran-porting
	a--;

	// init diagonalization matrix as the unit matrix
	double *v = new double[n * n];
	ij = 0;
	int i;
	for (i = 0; i < n; i++)
	{
		for (int j = 0; j < n; j++)
		{
			if (i == j)
			{
				v[ij++] = 1.0;
			}
			else
			{
				v[ij++] = 0.0;
			}
		}
	}
	// Fortran-porting
	v--;

	// compute weight of the non diagonal terms
	ij = 1;
	double a_norm = 0.0;
	for (i = 1; i <= n; i++)
	{
		for (int j = 1; j <= i; j++)
		{
			if (i != j)
			{
				double a_ij = a[ij];
				a_norm += a_ij * a_ij;
			}
			ij++;
		}
	}

	if (a_norm != 0.0)
	{
		double a_normEPS = a_norm * ZERO_EPSILON;
		double thr = a_norm;

		// rotations
		int nb_iter = 0;
		while (thr > a_normEPS && nb_iter < MAX_ITER)
		{
			nb_iter++;
			double thr_nn = thr / nn;

			for (int l = 1; l < n; l++)
			{
				for (int m = l + 1; m <= n; m++)
				{
					// compute sinx and cosx
					int lq = (l * l - l) / 2;
					int mq = (m * m - m) / 2;

					int lm = l + mq;
					double a_lm = a[lm];
					double a_lm_2 = a_lm * a_lm;

					if (a_lm_2 < thr_nn)
					{
						continue;
					}

					int ll = l + lq;
					int mm = m + mq;
					double a_ll = a[ll];
					double a_mm = a[mm];

					double delta = a_ll - a_mm;

					double x;
					if (delta == 0.0)
					{
						x = (double) -3.1415926535897932384626433832795 / 4;
					}
					else
					{
						x = (double) (-atan((a_lm + a_lm) / delta) / 2.0);
					}

					double sinx = sin(x);
					double cosx = cos(x);
					double sinx_2 = sinx * sinx;
					double cosx_2 = cosx * cosx;
					double sincos = sinx * cosx;

					// rotate L and M columns
					int ilv = n * (l - 1);
					int imv = n * (m - 1);

					int i;
					for (i = 1; i <= n; i++)
					{
						if ((i != l) && (i != m))
						{
							int iq = (i * i - i) / 2;

							int im;
							if (i < m)
							{
								im = i + mq;
							}
							else
							{
								im = m + iq;
							}

							double a_im = a[im];

							int il;
							if (i < l)
							{
								il = i + lq;
							}
							else
							{
								il = l + iq;
							}
							double a_il = a[il];

							a[il] = a_il * cosx - a_im * sinx;
							a[im] = a_il * sinx + a_im * cosx;
						}

						ilv++;
						imv++;

						double v_ilv = v[ilv];
						double v_imv = v[imv];

						v[ilv] = cosx * v_ilv - sinx * v_imv;
						v[imv] = sinx * v_ilv + cosx * v_imv;
					}

					x = a_lm * sincos;
					x += x;

					a[ll] = a_ll * cosx_2 + a_mm * sinx_2 - x;
					a[mm] = a_ll * sinx_2 + a_mm * cosx_2 + x;
					a[lm] = 0.0;

					thr = fabs(thr - a_lm_2);
				}
			}
		}
	}

	// convert indices and copy eigen values
	a++;
	for (i = 0; i < n; i++)
	{
		int k = i + (i * (i + 1)) / 2;
		eigen_values[i] = a[k];
	}

	delete[] a;

	// sort eigen values and vectors
	int *index = new int[n];
	for (i = 0; i < n; i++)
	{
		index[i] = i;
	}

	for (i = 0; i < (n - 1); i++)
	{
		double x = eigen_values[i];
		int k = i;

		for (int j = i + 1; j < n; j++)
		{
			if (x < eigen_values[j])
			{
				k = j;
				x = eigen_values[j];
			}
		}
		eigen_values[k] = eigen_values[i];
		eigen_values[i] = x;

		int jj = index[k];
		index[k] = index[i];
		index[i] = jj;
	}

	// save eigen vectors
	v++; // back to C++
	ij = 0;
	for (int k = 0; k < n; k++)
	{
		int ik = index[k] * n;
		for (int i = 0; i < n; i++)
		{
			eigen_vectors[ij++] = v[ik++];
		}
	}

	delete[] v;
	delete[] index;
}

