/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <assert.h>
#include <float.h>

#include "vector3.h"
#include "bestfitsphere.h"

TBestFitSphere::TBestFitSphere(void)
:d_Position(0,0,0),
 d_Radius(0),
 d_FormError(0)
{
}

TBestFitSphere::~TBestFitSphere(void)
{

}

bool TBestFitSphere::Fit(
	const int 							NumPoints,
	const TVector3 						*Points)
{
	TVector3 							AveragePoints;
	TVector3							InitSphereCenter;
	TVector3 							DifferenceVector;
	TVector3							AverageErrorVector;
	double								DifferenceLength;
	double								InvDifferenceLength;
	int									IterationCntr;
	int									Cntr;
	double								InvNumberOfPoints;
	static const int					MAX_ITERATIONS(10000);
	static const double					ZERO_EPSILON(0.000000001);

	d_Position = TVector3(0, 0, 0);
	d_Radius = 0;
	d_FormError = 0;

	if(NumPoints < 4)
	{
		return false;
	}

	assert(Points);

	InvNumberOfPoints = static_cast<double>(1.0)/static_cast<double>(NumPoints);

	AveragePoints = TVector3(0,0,0);
	for (Cntr = 0; Cntr < NumPoints; Cntr++)
	{
		AveragePoints += Points[Cntr];
	}

	AveragePoints *= InvNumberOfPoints;

	d_Position = AveragePoints;
	
	for (IterationCntr = 0; IterationCntr < MAX_ITERATIONS; ++IterationCntr)
	{
		InitSphereCenter = d_Position;

		d_Radius = 0.0;
		AverageErrorVector = TVector3(0,0,0);

		for (Cntr = 0; Cntr < NumPoints; Cntr++)
		{
			DifferenceVector = Points[Cntr] - d_Position;
			DifferenceLength = DifferenceVector.Length();

			if (DifferenceLength > ZERO_EPSILON)
			{
				d_Radius += DifferenceLength;
				InvDifferenceLength = static_cast<double>(1.0)/DifferenceLength;
				AverageErrorVector -= InvDifferenceLength * DifferenceVector;
			}
		}

		d_Radius *= InvNumberOfPoints;
		AverageErrorVector *= InvNumberOfPoints;

		d_Position = AveragePoints + d_Radius * AverageErrorVector;

		DifferenceVector = d_Position - InitSphereCenter;
		
		if ((fabs(DifferenceVector.x) < ZERO_EPSILON) &&
				(fabs(DifferenceVector.y) < ZERO_EPSILON) &&
				(fabs(DifferenceVector.z) < ZERO_EPSILON))
		{
			break;
		}
	}

	if (IterationCntr < MAX_ITERATIONS)
	{
		this->CalcFormError(NumPoints,Points);
		return true;
	}

	return false;
}

void TBestFitSphere::CalcFormError(
	int 								NumPoints,
	const TVector3 						*Points)
{
	int 								Cntr;
	TVector3 							PRad;
	double 								Rad, RadMax,RadMin;

	if(NumPoints < 4)
	{
		d_FormError = 0;
		return;
	}

	assert(Points);

	for (Cntr = 0; Cntr < NumPoints; ++Cntr)
	{
		PRad = Points[Cntr] - d_Position;
		Rad = PRad.Length();

		if (Cntr == 0)
		{
			RadMax = Rad;
			RadMin = Rad;
		}
		else
		{
			if(Rad > RadMax)
			{
				RadMax = Rad;
			}
			else if(Rad < RadMin)
			{
				RadMin = Rad;
			}
		}
	}

	d_FormError = RadMax - RadMin;
}

