/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QPaintEvent>
#include <QResizeEvent>
#include <QPainter>
#include <QPen>
#include <QBrush>
#include <QColor>
#include <QPoint>
#include <QFontMetrics>

#include "gaugewidget.h"

static const int						MINIMUM_RECT_DIMENSION(12);

TGaugeWidget::TGaugeWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	// defaults
	d_minimum_value = -200.13;
	d_maximum_value = 100.13;
	d_value = -25.0;
	d_display_precision = 2;
	d_minimum_size = QSize(100,50);
	
	d_indicator_type = TGaugeWidget::POINTER;
	d_indicator_color = QColor(0,0,0);
	
	d_update_layout = false;
	d_enable_value_display = false;
	
	d_user_id = -1;
}

TGaugeWidget::~TGaugeWidget(void)
{
}

void TGaugeWidget::setFont(
	const QFont							&font)
{
	QWidget::setFont(font);
	
	d_update_layout = true;
	this->update();
}

void TGaugeWidget::Set_Minimum_Value(
	const double						&value)
{
	d_minimum_value = value;
	
	d_update_layout = true;
	this->update();
}

void TGaugeWidget::Set_Maximum_Value(
	const double						&value)
{
	d_maximum_value = value;
	
	d_update_layout = true;
	this->update();
}

void TGaugeWidget::Set_Value(
	const double						&value)
{
	d_value = value;
	this->update();
}

void TGaugeWidget::Enable_Value_Display(
	const bool							state)
{
	d_enable_value_display = state;
	
	d_update_layout = true;
	this->update();
}

void TGaugeWidget::Set_Value_Display_Suffix(
	const QString						&suffix_text)
{
	d_value_suffix_text = suffix_text;
	this->update();
}

void TGaugeWidget::Set_Display_Text(
	const QString						&display_text)
{
	d_display_text = display_text;
	this->update();
}

void TGaugeWidget::Set_Precision(
	const int							precision)
{
	if(precision < 0 || precision > 9)
	{
		return;
	}
	
	d_display_precision = precision;
	
	d_update_layout = true;
	this->update();
}

void TGaugeWidget::Set_Indicator_Type(
	const TGaugeWidget::TIndicatorType 	type)
{
	d_indicator_type = type;
	this->update();
}

void TGaugeWidget::Set_Indicator_Color(
	QColor 								color)
{
	d_indicator_color = color;
	this->update();
}

void TGaugeWidget::paintEvent(
	QPaintEvent							*event)
{
	Q_UNUSED(event);
	
	QPainter							painter(this);
	QRect								widget_rect;
	QRect								rect;
	QRect								bar_rect;
	QPoint								p1,p2;
	QPoint								pnts[5];
	QString								text;
	QBrush								black_brush(Qt::black);
	QBrush								background_brush(QColor(212,212,224));
	QBrush								indicator_brush(d_indicator_color);
	double								min_value,max_value,value;
	double								increment;
	double								minor_increment;
	int									count;
	int									xi,yi;
	int									cntr;
	int									minor_cntr;
	static const int					MINOR_LINE_COUNT(5);
	
	widget_rect = this->rect();
	
	if(d_update_layout)
	{
		this->Update_Layout(widget_rect);
	}
	
	if(!d_update_layout)
	{
		
		if(d_maximum_value > d_minimum_value)
		{
			min_value = d_minimum_value;
			max_value = d_maximum_value;
		}
		else
		{
			min_value = d_maximum_value;
			max_value = d_minimum_value;
		}
		
		painter.setPen(Qt::lightGray);
		
		// widget border
		painter.drawLine(widget_rect.topLeft(),widget_rect.topRight());
		painter.drawLine(widget_rect.topRight(),widget_rect.bottomRight());
		painter.drawLine(widget_rect.bottomRight(),widget_rect.bottomLeft());
		painter.drawLine(widget_rect.bottomLeft(),widget_rect.topLeft());

		painter.fillRect(d_rect_gauge,background_brush);
		
		painter.setPen(Qt::gray);

		if(d_enable_value_display)
		{
			painter.fillRect(d_rect_value,background_brush);
			painter.drawLine(d_rect_value.topLeft(),d_rect_value.topRight());
		}

		// draw legend and scale
		painter.setPen(Qt::black);
		painter.setBrush(black_brush);
		
		rect = d_rect_text;
		
		count = rect.height() / d_text_row_height;
		
		if(!(count % 2))
		{
			--count;
		}
		
		if(count > 2)
		{
			value = max_value;
			
			increment = (min_value - max_value) / static_cast<double>(count - 1);
			minor_increment = increment / static_cast<double>(MINOR_LINE_COUNT);
			
			for(cntr = 0;cntr < count;++cntr)
			{
				d_viewport_mat.Position_To_Viewport(0,value,&xi,&yi);
				
				rect.setTop(yi - d_text_row_height / 2);
				rect.setHeight(d_text_row_height);
				
				text = QString(" %1").arg(value,0,'f',d_display_precision);
				painter.drawText(rect,Qt::AlignRight|Qt::AlignVCenter,text);
				
				// draw major line
				p1.setX(d_rect_scale.left());
				p2.setX(d_rect_scale.right());
				
				p1.setY(yi);
				p2.setY(yi);
				
				painter.drawLine(p1,p2);
				
				// draw minor lines
				if((count - cntr) > 1)
				{
					p1.setX(d_rect_scale.center().x());
					p2.setX(d_rect_scale.right());
					
					for(minor_cntr = 0;minor_cntr < (MINOR_LINE_COUNT - 1);++minor_cntr)
					{
						value += minor_increment;
						
						d_viewport_mat.Position_To_Viewport(0,value,&xi,&yi);
						
						// draw minor line
						p1.setY(yi);
						p2.setY(yi);
						
						painter.drawLine(p1,p2);
					}
					
					value += minor_increment;
				}
			}
		}
		else
		{
			increment = (min_value - max_value);
			minor_increment = increment / static_cast<double>(MINOR_LINE_COUNT);

			d_viewport_mat.Position_To_Viewport(0,max_value,&xi,&yi);
			
			rect.setTop(yi - d_text_row_height / 2);
			rect.setHeight(d_text_row_height);
			
			text = QString(" %1").arg(max_value,0,'f',d_display_precision);
			painter.drawText(rect,Qt::AlignRight|Qt::AlignVCenter,text);

			// draw major line
			p1.setX(d_rect_scale.left());
			p2.setX(d_rect_scale.right());
			
			p1.setY(yi);
			p2.setY(yi);
			
			painter.drawLine(p1,p2);
			
			p1.setX(d_rect_scale.center().x());
			p2.setX(d_rect_scale.right());
			
			value = max_value;
			
			for(minor_cntr = 0;minor_cntr < (MINOR_LINE_COUNT - 1);++minor_cntr)
			{
				value += minor_increment;
				
				d_viewport_mat.Position_To_Viewport(0,value,&xi,&yi);
				
				// draw minor line
				p1.setY(yi);
				p2.setY(yi);
				
				painter.drawLine(p1,p2);
			}
			
//  'value' is not used past this point
//			value += minor_increment;


			d_viewport_mat.Position_To_Viewport(0,min_value,&xi,&yi);
			
			rect.setTop(yi - d_text_row_height / 2);
			rect.setHeight(d_text_row_height);
			
			text = QString(" %1").arg(min_value,0,'f',d_display_precision);
			painter.drawText(rect,Qt::AlignRight|Qt::AlignVCenter,text);
			
			
			// draw major line
			p1.setX(d_rect_scale.left());
			p2.setX(d_rect_scale.right());
			
			p1.setY(yi);
			p2.setY(yi);
			
			painter.drawLine(p1,p2);
		}
		
		if(d_enable_value_display)
		{
			if(d_display_text.length())
			{
				painter.drawText(d_rect_value,Qt::AlignLeft|Qt::AlignVCenter,d_display_text);
			}
			else
			{
				text = QString("%1 %2").arg(d_value,0,'f',d_display_precision).arg(d_value_suffix_text);
				painter.drawText(d_rect_value,Qt::AlignLeft|Qt::AlignVCenter,text);
			}
		}
		
		// draw indicator pointer
		painter.setBrush(indicator_brush);
		
		if(d_indicator_type == TGaugeWidget::POINTER)
		{
			// draw value
			if(d_value > max_value)
			{
				d_viewport_mat.Position_To_Viewport(0,max_value,&xi,&yi);
				
				pnts[0].setX(d_rect_gauge.left()); pnts[0].setY(yi);
				pnts[1].setX(d_rect_gauge.center().x()); pnts[1].setY(d_rect_gauge.top());
				pnts[2].setX(d_rect_gauge.right()); pnts[2].setY(yi);
				
				painter.drawPolygon(pnts,3);

			}
			else if(d_value < min_value)
			{
				d_viewport_mat.Position_To_Viewport(0,min_value,&xi,&yi);
				
				pnts[0].setX(d_rect_gauge.left()); pnts[0].setY(yi);
				pnts[1].setX(d_rect_gauge.center().x()); pnts[1].setY(d_rect_gauge.bottom());
				pnts[2].setX(d_rect_gauge.right()); pnts[2].setY(yi);
				
				painter.drawPolygon(pnts,3);
			}
			else
			{
				d_viewport_mat.Position_To_Viewport(0,d_value,&xi,&yi);
				
				pnts[0].setX(d_rect_gauge.left()); pnts[0].setY(yi);
				pnts[1].setX(d_rect_gauge.left() + d_text_row_height); pnts[1].setY(yi + d_text_row_height / 2);
				pnts[2].setX(d_rect_gauge.right()); pnts[2].setY(yi + d_text_row_height / 2);
				pnts[3].setX(d_rect_gauge.right()); pnts[3].setY(yi - d_text_row_height / 2);
				pnts[4].setX(d_rect_gauge.left() + d_text_row_height); pnts[4].setY(yi - d_text_row_height / 2);
				
				painter.drawPolygon(pnts,5);
			}
		}
		else	// draw indicator bar
		{
			rect = d_rect_gauge.adjusted(1,0,0,0);
			
			d_viewport_mat.Position_To_Viewport(0,d_value,&xi,&yi);
			rect.setTop(yi);
			
			d_viewport_mat.Position_To_Viewport(0,0.0,&xi,&yi);
			rect.setBottom(yi);
		
			bar_rect = d_rect_gauge.intersected(rect.normalized());
			painter.fillRect(bar_rect,indicator_brush);
		}
	}
}

void TGaugeWidget::resizeEvent(
	QResizeEvent						*event)
{
	QWidget::resizeEvent(event);
	d_update_layout = true;
}

void TGaugeWidget::Update_Layout(
	const QRect							&rect)
{
	int									scale_gauge_width;
	
	this->Update_Minimum_Size(rect);
	
	if(rect.width() < MINIMUM_RECT_DIMENSION)
	{
		return;
	}
	
	if(rect.height() < MINIMUM_RECT_DIMENSION)
	{
		return;
	}
	
	scale_gauge_width = rect.width() - d_rect_text.width();
	
	d_rect_gauge = rect;
	d_rect_gauge.setWidth(scale_gauge_width / 2);
	d_rect_gauge.moveRight(rect.right());
	d_rect_gauge.setBottom(d_rect_text.bottom());

	d_rect_scale = rect;
	d_rect_scale.setLeft(d_rect_text.right());
	d_rect_scale.setRight(d_rect_gauge.left());
	d_rect_scale.setBottom(d_rect_text.bottom());
	
	if(d_enable_value_display)
	{
		d_rect_value = rect;
		d_rect_value.setTop(d_rect_text.bottom());
		
		d_rect_value.adjust(1,0,-1,-1);
	}

	d_rect_text.adjust(1,1,-1,-1);
	d_rect_scale.adjust(2,1,0,-1);
	d_rect_gauge.adjust(1,1,-1,-1);
	
	d_update_layout = false;
	
	this->Update_Viewport();
}

void TGaugeWidget::Update_Viewport(void)
{
	double                              min_x,max_x,min_y,max_y;
	TViewportMat                        W,S,V;
	static const double					MINIMUM_RANGE(0.0001);
	
	max_x = 1.0;
	min_x = 0.0;
	max_y = d_maximum_value;
	min_y = d_minimum_value;
	
	if((max_y - min_y) < MINIMUM_RANGE)
	{
		max_y += (MINIMUM_RANGE / 2.0);
		min_y -= (MINIMUM_RANGE / 2.0);
	}
	
	W.tx[0] = 1;
	W.tx[1] = 0;
	W.ty[0] = 0;
	W.ty[1] = 1;
	W.to[0] = -min_x;
	W.to[1] = -min_y;
	
	S.tx[0] = (d_rect_text.right() - d_rect_text.left()) / (max_x - min_x);
	S.tx[1] = 0;
	S.ty[0] = 0;
	S.ty[1] = (d_rect_text.top() - d_rect_text.bottom() + d_text_row_height) / (max_y - min_y);
	S.to[0] = 0;
	S.to[1] = 0;
	
	V.tx[0] = 1;
	V.tx[1] = 0;
	V.ty[0] = 0;
	V.ty[1] = 1;
	V.to[0] = d_rect_text.left();
	V.to[1] = d_rect_text.bottom() - d_text_row_height / 2;
	
	d_viewport_mat = W;
	d_viewport_mat *=S;
	d_viewport_mat *=V;
}

void TGaugeWidget::Update_Minimum_Size(
	const QRect							&widget_rect)
{
	QRect								minimum_rect;
	QRect								maximum_rect;
	QRect								rect;
	QString								text;
	QFontMetrics						font_metrics(this->font());
	
	text = QString(" %1").arg(d_minimum_value,0,'f',d_display_precision);
	minimum_rect = font_metrics.boundingRect(text);

	text = QString(" %1").arg(d_maximum_value,0,'f',d_display_precision);
	maximum_rect = font_metrics.boundingRect(text);

	rect = minimum_rect.united(maximum_rect);

	d_rect_text = widget_rect;
	d_rect_text.setWidth(rect.width());
	
	d_text_row_height = rect.height();
	
	if(d_enable_value_display)
	{
		d_minimum_size.setHeight(d_text_row_height * 4);
		d_rect_text.setBottom(widget_rect.bottom() - d_text_row_height);
	}
	else
	{
		d_minimum_size.setHeight(d_text_row_height * 3);
	}
	
	d_minimum_size.setWidth(rect.width() + font_metrics.averageCharWidth() * 8);
	
	this->updateGeometry();
}
