/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#ifndef GAUGEWIDGETHEADERFILE
#define GAUGEWIDGETHEADERFILE

#include <QWidget>
#include <QFont>
#include <QSize>
#include <QRect>
#include <QColor>

class QPaintEvent;
class QResizeEvent;

class TGaugeWidget : public QWidget
{
Q_OBJECT

public:
	
enum TIndicatorType
{
	POINTER = 0,
	BAR_GRAPH
};

// CREATORS
	TGaugeWidget(const QWidget *parent=0, Qt::WindowFlags flags=Qt::WindowFlags());
	virtual ~TGaugeWidget(void);

// ACCESSORS
	virtual QSize minimumSizeHint(void) const{return d_minimum_size;}
	
	double Minimum(void) const {return d_minimum_value;}
	double Maximum(void) const {return d_maximum_value;}
	double Value(void) const {return d_value;}
	
	int Precision(void) const {return d_display_precision;}
	int User_Id(void) const {return d_user_id;}

// MANIPULATORS
	virtual void setFont(const QFont &font);
	
	void Set_Minimum_Value(const double &value);
	void Set_Maximum_Value(const double &value);
	void Set_Value(const double &value);
	
	void Enable_Value_Display(const bool state);
	void Set_Value_Display_Suffix(const QString &suffix_text);
	void Set_Display_Text(const QString &display_text);
	
	void Set_Precision(const int precision);
	void Set_Indicator_Type(const TIndicatorType type);
	void Set_Indicator_Color(QColor color);
	
	void Set_User_Id(const int id) {d_user_id = id;}

signals:

private slots:
	
protected:
	virtual void paintEvent(QPaintEvent *event);
	virtual void resizeEvent(QResizeEvent *event);
	
private:
	
struct TViewportMat
{
	double								tx[2];
	double								ty[2];
	double								to[2];
	
	TViewportMat& operator*=(const TViewportMat &rhs)
	{
		double							x,y,o[2];
		
		x = tx[0] * rhs.tx[0];
		y = ty[1] * rhs.ty[1];
		o[0] = to[0] * rhs.tx[0] + rhs.to[0];
		o[1] = to[1] * rhs.ty[1] + rhs.to[1];
		
		tx[0] = x;
		ty[1] = y;
		to[0] = o[0];
		to[1] = o[1];
		
		return *this;
	}
	
	void Position_To_Viewport(const double Xd, const double Yd, int *X, int *Y) const
	{
		*X = static_cast<int>(Xd * tx[0] + to[0]);
		*Y = static_cast<int>(Yd * ty[1] + to[1]);
	}
	
	void Viewport_To_Position(const int X, const int Y, double *Xd, double *Yd) const
	{
		*Xd = (static_cast<double>(X) - to[0]) / tx[0];
		*Yd = (static_cast<double>(Y) - to[1]) / ty[1];
	}
};

	TViewportMat						d_viewport_mat;
	TIndicatorType						d_indicator_type;
	QColor								d_indicator_color;

	QRect								d_rect_text;
	QRect								d_rect_scale;
	QRect								d_rect_gauge;
	QRect								d_rect_value;
	QSize								d_minimum_size;
	QString								d_value_suffix_text;
	QString								d_display_text;
	double								d_minimum_value;
	double								d_maximum_value;
	double								d_value;
	int									d_display_precision;
	int									d_text_row_height;
	int									d_user_id;
	bool								d_update_layout;
	bool								d_enable_value_display;

	
	void Update_Layout(const QRect &rect);
	void Update_Viewport(void);
	void Update_Minimum_Size(const QRect &widget_rect);		// calculates d_rect_text to minimum size

// NOT IMPLEMENTED
	TGaugeWidget(const TGaugeWidget&);
	TGaugeWidget& operator=(const TGaugeWidget&);
};

#endif
