/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#ifndef GRAPHWIDGETHEADERFILE
#define GRAPHWIDGETHEADERFILE

#include <QVariant>
#include <QWidget>
#include <QRect>
#include <QPainter>
#include <QPixmap>

#include <vector>

#include "vector2.h"

class QWidget;
class QPaintEvent;
class QResizeEvent;
class QMouseEvent;
class QPainter;

class TGraphWidget : public QWidget
{
	Q_OBJECT

public:

enum TGraphType
{
	DEFAULT_GRAPH=0,
	LINE_GRAPH,
	BAR_GRAPH,
	POINT_GRAPH,
	DOT_GRAPH
};

enum TLegendMode
{
	NO_LEGEND=0,
	DEFAULT_LEGEND_PRECISION,
	INTEGER_LEGEND_PRECISION,
	LOG10_LEGEND_PRECISION
};


// CREATORS
	TGraphWidget(const QWidget *parent=0,Qt::WindowFlags flags=Qt::WindowFlags());
	virtual ~TGraphWidget(void);

// ACCESSORS
	int NumberDataSets(void) const {return static_cast<int>(d_data_sets.size());}
	int DataSetSize(const int set) const;
	TVector2 GetPoint(const int set,const int index) const;

// MANIPULATORS	
	void VerticalLegend(const QString &s) {d_vertical_legend_title = s;}
	void HorizontalLegend(const QString &s) {d_horizontal_legend_title = s;}
	void Title(const QString &s) {d_title = s;}
	void SetHLegendMode(const TLegendMode state) {d_horizontal_legend_mode = state;}
	void SetVLegendMode(const TLegendMode state) {d_vertical_legend_mode = state;}
	void SetGraphType(const TGraphType type) {d_graph_type = type;}

	bool DeleteDataSet(const int index);
	void DeleteDataSets(void);

	int CreateDataSet(const QColor color,const TGraphWidget::TGraphType type = TGraphWidget::DEFAULT_GRAPH);
	bool AddPoint(const int set,const TVector2 &point);
	bool AddPoints(const int set,const std::vector<TVector2> &points);
	bool SetPoints(const int set,const std::vector<TVector2> &points);
	bool ModifyPoint(const int set,const int index,const TVector2 &new_pnt);
	bool DeletePoint(const int set,const int index);
	bool DeletePoints(const int set);
	
	void UpdateGraph(void);
	
	bool PaintToDevice(QPainter *painter,const QRect &rect,const bool enable_border = true);
	bool PaintToDevice_GraphOnly(QPainter *Device,const QRect &DeviceRect,const bool enable_border = true);
	
	void SetScaleMinimumY(const double &minimum_y,const bool bidirectional = true);	// option EnableYIncludesZero must be true
	void EnableYIncludesZero(const bool state);
	
	void LockAspectRatio(const bool state) {d_lock_aspect = state;}
	void EnableMouseCoordinateDisplay(const bool state) {d_show_mouse_tracking = state;}
	
	// direct rendering speeds up drawing considerably but disables pan and zoom functions
	// default is off
	void Enable_Direct_Render(const bool state) {d_enable_direct_rendering = state;}

	
// GenerateRGBColor always creates color.  Returns false and black when set is above 96
	static bool GenerateRGBColor(const int Set,int *R,int *G,int *B);
	
signals:
	void Mouse_Click(const double&,const double&);	// xy position in graph coordinates
	

public slots:
	void ZoomOut(void);
	void ScaleToFit(void);

protected:
	
	virtual void paintEvent(QPaintEvent*);
	virtual void resizeEvent(QResizeEvent*);
	virtual void mousePressEvent(QMouseEvent*);
	virtual void mouseMoveEvent(QMouseEvent*);
	virtual void mouseReleaseEvent(QMouseEvent*);
	
private:

// NOT IMPLEMENTED
	TGraphWidget(const TGraphWidget&);
	TGraphWidget& operator=(const TGraphWidget&);	

struct TViewportMat
{
	double								tx[2];
	double								ty[2];
	double								to[2];

	TViewportMat& operator*=(const TViewportMat &rhs)
	{
		double							x,y,o[2];

		x = tx[0] * rhs.tx[0];
		y = ty[1] * rhs.ty[1];
		o[0] = to[0] * rhs.tx[0] + rhs.to[0];
		o[1] = to[1] * rhs.ty[1] + rhs.to[1];

		tx[0] = x;
		ty[1] = y;
		to[0] = o[0];
		to[1] = o[1];

		return *this;
	}

	void Position_To_Viewport(const double Xd, const double Yd, int *X, int *Y) const
	{
		*X = static_cast<int>(Xd * tx[0] + to[0]);
		*Y = static_cast<int>(Yd * ty[1] + to[1]);
	}

	void Viewport_To_Position(const int X, const int Y, double *Xd, double *Yd) const
	{
		*Xd = (static_cast<double>(X) - to[0]) / tx[0];
		*Yd = (static_cast<double>(Y) - to[1]) / ty[1];
	}
	
};

struct TDataSet
{
	std::vector<TVector2>				dataset_points;
	QColor								dataset_color;
	TGraphType							dataset_graph_type;
};

	TViewportMat						d_viewport_mat;
	std::vector<TDataSet>				d_data_sets;

	QRect								d_graph_rect;
	QRect								d_vertical_legend_rect;
	QRect								d_vertical_legend_title_rect;
	QRect								d_horizontal_legend_rect;
	QRect								d_horizontal_legend_title_rect;
	QRect								d_title_rect;
	QRect								d_viewport_rect;
	QRect								d_mouse_down_rect;

	QString								d_vertical_legend_title;
	QString								d_horizontal_legend_title;
	QString								d_title;
	
	int									d_horizontal_text_total;
	int									d_horizontal_text_precision;
	int									d_vertical_text_total;
	int									d_vertical_text_precision;
	
	QSize								d_vertical_label_size;
	QSize								d_horizontal_label_size;
	TGraphType							d_graph_type;

	double								d_data_range_x;
	double								d_data_range_y;
	QPixmap								d_widget_pixmap;

	QString								d_tracking_text;

	bool								d_draw_mouse_rect;
	bool								d_left_mouse_down;
	bool								d_left_mouse_moving;
	bool								d_right_mouse_down;
	bool								d_right_mouse_moving;
	bool								d_show_mouse_tracking;
	bool								d_enable_direct_rendering;
	bool								d_y_includes_zero;
	bool								d_bidirectional_y_min_value;
	
	bool								d_draw_border;
	bool								d_process_scale_to_fit;
	bool								d_process_zoom_out;
	TLegendMode							d_horizontal_legend_mode;
	TLegendMode							d_vertical_legend_mode;
	
	double								d_y_min_value;
	bool								d_lock_aspect;
	
	
	void Draw_Background_Pixmap(void);
	void Draw_Background_Scaled_Pixmap(const QRect &rect);	// zoom in/out functions
	void Direct_Render(QPainter *painter);
	void Update_Viewport_To_Data(void);
	void Update_Viewport_To_Rect(const QRect &rect);
	void Draw_Legend(QPainter*);
	void Draw_Graph(QPainter*);
	void Calculate_Legend_Display_Precision(QPainter*);
	void Update_Left_Down_Box(void);
	
	int Forward_Log10_Position_Horizontal(const int x) const;
	int Forward_Log10_Position_Vertical(const int y) const;
	int Reverse_Log10_Position_Horizontal(const int x) const;
	int Reverse_Log10_Position_Vertical(const int y) const;

	int Calculate_Log10_Position_Horizontal(const double &x) const;
	double Calculate_Log10_Value_Horizontal(const int x) const;
	int Calculate_Log10_Position_Vertical(const double &y) const;
	double Calculate_Log10_Value_Vertical(const int y) const;

};

#endif
