/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <math.h>
#include <assert.h>
#include <float.h>
#include <string.h>

#include "matrix.h"

TMatrix::TMatrix(void)
: d_Data(0),
  d_Columns(0),
  d_Rows(0)
{

}

TMatrix::TMatrix(
	const int							Rows,
	const int							Columns)
{
	assert(Rows);
	assert(Columns);

	d_Columns = Columns;
	d_Rows = Rows;

	d_Data = new double[d_Columns * d_Rows];

	memset(d_Data,0,sizeof(double) * d_Columns * d_Rows);
}

TMatrix::TMatrix(
	const TMatrix						&rhs)
: d_Data(0),
d_Columns(0),
d_Rows(0)
{
	if(rhs.d_Data)
	{
		d_Columns = rhs.d_Columns;
		d_Rows = rhs.d_Rows;

		d_Data = new double[d_Columns * d_Rows];
		memcpy(d_Data,rhs.d_Data,sizeof(double) * d_Columns * d_Rows);
	}
}

TMatrix& TMatrix::operator=(
	const TMatrix						&rhs)
{

	if(&rhs != this)
	{
		if(d_Data)
		{
			delete[] d_Data;
			d_Columns = 0;
			d_Rows = 0;
		}

		if(rhs.d_Data)
		{
			d_Columns = rhs.d_Columns;
			d_Rows = rhs.d_Rows;

			d_Data = new double[d_Columns * d_Rows];
			memcpy(d_Data,rhs.d_Data,sizeof(double) * d_Columns * d_Rows);
		}
	}

	return *this;
}

TMatrix::~TMatrix(void)
{
	if(d_Data)
	{
		delete[] d_Data;
		d_Rows = 0;
		d_Columns = 0;
	}
}

void TMatrix::Transpose(void)
{
	int									Row;
	int									Column;
	int									iVal;
	double								*Data;

	if(d_Data)
	{
		Data = new double[d_Columns * d_Rows];

		for(Row = 0;Row < d_Rows;++Row)
		{
			for(Column = 0;Column < d_Columns;++Column)
			{
				Data[Row * d_Columns + Column] = d_Data[Column * d_Rows + Row];
			}
		}

		iVal = d_Rows;
		d_Rows = d_Columns;
		d_Columns = iVal;

		memcpy(d_Data,Data,sizeof(double) * d_Columns * d_Rows);

		delete[] Data;
	}
}

void TMatrix::Resize(
	const int							Rows,
	const int							Columns)
{
	assert(Columns);
	assert(Rows);

	if(d_Data)
	{
		delete[] d_Data;
	}

	d_Columns = Columns;
	d_Rows = Rows;

	d_Data = new double[d_Columns * d_Rows];

	memset(d_Data,0,sizeof(double) * d_Columns * d_Rows);
}

void TMatrix::SwapRows(
	const int							Row1,
	const int							Row2)
{
	double								dVal;
	int									Cntr;

	assert(Row1 < d_Rows);
	assert(Row2 < d_Rows);

	if(Row1 != Row2)
	{
		for(Cntr = 0;Cntr < d_Columns;++Cntr)
		{
			dVal = d_Data[Cntr * d_Rows + Row1];
			d_Data[Cntr * d_Rows + Row1] = d_Data[Cntr * d_Rows + Row2];
			d_Data[Cntr * d_Rows + Row2] = dVal;
		}
	}
}

void TMatrix::SwapColumns(
	const int							Column1,
	const int							Column2)
{
	double								dVal;
	int									Cntr;

	assert(Column1 < d_Columns);
	assert(Column2 < d_Columns);

	if(Column1 != Column2)
	{
		for(Cntr = 0;Cntr < d_Rows;++Cntr)
		{
			dVal = d_Data[Column1 * d_Rows + Cntr];
			d_Data[Column1 * d_Rows + Cntr] = d_Data[Column2 * d_Rows + Cntr];
			d_Data[Column2 * d_Rows + Cntr] = dVal;
		}
	}
}

TMatrix& TMatrix::operator+=(
	const TMatrix						&rhs)
{
	int									Size;

	assert(d_Rows == rhs.d_Rows);
	assert(d_Columns == rhs.d_Columns);

	if(this != &rhs)
	{
		Size = d_Rows * d_Columns;

		for(int Cntr = 0;Cntr < Size;++Cntr)
		{
			d_Data[Cntr] += rhs.d_Data[Cntr];
		}
	}

	return *this;
}

TMatrix& TMatrix::operator-=(
	const TMatrix						&rhs)
{
	int									Size;

	assert(d_Rows == rhs.d_Rows);
	assert(d_Columns == rhs.d_Columns);

	if(this != &rhs)
	{
		Size = d_Rows * d_Columns;

		for(int Cntr = 0;Cntr < Size;++Cntr)
		{
			d_Data[Cntr] -= rhs.d_Data[Cntr];
		}
	}

	return *this;
}

TMatrix operator+(
	const TMatrix						&lhs,
	const TMatrix						&rhs)
{
	TMatrix								Res(lhs);

	Res+= rhs;
	return Res;
}

TMatrix operator-(
	const TMatrix						&lhs,
	const TMatrix						&rhs)
{
	TMatrix								Res(lhs);

	Res-= rhs;
	return Res;
}

void TMatrix::Zero(void)
{
	if(d_Data)
	{
		memset(d_Data,0,sizeof(double) * d_Columns * d_Rows);
	}
}

/*
void TMatrix::Debug(void)
{
	int							Column;
	int							Row;

	debug << _T("TMatrix Data ") << std::endl;

	for(Row = 0;Row < d_Rows;++Row)
	{
		for(Column = 0;Column < d_Columns;++Column)
		{
			debug << _T(" ") << this->At(Row,Column);
		}

		debug << std::endl;
	}
}
*/
bool TMatrix::Invert(void)
{
	double								**inv,tmp;
	int									i,j,k,retval;
	int									Size;

	assert(d_Rows == d_Columns);

	Size = d_Rows * d_Columns;

	retval = 0;

	inv = new double *[Size];

	for (i=0;i<Size;i++) 
	{
		inv[i] = new double [Size];
	}

	// Initialize identity matrix
	for (i=0;i<Size;i++) 
	{
		for (j=0;j<Size;j++) 
		{
			inv[i][j] = 0.0;
		}

		inv[i][i] = 1.0;
	}

	for (k=0;k<Size;k++) 
	{
		tmp = d_Data[k * d_Columns + k];//a[k][k];

		if (tmp == 0) 
		{
			retval = 1;
			goto _100;
		}

		for (j=0;j<Size;j++) 
		{
			//            if (j>k) a[k][j] /= tmp;    // Don't bother with previous entries
			if (j>k) d_Data[k * d_Columns + j] /= tmp;    // Don't bother with previous entries
			inv[k][j] /= tmp;
		}

		for (i=0;i<Size;i++) 
		{             // Loop over rows
			if (i == k) continue;
			tmp = d_Data[i*d_Columns + k];//a[i][k];

			for (j=0;j<Size;j++) 
			{
				//                if (j>k) a[i][j] -= d_Data[k * d_Columns + j] * tmp;//a[k][j]*tmp;
				if (j>k) d_Data[i * d_Columns + j] -= d_Data[k * d_Columns + j] * tmp;//a[k][j]*tmp;
				inv[i][j] -= inv[k][j]*tmp;
			}
		}
	}

	// Copy inverse to source matrix
	for (i=0;i<Size;i++) 
	{
		for (j=0;j<Size;j++) 
		{
			//            a[i][j] = inv[i][j];
			d_Data[i * d_Columns + j] = inv[i][j];
		}
	}
_100:
	for (i=0;i<Size;i++) 
	{
		delete [] inv[i];
	}

	delete [] inv;
	return (retval == 0);
}

TMatrix operator*(
	const TMatrix						&lhs,
	const TMatrix						&rhs)
{
	double								dVal;
	int									Row,Column,n;

	assert(lhs.Columns() == rhs.Rows());

	TMatrix						Result(lhs.Rows(),rhs.Columns());

	for(Row = 0;Row < lhs.Rows();++Row)
	{
		for(Column = 0;Column < rhs.Columns();++Column)
		{
			dVal = 0;

			for(n = 0;n < lhs.Columns();++n)
			{
				dVal += lhs.At(Row,n) * rhs.At(n,Column);
			}

			Result.At(Row,Column,dVal);
		}
	}

	return Result;
}

bool MatrixInverse (
	const TMatrix						&rkA,
	TMatrix								&rkInvA)
{
    // computations are performed in-place
    assert(rkA.Rows() == rkA.Columns());

//    int iSize = rkInvA.GetRows();
    int iSize = rkInvA.Rows();
    rkInvA = rkA;

    int* aiColIndex = new int[iSize];
    assert( aiColIndex );

    int* aiRowIndex = new int[iSize];
    assert( aiRowIndex );

    bool* abPivoted = new bool[iSize];
    assert( abPivoted );

    memset(abPivoted,0,iSize*sizeof(bool));

    int i1, i2, iRow = 0, iCol = 0;
    double fSave;

    // elimination by full pivoting
    for (int i0 = 0; i0 < iSize; i0++)
    {
        // search matrix (excluding pivoted rows) for maximum absolute entry
        double fMax = 0.0;
        for (i1 = 0; i1 < iSize; i1++)
        {
            if ( !abPivoted[i1] )
            {
                for (i2 = 0; i2 < iSize; i2++)
                {
                    if ( !abPivoted[i2] )
                    {
//                        Real fAbs = Math<Real>::FAbs(rkInvA[i1][i2]);
                        double fAbs = fabs(rkInvA.At(i1,i2));
                        if(fAbs > fMax)
                        {
                            fMax = fAbs;
                            iRow = i1;
                            iCol = i2;
                        }
                    }
                }
            }
        }

        if(fMax == 0.0)
        {
            // matrix is not invertible
            delete[] aiColIndex;
            delete[] aiRowIndex;
            delete[] abPivoted;
            return false;
        }

        abPivoted[iCol] = true;

        // swap rows so that A[iCol][iCol] contains the pivot entry
        if ( iRow != iCol )
	   {
//            rkInvA.SwapRows(iRow,iCol);
            rkInvA.SwapRows(iRow,iCol);
//			rkInvA.SwapColumns(iRow,iCol);
	   }

        // keep track of the permutations of the rows
        aiRowIndex[i0] = iRow;
        aiColIndex[i0] = iCol;

        // scale the row so that the pivot entry is 1
//        Real fInv = 1.0/rkInvA[iCol][iCol];
        double fInv = 1.0/rkInvA.At(iCol,iCol);
        rkInvA.At(iCol,iCol,1.0);
        for (i2 = 0; i2 < iSize; i2++)
        {
//            rkInvA[iCol][i2] *= fInv;
			rkInvA.Multiply(iCol,i2,fInv);
        }

        // zero out the pivot column locations in the other rows
        for (i1 = 0; i1 < iSize; i1++)
        {
            if ( i1 != iCol )
            {
//                fSave = rkInvA[i1][iCol];
                fSave = rkInvA.At(i1,iCol);//[i1][iCol];
//                rkInvA[i1][iCol] = (Real)0.0;
                rkInvA.At(i1,iCol,0.);//[i1][iCol] = (Real)0.0;
                for (i2 = 0; i2 < iSize; i2++)
                {
//                    rkInvA[i1][i2] -= rkInvA[iCol][i2]*fSave;
                    rkInvA.Subtract(i1,i2,rkInvA.At(iCol,i2)*fSave);
                }
            }
        }
    }

    // reorder rows so that A[][] stores the inverse of the original matrix
    for (i1 = iSize-1; i1 >= 0; i1--)
    {
        if ( aiRowIndex[i1] != aiColIndex[i1] )
        {
            for (i2 = 0; i2 < iSize; i2++)
            {
//                fSave = rkInvA[i2][aiRowIndex[i1]];
  //              rkInvA[i2][aiRowIndex[i1]] = rkInvA[i2][aiColIndex[i1]];
    //            rkInvA[i2][aiColIndex[i1]] = fSave;
                fSave = rkInvA.At(i2,aiRowIndex[i1]);
                rkInvA.At(i2,aiRowIndex[i1],rkInvA.At(i2,aiColIndex[i1]));
                rkInvA.At(i2,aiColIndex[i1],fSave);
            }
        }
    }

    delete[] aiColIndex;
    delete[] aiRowIndex;
    delete[] abPivoted;
    return true;
}
