/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <Qt>
#include <QAction>
#include <QApplication>
#include <QLineEdit>
#include <QCheckBox>
#include <QToolButton>
#include <QComboBox>
#include <QDateTimeEdit>
#include <QToolButton>
#include <QFileDialog>
#include <QHBoxLayout>
#include <QResizeEvent>
#include <QSize>
#include <QSpacerItem>
#include <assert.h>

#include "optioneditor.h"

static const int						LINE_EDITOR 	= 1;
static const int						PATH_EDITOR		= 2;
static const int						CHECK_OPTION	= 3;
static const int						DATETIME_EDITOR	= 4;

TPathEdit::TPathEdit(
	const QWidget						*parent,
	Qt::WindowFlags						flags)
:QLineEdit(const_cast<QWidget*>(parent))
{
	Q_UNUSED(flags);
	
	QHBoxLayout 						*hboxLayout;
	QSpacerItem							*spacerItem;

	hboxLayout = new QHBoxLayout(this);
	hboxLayout->setSpacing(0);
	hboxLayout->setContentsMargins(0,0,0,0);
	
	spacerItem = new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum);
	hboxLayout->addItem(spacerItem);
	
	d_BrowseButton = new QToolButton(this);
	hboxLayout->addWidget(d_BrowseButton);
	
	d_BrowseButton->setText(QStringLiteral("..."));
	d_row = 0;
	d_column = 0;
	
	connect(d_BrowseButton,&QToolButton::clicked,this,&TPathEdit::Browse_Path_Clicked);
}

TPathEdit::~TPathEdit(void)
{
}

void TPathEdit::resizeEvent(
	QResizeEvent 						*event)
{
	QSize								adjusted_size;
	
	adjusted_size = event->size();
	adjusted_size.rwidth() -= d_BrowseButton->size().width();
	
	QResizeEvent resize_event(adjusted_size,this->size());
	QLineEdit::resizeEvent(&resize_event);
}

void TPathEdit::Browse_Path_Clicked(void)
{
	// queued connection is used
	emit Browse_Path(this->text(),d_row,d_column);
}

TOptionEditorWidget::TOptionEditorWidget(
	const QWidget						*parent)
:QTreeWidget(const_cast<QWidget*>(parent))
{
	d_item_delegate = new TOptionItemDelegate();
	
    this->setItemDelegate(d_item_delegate);
	this->setLayoutDirection(Qt::LeftToRight);
	this->setEditTriggers(QAbstractItemView::AllEditTriggers);
	this->setAlternatingRowColors(true);
	this->setTextElideMode(Qt::ElideRight);
	this->setRootIsDecorated(false);
	this->setItemsExpandable(false);
	this->setAnimated(false);
	
	d_item_delegate->Set_Child_Dialog_Parent(this);
	
	this->setColumnCount(2);
	this->clear();
	
	connect(d_item_delegate,&TOptionItemDelegate::List_Item_Changed,this,&TOptionEditorWidget::Edit_List_Changed);
}

TOptionEditorWidget::~TOptionEditorWidget(void)
{
	delete d_item_delegate;
}

QString TOptionEditorWidget::Option_Name(
	const int 							index) const
{
	QTreeWidgetItem						*item;
	
	item = this->topLevelItem(index);
	
	if(item)
	{
		return item->text(0);
	}
	
	return QString();
}

QString TOptionEditorWidget::Option_Value(
	const int 							index) const
{
	QTreeWidgetItem						*item;
		
	item = this->topLevelItem(index);
	
	if(item)
	{
		if(item->data(1,Qt::UserRole).toInt() == CHECK_OPTION)
		{
			if(item->checkState(1) == Qt::Checked)
			{
				return QString("Yes");
			}

			return QString("No");
		}

		return item->text(1);
	}
	
	return QString();
}

int TOptionEditorWidget::Add_Line_Editor_Option(
	const QString 						&Title,
	const QString 						&DefaultValue,
	const QString						&Column2,
	const QString						&Column3,
	const QString						&Column4,
	const QString						&Column5,
	const QString						&Column6)
{
	QTreeWidgetItem 					*OptionItem;
	 
	OptionItem = new QTreeWidgetItem(this);
	OptionItem->setText(0, Title);
	OptionItem->setText(1, DefaultValue);
	OptionItem->setText(2, Column2);
	OptionItem->setText(3, Column3);
	OptionItem->setText(4, Column4);	
	OptionItem->setText(5, Column5);
	OptionItem->setText(6, Column6);
	OptionItem->setData(1,Qt::UserRole,LINE_EDITOR);
	OptionItem->setFlags(Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled);
	
	return this->topLevelItemCount()-1;
}

int TOptionEditorWidget::Add_Line_Editor_Option(
	const QString 						&Title,
	const int 							DefaultValue,
	const QString						&Column2,
	const QString						&Column3,
	const QString						&Column4,
	const QString						&Column5,
	const QString						&Column6)
{
	return this->Add_Line_Editor_Option(Title,QString("%1").arg(DefaultValue),Column2,Column3,Column4,Column5,Column6);
}


int TOptionEditorWidget::Add_Path_Editor_Option(
	const QString 						&Title,
	const QString 						&DefaultPath)
{
	QTreeWidgetItem 					*OptionItem;
	 
	OptionItem = new QTreeWidgetItem(this);
	OptionItem->setText(0, Title);
	OptionItem->setText(1, DefaultPath);
	OptionItem->setData(1,Qt::UserRole,PATH_EDITOR);
	OptionItem->setFlags(Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled);
	
	return this->topLevelItemCount()-1;
}

int TOptionEditorWidget::Add_Checkbox_Option(
	const QString 						&Title,
	const bool							State,
	const QString						&Column2,
	const QString						&Column3,
	const QString						&Column4,
	const QString						&Column5,
	const QString						&Column6)
{
	QTreeWidgetItem 					*OptionItem;

	OptionItem = new QTreeWidgetItem(this);
	OptionItem->setText(0, Title);
	if(State)
	{
		OptionItem->setCheckState(1,Qt::Checked);//setText(1, "Yes");
	}
	else
	{
		OptionItem->setCheckState(1,Qt::Unchecked);//setText(1, "No");
	}

	OptionItem->setData(1,Qt::UserRole,CHECK_OPTION);
	OptionItem->setText(2, Column2);
	OptionItem->setText(3, Column3);
	OptionItem->setText(4, Column4);
	OptionItem->setText(5, Column5);
	OptionItem->setText(6, Column6);
	OptionItem->setFlags(Qt::ItemIsSelectable | Qt::ItemIsUserCheckable | Qt::ItemIsEditable | Qt::ItemIsEnabled);

	return this->topLevelItemCount()-1;
}

int TOptionEditorWidget::Add_DateTime_Option(
	const QString 						&Title,
	const QDateTime 					&DefaultDateTime)
{
	QTreeWidgetItem 					*OptionItem;
	 
	OptionItem = new QTreeWidgetItem(this);
	OptionItem->setText(0, Title);
	OptionItem->setText(1, DefaultDateTime.toString(QString("ddd MMM dd hh:mm:ss yyyy AP")));
	OptionItem->setData(1,Qt::UserRole,DATETIME_EDITOR);
	OptionItem->setFlags(Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled);
	
	return this->topLevelItemCount()-1;
}


int TOptionEditorWidget::Add_Combobox_Option(
	const QString 						&Title,
	const QStringList 					&Items,
	const int 							DefaultIndex,
	const QString						&Column2,
	const QString						&Column3,
	const QString						&Column4,
	const QString						&Column5,
	const QString						&Column6)
{
	QTreeWidgetItem 					*OptionItem;
	
	assert(DefaultIndex < Items.size());

	OptionItem = new QTreeWidgetItem(this);
	OptionItem->setText(0, Title);
	OptionItem->setText(1, Items[DefaultIndex]);
	OptionItem->setData(1,Qt::UserRole,Items);
	OptionItem->setText(2, Column2);
	OptionItem->setText(3, Column3);
	OptionItem->setText(4, Column4);
	OptionItem->setText(5, Column5);
	OptionItem->setText(6, Column6);
	OptionItem->setFlags(Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled);
	
	return this->topLevelItemCount()-1;
}

void TOptionEditorWidget::Change_Combobox_Option(
	const int 							ItemIndex,
	const int 							ListIndex)
{
	QTreeWidgetItem						*item;
	QVariant							item_data;
	QStringList							item_list;
	
	item = this->topLevelItem(ItemIndex);

	if(item)
	{
		item_data = item->data(1,Qt::UserRole);
		
		if(item_data.canConvert(QVariant::StringList))
		{
			item_list=item_data.toStringList();
			
			if(ListIndex < item_list.size())
			{
				this->Change_Option_Text(ItemIndex,item_list[ListIndex]);
			}
		}
	}
}

bool TOptionEditorWidget::Change_Option_Text(
	const int 							index,
	const QString 						&text)
{
	QTreeWidgetItem						*item;
	
	if(this->topLevelItemCount() < index)
	{
		return false;
	}
	
	item = this->topLevelItem(index);
	
	if(item)
	{
		item->setText(1,text);
		return true;
	}
	
	return false;
}

bool TOptionEditorWidget::Change_Option_Text(
	const int 							index,
	const int 							value)
{
	return this->Change_Option_Text(index,QString("%1").arg(value));
}

void TOptionEditorWidget::Enable_Option(
	const int 							index,
	const bool							state)
{
	QTreeWidgetItem						*item;
	Qt::ItemFlags						flags;
	
	if(index < 0)
	{
		return;
	}
	
	if(!(this->topLevelItemCount() < index))
	{
		item = this->topLevelItem(index);
		
		if(item)
		{
			flags = item->flags();
			
			if(state)
			{
				flags |= Qt::ItemIsEnabled;
			}
			else
			{
				flags &= ~Qt::ItemIsEnabled;
			}
			
			item->setFlags(flags);
		}
	}
}

void TOptionEditorWidget::Edit_List_Changed(
	int 								index,
	const QString 						&text)
{
	this->Change_Option_Text(index,text);
	emit Option_Combo_Item_Changed(index,text);
}

void TOptionEditorWidget::Path_Edit_Browse(
	const QString						default_path,
	const int							row,
	const int							column)
{
	QString								path;
	QTreeWidgetItem						*item;
	
	 path = QFileDialog::getExistingDirectory(
						 this,
						 QStringLiteral("Select a directory path"),
						 default_path,
						 QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks | QFileDialog::DontUseNativeDialog);

	if(!path.isEmpty())
	{
		if(!(path.endsWith("/")))
		{
			path += "/";
		}
		
		path = QDir::toNativeSeparators(path);
		
		item = this->topLevelItem(row);
		
		if(item)
		{
			item->setText(column,path);
		}
	}
}

TOptionItemDelegate::TOptionItemDelegate(
	const QObject						*parent)
:QItemDelegate(const_cast<QObject*>(parent)),
	d_active_row(0),
	d_active_column(0)
{
}

TOptionItemDelegate::~TOptionItemDelegate(void)
{

}

QWidget* TOptionItemDelegate::createEditor(
	QWidget								*parent,
	const QStyleOptionViewItem			&style_option_view_item,
    const QModelIndex 					&model_index) const
{
	Q_UNUSED(style_option_view_item);
	
	TPathEdit							*path_edit;
	QLineEdit							*line_edit;
	QDateTimeEdit						*date_time_edit;
	QComboBox							*list_combo;
	
    if(model_index.column() == 1)
    {
    	if(model_index.data(Qt::UserRole).canConvert(QVariant::Int))
    	{
			if(model_index.data(Qt::UserRole).toInt() == LINE_EDITOR)
			{
				line_edit = new QLineEdit(parent);
				line_edit->setFrame(false);
				line_edit->setAutoFillBackground(true);
				return line_edit;			
			}
			else if(model_index.data(Qt::UserRole).toInt() == PATH_EDITOR)
			{
				path_edit = new TPathEdit(parent);
				path_edit->setFrame(false);
				path_edit->setAutoFillBackground(true);
				
				path_edit->Set_Row(model_index.row());
				path_edit->Set_Column(model_index.column());
				connect(path_edit,&TPathEdit::Browse_Path,d_option_editor_widget,&TOptionEditorWidget::Path_Edit_Browse,Qt::QueuedConnection);

				return path_edit;
			}
			else if(model_index.data(Qt::UserRole).toInt() == DATETIME_EDITOR)
			{
				date_time_edit = new QDateTimeEdit(parent);
				date_time_edit->setFrame(false);
				date_time_edit->setAutoFillBackground(true);
				date_time_edit->setDisplayFormat(QString("ddd MMM dd hh:mm:ss yyyy AP"));
				return date_time_edit;
			}
    	}
    	else	// assumed to be a combo box and data is QStringList
    	{
    		if(model_index.data(Qt::UserRole).canConvert(QVariant::StringList))
    		{
				list_combo = new QComboBox(parent);
				list_combo->setFrame(false);
				list_combo->addItems(model_index.data(Qt::UserRole).toStringList());
				list_combo->setAutoFillBackground(true);
				
				connect(list_combo,static_cast<void (QComboBox::*)(const int)>(&QComboBox::currentIndexChanged),this,&TOptionItemDelegate::Combo_Item_Changed);
				return list_combo;
    		}
    	}
    }

    return 0;
}

void TOptionItemDelegate::commitAndCloseEditor()
{
	TPathEdit							*path_edit;
	QLineEdit							*line_edit;
	QComboBox							*list_combo;
	QDateTimeEdit						*date_time_edit;

	path_edit = qobject_cast<TPathEdit*>(sender());
	
	if(path_edit)
	{
		emit commitData(path_edit);
		emit closeEditor(path_edit);
		return;
	}
	
	line_edit = qobject_cast<QLineEdit*>(sender());
	
	if(line_edit)
	{
		emit commitData(line_edit);
		emit closeEditor(line_edit);
		return;
	}

	date_time_edit = qobject_cast<QDateTimeEdit*>(sender());
	
	if(date_time_edit)
	{
		emit commitData(date_time_edit);
		emit closeEditor(date_time_edit);
		return;
	}
	
	list_combo = qobject_cast<QComboBox*>(sender());
	
	if(list_combo)
	{
		emit commitData(list_combo);
		emit closeEditor(list_combo);
		return;
	}
}

void TOptionItemDelegate::Combo_Item_Changed(
	const int 							index)
{
	QString								text;
	QComboBox							*combo;
	
	combo = qobject_cast<QComboBox*>(this->sender());
	
	if(combo)
	{
		if(!(index < 0) && (index < combo->count()))
		{
			text = combo->itemText(index);
			emit List_Item_Changed(d_active_row,text);
		}
	}
}

void TOptionItemDelegate::setEditorData(
	QWidget 							*editor,
	const QModelIndex 					&model_index) const
{
	int									offset;
	bool								prev_state;
	
	d_active_row = model_index.row();
	d_active_column = model_index.column();
	
    if(model_index.data(Qt::UserRole).canConvert(QVariant::Int))	// one of the text editors
    {
		if(model_index.data(Qt::UserRole).toInt() == LINE_EDITOR)
		{
		    QLineEdit *line_edit = qobject_cast<QLineEdit *>(editor);
		    
		    if(line_edit)
		    {
		        line_edit->setText(model_index.model()->data(model_index, Qt::EditRole).toString());
		    }
		    
		}
		else if(model_index.data(Qt::UserRole).toInt() == PATH_EDITOR)
		{
			TPathEdit *path_edit = qobject_cast<TPathEdit *>(editor);
		    
		    if(path_edit)
		    {
		        path_edit->setText(model_index.model()->data(model_index, Qt::EditRole).toString());
		    }
		}

		else if(model_index.data(Qt::UserRole).toInt() == DATETIME_EDITOR)
		{
			QDateTimeEdit *date_time_edit = qobject_cast<QDateTimeEdit*>(editor);
		    
		    if(date_time_edit)
		    {
		    	date_time_edit->setDateTime(
		    						QDateTime::fromString(model_index.model()->data(model_index, Qt::EditRole).toString(),
		    						QString("ddd MMM dd hh:mm:ss yyyy AP")));
		    }
		}
		
    }
    else if(model_index.data(Qt::UserRole).canConvert(QVariant::StringList))	// assume the combobox
    {
		QComboBox *combo_box = qobject_cast<QComboBox *>(editor);
		    
		if(combo_box)
		{
			prev_state = combo_box->blockSignals(true);

			offset = combo_box->findText(model_index.model()->data(model_index, Qt::EditRole).toString());
			combo_box->setCurrentIndex(offset);

			combo_box->blockSignals(prev_state);
		}
    }
}

void TOptionItemDelegate::setModelData(
	QWidget 							*editor,
	QAbstractItemModel 					*item_model,
    const QModelIndex					&model_index) const
{
    if(model_index.data(Qt::UserRole).canConvert(QVariant::Int))	// one of the text editors
    {
		if(model_index.data(Qt::UserRole).toInt() == LINE_EDITOR)
		{
		    QLineEdit *line_edit = qobject_cast<QLineEdit *>(editor);
		    
		    if(line_edit)
		    {
        		item_model->setData(model_index, line_edit->text());
		    }
		    
		}
		else if(model_index.data(Qt::UserRole).toInt() == PATH_EDITOR)
		{
			TPathEdit *path_edit = qobject_cast<TPathEdit *>(editor);
		    
		    if(path_edit)
		    {
        		item_model->setData(model_index, path_edit->text());
		    }
		}
		else if(model_index.data(Qt::UserRole).toInt() == DATETIME_EDITOR)
		{
			QDateTimeEdit *date_time_edit = qobject_cast<QDateTimeEdit*>(editor);
		    
		    if(date_time_edit)
		    {
        		item_model->setData(model_index, date_time_edit->dateTime().toString(QString("ddd MMM dd hh:mm:ss yyyy AP")));
		    }
		}
    }
    else if(model_index.data(Qt::UserRole).canConvert(QVariant::StringList))	// assume the combobox
    {
		QComboBox *combo_box = qobject_cast<QComboBox *>(editor);
		    
		if(combo_box)
		{
        	item_model->setData(model_index, combo_box->currentText());
		}
    }
}

QSize TOptionItemDelegate::sizeHint(
	const QStyleOptionViewItem 			&style_option_view_item,
	const QModelIndex 					&index) const
{
	QSize								size;

	size = QItemDelegate::sizeHint(style_option_view_item,index);

	size.setHeight(size.height() + size.height()/4);

	return size;
}

