/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QElapsedTimer>
#include <cstring>
// #include <errno.h>
// #include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h> 
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <assert.h>

#include "serial.h"

static const int INVALID_HANDLE_VALUE = -1;

TSerial::TSerial(void)
{
	this->Intialize_Variables();
}

TSerial::~TSerial(void)
{
}

bool TSerial::isOpen(void) const
{
	return (d_comm_handle != INVALID_HANDLE_VALUE);
}

int TSerial::bytesAvailable(void) const
{
	int								bytes(0);

	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return bytes;
	}

	if(::ioctl(d_comm_handle, FIONREAD, &bytes) < 0)	// TIOCINQ?
	{
		d_last_error = QString("ioctl error: %1").arg(errno);
		bytes = -1;
	}

	return bytes;
}

int TSerial::read(
	char								*data,
	int								max_size) const
{
	int								read_size(0);
	
	assert(data);
	
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return static_cast<int>(read_size);
	}
	
	read_size = ::read(d_comm_handle, data, max_size);
	
	if(read_size < 0)
	{
		d_last_error = QString("read error: %1").arg(errno);
		return -1;
	}
	
	return read_size;
}

QString TSerial::portName(void) const
{
	return d_port_name;
}

QString TSerial::baudRate(void) const
{
	return QString("%1").arg(this->Get_Current_Baud_Rate());
}

QString TSerial::dataBits(void) const
{
	int									data_bits;
	
	if((d_termios.c_cflag & CS5) == CS5)
	{
		data_bits = 5;
	}
	if((d_termios.c_cflag & CS6) == CS6)
	{
		data_bits = 6;
	}
	if((d_termios.c_cflag & CS7) == CS7)
	{
		data_bits = 7;
	}
	if((d_termios.c_cflag & CS8) == CS8)
	{
		data_bits = 8;
	}
	else
	{
		data_bits = 8;
	}

	return QString("%1").arg(data_bits);
}

QString TSerial::parity(void) const
{
	if((d_termios.c_cflag & PARENB) == PARENB)
	{
		if((d_termios.c_cflag & PARODD) == PARODD)
		{
			return QStringLiteral("ODD");
		}
		
		return QStringLiteral("EVEN");
	}
	return QStringLiteral("NONE");
}

QString TSerial::stopBits(void) const
{
	if((d_termios.c_cflag & CSTOPB) == CSTOPB)
	{
		return QStringLiteral("2");
	}
	return QStringLiteral("1");
}

QString TSerial::flowControl(void) const
{
	if((d_termios.c_cflag & CRTSCTS) == CRTSCTS)
	{
		return QStringLiteral("HARDWARE");
	}
	else
	{
		if((d_termios.c_cflag & IXON) == IXON ||
		   (d_termios.c_cflag & IXOFF) == IXOFF ||
		   (d_termios.c_cflag & IXANY) == IXANY)
		{
			return QStringLiteral("XON/XOFF");
		}
	}

	return QStringLiteral("NONE");
}


bool TSerial::setPortName(
	const QString						&name)
{
	d_port_name = name;
	
	return (name.length() > 0);
}

bool TSerial::setBaudRate(
	const TLibSerialDeviceEnum::TBaudRate	baud_rate)
{
	speed_t								speed;

	d_baud_rate = baud_rate;
	
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return true;
	}
	
	speed = this->Convert_Baud(baud_rate);
	
	::cfsetispeed(&d_termios, speed);
	::cfsetospeed(&d_termios, speed);

	d_termios.c_cflag |= (CLOCAL | CREAD);

	return this->Update_Comm_Parameters();
}

bool TSerial::setDataBits(
	const TLibSerialDeviceEnum::TDataBits data_bits)
{
	d_data_bits = data_bits;
	
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return true;
	}
	
	switch(data_bits)
	{
		case TLibSerialDeviceEnum::SD_DATA_BITS_5:
			d_termios.c_cflag &= (~CSIZE);
			d_termios.c_cflag |= CS5;
			break;
			
		case TLibSerialDeviceEnum::SD_DATA_BITS_6:
			d_termios.c_cflag &= (~CSIZE);
			d_termios.c_cflag |= CS6;
			break;
			
		case TLibSerialDeviceEnum::SD_DATA_BITS_7:
			d_termios.c_cflag &= (~CSIZE);
			d_termios.c_cflag |= CS7;
			break;
			
		case TLibSerialDeviceEnum::SD_DATA_BITS_8:
			d_termios.c_cflag &= (~CSIZE);
			d_termios.c_cflag |= CS8;
			break;
	}
	
	return this->Update_Comm_Parameters();
}

bool TSerial::setParity(
	const TLibSerialDeviceEnum::TParity parity)
{
	d_parity = parity;
	
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return true;
	}
	
	switch(parity)
	{
		case TLibSerialDeviceEnum::SD_PARITY_NONE:
			d_termios.c_cflag &= (~PARENB);
			break;
			
		case TLibSerialDeviceEnum::SD_PARITY_ODD:
			d_termios.c_cflag |= (PARENB | PARODD);
			break;
			
		case TLibSerialDeviceEnum::SD_PARITY_EVEN:
			d_termios.c_cflag &= (~PARODD);
			d_termios.c_cflag |= PARENB;
			break;
	}
	
	
	return this->Update_Comm_Parameters();
}

bool TSerial::setStopBits(
	const TLibSerialDeviceEnum::TStopBits stop_bits)
{
	d_stop_bits = stop_bits;
	
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return true;
	}
	
	switch(stop_bits)
	{
		case TLibSerialDeviceEnum::SD_STOP_BITS_1:
			d_termios.c_cflag &= (~CSTOPB);
			break;
			
		case TLibSerialDeviceEnum::SD_STOP_BITS_1_5:	// not supported, default to 1
			d_termios.c_cflag &= (~CSTOPB);
			break;
			
		case TLibSerialDeviceEnum::SD_STOP_BITS_2:
			d_termios.c_cflag |= CSTOPB;
			break;
	}
	
	return this->Update_Comm_Parameters();
}

bool TSerial::setFlowControl(
	const TLibSerialDeviceEnum::TFlowControl flow_control)
{
	d_flow_control = flow_control;
	
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return true;
	}
	
	switch(flow_control)
	{
		case TLibSerialDeviceEnum::SD_FLOW_NONE:
			d_termios.c_cflag &= (~CRTSCTS);
			d_termios.c_iflag &= (~(IXON | IXOFF | IXANY));
			break;
			
		case TLibSerialDeviceEnum::SD_FLOW_HARDWARE:
			d_termios.c_cflag |= CRTSCTS;
			d_termios.c_iflag &= (~(IXON | IXOFF | IXANY));
			break;
			
		case TLibSerialDeviceEnum::SD_FLOW_XON_XOFF:
			d_termios.c_cflag &= (~CRTSCTS);
			d_termios.c_iflag |= (IXON | IXOFF | IXANY);
			break;
	}
	
	return this->Update_Comm_Parameters();
}

bool TSerial::setDataTerminalReady(
	const bool							state)
{
	int									line_mask(0);

	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return false;
	}
	
	if (-1 == ioctl(d_comm_handle, TIOCMGET, &line_mask))
	{
		return false;
	}
	
	if(state)
	{
		line_mask |= TIOCM_DTR;
	}
	else
	{
		line_mask &= (~TIOCM_DTR);
	}
	
	if(::ioctl(d_comm_handle, TIOCMSET, &line_mask) < 0)
	{
		d_last_error = QString("ioctl error: %1").arg(errno);
		return false;
	}
	
	return true;
}

bool TSerial::setRequestToSend(
	const bool							state)
{
	int									line_mask(0);
	
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return false;
	}
	
	if(::ioctl(d_comm_handle, TIOCMGET, &line_mask) < 0)
	{
		d_last_error = QString("ioctl error: %1").arg(errno);
		return false;
	}
	
	if(state)
	{
		line_mask |= TIOCM_RTS;
	}
	else
	{
		line_mask &= (~TIOCM_RTS);
	}
	
	if (::ioctl(d_comm_handle, TIOCMSET, &line_mask) < 0)
	{
		d_last_error = QString("ioctl error: %1").arg(errno);
		return false;
	}
	
	return true;
}

int TSerial::write(
	const char							*data,
	int									max_size,
	const bool							block_until_written)
{
	QElapsedTimer						timer;
	int									written_size(0);
	int									block_write_size;
	int									block_written_size(0);
	unsigned int						data_offset;
	unsigned int						sleep_time;
	unsigned int						expected_write_time;
	unsigned int						actual_write_time;
	unsigned int						current_baud_rate;
	unsigned int						current_byte_size;
	double								byte_write_time;
	unsigned int						frame_bit_count;
	const int							WRITE_BLOCK_SIZE(8);	// 8 bytes transferred per interval
//	int									written_size(0);
	
	assert(data);
	
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return static_cast<int>(written_size);
	}
	
	if(block_until_written)
	{
		// calculated write time
		
		// find size of transmission frame
		// 1 start bit + data bits + parity + stop
		
		frame_bit_count = 1;
		
		if((d_termios.c_cflag & CS5) == CS5)
		{
			current_byte_size = 5;
		}
		if((d_termios.c_cflag & CS6) == CS6)
		{
			current_byte_size = 6;
		}
		if((d_termios.c_cflag & CS7) == CS7)
		{
			current_byte_size = 7;
		}
		if((d_termios.c_cflag & CS8) == CS8)
		{
			current_byte_size = 8;
		}
		else
		{
			current_byte_size = 8;
		}
		
		frame_bit_count += current_byte_size;
		
		if((d_termios.c_cflag & PARENB) == PARENB) // check if parity enabled
		{
			frame_bit_count++;
		}
		
		if((d_termios.c_cflag & CSTOPB) == CSTOPB) // second stop bit is enabled
		{
			frame_bit_count += 2;
		}
		else
		{
			frame_bit_count++;
		}
		
		current_baud_rate = Get_Current_Baud_Rate();
		
		if(current_baud_rate == 0)			// this should never be zero but just in case
		{
			current_baud_rate = 8;
		}
		
		byte_write_time = 1000 / static_cast<double>(current_baud_rate);
		byte_write_time *= static_cast<double>(frame_bit_count);
		
		data_offset = 0;
		written_size = 0;
		
		timer.start();
		
		do
		{
			block_write_size = max_size - data_offset;
			
			if(block_write_size > WRITE_BLOCK_SIZE)
			{
				block_write_size = WRITE_BLOCK_SIZE;
			}
			
			expected_write_time = static_cast<unsigned int>(0.5 + byte_write_time * static_cast<double>(block_write_size));
			
			block_written_size = ::write(d_comm_handle, data + data_offset, block_write_size);
			
			if(block_written_size < 0)
			{
				d_last_error = QString("write error: %1").arg(errno);
				return -1;
			}
			
			written_size += block_written_size;
			
			data_offset += static_cast<unsigned int>(block_write_size);
			
			// block for time expected to write block of data
			actual_write_time = timer.restart();
			
//			qWarning(QString("expected write time: %1 actual %2").arg(expected_write_time).arg(actual_write_time).toLatin1());
			
			
			if(expected_write_time > actual_write_time)
			{
				sleep_time = 1000 * expected_write_time - actual_write_time;
				::usleep(sleep_time);
			}
			
		} while (written_size < max_size);
		
		return static_cast<int>(written_size);
	}
	
	// non blocking mode.  May be blocked by the OS but not by me
	written_size = ::write(d_comm_handle, data, max_size);
	
	if(written_size < 0)
	{
		d_last_error = QString("write error: %1").arg(errno);
		return -1;
	}

	return static_cast<int>(written_size);

}

bool TSerial::purge(void)
{
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		d_last_error = QStringLiteral("Device not open.");
		return false;
	}
	
	if(::tcflush(d_comm_handle, TCIOFLUSH) < 0)
	{
		d_last_error = QString("tcflush error: %1").arg(errno);
		return false;
	}
	
	return true;
}

bool TSerial::open()
{
	d_comm_handle = ::open(d_port_name.toUtf8().constData(), O_RDWR | O_NOCTTY | O_NDELAY);
	
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		d_last_error = QString("open error: %1").arg(errno);
		return false;
	}

	::fcntl(d_comm_handle, F_SETFL, FNDELAY);	// return immediately on read if no data exists
//	::fcntl(d_comm_handle, F_SETFL, 0);			// blocking I/O
	::ioctl(d_comm_handle, TIOCEXCL);				// user exclusive mode

//	fcntl(d_comm_handle, F_SETFL, 0);

	this->purge();
	
	// get original and working copy of termios structure
	if(::tcgetattr(d_comm_handle, &d_termios_original) < 0)
	{
		d_last_error = QString("tcgetattr error: %1").arg(errno);
		return false;
	}

	memcpy(reinterpret_cast<void *>(&d_termios_original),
		   reinterpret_cast<void *>(&d_termios),
		   sizeof(termios));

	
	// set all remaining parameters
	if(!this->setBaudRate(d_baud_rate))
	{
		this->close();
		return false;
	}
	
	if(!this->setDataBits(d_data_bits))
	{
		this->close();
		return false;
	}
	
	if(!this->setParity(d_parity))
	{
		this->close();
		return false;
	}
	
	if(!this->setStopBits(d_stop_bits))
	{
		this->close();
		return false;
	}
	
	if(!this->setFlowControl(d_flow_control))
	{
		this->close();
		return false;
	}

	return true;
}

void TSerial::close(void)
{
	if(d_comm_handle != INVALID_HANDLE_VALUE)
	{
		::tcsetattr(d_comm_handle, TCSANOW, &d_termios_original);
		
		::close(d_comm_handle);
		
		d_comm_handle = INVALID_HANDLE_VALUE;
	}
}

speed_t TSerial::Convert_Baud(
	const TLibSerialDeviceEnum::TBaudRate	&baud_rate) const
{
	switch(baud_rate)
	{
		case TLibSerialDeviceEnum::SD_BAUD_110:
#ifdef B110
			return B110;
#endif
			break;
		case TLibSerialDeviceEnum::SD_BAUD_300:
#ifdef B300
			return B300;
#endif
			break;
		case TLibSerialDeviceEnum::SD_BAUD_600:
#ifdef B600
			return B600;
#endif
			break;
		case TLibSerialDeviceEnum::SD_BAUD_1200:
#ifdef B1200
			return B1200;
#endif
			break;
		case TLibSerialDeviceEnum::SD_BAUD_2400:
#ifdef B2400
			return B2400;
#endif
			break;
		case TLibSerialDeviceEnum::SD_BAUD_4800:
#ifdef B4800
			return B4800;
#endif
			break;
		case TLibSerialDeviceEnum::SD_BAUD_9600:
#ifdef B9600
			return B9600;
#endif
			break;
		case TLibSerialDeviceEnum::SD_BAUD_19200:
#ifdef B19200
			return B19200;
#endif
			break;
		case TLibSerialDeviceEnum::SD_BAUD_38400:
#ifdef B38400
			return B38400;
#endif
			break;
		case TLibSerialDeviceEnum::SD_BAUD_57600:
#ifdef B57600
			return B57600;
#endif
			break;
		case TLibSerialDeviceEnum::SD_BAUD_115200:
#ifdef B115200
			return B115200;
#endif
			break;

	}
	
	return 0;
}

unsigned int TSerial::Get_Current_Baud_Rate(void) const
{
	// assumption that the current baud rate is the same
	// as the d_baud_rate parameter
	switch(d_baud_rate)
	{
		case TLibSerialDeviceEnum::SD_BAUD_110:
			return 110;
		case TLibSerialDeviceEnum::SD_BAUD_300:
			return 300;
		case TLibSerialDeviceEnum::SD_BAUD_600:
			return 600;
		case TLibSerialDeviceEnum::SD_BAUD_1200:
			return 1200;
		case TLibSerialDeviceEnum::SD_BAUD_2400:
			return 2400;
		case TLibSerialDeviceEnum::SD_BAUD_4800:
			return 4800;
		case TLibSerialDeviceEnum::SD_BAUD_9600:
			return 9600;
		case TLibSerialDeviceEnum::SD_BAUD_19200:
			return 19200;
		case TLibSerialDeviceEnum::SD_BAUD_38400:
			return 38400;
		case TLibSerialDeviceEnum::SD_BAUD_57600:
			return 57600;
		case TLibSerialDeviceEnum::SD_BAUD_115200:
			return 115200;
			
	}
	
	return 0;
}


bool TSerial::Update_Comm_Parameters(void)
{
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		d_last_error = QStringLiteral("Device not open.");
		return false;
	}
	
	if(::tcsetattr(d_comm_handle, TCSANOW, &d_termios) < 0)
	{
		d_last_error = QString("tcflush error: %1").arg(errno);
		return false;
	}
	
	return true;
}

void TSerial::Intialize_Variables(void)
{
	memset(reinterpret_cast<void *>(&d_termios), 0, sizeof(d_termios));
	memset(reinterpret_cast<void *>(&d_termios_original), 0, sizeof(d_termios));
	
	d_comm_handle = INVALID_HANDLE_VALUE;
	
	// set defaults
	d_baud_rate = TLibSerialDeviceEnum::SD_BAUD_9600;
	d_data_bits = TLibSerialDeviceEnum::SD_DATA_BITS_8;
	d_parity = TLibSerialDeviceEnum::SD_PARITY_NONE;
	d_stop_bits = TLibSerialDeviceEnum::SD_STOP_BITS_1;
	d_flow_control = TLibSerialDeviceEnum::SD_FLOW_NONE;
}
