/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QElapsedTimer>

#include <assert.h>
#include "serial.h"

TSerial::TSerial(void)
{
	this->Intialize_Variables();
}

TSerial::~TSerial(void)
{
}

bool TSerial::isOpen(void) const
{
	return (d_comm_handle != INVALID_HANDLE_VALUE);
}

int TSerial::bytesAvailable(void) const
{
	DWORD								error(0);
	COMSTAT								comstat;
	int								bytes(0);
	
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return bytes;
	}
	
	memset(reinterpret_cast<void *>(&comstat), 0, sizeof(COMSTAT));

	if(FALSE == ::ClearCommError(d_comm_handle, &error, &comstat))
	{
		d_last_error = QString("ClearCommError error: %1").arg(errno);
		return -1;
	}
	
	if(error)
	{
		switch(error)
		{
			case CE_BREAK:
				d_last_error = QStringLiteral("ClearCommError RX Break");
				return -1;
				
			case CE_FRAME:
				d_last_error = QStringLiteral("ClearCommError Framing Error");
				return -1;
				
			case CE_OVERRUN:
				d_last_error = QStringLiteral("ClearCommError Overrun Error");
				return -1;
				
			case CE_RXOVER:
				d_last_error = QStringLiteral("ClearCommError RX Buffer Overflow");
				return -1;
				
			case CE_RXPARITY:
				d_last_error = QStringLiteral("ClearCommError RX Parity Error");
				return -1;
				
			default:
				d_last_error = QString("ClearCommError error: %1").arg(GetLastError());
				return -1;
				
		}
	}
	
	
	return static_cast<int>(comstat.cbInQue);
}

int TSerial::read(
	char								*data,
	int									max_size) const
{
	DWORD								read_size(0);

	assert(data);
	
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return static_cast<int>(read_size);
	}
	
	if(FALSE != ::ReadFile(d_comm_handle, data, max_size, &read_size, NULL))
	{
		return static_cast<int>(read_size);
	}
	
	d_last_error = QString("ReadFile error: %1").arg(GetLastError());
	return -1;
}

QString TSerial::portName(void) const
{
	return d_port_name;
}

QString TSerial::baudRate(void) const
{
	return QString("%1").arg(this->Convert_Baud(d_baud_rate));
}

QString TSerial::dataBits(void) const
{
	return QString("%1").arg(d_device_control_block.ByteSize);
}

QString TSerial::parity(void) const
{
	if(d_device_control_block.Parity == ODDPARITY)
	{
		return QStringLiteral("ODD");
	}
	else if(d_device_control_block.Parity == EVENPARITY)
	{
		return QStringLiteral("EVEN");
	}
	return QStringLiteral("NONE");
}

QString TSerial::stopBits(void) const
{
	if(d_device_control_block.StopBits == ONESTOPBIT)
	{
		return QStringLiteral("1");
	}
	else if(d_device_control_block.StopBits == ONE5STOPBITS)
	{
		return QStringLiteral("1.5");
	}
	return QStringLiteral("2");
}

QString TSerial::flowControl(void) const
{
	if(d_device_control_block.fRtsControl == RTS_CONTROL_HANDSHAKE)
	{
		return QStringLiteral("HARDWARE");
	}
	else if(d_device_control_block.fRtsControl == RTS_CONTROL_DISABLE &&
			d_device_control_block.fInX == TRUE &&
			d_device_control_block.fOutX == TRUE)
	{
		return QStringLiteral("XON/XOFF");
	}
	
	return QStringLiteral("NONE");
}

bool TSerial::setPortName(
	const QString						&name)
{
	d_port_name = name;
	
	return (name.length() > 0);
}

bool TSerial::setBaudRate(
	const TLibSerialDeviceEnum::TBaudRate	baud_rate)
{
	DWORD								speed;
	
	d_baud_rate = baud_rate;

	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return true;
	}
	
	speed = this->Convert_Baud(baud_rate);
	
	d_device_control_block.BaudRate = speed;
	
	return this->Update_Comm_Parameters();
}

bool TSerial::setDataBits(
	const TLibSerialDeviceEnum::TDataBits data_bits)
{
	d_data_bits = data_bits;
	
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return true;
	}

	switch(data_bits)
	{
		case TLibSerialDeviceEnum::SD_DATA_BITS_5:
			d_device_control_block.ByteSize = 5;
			break;
			
		case TLibSerialDeviceEnum::SD_DATA_BITS_6:
			d_device_control_block.ByteSize = 6;
			break;
			
		case TLibSerialDeviceEnum::SD_DATA_BITS_7:
			d_device_control_block.ByteSize = 7;
			break;
			
		case TLibSerialDeviceEnum::SD_DATA_BITS_8:
			d_device_control_block.ByteSize = 8;
			break;
	}

	return this->Update_Comm_Parameters();
	
}

bool TSerial::setParity(
	const TLibSerialDeviceEnum::TParity parity)
{
	d_parity = parity;
	
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return true;
	}

	switch(parity)
	{
		case TLibSerialDeviceEnum::SD_PARITY_NONE:
			d_device_control_block.Parity = NOPARITY;
			d_device_control_block.fParity = FALSE;
			break;
			
		case TLibSerialDeviceEnum::SD_PARITY_ODD:
			d_device_control_block.Parity = ODDPARITY;
			d_device_control_block.fParity = TRUE;
			break;
			
		case TLibSerialDeviceEnum::SD_PARITY_EVEN:
			d_device_control_block.Parity = EVENPARITY;
			d_device_control_block.fParity = TRUE;
			break;
	}


	return this->Update_Comm_Parameters();
}

bool TSerial::setStopBits(
	const TLibSerialDeviceEnum::TStopBits stop_bits)
{
	d_stop_bits = stop_bits;
	
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return true;
	}
	
	switch(stop_bits)
	{
		case TLibSerialDeviceEnum::SD_STOP_BITS_1:
			d_device_control_block.StopBits = ONESTOPBIT;
			break;
			
		case TLibSerialDeviceEnum::SD_STOP_BITS_1_5:
			d_device_control_block.StopBits = ONE5STOPBITS;
			break;
			
		case TLibSerialDeviceEnum::SD_STOP_BITS_2:
			d_device_control_block.StopBits = TWOSTOPBITS;
			break;
	}

	return this->Update_Comm_Parameters();
}

bool TSerial::setFlowControl(
	const TLibSerialDeviceEnum::TFlowControl flow_control)
{
	d_flow_control = flow_control;
	
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return true;
	}

	switch(flow_control)
	{
		case TLibSerialDeviceEnum::SD_FLOW_NONE:
			d_device_control_block.fRtsControl = RTS_CONTROL_DISABLE;
			d_device_control_block.fOutxCtsFlow = FALSE;
			d_device_control_block.fInX = FALSE;
			d_device_control_block.fOutX = FALSE;
			break;
			
		case TLibSerialDeviceEnum::SD_FLOW_HARDWARE:
			d_device_control_block.fRtsControl = RTS_CONTROL_HANDSHAKE;
			d_device_control_block.fOutxCtsFlow = TRUE;
			d_device_control_block.fInX = FALSE;
			d_device_control_block.fOutX = FALSE;
			break;
			
		case TLibSerialDeviceEnum::SD_FLOW_XON_XOFF:
			d_device_control_block.fRtsControl = RTS_CONTROL_DISABLE;
			d_device_control_block.fOutxCtsFlow = FALSE;
			d_device_control_block.fInX = TRUE;
			d_device_control_block.fOutX = TRUE;
			break;
	}

	return this->Update_Comm_Parameters();
}

bool TSerial::setDataTerminalReady(
	const bool							state)
{
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return false;
	}

	if(TRUE != ::EscapeCommFunction(d_comm_handle, state ? SETDTR : CLRDTR))
	{
		d_last_error = QString("EscapeCommFunction error: %1").arg(GetLastError());
		return false;
	}
	
	return true;
}

bool TSerial::setRequestToSend(
	const bool							state)
{
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return false;
	}
	
	if(TRUE != ::EscapeCommFunction(d_comm_handle, state ? SETRTS : CLRRTS))
	{
		d_last_error = QString("EscapeCommFunction error: %1").arg(GetLastError());
		return false;
	}
	
	return true;
}

int TSerial::write(
	const char							*data,
	int									max_size,
	const bool							block_until_written)
{
	QElapsedTimer						timer;
	DWORD								written_size(0);
	DWORD								expected_write_time;
	DWORD								actual_write_time;
	DWORD								block_write_size;
	DWORD								block_written_size(0);
	DWORD								data_offset;
	DWORD								sleep_time;
	double								byte_write_time;
	DWORD								frame_bit_count;
	const int							WRITE_BLOCK_SIZE(8);	// 8 bytes transferred per interval
	
	assert(data);
	
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		return static_cast<int>(written_size);
	}
	
	if(block_until_written)
	{
		// calculated write time
		
		// find size of transmission frame
		// 1 start bit + data bits + parity + stop
		
		frame_bit_count = 1;
		frame_bit_count += d_device_control_block.ByteSize;
		
		if(d_device_control_block.Parity != NOPARITY)
		{
			frame_bit_count++;
		}
		
		if(d_device_control_block.StopBits == ONESTOPBIT)
		{
			frame_bit_count++;
		}
		else
		{
			frame_bit_count += 2;
		}
		
		byte_write_time = 1000 / static_cast<double>(d_device_control_block.BaudRate);
		byte_write_time *= static_cast<double>(frame_bit_count);
		
		data_offset = 0;
		written_size = 0;
		
		timer.start();
		
		do
		{
			block_write_size = max_size - data_offset;
			
			if(block_write_size > WRITE_BLOCK_SIZE)
			{
				block_write_size = WRITE_BLOCK_SIZE;
			}
			
			expected_write_time = static_cast<DWORD>(0.5 + byte_write_time * static_cast<double>(block_write_size));
			
			if(TRUE != ::WriteFile(d_comm_handle, data + data_offset, block_write_size, &block_written_size, NULL))
			{
				d_last_error = QString("WriteFile error: %1").arg(GetLastError());
				return -1;
			}
			
			written_size += block_written_size;
			
			data_offset += block_write_size;
			
			// block for time expected to write block of data
			actual_write_time = timer.restart();
			
			if(expected_write_time > actual_write_time)
			{
				sleep_time = expected_write_time - actual_write_time;
				::Sleep(sleep_time);
			}
			
		} while (written_size < static_cast<DWORD>(max_size));
			
		return static_cast<int>(written_size);
	}
	
	// non blocking mode.  may be blocked by the OS but not by me
	if(TRUE != ::WriteFile(d_comm_handle, data, max_size, &written_size, NULL))
	{
		d_last_error = QString("WriteFile error: %1").arg(GetLastError());
		return -1;
	}
	
	return static_cast<int>(written_size);

}

bool TSerial::purge(void)
{
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		d_last_error = QStringLiteral("Device not open.");
		return false;
	}
	
	if(TRUE != ::PurgeComm(d_comm_handle, PURGE_TXABORT | PURGE_RXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR))
	{
		d_last_error = QString("PurgeComm error: %1").arg(GetLastError());
		return false;
	}
	
	return true;
}

bool TSerial::open()
{
	QString								port_name;
//	QByteArray							port_name_unicode;

	port_name = "\\\\.\\" + d_port_name;
//	port_name_unicode = QByteArray((const char *)port_name.utf16(), port_name.size() * 2 + 1);	// include terminator char

	d_comm_handle = ::CreateFile(reinterpret_cast<const wchar_t*>(port_name.utf16()),
					   GENERIC_READ | GENERIC_WRITE,
					   0,
					   NULL,
					   OPEN_EXISTING,
					   0,//FILE_FLAG_OVERLAPPED,
					   NULL);
	
	if(INVALID_HANDLE_VALUE == d_comm_handle)
	{
		d_last_error = QString("open error: %1").arg(GetLastError());
		return false;
	}
	
	this->purge();
	
	// get original and make a working copy of the device control block data DCB
	memset(reinterpret_cast<void *>(&d_device_control_block_original), 0, sizeof(DCB));
	d_device_control_block_original.DCBlength = sizeof(DCB);

	if(!::GetCommState(d_comm_handle, &d_device_control_block_original))
	{
		d_last_error = QString("GetCommState error: %1").arg(GetLastError());
		this->close();
		return false;
	}
	
	memcpy(reinterpret_cast<void *>(&d_device_control_block),
		   reinterpret_cast<void *>(&d_device_control_block_original),
		   sizeof(DCB));
	
	// get original and make a working copy of the comm timeout data COMMTIMEOUTS
	if (!::GetCommTimeouts(d_comm_handle, &d_comm_timeouts_original))
	{
		d_last_error = QString("GetCommTimeouts error: %1").arg(GetLastError());
		this->close();
		return false;
	}
	
	memcpy(reinterpret_cast<void *>(&d_comm_timeouts),
		   reinterpret_cast<void *>(&d_comm_timeouts_original),
		   sizeof(COMMTIMEOUTS));
	
	d_comm_timeouts.ReadIntervalTimeout = MAXDWORD;
	d_comm_timeouts.ReadTotalTimeoutMultiplier = 0;
	d_comm_timeouts.ReadTotalTimeoutConstant = 0;
	
	// set to blocking write
	d_comm_timeouts.WriteTotalTimeoutMultiplier = 0;
	d_comm_timeouts.WriteTotalTimeoutConstant = 0;
		
	if(!::SetCommTimeouts(d_comm_handle, &d_comm_timeouts))
	{
		this->close();
		return false;
	}
	
	if(!this->setBaudRate(d_baud_rate))
	{
		this->close();
		return false;
	}
	
	if(!this->setDataBits(d_data_bits))
	{
		this->close();
		return false;
	}
	
	if(!this->setParity(d_parity))
	{
		this->close();
		return false;
	}
	
	if(!this->setStopBits(d_stop_bits))
	{
		this->close();
		return false;
	}
	
	if(!this->setFlowControl(d_flow_control))
	{
		this->close();
		return false;
	}
	
	return true;
}

void TSerial::close(void)
{
	if(d_comm_handle != INVALID_HANDLE_VALUE)
	{
		// restore old settings
		::SetCommState(d_comm_handle, &d_device_control_block);
		::SetCommTimeouts(d_comm_handle, &d_comm_timeouts_original);
		
		::CloseHandle(d_comm_handle);
		
		d_comm_handle = INVALID_HANDLE_VALUE;
	}
}

DWORD TSerial::Convert_Baud(
	const TLibSerialDeviceEnum::TBaudRate &baud_rate) const
{
	switch(baud_rate)
	{
		case TLibSerialDeviceEnum::SD_BAUD_110:
			return 110;
			
		case TLibSerialDeviceEnum::SD_BAUD_300:
			return 300;
			
		case TLibSerialDeviceEnum::SD_BAUD_600:
			return 600;
			
		case TLibSerialDeviceEnum::SD_BAUD_1200:
			return 1200;
			
		case TLibSerialDeviceEnum::SD_BAUD_2400:
			return 2400;
			
		case TLibSerialDeviceEnum::SD_BAUD_4800:
			return 4800;
			
		case TLibSerialDeviceEnum::SD_BAUD_9600:
			return 9600;
			
		case TLibSerialDeviceEnum::SD_BAUD_19200:
			return 19200;
			
		case TLibSerialDeviceEnum::SD_BAUD_38400:
			return 38400;
			
		case TLibSerialDeviceEnum::SD_BAUD_57600:
			return 57600;
			
		case TLibSerialDeviceEnum::SD_BAUD_115200:
			return 115200;
	}
	
	return 0;
}

bool TSerial::Update_Comm_Parameters(void)
{
	if(d_comm_handle == INVALID_HANDLE_VALUE)
	{
		d_last_error = QStringLiteral("Device not open.");
		return false;
	}
	
	if(TRUE != ::SetCommState(d_comm_handle, &d_device_control_block))
	{
		d_last_error = QString("SetCommState error: %1").arg(GetLastError());
		return false;
	}
	
	return true;
}

void TSerial::Intialize_Variables(void)
{
	memset(reinterpret_cast<void *>(&d_device_control_block), 0, sizeof(DCB));
	memset(reinterpret_cast<void *>(&d_device_control_block_original), 0, sizeof(DCB));
	
	memset(reinterpret_cast<void *>(&d_comm_timeouts), 0, sizeof(COMMTIMEOUTS));
	memset(reinterpret_cast<void *>(&d_comm_timeouts_original), 0, sizeof(COMMTIMEOUTS));
	
	d_device_control_block.DCBlength = sizeof(DCB);
	d_device_control_block_original.DCBlength = sizeof(DCB);
	
	d_comm_handle = INVALID_HANDLE_VALUE;
	
	// set defaults
	d_baud_rate = TLibSerialDeviceEnum::SD_BAUD_9600;
	d_data_bits = TLibSerialDeviceEnum::SD_DATA_BITS_8;
	d_parity = TLibSerialDeviceEnum::SD_PARITY_NONE;
	d_stop_bits = TLibSerialDeviceEnum::SD_STOP_BITS_1;
	d_flow_control = TLibSerialDeviceEnum::SD_FLOW_NONE;
}

