/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include "command_feature.h"

TCommandFeature::TCommandFeature(
	const int							memory_id)
{
	d_memory_id = memory_id;
	
	d_display_format = TCommandFeature::FORMAT_DEFAULT;
	
	d_nominal_ijk.Set(0.0,0.0,1.0);
	d_actual_ijk.Set(0.0,0.0,1.0);
	d_nominal_projection_axis.Set(0.0,0.0,1.0);
	d_touch_approach_vector.Set(0.0,0.0,1.0);
	
	d_nominal_diameter = 0;
	d_actual_diameter = 0;
	d_form_error = 0;

	d_nominal_diameter_type = TCommandFeature::DIAMETER_TYPE_NONE;
	d_actual_diameter_type = TCommandFeature::DIAMETER_TYPE_NONE;
}

TCommandFeature::~TCommandFeature(void)
{
}

QString TCommandFeature::Feature_Item_Text(
	const TReferenceSystem 				&refsys)
{
	QString								line;
	TVector3							pnt;
	TVector3							vec;
	
	if(d_display_format == TCommandFeature::FORMAT_NONE)
	{
		return line;
	}

	//           1         2         3         4         5         6         7         8
	// 012345678901234567890123456789012345678901234567890123456789012345678901234567890
	// xxxxxx +99999.9999 +99999.9999 +99999.9999 +99999.9999 +99999.9999 +99999.9999
	//  name |     x     |     y     |     z     |
	
	if(d_memory_id < 10)
	{
		line = QString("Memory[%1]   ").arg(d_memory_id);
	}
	else if(d_memory_id < 100)
	{
		line = QString("Memory[%1]  ").arg(d_memory_id);
	}
	else if(d_memory_id < 1000)
	{
		line = QString("Memory[%1] ").arg(d_memory_id);
	}
	else
	{
		line = QString("Memory[%1]").arg(d_memory_id);
	}
	
	pnt = refsys.FromWorld(d_actual_xyz);
	vec = refsys.AxisFromWorld(d_actual_ijk);

	switch(d_display_format)
	{
		case TCommandFeature::FORMAT_NONE:
		case TCommandFeature::FORMAT_DEFAULT:	// Defined by user
			break;
			
		case TCommandFeature::FORMAT_XYZ:
			
			line += QString("%1").arg(pnt.x,12,'f',4);
			line += QString("%1").arg(pnt.y,12,'f',4);
			line += QString("%1").arg(pnt.z,12,'f',4);

			break;

		case TCommandFeature::FORMAT_XYZIJK:
									
			line += QString("%1").arg(pnt.x,12,'f',4);
			line += QString("%1").arg(pnt.y,12,'f',4);
			line += QString("%1").arg(pnt.z,12,'f',4);

			line += QString("%1").arg(vec.i,12,'f',8);
			line += QString("%1").arg(vec.j,12,'f',8);
			line += QString("%1").arg(vec.k,12,'f',8);

			break;
			
		case TCommandFeature::FORMAT_XYZD:
						
			line += QString("%1").arg(pnt.x,12,'f',4);
			line += QString("%1").arg(pnt.y,12,'f',4);
			line += QString("%1").arg(pnt.z,12,'f',4);

			line += QString("%1").arg(d_actual_diameter,12,'f',4);
			
			break;
			
		case TCommandFeature::FORMAT_XYZIJKD:
			
			line += QString("%1").arg(pnt.x,12,'f',4);
			line += QString("%1").arg(pnt.y,12,'f',4);
			line += QString("%1").arg(pnt.z,12,'f',4);

			line += QString("%1").arg(vec.i,12,'f',8);
			line += QString("%1").arg(vec.j,12,'f',8);
			line += QString("%1").arg(vec.k,12,'f',8);
			line += QString("%1").arg(d_actual_diameter,12,'f',4);
			
			break;
			
		case TCommandFeature::FORMAT_XYZDF:
			
			line += QString("%1").arg(pnt.x,12,'f',4);
			line += QString("%1").arg(pnt.y,12,'f',4);
			line += QString("%1").arg(pnt.z,12,'f',4);

			line += QString("%1").arg(d_actual_diameter,12,'f',4);
			line += QString("%1").arg(d_form_error,12,'f',4);
			
			break;
	}

	return line;
}

bool TCommandFeature::Waiting_For_Measured_Points(void) const
{
	int									nominal_point_count(0);
	int									actual_point_count(0);
	
	nominal_point_count = this->Nominal_Point_Count();
	actual_point_count = this->Actual_Point_Count();
	
	if(actual_point_count < nominal_point_count)
	{
		return true;
	}
	
	return false;
}

int TCommandFeature::Nominal_Point_Count(void) const
{
	std::vector<TPathPoint>::const_iterator iter;
	int									nominal_point_count(0);

	for(iter = d_nominal_points.begin();iter != d_nominal_points.end();++iter)
	{
		if((*iter).Type() == TPathPoint::TOUCH_POINT)
		{
			++nominal_point_count;
		}
	}
	
	return nominal_point_count;
}

int TCommandFeature::Actual_Point_Count(void) const
{
	std::vector<TPathPoint>::const_iterator iter;
	int									actual_point_count(0);
	
	for(iter = d_actual_points.begin();iter != d_actual_points.end();++iter)
	{
		if((*iter).Type() == TPathPoint::TOUCH_POINT)
		{
			++actual_point_count;
		}
	}
	
	return actual_point_count;
}

std::vector<TVector3> TCommandFeature::Touch_Points(void) const
{
	std::vector<TPathPoint>::const_iterator iter;
	std::vector<TVector3>				pnts;
	
	for(iter = d_actual_points.begin();iter != d_actual_points.end();++iter)
	{
		if((*iter).Type() == TPathPoint::TOUCH_POINT)
		{
			pnts.push_back((*iter).XYZ());
		}
	}
	
	return pnts;
}

QString TCommandFeature::Output_Title_XYZ(void)
{
	return QStringLiteral("Name               X           Y           Z\n------------|-----------|-----------|-----------");
}

QString TCommandFeature::Output_Title_XYZIJK(void)
{
	return QStringLiteral("Name               X           Y           Z           I            J           K\n------------|-----------|-----------|-----------|-----------|-----------|-----------");
}

QString TCommandFeature::Output_Title_XYZD(void)
{
	return QStringLiteral("Name              X           Y           Z           D\n------------|-----------|-----------|-----------|-----------");
}

QString TCommandFeature::Output_Title_XYZIJKD(void)
{
	return QStringLiteral("Name               X           Y           Z           I            J           K          D\n------------|-----------|-----------|-----------|-----------|-----------|-----------|-----------");
}

QString TCommandFeature::Output_Title_XYZDF(void)
{
	return QStringLiteral("Name              X           Y           Z           D           F\n------------|-----------|-----------|-----------|-----------|-----------");
}

void TCommandFeature::Reset_Nominal(void)
{
	d_nominal_points.clear();
}

void TCommandFeature::Reset_Actual(void)
{
	d_actual_points.clear();
}

void TCommandFeature::Add_Path_Point(
	const TPathPoint					&pnt)
{
	d_nominal_points.push_back(pnt);
}

void TCommandFeature::Add_Path_Move_Point(
	const TVector3						&pnt)
{
	TPathPoint							path_point;
	
	path_point.Set_XYZ(pnt);
	path_point.Set_Type(TPathPoint::MOVE_POINT);
	
	d_nominal_points.push_back(path_point);
}

void TCommandFeature::Add_Path_Touch_Point(
	const TVector3						&pnt,
	const TVector3						&ijk)
{
	TPathPoint							path_point;
		
	path_point.Set_XYZ(pnt);
	path_point.Set_IJK(ijk);
	path_point.Set_Type(TPathPoint::TOUCH_POINT);
	
	d_nominal_points.push_back(path_point);
}


void TCommandFeature::Add_Measured_Point(
	const TVector3						&xyz,
	const TVector3						&ijk)
{
	TPathPoint							path_point;
	
	path_point.Set_Type(TPathPoint::TOUCH_POINT);
	path_point.Set_XYZ(xyz);
	path_point.Set_IJK(ijk);
	
	d_actual_points.push_back(path_point);
}

void TCommandFeature::Erase_Last_Measured_Point(void)
{
	std::vector<TPathPoint>::iterator	iter;
	
	for(iter = d_actual_points.end();iter != d_actual_points.begin();--iter)
	{
		if((*(iter-1)).Type() ==TPathPoint::TOUCH_POINT)
		{
			d_actual_points.erase(iter-1);
			return;
		}
	}
}

void TCommandFeature::Clear_Nominal(void)
{
	d_nominal_xyz.Set(0.0,0.0,0.0);
	d_nominal_ijk.Set(0.0,0.0,1.0);
	d_nominal_diameter = 0.0;
}

void TCommandFeature::Clear_Actual(void)
{
	d_actual_xyz.Set(0.0,0.0,0.0);
	d_actual_ijk.Set(0.0,0.0,1.0);
	d_actual_diameter = 0.0;
	d_form_error = 0.0;
}

