/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#ifndef COMMANDFEATUREHEADERFILE
#define COMMANDFEATUREHEADERFILE

#include <QString>
#include <vector>

#include "../../../core/vector3.h"
#include "../../../core/referencesystem.h"

#include "command.h"
#include "pathpoint.h"

class TCommandFeature : public TCommand
{
public:

enum TDisplayFormat
{
	FORMAT_NONE = 0,
	FORMAT_DEFAULT,						// output is enabled but layout set by user
	FORMAT_XYZ,
	FORMAT_XYZIJK,
	FORMAT_XYZD,
	FORMAT_XYZIJKD,
	FORMAT_XYZDF
};
	
enum TDiameterType
{
	DIAMETER_TYPE_NONE = 0,
	DIAMETER_TYPE_ID,
	DIAMETER_TYPE_OD
};
	
	// CREATORS
	TCommandFeature(const int memory_id);
	virtual ~TCommandFeature(void) = 0;
	
	// ACCESSORS
	int Memory_Id(void) const {return d_memory_id;}
	
	TDisplayFormat Display_Format(void) const {return d_display_format;}
	QString Feature_Item_Text(const TReferenceSystem &refsys);	// output a formatted line of text

	bool Waiting_For_Measured_Points(void) const;
	
	int Nominal_Point_Count(void) const;
	int Actual_Point_Count(void) const;
	
	TVector3 Nominal_XYZ(void) const {return d_nominal_xyz;}
	TVector3 Nominal_IJK(void) const {return d_nominal_ijk;}
	TVector3 Actual_XYZ(void) const {return d_actual_xyz;}
	TVector3 Actual_IJK(void) const {return d_actual_ijk;}
	
	double Form(void) const {return d_form_error;}
	double Nominal_Diameter(void) const {return d_nominal_diameter;}
	double Actual_Diameter(void) const {return d_actual_diameter;}
	TDiameterType Nominal_Diameter_Type(void) const {return d_nominal_diameter_type;}
	TDiameterType Actual_Diameter_Type(void) const {return d_actual_diameter_type;}
	
	std::vector<TPathPoint> Path_Points(void) const {return d_nominal_points;}
	std::vector<TVector3> Touch_Points(void) const;									// tip center, not compensated
	TVector3 Touch_Point_Approach_Vector(void) const {return d_touch_approach_vector;}	// approach vector for point, line, plane

	static QString Output_Title_XYZ(void);
	static QString Output_Title_XYZIJK(void);
	static QString Output_Title_XYZD(void);
	static QString Output_Title_XYZIJKD(void);
	static QString Output_Title_XYZDF(void);

	// MANIPULATORS
	void Set_Output_Format(TDisplayFormat format) {d_display_format = format;}
	void Set_Nominal_Diameter_Type(TDiameterType type) {d_nominal_diameter_type = type;}
	void Set_Nominal_Projection_Axis(const TVector3 &axis) {d_nominal_projection_axis = axis;}
	
	void Reset_Nominal(void);
	void Reset_Actual(void);
	
	void Add_Path_Point(const TPathPoint &pnt);
	void Add_Path_Move_Point(const TVector3 &pnt);
	void Add_Path_Touch_Point(const TVector3 &pnt,const TVector3 &ijk);
	
	virtual bool Calculate_Feature(const TVector3 &projection_axis,const double &tip_radius) = 0;
	
	void Add_Measured_Point(const TVector3 &xyz,const TVector3 &ijk);
	void Erase_Last_Measured_Point(void);
	
protected:
	int									d_memory_id;
	TVector3							d_nominal_xyz;
	TVector3							d_nominal_ijk;
	TVector3							d_actual_xyz;
	TVector3							d_actual_ijk;
	TVector3							d_nominal_projection_axis;
	TVector3							d_touch_approach_vector;
	double								d_nominal_diameter;
	double								d_actual_diameter;
	double								d_form_error;
	TDisplayFormat						d_display_format;

	TDiameterType						d_nominal_diameter_type;
	TDiameterType						d_actual_diameter_type;
	
	std::vector<TPathPoint>				d_nominal_points;
	std::vector<TPathPoint>				d_actual_points;
	
	void Clear_Nominal(void);
	void Clear_Actual(void);

private:

	// NOT IMPLEMENTED
	TCommandFeature(const TCommandFeature&);
	TCommandFeature& operator=(const TCommandFeature&);
};


#endif
