/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QThread>
#include <QString>
#include <QStringList>
#include <QDateTime>
#include <assert.h>

#if defined  INCLUDE_SRC_EXTRA

#include "../src_extra/controller_dc_code.h"

#endif
 
#include "controller_dc.h"

TControllerDC::TControllerDC(
								   const TLinkProtocol::TProtocolType	protocol_type)
:TController(protocol_type)
{
	d_controller_type = TController::CONTROLLER_DC;
}

TControllerDC::~TControllerDC(void)
{
}

bool TControllerDC::Connect(void)
{
	if(!d_link_protocol.Connect())
	{
		this->Add_Event_Text(QString("ERR: %1").arg(d_link_protocol.Get_Last_Error()));
		return false;
	}
		
	d_current_touch_speed = 4.0;
	d_current_approach_distance = 3.0;
	
	if(this->Clear_Error())
	{
		return true;
	}

	return false;
}

bool TControllerDC::Initialize(void)
{
	QString								write_string;
	QString								text;
	QStringList							list;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	double								param[3];
	TController::TSendResult			write_status;
	static const double					MIN_PARAM_VALUE(1.0);
	
	d_position_updates_enabled = false;
	d_position_precision_updates_enabled = false;

	// Disable temperature compensation
	write_string = QStringLiteral("DISABLE TEMP");
	expected_responses.clear();
	expected_responses.push_back(QStringLiteral("%"));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");

	write_status = this->Send_Command(
									 write_string.toLatin1(),
									 MODE_WAIT_RESPONSE,
									 expected_responses,
									 &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR: DISABLE TEMP command not accepted."));
		
		return false;
	}
	
	// RTC Time
	write_string = QString("RTCTIME %1").arg(QDateTime::currentDateTime().toString("yyyy,MM,dd,HH,mm,ss"));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%"));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(
										write_string.toLatin1(),
										MODE_WAIT_RESPONSE,
										expected_responses,
										&actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		return false;
	}
	
	if(!this->Initialize_Software_Authentication())
	{
		return false;
	}
	
	// verify machine is homed
	write_string = QString("CMHWST");
	
	expected_responses.clear();
	expected_responses.push_back(QString("CMHWST "));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  CMHWST command not accepted."));
		
		return false;
	}
	
	d_is_homed = false;
	
	if(actual_responses.size())
	{
		text = actual_responses[actual_responses.size()-1].simplified().mid(6);	// remove CMHWST
		
		list = text.split(',');
		
		if(list.size() > 0)
		{
			if(list[0].toInt() & 0x01)
			{
				d_is_homed = true;
			}
		}
	}
	
	this->Add_Event_Text(QString("Controller Homed: %1").arg(d_is_homed ? QStringLiteral("YES"):QStringLiteral("NO")));
	
	// get max speed
	// does not fail (uses default instead)
	write_string = QString("SHOW MAXVEL");
	
	expected_responses.clear();
	expected_responses.push_back(QString("MAXVEL "));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  SHOW MAXVEL command not accepted."));
	}
	else if(actual_responses.size())
	{
		text = actual_responses[0].mid(7).trimmed();	// remove MAXVEL
		list = text.split(',');
		
		if(list.size() > 2)		// only interested in first three values
		{
			param[0] = list[0].toDouble();	// x
			param[1] = list[1].toDouble();	// y
			param[2] = list[2].toDouble();	// z
			
			if(param[0] > MIN_PARAM_VALUE &&
			   param[1] > MIN_PARAM_VALUE &&
			   param[2] > MIN_PARAM_VALUE)
			{
				d_max_speed = param[0];
				
				if(param[1] < d_max_speed) d_max_speed = param[1];
				if(param[2] < d_max_speed) d_max_speed = param[2];
			}
		}
	}
	
	// get max acceleration
	// does not fail (uses default instead)
	write_string = QString("SHOW MAXACC");
	
	expected_responses.clear();
	expected_responses.push_back(QString("MAXACC "));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  SHOW MAXACC command not accepted."));
	}
	else if(actual_responses.size())
	{
		text = actual_responses[0].mid(7).trimmed();	// remove MAXACC
		list = text.split(',');
		
		if(list.size() > 2)		// only interested in first three values
		{
			param[0] = list[0].toDouble();	// x
			param[1] = list[1].toDouble();	// y
			param[2] = list[2].toDouble();	// z
			
			if(param[0] > MIN_PARAM_VALUE &&
			   param[1] > MIN_PARAM_VALUE &&
			   param[2] > MIN_PARAM_VALUE)
			{
				d_max_acceleration = param[0];
				
				if(param[1] < d_max_acceleration) d_max_acceleration = param[1];
				if(param[2] < d_max_acceleration) d_max_acceleration = param[2];
			}
		}
	}
	

	return true;
}

bool TControllerDC::Refresh_Tools(void)
{
	return true;
}

bool TControllerDC::Send_Home(void)
{
	QString								write_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	TController::TSendResult			write_status;
	
	d_is_homed = false;
	
	write_string = QString("AUTZER");
		
	expected_responses.clear();
	expected_responses.push_back(QString("%"));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(
									 write_string.toLatin1(),
									 MODE_QUEUE_RESPONSE,
									 expected_responses,
									 &actual_responses);
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  AUTZER command not accepted."));
		return false;
	}
	
	return true;
}

void TControllerDC::Disconnect(void)
{
}

void TControllerDC::Close(void)
{
	d_link_protocol.Close();
}

bool TControllerDC::Clear_Error(void)
{
	QString								write_string;
	QString								text;
	QByteArray							reset_sequence;
	QByteArray							ctrl_cb;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TErrorData				error_data;
	TController::TSendResult			write_status;
	static const int					SLEEP_TIME(100);
	
	reset_sequence.resize(2);
	ctrl_cb.resize(3);
	
	reset_sequence[0] = static_cast<char>(0x0b);
	reset_sequence[1] = static_cast<char>(0x0a);
	
	ctrl_cb[0] = static_cast<char>(0x0e);
	ctrl_cb[1] = static_cast<char>(0x03);
	ctrl_cb[2] = static_cast<char>(0x02);
	
	expected_responses.clear();
	
	this->Add_Event_Text("Sending reset sequence ...");
	
	write_status = this->Send_Command(reset_sequence,
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  Reset sequence not accepted."));
		
		return false;
	}
	

	QThread::msleep(SLEEP_TIME);
	
	// READY
	expected_responses.clear();
	expected_responses.push_back(QString("READY"));
	
	this->Add_Event_Text("Sending CTRL_C, CTRL_B...");
	
	write_status = this->Send_Command(ctrl_cb,
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);

	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  Reset sequence did not return READY."));
		
		return false;
	}
	
	QThread::msleep(SLEEP_TIME);


	return true;
}

void TControllerDC::Check_Machine_Status(void)
{
	QString								write_string;
	QString								text;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TErrorData				error_data;
	TController::TSendResult			write_status;
	
	d_is_machine_error_state = false;
	d_is_machine_estop = false;

	expected_responses.clear();
	
	// check for emergency stop
	write_string = QStringLiteral("SHOW ESTOP");
	
	expected_responses.clear();
	expected_responses.push_back(QString("ESTOP "));	// ESTOP FALSE | TRUE
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  Machine did not return response to ESTOP command."));
		
		d_is_machine_error_state = true;
		return;
	}
		
	if(actual_responses.size())
	{
		text = actual_responses[0];
		
		if(text.contains(QStringLiteral("TRUE"),Qt::CaseInsensitive))
		{
			d_is_machine_estop = true;
		}
	}
}

void TControllerDC::Abort(void)
{
	QByteArray							reset_sequence;
	QByteArray							abort_sequence;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	TController::TSendResult			write_status;
	static const int					sleep_time = 500;
	
	abort_sequence.resize(1);
	
	abort_sequence[0] = static_cast<char>(0x03);
	
	expected_responses.clear();
	
	this->Add_Event_Text("Sending abort ^C...");
	
	write_status = this->Send_Command(abort_sequence,
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  Abort sequence not accepted."));
	}
	
	QThread::msleep(sleep_time);
	
	d_command_response_queue.clear();
	d_sensor_ids.clear();
}

bool TControllerDC::Set_Tool_Type(
	const QString						&tool_type)
{
	QString								write_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	TController::TSendResult			write_status;
	
	if(tool_type.compare(QStringLiteral("TTP"),Qt::CaseInsensitive) != 0)
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Probe type must be TTP"));
		return false;
	}
	
	// Set probe type to trigger
	write_string = QString("PRBTYP TP2");
	
	expected_responses.clear();
	expected_responses.push_back(QString("%"));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_QUEUE_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  PRBTYP command not accepted."));
		return false;
	}
	
	return true;
}
bool TControllerDC::Set_Tool_Data(
	const TVector3						&xyz,
	const double						&tip_diameter)
{
	QString								write_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	TController::TSendResult			write_status;
	
	d_tool_offset = xyz;
	d_tip_radius = tip_diameter / 2.0;
	
	// PRBPIN
	write_string = QString("PRBPIN %1, %2, %3, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, %4")
									.arg(xyz.x,0,'f',4)
									.arg(xyz.y,0,'f',4)
									.arg(xyz.z,0,'f',4)
									.arg(tip_diameter,0,'f',6);
	
	expected_responses.clear();
	expected_responses.push_back(QString("%"));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_QUEUE_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  PRBPIN data not accepted."));
		return false;
	}
	
	return true;
}

bool TControllerDC::Set_Tool_Angles(
	const double						&angle_a,
	const double						&angle_b)
{
	QString								write_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	TController::TSendResult			write_status;
	
	// PHBHTY
	write_string = QString("PRBHTY PH9,%1, %2").arg(angle_a,0,'f',1).arg(angle_b,0,'f',1);
	
	expected_responses.clear();
	expected_responses.push_back(QString("%"));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(
									  write_string.toLatin1(),
									  MODE_QUEUE_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  PRBHTY command not accepted."));
		return false;
	}
	
	return true;
}

bool TControllerDC::Set_Move_Speed(
	const double						&value)
{
	QString								write_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;
	double								actual_speed(value);
	
	if(actual_speed > d_max_speed)
	{
		this->Add_Event_Text(QStringLiteral("WAR:  Move speed greater than controllers max move speed."));
		actual_speed = d_max_speed;
	}

	write_string = QString("MOVPAR %1, %2, %3").arg(actual_speed,0,'f',5).arg(actual_speed,0,'f',5).arg(actual_speed,0,'f',5);
	
	expected_responses.clear();
	expected_responses.push_back(QString("%"));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,	
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  MOVPAR command not accepted."));
		
		return false;
	}
	
	return true;
	
}

bool TControllerDC::Set_Touch_Speed(
	const double						&value)
{
	QString								write_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;

	d_current_touch_speed = value;
	
	// touch speed, probing accuracy, prehit/retract.
	write_string = QString("PRBLPA %1,,%2").arg(d_current_touch_speed,0,'f',5).arg(d_current_approach_distance,0,'f',5);
	
	expected_responses.clear();
	expected_responses.push_back(QString("%"));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,	
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  PRBLPA command not accepted."));
		
		return false;
	}
	
	return true;
}

bool TControllerDC::Set_Acceleration(
	const double						&value)
{
	QString								write_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;
	double								actual_acceleration(value);
	
	if(actual_acceleration > d_max_acceleration)
	{
		this->Add_Event_Text(QStringLiteral("WAR:  Acceleration greater than controllers max acceleration."));
		actual_acceleration = d_max_acceleration;
	}

	write_string = QString("ACCEL %1, %2, %3").arg(actual_acceleration,0,'f',5).arg(actual_acceleration,0,'f',5).arg(actual_acceleration,0,'f',5);
	
	expected_responses.clear();
	expected_responses.push_back(QString("%"));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,	
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  ACCEL command not accepted."));
		
		return false;
	}
	
	return true;
}

bool TControllerDC::Set_Approach_Distance(
	const double						&value)
{
	QString								write_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;

	d_current_approach_distance = value;
	
	// touch speed, probing accuracy, prehit/retract.
	write_string = QString("PRBLPA %1,,%2").arg(d_current_touch_speed,0,'f',5).arg(d_current_approach_distance,0,'f',5);
	
	expected_responses.clear();
	expected_responses.push_back(QString("%"));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,	
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  PRBLPA command not accepted."));
		
		return false;
	}
	
	// search distance, probe mode, max force
	write_string = QString("PRBGPA %1").arg(d_current_approach_distance * 3.0,0,'f',5);
	
	expected_responses.clear();
	expected_responses.push_back(QString("%"));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,	
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  PRBGPA command not accepted."));
		
		return false;
	}
	
	
	return true;
}


bool TControllerDC::Enable_Blended_Moves(void)
{
	QString								write_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;

	// enable fly mode
	write_string = QString("ENABLE FLY");
	
	expected_responses.clear();
	expected_responses.push_back(QString("%"));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  ENABLE FLY command not accepted."));
		
		return false;
	}
	
	return true;
}

bool TControllerDC::Disable_Blended_Moves(void)
{
	QString								write_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;

	// disable fly mode
	write_string = QString("DISABLE FLY");
	
	expected_responses.clear();
	expected_responses.push_back(QString("%"));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  DISABLE FLY command not accepted."));
		
		return false;
	}
	
	return true;
}


bool TControllerDC::Get_Position(
	TVector3							* const pnt)
{
	QString								write_string;
	QString								text;
	QStringList							text_list;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;
	bool								valid;
	
	assert(pnt);
	
	write_string = QStringLiteral("GETPOS");
	
	expected_responses.clear();
	expected_responses.push_back(QString("POS "));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  GETPOS command not accepted."));
		
		return false;
	}
	
	if(actual_responses.size())
	{
		text_list = actual_responses[0].split(',');
		
		// POS 123.456, 123,456, 123.456
		if(text_list.size() > 2)
		{
			text = text_list[0].simplified();
			
			text.remove(0,3);	// remove 'POS'
			
			(*pnt).x = text.toDouble(&valid);
			
			if(!valid)
			{
				this->Add_Event_Text("ERR:  Invalid X data returned from GETPOS command.");
				return false;
			}
			
			(*pnt).y = text_list[1].toDouble(&valid);
			
			if(!valid)
			{
				this->Add_Event_Text("ERR:  Invalid Y data returned from GETPOS command.");
				return false;
			}
			
			(*pnt).z = text_list[2].toDouble(&valid);
			
			if(!valid)
			{
				this->Add_Event_Text("ERR:  Invalid Z data returned from GETPOS command.");
				return false;
			}
			
			d_current_position = (*pnt);
			
		}
		else
		{
			this->Add_Event_Text("ERR:  Invalid data returned from GETPOS command.");
			return false;
		}
	}
	else
	{
		this->Add_Event_Text("ERR:  No data returned from GETPOS command.");
		return false;
	}
	
	return true;
}

bool TControllerDC::Get_Position_DRO(
	TVector3							* const pnt)
{
	QString								write_string;
	QString								text;
	QStringList							text_list;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;

	assert(pnt);
		
	if(d_position_precision_updates_enabled && 
		!d_command_response_queue.size())	// ignore precision requests if commands are currently executing
	{
		if(this->Get_Position(pnt))
		{
			return true;
		}
	}
	
	expected_responses.clear();
	
	write_string = QStringLiteral(">2>GETPOS\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  >2>GETPOS command not accepted."));
		
		return false;
	}
	
	// not the most 'current' position but okay for DRO
	*pnt = d_current_position;
	
	return true;
}

bool TControllerDC::Move_To(
	const TVector3						&pnt)
{
	QString								write_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	TController::TSendResult			write_status;

	if(d_mode == TController::MODE_DCC)
	{
		write_string = QString("MOVABS %1,%2,%3").arg(pnt.x,12,'f',5).arg(pnt.y,12,'f',5).arg(pnt.z,12,'f',5);
		
		expected_responses.clear();
		expected_responses.push_back(QString("%"));
		
		this->Add_Event_Text(write_string);
		write_string += QStringLiteral("\r\n");
		
		write_status = this->Send_Command(write_string.toLatin1(),
										  MODE_QUEUE_RESPONSE,
										  expected_responses,
										  &actual_responses);
		if(write_status != TController::SEND_SUCCESS)
		{
			this->Add_Event_Text(d_last_error);
			this->Add_Event_Text(QStringLiteral("ERR:  MOVEABS command not accepted."));
			
			return false;
		}
	}
	
	return true;
}

bool TControllerDC::Touch(
	const TVector3						&target,
	const TVector3						&approach)
{
	QString								write_string;
	QString								text;
	QStringList							text_list;
	TVector3							surface_normal_vector;
	TVector3							pnt;
	TController::TTouchPoint			touch_point;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	TController::TSendResult			write_status;
	static const double					ZERO_EPSILON(0.00001);

	
	surface_normal_vector = -1.0 * approach;
	
	if(surface_normal_vector.Length() > ZERO_EPSILON)
	{
		surface_normal_vector.Normal();
	}
	else
	{
		this->Add_Event_Text("ERR:  Touch point IJK not defined.");
		return false;
	}
	
	if(d_mode == TController::MODE_DCC)
	{
		write_string = QString("PROBE %1,%2,%3,,%4,%5,%6")
								.arg(target.x,12,'f',5)
								.arg(target.y,12,'f',5)
								.arg(target.z,12,'f',5)
								.arg(surface_normal_vector.i,9,'f',6)
								.arg(surface_normal_vector.j,9,'f',6)
								.arg(surface_normal_vector.k,9,'f',6);
		
		expected_responses.clear();
		expected_responses.push_back(QString("POINT "));
		
		this->Add_Event_Text(write_string);
		write_string += QStringLiteral("\r\n");
		
		write_status = this->Send_Command(write_string.toLatin1(),
										  MODE_QUEUE_RESPONSE,
										  expected_responses,
										  &actual_responses);				
		if(write_status != TController::SEND_SUCCESS)
		{
			this->Add_Event_Text(d_last_error);
			this->Add_Event_Text(QStringLiteral("ERR:  PROBE command not accepted."));
			
			return false;
		}
	}
	
	return true;
}


bool TControllerDC::Get_X_Temperature(
	const QString						&sensor_list,
	double								* const value,
	bool								* const valid)
{
	return this->Get_Sensor_Data(sensor_list,value,valid);
}

bool TControllerDC::Get_Y_Temperature(
	const QString						&sensor_list,
	double								* const value,
	bool								* const valid)
{
	return this->Get_Sensor_Data(sensor_list,value,valid);
}

bool TControllerDC::Get_Z_Temperature(
	const QString						&sensor_list,
	double								* const value,
	bool								* const valid)
{
	return this->Get_Sensor_Data(sensor_list,value,valid);
}

bool TControllerDC::Get_Part_Temperature(
	const QString						&sensor_list,
	double								* const value,
	bool								* const valid)
{
	return this->Get_Sensor_Data(sensor_list,value,valid);
}

bool TControllerDC::Get_Sensor_Value(
	int 								sensor_id)
{
	QString								write_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;

	if(!(sensor_id > 0))
	{
		return false;
	}
	
	write_string = QString("READTP %1").arg(sensor_id);
	
	expected_responses.clear();
	expected_responses.push_back(QString("READTP "));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
										MODE_QUEUE_RESPONSE,
										expected_responses,
										&actual_responses);
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  READTP command not accepted."));
		
		return false;
	}
	
	d_sensor_ids.push_back(sensor_id);
	
	return true;
}

bool TControllerDC::Set_Part_Expansion_Coefficient(
	const double						&expansion_coefficient)
{
	Q_UNUSED(expansion_coefficient);
	
//	TODO - DC set part expansion coefficient
	return true;
}

QString TControllerDC::Get_X_Sensors(void)
{
	QString								sensors("-1");
	QString								write_string;
	QString								text;
	QStringList							list;
	QStringList::const_iterator			list_iter;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	int									ival;
	TController::TSendResult			write_status;

	write_string = QString("SHOW X_SENSAXIS");
	
	expected_responses.clear();
	expected_responses.push_back(QString("X_SENSAXIS "));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  SHOW X_SENSAXIS command not accepted."));
		
		return sensors;
	}
	
	if(actual_responses.size())
	{
		text = actual_responses[0].mid(10).trimmed();	// remove X_SENSAXIS
		list = text.split(',');
		
		sensors = QString();
		for(list_iter = list.begin();list_iter != list.end();++list_iter)
		{
			ival = (*list_iter).toInt();
			
			if(ival > 0)
			{
				if(sensors.length())
				{
					sensors.append(QString(",%1").arg(ival));
				}
				else
				{
					sensors = QString("%1").arg(ival);
				}
			}
		}
		
		if(sensors.length() == 0)
		{
			sensors = QStringLiteral("-1");
		}
	}
	
	return sensors;
}

QString TControllerDC::Get_Y_Sensors(void)
{
	QString								sensors("-1");
	QString								write_string;
	QString								text;
	QStringList							list;
	QStringList::const_iterator			list_iter;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	int									ival;
	TController::TSendResult			write_status;

	write_string = QString("SHOW Y_SENSAXIS");
	
	expected_responses.clear();
	expected_responses.push_back(QString("Y_SENSAXIS "));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  SHOW Y_SENSAXIS command not accepted."));
		
		return sensors;
	}
	
	if(actual_responses.size())
	{
		text = actual_responses[0].mid(10).trimmed();	// remove Y_SENSAXIS
		list = text.split(',');
		
		sensors = QString();
		for(list_iter = list.begin();list_iter != list.end();++list_iter)
		{
			ival = (*list_iter).toInt();
			
			if(ival > 0)
			{
				if(sensors.length())
				{
					sensors.append(QString(",%1").arg(ival));
				}
				else
				{
					sensors = QString("%1").arg(ival);
				}
			}
		}
		
		if(sensors.length() == 0)
		{
			sensors = QStringLiteral("-1");
		}
	}
	
	return sensors;
}

QString TControllerDC::Get_Z_Sensors(void)
{
	QString								sensors("-1");
	QString								write_string;
	QString								text;
	QStringList							list;
	QStringList::const_iterator			list_iter;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	int									ival;
	TController::TSendResult			write_status;

	write_string = QString("SHOW Z_SENSAXIS");
	
	expected_responses.clear();
	expected_responses.push_back(QString("Z_SENSAXIS "));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  SHOW Z_SENSAXIS command not accepted."));
		
		return sensors;
	}
	
	if(actual_responses.size())
	{
		text = actual_responses[0].mid(10).trimmed();	// remove Z_SENSAXIS
		list = text.split(',');
		
		sensors = QString();
		for(list_iter = list.begin();list_iter != list.end();++list_iter)
		{
			ival = (*list_iter).toInt();
			
			if(ival > 0)
			{
				if(sensors.length())
				{
					sensors.append(QString(",%1").arg(ival));
				}
				else
				{
					sensors = QString("%1").arg(ival);
				}
			}
		}
		
		if(sensors.length() == 0)
		{
			sensors = QStringLiteral("-1");
		}
	}
	
	return sensors;
}

QString TControllerDC::Get_Part_Sensors(void)
{
	QString								sensors("-1");
	QString								write_string;
	QString								text;
	QStringList							list;
	QStringList::const_iterator			list_iter;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	int									ival;
	TController::TSendResult			write_status;

	write_string = QString("SHOW SENSWKP");
	
	expected_responses.clear();
	expected_responses.push_back(QString("SENSWKP "));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  SHOW SENSWKP command not accepted."));
		
		return sensors;
	}
	
	if(actual_responses.size())
	{
		text = actual_responses[0].mid(7).trimmed();	// remove SENSWKP
		list = text.split(',');
		
		sensors = QString();
		for(list_iter = list.begin();list_iter != list.end();++list_iter)
		{
			ival = (*list_iter).toInt();
			
			if(ival > 0)
			{
				if(sensors.length())
				{
					sensors.append(QString(",%1").arg(ival));
				}
				else
				{
					sensors = QString("%1").arg(ival);
				}
			}
		}
		
		if(sensors.length() == 0)
		{
			sensors = QStringLiteral("-1");
		}
	}
	
	return sensors;
}

void TControllerDC::Process_Rx_Data(void)
{
	int									index;
	QString								text;
	TController::TErrorData				error_data;
	
	d_rx_data += this->ReadPort();
	
	do
	{
		index = d_rx_data.indexOf('\n');
		
		if(!(index < 0))
		{
			text = QString::fromLatin1(d_rx_data.mid(0,index)).simplified();
			d_rx_data.remove(0,index + 1);
			
#ifdef ENABLE_DEBUG_MODE
			qWarning(QString("RECV: %1").arg(text).toLatin1());
#endif

			if(text.contains(QStringLiteral("ERROR "),Qt::CaseInsensitive))
			{
				if(text.contains(QStringLiteral("0x14000022 Protocol authorization needed")) ||
				   text.contains(QStringLiteral("0x14000014 Protocol authorization violation")))
				{
					this->Initialize_Software_Authentication();
				}
				else
				{
					error_data.text = text;
					
					if(text.contains(QStringLiteral("ABORT "),Qt::CaseInsensitive))
					{
						error_data.severity = 0;
					}
					else
					{
						error_data.severity = 1;
					}

					d_error_data.push_back(error_data);
				}
			}
			else if(text.contains(QStringLiteral("SRCNUM "),Qt::CaseInsensitive))
			{
				this->Complete_Software_Authentication(text);
			}
			else if(text.contains(QStringLiteral("POINT "),Qt::CaseInsensitive))
			{
				this->Process_Touch_Point(text);
			}
			else if(text.contains(QStringLiteral("POS "),Qt::CaseInsensitive))
			{
				this->Process_Pos(text);
			}
			else if(text.contains(QStringLiteral("READTP "),Qt::CaseInsensitive))
			{
				this->Process_Sensor(text);
			}
			else if(text.startsWith(QStringLiteral("KEYDELPNT")))	// KEYPRINT not handled
			{
				d_key_press.push_back(TController::KEY_ERASE_HIT);
			}
			else if(text.startsWith(QStringLiteral("KEYDONE")))
			{
				d_key_press.push_back(TController::KEY_DONE);
			}
		}
		
	}while(!(index < 0));
}


void TControllerDC::Process_Touch_Point(
	const QString						&text)
{
	QString								line;
	QStringList							list;
	TController::TTouchPoint			touch_point;
	int									index;
	bool								valid;
	static const double					ZERO_EPSILON(0.00001);

	index = text.indexOf(QStringLiteral("POINT "),Qt::CaseInsensitive);
	
	if(index < 0)
	{
		this->Add_Event_Text("ERR:  Process_Touch_Point data from POINT X, Y, Z, W, I, J, K");
		return;
	}
	
	line = text.mid(index + 5);	// end of POINT
	
	list = line.split(',');
	
	if(list.size() > 6)
	{
		touch_point.xyz.x = list[0].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text("ERR:  Process_Touch_Point X data from POINT X, Y, Z, W, I, J, K");
			return;
		}
		
		touch_point.xyz.y = list[1].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text("ERR:  Process_Touch_Point Y data from POINT X, Y, Z, W, I, J, K");
			return;
		}
		
		touch_point.xyz.z = list[2].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text("ERR:  Process_Touch_Point Z data from POINT X, Y, Z, W, I, J, K");
			return;
		}
		
		touch_point.ijk.i = list[4].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text("ERR:  Process_Touch_Point I data from POINT X, Y, Z, W, I, J, K");
			return;
		}
		
		touch_point.ijk.j = list[5].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text("ERR:  Process_Touch_Point J data from POINT X, Y, Z, W, I, J, K");
			return;
		}
		
		touch_point.ijk.k = list[6].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text("ERR:  Process_Touch_Point K data from POINT X, Y, Z, W, I, J, K");
			return;
		}
	}
	
	else
	{
		this->Add_Event_Text("ERR:  Process_Touch_Point data from POINT X, Y, Z, W, I, J, K");
		return;
	}
	
	if(touch_point.ijk.Length() > ZERO_EPSILON)
	{
		touch_point.ijk.Normal();
	}
	else
	{
		touch_point.ijk.Set(0,0,1);
	}
	
	touch_point.ijk *= (-1.0);	// returned point is surface normal.  must be approach direction
	
	d_current_position = touch_point.xyz - touch_point.ijk * d_current_approach_distance;
	
	d_touch_points.push_back(touch_point);
}

void TControllerDC::Process_Pos(
	const QString						&text)
{
	QString								line;
	TVector3							pnt;
	QStringList							list;
	TController::TTouchPoint			touch_point;
	int									index;
	bool								valid;
	
	index = text.indexOf(QStringLiteral("POS "),Qt::CaseInsensitive);
	
	if(index < 0)
	{
		this->Add_Event_Text("ERR:  Process_Pos data from POS X, Y, Z, W, A, B, C");
		return;
	}
	
	line = text.mid(index + 3);	// end of POS
	
	list = line.split(',');
	
	
	// POS 123.456, 123,456, 123.456
	if(list.size() > 2)
	{
		pnt.x = list[0].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text("ERR:  Invalid X data returned from POS line.");
			return;
		}
		
		pnt.y = list[1].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text("ERR:  Invalid Y data returned from POS line.");
			return;
		}
		
		pnt.z = list[2].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text("ERR:  Invalid Z data returned from POS line.");
			return;
		}
		
		d_current_position = pnt;
	}
	else
	{
		this->Add_Event_Text("ERR:  Invalid data returned from POS line.");
		return;
	}
}

void TControllerDC::Process_Sensor(
	const QString						&text)
{
	TController::TSensorData			sensor_data;
	int									index;
	QString								sub_text;
	bool								state;
	
	index = text.indexOf(QStringLiteral("READTP "));
	
	if(index < 0)
	{
		this->Add_Event_Text("ERR:  Process_Sensor READTP not found");
		return;
	}
	
	sub_text = text.mid(index + 6);	// remove READTP
	sensor_data.value = sub_text.toDouble(&state);
	
	if(state)
	{
		// get first sensor_id name
		if(d_sensor_ids.size())
		{
			sensor_data.sensor_id = d_sensor_ids[0];
			d_sensor_ids.erase(d_sensor_ids.begin());
			
			// make sure it is valid then store it
			if(sensor_data.sensor_id > 0)
			{
				d_sensor_data.push_back(sensor_data);
			}
		}
	}
}

bool TControllerDC::Get_Sensor_Data(
	const QString						&sensor_list,
	double								* const value,
	bool								* const valid)
{
	QString								write_string;
	QString								text;
	QStringList							text_list;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	std::vector<int>					sensors;
	std::vector<int>::iterator			sensor_iter;
	std::vector<double>					value_list;
	std::vector<double>::iterator		value_iter;
	QStringList							list;
	QStringList::const_iterator			iter;
	double								dval;
	int									sensor_id;
	TController::TSendResult			write_status;
	bool								state;
	
	assert(value);
	assert(valid);
	
	*valid = false;
	
	list = sensor_list.split(',');
	
	for(iter = list.begin();iter != list.end();++iter)
	{
		sensor_id = (*iter).toInt(&state);
		
		if(valid && !(sensor_id < 0))
		{
			sensors.push_back(sensor_id);
		}
	}
	
	if(sensors.size() == 0 || sensor_list.length() == 0)
	{
		return true;
	}
	
	// get sensor values
	for(sensor_iter = sensors.begin();sensor_iter != sensors.end();++sensor_iter)
	{
		expected_responses.clear();
		
		write_string = QString("READTP %1").arg(*sensor_iter);
		
		expected_responses.clear();
		expected_responses.push_back(QString("READTP "));
		
		this->Add_Event_Text(write_string);
		write_string += QStringLiteral("\r\n");
		
		write_status = this->Send_Command(write_string.toLatin1(),
										  MODE_WAIT_RESPONSE,
										  expected_responses,
										  &actual_responses);
		
		for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
		{
			this->Add_Event_Text(*str_iter);
		}
		
		if(write_status != TController::SEND_SUCCESS)
		{
			this->Add_Event_Text(d_last_error);
			this->Add_Event_Text(QStringLiteral("ERR:  READTP command not accepted."));
			
			return false;
		}
		
		if(actual_responses.size())
		{
			text = actual_responses[0].simplified().mid(6);	// remove READTP
			
			dval = text.toDouble(&state);
			
			if(state)
			{
				value_list.push_back(dval);
			}
		}
	}
	
	if(value_list.size())
	{
		dval = 0;
		
		for(value_iter = value_list.begin();value_iter != value_list.end();++value_iter)
		{
			dval += (*value_iter);
		}
		
		dval /= static_cast<double>(value_list.size());
		
		*valid = true;
		*value = dval;
	}
	
	return true;
}

bool TControllerDC::Initialize_Software_Authentication(void)
{
	QString								write_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	TController::TSendResult			write_status;

	this->Add_Event_Text(QStringLiteral("Protocol Authorization Requred"));

	write_string = QStringLiteral("SHOW SRCNUM");
	
	expected_responses.clear();
	expected_responses.push_back(QString("SRCNUM "));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(
										write_string.toLatin1(),
										MODE_QUEUE_RESPONSE,
										expected_responses,
										&actual_responses);
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  SHOW SRCNUM command not accepted."));
		
		return false;
	}
	
	return true;
}

bool TControllerDC::Complete_Software_Authentication(
	const QString						&srcnum)
{
#if defined INCLUDE_SRC_EXTRA
	
	TControllerDCCode					dc_code;
	QString								write_string;
	QString								value_string;
	QString								srcnum_data(srcnum);
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	QStringList::const_iterator			iter;
	TController::TSendResult			write_status;

	this->Add_Event_Text(srcnum);

	if(srcnum_data.length() > 72)	// SRCNUM 014a03b2c2603c...
	{
		srcnum_data.remove(0,7);	// remove 'SRCNUM '
		value_string = QString::fromLatin1(dc_code.SRCNUM_Response(srcnum_data.toLatin1()));
	}
	else
	{
		this->Add_Event_Text("ERR:  Invalid SRCNUM data format.");
		return false;
	}
	
	// CODENUM
	write_string = QString("CODENUM %1").arg(value_string);
	
	expected_responses.clear();
	expected_responses.push_back(QString("%"));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(
									  write_string.toLatin1(),
									  MODE_QUEUE_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		return false;
	}
#else
	
	Q_UNUSED(srcnum);

#pragma message "DC Authentication response code not activated."
	
#endif
	
	return true;
}

