/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#ifndef DRIVERHEADERFILE
#define DRIVERHEADERFILE

#include <QThread>
#include <QMutex>
#include <QString>
#include <QByteArray>
#include <vector>

#include "linkprotocol.h"
#include "controller.h"

class TDriver : public QThread
{
	Q_OBJECT
public:
	
enum TDriverState
{
	DRIVER_STATE_NOT_CONNECTED = 0,
	DRIVER_STATE_CONNECTING,
	DRIVER_STATE_CONNECTED
};

enum TCommandType
{
	DRIVER_COMMAND_NULL = 0,
	DRIVER_SEND_HOME,
	DRIVER_SET_MODE_MANUAL,
	DRIVER_SET_MODE_DCC,
	DRIVER_GET_POSITION,
	DRIVER_MOVE_TO,
	DRIVER_TOUCH_POINT,
	DRIVER_QUERY_TOOL_DATA,
	DRIVER_SET_TOOL_NAME,
	DRIVER_SET_TOOL_TYPE,
	DRIVER_SET_TOOL_DATA,
	DRIVER_SET_TOOL_AB_ANGLES,
	DRIVER_REFRESH_TOOLS,
	DRIVER_SET_MOVE_SPEED,
	DRIVER_SET_TOUCH_SPEED,
	DRIVER_SET_ACCELERATION,
	DRIVER_SET_APPROACH_DISTANCE,
	DRIVER_ENABLE_BLENDED_MOVES,
	DRIVER_DISABLE_BLENDED_MOVES,
	DRIVER_GET_X_TEMPERATURE,
	DRIVER_GET_Y_TEMPERATURE,
	DRIVER_GET_Z_TEMPERATURE,
	DRIVER_GET_PART_TEMPERATURE,
	DRIVER_SET_PART_CTE,
	DRIVER_GET_SENSOR_IDS,
	DRIVER_GET_SENSOR_VALUE,
	DRIVER_SYNC
};
	
enum TCoordinateUpdateSpeed
{
	UPDATE_SPEED_DEFAULT = 0,
	UPDATE_SPEED_HIGH
};
	
struct TCommand
{
	TCommandType						command_type;
	TVector3							xyz;
	TVector3							ijk;
	QString								text;
	int									ivalue;
	double								dvalue1;
	double								dvalue2;
};

	// CREATORS
	TDriver(QObject * const parent);
	~TDriver(void);
	
	// ACCESSORS
//	bool Is_Manual_Mode(void) const {return d_controller_in_manual_mode;}
	
	bool Is_Manual_Mode(void) const {return (d_machine_state & MACHINE_MANUAL_MODE);}
	bool Is_EStop(void) const {return (d_machine_state & MACHINE_ESTOP);}
	bool Is_Homed(void) const {return (d_machine_state & MACHINE_HOMED);}

	// MANIPULATORS
	void Reset(const TController::TControllerType &controller_type);
	void Add_Command(const TDriver::TCommand &command,const bool insert_front = false);		// commands added to front are executed next
	
	void Run(void);
	void Stop(void);
	void Pause(void);
	void Resume(void);
	void Enable_Position_Updates(const bool state,const TCoordinateUpdateSpeed update_speed);
	void Set_Command_Queue_Size(const int size);		// set to -1 to use default
	
	void Close_Driver(void) {this->Set_State_Exit();}
	
	void Set_Protocol_Type(const TLinkProtocol::TProtocolType type) {d_protocol_type = type;}
	
	void Set_Serial_DeviceName(const QString &device_name) {d_serial_device_name = device_name;}
	void Set_Serial_Baud(const TLibSerialDeviceEnum::TBaudRate baud) {d_serial_baud_rate = baud;}
	void Set_Serial_Data(const TLibSerialDeviceEnum::TDataBits data) {d_serial_data_bits = data;}
	void Set_Serial_Parity(const TLibSerialDeviceEnum::TParity parity) {d_serial_parity = parity;}
	void Set_Serial_StopBits(const TLibSerialDeviceEnum::TStopBits stop) {d_serial_stop_bits = stop;}
	void Set_Serial_Flow(const TLibSerialDeviceEnum::TFlowControl flow) {d_serial_flow = flow;}
	void Set_Serial_ReadTimeout(const int timeout) {d_serial_read_timeout = timeout;}
	
	void Set_Socket_Hostname(const QString &name) {d_socket_hostname = name;}
	void Set_Socket_Port(const int port) {d_socket_port = port;}

signals:
	void Connecting(void);
	void Connected(void);
	void Update_Machine_State(const bool show_warning);				
	
	void Event(const QString&);									// anything other than fatal errors
	void Error(const QString&,const int);						// fatal errors text, severity
	void Key_Done(void);									
	void Key_Erase_Hit(void);

	void Position(const double&,const double&,const double&);	// xyz machine position.
	void Touch_Point(const double&,const double&,const double&,const double&,const double&,const double&);	// xyzijk
	
	void Tool_Index_Confirmation_AB(const double&, const double&,int *  const state);	// AB Angles, result.  Result = 0 no index, 1 = manual_index, 2 = controller_index
	void Tool_Index_Confirmation_Name(const QString&,int *  const state);	// Name, result.  Result = 0 no index, 1 = manual_index, 2 = controller_index
	void Tool_Name_Changed(const QString&);
	void Tool_Update_Error(void);
	
	void Query_Tool_Data(const QString &name,const double &x,const double &y,const double &z, const double &i,const double &j, const double &k,const double &d);
	
	void Approach_Distance(const double&);
	
	void Clear_Ipp_Tools(void);
	void Add_Ipp_Tool(const QString&);
	void Add_Ipp_Tools_Complete(void);
	
	void Temperature_X(const double&);
	void Temperature_Y(const double&);
	void Temperature_Z(const double&);
	void Temperature_Part(const double&);
	void Sensor_Value(int,const double&);

	void Sensors_X(const QString&);
	void Sensors_Y(const QString&);
	void Sensors_Z(const QString&);
	void Sensors_Part(const QString&);
	
	void Driver_Sync(const int);
	
#ifdef ENABLE_DEBUG_MODE
	void Clear_Queued_Commands(void);
	void Add_Queued_Command(const QString &text);

	void Clear_Response_Commands(void);
	void Add_Response_Command(const QString &text);
#endif
	
protected:
	virtual void run(void);
	
private:
	
	static const int					STATE_NULL					= 0x00000000;
	static const int					STATE_EXIT					= 0x00000001;
	static const int					STATE_STOPPED				= 0x00000002;
	static const int					STATE_RUNNING				= 0x00000004;
	static const int					STATE_PAUSED				= 0x00000008;
	static const int					STATE_RESUME				= 0x00000010;
	static const int					STATE_ABORT					= 0x00000020;
	
	static const int					MACHINE_NORMAL				= 0x00000000;
	static const int					MACHINE_MANUAL_MODE			= 0x00000001;
	static const int					MACHINE_ESTOP				= 0x00000002;
	static const int					MACHINE_HOMED				= 0x00000004;

	TLinkProtocol::TProtocolType		d_protocol_type;
	
	QString								d_serial_device_name;
	TLibSerialDeviceEnum::TBaudRate		d_serial_baud_rate;
	TLibSerialDeviceEnum::TDataBits		d_serial_data_bits;
	TLibSerialDeviceEnum::TParity		d_serial_parity;
	TLibSerialDeviceEnum::TStopBits		d_serial_stop_bits;
	TLibSerialDeviceEnum::TFlowControl	d_serial_flow;
	int									d_serial_read_timeout;
	
	QString								d_socket_hostname;
	int									d_socket_port;
	
	std::vector<TDriver::TCommand>		d_commands;
	std::vector<TDriver::TCommand>		d_queued_commands;
	TController::TControllerType		d_controller_type;
	
	int									d_controller_queue_size;
	
	int									d_driver_state;
	int									d_machine_state;
	
	bool								d_abort_command_signal;
//	bool								d_controller_in_manual_mode;
	bool								d_position_updates_enabled;
	TCoordinateUpdateSpeed				d_position_update_speed;
	
	mutable QMutex						d_mutex;
	
	bool Is_State_Exit(void) const {return (d_driver_state & STATE_EXIT);}
	bool Is_State_Stopped(void) const {return (d_driver_state & STATE_STOPPED);}
	bool Is_State_Running(void) const {return (d_driver_state & STATE_RUNNING);}
	bool Is_State_Paused(void) const {return (d_driver_state & STATE_PAUSED);}
	bool Is_State_Resume(void) const {return (d_driver_state & STATE_RESUME);}
	bool Is_State_Abort(void) const {return (d_driver_state & STATE_ABORT);}
	
	void Set_State_Exit(void)
	{
		d_driver_state |= STATE_EXIT;
		
		d_abort_command_signal = true;
	}
	
	void Set_State_Stopped(void)
	{
		d_driver_state |= STATE_STOPPED;
		d_driver_state |= STATE_ABORT;
		
		d_driver_state &= (~STATE_RUNNING);
		d_driver_state &= (~STATE_PAUSED);
		d_driver_state &= (~STATE_RESUME);
		
		d_abort_command_signal = true;
	}
	
	void Set_State_Running(void)
	{
		d_driver_state |= STATE_RUNNING;
		
		d_driver_state &= (~STATE_STOPPED);
		d_driver_state &= (~STATE_PAUSED);
		d_driver_state &= (~STATE_RESUME);
		d_driver_state &= (~STATE_ABORT);
		
		d_abort_command_signal = false;
	}
	
	void Set_State_Paused(void)
	{
		d_driver_state |= STATE_PAUSED;
		d_driver_state |= STATE_STOPPED;
		d_driver_state |= STATE_ABORT;
		
		d_driver_state &= (~STATE_RESUME);
		d_driver_state &= (~STATE_RUNNING);
		
		d_abort_command_signal = true;
	}
	
	void Set_State_Resume(void)
	{
		d_driver_state |= STATE_RESUME;
		
		d_driver_state &= (~STATE_STOPPED);
		d_driver_state &= (~STATE_RUNNING);
		d_driver_state &= (~STATE_PAUSED);
		d_driver_state &= (~STATE_ABORT);

		d_abort_command_signal = false;
	}
	
	void Set_Machine_Manual_Mode(void) {d_machine_state |= MACHINE_MANUAL_MODE;}
	void Set_Machine_EStop(void) {d_machine_state |= MACHINE_ESTOP;}
	void Set_Machine_Homed(void) {d_machine_state |= MACHINE_HOMED;}

	void Clear_State_Stopped(void) {d_driver_state &= (~STATE_STOPPED);}
	void Clear_State_Running(void) {d_driver_state &= (~STATE_RUNNING);}
	void Clear_State_Paused(void) {d_driver_state &= (~STATE_PAUSED);}
	void Clear_State_Resume(void) {d_driver_state &= (~STATE_RESUME);}
	
	void Clear_State_Abort(void) 
	{
		d_driver_state &= (~STATE_ABORT);
		d_abort_command_signal = false;
	}
	
	void Clear_State_Exit(void)
	{
		d_driver_state &= (~STATE_EXIT);
		d_abort_command_signal = false;
	}
	
	void Clear_Machine_Manual_Mode(void) {d_machine_state &= (~MACHINE_MANUAL_MODE);}
	void Clear_Machine_EStop(void) {d_machine_state &= (~MACHINE_ESTOP);}
	void Clear_Machine_Homed(void) {d_machine_state &= (~MACHINE_HOMED);}

	int Command_Count(void) const;
	bool Get_Next_Command(TDriver::TCommand * const command);
	void Put_Back_Command(const TDriver::TCommand &command);
	void Put_Back_Commands(const std::vector<TDriver::TCommand> &commands);
	
	void Purge_Commands(void);
	
#ifdef ENABLE_DEBUG_MODE
	void Debug_Show_Execution_Queues(TController * const controller);
#endif
};

#endif
