/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QComboBox>
#include <QFrame>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QPushButton>
#include <QToolButton>
#include <QSpinBox>
#include <QPlainTextEdit>
#include <QSizePolicy>
#include <QSpacerItem>
#include <QStringList>
#include <QCloseEvent>
#include <QFont>
#include <cmath>

#include "exercisedialog.h"

TExerciseDialog::TExerciseDialog(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QDialog(const_cast<QWidget*>(parent),flags)
{
	QFrame								*lower_hseparator_line;
	QFrame								*upper_hseparator_line;
	QFrame								*middle_hseparator_line;
	QGridLayout							*dialog_layout;
	QHBoxLayout							*button_hlayout;
	QHBoxLayout							*header_hlayout;
	QLabel								*corner_1_y_label;
	QLabel								*corner_1_label;
	QLabel								*corner_1_x_label;
	QLabel								*corner_1_z_label;
	QLabel								*corner_2_label;
	QLabel								*corner_2_y_label;
	QLabel								*corner_2_z_label;
	QLabel								*corner_2_x_label;
	QLabel								*pattern_label;
	QLabel								*tool_label;
	QLabel								*count_label;
	QSpacerItem							*button_hspacer;
	QSpacerItem							*header_hspacer;
	QFont								text_font(QStringLiteral("courier new"));
	
	this->resize(601,501);

	text_font.setFixedPitch(true);

	dialog_layout = new QGridLayout(this);
	
	header_hlayout = new QHBoxLayout();
	
	tool_label = new QLabel(this);
	header_hlayout->addWidget(tool_label);
	
	d_active_tool_combo = new QComboBox(this);
	header_hlayout->addWidget(d_active_tool_combo);
	
	pattern_label = new QLabel(this);
	header_hlayout->addWidget(pattern_label);
	
	d_pattern_combo = new QComboBox(this);
	header_hlayout->addWidget(d_pattern_combo);
	
	count_label = new QLabel(this);
	header_hlayout->addWidget(count_label);
	
	d_count_spin = new QSpinBox(this);
	header_hlayout->addWidget(d_count_spin);
	
	header_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	header_hlayout->addItem(header_hspacer);
	dialog_layout->addLayout(header_hlayout,0,0,1,6);
	
	upper_hseparator_line = new QFrame(this);
	upper_hseparator_line->setFrameShape(QFrame::HLine);
	upper_hseparator_line->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(upper_hseparator_line,1,0,1,6);
	
	corner_1_label = new QLabel(this);
	corner_1_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(corner_1_label,2,0,1,1);
	
	corner_1_x_label = new QLabel(this);
	dialog_layout->addWidget(corner_1_x_label,2,1,1,1);
	
	d_get_corner_1_button = new QToolButton(this);
	dialog_layout->addWidget(d_get_corner_1_button,3,0,1,1);
	
	d_move_corner_1_button = new QToolButton(this);
	dialog_layout->addWidget(d_move_corner_1_button,4,0,1,1);

	d_corner_1_x_edit = new QLineEdit(this);
	d_corner_1_x_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_corner_1_x_edit,2,2,1,1);
	
	corner_1_y_label = new QLabel(this);
	dialog_layout->addWidget(corner_1_y_label,3,1,1,1);

	d_corner_1_y_edit = new QLineEdit(this);
	d_corner_1_y_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_corner_1_y_edit,3,2,1,1);

	corner_1_z_label = new QLabel(this);
	dialog_layout->addWidget(corner_1_z_label,4,1,1,1);
	
	d_corner_1_z_edit = new QLineEdit(this);
	d_corner_1_z_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_corner_1_z_edit,4,2,1,1);

	corner_2_label = new QLabel(this);
	corner_2_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(corner_2_label,2,3,1,1);

	corner_2_x_label = new QLabel(this);
	dialog_layout->addWidget(corner_2_x_label,2,4,1,1);
	
	d_get_corner_2_button = new QToolButton(this);
	dialog_layout->addWidget(d_get_corner_2_button,3,3,1,1);
	
	d_move_corner_2_button = new QToolButton(this);
	dialog_layout->addWidget(d_move_corner_2_button,4,3,1,1);

	d_corner_2_x_edit = new QLineEdit(this);
	d_corner_2_x_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_corner_2_x_edit,2,5,1,1);
	
	corner_2_y_label = new QLabel(this);
	dialog_layout->addWidget(corner_2_y_label,3,4,1,1);

	d_corner_2_y_edit = new QLineEdit(this);
	d_corner_2_y_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_corner_2_y_edit,3,5,1,1);

	corner_2_z_label = new QLabel(this);
	dialog_layout->addWidget(corner_2_z_label,4,4,1,1);
	
	d_corner_2_z_edit = new QLineEdit(this);
	d_corner_2_z_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_corner_2_z_edit,4,5,1,1);

	middle_hseparator_line = new QFrame(this);
	middle_hseparator_line->setFrameShape(QFrame::HLine);
	middle_hseparator_line->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(middle_hseparator_line,5,0,1,6);
	
	d_log_text = new QPlainTextEdit(this);
	d_log_text->setFont(text_font);
	d_log_text->setLineWrapMode(QPlainTextEdit::NoWrap);
	d_log_text->setReadOnly(true);
	dialog_layout->addWidget(d_log_text,6,0,1,6);

	lower_hseparator_line = new QFrame(this);
	lower_hseparator_line->setFrameShape(QFrame::HLine);
	lower_hseparator_line->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(lower_hseparator_line,7,0,1,6);

	button_hlayout = new QHBoxLayout();

	button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_hlayout->addItem(button_hspacer);

	d_close_button = new QPushButton(this);
	d_close_button->setAutoDefault(false);
	button_hlayout->addWidget(d_close_button);

	d_stop_button = new QPushButton(this);
	d_stop_button->setAutoDefault(false);
	button_hlayout->addWidget(d_stop_button);

	d_start_button = new QPushButton(this);
	d_start_button->setAutoDefault(false);
	button_hlayout->addWidget(d_start_button);
	dialog_layout->addLayout(button_hlayout,8,0,1,6);
	
	QWidget::setTabOrder(d_active_tool_combo,d_pattern_combo);
	QWidget::setTabOrder(d_pattern_combo,d_count_spin);
	QWidget::setTabOrder(d_count_spin,d_get_corner_1_button);
	QWidget::setTabOrder(d_get_corner_1_button,d_move_corner_1_button);
	QWidget::setTabOrder(d_move_corner_1_button,d_corner_1_x_edit);
	QWidget::setTabOrder(d_corner_1_x_edit,d_corner_1_y_edit);
	QWidget::setTabOrder(d_corner_1_y_edit,d_corner_1_z_edit);
	QWidget::setTabOrder(d_corner_1_z_edit,d_get_corner_2_button);
	QWidget::setTabOrder(d_get_corner_2_button,d_move_corner_2_button);
	QWidget::setTabOrder(d_move_corner_2_button,d_corner_2_x_edit);
	QWidget::setTabOrder(d_corner_2_x_edit,d_corner_2_y_edit);
	QWidget::setTabOrder(d_corner_2_y_edit,d_corner_2_z_edit);
	QWidget::setTabOrder(d_corner_2_z_edit,d_start_button);
	QWidget::setTabOrder(d_start_button,d_stop_button);
	QWidget::setTabOrder(d_stop_button,d_close_button);
	
	// defaults
	this->Set_State(TExerciseDialog::STATE_STOPPED);
	d_count_spin->setRange(1,1000);
	d_count_spin->setValue(10);
	
	d_waiting_state = TExerciseDialog::NOT_WAITING;
	
	d_pattern_combo->addItem(QStringLiteral("Simple"),TExerciseDialog::SEQUENCE_SIMPLE);
	d_pattern_combo->addItem(QStringLiteral("Ellipse"),TExerciseDialog::SEQUENCE_ELLIPSE);
	d_pattern_combo->addItem(QStringLiteral("Star Burst"),TExerciseDialog::SEQUENCE_STARBURST);
	d_pattern_combo->addItem(QStringLiteral("Zigzag"),TExerciseDialog::SEQUENCE_ZIGZAG);
	d_pattern_combo->setCurrentIndex(0);

	this->setWindowTitle(QStringLiteral("Exercise CMM"));
	
	tool_label->setText(QStringLiteral("Tool:"));
	pattern_label->setText(QStringLiteral("Pattern:"));
	count_label->setText(QStringLiteral("Cycle Count:"));
	corner_1_label->setText(QStringLiteral("Corner 1:"));
	corner_1_x_label->setText(QStringLiteral("X:"));
	corner_1_y_label->setText(QStringLiteral("Y:"));
	corner_1_z_label->setText(QStringLiteral("Z:"));
	corner_2_label->setText(QStringLiteral("Corner 2:"));
	corner_2_x_label->setText(QStringLiteral("X:"));
	corner_2_y_label->setText(QStringLiteral("Y:"));
	corner_2_z_label->setText(QStringLiteral("Z:"));
	d_close_button->setText(QStringLiteral("Close"));
	d_stop_button->setText(QStringLiteral("Stop"));
	d_start_button->setText(QStringLiteral("Start"));
	d_get_corner_1_button->setText(QStringLiteral("Get Pos"));
	d_get_corner_2_button->setText(QStringLiteral("Get Pos"));
	d_move_corner_1_button->setText(QStringLiteral("Move To"));
	d_move_corner_2_button->setText(QStringLiteral("Move To"));
	
	d_corner_1_x_edit->setText(QStringLiteral("100.0000"));
	d_corner_1_y_edit->setText(QStringLiteral("100.0000"));
	d_corner_1_z_edit->setText(QStringLiteral("-100.0000"));
	
	d_corner_2_x_edit->setText(QStringLiteral("200.0000"));
	d_corner_2_y_edit->setText(QStringLiteral("200.0000"));
	d_corner_2_z_edit->setText(QStringLiteral("-200.0000"));

	connect(d_active_tool_combo,static_cast<void (QComboBox::*)(int)>(&QComboBox::activated),this,&TExerciseDialog::Tool_Selection_Changed);
	
	connect(d_close_button,&QPushButton::clicked,this,&TExerciseDialog::close);
	connect(d_start_button,&QPushButton::clicked,this,&TExerciseDialog::Start);
	connect(d_stop_button,&QPushButton::clicked,this,&TExerciseDialog::Stop);
	connect(d_get_corner_1_button,&QPushButton::clicked,this,&TExerciseDialog::Get_Corner1);
	connect(d_get_corner_2_button,&QPushButton::clicked,this,&TExerciseDialog::Get_Corner2);
	connect(d_move_corner_1_button,&QPushButton::clicked,this,&TExerciseDialog::Move_Corner1);
	connect(d_move_corner_2_button,&QPushButton::clicked,this,&TExerciseDialog::Move_Corner2);
}

TExerciseDialog::~TExerciseDialog(void)
{
}

std::vector<TDriver::TCommand> TExerciseDialog::Get_Next_Driver_Commands(void)
{
	std::vector<TDriver::TCommand>		commands;
	
	commands = d_commands;
	d_commands.clear();
	
	return commands;
}

void TExerciseDialog::Reset(
	const QStringList					&tool_list,
	const QString						&active_tool_name)
{
	int									index;
	bool								prev_state;
	
	this->Set_State(TExerciseDialog::STATE_STOPPED);
	
	prev_state = d_active_tool_combo->blockSignals(true);
	
	d_active_tool_combo->clear();
	d_active_tool_combo->addItems(tool_list);
	
	// use active tool name or fist index (no name)
	if(active_tool_name.length())
	{
		index = d_active_tool_combo->findText(active_tool_name);
		
		if(!(index < 0))
		{
			d_active_tool_combo->setCurrentIndex(index);
		}
		else
		{
			d_active_tool_combo->setCurrentIndex(0);	// default to '-- select tool --'
		}
	}
	else
	{
		d_active_tool_combo->setCurrentIndex(0);	// default to '-- select tool --'
	}
	
	d_active_tool_combo->blockSignals(prev_state);
	d_active_tool_combo->setEnabled(true);
}

void TExerciseDialog::Reset_Tool_Selection(void)
{
	bool								prev_state;
	
	if(d_active_tool_combo->count())
	{
		prev_state = d_active_tool_combo->blockSignals(true);
		d_active_tool_combo->setCurrentIndex(0);	// default to '-- select tool --'
		d_active_tool_combo->blockSignals(prev_state);
	}
}


void TExerciseDialog::Sequence_Canceled(void)
{
//	this->Add_Log_Text(QStringLiteral("Sequece cancelled."));
//	this->Set_State(TExerciseDialog::STATE_STOPPED);
	
	this->Stop();
}

void TExerciseDialog::Driver_Error(
	const QString						&error,
	const int							severity)
{
	this->Add_Log_Text(QString("ERR:  Severity %1 '%2'").arg(severity).arg(error));
	
	if(d_current_state != TExerciseDialog::STATE_STOPPED)
	{
		// send stop and reset for sever errors
		if(severity > 0)
		{
			this->Add_Log_Text(QStringLiteral("INF:  Move Sequence Cancelled"));
			this->Stop();
		}
	}
}

void TExerciseDialog::Set_Position(
	const TVector3 						&pos)
{
	switch(d_waiting_state)
	{
		case TExerciseDialog::NOT_WAITING:
			break;
			
		case TExerciseDialog::WAITING_CORNER1:
			
			d_corner_1_x_edit->setText(QString("%1").arg(pos.x,0,'f',4));
			d_corner_1_y_edit->setText(QString("%1").arg(pos.y,0,'f',4));
			d_corner_1_z_edit->setText(QString("%1").arg(pos.z,0,'f',4));
			
			break;
			
		case TExerciseDialog::WAITING_CORNER2:
			
			d_corner_2_x_edit->setText(QString("%1").arg(pos.x,0,'f',4));
			d_corner_2_y_edit->setText(QString("%1").arg(pos.y,0,'f',4));
			d_corner_2_z_edit->setText(QString("%1").arg(pos.z,0,'f',4));
			
			break;
	}
	
	d_waiting_state = TExerciseDialog::NOT_WAITING;
}

void TExerciseDialog::Driver_Sync(
	const int							value)
{
	int									count;
	
	count = d_count_spin->value();
	
	if(value > count)
	{
		this->Add_Log_Text(QStringLiteral("Sequence Complete."));
		this->Set_State(TExerciseDialog::STATE_STOPPED);
		
		emit Sequence_End();
	}
	else
	{
		this->Add_Log_Text(QString("Execution cycle %1 of %2 in process...").arg(value).arg(count));
	}
}

void TExerciseDialog::Start(void)
{
	TExerciseDialog::TExerciseSequence	sequence;
	std::vector<TDriver::TCommand>		cycle_commands;
	TDriver::TCommand					command;
	TVector3							p1,p2,pt;
	TVector3							center;
	TVector3							vec;
	TVector3							move_table[4];
	double								angle_xy,angle_z,angle_offset;
	double								radius_x,radius_y,radius_z;
	double								length;
	double								radius;
	int									cntr;
	int									count;
	int									index;
	bool								state(false);
	static const double					PI2(6.28318530717959);
	static const double					MIN_LENGTH(20.0);
	static const double					MIN_XY_RADIUS(MIN_LENGTH / 2.0);
	static const int					ANGLE_COUNT(12);
	static const double					ANGLE_INCREMENT(0.523598775598299);	// 60 deg.

	d_commands.clear();
	
	this->Clear_Log();
	
	index = d_active_tool_combo->currentIndex();
	
	if(index < 1)
	{
		this->Add_Log_Text(QStringLiteral("ERR:  Tool must be set prior to measurement!"));
		return;
	}
	
	p1.x = d_corner_1_x_edit->text().toDouble(&state);
	if(!state)
	{
		this->Add_Log_Text(QStringLiteral("ERR:  Corner 1, X position invalid"));
		return;
	}
	
	p1.y = d_corner_1_y_edit->text().toDouble(&state);
	if(!state)
	{
		this->Add_Log_Text(QStringLiteral("ERR:  Corner 1, Y position invalid"));
		return;
	}
	
	p1.z = d_corner_1_z_edit->text().toDouble(&state);
	if(!state)
	{
		this->Add_Log_Text(QStringLiteral("ERR:  Corner 1, Z position invalid"));
		return;
	}
	
	p2.x = d_corner_2_x_edit->text().toDouble(&state);
	if(!state)
	{
		this->Add_Log_Text(QStringLiteral("ERR:  Corner 2, X position invalid"));
		return;
	}
	
	p2.y = d_corner_2_y_edit->text().toDouble(&state);
	if(!state)
	{
		this->Add_Log_Text(QStringLiteral("ERR:  Corner 2, Y position invalid"));
		return;
	}
	
	p2.z = d_corner_2_z_edit->text().toDouble(&state);
	if(!state)
	{
		this->Add_Log_Text(QStringLiteral("ERR:  Corner 2, Z position invalid"));
		return;
	}
	
	pt = p2 - p1;
	length = pt.Length();
	
	if(length < MIN_LENGTH)
	{
		this->Add_Log_Text(QString("ERR:  Length between Corner 1 and Corner 2 less than %1 mm").arg(MIN_LENGTH,0,'f',1));
		return;
	}
	
	vec = pt;
	vec.Normal();
	
	radius = length / 2.0;
	
	center = (p1 + p2) / 2.0;
	radius -= 5.0;		// make sure move pattern stays within the machine volume by 5 mm
	
	p1 = center - (vec * radius);
	p2 = center + (vec * radius);
	
	// find angle offset in XY plane
	// use point with highest 'z' as so pattern follows slope
	
	if(p1.z > p2.z)
	{
		pt = p1 - center;
	}
	else
	{
		pt = p2 - center;
	}
	
	pt.z = 0;

	if(pt.Length() < MIN_XY_RADIUS)
	{
		this->Add_Log_Text(QString("ERR:  XY Length from center to Corner 1 less than %1 mm").arg(MIN_XY_RADIUS,0,'f',1));
		return;
	}
	
	pt.Normal();
	
	angle_offset = acos(pt.x);
	
	if(pt.y < 0)
	{
		angle_offset = PI2 - angle_offset;
	}
	
	this->Add_Log_Text(QStringLiteral("Generating move sequence..."));

	sequence = static_cast<TExerciseDialog::TExerciseSequence>(d_pattern_combo->currentData(Qt::UserRole).toInt());
	count = d_count_spin->value();
	
	command.command_type = TDriver::DRIVER_MOVE_TO;
	
	radius_x = fabs(p1.x - center.x);
	radius_y = fabs(p1.y - center.y);
	radius_z = fabs(p1.z - center.z);

	switch(sequence)
	{
		case TExerciseDialog::SEQUENCE_SIMPLE:
			
			command.xyz = p1;
			cycle_commands.push_back(command);
			
			command.xyz = p2;
			cycle_commands.push_back(command);
			
			break;
			
		case TExerciseDialog::SEQUENCE_ELLIPSE:
			
			angle_xy = angle_offset;
			angle_z = 0.0;
			
			for(cntr = 0;cntr < ANGLE_COUNT;++cntr)
			{
				pt.x = center.x + cos(angle_xy) * radius_x;
				pt.y = center.y + sin(angle_xy) * radius_y;
				pt.z = center.z + cos(angle_z) * radius_z;
				
				command.xyz = pt;
				cycle_commands.push_back(command);
				
				angle_xy += ANGLE_INCREMENT;
				angle_z += ANGLE_INCREMENT;
			}
			
			break;
			
		case TExerciseDialog::SEQUENCE_STARBURST:
			
			angle_xy = angle_offset;
			angle_z = 0.0;
			
			for(cntr = 0;cntr < ANGLE_COUNT;++cntr)
			{
				command.xyz = center;
				cycle_commands.push_back(command);

				pt.x = center.x + cos(angle_xy) * radius_x;
				pt.y = center.y + sin(angle_xy) * radius_y;
				pt.z = center.z + cos(angle_z) * radius_z;
				
				command.xyz = pt;
				cycle_commands.push_back(command);
				
				angle_xy += ANGLE_INCREMENT;
				angle_z += ANGLE_INCREMENT;
			}
			break;
			
		case TExerciseDialog::SEQUENCE_ZIGZAG:
			move_table[0] = p1;
			
			move_table[1] = p1;
			move_table[1].x = p2.x;
			move_table[1].z = (p1.z + p2.z)/2.0;
			
			move_table[2] = p2;
			move_table[2].x = p1.x;
			move_table[2].z = (p1.z + p2.z)/2.0;
			
			move_table[3] = p2;
			
			command.xyz = move_table[0]; cycle_commands.push_back(command);
			command.xyz = move_table[1]; cycle_commands.push_back(command);
			command.xyz = move_table[2]; cycle_commands.push_back(command);
			command.xyz = move_table[3]; cycle_commands.push_back(command);
			command.xyz = move_table[1]; cycle_commands.push_back(command);
			command.xyz = move_table[2]; cycle_commands.push_back(command);
			command.xyz = move_table[0]; cycle_commands.push_back(command);

			break;
	}
	
	for(cntr = 0;cntr < count;++cntr)
	{
		command.command_type = TDriver::DRIVER_SYNC;
		command.ivalue = cntr + 1;
		
		d_commands.push_back(command);

		d_commands.insert(d_commands.end(),cycle_commands.begin(),cycle_commands.end());
	}
	
	command.command_type = TDriver::DRIVER_SYNC;
	command.ivalue = count + 1;
	
	d_commands.push_back(command);

	this->Add_Log_Text(QStringLiteral("Executing move sequence..."));

	this->Set_State(TExerciseDialog::STATE_RUNNING);

	emit Sequence_Start();
}

void TExerciseDialog::Stop(void)
{
	this->Add_Log_Text(QStringLiteral("Sequece cancelled."));
	
	this->Set_State(TExerciseDialog::STATE_STOPPED);

	emit Sequence_Cancel();
}

void TExerciseDialog::Get_Corner1(void)
{
	if(d_active_tool_combo->currentIndex() < 1)
	{
		this->Add_Log_Text(QStringLiteral("Tool must be set prior to reading machine position!"));
		return;
	}

	d_waiting_state = TExerciseDialog::WAITING_CORNER1;
	
	emit Get_Position();
}

void TExerciseDialog::Get_Corner2(void)
{
	if(d_active_tool_combo->currentIndex() < 1)
	{
		this->Add_Log_Text(QStringLiteral("Tool must be set prior to reading machine position!"));
		return;
	}

	d_waiting_state = TExerciseDialog::WAITING_CORNER2;
	
	emit Get_Position();
}

void TExerciseDialog::Move_Corner1(void)
{
	double								x,y,z;
	bool								state(false);
	
	if(d_active_tool_combo->currentIndex() < 1)
	{
		this->Add_Log_Text(QStringLiteral("Tool must be set prior to moving machine!"));
		return;
	}
	
	x = d_corner_1_x_edit->text().toDouble(&state);
	if(!state) return;
	
	y = d_corner_1_y_edit->text().toDouble(&state);
	if(!state) return;
	
	z = d_corner_1_z_edit->text().toDouble(&state);
	if(!state) return;
	
	emit Move_Position(x,y,z);
}

void TExerciseDialog::Move_Corner2(void)
{
	double								x,y,z;
	bool								state(false);
	
	if(d_active_tool_combo->currentIndex() < 1)
	{
		this->Add_Log_Text(QStringLiteral("Tool must be set prior to moving machine!"));
		return;
	}
	
	x = d_corner_2_x_edit->text().toDouble(&state);
	if(!state) return;
	
	y = d_corner_2_y_edit->text().toDouble(&state);
	if(!state) return;
	
	z = d_corner_2_z_edit->text().toDouble(&state);
	if(!state) return;
	
	emit Move_Position(x,y,z);
}

void TExerciseDialog::Tool_Selection_Changed(
	int									index)
{
	QString								text;
	
	text = d_active_tool_combo->itemText(index);
	
	emit Change_Tool(text);
}
void TExerciseDialog::closeEvent(
	QCloseEvent							*event)
{
	if(d_current_state == TExerciseDialog::STATE_RUNNING)
	{
		emit Stop();
	}
	
	event->accept();
}

void TExerciseDialog::Set_State(
	const TExecutionState				state)
{
	d_current_state = state;
	
	switch(state)
	{
		case TExerciseDialog::STATE_STOPPED:
			
			d_get_corner_1_button->setEnabled(true);
			d_get_corner_2_button->setEnabled(true);

			d_move_corner_1_button->setEnabled(true);
			d_move_corner_2_button->setEnabled(true);

			d_start_button->setEnabled(true);
			d_stop_button->setEnabled(false);
			d_close_button->setEnabled(true);
			
			d_active_tool_combo->setEnabled(true);
			d_pattern_combo->setEnabled(true);
			d_count_spin->setEnabled(true);
			
			break;
			
			
		case TExerciseDialog::STATE_RUNNING:
			
			d_get_corner_1_button->setEnabled(false);
			d_get_corner_2_button->setEnabled(false);
			
			d_move_corner_1_button->setEnabled(false);
			d_move_corner_2_button->setEnabled(false);

			d_start_button->setEnabled(false);
			d_stop_button->setEnabled(true);
			d_close_button->setEnabled(false);
			
			d_active_tool_combo->setEnabled(false);
			d_pattern_combo->setEnabled(false);
			d_count_spin->setEnabled(false);

			break;
	}
}

void TExerciseDialog::Clear_Log(void)
{
	d_log_text->clear();
}
void TExerciseDialog::Add_Log_Text(
	const QString						&text)
{
	d_log_text->moveCursor(QTextCursor::End);
	d_log_text->appendPlainText(text);
}

