/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QGridLayout>
#include <QLabel>
#include <QLineEdit>
#include <QComboBox>
#include <QSpacerItem>
#include <assert.h>

#include "inputwidgetitem.h"

#include "inputwidget.h"

static const int 						STATIC_WIDGET_ROWS(2);

TInputWidget::TInputWidget(
	const TLayoutMode					mode,
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QWidget								*static_spacer_widget;
	
	assert(STATIC_WIDGET_ROWS > 1);

	d_layout_mode = mode;
	
	d_entry_layout = new QGridLayout(this);
	d_entry_layout->setContentsMargins(0,0,0,0);

	static_spacer_widget = new QWidget(this);
	static_spacer_widget->setMinimumHeight(10);
	d_entry_layout->addWidget(static_spacer_widget,STATIC_WIDGET_ROWS-1,0,1,3);

	d_entry_vspacer = new QSpacerItem(0,0,QSizePolicy::Minimum,QSizePolicy::Expanding);
	d_entry_layout->addItem(d_entry_vspacer,STATIC_WIDGET_ROWS,0,1,3);
	
	// defaults
	d_entry_count = 0;
	d_static_widget_1 = 0;
	d_static_widget_2 = 0;
	d_static_widget_3 = 0;
}

TInputWidget::~TInputWidget(void)
{
}

QString TInputWidget::Entry_Label_Text(
	const int							index) const
{
	assert(!(index < 0));
	assert(index < d_entry_count);
	
	return d_input_entries[index].input_label->text();
}

QString TInputWidget::Entry_Item_Text(
	const int							index) const
{
	QString								text;
	
	assert(!(index < 0));
	assert(index < d_entry_count);
		
	if(d_input_entries[index].input_item->Type() == TInputWidgetItem::TYPE_TEXT)
	{
		text = d_input_entries[index].input_item->Widget_Edit()->text();
	}
	else if(d_input_entries[index].input_item->Type() == TInputWidgetItem::TYPE_COMBO)
	{
		text = d_input_entries[index].input_item->Widget_Combo()->currentText();
	}
	
	return text;
}

int TInputWidget::Entry_Item_Data(
	const int							index) const
{
	int									data(0);
	
	assert(!(index < 0));
	assert(index < d_entry_count);
	
	if(d_input_entries[index].input_item->Type() == TInputWidgetItem::TYPE_COMBO)
	{
		data = d_input_entries[index].input_item->Widget_Combo()->currentData().toInt();
	}
	
	return data;
}

void TInputWidget::Set_Entry_Count(
	const int							count)
{
	int									cntr;
	int									current_size;
	TInputEntry							input_entry;
	std::vector<TInputEntry>::iterator	iter;
	bool								init;
	TInputWidgetItem					*prev_item(0);
	QSpacerItem							*entry_hspacer;
	
	current_size = static_cast<int>(d_input_entries.size());
	
	if(count > current_size)
	{
		d_entry_layout->removeItem(d_entry_vspacer);

		for(cntr = current_size; cntr < count;++cntr)
		{
			input_entry.input_label = new QLabel(this);
			input_entry.input_label->setAlignment(Qt::AlignRight | Qt::AlignTrailing | Qt::AlignVCenter);
	
			input_entry.input_item = new TInputWidgetItem(this);
			entry_hspacer = new QSpacerItem(0,0,QSizePolicy::MinimumExpanding,QSizePolicy::Preferred);

			if(d_layout_mode == TInputWidget::MODE_LEFT)
			{
				d_entry_layout->addWidget(input_entry.input_label,cntr + STATIC_WIDGET_ROWS,0,1,1);
				d_entry_layout->addWidget(input_entry.input_item,cntr + STATIC_WIDGET_ROWS,1,1,1);
				d_entry_layout->addItem(entry_hspacer,cntr+STATIC_WIDGET_ROWS,2,1,1);
			}
			else	// TInputWidget::MODE_RIGHT
			{
				d_entry_layout->addItem(entry_hspacer,cntr+STATIC_WIDGET_ROWS,0,1,1);
				d_entry_layout->addWidget(input_entry.input_label,cntr + STATIC_WIDGET_ROWS,1,1,1);
				d_entry_layout->addWidget(input_entry.input_item,cntr + STATIC_WIDGET_ROWS,2,1,1);
			}
			
			d_input_entries.push_back(input_entry);
		}
		
		d_entry_layout->addItem(d_entry_vspacer,count+STATIC_WIDGET_ROWS,0,1,3);
	}
	
	for(cntr = 0,iter = d_input_entries.begin();iter != d_input_entries.end();++iter,++cntr)
	{
		if(cntr < count)
		{
			(*iter).input_label->show();
			(*iter).input_item->show();
		}
		else
		{
			(*iter).input_label->hide();
			(*iter).input_item->hide();
		}
	}
	
	// tab order
	init = true;
	for(iter = d_input_entries.begin();iter != d_input_entries.end();++iter)
	{
		if(init)
		{
			init = false;
		}
		else
		{
			this->setTabOrder(prev_item,(*iter).input_item);
		}
					
		prev_item = (*iter).input_item;
	}
	
	d_entry_count = count;
}

void TInputWidget::Configure_Entry_As_Text(
	const int 							index)
{
	assert(!(index < 0));
	assert(index < d_entry_count);
	
	d_input_entries[index].input_item->Set_Type(TInputWidgetItem::TYPE_TEXT);
}

void TInputWidget::Configure_Entry_As_Combo(
	const int 							index)
{
	assert(!(index < 0));
	assert(index < d_entry_count);
	
	d_input_entries[index].input_item->Set_Type(TInputWidgetItem::TYPE_COMBO);
	
	connect(d_input_entries[index].input_item->Widget_Combo(),static_cast<void (QComboBox::*)(int)>(&QComboBox::currentIndexChanged),this,&TInputWidget::Selection_Changed);
}

void TInputWidget::Set_Entry_Label_Text(
	const int							index,
	const QString						&text)
{
	assert(!(index < 0));
	assert(index < d_entry_count);
	
	d_input_entries[index].input_label->setText(text);
}

void TInputWidget::Set_Entry_Item_Text(
	  const int							index,
	  const QString						&text)
{
	assert(!(index < 0));
	assert(index < d_entry_count);
		
	d_input_entries[index].input_item->Widget_Edit()->setText(text);
}


void TInputWidget::Clear_Entry_Item_Combo(
	const int							index)
{
	assert(!(index < 0));
	assert(index < d_entry_count);
	
	d_input_entries[index].input_item->Widget_Combo()->clear();
}

void TInputWidget::Add_Entry_Combo_Item(
	const int							index,
	const QString						&text,
	const int							option_data)
{
	bool								prev_state;
	
	assert(!(index < 0));
	assert(index < d_entry_count);
	
	prev_state = d_input_entries[index].input_item->Widget_Combo()->blockSignals(true);
	d_input_entries[index].input_item->Widget_Combo()->addItem(text,option_data);
	d_input_entries[index].input_item->Widget_Combo()->blockSignals(prev_state);
}

void TInputWidget::Add_Entry_Combo_Items(
	const int 							index,
	const QStringList 					&options)
{
	bool								prev_state;

	assert(!(index < 0));
	assert(index < d_entry_count);
	
	prev_state = d_input_entries[index].input_item->Widget_Combo()->blockSignals(true);
	d_input_entries[index].input_item->Widget_Combo()->addItems(options);
	d_input_entries[index].input_item->Widget_Combo()->blockSignals(prev_state);
}

void TInputWidget::Select_Entry_Item_Combo_Text(
	const int							index,
	const QString						&text)
{
	bool								prev_state;
	int									option_index(-1);
	
	assert(!(index < 0));
	assert(index < d_entry_count);
	
	if(text.length())
	{
		option_index = d_input_entries[index].input_item->Widget_Combo()->findText(text);
	}
	
	if(!(option_index < 0))
	{
		prev_state = d_input_entries[index].input_item->Widget_Combo()->blockSignals(true);
		d_input_entries[index].input_item->Widget_Combo()->setCurrentIndex(option_index);
		d_input_entries[index].input_item->Widget_Combo()->blockSignals(prev_state);
	}
}

void TInputWidget::Select_Entry_Item_Combo_Data(
	const int							index,
	const int							option_data)
{
	bool								prev_state;
	int									option_index;
	
	assert(!(index < 0));
	assert(index < d_entry_count);
	
	option_index = d_input_entries[index].input_item->Widget_Combo()->findData(option_data);
	
	if(!(option_index < 0))
	{
		prev_state = d_input_entries[index].input_item->Widget_Combo()->blockSignals(true);
		d_input_entries[index].input_item->Widget_Combo()->setCurrentIndex(option_index);
		d_input_entries[index].input_item->Widget_Combo()->blockSignals(prev_state);
	}
}

void TInputWidget::Enable_Entry_Item(
	const int							index,
	const bool							state)
{
	assert(!(index < 0));
	assert(index < d_entry_count);
	
	d_input_entries[index].input_label->setEnabled(state);
	d_input_entries[index].input_item->setEnabled(state);
}

void TInputWidget::Enable_Entry_Items(
	const bool							state)
{
	std::vector<TInputEntry>::iterator iter_input;

	for(iter_input = d_input_entries.begin();iter_input != d_input_entries.end();++iter_input)
	{
		(*iter_input).input_label->setEnabled(state);
		(*iter_input).input_item->setEnabled(state);
	}
}

void TInputWidget::Set_Static_Widget(
	QWidget 							* const widget,
	const int							column,
	const int							column_count)
{
	bool								column_clear[3];
	int									cntr;
	
	assert(!(column < 0));
	assert(column < 2);
	assert(!(column_count < 1));
	assert((column + column_count) < 4);
	
	memset(column_clear,0,sizeof(column_clear));
	
	for(cntr = column;cntr < column_count;++cntr)
	{
		column_clear[cntr] = true;
	}
	
	if(column_clear[0] && d_static_widget_1)
	{
		d_entry_layout->removeWidget(d_static_widget_1);
	}
	
	if(column_clear[1] && d_static_widget_2)
	{
		d_entry_layout->removeWidget(d_static_widget_2);
	}
	
	if(column_clear[2] && d_static_widget_3)
	{
		d_entry_layout->removeWidget(d_static_widget_3);
	}
	
	switch(column)
	{
		case 0:
			d_static_widget_1 = widget;
			break;
			
		case 1:
			d_static_widget_2 = widget;
			break;
			
		case 2:
			d_static_widget_3 = widget;
			break;
	}

	
	if(widget)
	{
		d_entry_layout->addWidget(widget,0,column,1,column_count);
	}
}

void TInputWidget::Selection_Changed(
	int									combo_index)
{
	std::vector<TInputEntry>::const_iterator iter;
	QObject								*sender;
	int									sender_index;
	int									data;
	
	sender = this->sender();
	
	if(sender)
	{
		// find option index
		for(sender_index = 0,iter = d_input_entries.begin();iter != d_input_entries.end();++iter,++sender_index)
		{
			if((*iter).input_item->Type() == TInputWidgetItem::TYPE_COMBO)
			{
				if(sender == (*iter).input_item->Widget_Combo())
				{
					if((*iter).input_item->Widget_Combo()->isVisible() && (combo_index < 0) == false)
					{
						data = (*iter).input_item->Widget_Combo()->itemData(combo_index).toInt();
						
						emit Selection_Changed_Combo(sender_index,data);
						return;
					}
				}
			}
		}
	}
}



