/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#ifndef LASERDIALOGHEADERFILE
#define LASERDIALOGHEADERFILE

#include <QDialog>
#include <vector>

#include "../../../core/vector2.h"
#include "../../../core/vector3.h"

#include "driver.h"
#include "controller.h"
#include "toolwidget.h"

class QComboBox;
class QDoubleSpinBox;
class QSpinBox;
class QLabel;
class QLineEdit;
class QToolButton;
class QPushButton;
class QCloseEvent;
class QPlainTextEdit;

class TLaserOffsetDialog;
class TLaserInputEdit;
class TGraphWidget;
class TMessageBox;

class TCoordinateDisplayWidget;

class TLaserDialog : public QDialog
{
Q_OBJECT

public:
	
enum TState
{
	STATE_STOPPED = 0,
	STATE_RUNNING
};
	
enum TMeasurementPattern
{
	MEASUREMENT_DATA_COLLECTION = 0,
	MEASUREMENT_10360_2
};
	
enum TMode
{
	MODE_UNIDIRECTIONAL = 0,
	MODE_BIDIRECTIONAL
};
	
enum TLaserMeasurementType
{
	LASER_SCALE = 0,
	LASER_STRAIGHTNESS_X,
	LASER_STRAIGHTNESS_Y,
	LASER_STRAIGHTNESS_Z,
	LASER_ANGULAR
};
	
enum TDirection
{
	DIRECTION_FORWARD = 0,
	DIRECTION_REVERSE
};
	
struct TParameters
{
	QString								description;
	
	double								increment;
	double								premove_offset;
	double								start_x;
	double								start_y;
	double								start_z;
	double								end_x;
	double								end_y;
	double								end_z;
	double								tool_offset_x;
	double								tool_offset_y;
	double								tool_offset_z;
	
	int									measurement_pattern;
	int									data_collection_mode;
	int									measurement_type;
	int									filter_level;
};

// CREATORS
	TLaserDialog(const QWidget *parent=0, Qt::WindowFlags flags=Qt::WindowFlags());
	~TLaserDialog(void);

// ACCESSORS
	TLaserDialog::TParameters Parameters(void);
	std::vector<TDriver::TCommand> Get_Next_Driver_Commands(void);

// MANIPULATORS
	void Sequence_Canceled(void);

	void Set_Controller_Type(TController::TControllerType type);
	void Set_Parameters(TLaserDialog::TParameters &parameters);
	void Set_Tool_Data(const TToolWidget::TToolItem &tool_item);

	void Reset(const QString &output_path,const QStringList &tool_list,const QString &active_tool = QString());
	void Reset_Tool_Selection(void);

	void Update_Position(const TVector3 &xyz);
	void Driver_Error(const QString &error,const int severity);

signals:
	void Change_Tool(const QString&);
	void Set_Active_Tool_Offsets(const double&, const double&,const double&,const double&);		// xyzd
	void Move_Position(const double&, const double&,const double&);					// xyz
	void Sequence_Start(void);
	void Sequence_End(void);
	void Stop_Driver(void);

private slots:
	void Close(void);
	void Set_Tool_Offsets(void);
	void Tool_Selection_Changed(int);
	void Measurement_Pattern_Changed(int);
	void Start_Position_Edited(void);
	void End_Position_Edited(void);
	void Get_Start_Position(void);
	void Get_End_Position(void);
	void Offset_Start_Position(void);
	void Offset_End_Position(void);
	void Move_Start_Position(void);
	void Move_End_Position(void);
	void Laser_Value_Changed(void);
	void Next(void);
	void Previous(void);
	void Start(void);
	void Stop(void);
	void Add_Log_Text(const QString&);

protected:
	virtual void closeEvent(QCloseEvent *event);

private:
	
struct TMeasuredData
{
	double								nominal_target;
	double								actual_target;
	double								laser_value;
	double								deviation;
	TVector3							position;
	int									index;			// secondary identification of data
};
	
	QComboBox							*d_active_tool_combo;
	QComboBox							*d_measurement_pattern_combo;
	QComboBox							*d_data_collection_mode_combo;
	QComboBox							*d_laser_measurement_type_combo;
	QDoubleSpinBox						*d_increment_spin;
	QDoubleSpinBox						*d_premove_offset_spin;
	QLabel								*d_icon_label;
	QLineEdit							*d_actual_position_edit;
	QLineEdit							*d_description_edit;
	QLineEdit							*d_status_edit;
	QLineEdit							*d_end_x_edit;
	QLineEdit							*d_end_y_edit;
	QLineEdit							*d_end_z_edit;
	QLineEdit							*d_start_x_edit;
	QLineEdit							*d_start_y_edit;
	QLineEdit							*d_start_z_edit;
	QLineEdit							*d_target_position_edit;
	QLineEdit							*d_tool_x_edit;
	QLineEdit							*d_tool_y_edit;
	QLineEdit							*d_tool_z_edit;
	QLabel								*d_start_map_x_label;
	QLabel								*d_start_map_y_label;
	QLabel								*d_start_map_z_label;
	QLabel								*d_end_map_x_label;
	QLabel								*d_end_map_y_label;
	QLabel								*d_end_map_z_label;
	QLabel								*d_measurement_length_label;
	QPlainTextEdit						*d_information_text;
	QPushButton							*d_end_getpos_button;
	QPushButton							*d_end_move_button;
	QPushButton							*d_end_offset_button;
	QPushButton							*d_start_getpos_button;
	QPushButton							*d_start_move_button;
	QPushButton							*d_start_offset_button;
	QPushButton							*d_set_offsets_button;
	QSpinBox							*d_filter_level_spin;
	QToolButton							*d_close_button;
	QToolButton							*d_next_button;
	QToolButton							*d_play_button;
	QToolButton							*d_previous_button;
	QToolButton							*d_stop_button;
	QWidget								*d_laser_widget;
	QWidget								*d_options_widget;
	QWidget								*d_start_end_points_widget;
	
	TLaserInputEdit						*d_laser_value_edit;
	TCoordinateDisplayWidget			*d_coordinate_display_widget;
	TLaserOffsetDialog					*d_laser_offset_Dialog;
	TGraphWidget						*d_graph_widget;
	TMessageBox							*d_message_box;

	TController::TControllerType		d_controller_type;
	TState								d_current_state;
	TVector3							d_measurement_axis;
	TVector3							d_start_point;
	TVector3							d_end_point;
	double								d_data_collection_target;
	double								d_10360_targets[30];
	int									d_10360_target_index;
	TDirection							d_direction;
	bool								d_input_data_error;
	QString								d_output_path;
	std::vector<TVector3>				d_filter_data;
	TVector3							d_filter_position;
	std::vector<TMeasuredData>			d_measured_data;
	
	std::vector<TDriver::TCommand>		d_commands;

	TVector3 Get_Start_Point(bool * const state = 0) const;
	TVector3 Get_End_Point(bool * const state = 0) const;
	TVector3 Get_Tool_Offsets(bool * const state = 0) const;

	void Write_Output_File(void);

	void Set_State(const TState);
	void Update_Actual_Position(void);
	
	void Update_Measurement_Axis(void);
	void Update_Map_Points(void);
	void Update_Measurement_Length(void);
	
	void Set_Start_Point(const TVector3 &xyz);
	void Set_Start_Map_Point(const TVector3 &xyz);
	void Set_End_Point(const TVector3 &xyz);
	void Set_End_Map_Point(const TVector3 &xyz);
	
	void Data_Collection_Next(const bool stop_on_end);
	void Data_Collection_Previous(const bool stop_on_end);
	void Data_10360_Next(const bool stop_on_end);
	void Data_10360_Previous(const bool stop_on_end);

	void Add_Measurement(const TLaserDialog::TMeasuredData &measured_data);
	void Update_Graph_Display(void);
	bool Fit_Line(const std::vector<TVector2> &pnts,double *slope,double *yintercept=0);
	
	void Clear_Log(void);


// NOT IMPLEMENTED
	TLaserDialog(const TLaserDialog&);
	TLaserDialog& operator=(const TLaserDialog&);
};

#endif
