/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QThread>

#include "linkprotocol.h"

TLinkProtocol::TLinkProtocol(
	const TLinkProtocol::TProtocolType	type)
{
	d_protocol_type = type;	
	
	d_socket_device=0;
	d_socket_device_thread = 0;

}

TLinkProtocol::~TLinkProtocol(void)
{
	this->Close();
	
	if(d_socket_device)
	{
		delete d_socket_device;
		d_socket_device=0;
	}
}

qint64 TLinkProtocol::bytesAvailable(void) const
{
	qint64								available(0);

	switch(d_protocol_type)
	{
		case TLinkProtocol::PROTOCOL_NONE:
			available = 0;
			break;

		case TLinkProtocol::PROTOCOL_SERIAL:
			if(d_serial_device.isOpen())
			{
				available = d_serial_device.bytesAvailable();
			}
			else
			{
				d_last_error = QStringLiteral("Connection closed.");
				available = 0;
			}
			break;
			
		case TLinkProtocol::PROTOCOL_ETHERNET:
			if(d_socket_device)
			{
				if(d_socket_device->isOpen())
				{
					d_socket_device->waitForReadyRead(1);	//	This 'wakes up' the QIODevice
					available = d_socket_device->bytesAvailable();
				}
				else
				{
					d_last_error = QStringLiteral("Connection closed.");
					available = 0;
				}
			}
			break;
	}	
	
	return available;
}

qint64 TLinkProtocol::read(
	char								*data, 
	qint64								max_size)
{
	qint64								result(0);
	
	switch(d_protocol_type)
	{
		case TLinkProtocol::PROTOCOL_NONE:
			break;

		case TLinkProtocol::PROTOCOL_SERIAL:
			
			if(d_serial_device.isOpen())
			{
				result = d_serial_device.read(data,max_size);
				
				if(result < 0)
				{
					d_last_error = d_serial_device.errorString();
				}
			}
			else
			{
				d_last_error = QStringLiteral("Connection closed.");
				result = -1;
			}

			break;
			
		case TLinkProtocol::PROTOCOL_ETHERNET:
			
			if(d_socket_device)
			{
				if(d_socket_device->isOpen())
				{
					d_socket_device->waitForReadyRead(1);	//	This 'wakes up' the QIODevice
					
					result = d_socket_device->read(data,max_size);
					
					if(result < 0)
					{
						d_last_error = d_socket_device->errorString();
					}
				}
				else
				{
					d_last_error = QStringLiteral("Connection closed.");
					result = -1;
				}
			}
			
			break;
	}	

	return result;
}

QByteArray TLinkProtocol::read(
	qint64								max_size)
{
	QByteArray							data;

	switch(d_protocol_type)
	{
		case TLinkProtocol::PROTOCOL_NONE:
			break;
			
		case TLinkProtocol::PROTOCOL_SERIAL:
			
			if(d_serial_device.isOpen())
			{
				data = d_serial_device.read(max_size);
			}
			else
			{
				d_last_error = QStringLiteral("Connection closed.");
			}
			break;
			
		case TLinkProtocol::PROTOCOL_ETHERNET:
			if(d_socket_device)
			{
				if(d_socket_device->isOpen())
				{
					d_socket_device->waitForReadyRead(1);	//	This 'wakes up' the QIODevice
					data = d_socket_device->read(max_size);
				}
				else
				{
					d_last_error = QStringLiteral("Connection closed.");
				}
			}
			break;
	}

	
	return data;
}

bool TLinkProtocol::Connect(void)
{
	this->Close();
	
	switch(d_protocol_type)
	{
		case TLinkProtocol::PROTOCOL_NONE:
			return true;
			
		case TLinkProtocol::PROTOCOL_SERIAL:
			
			d_serial_device.setPortName(d_serial_devicename);
			d_serial_device.setBaudRate(d_serial_baud);
			d_serial_device.setDataBits(d_serial_data);
			d_serial_device.setParity(d_serial_parity);
			d_serial_device.setStopBits(d_serial_stopbits);
			d_serial_device.setFlowControl(d_serial_flow);
			
			d_serial_device.open();
			
			if(!d_serial_device.isOpen())
			{
				d_last_error = QString("Could not open serial port. %1").arg(d_serial_device.errorString());
				return false;
			}
			
			d_serial_device.setDataTerminalReady(true);
			d_serial_device.setRequestToSend(true);
			return true;
			
		case TLinkProtocol::PROTOCOL_ETHERNET:

			d_socket_device = new QTcpSocket();
			d_socket_device_thread = QThread::currentThread();
			
			d_socket_device->connectToHost(QHostAddress(d_socket_hostname),d_socket_port);
			
			if(!d_socket_device->waitForConnected(3000))
			{
				d_last_error = QStringLiteral("Connection timed out.  Wrong IP address?");
				this->Close();
				return false;
			}
			
			if(!d_socket_device->isOpen())
			{
				d_last_error = QStringLiteral("Could not open Socket connection.  Make sure the IP address and port number is correct.");
				
				delete d_socket_device;
				
				d_socket_device=0;
				d_socket_device_thread = 0;
				
				return false;
			}
			
			return true;
	}
	
	return false;
}

void TLinkProtocol::Close(void)
{
	QThread								*current_thread(QThread::currentThread());

	switch(d_protocol_type)
	{
		case TLinkProtocol::PROTOCOL_NONE:
			break;
			
		case TLinkProtocol::PROTOCOL_SERIAL:
			
			if(d_serial_device.isOpen())
			{
				d_serial_device.close();
			}
			break;
			
		case TLinkProtocol::PROTOCOL_ETHERNET:
			
			if(d_socket_device)
			{
				// verify which thread this is call is coming from
				// if the thread is wrong the program blows up
				// preferred to leave it dangling in this case
				// it shouldn't happen
				
				if(current_thread == d_socket_device_thread)
				{
					if(d_socket_device->isOpen())
					{
						d_socket_device->close();
					}
					
					delete d_socket_device;
					
					d_socket_device=0;
					d_socket_device_thread = 0;
				}
				else
				{
//					qWarning("WAR:  TLinkProtoco::Close() called from wrong thread");
					d_socket_device = 0;
				}
			}
			break;
	}
}

bool TLinkProtocol::getChar(
	char								*c)
{
	switch(d_protocol_type)
	{
		case TLinkProtocol::PROTOCOL_NONE:
			break;
			
		case TLinkProtocol::PROTOCOL_SERIAL:
			
			if(d_serial_device.isOpen())
			{
				return d_serial_device.getChar(c);
			}
			else
			{
				d_last_error = QStringLiteral("Connection closed.");
				return false;
			}

			break;
			
			
		case TLinkProtocol::PROTOCOL_ETHERNET:
			
			if(d_socket_device)
			{
				if(d_socket_device->isOpen())
				{
					return d_socket_device->getChar(c);
				}
				else
				{
					d_last_error = QStringLiteral("Connection closed.");
					return false;
				}
			}
			break;
	}

	return false;
}

qint64 TLinkProtocol::write(
	const char							*data,
    qint64								max_size)
{
	qint64								result(0);
		
	switch(d_protocol_type)
	{
		case TLinkProtocol::PROTOCOL_NONE:
			break;
			
		case TLinkProtocol::PROTOCOL_SERIAL:
			if(d_serial_device.isOpen())
			{
				result = d_serial_device.write(data,max_size,true);
				
				if(result < 0)
				{
					d_last_error = d_serial_device.errorString();
				}
			}
			else
			{
				d_last_error = QStringLiteral("Connection closed.");
				result = -1;
			}

			break;
			
		case TLinkProtocol::PROTOCOL_ETHERNET:

			if(d_socket_device)
			{
				if(d_socket_device->isOpen())
				{
					result = d_socket_device->write(data,max_size);
					
					if(result < 0)
					{
						d_last_error = d_socket_device->errorString();
					}
					else
					{
						d_socket_device->waitForBytesWritten(-1);
					}
				}
				else
				{
					d_last_error = QStringLiteral("Connection closed.");
					result = -1;
				}
			}
			
			break;
	}	

	return result;
}

qint64 TLinkProtocol::write(
	const QByteArray					&byte_array)
{
	qint64								result(0);
	
#ifdef ENABLE_DEBUG_MODE
	qWarning(QString("SEND: %1").arg(QString(byte_array).trimmed()).toLatin1());
#endif
	
	switch(d_protocol_type)
	{
		case TLinkProtocol::PROTOCOL_NONE:
			break;
			
		case TLinkProtocol::PROTOCOL_SERIAL:
			
			if(d_serial_device.isOpen())
			{
				result = d_serial_device.write(byte_array,true);
				
				if(result < 0)
				{
					d_last_error = d_serial_device.errorString();
				}
			}
			else
			{
				d_last_error = QStringLiteral("Connection closed.");
				result = -1;
			}

			break;
			
		case TLinkProtocol::PROTOCOL_ETHERNET:
			
			if(d_socket_device)
			{
				
				if(d_socket_device->isOpen())
				{
					result = d_socket_device->write(byte_array);
					
					if(result < 0)
					{
						d_last_error = d_socket_device->errorString();
					}
					else
					{
						d_socket_device->waitForBytesWritten(-1);
					}
				}
				else
				{
					d_last_error = QStringLiteral("Connection closed.");
					result = -1;
				}
			}
			
			break;
	}	

	return result;
}


