/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QDateTime>

#include "measure_routines.h"

static const double						ZERO_EPSILON(0.000001);

static const double						MIN_STEPGAUGE_PLANE_POINT_SPACING_Y(2.0);
static const double						MIN_SPHERE_DIAMETER(4.0);
static const double						MIN_GAUGEBLOCK_LENGTH(2.0);
static const double						MIN_RINGGAUGE_DIAMETER(10.0);
static const double						MIN_PINGAUGE_DIAMETER(10.0);
static const double						MIN_PLANE_SPACER(1.0);
static const double						MIN_MOVE_DISTANCE(10.0);
static const double						MAX_RINGGAUGE_DIAMETER_ERROR(1.0);
static const double						MAX_PINGAUGE_DIAMETER_ERROR(1.0);

static const int						MEMORY_STEPGAUGE_PLANE_MANUAL(10);
static const int						MEMORY_STEPGAUGE_LINE_MANUAL(11);
static const int						MEMORY_STEPGAUGE_POINT_START_MANUAL(12);
static const int						MEMORY_STEPGAUGE_POINT_END_MANUAL(13);
static const int						MEMORY_STEPGAUGE_PLANE_DCC(14);
static const int						MEMORY_STEPGAUGE_LINE_DCC(15);
static const int						MEMORY_STEPGAUGE_POINT_START_DCC(16);
static const int						MEMORY_STEPGAUGE_POINT_END_DCC(17);
static const int						MEMORY_STEPGAUGE_POINT_START_MEASURE(18);

static const int						MEMORY_GAUGEBLOCK_PLANE_MANUAL(100);
static const int						MEMORY_GAUGEBLOCK_PLANE_DCC(101);
static const int						MEMORY_GAUGEBLOCK_POINT_MEASURE(102);

static const int						MEMORY_PLANE_REPEAT_MANUAL(200);
static const int						MEMORY_PLANE_REPEAT_DCC(201);
static const int						MEMORY_POINT_REPEAT_MEASURE(202);

static const int						MEMORY_SPHERE_POINT_MANUAL(300);
static const int						MEMORY_SPHERE_DCC(301);
static const int						MEMORY_SPHERE_CALIBRATE_TOOL(304);
static const int						MEMORY_SPHERE_PFTU_MEASURE(305);
static const int						MEMORY_SPHERE_REPEAT_MEASURE(306);

static const int						MEMORY_BALLBAR_POINT1_MANUAL(400);
static const int						MEMORY_BALLBAR_POINT2_MANUAL(401);
static const int						MEMORY_BALLBAR_SPHERE1_DCC(402);
static const int						MEMORY_BALLBAR_SPHERE2_DCC(403);
static const int						MEMORY_BALLBAR_SPHERES_MEASURE(404);

static const int						MEMORY_BALLBAR_10360_POINT0_MANUAL(500);
static const int						MEMORY_BALLBAR_10360_POINT5_MANUAL(501);
static const int						MEMORY_BALLBAR_10360_SPHERE0_DCC(502);
static const int						MEMORY_BALLBAR_10360_SPHERE5_DCC(503);
static const int						MEMORY_BALLBAR_10360_SPHERES_MEASURE(504);

static const int						MEMORY_RINGGAUGE_PLANE_MANUAL(600);
static const int						MEMORY_RINGGAUGE_CIRCLE_MANUAL(601);
static const int						MEMORY_RINGGAUGE_PLANE_DCC(602);
static const int						MEMORY_RINGGAUGE_CIRCLE_DCC(603);
static const int						MEMORY_RINGGAUGE_CIRCLE_MEASURE(604);

static const int						MEMORY_PINGAUGE_PLANE_MANUAL(700);
static const int						MEMORY_PINGAUGE_CIRCLE_MANUAL(701);
static const int						MEMORY_PINGAUGE_PLANE_DCC(702);
static const int						MEMORY_PINGAUGE_CIRCLE_DCC(703);
static const int						MEMORY_PINGAUGE_CIRCLE_MEASURE(704);

static const int						MEMORY_SPHERE1_ROLL_DCC(800);
static const int						MEMORY_SPHERE2_ROLL_DCC(801);
static const int						MEMORY_SPHERE3_ROLL_DCC(802);
static const int						MEMORY_SPHERE4_ROLL_DCC(803);

TMeasureRoutines::TMeasureRoutines(void)
{
	d_active_tool.xyz.Set(0,0,0);
	d_active_tool.ijk.Set(0,0,-1);
	d_active_tool.tip_diameter = 0.0;
	d_active_tool.angle_a = 0.0;
	d_active_tool.angle_b = 0.0;

	d_approach_distance = 4.0;
	
	d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
	d_stepgauge_line_approach_vector = TVector3(0,1,0);
	
	d_expansion_coeff_x = 10.0;
	d_expansion_coeff_y = 10.0;
	d_expansion_coeff_z = 10.0;
}

TMeasureRoutines::~TMeasureRoutines(void)
{
	d_command_sequence.Delete_Commands();
	d_feature_memory.clear();
}

TMeasureRoutines::TFeatureMemory TMeasureRoutines::Feature_Memory(
	const int 							memory_id,
	bool 								* const valid) const
{
	std::vector<TMeasureRoutines::TFeatureMemory>::const_iterator iter;
	
	assert(valid);
	*valid = false;
	
	for(iter = d_feature_memory.begin();iter != d_feature_memory.end();++iter)
	{
		if((*iter).memory_id == memory_id)
		{
			*valid = true;
			return (*iter);
		}
	}
	
	TMeasureRoutines::TFeatureMemory	fm;
	
	fm.ijk.Set(0,0,1);
	fm.diameter = 0.0;
	fm.form = 0.0;
	fm.memory_id = -1;
	
	return fm;
}

void TMeasureRoutines::Set_Tool_Data(
	const TToolWidget::TToolItem 		&tool_data)
{
	std::vector<TToolWidget::TToolItem>::iterator iter;
	
	if(d_active_tool.name == tool_data.name)
	{
		d_active_tool = tool_data;
	}
		
	for(iter = d_tool_items.begin();iter != d_tool_items.end();++iter)
	{
		if((*iter).name == tool_data.name)
		{
			(*iter) = tool_data;		
			return;
		}
	}
	
	d_tool_items.push_back(tool_data);
}

bool TMeasureRoutines::Set_Active_Tool_Name(
	const QString 						&name)
{
	bool								valid;
		
	d_active_tool = this->Get_Tool_Item(name,&valid);

	if(valid)
	{
		return true;
	}

	d_active_tool.name = name;
	d_active_tool.xyz.Set(0,0,0);
	d_active_tool.ijk.Set(0,0,-1);
	d_active_tool.tip_diameter = 0.0;
	d_active_tool.angle_a = 0.0;
	d_active_tool.angle_b = 0.0;
	
	return false;
}

void TMeasureRoutines::Set_Calibration_Tool_Data(
	const QString						&name,
	const std::vector<TToolWidget::TToolItem> &data)
{
	d_calibration_tool_name = name;
	d_calibration_tool_items = data;
		
	std::vector<TToolWidget::TToolItem>::const_iterator iter;
	
	for(iter = d_calibration_tool_items.begin();iter != d_calibration_tool_items.end();++iter)
	{
		this->Set_Tool_Data((*iter));
	}
}

void TMeasureRoutines::Set_Feature_Memory(
	const TMeasureRoutines::TFeatureMemory &fm)
{
	std::vector<TMeasureRoutines::TFeatureMemory>::iterator iter;
	
	for(iter = d_feature_memory.begin();iter != d_feature_memory.end();++iter)
	{
		if((*iter).memory_id == fm.memory_id)
		{
			(*iter) = fm;
			return;
		}
	}
	
	if(fm.memory_id > 0)
	{
		d_feature_memory.push_back(fm);
	}
}

void TMeasureRoutines::Remove_Feature_Memory(
	const int 							memory_id)
{
	std::vector<TMeasureRoutines::TFeatureMemory>::iterator iter;
	
	for(iter = d_feature_memory.begin();iter != d_feature_memory.end();++iter)
	{
		if((*iter).memory_id == memory_id)
		{
			d_feature_memory.erase(iter);
			return;
		}
	}
}

void TMeasureRoutines::Set_Axis_Expansion_Coefficients(
	const double						&x,
	const double						&y,
	const double						&z)
{
	d_expansion_coeff_x = x;
	d_expansion_coeff_y = y;
	d_expansion_coeff_z = z;
}

void TMeasureRoutines::Generate_Sequence(void)
{
	switch(d_current_sequence_function)
	{
		case MeasureTypes::SEQUENCE_NULL:
			break;
			
		case MeasureTypes::SEQUENCE_INIT_ROLL:
			if(!this->Initialize_Roll())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_CHANGE_TOOL_ROLL:
			if(!this->Change_Tool_Roll())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_MOVE_CLEAR_ROLL:
			if(!this->Move_Clear_Roll())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;

		case MeasureTypes::SEQUENCE_MEASURE_ROLL_SPHERE:
			if(!this->Measure_Roll_Sphere())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_STEPGAUGE_MANUAL:
			if(!this->Align_Stepgauge_Manual())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_GAUGEBLOCK_MANUAL:
			if(!this->Align_Gaugeblock_Manual())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_POINT_MANUAL:
			if(!this->Align_Point_Manual())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_SPHERE_MANUAL:
			if(!this->Align_Sphere_Manual())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_SPHERE_MANUAL_CALIBRATION:

			if(!this->Align_Sphere_Manual_Calibration())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_BALLBAR_B89_MANUAL:
			if(!this->Align_Ballbar_B89_Manual())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_BALLBAR_10360_MANUAL:
			if(!this->Align_Ballbar_10360_Manual())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;

		case MeasureTypes::SEQUENCE_ALIGN_RINGGAUGE_MANUAL:
			if(!this->Align_Ringgauge_Manual())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_PINGAUGE_MANUAL:
			if(!this->Align_Pingauge_Manual())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_STEPGAUGE_DCC:
			if(!this->Align_Stepgauge_DCC())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_GAUGEBLOCK_DCC:
			if(!this->Align_Gaugeblock_DCC())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_POINT_DCC:
			if(!this->Align_Point_DCC())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_SPHERE_DCC:
			if(!this->Align_Sphere_DCC())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_BALLBAR_B89_DCC:
			if(!this->Align_Ballbar_B89_DCC())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_BALLBAR_10360_DCC:
			if(!this->Align_Ballbar_10360_DCC())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;

		case MeasureTypes::SEQUENCE_ALIGN_RINGGAUGE_DCC:
			if(!this->Align_Ringgauge_DCC())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_PINGAUGE_DCC:
			if(!this->Align_Pingauge_DCC())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_STEPGAUGE_10360:
			
			this->Initialize_Output_File();
			
			if(!this->Measure_Stepgauge_10360())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_STEPGAUGE_B89:
			
			this->Initialize_Output_File();
			
			if(!this->Measure_Stepgauge_B89())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_STEPGAUGE_SQUARE:
			
			this->Initialize_Output_File();
			
			if(!this->Measure_Stepgauge_Square())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_GAUGE_BLOCK:
			
			this->Initialize_Output_File();
			
			if(!this->Measure_Gaugeblock())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_POINT_REPEAT:
			
			this->Initialize_Output_File();
			
			if(!this->Measure_Point_Repeat())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_SPHERE_REPEAT:
			
			this->Initialize_Output_File();
			
			if(!this->Measure_Sphere_Rpt())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_SPHERE_PFTU:
			
			this->Initialize_Output_File();
			
			if(!this->Measure_Sphere_Pftu())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_BALLBAR_B89:
			
			this->Initialize_Output_File();
			
			if(!this->Measure_Ballbar_B89())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_BALLBAR_10360:
			
			this->Initialize_Output_File();
			
			if(!this->Measure_Ballbar_10360())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;

		case MeasureTypes::SEQUENCE_MEASURE_RINGGAUGE:
			
			this->Initialize_Output_File();
			
			if(!this->Measure_Ringgauge())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_PINGAUGE:
			
			this->Initialize_Output_File();
			
			if(!this->Measure_Pingauge())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_STEPGAUGE_10360:
			if(!this->Report_Stepgauge_10360())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_STEPGAUGE_B89:
			if(!this->Report_Stepgauge_B89())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_STEPGAUGE_SQUARE:
			if(!this->Report_Stepgauge_Square())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_GAUGEBLOCK:
			if(!this->Report_Gaugeblock())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_POINT_REPEAT:
			if(!this->Report_Point_Repeat())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_SPHERE_REPEAT:
			if(!this->Report_Sphere_Rpt())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_SPHERE_PFTU:
			if(!this->Report_Sphere_Pftu())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_BALLBAR_B89:
			if(!this->Report_Ballbar())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_BALLBAR_10360:
			if(!this->Report_Ballbar_10360())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;

		case MeasureTypes::SEQUENCE_REPORT_RINGGAUGE:
			if(!this->Report_Ringgauge())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_PINGAUGE:
			if(!this->Report_Pingauge())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_ROLL:
			
			this->Initialize_Output_File();

			if(!this->Report_Roll())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;

		case MeasureTypes::SEQUENCE_CALIBRATE_TOOLS:
			
			this->Initialize_Output_File();
			
			if(!this->Calibrate_Tools())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_TOOLS:
			if(!this->Report_Tool())
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			}
			break;
	}
}

void TMeasureRoutines::Start_Sequence(void)
{
	d_command_sequence.Start_Sequence();
}

void TMeasureRoutines::Increment_Current_Sequence_Step(
	const MeasureTypes::TAlignmentOptions	alignment_option)
{
	switch(d_current_sequence_function)
	{
		case MeasureTypes::SEQUENCE_NULL:
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_STEPGAUGE_MANUAL:
			d_current_sequence_function = MeasureTypes::SEQUENCE_ALIGN_STEPGAUGE_DCC;
			break;
			
		case MeasureTypes::SEQUENCE_INIT_ROLL:
			d_current_sequence_function = MeasureTypes::SEQUENCE_CHANGE_TOOL_ROLL;
			break;
			
		case MeasureTypes::SEQUENCE_CHANGE_TOOL_ROLL:
			
			if(d_active_roll_sphere == MEMORY_SPHERE2_ROLL_DCC ||
			   d_active_roll_sphere == MEMORY_SPHERE4_ROLL_DCC)
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_MOVE_CLEAR_ROLL;
			}
			else
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_ALIGN_SPHERE_MANUAL;
			}

			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_ROLL_SPHERE:
			if(d_active_roll_sphere > MEMORY_SPHERE4_ROLL_DCC)
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_REPORT_ROLL;
			}
			else if(d_active_roll_sphere == MEMORY_SPHERE3_ROLL_DCC)
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_ALIGN_SPHERE_MANUAL;
			}
			else
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_CHANGE_TOOL_ROLL;
			}
			break;
			
		case MeasureTypes::SEQUENCE_MOVE_CLEAR_ROLL:
				d_current_sequence_function = MeasureTypes::SEQUENCE_ALIGN_SPHERE_DCC;
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_GAUGEBLOCK_MANUAL:
			d_current_sequence_function = MeasureTypes::SEQUENCE_ALIGN_GAUGEBLOCK_DCC;
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_POINT_MANUAL:
			d_current_sequence_function = MeasureTypes::SEQUENCE_ALIGN_POINT_DCC;
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_SPHERE_MANUAL:
		case MeasureTypes::SEQUENCE_ALIGN_SPHERE_MANUAL_CALIBRATION:
			d_current_sequence_function = MeasureTypes::SEQUENCE_ALIGN_SPHERE_DCC;
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_BALLBAR_B89_MANUAL:
			d_current_sequence_function = MeasureTypes::SEQUENCE_ALIGN_BALLBAR_B89_DCC;
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_BALLBAR_10360_MANUAL:
			d_current_sequence_function = MeasureTypes::SEQUENCE_ALIGN_BALLBAR_10360_DCC;
			break;

		case MeasureTypes::SEQUENCE_ALIGN_RINGGAUGE_MANUAL:
			d_current_sequence_function = MeasureTypes::SEQUENCE_ALIGN_RINGGAUGE_DCC;
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_PINGAUGE_MANUAL:
			d_current_sequence_function = MeasureTypes::SEQUENCE_ALIGN_PINGAUGE_DCC;
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_STEPGAUGE_DCC:
			
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;

			if(alignment_option != MeasureTypes::ALIGNMENT_ONLY)
			{
				if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_10360)
				{
					d_current_sequence_function = MeasureTypes::SEQUENCE_MEASURE_STEPGAUGE_10360;
				}
				else if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_B89)
				{
					d_current_sequence_function = MeasureTypes::SEQUENCE_MEASURE_STEPGAUGE_B89;
				}
				else if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_SQUARE)
				{
					d_current_sequence_function = MeasureTypes::SEQUENCE_MEASURE_STEPGAUGE_SQUARE;
				}
			}

			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_GAUGEBLOCK_DCC:
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			
			if(alignment_option != MeasureTypes::ALIGNMENT_ONLY)
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_MEASURE_GAUGE_BLOCK;
			}
			
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_POINT_DCC:
			
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			
			if(alignment_option != MeasureTypes::ALIGNMENT_ONLY)
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_MEASURE_POINT_REPEAT;
			}
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_SPHERE_DCC:
			
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			
			if(alignment_option != MeasureTypes::ALIGNMENT_ONLY)
			{
				if(d_configuration == MeasureTypes::MEASURE_SPHERE_REPEAT)
				{
					d_current_sequence_function = MeasureTypes::SEQUENCE_MEASURE_SPHERE_REPEAT;
				}
				else if(d_configuration == MeasureTypes::MEASURE_SPHERE_PFTU)
				{
					d_current_sequence_function = MeasureTypes::SEQUENCE_MEASURE_SPHERE_PFTU;
				}
				else if(d_configuration == MeasureTypes::CALIBRATE_TOOLS)
				{
					d_current_sequence_function = MeasureTypes::SEQUENCE_CALIBRATE_TOOLS;
				}
				else if(d_configuration == MeasureTypes::MEASURE_ROLL_OFFSET_TOOL)
				{
					d_current_sequence_function = MeasureTypes::SEQUENCE_MEASURE_ROLL_SPHERE;
				}
			}
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_BALLBAR_B89_DCC:
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			
			if(alignment_option != MeasureTypes::ALIGNMENT_ONLY)
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_MEASURE_BALLBAR_B89;
			}
			
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_BALLBAR_10360_DCC:
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			
			if(alignment_option != MeasureTypes::ALIGNMENT_ONLY)
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_MEASURE_BALLBAR_10360;
			}
			
			break;

		case MeasureTypes::SEQUENCE_ALIGN_RINGGAUGE_DCC:
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			
			if(alignment_option != MeasureTypes::ALIGNMENT_ONLY)
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_MEASURE_RINGGAUGE;
			}
			
			break;
			
		case MeasureTypes::SEQUENCE_ALIGN_PINGAUGE_DCC:
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			
			if(alignment_option != MeasureTypes::ALIGNMENT_ONLY)
			{
				d_current_sequence_function = MeasureTypes::SEQUENCE_MEASURE_PINGAUGE;
			}
			
			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_STEPGAUGE_10360:
			d_current_sequence_function = MeasureTypes::SEQUENCE_REPORT_STEPGAUGE_10360;
			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_STEPGAUGE_B89:
			d_current_sequence_function = MeasureTypes::SEQUENCE_REPORT_STEPGAUGE_B89;
			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_STEPGAUGE_SQUARE:
			d_current_sequence_function = MeasureTypes::SEQUENCE_REPORT_STEPGAUGE_SQUARE;
			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_GAUGE_BLOCK:
			d_current_sequence_function = MeasureTypes::SEQUENCE_REPORT_GAUGEBLOCK;
			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_POINT_REPEAT:
			d_current_sequence_function = MeasureTypes::SEQUENCE_REPORT_POINT_REPEAT;
			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_SPHERE_REPEAT:
			d_current_sequence_function = MeasureTypes::SEQUENCE_REPORT_SPHERE_REPEAT;
			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_SPHERE_PFTU:
			d_current_sequence_function = MeasureTypes::SEQUENCE_REPORT_SPHERE_PFTU;
			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_BALLBAR_B89:
			d_current_sequence_function = MeasureTypes::SEQUENCE_REPORT_BALLBAR_B89;
			break;
			
		case MeasureTypes::SEQUENCE_MEASURE_BALLBAR_10360:
			d_current_sequence_function = MeasureTypes::SEQUENCE_REPORT_BALLBAR_10360;
			break;

		case MeasureTypes::SEQUENCE_MEASURE_RINGGAUGE:
			d_current_sequence_function = MeasureTypes::SEQUENCE_REPORT_RINGGAUGE;
			break;

		case MeasureTypes::SEQUENCE_MEASURE_PINGAUGE:
			d_current_sequence_function = MeasureTypes::SEQUENCE_REPORT_PINGAUGE;
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_STEPGAUGE_10360:
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_STEPGAUGE_B89:
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			break;
		
		case MeasureTypes::SEQUENCE_REPORT_STEPGAUGE_SQUARE:
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_GAUGEBLOCK:
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_POINT_REPEAT:
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_SPHERE_REPEAT:
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_SPHERE_PFTU:
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_BALLBAR_B89:
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_BALLBAR_10360:
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			break;

		case MeasureTypes::SEQUENCE_REPORT_RINGGAUGE:
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_PINGAUGE:
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_ROLL:
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			break;

		case MeasureTypes::SEQUENCE_CALIBRATE_TOOLS:
			d_current_sequence_function = MeasureTypes::SEQUENCE_REPORT_TOOLS;
			break;
			
		case MeasureTypes::SEQUENCE_REPORT_TOOLS:
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			break;
	}
}

void TMeasureRoutines::Close_Output_File(void)
{
	if(d_output_file.isOpen())
	{
		d_output_file.close();
	}
}

void TMeasureRoutines::Process_Command(
	const MeasureTypes::TDisplayCoordinateSystem display_coordinate_system,
	TCommand							* const command)
{
	TCommandAlign						*command_align;
	TCommandFeature						*command_feature;
	TCommandFeature						*command_feature1;
	TCommandFeature						*command_feature2;
	TCommandMode						*command_mode;
	TCommandDimension					*command_dimension;
	TCommandAlign::TAlignStep			align_step;
	TCommandDimension::TDimensionItem	dimension_item;
	TVector3							xyz;
	TVector3							ijk;
	TVector3							xyz_actual_feat1;
	TVector3							xyz_nominal_feat1;
	TVector3							xyz_actual_feat2;
	TVector3							xyz_nominal_feat2;
	QString								text;
	double								dval;
	int									count;
	int									cntr;
	int									memory_id;
	int									memory_id_feat1;
	int									memory_id_feat2;
	
	assert(command);
		
	switch(command->Type())
	{
		case TCommand::COMMAND_NULL:
		case TCommand::COMMAND_MOVE:
		case TCommand::COMMAND_MOVE_CUBE:
		case TCommand::COMMAND_QUERY_TOOL:
		case TCommand::COMMAND_SET_TOOL:
		case TCommand::COMMAND_MODAL_PROMPT:
		case TCommand::COMMAND_MEASURE_PLANE:
		case TCommand::COMMAND_MEASURE_LINE:
		case TCommand::COMMAND_MEASURE_POINT:
		case TCommand::COMMAND_MEASURE_SPHERE:
		case TCommand::COMMAND_MEASURE_CIRCLE:
		case TCommand::COMMAND_MODAL_SYNC:
			break;
			
		case TCommand::COMMAND_DIMENSION:
			
			command_dimension = reinterpret_cast<TCommandDimension*>(command);
			
			memory_id_feat1 = command_dimension->Memory_Id_Feature1();
			memory_id_feat2 = command_dimension->Memory_Id_Feature2();
			
			command_feature1 = d_command_sequence.Get_Feature_Command(memory_id_feat1);
			command_feature2 = d_command_sequence.Get_Feature_Command(memory_id_feat2);
						
			if((memory_id_feat1 < 0) == false && command_feature1 == 0)
			{
				emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(memory_id_feat1));
			}
			
			if((memory_id_feat2 < 0) == false && command_feature2 == 0)
			{
				emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(memory_id_feat2));
			}
			
			if(command_feature1 && command_feature2)	// two feature dimensions
			{
				if(!(command_feature1->Waiting_For_Measured_Points() || command_feature2->Waiting_For_Measured_Points()))
				{
					switch(display_coordinate_system)
					{
						case MeasureTypes::COORDINATE_SYSTEM_MACHINE:
							
							xyz_actual_feat1 = command_feature1->Actual_XYZ();
							xyz_nominal_feat1 = command_feature1->Nominal_XYZ();
							
							xyz_actual_feat2 = command_feature2->Actual_XYZ();
							xyz_nominal_feat2 = command_feature2->Nominal_XYZ();
							
							break;
							
						case MeasureTypes::COORDINATE_SYSTEM_PART:
							
							xyz_actual_feat1 = d_refsys.FromWorld(command_feature1->Actual_XYZ());
							xyz_nominal_feat1 = command_feature1->Nominal_XYZ();
							
							xyz_actual_feat2 = d_refsys.FromWorld(command_feature2->Actual_XYZ());
							xyz_nominal_feat2 = command_feature2->Nominal_XYZ();
							
							break;
							
						case MeasureTypes::COORDINATE_SYSTEM_PART_ORIGIN_ONLY:
							
							xyz_actual_feat1 = command_feature1->Actual_XYZ() - d_refsys.Origin();
							xyz_nominal_feat1 = command_feature1->Nominal_XYZ();
							
							xyz_actual_feat2 = command_feature2->Actual_XYZ();
							xyz_nominal_feat2 = command_feature2->Nominal_XYZ();
							
							break;
							
					}
					
					count = command_dimension->Dimension_Item_Count();
					
					for(cntr = 0;cntr < count;++cntr)
					{
						dimension_item = command_dimension->Dimension_Item(cntr);
						
						switch(dimension_item.dimension_field)
						{
							case TCommandDimension::FIELD_X:
							case TCommandDimension::FIELD_Y:
							case TCommandDimension::FIELD_Z:
							case TCommandDimension::FIELD_FORM:
							case TCommandDimension::FIELD_DIAM:
								break;
								
							case TCommandDimension::FIELD_DIST_X:
								command_dimension->Set_Actual(dimension_item.dimension_field,xyz_actual_feat2.x - xyz_actual_feat1.x);
								command_dimension->Set_Nominal(dimension_item.dimension_field,xyz_nominal_feat2.x - xyz_nominal_feat1.x);
								break;
								
							case TCommandDimension::FIELD_DIST_Y:
								command_dimension->Set_Actual(dimension_item.dimension_field,xyz_actual_feat2.y - xyz_actual_feat1.y);
								command_dimension->Set_Nominal(dimension_item.dimension_field,xyz_nominal_feat2.y - xyz_nominal_feat1.y);
								break;
								
							case TCommandDimension::FIELD_DIST_Z:
								break;
								command_dimension->Set_Actual(dimension_item.dimension_field,xyz_actual_feat2.z - xyz_actual_feat1.z);
								command_dimension->Set_Nominal(dimension_item.dimension_field,xyz_nominal_feat2.z - xyz_nominal_feat1.z);
								break;
								
							case TCommandDimension::FIELD_DIST_XYZ:
								
								xyz = xyz_actual_feat1 -  xyz_actual_feat2;
								dval = xyz.Length();
								command_dimension->Set_Actual(dimension_item.dimension_field,dval);
								
								xyz = xyz_nominal_feat1 -  xyz_nominal_feat2;
								dval = xyz.Length();
								command_dimension->Set_Nominal(dimension_item.dimension_field,dval);
								
								break;
						}
						
						emit Add_Log_Text(command_dimension->Dimension_Item_Text(cntr));
					}
				}
			}
			else if(command_feature1)	// single feature dimension
			{
				if(!command_feature1->Waiting_For_Measured_Points())
				{
					
					switch(display_coordinate_system)
					{
						case MeasureTypes::COORDINATE_SYSTEM_MACHINE:
							
							xyz_actual_feat1 = command_feature1->Actual_XYZ();
							
							break;
							
						case MeasureTypes::COORDINATE_SYSTEM_PART:
							
							xyz_actual_feat1 = d_refsys.FromWorld(command_feature1->Actual_XYZ());
							
							break;
							
						case MeasureTypes::COORDINATE_SYSTEM_PART_ORIGIN_ONLY:
							
							xyz_actual_feat1 = command_feature1->Actual_XYZ() - d_refsys.Origin();
							
							break;
					}
					
					
					count = command_dimension->Dimension_Item_Count();
					
					for(cntr = 0;cntr < count;++cntr)
					{
						dimension_item = command_dimension->Dimension_Item(cntr);
						
						switch(dimension_item.dimension_field)
						{
							case TCommandDimension::FIELD_X:
								command_dimension->Set_Actual(dimension_item.dimension_field,xyz_actual_feat1.x);
								break;
								
							case TCommandDimension::FIELD_Y:
								command_dimension->Set_Actual(dimension_item.dimension_field,xyz_actual_feat1.y);
								break;
								
							case TCommandDimension::FIELD_Z:
								command_dimension->Set_Actual(dimension_item.dimension_field,xyz_actual_feat1.z);
								break;
								
							case TCommandDimension::FIELD_DIST_X:
							case TCommandDimension::FIELD_DIST_Y:
							case TCommandDimension::FIELD_DIST_Z:
							case TCommandDimension::FIELD_DIST_XYZ:
								break;
								
							case TCommandDimension::FIELD_FORM:
								command_dimension->Set_Actual(dimension_item.dimension_field,command_feature1->Form());
								command_dimension->Set_Nominal(dimension_item.dimension_field,0.0);
								break;
								
							case TCommandDimension::FIELD_DIAM:
								command_dimension->Set_Actual(dimension_item.dimension_field,command_feature1->Actual_Diameter());
								command_dimension->Set_Nominal(dimension_item.dimension_field,command_feature1->Nominal_Diameter());
								break;
						}
						
						emit Add_Log_Text(command_dimension->Dimension_Item_Text(cntr));
					}
				}
			}
			
			break;
			
		case TCommand::COMMAND_ALIGN:
			
			command_align = reinterpret_cast<TCommandAlign*>(command);
			
			count = command_align->Alignment_Step_Count();

			for(cntr = 0;cntr < count;++cntr)
			{
				align_step = command_align->Alignment_Step(cntr);
				
				memory_id = align_step.memory_id;
				
				if(memory_id < 0)
				{
					switch(align_step.command_function)
					{
						case TCommandAlign::TYPE_SKEW1:
							emit Add_Log_Text(QString("ERR:  Skew1 %1 requires a feature.").arg(this->Axis_Name(align_step.axis)));
							d_refsys.Skew1(ijk,align_step.axis);
							break;
							
						case TCommandAlign::TYPE_SKEW1_PROBE_AXIS:
							emit Add_Log_Text(QString("INF:  Skew1 %1 to <Probe_Axis>").arg(this->Axis_Name(align_step.axis)));
							d_refsys.Skew1(d_active_tool.ijk,align_step.axis);
							break;
							
						case TCommandAlign::TYPE_SKEW2:
							emit Add_Log_Text(QString("ERR:  Skew2 %1 requires a feature").arg(this->Axis_Name(align_step.axis)));
							d_refsys.Skew2(ijk,align_step.axis,align_step.projection_axis);
							break;
							
						case TCommandAlign::TYPE_SKEW2_PROBE_AXIS:
							emit Add_Log_Text(QString("INF:  Skew2 %1 to <Probe_Axis> about %2").arg(this->Axis_Name(align_step.axis)).arg(this->Axis_Name(align_step.projection_axis)));
							d_refsys.Skew2(d_active_tool.ijk,align_step.axis,align_step.projection_axis);
							break;
							
						case TCommandAlign::TYPE_SKEW2_STEM_AXIS:
							emit Add_Log_Text(QString("INF:  Skew2 %1 to <Stem> about %2").arg(this->Axis_Name(align_step.axis)).arg(this->Axis_Name(align_step.projection_axis)));
							d_refsys.Skew2(d_stem_vector_sphere,align_step.axis,align_step.projection_axis);
							break;
							
						case TCommandAlign::TYPE_ORIGIN:
							emit Add_Log_Text(QString("ERR:  Origin %1 requires a feature").arg(this->Axis_Name(align_step.axis)));
							d_refsys.Set_Origin(xyz,align_step.axis);
							break;
							
						case TCommandAlign::TYPE_ORIGIN_OFFSET:
							emit Add_Log_Text(QString("INF:  Origin offset of %1 in %2").arg(align_step.value,0,'f',4).arg(this->Axis_Name(align_step.axis)));
							d_refsys.Offset_Origin(align_step.value,align_step.axis);
							break;
					}
				}
				else
				{
					command_feature = d_command_sequence.Get_Feature_Command(memory_id);
					
					if(command_feature)
					{
						if(!command_feature->Waiting_For_Measured_Points())
						{
							
							xyz = command_feature->Actual_XYZ();
							ijk = command_feature->Actual_IJK();
							
							
							switch(align_step.command_function)
							{
								case TCommandAlign::TYPE_SKEW1:
									emit Add_Log_Text(QString("INF:  Skew1 %1 to Memory[%2]").arg(this->Axis_Name(align_step.axis)).arg(memory_id));
									d_refsys.Skew1(ijk,align_step.axis);
									break;
									
								case TCommandAlign::TYPE_SKEW1_PROBE_AXIS:
									emit Add_Log_Text(QString("INF:  Skew1 %1 to <Probe_Axis>").arg(this->Axis_Name(align_step.axis)));
									d_refsys.Skew1(d_active_tool.ijk,align_step.axis);
									break;
									
								case TCommandAlign::TYPE_SKEW2:
									emit Add_Log_Text(QString("INF:  Skew2 %1 to Memory[%2] about %3").arg(this->Axis_Name(align_step.axis)).arg(memory_id).arg(this->Axis_Name(align_step.projection_axis)));
									d_refsys.Skew2(ijk,align_step.axis,align_step.projection_axis);
									break;
									
								case TCommandAlign::TYPE_SKEW2_PROBE_AXIS:
									emit Add_Log_Text(QString("INF:  Skew2 %1 to <Probe_Axis> about %2").arg(this->Axis_Name(align_step.axis)).arg(this->Axis_Name(align_step.projection_axis)));
									d_refsys.Skew2(d_active_tool.ijk,align_step.axis,align_step.projection_axis);
									break;
									
								case TCommandAlign::TYPE_SKEW2_STEM_AXIS:
									emit Add_Log_Text(QString("INF:  Skew2 %1 to <Stem> about %2").arg(this->Axis_Name(align_step.axis)).arg(this->Axis_Name(align_step.projection_axis)));
									d_refsys.Skew2(d_stem_vector_sphere,align_step.axis,align_step.projection_axis);
									break;
									
								case TCommandAlign::TYPE_ORIGIN:
									emit Add_Log_Text(QString("INF:  Origin %1 to Memory[%2]").arg(this->Axis_Name(align_step.axis)).arg(memory_id));
									d_refsys.Set_Origin(xyz,align_step.axis);
									break;
									
								case TCommandAlign::TYPE_ORIGIN_OFFSET:
									emit Add_Log_Text(QString("INF:  Origin offset of %1 in %2").arg(align_step.value,0,'f',4).arg(this->Axis_Name(align_step.axis)));
									d_refsys.Offset_Origin(align_step.value,align_step.axis);
									break;
							}
						}
					}
					else
					{
						emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(memory_id));
					}
				}
			}
			
			break;
			
		case TCommand::COMMAND_MODAL_MODE:
			
			command_mode = reinterpret_cast<TCommandMode*>(command);
			
			if(command_mode->Is_Manual_Mode())
			{
				emit Add_Log_Text(QStringLiteral("Mode:  Manual"));
				d_manual_mode = true;
			}
			else
			{
				emit Add_Log_Text(QStringLiteral("Mode:  DCC"));
				d_manual_mode = false;
			}
			
			break;
	}
}

bool TMeasureRoutines::Tool_Item_Exists(
	const QString 						&name) const
{
	std::vector<TToolWidget::TToolItem>::const_iterator iter;
	
	if(name.length())
	{
		for(iter = d_tool_items.begin();iter != d_tool_items.end();++iter)
		{
			if((*iter).name == name)
			{
				return true;
			}
		}
	}

	return false;
}

TToolWidget::TToolItem TMeasureRoutines::Get_Tool_Item(
	const QString 						&name,
	bool 								* const valid) const
{
	std::vector<TToolWidget::TToolItem>::const_iterator iter;
	
	assert(valid);
	
	if(name.length())
	{
		for(iter = d_tool_items.begin();iter != d_tool_items.end();++iter)
		{
			if((*iter).name == name)
			{
				*valid = true;
				return (*iter);
			}
		}
	}
	
	*valid = false;
	
	return TToolWidget::TToolItem();
}

bool TMeasureRoutines::Initialize_Roll(void)
{
	TCommandMode						*command_mode;
	TCommandQueryTool					*command_query_tool;
	TCommandSync						*command_sync;
	TToolWidget::TToolItem				tool_item;
		
	this->Refsys_Clear_Alignment();
	
	d_active_roll_sphere = MEMORY_SPHERE1_ROLL_DCC;

	tool_item.xyz.Set(0,0,0);
	tool_item.ijk.Set(0,0,-1);
	tool_item.angle_a = 0.0;
	tool_item.angle_b = 0.0;
	tool_item.tip_diameter = 0.0;
		
	if(!this->Tool_Item_Exists(d_parameters.roll_tool1_name))
	{
		tool_item.name = d_parameters.roll_tool1_name;
		d_tool_items.push_back(tool_item);
	}

	if(!this->Tool_Item_Exists(d_parameters.roll_tool2_name))
	{
		tool_item.name = d_parameters.roll_tool2_name;
		d_tool_items.push_back(tool_item);
	}
		
	command_query_tool = new TCommandQueryTool;
	command_query_tool->Set_Name(d_parameters.roll_tool1_name);
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_query_tool));
	
	command_query_tool = new TCommandQueryTool;
	command_query_tool->Set_Name(d_parameters.roll_tool2_name);
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_query_tool));
	
	// manmove
	command_mode = new TCommandMode();
	command_mode->Set_Manual_Mode();
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));

	// sync
	// query tools must be completed before generating next steps
	command_sync = new TCommandSync();
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_sync));

	return true;
}

bool TMeasureRoutines::Change_Tool_Roll(void)
{
	TCommandPrompt						*command_prompt;
	TCommandSetTool						*command_change_tool;
	TCommandAlign						*command_align;
	TCommandAlign::TAlignStep			align_step;
	TToolWidget::TToolItem				tool_item;
	TToolWidget::TToolItem				tool_item_second;
	bool								valid;
				
	switch(d_active_roll_sphere)
	{
		case MEMORY_SPHERE1_ROLL_DCC:
			// prompt
			command_prompt = new TCommandPrompt(QStringLiteral("Measure 1 point on lower sphere normal to probe."));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));

			tool_item_second = this->Get_Tool_Item(d_parameters.roll_tool2_name,&valid);
			
			if(valid)
			{
				tool_item = this->Get_Tool_Item(d_parameters.roll_tool1_name,&valid);
			}
		
			if(!valid || (tool_item.ijk ^ tool_item_second.ijk) > -0.991444861374)	// 7.5 degrees
			{
				emit Add_Log_Text(QStringLiteral("ERR:  Tool angles must be opposing."));
				return false;
			}
			break;
			
		case MEMORY_SPHERE4_ROLL_DCC:
			// prompt
			command_prompt = new TCommandPrompt(QStringLiteral("Measure 1 point on upper sphere normal to probe."));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));

			tool_item = this->Get_Tool_Item(d_parameters.roll_tool1_name,&valid);
			break;
			
		case MEMORY_SPHERE2_ROLL_DCC:
			// prompt
			command_prompt = new TCommandPrompt(QStringLiteral("Measure 1 point on lower sphere normal to probe."));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));

			tool_item = this->Get_Tool_Item(d_parameters.roll_tool2_name,&valid);
			break;
			
		case MEMORY_SPHERE3_ROLL_DCC:
			// prompt
			command_prompt = new TCommandPrompt(QStringLiteral("Measure 1 point on upper sphere normal to probe."));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));

			tool_item = this->Get_Tool_Item(d_parameters.roll_tool2_name,&valid);
			break;
	}

	if(!valid)
	{
		emit Add_Log_Text(QString("ERR:  Tool data for '%1' not available.").arg(d_parameters.roll_tool1_name));
		return false;
	}

	command_change_tool = new TCommandSetTool;
	
	command_change_tool->Set_Name(tool_item.name);
	command_change_tool->Set_Offset(tool_item.xyz);
	command_change_tool->Set_Vector(tool_item.ijk);
	command_change_tool->Set_Angle_A(tool_item.angle_a);
	command_change_tool->Set_Angle_B(tool_item.angle_b);
	command_change_tool->Set_Tip_Diameter(tool_item.tip_diameter);

	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_change_tool));

	// initial alignment
	command_align = new TCommandAlign;
	align_step.command_function = TCommandAlign::TYPE_SKEW1_PROBE_AXIS;
	align_step.memory_id = -1;
	align_step.axis = TReferenceSystem::ZMINUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_SKEW2_STEM_AXIS;
	align_step.memory_id = -1;
	align_step.axis = TReferenceSystem::XMINUS;
	align_step.projection_axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	
	return true;
}

bool TMeasureRoutines::Move_Clear_Roll(void)
{
	TCommandMove						*command_move;
	TCommandMoveCube					*command_move_cube;
	TVector3							v1;
	double								sphere_nominal_diameter;
	double								sphere_radius;
	double								clear_dist;
	double								clear_dist_probe_axis;
	double								tip_radius;
	
	sphere_nominal_diameter = d_parameters.sphere_nominal_diameter;
	
	sphere_radius = sphere_nominal_diameter / 2.0;
	tip_radius = d_active_tool.tip_diameter / 2.0;
	
	clear_dist = d_approach_distance;
	clear_dist += tip_radius;
	clear_dist += sphere_radius;
	
	clear_dist_probe_axis = clear_dist + 50.0;
	
	// clearance moves
	v1 = TVector3(0,0,clear_dist_probe_axis);

	command_move_cube = new TCommandMoveCube;
	command_move_cube->Set_Axis(TCommandMoveCube::ZPLUS,d_artifact_settings.roll_clear_z);
	command_move_cube->Add_Point(v1);
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move_cube));
	
	command_move = new TCommandMove;
	command_move->Add_Point(v1);
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));

	return true;
}

bool TMeasureRoutines::Align_Stepgauge_Manual(void)
{
	TCommandMode						*command_mode;
	TCommandPrompt						*command_prompt;
	TCommandPlane						*command_plane;
	TCommandLine						*command_line;
	TCommandPoint						*command_point;
	TCommandAlign						*command_align;
	TCommandAlign::TAlignStep			align_step;
		
	// manmove
	command_mode = new TCommandMode();
	command_mode->Set_Manual_Mode();
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	
	// prompt
	command_prompt = new TCommandPrompt(QStringLiteral("Measure 3 points on reference plane of step gauge"));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));
	
	// plane
	command_plane = new TCommandPlane(MEMORY_STEPGAUGE_PLANE_MANUAL);
	command_plane->Add_Path_Touch_Point(TVector3(0,0,0),TVector3(0,0,-1));
	command_plane->Add_Path_Touch_Point(TVector3(1000,0,0),TVector3(0,0,-1));
	command_plane->Add_Path_Touch_Point(TVector3(1000,10,0),TVector3(0,0,-1));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_plane));
	
	// alignment
	command_align = new TCommandAlign;
	
	align_step.command_function = TCommandAlign::TYPE_SKEW1;
	align_step.memory_id = MEMORY_STEPGAUGE_PLANE_MANUAL;
	align_step.axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_STEPGAUGE_PLANE_MANUAL;
	align_step.axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	
	// prompt
	command_prompt = new TCommandPrompt(QStringLiteral("Measure 2 points on reference line of step gauge"));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));
	
	// line
	command_line = new TCommandLine(MEMORY_STEPGAUGE_LINE_MANUAL);
	command_line->Add_Path_Touch_Point(TVector3(0,0,-4),TVector3(0,1,0));
	command_line->Add_Path_Touch_Point(TVector3(1000,0,-4),TVector3(0,1,0));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_line));
	
	// alignment
	command_align = new TCommandAlign;
	
	align_step.command_function = TCommandAlign::TYPE_SKEW2;
	align_step.memory_id = MEMORY_STEPGAUGE_LINE_MANUAL;
	align_step.axis = TReferenceSystem::XPLUS;
	align_step.projection_axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_STEPGAUGE_LINE_MANUAL;
	align_step.axis = TReferenceSystem::YPLUS;
	command_align->Add_Align_Step(align_step);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	
	if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_B89)
	{
		// prompt
		command_prompt = new TCommandPrompt(QString("Measure 1 point on first step of step gauge at position %1 mm").arg(d_parameters.stepgauge_measurement_b89_start,0,'f',1));
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));
		
		// point start
		command_point = new TCommandPoint(MEMORY_STEPGAUGE_POINT_START_MANUAL);
		command_point->Add_Path_Touch_Point(TVector3(0,4,-4),TVector3(-1,0,0));
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
		
		// alignment
		command_align = new TCommandAlign;
		
		align_step.command_function = TCommandAlign::TYPE_ORIGIN;
		align_step.memory_id = MEMORY_STEPGAUGE_POINT_START_MANUAL;
		align_step.axis = TReferenceSystem::XPLUS;
		command_align->Add_Align_Step(align_step);
		
		align_step.command_function = TCommandAlign::TYPE_ORIGIN_OFFSET;
		align_step.memory_id = -1;
		align_step.value = (-1.0 * d_parameters.stepgauge_measurement_b89_start);
		align_step.axis = TReferenceSystem::XPLUS;
		command_align->Add_Align_Step(align_step);
		
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	}
	else
	{
		// prompt
		command_prompt = new TCommandPrompt(QStringLiteral("Measure 1 point on first step of step gauge"));
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));
		
		// point start
		command_point = new TCommandPoint(MEMORY_STEPGAUGE_POINT_START_MANUAL);
		command_point->Add_Path_Touch_Point(TVector3(0,4,-4),TVector3(1,0,0));
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
		
		// alignment
		command_align = new TCommandAlign;
		
		align_step.command_function = TCommandAlign::TYPE_ORIGIN;
		align_step.memory_id = MEMORY_STEPGAUGE_POINT_START_MANUAL;
		align_step.axis = TReferenceSystem::XPLUS;
		command_align->Add_Align_Step(align_step);
		
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	}
	
	// prompt
	command_prompt = new TCommandPrompt(QStringLiteral("Measure 1 point on last step of step gauge"));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));
	
	// point end
	command_point = new TCommandPoint(MEMORY_STEPGAUGE_POINT_END_MANUAL);
	command_point->Add_Path_Touch_Point(TVector3(1010,4,-4),TVector3(-1,0,0));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
	
	if(d_parameters.stepgauge_type == static_cast<int>(MeasureTypes::STEPGAUGE_STARRET))
	{
		// alignment change.  Set end point as zero in Y/Z
		command_align = new TCommandAlign;
		
		align_step.command_function = TCommandAlign::TYPE_ORIGIN;
		align_step.memory_id = MEMORY_STEPGAUGE_POINT_END_MANUAL;
		align_step.axis = TReferenceSystem::YPLUS;
		command_align->Add_Align_Step(align_step);
		
		align_step.command_function = TCommandAlign::TYPE_ORIGIN;
		align_step.memory_id = MEMORY_STEPGAUGE_POINT_END_MANUAL;
		align_step.axis = TReferenceSystem::ZPLUS;
		command_align->Add_Align_Step(align_step);
		
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	}
	
	return true;
}

bool TMeasureRoutines::Align_Gaugeblock_Manual(void)
{
	TCommandMode						*command_mode;
	TCommandPrompt						*command_prompt;
	TCommandPlane						*command_plane;
	TCommandAlign						*command_align;
	TCommandAlign::TAlignStep			align_step;
	double								nominal_length;
	
	nominal_length = d_parameters.gaugeblock_nominal_length;
	
	if(nominal_length < MIN_GAUGEBLOCK_LENGTH)
	{
		emit Add_Log_Text(QString("ERR:  Gauge Block nominal length too small.  Min length: %1").arg(MIN_GAUGEBLOCK_LENGTH,0,'f',3));
		return false;
	}
	
	if(d_artifact_settings.gaugeblock_pln_sp < MIN_PLANE_SPACER)
	{
		emit Add_Log_Text(QString("ERR:  Gauge Block plane spacer too small.  Min plane spacer: %1").arg(MIN_PLANE_SPACER,0,'f',3));
		return false;
	}
	
	// manmove
	command_mode = new TCommandMode();
	command_mode->Set_Manual_Mode();
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	
	// prompt
	command_prompt = new TCommandPrompt(QStringLiteral("Measure 3 points on measurement face of gauge block"));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));
	
	// plane
	command_plane = new TCommandPlane(MEMORY_GAUGEBLOCK_PLANE_MANUAL);
	command_plane->Add_Path_Touch_Point(TVector3(0,0,3),TVector3(-1,0,0));
	command_plane->Add_Path_Touch_Point(TVector3(0,-5,-3),TVector3(-1,0,0));
	command_plane->Add_Path_Touch_Point(TVector3(0,5,-3),TVector3(-1,0,0));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_plane));
	
	// alignment
	command_align = new TCommandAlign;
	
	align_step.command_function = TCommandAlign::TYPE_SKEW1;
	align_step.memory_id = MEMORY_GAUGEBLOCK_PLANE_MANUAL;
	align_step.axis = TReferenceSystem::XMINUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_SKEW2_PROBE_AXIS;
	align_step.memory_id = -1;
	align_step.axis = TReferenceSystem::ZMINUS;
	align_step.projection_axis = TReferenceSystem::XMINUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_GAUGEBLOCK_PLANE_MANUAL;
	align_step.axis = TReferenceSystem::XPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_GAUGEBLOCK_PLANE_MANUAL;
	align_step.axis = TReferenceSystem::YPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_GAUGEBLOCK_PLANE_MANUAL;
	align_step.axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	
	return true;
}

bool TMeasureRoutines::Align_Point_Manual(void)
{
	TCommandMode						*command_mode;
	TCommandPrompt						*command_prompt;
	TCommandPlane						*command_plane;
	TCommandAlign						*command_align;
	TCommandAlign::TAlignStep			align_step;
	double								move_distance;
	
	move_distance = d_parameters.point_repeat_move_distance;
	
	if(move_distance < MIN_MOVE_DISTANCE)
	{
		emit Add_Log_Text(QString("ERR:  Move distance is too small.  Minimum move distance: %1").arg(MIN_MOVE_DISTANCE,0,'f',3));
		return false;
	}
	
	if(d_artifact_settings.plane_sample_radius < MIN_PLANE_SPACER)
	{
		emit Add_Log_Text(QString("ERR:  Point sample plane spacer too small.  Min plane spacer: %1").arg(MIN_PLANE_SPACER,0,'f',3));
		return false;
	}
	
	// manmove
	command_mode = new TCommandMode();
	command_mode->Set_Manual_Mode();
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	
	// prompt
	command_prompt = new TCommandPrompt(QStringLiteral("Measure 3 points on the datum face."));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));
	
	// plane
	command_plane = new TCommandPlane(MEMORY_PLANE_REPEAT_MANUAL);
	command_plane->Add_Path_Touch_Point(TVector3(0,0,3),TVector3(-1,0,0));
	command_plane->Add_Path_Touch_Point(TVector3(0,-5,-3),TVector3(-1,0,0));
	command_plane->Add_Path_Touch_Point(TVector3(0,5,-3),TVector3(-1,0,0));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_plane));
	
	// alignment
	command_align = new TCommandAlign;
	
	align_step.command_function = TCommandAlign::TYPE_SKEW1;
	align_step.memory_id = MEMORY_PLANE_REPEAT_MANUAL;
	align_step.axis = TReferenceSystem::XPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_SKEW2_PROBE_AXIS;
	align_step.memory_id = -1;
	align_step.axis = TReferenceSystem::ZMINUS;
	align_step.projection_axis = TReferenceSystem::XMINUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_PLANE_REPEAT_MANUAL;
	align_step.axis = TReferenceSystem::XPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_PLANE_REPEAT_MANUAL;
	align_step.axis = TReferenceSystem::YPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_PLANE_REPEAT_MANUAL;
	align_step.axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	
	return true;
}

bool TMeasureRoutines::Align_Sphere_Manual(void)
{
	TCommandMode						*command_mode;
	TCommandPrompt						*command_prompt;
	TCommandPoint						*command_point;
	TCommandAlign						*command_align;
	TCommandAlign::TAlignStep			align_step;
	double								sphere_diameter;
	double								sphere_radius;
	
	sphere_diameter = d_parameters.sphere_nominal_diameter;
	
	if(sphere_diameter < MIN_SPHERE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Sphere diameter too small.  Minimum diameter: %1").arg(MIN_SPHERE_DIAMETER,0,'f',3));
		return false;
	}
	
	sphere_radius = sphere_diameter / 2.0;
	
	// manmove
	command_mode = new TCommandMode();
	command_mode->Set_Manual_Mode();
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	
	// prompt
	command_prompt = new TCommandPrompt(QStringLiteral("Measure 1 point on the sphere normal to probe"));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));
	
	// sphere
	command_point = new TCommandPoint(MEMORY_SPHERE_POINT_MANUAL);
	command_point->Add_Path_Touch_Point(TVector3(0,0,sphere_radius),TVector3(0,0,-1));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
	
	// alignment
	command_align = new TCommandAlign;
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_SPHERE_POINT_MANUAL;
	align_step.axis = TReferenceSystem::XPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_SPHERE_POINT_MANUAL;
	align_step.axis = TReferenceSystem::YPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_SPHERE_POINT_MANUAL;
	align_step.axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN_OFFSET;
	align_step.memory_id = -1;
	align_step.value = (-1.0 * sphere_radius);
	align_step.axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	
	return true;
}

bool TMeasureRoutines::Align_Sphere_Manual_Calibration(void)
{
	TCommandMode						*command_mode;
	TCommandPrompt						*command_prompt;
	TCommandPoint						*command_point;
	TCommandSetTool						*command_settool;
	TCommandAlign						*command_align;
	TCommandAlign::TAlignStep			align_step;
	double								sphere_diameter;
	double								sphere_radius;
	
	sphere_diameter = d_parameters.sphere_nominal_diameter;
	
	if(sphere_diameter < MIN_SPHERE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Sphere diameter too small.  Minimum diameter: %1").arg(MIN_SPHERE_DIAMETER,0,'f',3));
		return false;
	}

	if(!d_calibration_tool_items.size())
	{
		emit Add_Log_Text(QStringLiteral("ERR:  No probes selected for calibration."));
		return false;
	}

	sphere_radius = sphere_diameter / 2.0;
	
	// manmove
	command_mode = new TCommandMode();
	command_mode->Set_Manual_Mode();
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	
	// set tool
	command_settool = new TCommandSetTool;
	command_settool->Set_Name(d_calibration_tool_items[0].name);
	command_settool->Set_Offset(d_calibration_tool_items[0].xyz);
	command_settool->Set_Vector(d_calibration_tool_items[0].ijk);
	command_settool->Set_Angle_A(d_calibration_tool_items[0].angle_a);
	command_settool->Set_Angle_B(d_calibration_tool_items[0].angle_b);
	command_settool->Set_Tip_Diameter(d_calibration_tool_items[0].tip_diameter);
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_settool));
	
	// initial alignment
	command_align = new TCommandAlign;
	align_step.command_function = TCommandAlign::TYPE_SKEW1_PROBE_AXIS;
	align_step.memory_id = -1;
	align_step.axis = TReferenceSystem::ZMINUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_SKEW2_STEM_AXIS;
	align_step.memory_id = -1;
	align_step.axis = TReferenceSystem::XMINUS;
	align_step.projection_axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	
	// prompt
	command_prompt = new TCommandPrompt(QStringLiteral("Measure 1 point on the sphere normal to probe"));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));
	
	// sphere point normal to probe
	command_point = new TCommandPoint(MEMORY_SPHERE_POINT_MANUAL);
	command_point->Add_Path_Touch_Point(TVector3(0,0,sphere_radius),TVector3(0,0,-1));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
	
	// alignment
	command_align = new TCommandAlign;
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_SPHERE_POINT_MANUAL;
	align_step.axis = TReferenceSystem::XPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_SPHERE_POINT_MANUAL;
	align_step.axis = TReferenceSystem::YPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_SPHERE_POINT_MANUAL;
	align_step.axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN_OFFSET;
	align_step.memory_id = -1;
	align_step.value = (-1.0 * sphere_radius);
	align_step.axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	
	return true;
}

bool TMeasureRoutines::Align_Ballbar_B89_Manual(void)
{
	TCommandMode						*command_mode;
	TCommandPrompt						*command_prompt;
	TCommandPoint						*command_point;
	double								sphere_diameter;
	double								sphere_radius;
	
	sphere_diameter = d_parameters.ballbar_sphere_diameter;
	
	if(sphere_diameter < MIN_SPHERE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Ballbar sphere diameter too small.  Minimum diameter: %1").arg(MIN_SPHERE_DIAMETER,0,'f',3));
		return false;
	}
	
	sphere_radius = sphere_diameter / 2.0;
	
	this->Remove_Feature_Memory(MEMORY_BALLBAR_SPHERE1_DCC);
	this->Remove_Feature_Memory(MEMORY_BALLBAR_SPHERE2_DCC);

	// manmove
	command_mode = new TCommandMode();
	command_mode->Set_Manual_Mode();
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	
	this->Remove_Feature_Memory(MEMORY_BALLBAR_SPHERE1_DCC);
	this->Remove_Feature_Memory(MEMORY_BALLBAR_SPHERE2_DCC);

	// prompt
	command_prompt = new TCommandPrompt(QStringLiteral("Measure 1 point on first sphere normal to probe"));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));
	
	// sphere point
	command_point = new TCommandPoint(MEMORY_BALLBAR_POINT1_MANUAL);
	command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
	command_point->Add_Path_Touch_Point(TVector3(0,0,sphere_radius),TVector3(0,0,-1));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
	
	// prompt
	command_prompt = new TCommandPrompt(QStringLiteral("Measure 1 point on second sphere normal to probe"));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));
	
	// sphere point
	command_point = new TCommandPoint(MEMORY_BALLBAR_POINT2_MANUAL);
	command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
	command_point->Add_Path_Touch_Point(TVector3(1000,0,sphere_radius),TVector3(0,0,-1));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
	
	return true;
}

bool TMeasureRoutines::Align_Ballbar_10360_Manual(void)
{
	TCommandMode						*command_mode;
	TCommandPrompt						*command_prompt;
	TCommandPoint						*command_point;
	double								sphere_diameter;
	double								sphere_radius;
	
	sphere_diameter = d_parameters.ballbar_10360_sphere_diameter;
	
	if(sphere_diameter < MIN_SPHERE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Ballbar 10360 sphere diameter too small.  Minimum diameter: %1").arg(MIN_SPHERE_DIAMETER,0,'f',3));
		return false;
	}
	
	sphere_radius = sphere_diameter / 2.0;
	
	// manmove
	command_mode = new TCommandMode();
	command_mode->Set_Manual_Mode();
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	
	this->Remove_Feature_Memory(MEMORY_BALLBAR_10360_SPHERE0_DCC);
	this->Remove_Feature_Memory(MEMORY_BALLBAR_10360_SPHERE5_DCC);

	// prompt
	command_prompt = new TCommandPrompt(QStringLiteral("Measure 1 point on first sphere normal to probe"));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));
	
	// sphere point
	command_point = new TCommandPoint(MEMORY_BALLBAR_10360_POINT0_MANUAL);
	command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
	command_point->Add_Path_Touch_Point(TVector3(0,0,sphere_radius),TVector3(0,0,-1));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
	
	// prompt
	command_prompt = new TCommandPrompt(QStringLiteral("Measure 1 point on last sphere normal to probe"));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));
	
	// sphere point
	command_point = new TCommandPoint(MEMORY_BALLBAR_10360_POINT5_MANUAL);
	command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
	command_point->Add_Path_Touch_Point(TVector3(1000,0,sphere_radius),TVector3(0,0,-1));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
	
	return true;
}

bool TMeasureRoutines::Align_Ringgauge_Manual(void)
{
	TCommandMode						*command_mode;
	TCommandPrompt						*command_prompt;
	TCommandPlane						*command_plane;
	TCommandCircle						*command_circle;
	TCommandAlign						*command_align;
	TCommandAlign::TAlignStep			align_step;
	double								ringgauge_nominal_diameter;
	double								ringgauge_radius;
	
	ringgauge_nominal_diameter = d_parameters.ringgauge_nominal_diameter;
	
	if(ringgauge_nominal_diameter < MIN_RINGGAUGE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Ring gauge diameter too small.  Minimum diameter: %1").arg(MIN_RINGGAUGE_DIAMETER,0,'f',3));
		return false;
	}
	
	ringgauge_radius = ringgauge_nominal_diameter / 2.0;
	
	// manmove
	command_mode = new TCommandMode();
	command_mode->Set_Manual_Mode();
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	
	// prompt
	command_prompt = new TCommandPrompt(QStringLiteral("Measure 3 points on the datum face of the ring gauge"));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));
	
	// plane
	command_plane = new TCommandPlane(MEMORY_RINGGAUGE_PLANE_MANUAL);
	command_plane->Add_Path_Touch_Point(TVector3(0,0,0),TVector3(0,0,-1));
	command_plane->Add_Path_Touch_Point(TVector3(10,0,0),TVector3(0,0,-1));
	command_plane->Add_Path_Touch_Point(TVector3(10,10,0),TVector3(0,0,-1));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_plane));
	
	// alignment
	command_align = new TCommandAlign;
	
	align_step.command_function = TCommandAlign::TYPE_SKEW1;
	align_step.memory_id = MEMORY_RINGGAUGE_PLANE_MANUAL;
	align_step.axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_RINGGAUGE_PLANE_MANUAL;
	align_step.axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	
	
	// prompt
	command_prompt = new TCommandPrompt(QStringLiteral("Measure 4 points on the ring gauge diameter"));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));
	
	// circle
	command_circle = new TCommandCircle(MEMORY_RINGGAUGE_CIRCLE_MANUAL);
	command_circle->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_ID);
	command_circle->Set_Nominal_Projection_Axis(TVector3(0,0,1));
	
	command_circle->Add_Path_Touch_Point(TVector3(ringgauge_radius,0,d_artifact_settings.ringgauge_cir_z),TVector3(1,0,0));
	command_circle->Add_Path_Touch_Point(TVector3(0,ringgauge_radius,d_artifact_settings.ringgauge_cir_z),TVector3(0,1,0));
	command_circle->Add_Path_Touch_Point(TVector3(-ringgauge_radius,5,d_artifact_settings.ringgauge_cir_z),TVector3(-1,0,0));
	command_circle->Add_Path_Touch_Point(TVector3(0,-ringgauge_radius,d_artifact_settings.ringgauge_cir_z),TVector3(0,-1,0));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_circle));
	
	// alignment
	command_align = new TCommandAlign;
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_RINGGAUGE_CIRCLE_MANUAL;
	align_step.axis = TReferenceSystem::XPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_RINGGAUGE_CIRCLE_MANUAL;
	align_step.axis = TReferenceSystem::YPLUS;
	command_align->Add_Align_Step(align_step);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	
	return true;
}

bool TMeasureRoutines::Align_Pingauge_Manual(void)
{
	TCommandMode						*command_mode;
	TCommandPrompt						*command_prompt;
	TCommandPlane						*command_plane;
	TCommandCircle						*command_circle;
	TCommandAlign						*command_align;
	TCommandAlign::TAlignStep			align_step;
	double								pingauge_nominal_diameter;
	double								pingauge_radius;
	
	pingauge_nominal_diameter = d_parameters.pingauge_nominal_diameter;
	
	if(pingauge_nominal_diameter < MIN_PINGAUGE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Pin gauge diameter too small.  Minimum diameter: %1").arg(MIN_PINGAUGE_DIAMETER,0,'f',3));
		return false;
	}
	
	pingauge_radius = pingauge_nominal_diameter / 2.0;
	
	// manmove
	command_mode = new TCommandMode();
	command_mode->Set_Manual_Mode();
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	
	// prompt
	command_prompt = new TCommandPrompt(QStringLiteral("Measure 3 points on the datum face of the pin gauge"));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));
	
	// plane
	command_plane = new TCommandPlane(MEMORY_PINGAUGE_PLANE_MANUAL);
	command_plane->Add_Path_Touch_Point(TVector3(0,0,0),TVector3(0,0,-1));
	command_plane->Add_Path_Touch_Point(TVector3(10,0,0),TVector3(0,0,-1));
	command_plane->Add_Path_Touch_Point(TVector3(10,10,0),TVector3(0,0,-1));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_plane));
	
	// alignment
	command_align = new TCommandAlign;
	
	align_step.command_function = TCommandAlign::TYPE_SKEW1;
	align_step.memory_id = MEMORY_PINGAUGE_PLANE_MANUAL;
	align_step.axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_PINGAUGE_PLANE_MANUAL;
	align_step.axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	
	
	// prompt
	command_prompt = new TCommandPrompt(QStringLiteral("Measure 4 points on the pin gauge diameter"));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_prompt));
	
	// circle
	command_circle = new TCommandCircle(MEMORY_PINGAUGE_CIRCLE_MANUAL);
	command_circle->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_OD);
	command_circle->Set_Nominal_Projection_Axis(TVector3(0,0,1));
	
	command_circle->Add_Path_Touch_Point(TVector3(pingauge_radius,0,d_artifact_settings.pingauge_cir_z),TVector3(-1,0,0));
	command_circle->Add_Path_Touch_Point(TVector3(0,pingauge_radius,d_artifact_settings.pingauge_cir_z),TVector3(0,-1,0));
	command_circle->Add_Path_Touch_Point(TVector3(-pingauge_radius,5,d_artifact_settings.pingauge_cir_z),TVector3(1,0,0));
	command_circle->Add_Path_Touch_Point(TVector3(0,-pingauge_radius,d_artifact_settings.pingauge_cir_z),TVector3(0,1,0));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_circle));
	
	// alignment
	command_align = new TCommandAlign;
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_PINGAUGE_CIRCLE_MANUAL;
	align_step.axis = TReferenceSystem::XPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_PINGAUGE_CIRCLE_MANUAL;
	align_step.axis = TReferenceSystem::YPLUS;
	command_align->Add_Align_Step(align_step);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	
	return true;
}

bool TMeasureRoutines::Align_Stepgauge_DCC(void)
{
	TMeasureRoutines::TFeatureMemory	feature_memory;
	TCommandFeature						*command_feature;
	TCommandMode						*command_mode;
	TCommandPlane						*command_plane;
	TCommandLine						*command_line;
	TCommandPoint						*command_point;
	TCommandMove						*command_move;
	TCommandDimension					*command_dimension;
	TCommandAlign						*command_align;
	TCommandAlign::TAlignStep			align_step;
	TVector3							pnt;
	std::vector<TVector3>				points;
	double								clear_xy;
	double								measure_length(0.0);
	double								minimum_length;
	double								dval;
	double								starrett_clear_z;
	double								koba_clear_z;
	double								tip_radius;
	bool								valid;

	emit Update_Last_Configuration();
	
	tip_radius = d_active_tool.tip_diameter / 2.0;
	
	feature_memory = this->Feature_Memory(MEMORY_STEPGAUGE_POINT_END_MANUAL,&valid);
	
	if(valid)
	{
		pnt = d_refsys.FromWorld(feature_memory.xyz);
		pnt.y = 0;
		pnt.z = 0;
		measure_length = pnt.Length();
	}

	minimum_length = d_parameters.stepgauge_zero_step_spacing + d_parameters.stepgauge_nominal_block_spacing * 5.0;
	
	if(measure_length < minimum_length)
	{
		emit Add_Log_Text(QString("ERR:  Stepgauge too short for required measurements.  Actual length: %1  Minimum length: %2").arg(measure_length,0,'f',3).arg(minimum_length,0,'f',3));
		
		return false;
	}
	
	// get manual point locations if it happens to be a starrett gauge
	// plane, line, and start point
	// end point is at YZ zero
	command_feature = d_command_sequence.Get_Feature_Command(MEMORY_STEPGAUGE_PLANE_MANUAL);
	
	if(command_feature)
	{
		points = command_feature->Touch_Points();
		
		if(points.size() == 3)
		{
			d_stepgauge_manual_points[0] = d_refsys.FromWorld(points[0]) + TVector3(0,0,-1) * tip_radius;
			d_stepgauge_manual_points[1] = d_refsys.FromWorld(points[1]) + TVector3(0,0,-1) * tip_radius;
			d_stepgauge_manual_points[2] = d_refsys.FromWorld(points[2]) + TVector3(0,0,-1) * tip_radius;
		}
	}
	
	command_feature = d_command_sequence.Get_Feature_Command(MEMORY_STEPGAUGE_LINE_MANUAL);
	
	if(command_feature)
	{
		points = command_feature->Touch_Points();
		
		if(points.size() == 2)
		{
			d_stepgauge_line_approach_vector = d_refsys.AxisFromWorld(command_feature->Touch_Point_Approach_Vector());
			
			d_stepgauge_manual_points[3] = d_refsys.FromWorld(points[0]) + d_stepgauge_line_approach_vector * tip_radius;
			d_stepgauge_manual_points[4] = d_refsys.FromWorld(points[1]) + d_stepgauge_line_approach_vector * tip_radius;
		}
	}
	
	command_feature = d_command_sequence.Get_Feature_Command(MEMORY_STEPGAUGE_POINT_START_MANUAL);
	
	if(command_feature)
	{
		points = command_feature->Touch_Points();
		
		if(points.size() == 1)
		{
			d_stepgauge_manual_points[5] = d_refsys.FromWorld(points[0]) + TVector3(1,0,0) * tip_radius;
		}
	}
	
	clear_xy = d_approach_distance;
	clear_xy += tip_radius;
	
	// dcc mode
	command_mode = new TCommandMode();
	command_mode->Set_DCC_Mode();
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	
	if(d_parameters.stepgauge_type == static_cast<int>(MeasureTypes::STEPGAUGE_CHECKMASTER))
	{
		dval = fabs(d_artifact_settings.sg_checkmaster_pln_y1 - d_artifact_settings.sg_checkmaster_pln_y2);
		
		if(dval < MIN_STEPGAUGE_PLANE_POINT_SPACING_Y)
		{
			emit Add_Log_Text(QString("ERR:  Stepgauge plane spacing tool small.  Minimum spacing: %1").arg(MIN_STEPGAUGE_PLANE_POINT_SPACING_Y,0,'f',3));
			return false;
		}
		
		// plane
		command_plane = new TCommandPlane(MEMORY_STEPGAUGE_PLANE_DCC);
		
		command_plane->Add_Path_Touch_Point(TVector3(measure_length + d_artifact_settings.sg_checkmaster_pln_x_end,d_artifact_settings.sg_checkmaster_pln_y1,0),
											TVector3(0,0,-1));
		command_plane->Add_Path_Touch_Point(TVector3(measure_length + d_artifact_settings.sg_checkmaster_pln_x_end,d_artifact_settings.sg_checkmaster_pln_y2,0),
											TVector3(0,0,-1));
		
		command_plane->Add_Path_Move_Point(TVector3(measure_length + d_artifact_settings.sg_checkmaster_pln_x_end,d_artifact_settings.sg_checkmaster_pln_y2,d_artifact_settings.sg_checkmaster_clear_z));
		command_plane->Add_Path_Move_Point(TVector3(d_artifact_settings.sg_checkmaster_pln_x_start,d_artifact_settings.sg_checkmaster_pln_y2,d_artifact_settings.sg_checkmaster_clear_z));
		
		command_plane->Add_Path_Touch_Point(TVector3(d_artifact_settings.sg_checkmaster_pln_x_start,d_artifact_settings.sg_checkmaster_pln_y2,0),
											TVector3(0,0,-1));
		command_plane->Add_Path_Touch_Point(TVector3(d_artifact_settings.sg_checkmaster_pln_x_start,d_artifact_settings.sg_checkmaster_pln_y1,0),
											TVector3(0,0,-1));
		
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_plane));
		
		// move points
		command_move = new TCommandMove;
		command_move->Add_Point(TVector3(d_artifact_settings.sg_checkmaster_pln_x_start, d_artifact_settings.sg_checkmaster_pln_y1, d_artifact_settings.sg_checkmaster_clear_z));
		command_move->Add_Point(TVector3(d_artifact_settings.sg_checkmaster_ln_x_start, -clear_xy, d_artifact_settings.sg_checkmaster_clear_z));
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		
		// alignment
		command_align = new TCommandAlign;
		
		align_step.command_function = TCommandAlign::TYPE_SKEW1;
		align_step.memory_id = MEMORY_STEPGAUGE_PLANE_DCC;
		align_step.axis = TReferenceSystem::ZPLUS;
		command_align->Add_Align_Step(align_step);
		
		align_step.command_function = TCommandAlign::TYPE_ORIGIN;
		align_step.memory_id = MEMORY_STEPGAUGE_PLANE_DCC;
		align_step.axis = TReferenceSystem::ZPLUS;
		command_align->Add_Align_Step(align_step);
		
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
		
		// line
		// measured in reverse order from manual version (efficient)
		command_line = new TCommandLine(MEMORY_STEPGAUGE_LINE_DCC);
		
		
		command_line->Add_Path_Touch_Point(TVector3(d_artifact_settings.sg_checkmaster_ln_x_start,0,d_artifact_settings.sg_checkmaster_ln_z)
										   ,TVector3(0,1,0));
		
		command_line->Add_Path_Move_Point(TVector3(d_artifact_settings.sg_checkmaster_ln_x_start,-clear_xy,d_artifact_settings.sg_checkmaster_clear_z));
		command_line->Add_Path_Move_Point(TVector3(measure_length + d_artifact_settings.sg_checkmaster_ln_x_end,-clear_xy,d_artifact_settings.sg_checkmaster_clear_z));
		
		command_line->Add_Path_Touch_Point(TVector3(measure_length + d_artifact_settings.sg_checkmaster_ln_x_end,0,d_artifact_settings.sg_checkmaster_ln_z),
										   TVector3(0,1,0));
		
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_line));
		
		// alignment
		command_align = new TCommandAlign;
		
		align_step.command_function = TCommandAlign::TYPE_SKEW2;
		align_step.memory_id = MEMORY_STEPGAUGE_LINE_DCC;
		align_step.axis = TReferenceSystem::XPLUS;
		align_step.projection_axis = TReferenceSystem::ZPLUS;
		command_align->Add_Align_Step(align_step);
		
		align_step.command_function = TCommandAlign::TYPE_ORIGIN;
		align_step.memory_id = MEMORY_STEPGAUGE_LINE_DCC;
		align_step.axis = TReferenceSystem::YPLUS;
		command_align->Add_Align_Step(align_step);
		
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
		
		
		// move points
		command_move = new TCommandMove;
		command_move->Add_Point(TVector3(measure_length + d_artifact_settings.sg_checkmaster_ln_x_end, -clear_xy, d_artifact_settings.sg_checkmaster_clear_z));
		command_move->Add_Point(TVector3(measure_length + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		
		// point end
		command_point = new TCommandPoint(MEMORY_STEPGAUGE_POINT_END_DCC);
		command_point->Add_Path_Touch_Point(TVector3(measure_length,d_artifact_settings.sg_checkmaster_pnt_y,d_artifact_settings.sg_checkmaster_pnt_z),
											TVector3(-1,0,0));
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));

		if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_B89)
		{
			// move points
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(measure_length + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			command_move->Add_Point(TVector3(d_parameters.stepgauge_measurement_b89_start + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// point start
			command_point = new TCommandPoint(MEMORY_STEPGAUGE_POINT_START_DCC);
			command_point->Add_Path_Touch_Point(TVector3(d_parameters.stepgauge_measurement_b89_start,d_artifact_settings.sg_checkmaster_pnt_y,d_artifact_settings.sg_checkmaster_pnt_z),
												TVector3(-1,0,0));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			// alignment
			command_align = new TCommandAlign;
			
			align_step.command_function = TCommandAlign::TYPE_ORIGIN;
			align_step.memory_id = MEMORY_STEPGAUGE_POINT_START_DCC;
			align_step.axis = TReferenceSystem::XPLUS;
			command_align->Add_Align_Step(align_step);
			
			align_step.command_function = TCommandAlign::TYPE_ORIGIN_OFFSET;
			align_step.memory_id = -1;
			align_step.value = (-1.0 * d_parameters.stepgauge_measurement_b89_start);
			align_step.axis = TReferenceSystem::XPLUS;
			command_align->Add_Align_Step(align_step);

			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
			
			// move points
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(d_parameters.stepgauge_measurement_b89_start + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));

		}
		else
		{
			// move points
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(measure_length + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			command_move->Add_Point(TVector3(-clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// point start
			command_point = new TCommandPoint(MEMORY_STEPGAUGE_POINT_START_DCC);
			command_point->Add_Path_Touch_Point(TVector3(0,d_artifact_settings.sg_checkmaster_pnt_y,d_artifact_settings.sg_checkmaster_pnt_z),
												TVector3(1,0,0));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			// alignment
			command_align = new TCommandAlign;
			
			align_step.command_function = TCommandAlign::TYPE_ORIGIN;
			align_step.memory_id = MEMORY_STEPGAUGE_POINT_START_DCC;
			align_step.axis = TReferenceSystem::XPLUS;
			command_align->Add_Align_Step(align_step);
			
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
			
			// move points
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(-clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));

		}
		
		command_dimension = new TCommandDimension;
		command_dimension->Set_Memory_Id_Feature1(MEMORY_STEPGAUGE_POINT_START_DCC);
		command_dimension->Set_Memory_Id_Feature2(MEMORY_STEPGAUGE_POINT_END_DCC);
		command_dimension->Add_Dimension(TCommandDimension::FIELD_DIST_X,QStringLiteral("Init-Len"),0.0);
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));

	}
	else if(d_parameters.stepgauge_type == static_cast<int>(MeasureTypes::STEPGAUGE_KOBA))
	{
		koba_clear_z = d_artifact_settings.sg_koba_clear_probe + d_artifact_settings.sg_koba_pnt_z;
		
		if(koba_clear_z < 10.0)
		{
			koba_clear_z = 10.0;
		}
		
		dval = fabs(d_artifact_settings.sg_koba_pln_y1 - d_artifact_settings.sg_koba_pln_y2);
		
		if(dval < MIN_STEPGAUGE_PLANE_POINT_SPACING_Y)
		{
			emit Add_Log_Text(QString("ERR:  Stepgauge plane spacing tool small.  Minimum spacing: %1").arg(MIN_STEPGAUGE_PLANE_POINT_SPACING_Y,0,'f',3));
			return false;
		}
		
		// plane
		command_plane = new TCommandPlane(MEMORY_STEPGAUGE_PLANE_DCC);
		
		command_plane->Add_Path_Touch_Point(TVector3(measure_length + d_artifact_settings.sg_koba_pln_x_end,d_artifact_settings.sg_koba_pln_y1,0),
											TVector3(0,0,-1));
		command_plane->Add_Path_Touch_Point(TVector3(measure_length + d_artifact_settings.sg_koba_pln_x_end,d_artifact_settings.sg_koba_pln_y2,0),
											TVector3(0,0,-1));
		
		command_plane->Add_Path_Move_Point(TVector3(measure_length + d_artifact_settings.sg_koba_pln_x_end,d_artifact_settings.sg_koba_pln_y2,koba_clear_z));
		command_plane->Add_Path_Move_Point(TVector3(d_artifact_settings.sg_koba_pln_x_start,d_artifact_settings.sg_koba_pln_y2,koba_clear_z));
		
		command_plane->Add_Path_Touch_Point(TVector3(d_artifact_settings.sg_koba_pln_x_start,d_artifact_settings.sg_koba_pln_y2,0),
											TVector3(0,0,-1));
		command_plane->Add_Path_Touch_Point(TVector3(d_artifact_settings.sg_koba_pln_x_start,d_artifact_settings.sg_koba_pln_y1,0),
											TVector3(0,0,-1));
		
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_plane));
		
		// move points
		command_move = new TCommandMove;
		command_move->Add_Point(TVector3(d_artifact_settings.sg_koba_pln_x_start, d_artifact_settings.sg_koba_pln_y1, koba_clear_z));
		command_move->Add_Point(TVector3(d_artifact_settings.sg_koba_ln_x_start, -clear_xy, koba_clear_z));
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		
		// alignment
		command_align = new TCommandAlign;
		
		align_step.command_function = TCommandAlign::TYPE_SKEW1;
		align_step.memory_id = MEMORY_STEPGAUGE_PLANE_DCC;
		align_step.axis = TReferenceSystem::ZPLUS;
		command_align->Add_Align_Step(align_step);
		
		align_step.command_function = TCommandAlign::TYPE_ORIGIN;
		align_step.memory_id = MEMORY_STEPGAUGE_PLANE_DCC;
		align_step.axis = TReferenceSystem::ZPLUS;
		command_align->Add_Align_Step(align_step);
		
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
		
		// line
		// measured in reverse order from manual version (efficient)
		command_line = new TCommandLine(MEMORY_STEPGAUGE_LINE_DCC);
		
		
		command_line->Add_Path_Touch_Point(TVector3(d_artifact_settings.sg_koba_ln_x_start,0,d_artifact_settings.sg_koba_ln_z)
										   ,d_stepgauge_line_approach_vector);
		
		if(d_stepgauge_line_approach_vector.y > 0)
		{
			command_line->Add_Path_Move_Point(TVector3(d_artifact_settings.sg_koba_ln_x_start,-clear_xy,koba_clear_z));
			command_line->Add_Path_Move_Point(TVector3(measure_length + d_artifact_settings.sg_koba_ln_x_end,-clear_xy,koba_clear_z));
		}
		else
		{
			command_line->Add_Path_Move_Point(TVector3(d_artifact_settings.sg_koba_ln_x_start,clear_xy,koba_clear_z));
			command_line->Add_Path_Move_Point(TVector3(measure_length + d_artifact_settings.sg_koba_ln_x_end,clear_xy,koba_clear_z));
		}
		
		command_line->Add_Path_Touch_Point(TVector3(measure_length + d_artifact_settings.sg_koba_ln_x_end,0,d_artifact_settings.sg_koba_ln_z),
										   d_stepgauge_line_approach_vector);
		
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_line));
		
		
		// alignment
		command_align = new TCommandAlign;
		
		align_step.command_function = TCommandAlign::TYPE_SKEW2;
		align_step.memory_id = MEMORY_STEPGAUGE_LINE_DCC;
		align_step.axis = TReferenceSystem::XPLUS;
		align_step.projection_axis = TReferenceSystem::ZPLUS;
		command_align->Add_Align_Step(align_step);
		
		align_step.command_function = TCommandAlign::TYPE_ORIGIN;
		align_step.memory_id = MEMORY_STEPGAUGE_LINE_DCC;
		align_step.axis = TReferenceSystem::YPLUS;
		command_align->Add_Align_Step(align_step);
		
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
		
		// move points
		command_move = new TCommandMove;
		
		if(d_stepgauge_line_approach_vector.y > 0)
		{
			command_move->Add_Point(TVector3(measure_length + d_artifact_settings.sg_koba_ln_x_end, -clear_xy, koba_clear_z));
		}
		else
		{
			command_move->Add_Point(TVector3(measure_length + d_artifact_settings.sg_koba_ln_x_end, clear_xy, koba_clear_z));
		}
		
		command_move->Add_Point(TVector3(measure_length + clear_xy, d_artifact_settings.sg_koba_pnt_y, koba_clear_z));
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		
		// point end
		command_point = new TCommandPoint(MEMORY_STEPGAUGE_POINT_END_DCC);
		command_point->Add_Path_Touch_Point(TVector3(measure_length,d_artifact_settings.sg_koba_pnt_y,d_artifact_settings.sg_koba_pnt_z),
											TVector3(-1,0,0));
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));

		if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_B89)
		{
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(measure_length + clear_xy, d_artifact_settings.sg_koba_pnt_y, koba_clear_z));
			command_move->Add_Point(TVector3(d_parameters.stepgauge_measurement_b89_start + clear_xy, d_artifact_settings.sg_koba_pnt_y, koba_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// point start
			command_point = new TCommandPoint(MEMORY_STEPGAUGE_POINT_START_DCC);
			command_point->Add_Path_Touch_Point(TVector3(d_parameters.stepgauge_measurement_b89_start,d_artifact_settings.sg_koba_pnt_y,d_artifact_settings.sg_koba_pnt_z),
												TVector3(-1,0,0));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			// alignment
			command_align = new TCommandAlign;
			
			align_step.command_function = TCommandAlign::TYPE_ORIGIN;
			align_step.memory_id = MEMORY_STEPGAUGE_POINT_START_DCC;
			align_step.axis = TReferenceSystem::XPLUS;
			command_align->Add_Align_Step(align_step);
			
			align_step.command_function = TCommandAlign::TYPE_ORIGIN_OFFSET;
			align_step.memory_id = -1;
			align_step.value = (-1.0 * d_parameters.stepgauge_measurement_b89_start);
			align_step.axis = TReferenceSystem::XPLUS;
			command_align->Add_Align_Step(align_step);

			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
			
			// move points
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(d_parameters.stepgauge_measurement_b89_start + clear_xy, d_artifact_settings.sg_koba_pnt_y, koba_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		}
		else
		{
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(measure_length + clear_xy, d_artifact_settings.sg_koba_pnt_y, koba_clear_z));
			command_move->Add_Point(TVector3(-clear_xy, d_artifact_settings.sg_koba_pnt_y, koba_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// point start
			command_point = new TCommandPoint(MEMORY_STEPGAUGE_POINT_START_DCC);
			command_point->Add_Path_Touch_Point(TVector3(0,d_artifact_settings.sg_koba_pnt_y,d_artifact_settings.sg_koba_pnt_z),
												TVector3(1,0,0));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			// alignment
			command_align = new TCommandAlign;
			
			align_step.command_function = TCommandAlign::TYPE_ORIGIN;
			align_step.memory_id = MEMORY_STEPGAUGE_POINT_START_DCC;
			align_step.axis = TReferenceSystem::XPLUS;
			command_align->Add_Align_Step(align_step);
			
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
			
			// move points
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(-clear_xy, d_artifact_settings.sg_koba_pnt_y, koba_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		}
		
		command_dimension = new TCommandDimension;
		command_dimension->Set_Memory_Id_Feature1(MEMORY_STEPGAUGE_POINT_START_DCC);
		command_dimension->Set_Memory_Id_Feature2(MEMORY_STEPGAUGE_POINT_END_DCC);
		command_dimension->Add_Dimension(TCommandDimension::FIELD_DIST_X,QStringLiteral("Init-Len"),0.0);
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
	}
	else if(d_parameters.stepgauge_type == static_cast<int>(MeasureTypes::STEPGAUGE_STARRET))
	{
		starrett_clear_z = d_artifact_settings.sg_webber_clear_z + d_stepgauge_manual_points[0].z;
		
		// plane
		command_plane = new TCommandPlane(MEMORY_STEPGAUGE_PLANE_DCC);
		
		command_plane->Add_Path_Move_Point(TVector3(d_stepgauge_manual_points[0].x,d_stepgauge_manual_points[0].y,starrett_clear_z));
		command_plane->Add_Path_Touch_Point(d_stepgauge_manual_points[0],TVector3(0,0,-1));
		command_plane->Add_Path_Move_Point(TVector3(d_stepgauge_manual_points[0].x,d_stepgauge_manual_points[0].y,starrett_clear_z));
		
		command_plane->Add_Path_Move_Point(TVector3(d_stepgauge_manual_points[1].x,d_stepgauge_manual_points[1].y,starrett_clear_z));
		command_plane->Add_Path_Touch_Point(d_stepgauge_manual_points[1],TVector3(0,0,-1));
		command_plane->Add_Path_Move_Point(TVector3(d_stepgauge_manual_points[1].x,d_stepgauge_manual_points[1].y,starrett_clear_z));
		
		command_plane->Add_Path_Move_Point(TVector3(d_stepgauge_manual_points[2].x,d_stepgauge_manual_points[2].y,starrett_clear_z));
		command_plane->Add_Path_Touch_Point(d_stepgauge_manual_points[2],TVector3(0,0,-1));
		
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_plane));
		
		// move points
		command_move = new TCommandMove;
		command_move->Add_Point(TVector3(d_stepgauge_manual_points[2].x,d_stepgauge_manual_points[2].y,starrett_clear_z));
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		
		// alignment
		command_align = new TCommandAlign;
		
		align_step.command_function = TCommandAlign::TYPE_SKEW1;
		align_step.memory_id = MEMORY_STEPGAUGE_PLANE_DCC;
		align_step.axis = TReferenceSystem::ZPLUS;
		command_align->Add_Align_Step(align_step);
		
		align_step.command_function = TCommandAlign::TYPE_ORIGIN;
		align_step.memory_id = MEMORY_STEPGAUGE_PLANE_DCC;
		align_step.axis = TReferenceSystem::ZPLUS;
		command_align->Add_Align_Step(align_step);
		
		align_step.command_function = TCommandAlign::TYPE_ORIGIN_OFFSET;
		align_step.memory_id = -1;
		align_step.value = (-1.0 * d_stepgauge_manual_points[0].z);
		align_step.axis = TReferenceSystem::ZPLUS;
		command_align->Add_Align_Step(align_step);
		
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
		
		// line
		command_line = new TCommandLine(MEMORY_STEPGAUGE_LINE_DCC);
		
		command_line->Add_Path_Move_Point(TVector3(d_stepgauge_manual_points[3].x,d_stepgauge_manual_points[3].y,starrett_clear_z) - d_stepgauge_line_approach_vector * clear_xy);
		command_line->Add_Path_Touch_Point(d_stepgauge_manual_points[3],d_stepgauge_line_approach_vector);
		command_line->Add_Path_Move_Point(TVector3(d_stepgauge_manual_points[3].x,d_stepgauge_manual_points[3].y,starrett_clear_z) - d_stepgauge_line_approach_vector * clear_xy);
		
		command_line->Add_Path_Move_Point(TVector3(d_stepgauge_manual_points[4].x,d_stepgauge_manual_points[4].y,starrett_clear_z) - d_stepgauge_line_approach_vector * clear_xy);
		command_line->Add_Path_Touch_Point(d_stepgauge_manual_points[4],d_stepgauge_line_approach_vector);
		
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_line));
		
		// alignment
		command_align = new TCommandAlign;
		
		align_step.command_function = TCommandAlign::TYPE_SKEW2;
		align_step.memory_id = MEMORY_STEPGAUGE_LINE_DCC;
		align_step.axis = TReferenceSystem::XPLUS;
		align_step.projection_axis = TReferenceSystem::ZPLUS;
		command_align->Add_Align_Step(align_step);
		
		align_step.command_function = TCommandAlign::TYPE_ORIGIN;
		align_step.memory_id = MEMORY_STEPGAUGE_LINE_DCC;
		align_step.axis = TReferenceSystem::YPLUS;
		command_align->Add_Align_Step(align_step);
		
		align_step.command_function = TCommandAlign::TYPE_ORIGIN_OFFSET;
		align_step.memory_id = -1;
		align_step.value = (-1.0 * d_stepgauge_manual_points[3].y);
		align_step.axis = TReferenceSystem::YPLUS;
		command_align->Add_Align_Step(align_step);
		
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
		
		// move points
		command_move = new TCommandMove;
		command_move->Add_Point(TVector3(d_stepgauge_manual_points[4].x,d_stepgauge_manual_points[4].y ,starrett_clear_z) - d_stepgauge_line_approach_vector * clear_xy);
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		
		// point end
		command_point = new TCommandPoint(MEMORY_STEPGAUGE_POINT_END_DCC);
		command_point->Add_Path_Move_Point(TVector3(measure_length + clear_xy,d_stepgauge_manual_points[5].y,starrett_clear_z));
		command_point->Add_Path_Touch_Point(TVector3(measure_length,d_stepgauge_manual_points[5].y,d_stepgauge_manual_points[5].z),TVector3(-1,0,0));
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));

		if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_B89)
		{
			// move points
			command_move = new TCommandMove;
			command_point->Add_Path_Move_Point(TVector3(measure_length + clear_xy,d_stepgauge_manual_points[5].y,starrett_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// point start
			command_point = new TCommandPoint(MEMORY_STEPGAUGE_POINT_START_DCC);
			command_point->Add_Path_Move_Point(TVector3(d_parameters.stepgauge_measurement_b89_start + clear_xy,d_stepgauge_manual_points[5].y,starrett_clear_z));
			command_point->Add_Path_Touch_Point(TVector3(d_parameters.stepgauge_measurement_b89_start,d_stepgauge_manual_points[5].y,d_stepgauge_manual_points[5].z),TVector3(-1,0,0));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			// alignment
			command_align = new TCommandAlign;
			
			align_step.command_function = TCommandAlign::TYPE_ORIGIN;
			align_step.memory_id = MEMORY_STEPGAUGE_POINT_START_DCC;
			align_step.axis = TReferenceSystem::XPLUS;
			command_align->Add_Align_Step(align_step);
			
			align_step.command_function = TCommandAlign::TYPE_ORIGIN_OFFSET;
			align_step.memory_id = -1;
			align_step.value = (-1.0 * d_parameters.stepgauge_measurement_b89_start);
			align_step.axis = TReferenceSystem::XPLUS;
			command_align->Add_Align_Step(align_step);

			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
			
			// move points
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(d_parameters.stepgauge_measurement_b89_start + clear_xy,d_stepgauge_manual_points[5].y,starrett_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));		}
		else
		{
			// move points
			command_move = new TCommandMove;
			command_point->Add_Path_Move_Point(TVector3(measure_length + clear_xy,d_stepgauge_manual_points[5].y,starrett_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// point start
			command_point = new TCommandPoint(MEMORY_STEPGAUGE_POINT_START_DCC);
			command_point->Add_Path_Move_Point(TVector3(-clear_xy,d_stepgauge_manual_points[5].y,starrett_clear_z));
			command_point->Add_Path_Touch_Point(TVector3(0,d_stepgauge_manual_points[5].y,d_stepgauge_manual_points[5].z),TVector3(1,0,0));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			// alignment
			command_align = new TCommandAlign;
			
			align_step.command_function = TCommandAlign::TYPE_ORIGIN;
			align_step.memory_id = MEMORY_STEPGAUGE_POINT_START_DCC;
			align_step.axis = TReferenceSystem::XPLUS;
			command_align->Add_Align_Step(align_step);
			
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
			
			// move points
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(-clear_xy,d_stepgauge_manual_points[5].y,starrett_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		}
		
		command_dimension = new TCommandDimension;
		command_dimension->Set_Memory_Id_Feature1(MEMORY_STEPGAUGE_POINT_START_DCC);
		command_dimension->Set_Memory_Id_Feature2(MEMORY_STEPGAUGE_POINT_END_DCC);
		command_dimension->Add_Dimension(TCommandDimension::FIELD_DIST_X,QStringLiteral("Init-Len"),0.0);
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
	}
	
	return true;
}

bool TMeasureRoutines::Align_Gaugeblock_DCC(void)
{
	TCommandMode						*command_mode;
	TCommandPlane						*command_plane;
	TCommandAlign						*command_align;
	TCommandMove						*command_move;
	TCommandAlign::TAlignStep			align_step;
	double								clear_xy;
	double								nominal_length;
	double								tip_radius;

	nominal_length = d_parameters.gaugeblock_nominal_length;
	
	if(nominal_length < MIN_GAUGEBLOCK_LENGTH)
	{
		emit Add_Log_Text(QString("ERR:  Gauge Block nominal length too small.  Min length: %1").arg(MIN_GAUGEBLOCK_LENGTH,0,'f',3));
		return false;
	}
	
	if(d_artifact_settings.gaugeblock_pln_sp < MIN_PLANE_SPACER)
	{
		emit Add_Log_Text(QString("ERR:  Gauge Block plane spacer too small.  Min plane spacer: %1").arg(MIN_PLANE_SPACER,0,'f',3));
		return false;
	}

	
	emit Update_Last_Configuration();
	tip_radius = d_active_tool.tip_diameter / 2.0;
	
	clear_xy = d_approach_distance;
	clear_xy += tip_radius;
	
	// dcc mode
	command_mode = new TCommandMode();
	command_mode->Set_DCC_Mode();
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	
	// move points
	command_move = new TCommandMove;
	command_move->Add_Point(TVector3(-clear_xy, 0, d_artifact_settings.gaugeblock_clear_z));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	
	// plane
	command_plane = new TCommandPlane(MEMORY_GAUGEBLOCK_PLANE_DCC);
	command_plane->Add_Path_Touch_Point(TVector3(0,d_artifact_settings.gaugeblock_pln_y - d_artifact_settings.gaugeblock_pln_sp,
												 d_artifact_settings.gaugeblock_pln_z + d_artifact_settings.gaugeblock_pln_sp),
										TVector3(1,0,0));
	command_plane->Add_Path_Touch_Point(TVector3(0,d_artifact_settings.gaugeblock_pln_y,
												 d_artifact_settings.gaugeblock_pln_z + d_artifact_settings.gaugeblock_pln_sp),
										TVector3(1,0,0));
	command_plane->Add_Path_Touch_Point(TVector3(0,d_artifact_settings.gaugeblock_pln_y + d_artifact_settings.gaugeblock_pln_sp,
												 d_artifact_settings.gaugeblock_pln_z + d_artifact_settings.gaugeblock_pln_sp),
										TVector3(1,0,0));
	command_plane->Add_Path_Touch_Point(TVector3(0,d_artifact_settings.gaugeblock_pln_y + d_artifact_settings.gaugeblock_pln_sp,
												 d_artifact_settings.gaugeblock_pln_z),
										TVector3(1,0,0));
	command_plane->Add_Path_Touch_Point(TVector3(0,d_artifact_settings.gaugeblock_pln_y,
												 d_artifact_settings.gaugeblock_pln_z),
										TVector3(1,0,0));
	command_plane->Add_Path_Touch_Point(TVector3(0,d_artifact_settings.gaugeblock_pln_y - d_artifact_settings.gaugeblock_pln_sp,
												 d_artifact_settings.gaugeblock_pln_z)
										,TVector3(1,0,0));
	command_plane->Add_Path_Touch_Point(TVector3(0,d_artifact_settings.gaugeblock_pln_y - d_artifact_settings.gaugeblock_pln_sp,
												 d_artifact_settings.gaugeblock_pln_z - d_artifact_settings.gaugeblock_pln_sp),
										TVector3(1,0,0));
	command_plane->Add_Path_Touch_Point(TVector3(0,d_artifact_settings.gaugeblock_pln_y,
												 d_artifact_settings.gaugeblock_pln_z - d_artifact_settings.gaugeblock_pln_sp)
										,TVector3(1,0,0));
	command_plane->Add_Path_Touch_Point(TVector3(0,d_artifact_settings.gaugeblock_pln_y + d_artifact_settings.gaugeblock_pln_sp,
												 d_artifact_settings.gaugeblock_pln_z - d_artifact_settings.gaugeblock_pln_sp),
										TVector3(1,0,0));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_plane));
	
	// alignment
	command_align = new TCommandAlign;
	
	align_step.command_function = TCommandAlign::TYPE_SKEW1;
	align_step.memory_id = MEMORY_GAUGEBLOCK_PLANE_DCC;
	align_step.axis = TReferenceSystem::XMINUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_SKEW2_PROBE_AXIS;
	align_step.memory_id = -1;
	align_step.axis = TReferenceSystem::ZMINUS;
	align_step.projection_axis = TReferenceSystem::XMINUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_GAUGEBLOCK_PLANE_DCC;
	align_step.axis = TReferenceSystem::XPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_GAUGEBLOCK_PLANE_DCC;
	align_step.axis = TReferenceSystem::YPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_GAUGEBLOCK_PLANE_DCC;
	align_step.axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	
	// move points
	command_move = new TCommandMove;
	command_move->Add_Point(TVector3(-clear_xy, 0, d_artifact_settings.gaugeblock_clear_z));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	
	return true;
}

bool TMeasureRoutines::Align_Point_DCC(void)
{
	TCommandMode						*command_mode;
	TCommandPlane						*command_plane;
	TCommandAlign						*command_align;
	TCommandAlign::TAlignStep			align_step;
	TVector3							p1,p2;
	double								move_distance;

	move_distance = d_parameters.point_repeat_move_distance;
	
	if(move_distance < MIN_MOVE_DISTANCE)
	{
		emit Add_Log_Text(QString("ERR:  Move distance is too small.  Minimum move distance: %1").arg(MIN_MOVE_DISTANCE,0,'f',3));
		return false;
	}
	
	if(d_artifact_settings.plane_sample_radius < MIN_PLANE_SPACER)
	{
		emit Add_Log_Text(QString("ERR:  Point sample plane spacer too small.  Min plane spacer: %1").arg(MIN_PLANE_SPACER,0,'f',3));
		return false;
	}

	emit Update_Last_Configuration();

	// dcc mode
	command_mode = new TCommandMode();
	command_mode->Set_DCC_Mode();
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	
	// plane
	command_plane = new TCommandPlane(MEMORY_PLANE_REPEAT_DCC);
	
	command_plane->Add_Path_Touch_Point(TVector3(0,0,d_artifact_settings.plane_sample_radius),TVector3(-1,0,0));
	command_plane->Add_Path_Touch_Point(TVector3(0,d_artifact_settings.plane_sample_radius * 0.707107,
												 -1 * d_artifact_settings.plane_sample_radius),TVector3(-1,0,0));
	command_plane->Add_Path_Touch_Point(TVector3(0,-1 * d_artifact_settings.plane_sample_radius * 0.707107,
												 -1 * d_artifact_settings.plane_sample_radius),TVector3(-1,0,0));
	

	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_plane));
	
	// alignment
	command_align = new TCommandAlign;
	
	align_step.command_function = TCommandAlign::TYPE_SKEW1;
	align_step.memory_id = MEMORY_PLANE_REPEAT_DCC;
	align_step.axis = TReferenceSystem::XPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_SKEW2_PROBE_AXIS;
	align_step.memory_id = -1;
	align_step.axis = TReferenceSystem::ZMINUS;
	align_step.projection_axis = TReferenceSystem::XMINUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_PLANE_REPEAT_DCC;
	align_step.axis = TReferenceSystem::XPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_PLANE_REPEAT_DCC;
	align_step.axis = TReferenceSystem::YPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_PLANE_REPEAT_DCC;
	align_step.axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	
	return true;
}

bool TMeasureRoutines::Align_Sphere_DCC(void)
{
	TCommandMode						*command_mode;
	TCommandMove						*command_move;
	TCommandSphere						*command_sphere;
	TCommandAlign						*command_align;
	TCommandAlign::TAlignStep			align_step;
	TVector3							point_table[5];
	TVector3							p1,p2;
	double								sphere_diameter;
	double								sphere_radius;
	double								clear_dist;
	double								clear_z;
	double								tip_radius;

	emit Update_Last_Configuration();
	
	tip_radius = d_active_tool.tip_diameter / 2.0;
	
	sphere_diameter = d_parameters.sphere_nominal_diameter;
	
	if(sphere_diameter < MIN_SPHERE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Sphere diameter too small.  Minimum diameter: %1").arg(MIN_SPHERE_DIAMETER,0,'f',3));
		return false;
	}
	
	sphere_radius = sphere_diameter / 2.0;
	
	clear_dist = d_approach_distance;
	clear_dist += tip_radius;
	clear_dist += sphere_radius;
	
	// approach vector
	point_table[0].Set(0.000000000,0.000000000,-1.000000000);
	point_table[1].Set(-0.500000000,-0.500000000,-0.707106781);
	point_table[2].Set(0.500000000,-0.500000000,-0.707106781);
	point_table[3].Set(0.500000000,0.500000000,-0.707106781);
	point_table[4].Set(-0.500000000,0.500000000,-0.707106781);
	
	if(d_manual_mode)
	{
		// dcc mode
		command_mode = new TCommandMode();
		command_mode->Set_DCC_Mode();
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	}
	
	// sphere
	command_sphere = new TCommandSphere(MEMORY_SPHERE_DCC);
	command_sphere->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_OD);
	
	if(d_configuration == MeasureTypes::MEASURE_ROLL_OFFSET_TOOL)
	{
		switch(d_active_roll_sphere)
		{
			case MEMORY_SPHERE1_ROLL_DCC:
				p1 = -clear_dist * point_table[0];
				break;
				
			case MEMORY_SPHERE2_ROLL_DCC:
				p1 = -(clear_dist + 10.0) * point_table[0];		// Stop machine 10 mm prior for safety
				break;
				
			case MEMORY_SPHERE3_ROLL_DCC:
				p1 = -clear_dist * point_table[0];
				break;
				
			case MEMORY_SPHERE4_ROLL_DCC:
				p1 = -(clear_dist + 10.0) * point_table[0];		// Stop machine 10 mm prior for safety
				break;
		}

	}
	else
	{
		p1 = -clear_dist * point_table[0];
	}
	
	clear_z = p1.z;
	
	command_sphere->Add_Path_Move_Point(p1);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[0],point_table[0]);
	
	p1 = -clear_dist * point_table[1];
	p2 = p1;
	p2.z = clear_z;
	
	command_sphere->Add_Path_Move_Point(p2);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[1],point_table[1]);
	command_sphere->Add_Path_Move_Point(p2);
	
	p1 = -clear_dist * point_table[2];
	p2 = p1;
	p2.z = clear_z;
	
	command_sphere->Add_Path_Move_Point(p2);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[2],point_table[2]);
	command_sphere->Add_Path_Move_Point(p2);
	
	p1 = -clear_dist * point_table[3];
	p2 = p1;
	p2.z = clear_z;
	
	command_sphere->Add_Path_Move_Point(p2);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[3],point_table[3]);
	command_sphere->Add_Path_Move_Point(p2);
	
	p1 = -clear_dist * point_table[4];
	p2 = p1;
	p2.z = clear_z;
	
	command_sphere->Add_Path_Move_Point(p2);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[4],point_table[4]);
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_sphere));
	
	// move points
	command_move = new TCommandMove;
	command_move->Add_Point(p2);
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	
	// alignment
	command_align = new TCommandAlign;
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_SPHERE_DCC;
	align_step.axis = TReferenceSystem::XPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_SPHERE_DCC;
	align_step.axis = TReferenceSystem::YPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_SPHERE_DCC;
	align_step.axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	
	return true;
}

bool TMeasureRoutines::Align_Ballbar_B89_DCC(void)
{
	TMeasureRoutines::TFeatureMemory	feature_memory;
	TCommandMode						*command_mode;
	TCommandMove						*command_move;
	TCommandSphere						*command_sphere;
	TVector3							point_table[5];
	TVector3							pt1;
	TVector3							pt2;
	TVector3							sph1;
	TVector3							sph2;
	TVector3							vec;
	TVector3							bar_axis;
	TVector3							clearance_axis;
	double								sphere_diameter;
	double								sphere_radius;
	double								ballbar_length;
	double								clear_dist;
	double								dval;
	double								clear_z;
	double								tip_radius;
	bool								valid;
	static const double					BALLBAR_LENGTH_MIN(10.0);

	emit Update_Last_Configuration();
	
	tip_radius = d_active_tool.tip_diameter / 2.0;
	
	sphere_diameter = d_parameters.ballbar_sphere_diameter;
	
	if(sphere_diameter < MIN_SPHERE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Ballbar sphere diameter too small.  Minimum diameter: %1").arg(MIN_SPHERE_DIAMETER,0,'f',3));
		return false;
	}

	sphere_radius = sphere_diameter / 2.0;
	
	// approach vectors
	point_table[0].Set(0.000000000,0.000000000,-1.000000000);
	point_table[1].Set(0.000000000,0.707106781,-0.707106781);
	point_table[2].Set(-0.707106781,0.000000000,-0.707106781);
	point_table[3].Set(0.000000000,-0.707106781,-0.707106781);
	point_table[4].Set(0.707106781,0.000000000,-0.707106781);
	
	
	feature_memory = this->Feature_Memory(MEMORY_BALLBAR_SPHERE1_DCC,&valid);
	
	if(valid)
	{
		sph1 = feature_memory.xyz;
	}
	else
	{
		feature_memory = this->Feature_Memory(MEMORY_BALLBAR_POINT1_MANUAL,&valid);
		
		if(!valid)
		{
			emit Add_Log_Text(QStringLiteral("ERR:  Ballbar sphere1 position unknown"));
			return false;
		}
		
		sph1 = feature_memory.xyz + d_active_tool.ijk * sphere_radius;
	}
	
	feature_memory = this->Feature_Memory(MEMORY_BALLBAR_SPHERE2_DCC,&valid);
	
	if(valid)
	{
		sph2 = feature_memory.xyz;
	}
	else
	{
		feature_memory = this->Feature_Memory(MEMORY_BALLBAR_POINT2_MANUAL,&valid);
		
		if(!valid)
		{
			emit Add_Log_Text(QStringLiteral("ERR:  Ballbar sphere2 position unknown"));
			return false;
		}
		
		sph2 = feature_memory.xyz + d_active_tool.ijk * sphere_radius;
	}
	
	bar_axis = sph2 - sph1;
	
	ballbar_length = bar_axis.Length();
	
	if(ballbar_length < BALLBAR_LENGTH_MIN)
	{
		emit Add_Log_Text(QString("ERR:  Ballbar length too short:  Minimum length: %1").arg(BALLBAR_LENGTH_MIN,0,'f',3));
		return false;
	}
	
	bar_axis.Normal();
	
	// check to make sure at least 30 degree separator between bar_axis and active tool
	dval = fabs(bar_axis ^ d_active_tool.ijk);
	
	if(dval > 0.866025403784439)
	{
		emit Add_Log_Text(QStringLiteral("ERR:  Ballbar axis too close to active probe tool axis.  Minimum: 30 Deg"));
		return false;
	}
		
	d_refsys.Skew1(d_active_tool.ijk,TReferenceSystem::ZMINUS);
	d_refsys.Skew2(bar_axis,TReferenceSystem::XPLUS,TReferenceSystem::ZPLUS);
	
	sph1 = d_refsys.FromWorld(sph1);
	sph2 = d_refsys.FromWorld(sph2);
	
	clear_dist = d_approach_distance;
	clear_dist += tip_radius;
	clear_dist += sphere_radius;
	
	vec = d_active_tool.ijk * bar_axis;
	clearance_axis = vec * bar_axis;
	
	if(clearance_axis.Length() > 0.001)
	{
		clearance_axis.Normal();
	}
	else
	{
		clearance_axis.Set(0,0,1);
	}
	
	if((clearance_axis ^ d_active_tool.ijk) > 0)
	{
		clearance_axis *= (-1.0);
	}
	
	clearance_axis = d_refsys.AxisFromWorld(clearance_axis);
	
	// dcc mode
	command_mode = new TCommandMode();
	command_mode->Set_DCC_Mode();
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	
	// sphere2
	command_sphere = new TCommandSphere(MEMORY_BALLBAR_SPHERE2_DCC);
	command_sphere->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_OD);
	
	command_sphere->Set_Output_Format(TCommandFeature::FORMAT_DEFAULT);
	
	// sphere 2 pattern uses0,1,2,3
	command_sphere->Add_Path_Move_Point(sph2  + clearance_axis * d_artifact_settings.ballbar_clear_distance);
	
	pt1 = sph2 - clear_dist * point_table[0];
	clear_z = pt1.z;
	
	command_sphere->Add_Path_Touch_Point(sph2 - sphere_radius * point_table[0],point_table[0]);
	command_sphere->Add_Path_Move_Point(pt1);
	
	pt1 = sph2 - clear_dist * point_table[1];
	pt2 = pt1;
	pt2.z = clear_z;
	
	command_sphere->Add_Path_Move_Point(pt2);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Touch_Point(sph2 - sphere_radius * point_table[1],point_table[1]);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Move_Point(pt2);
	
	pt1 = sph2 - clear_dist * point_table[2];
	pt2 = pt1;
	pt2.z = clear_z;
	
	command_sphere->Add_Path_Move_Point(pt2);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Touch_Point(sph2 - sphere_radius * point_table[2],point_table[2]);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Move_Point(pt2);
	
	pt1 = sph2 - clear_dist * point_table[3];
	pt2 = pt1;
	pt2.z = clear_z;
	
	command_sphere->Add_Path_Move_Point(pt2);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Touch_Point(sph2 - sphere_radius * point_table[3],point_table[3]);
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_sphere));
	
	// move points
	command_move = new TCommandMove;
	command_sphere->Add_Path_Move_Point(pt1);
	command_move->Add_Point(sph2 + clearance_axis * d_artifact_settings.ballbar_clear_distance);
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	
	// sphere1
	command_sphere = new TCommandSphere(MEMORY_BALLBAR_SPHERE1_DCC);
	command_sphere->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_OD);
	
	command_sphere->Set_Output_Format(TCommandFeature::FORMAT_DEFAULT);
	
	// sphere 1 pattern uses0,1,4,3
	command_sphere->Add_Path_Move_Point(sph1  + clearance_axis * d_artifact_settings.ballbar_clear_distance);
	
	pt1 = sph1 - clear_dist * point_table[0];
	clear_z = pt1.z;
	
	command_sphere->Add_Path_Touch_Point(sph1 - sphere_radius * point_table[0],point_table[0]);
	command_sphere->Add_Path_Move_Point(pt1);
	
	pt1 = sph1 - clear_dist * point_table[1];
	pt2 = pt1;
	pt2.z = clear_z;
	
	command_sphere->Add_Path_Move_Point(pt2);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Touch_Point(sph1 - sphere_radius * point_table[1],point_table[1]);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Move_Point(pt2);
	
	pt1 = sph1 - clear_dist * point_table[4];
	pt2 = pt1;
	pt2.z = clear_z;
	
	command_sphere->Add_Path_Move_Point(pt2);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Touch_Point(sph1 - sphere_radius * point_table[4],point_table[4]);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Move_Point(pt2);
	
	pt1 = sph1 - clear_dist * point_table[3];
	pt2 = pt1;
	pt2.z = clear_z;
	
	command_sphere->Add_Path_Move_Point(pt2);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Touch_Point(sph1 - sphere_radius * point_table[3],point_table[3]);
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_sphere));
	
	// move points
	command_move = new TCommandMove;
	command_move->Add_Point(pt1);
	command_move->Add_Point(sph1 + clearance_axis * d_artifact_settings.ballbar_clear_distance);
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	
	return true;
}

bool TMeasureRoutines::Align_Ballbar_10360_DCC(void)
{
	TMeasureRoutines::TFeatureMemory	feature_memory;
	TCommandMode						*command_mode;
	TCommandMove						*command_move;
	TCommandSphere						*command_sphere;
	TVector3							point_table[5];
	TVector3							sph0;
	TVector3							sph5;
	TVector3							pt1;
	TVector3							pt2;
	TVector3							vec;
	TVector3							bar_axis;
	TVector3							clearance_axis;
	double								sphere_diameter;
	double								sphere_radius;
	double								length;
	double								clear_dist;
	double								dval;
	double								clear_z;
	double								tip_radius;
	bool								valid;

	emit Update_Last_Configuration();
	
	tip_radius = d_active_tool.tip_diameter / 2.0;
	
	sphere_diameter = d_parameters.ballbar_10360_sphere_diameter;
	
	if(sphere_diameter < MIN_SPHERE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Ballbar 10360 sphere diameter too small.  Minimum diameter: %1").arg(MIN_SPHERE_DIAMETER,0,'f',3));
		return false;
	}

	sphere_radius = sphere_diameter / 2.0;
	
	// approach vectors
	point_table[0].Set(0.000000000,0.000000000,-1.000000000);
	point_table[1].Set(0.000000000,0.707106781,-0.707106781);
	point_table[2].Set(-0.707106781,0.000000000,-0.707106781);
	point_table[3].Set(0.000000000,-0.707106781,-0.707106781);
	point_table[4].Set(0.707106781,0.000000000,-0.707106781);
	
	
	feature_memory = this->Feature_Memory(MEMORY_BALLBAR_10360_SPHERE0_DCC,&valid);
	
	if(valid)
	{
		sph0 = feature_memory.xyz;
	}
	else
	{
		feature_memory = this->Feature_Memory(MEMORY_BALLBAR_10360_POINT0_MANUAL,&valid);
		
		if(!valid)
		{
			emit Add_Log_Text(QStringLiteral("ERR:  Ballbar 10360 sphere zero position unknown"));
			return false;
		}
		
		sph0 = feature_memory.xyz + d_active_tool.ijk * sphere_radius;
	}
	
	feature_memory = this->Feature_Memory(MEMORY_BALLBAR_10360_SPHERE5_DCC,&valid);
	
	if(valid)
	{
		sph5 = feature_memory.xyz;
	}
	else
	{
		feature_memory = this->Feature_Memory(MEMORY_BALLBAR_10360_POINT5_MANUAL,&valid);
		
		if(!valid)
		{
			emit Add_Log_Text(QStringLiteral("ERR:  Ballbar 10360 end sphere position unknown"));
			return false;
		}
		
		sph5 = feature_memory.xyz + d_active_tool.ijk * sphere_radius;
	}
	
	bar_axis = sph5 - sph0;
	
	length = bar_axis.Length();
		
	if(length < (d_parameters.ballbar_10360_sphere_spacing * 5.0))
	{
		emit Add_Log_Text(QString("ERR:  Ballbar 10360 length too short:  Minimum length: %1").arg(d_parameters.ballbar_10360_sphere_spacing * 5.0,0,'f',3));
		return false;
	}
	
	bar_axis.Normal();
	   
	// check to make sure at least 30 degree separator between bar_axis and active tool
	dval = fabs(bar_axis ^ d_active_tool.ijk);
	
	if(dval > 0.866025403784439)
	{
		emit Add_Log_Text(QStringLiteral("ERR:  Ballbar 10360 axis too close to active probe tool axis.  Minimum: 30 Deg"));
		return false;
	}
	
	d_refsys.Skew1(d_active_tool.ijk,TReferenceSystem::ZMINUS);
	d_refsys.Skew2(bar_axis,TReferenceSystem::XPLUS,TReferenceSystem::ZPLUS);
	
	sph0 = d_refsys.FromWorld(sph0);
	sph5 = d_refsys.FromWorld(sph5);

	clear_dist = d_approach_distance;
	clear_dist += tip_radius;
	clear_dist += sphere_radius;
	
	vec = d_active_tool.ijk * bar_axis;
	clearance_axis = vec * bar_axis;
	
	if(clearance_axis.Length() > 0.001)
	{
		clearance_axis.Normal();
	}
	else
	{
		clearance_axis.Set(0,0,1);
	}
	
	if((clearance_axis ^ d_active_tool.ijk) > 0)
	{
		clearance_axis *= (-1.0);
	}
	
	clearance_axis = d_refsys.AxisFromWorld(clearance_axis);
	
	// dcc mode
	command_mode = new TCommandMode();
	command_mode->Set_DCC_Mode();
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	
	// sphere2
	command_sphere = new TCommandSphere(MEMORY_BALLBAR_10360_SPHERE5_DCC);
	command_sphere->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_OD);
	command_sphere->Set_Output_Format(TCommandFeature::FORMAT_DEFAULT);
	
	// sphere 2 pattern
	command_sphere->Add_Path_Move_Point(sph5  + clearance_axis * d_artifact_settings.ballbar_10360_clear_distance);
	
	pt1 = sph5 - clear_dist * point_table[0];
	clear_z = pt1.z;
	
	command_sphere->Add_Path_Touch_Point(sph5 - sphere_radius * point_table[0],point_table[0]);
	command_sphere->Add_Path_Move_Point(pt1);
	
	pt1 = sph5 - clear_dist * point_table[1];
	pt2 = pt1;
	pt2.z = clear_z;
	
	command_sphere->Add_Path_Move_Point(pt2);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Touch_Point(sph5 - sphere_radius * point_table[1],point_table[1]);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Move_Point(pt2);
	
	pt1 = sph5 - clear_dist * point_table[2];
	pt2 = pt1;
	pt2.z = clear_z;
	
	command_sphere->Add_Path_Move_Point(pt2);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Touch_Point(sph5 - sphere_radius * point_table[2],point_table[2]);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Move_Point(pt2);
	
	pt1 = sph5 - clear_dist * point_table[3];
	pt2 = pt1;
	pt2.z = clear_z;
	
	command_sphere->Add_Path_Move_Point(pt2);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Touch_Point(sph5 - sphere_radius * point_table[3],point_table[3]);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Move_Point(pt2);
	
	pt1 = sph5 - clear_dist * point_table[4];
	pt2 = pt1;
	pt2.z = clear_z;
	
	command_sphere->Add_Path_Move_Point(pt2);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Touch_Point(sph5 - sphere_radius * point_table[4],point_table[4]);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_sphere));
	
	// move points
	command_move = new TCommandMove;
	command_sphere->Add_Path_Move_Point(pt1);
	command_move->Add_Point(sph5 + clearance_axis * d_artifact_settings.ballbar_10360_clear_distance);
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	
	// sphere1
	command_sphere = new TCommandSphere(MEMORY_BALLBAR_10360_SPHERE0_DCC);
	command_sphere->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_OD);
	command_sphere->Set_Output_Format(TCommandFeature::FORMAT_DEFAULT);
	
	// sphere 1 pattern
	command_sphere->Add_Path_Move_Point(sph0  + clearance_axis * d_artifact_settings.ballbar_10360_clear_distance);
	
	pt1 = sph0 - clear_dist * point_table[0];
	clear_z = pt1.z;
	
	command_sphere->Add_Path_Touch_Point(sph0 - sphere_radius * point_table[0],point_table[0]);
	command_sphere->Add_Path_Move_Point(pt1);
	
	pt1 = sph0 - clear_dist * point_table[1];
	pt2 = pt1;
	pt2.z = clear_z;
	
	command_sphere->Add_Path_Move_Point(pt2);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Touch_Point(sph0 - sphere_radius * point_table[1],point_table[1]);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Move_Point(pt2);
	
	pt1 = sph0 - clear_dist * point_table[2];
	pt2 = pt1;
	pt2.z = clear_z;
	
	command_sphere->Add_Path_Move_Point(pt2);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Touch_Point(sph0 - sphere_radius * point_table[2],point_table[2]);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Move_Point(pt2);
	
	pt1 = sph0 - clear_dist * point_table[3];
	pt2 = pt1;
	pt2.z = clear_z;
	
	command_sphere->Add_Path_Move_Point(pt2);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Touch_Point(sph0 - sphere_radius * point_table[3],point_table[3]);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Move_Point(pt2);
	
	pt1 = sph0 - clear_dist * point_table[4];
	pt2 = pt1;
	pt2.z = clear_z;
	
	command_sphere->Add_Path_Move_Point(pt2);
	command_sphere->Add_Path_Move_Point(pt1);
	command_sphere->Add_Path_Touch_Point(sph0 - sphere_radius * point_table[4],point_table[4]);
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_sphere));
	
	// move points
	command_move = new TCommandMove;
	command_move->Add_Point(pt1);
	command_move->Add_Point(sph0 + clearance_axis * d_artifact_settings.ballbar_10360_clear_distance);
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	
	return true;
}

bool TMeasureRoutines::Align_Ringgauge_DCC(void)
{
	TMeasureRoutines::TFeatureMemory	feature_memory;
	TCommandMode						*command_mode;
	TCommandMove						*command_move;
	TCommandPlane						*command_plane;
	TCommandCircle						*command_circle;
	TCommandAlign						*command_align;
	TCommandAlign::TAlignStep			align_step;
	double								ringgauge_actual_diameter;
	double								ringgauge_nominal_diameter;
	double								ringgauge_radius;
	double								tip_radius;
	bool								valid(false);

	emit Update_Last_Configuration();
	
	ringgauge_nominal_diameter = d_parameters.ringgauge_nominal_diameter;
	
	if(ringgauge_nominal_diameter < MIN_RINGGAUGE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Ring gauge diameter too small.  Minimum diameter: %1").arg(MIN_RINGGAUGE_DIAMETER,0,'f',3));
		return false;
	}
	
	tip_radius = d_active_tool.tip_diameter / 2.0;
	ringgauge_radius = ringgauge_nominal_diameter / 2.0;
	
	feature_memory = this->Feature_Memory(MEMORY_RINGGAUGE_CIRCLE_MANUAL,&valid);
	
	if(valid)
	{
		ringgauge_actual_diameter = feature_memory.diameter;
	}
	else
	{
		emit Add_Log_Text(QStringLiteral("ERR:  Manual measurement of ringgauge diameter not available"));
		return false;
	}

	d_ringgauge_diameter_error = fabs(ringgauge_actual_diameter - ringgauge_nominal_diameter);
	
	if(d_ringgauge_diameter_error > MAX_RINGGAUGE_DIAMETER_ERROR)
	{
		emit Add_Log_Text(QString("ERR:  Actual ring gauge diameter different than nominal diameter:  Error %1").arg(d_ringgauge_diameter_error,0,'f',3));
		return false;
	}
	
	// dcc
	command_mode = new TCommandMode();
	command_mode->Set_DCC_Mode();
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	
	// move points
	command_move = new TCommandMove;
	command_move->Add_Point(TVector3(ringgauge_radius + d_artifact_settings.ringgauge_pln_offset, 0, d_artifact_settings.ringgauge_clear_z));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	
	// plane
	command_plane = new TCommandPlane(MEMORY_RINGGAUGE_PLANE_DCC);
	command_plane->Add_Path_Touch_Point(TVector3(ringgauge_radius + d_artifact_settings.ringgauge_pln_offset,0,0),TVector3(0,0,-1));
	command_plane->Add_Path_Touch_Point(TVector3(0,ringgauge_radius + d_artifact_settings.ringgauge_pln_offset,0),TVector3(0,0,-1));
	command_plane->Add_Path_Touch_Point(TVector3(-(ringgauge_radius + d_artifact_settings.ringgauge_pln_offset),0,0),TVector3(0,0,-1));
	command_plane->Add_Path_Touch_Point(TVector3(0,-(ringgauge_radius + d_artifact_settings.ringgauge_pln_offset),0),TVector3(0,0,-1));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_plane));
	
	// alignment
	command_align = new TCommandAlign;
	
	align_step.command_function = TCommandAlign::TYPE_SKEW1;
	align_step.memory_id = MEMORY_RINGGAUGE_PLANE_DCC;
	align_step.axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_RINGGAUGE_PLANE_DCC;
	align_step.axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	
	// move points
	command_move = new TCommandMove;
	command_move->Add_Point(TVector3(0, -(ringgauge_radius + d_artifact_settings.ringgauge_pln_offset), d_artifact_settings.ringgauge_clear_z));
	command_move->Add_Point(TVector3(ringgauge_radius - d_approach_distance - tip_radius, 0, d_artifact_settings.ringgauge_clear_z));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	
	// circle
	command_circle = new TCommandCircle(MEMORY_RINGGAUGE_CIRCLE_DCC);
	command_circle->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_ID);
	command_circle->Set_Nominal_Projection_Axis(TVector3(0,0,1));
	
	command_circle->Add_Path_Touch_Point(TVector3(ringgauge_radius,0,d_artifact_settings.ringgauge_cir_z),TVector3(1,0,0));
	command_circle->Add_Path_Touch_Point(TVector3(0,ringgauge_radius,d_artifact_settings.ringgauge_cir_z),TVector3(0,1,0));
	command_circle->Add_Path_Touch_Point(TVector3(-ringgauge_radius,0,d_artifact_settings.ringgauge_cir_z),TVector3(-1,0,0));
	command_circle->Add_Path_Touch_Point(TVector3(0,-ringgauge_radius,d_artifact_settings.ringgauge_cir_z),TVector3(0,-1,0));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_circle));
	
	// move points
	command_move = new TCommandMove;
	command_move->Add_Point(TVector3(0,-(ringgauge_radius - d_approach_distance - tip_radius), d_artifact_settings.ringgauge_clear_z));
	command_move->Add_Point(TVector3(0, 0, d_artifact_settings.ringgauge_clear_z));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	
	// alignment
	command_align = new TCommandAlign;
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_RINGGAUGE_CIRCLE_DCC;
	align_step.axis = TReferenceSystem::XPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_RINGGAUGE_CIRCLE_DCC;
	align_step.axis = TReferenceSystem::YPLUS;
	command_align->Add_Align_Step(align_step);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	
	return true;
}

bool TMeasureRoutines::Align_Pingauge_DCC(void)
{
	TMeasureRoutines::TFeatureMemory	feature_memory;
	TCommandMode						*command_mode;
	TCommandMove						*command_move;
	TCommandPlane						*command_plane;
	TCommandCircle						*command_circle;
	TCommandAlign						*command_align;
	TCommandAlign::TAlignStep			align_step;
	double								pingauge_actual_diameter;
	double								pingauge_nominal_diameter;
	double								pingauge_radius;
	double								tip_radius;
	bool								valid;

	emit Update_Last_Configuration();
	
	pingauge_nominal_diameter = d_parameters.pingauge_nominal_diameter;
	
	if(pingauge_nominal_diameter < MIN_PINGAUGE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Pin gauge diameter too small.  Minimum diameter: %1").arg(MIN_PINGAUGE_DIAMETER,0,'f',3));
		return false;
	}
	
	pingauge_radius = pingauge_nominal_diameter / 2.0;
	tip_radius = d_active_tool.tip_diameter / 2.0;

	feature_memory = this->Feature_Memory(MEMORY_PINGAUGE_CIRCLE_MANUAL,&valid);
	
	if(valid)
	{
		pingauge_actual_diameter = feature_memory.diameter;
	}
	else
	{
		emit Add_Log_Text(QStringLiteral("ERR:  Manual measurement of pingauge diameter not available"));
		return false;
	}
	
	d_pingauge_diameter_error = fabs(pingauge_actual_diameter - pingauge_nominal_diameter);
	
	if(d_pingauge_diameter_error > MAX_PINGAUGE_DIAMETER_ERROR)
	{
		emit Add_Log_Text(QString("ERR:  Actual pin gauge diameter different than nominal diameter:  Error %1").arg(d_pingauge_diameter_error,0,'f',3));
		return false;
	}
	
	// dcc
	command_mode = new TCommandMode();
	command_mode->Set_DCC_Mode();
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	
	// move points
	command_move = new TCommandMove;
	command_move->Add_Point(TVector3(pingauge_radius + d_artifact_settings.pingauge_pln_offset, 0, d_artifact_settings.pingauge_clear_z));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	
	// plane
	command_plane = new TCommandPlane(MEMORY_PINGAUGE_PLANE_DCC);
	command_plane->Add_Path_Touch_Point(TVector3(pingauge_radius + d_artifact_settings.pingauge_pln_offset,0,0),TVector3(0,0,-1));
	command_plane->Add_Path_Touch_Point(TVector3(0,pingauge_radius + d_artifact_settings.pingauge_pln_offset,0),TVector3(0,0,-1));
	command_plane->Add_Path_Touch_Point(TVector3(-(pingauge_radius + d_artifact_settings.pingauge_pln_offset),0,0),TVector3(0,0,-1));
	command_plane->Add_Path_Touch_Point(TVector3(0,-(pingauge_radius + d_artifact_settings.pingauge_pln_offset),0),TVector3(0,0,-1));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_plane));
	
	// alignment
	command_align = new TCommandAlign;
	
	align_step.command_function = TCommandAlign::TYPE_SKEW1;
	align_step.memory_id = MEMORY_PINGAUGE_PLANE_DCC;
	align_step.axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_PINGAUGE_PLANE_DCC;
	align_step.axis = TReferenceSystem::ZPLUS;
	command_align->Add_Align_Step(align_step);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	
	// move points
	command_move = new TCommandMove;
	command_move->Add_Point(TVector3(0, -(pingauge_radius + d_artifact_settings.pingauge_pln_offset), d_artifact_settings.pingauge_clear_z));
	command_move->Add_Point(TVector3(pingauge_radius + d_approach_distance + tip_radius, 0, d_artifact_settings.pingauge_clear_z));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	
	// circle
	command_circle = new TCommandCircle(MEMORY_PINGAUGE_CIRCLE_DCC);
	command_circle->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_OD);
	command_circle->Set_Nominal_Projection_Axis(TVector3(0,0,1));
	
	command_circle->Add_Path_Touch_Point(TVector3(pingauge_radius,0,d_artifact_settings.pingauge_cir_z),TVector3(-1,0,0));
	command_circle->Add_Path_Move_Point(TVector3(pingauge_radius + d_approach_distance + tip_radius,0,d_artifact_settings.pingauge_clear_z));
	command_circle->Add_Path_Move_Point(TVector3(0,pingauge_radius + d_approach_distance + tip_radius,d_artifact_settings.pingauge_clear_z));
	command_circle->Add_Path_Touch_Point(TVector3(0,pingauge_radius,d_artifact_settings.pingauge_cir_z),TVector3(0,-1,0));
	command_circle->Add_Path_Move_Point(TVector3(0,pingauge_radius + d_approach_distance + tip_radius,d_artifact_settings.pingauge_clear_z));
	command_circle->Add_Path_Move_Point(TVector3(-(pingauge_radius + d_approach_distance + tip_radius),0,d_artifact_settings.pingauge_clear_z));
	command_circle->Add_Path_Touch_Point(TVector3(-pingauge_radius,0,d_artifact_settings.pingauge_cir_z),TVector3(1,0,0));
	command_circle->Add_Path_Move_Point(TVector3(-(pingauge_radius + d_approach_distance + tip_radius),0,d_artifact_settings.pingauge_clear_z));
	command_circle->Add_Path_Move_Point(TVector3(0,-(pingauge_radius + d_approach_distance + tip_radius),d_artifact_settings.pingauge_clear_z));
	command_circle->Add_Path_Touch_Point(TVector3(0,-pingauge_radius,d_artifact_settings.pingauge_cir_z),TVector3(0,1,0));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_circle));
	
	// move points
	command_move = new TCommandMove;
	command_move->Add_Point(TVector3(0,-(pingauge_radius + d_approach_distance + tip_radius), d_artifact_settings.pingauge_clear_z));
	command_move->Add_Point(TVector3(0, 0, d_artifact_settings.pingauge_clear_z));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	
	// alignment
	command_align = new TCommandAlign;
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_PINGAUGE_CIRCLE_DCC;
	align_step.axis = TReferenceSystem::XPLUS;
	command_align->Add_Align_Step(align_step);
	
	align_step.command_function = TCommandAlign::TYPE_ORIGIN;
	align_step.memory_id = MEMORY_PINGAUGE_CIRCLE_DCC;
	align_step.axis = TReferenceSystem::YPLUS;
	command_align->Add_Align_Step(align_step);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
	
	return true;
}

bool TMeasureRoutines::Measure_Stepgauge_10360(void)
{
	TMeasureRoutines::TFeatureMemory	feature_memory;
	TCommandPoint						*command_point;
	TCommandMove						*command_move;
	TCommandDimension					*command_dimension;
	TCommandMode						*command_mode;
	QString								text;
	TVector3							v1;
	TVector3							probe_axis;
	double								targets[6];
	double								increment;
	double								clear_xy;
	double								starrett_clear_z;
	double								measure_length(0.0);
	int									target_count;
	int									ival;
	int									memory_id(MEMORY_STEPGAUGE_POINT_START_MEASURE);
	int									memory_p1;
	int									memory_p2;
	int									cntr;
	int									repeat_cntr;
	double								tip_radius;
	bool								valid;

	emit Add_Log_Text(TCommandDimension::Dimension_Item_Text_Title_Err());
	
	feature_memory = this->Feature_Memory(MEMORY_STEPGAUGE_POINT_END_DCC,&valid);
	
	if(valid)
	{
		v1 = d_refsys.FromWorld(feature_memory.xyz);
		measure_length = v1.Length();
	}
	
	tip_radius = d_active_tool.tip_diameter / 2.0;
	
	if(d_output_file.isOpen())
	{
		d_output_file.write("Alignment_Begin:\n");
		
		v1 = d_refsys.Rotation_Mat().X();
		text = QString("X_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Y();
		text = QString("Y_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Z();
		text = QString("Z_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Origin();
		text = QString("Translation:%1,%2,%3\n").arg(v1.x,0,'f',5).arg(v1.y,0,'f',5).arg(v1.z,0,'f',5);
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Alignment_End:\n\n");
		
		d_output_file.write("Temperature_Begin:\n");
		
		text = QString("X_Scale:%1\n").arg(d_parameters.axis_x_temperature,0,'f',3);
		d_output_file.write(text.toLatin1());
		
		text = QString("Y_Scale:%1\n").arg(d_parameters.axis_y_temperature,0,'f',3);
		d_output_file.write(text.toLatin1());
		
		text = QString("Z_Scale:%1\n").arg(d_parameters.axis_z_temperature,0,'f',3);
		d_output_file.write(text.toLatin1());
		
		text = QString("Gauge:%1\n").arg(d_parameters.part_temperature,0,'f',3);
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Temperature_End:\n\n");
		
		text = QString("Probe_Offset:%1,%2,%3\n").arg(d_active_tool.xyz.x,0,'f',4).arg(d_active_tool.xyz.y,0,'f',4).arg(d_active_tool.xyz.z,0,'f',4);
		d_output_file.write(text.toLatin1());
		
		text = QString("Probe_Vector:%1,%2,%3\n\n").arg(d_active_tool.ijk.x,0,'f',12).arg(d_active_tool.ijk.y,0,'f',12).arg(d_active_tool.ijk.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Measurement_Begin:\n\n");
		
		text = QString("Description:%1\n").arg(d_parameters.stepgauge_10360_position_name);
		d_output_file.write(text.toLatin1());
		
		text = QString("Reference_Step_Position:%1\n\n").arg(d_parameters.stepgauge_measurement_zero_position,0,'f',3);
		d_output_file.write(text.toLatin1());
	}
	
	clear_xy = d_approach_distance;
	clear_xy += tip_radius;
	
	target_count = static_cast<int>(0.5 + (measure_length - d_parameters.stepgauge_zero_step_spacing) / d_parameters.stepgauge_nominal_block_spacing);
	
	targets[5] = d_parameters.stepgauge_zero_step_spacing + static_cast<double>(target_count) * d_parameters.stepgauge_nominal_block_spacing;
	targets[0] = d_parameters.stepgauge_zero_step_spacing;
	
	ival = target_count / 5;
	
	if(ival < 1)
	{
		emit Add_Log_Text(QString("ERR:  Stepgauge too short for required measurements.  Length: %1").arg(measure_length,0,'f',3));
		return false;
	}
	
	increment = static_cast<double>(ival) * d_parameters.stepgauge_nominal_block_spacing;
	
	targets[4] = targets[5] - increment;
	targets[3] = targets[4] - increment;
	targets[2] = targets[3] - increment;
	targets[1] = targets[2] - increment;
	
	if(d_manual_mode)
	{
		// dcc mode
		command_mode = new TCommandMode();
		command_mode->Set_DCC_Mode();
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	}
	
	if(d_parameters.stepgauge_type == static_cast<int>(MeasureTypes::STEPGAUGE_CHECKMASTER))
	{
		for(cntr = 0;cntr < 6;++cntr)
		{
			for(repeat_cntr = 0;repeat_cntr < 3;++repeat_cntr)
			{
				// move points
				command_move = new TCommandMove;
				command_move->Add_Point(TVector3(-clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
				
				// point start
				memory_p1 = memory_id++;
				command_point = new TCommandPoint(memory_p1);
				command_point->Add_Path_Touch_Point(TVector3(0,d_artifact_settings.sg_checkmaster_pnt_y,d_artifact_settings.sg_checkmaster_pnt_z),TVector3(1,0,0));
				command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
				
				command_dimension = new TCommandDimension;
				command_dimension->Set_Memory_Id_Feature1(memory_p1);
				command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("PtA.X"),0.0);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
				
				// move points
				command_move = new TCommandMove;
				command_move->Add_Point(TVector3(-clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
				command_move->Add_Point(TVector3(targets[cntr] + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
				
				// point end
				memory_p2 = memory_id++;
				command_point = new TCommandPoint(memory_p2);
				command_point->Add_Path_Touch_Point(TVector3(targets[cntr],d_artifact_settings.sg_checkmaster_pnt_y,d_artifact_settings.sg_checkmaster_pnt_z),TVector3(-1,0,0));
				command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
				
				command_dimension = new TCommandDimension;
				command_dimension->Set_Memory_Id_Feature1(memory_p2);
				command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("PtB.X"),0.0);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
				
				command_dimension = new TCommandDimension;
				command_dimension->Set_Memory_Id_Feature1(memory_p1);
				command_dimension->Set_Memory_Id_Feature2(memory_p2);
				command_dimension->Add_Dimension(TCommandDimension::FIELD_DIST_X,QStringLiteral("PtB-PtA"),0.0,0.0);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
				
				// move points
				command_move = new TCommandMove;
				command_move->Add_Point(TVector3(targets[cntr] + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			}
		}
	}
	else if(d_parameters.stepgauge_type == static_cast<int>(MeasureTypes::STEPGAUGE_KOBA))
	{
		probe_axis = d_refsys.AxisFromWorld(d_active_tool.ijk);
		
		for(cntr = 0;cntr < 6;++cntr)
		{
			for(repeat_cntr = 0;repeat_cntr < 3;++repeat_cntr)
			{
				// move points
				command_move = new TCommandMove;
				v1.Set(-clear_xy,d_artifact_settings.sg_koba_pnt_y, d_artifact_settings.sg_koba_pnt_z);
				v1 -= (probe_axis * d_artifact_settings.sg_koba_clear_probe);
				command_move->Add_Point(v1);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
				
				// point start
				memory_p1 = memory_id++;
				command_point = new TCommandPoint(memory_p1);
				command_point->Add_Path_Touch_Point(TVector3(0,d_artifact_settings.sg_koba_pnt_y,d_artifact_settings.sg_koba_pnt_z),TVector3(1,0,0));
				command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
				
				command_dimension = new TCommandDimension;
				command_dimension->Set_Memory_Id_Feature1(memory_p1);
				command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("PtA.X"),0.0);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
				
				// move points
				command_move = new TCommandMove;
				command_move->Add_Point(v1);
				v1.x = targets[cntr] + clear_xy;
				command_move->Add_Point(v1);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
				
				// point end
				memory_p2 = memory_id++;
				command_point = new TCommandPoint(memory_p2);
				command_point->Add_Path_Touch_Point(TVector3(targets[cntr],d_artifact_settings.sg_koba_pnt_y,d_artifact_settings.sg_koba_pnt_z),TVector3(-1,0,0));
				command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
				
				command_dimension = new TCommandDimension;
				command_dimension->Set_Memory_Id_Feature1(memory_p2);
				command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("PtB.X"),0.0);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
				
				command_dimension = new TCommandDimension;
				command_dimension->Set_Memory_Id_Feature1(memory_p1);
				command_dimension->Set_Memory_Id_Feature2(memory_p2);
				command_dimension->Add_Dimension(TCommandDimension::FIELD_DIST_X,QStringLiteral("PtB-PtA"),0.0,0.0);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
				
				// move points
				command_move = new TCommandMove;
				command_move->Add_Point(v1);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			}
		}
	}
	else if(d_parameters.stepgauge_type == static_cast<int>(MeasureTypes::STEPGAUGE_STARRET))
	{
		starrett_clear_z = d_artifact_settings.sg_webber_clear_z + d_stepgauge_manual_points[0].z;
		
		for(cntr = 0;cntr < 6;++cntr)
		{
			for(repeat_cntr = 0;repeat_cntr < 3;++repeat_cntr)
			{
				// move points
				command_move = new TCommandMove;
				command_move->Add_Point(TVector3(-clear_xy, d_stepgauge_manual_points[5].y, starrett_clear_z));
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
				
				// point start
				memory_p1 = memory_id++;
				command_point = new TCommandPoint(memory_p1);
				command_point->Add_Path_Touch_Point(TVector3(0,d_stepgauge_manual_points[5].y,d_stepgauge_manual_points[5].z),TVector3(1,0,0));
				command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
				
				command_dimension = new TCommandDimension;
				command_dimension->Set_Memory_Id_Feature1(memory_p1);
				command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("PtA.X"),0.0);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
				
				// move points
				command_move = new TCommandMove;
				command_move->Add_Point(TVector3(-clear_xy, d_stepgauge_manual_points[5].y, starrett_clear_z));
				command_move->Add_Point(TVector3(targets[cntr] + clear_xy, 0, starrett_clear_z));
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
				
				// point end
				memory_p2 = memory_id++;
				command_point = new TCommandPoint(memory_p2);
				command_point->Add_Path_Touch_Point(TVector3(targets[cntr],0,0),TVector3(-1,0,0));
				command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
				
				command_dimension = new TCommandDimension;
				command_dimension->Set_Memory_Id_Feature1(memory_p2);
				command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("PtB.X"),0.0);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
				
				command_dimension = new TCommandDimension;
				command_dimension->Set_Memory_Id_Feature1(memory_p1);
				command_dimension->Set_Memory_Id_Feature2(memory_p2);
				command_dimension->Add_Dimension(TCommandDimension::FIELD_DIST_X,QStringLiteral("PtB-PtA"),0.0,0.0);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
				
				// move points
				command_move = new TCommandMove;
				command_move->Add_Point(TVector3(targets[cntr] + clear_xy, 0, starrett_clear_z));
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			}
		}
	}
	return true;
}

bool TMeasureRoutines::Measure_Stepgauge_B89(void)
{
	TMeasureRoutines::TFeatureMemory	feature_memory;
	TCommandPoint						*command_point;
	TCommandMove						*command_move;
	TCommandDimension					*command_dimension;
	TCommandMode						*command_mode;
	QString								text;
	TVector3							v1;
	TVector3							probe_axis;
	double								start_position;
	double								position;
	double								clear_xy;
	double								starrett_clear_z;
	double								measure_length(0.0);
	int									target_count;
	int									target_cntr;
	int									measurement_count;
	int									measurement_cntr;
	int									memory_id(MEMORY_STEPGAUGE_POINT_START_MEASURE);
	int									memory_p;
	double								tip_radius;
	bool								valid;

	emit Add_Log_Text(TCommandDimension::Dimension_Item_Text_Title_Err());
	
	feature_memory = this->Feature_Memory(MEMORY_STEPGAUGE_POINT_END_DCC,&valid);
	
	if(valid)
	{
		v1 = d_refsys.FromWorld(feature_memory.xyz);
		measure_length = v1.Length();
	}

	tip_radius = d_active_tool.tip_diameter / 2.0;
	
	if(d_output_file.isOpen())
	{
		d_output_file.write("Alignment_Begin:\n");
		
		v1 = d_refsys.Rotation_Mat().X();
		text = QString("X_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Y();
		text = QString("Y_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Z();
		text = QString("Z_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Origin();
		text = QString("Translation:%1,%2,%3\n").arg(v1.x,0,'f',5).arg(v1.y,0,'f',5).arg(v1.z,0,'f',5);
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Alignment_End:\n\n");
		
		d_output_file.write("Temperature_Begin:\n");
		
		text = QString("X_Scale:%1\n").arg(d_parameters.axis_x_temperature,0,'f',3);
		d_output_file.write(text.toLatin1());
		
		text = QString("Y_Scale:%1\n").arg(d_parameters.axis_y_temperature,0,'f',3);
		d_output_file.write(text.toLatin1());
		
		text = QString("Z_Scale:%1\n").arg(d_parameters.axis_z_temperature,0,'f',3);
		d_output_file.write(text.toLatin1());
		
		text = QString("Gauge:%1\n").arg(d_parameters.part_temperature,0,'f',3);
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Temperature_End:\n\n");
		
		text = QString("Probe_Offset:%1,%2,%3\n").arg(d_active_tool.xyz.x,0,'f',4).arg(d_active_tool.xyz.y,0,'f',4).arg(d_active_tool.xyz.z,0,'f',4);
		d_output_file.write(text.toLatin1());
		
		text = QString("Probe_Vector:%1,%2,%3\n\n").arg(d_active_tool.ijk.x,0,'f',12).arg(d_active_tool.ijk.y,0,'f',12).arg(d_active_tool.ijk.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Measurement_Begin:\n\n");
		
		text = QString("Description:%1\n").arg(d_parameters.stepgauge_b89_position_name);
		d_output_file.write(text.toLatin1());
		
		text = QString("Reference_Step_Position:%1\n\n").arg(0.0,0,'f',3);	// not measured but implied by stepgauge_measurement_b89_start
		d_output_file.write(text.toLatin1());
	}
	
	clear_xy = d_approach_distance;
	clear_xy += tip_radius;
	start_position = d_parameters.stepgauge_measurement_b89_start;

	target_count = 1 + static_cast<int>((measure_length + 2.0 - start_position) / d_parameters.stepgauge_b89_nominal_block_spacing);
	
	if(target_count < 1)
	{
		emit Add_Log_Text(QString("ERR:  Stepgauge too short for required measurements.  Length: %1").arg(measure_length,0,'f',3));
		return false;
	}
	
	measurement_count = d_parameters.stepgauge_b89_measurement_count;
	
	if(measurement_count < 1)
	{
		emit Add_Log_Text(QStringLiteral("ERR:  Stepgauge B89 measurement count must be greater than zero."));
		return false;
	}
		
	if(start_position < 5.0)
	{
		emit Add_Log_Text(QStringLiteral("ERR:  Stepgauge B89 start position cannot be smaller than 5 mm."));
		return false;
	}

	if(d_manual_mode)
	{
		// dcc mode
		command_mode = new TCommandMode();
		command_mode->Set_DCC_Mode();
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	}
	if(d_parameters.stepgauge_type == static_cast<int>(MeasureTypes::STEPGAUGE_CHECKMASTER))
	{
		for(measurement_cntr = 0;measurement_cntr < measurement_count;++measurement_cntr)
		{
			position = start_position;

			for(target_cntr = 0;target_cntr < target_count;++target_cntr)
			{
				// move points
				command_move = new TCommandMove;
				command_move->Add_Point(TVector3(position + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
				
				memory_p = memory_id++;
				
				command_point = new TCommandPoint(memory_p);
				command_point->Add_Path_Touch_Point(TVector3(position,d_artifact_settings.sg_checkmaster_pnt_y,d_artifact_settings.sg_checkmaster_pnt_z),TVector3(-1,0,0));
				command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
				
				command_dimension = new TCommandDimension;
				command_dimension->Set_Memory_Id_Feature1(memory_p);
				command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("Pt.X"),0.0);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
				
				// moves
				command_move = new TCommandMove;
				command_move->Add_Point(TVector3(position + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
				
				position += d_parameters.stepgauge_b89_nominal_block_spacing;
			}
			
		}
	}
	else if(d_parameters.stepgauge_type == static_cast<int>(MeasureTypes::STEPGAUGE_KOBA))
	{
		probe_axis = d_refsys.AxisFromWorld(d_active_tool.ijk);
		
		for(measurement_cntr = 0;measurement_cntr < measurement_count;++measurement_cntr)
		{
			position = start_position;
			
			for(target_cntr = 0;target_cntr < target_count;++target_cntr)
			{
				// move points
				command_move = new TCommandMove;
				v1.Set(position + clear_xy,d_artifact_settings.sg_koba_pnt_y, d_artifact_settings.sg_koba_pnt_z);
				v1 -= (probe_axis * d_artifact_settings.sg_koba_clear_probe);
				command_move->Add_Point(v1);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
				
				memory_p = memory_id++;
				command_point = new TCommandPoint(memory_p);
				command_point->Add_Path_Touch_Point(TVector3(position,d_artifact_settings.sg_koba_pnt_y,d_artifact_settings.sg_koba_pnt_z),TVector3(-1,0,0));
				command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
				
				command_dimension = new TCommandDimension;
				command_dimension->Set_Memory_Id_Feature1(memory_p);
				command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("Pt.X"),0.0);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
				
				// moves
				command_move = new TCommandMove;
				command_move->Add_Point(v1);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
				
				position += d_parameters.stepgauge_b89_nominal_block_spacing;
			}
		}
	}
	else if(d_parameters.stepgauge_type == static_cast<int>(MeasureTypes::STEPGAUGE_STARRET))
	{
		starrett_clear_z = d_artifact_settings.sg_webber_clear_z + d_stepgauge_manual_points[0].z;
		
		for(measurement_cntr = 0;measurement_cntr < measurement_count;++measurement_cntr)
		{
			position = start_position;
			
			for(target_cntr = 0;target_cntr < target_count;++target_cntr)
			{
				// move points
				command_move = new TCommandMove;
				command_move->Add_Point(TVector3(position + clear_xy, d_stepgauge_manual_points[5].y, starrett_clear_z));
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
				
				memory_p = memory_id++;
				command_point = new TCommandPoint(memory_p);
				command_point->Add_Path_Touch_Point(TVector3(position,d_stepgauge_manual_points[5].y,d_stepgauge_manual_points[5].z),TVector3(-1,0,0));
				command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
				
				command_dimension = new TCommandDimension;
				command_dimension->Set_Memory_Id_Feature1(memory_p);
				command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("Pt.X"),0.0);
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
				
				// moves
				command_move = new TCommandMove;
				command_move->Add_Point(TVector3(position + clear_xy, d_stepgauge_manual_points[5].y, starrett_clear_z));
				d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
				
				position += d_parameters.stepgauge_b89_nominal_block_spacing;
			}
		}
	}
	
	return true;
}

bool TMeasureRoutines::Measure_Stepgauge_Square(void)
{
	TMeasureRoutines::TFeatureMemory	feature_memory;
	TCommandPoint						*command_point;
	TCommandMove						*command_move;
	TCommandDimension					*command_dimension;
	TCommandMode						*command_mode;
	QString								text;
	TVector3							v1;
	TVector3							probe_axis;
	double								targets[6];
	double								increment;
	double								clear_xy;
	double								starrett_clear_z;
	double								measure_length(0.0);
	int									target_count;
	int									ival;
	int									memory_id(MEMORY_STEPGAUGE_POINT_START_MEASURE);
	int									memory_p;
	int									measurement_count;
	int									measurement_cntr;
	double								tip_radius;
	bool								valid;

	emit Add_Log_Text(TCommandDimension::Dimension_Item_Text_Title_Meas());
	
	feature_memory = this->Feature_Memory(MEMORY_STEPGAUGE_POINT_END_DCC,&valid);
	
	if(valid)
	{
		v1 = d_refsys.FromWorld(feature_memory.xyz);
		measure_length = v1.Length();
	}
	
	tip_radius = d_active_tool.tip_diameter / 2.0;

	if(d_output_file.isOpen())
	{
		d_output_file.write("Alignment_Begin:\n");
		
		v1 = d_refsys.Rotation_Mat().X();
		text = QString("X_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Y();
		text = QString("Y_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Z();
		text = QString("Z_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Origin();
		text = QString("Translation:%1,%2,%3\n").arg(v1.x,0,'f',5).arg(v1.y,0,'f',5).arg(v1.z,0,'f',5);
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Alignment_End:\n\n");
		
		d_output_file.write("Temperature_Begin:\n");
		
		text = QString("X_Scale:%1\n").arg(d_parameters.axis_x_temperature,0,'f',3);
		d_output_file.write(text.toLatin1());
		
		text = QString("Y_Scale:%1\n").arg(d_parameters.axis_y_temperature,0,'f',3);
		d_output_file.write(text.toLatin1());
		
		text = QString("Z_Scale:%1\n").arg(d_parameters.axis_z_temperature,0,'f',3);
		d_output_file.write(text.toLatin1());
		
		text = QString("Gauge:%1\n").arg(d_parameters.part_temperature,0,'f',3);
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Temperature_End:\n\n");
		
		text = QString("Probe_Offset:%1,%2,%3\n").arg(d_active_tool.xyz.x,0,'f',4).arg(d_active_tool.xyz.y,0,'f',4).arg(d_active_tool.xyz.z,0,'f',4);
		d_output_file.write(text.toLatin1());
		
		text = QString("Probe_Vector:%1,%2,%3\n\n").arg(d_active_tool.ijk.x,0,'f',12).arg(d_active_tool.ijk.y,0,'f',12).arg(d_active_tool.ijk.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Measurement_Begin:\n\n");
		
		text = QString("Description:%1\n").arg(d_parameters.stepgauge_square_position_name);
		d_output_file.write(text.toLatin1());
		
		text = QString("Reference_Step_Position:%1\n\n").arg(d_parameters.stepgauge_measurement_zero_position,0,'f',3);
		d_output_file.write(text.toLatin1());
	}
	
	clear_xy = d_approach_distance;
	clear_xy += tip_radius;
	
	measurement_count = d_parameters.stepgauge_square_measurement_count;
	
	if(measurement_count < 1)
	{
		emit Add_Log_Text(QStringLiteral("ERR:  Stepgauge squareness measurement count must be greater than zero."));
		return false;
	}
	
	target_count = static_cast<int>(0.5 + (measure_length - d_parameters.stepgauge_zero_step_spacing) / d_parameters.stepgauge_nominal_block_spacing);
	
	targets[5] = d_parameters.stepgauge_zero_step_spacing + static_cast<double>(target_count) * d_parameters.stepgauge_nominal_block_spacing;
	targets[0] = d_parameters.stepgauge_zero_step_spacing;
	
	ival = target_count / 5;
	
	if(ival < 1)
	{
		emit Add_Log_Text(QString("ERR:  Stepgauge too short for required measurements.  Length: %1").arg(measure_length,0,'f',3));
		return false;
	}
	
	increment = static_cast<double>(ival) * d_parameters.stepgauge_nominal_block_spacing;
	
	targets[4] = targets[5] - increment;
	targets[3] = targets[4] - increment;
	targets[2] = targets[3] - increment;
	targets[1] = targets[2] - increment;
	
	if(d_manual_mode)
	{
		// dcc mode
		command_mode = new TCommandMode();
		command_mode->Set_DCC_Mode();
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	}
	
	if(d_parameters.stepgauge_type == static_cast<int>(MeasureTypes::STEPGAUGE_CHECKMASTER))
	{
		for(measurement_cntr = 0;measurement_cntr < measurement_count;++measurement_cntr)
		{
			// move points
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(-clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// point start
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(0,d_artifact_settings.sg_checkmaster_pnt_y,d_artifact_settings.sg_checkmaster_pnt_z),TVector3(1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("PZ"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(-clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			command_move->Add_Point(TVector3(targets[0] + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// target[0]
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(targets[0],d_artifact_settings.sg_checkmaster_pnt_y,d_artifact_settings.sg_checkmaster_pnt_z),TVector3(-1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("P0"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(targets[0] + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			command_move->Add_Point(TVector3(targets[1] + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// target[1]
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(targets[1],d_artifact_settings.sg_checkmaster_pnt_y,d_artifact_settings.sg_checkmaster_pnt_z),TVector3(-1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("P1"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(targets[1] + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			command_move->Add_Point(TVector3(targets[2] + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// target[2]
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(targets[2],d_artifact_settings.sg_checkmaster_pnt_y,d_artifact_settings.sg_checkmaster_pnt_z),TVector3(-1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("P2"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(targets[2] + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			command_move->Add_Point(TVector3(targets[3] + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// target[3]
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(targets[3],d_artifact_settings.sg_checkmaster_pnt_y,d_artifact_settings.sg_checkmaster_pnt_z),TVector3(-1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("P3"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(targets[3] + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			command_move->Add_Point(TVector3(targets[4] + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// target[4]
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(targets[4],d_artifact_settings.sg_checkmaster_pnt_y,d_artifact_settings.sg_checkmaster_pnt_z),TVector3(-1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("P4"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(targets[4] + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			command_move->Add_Point(TVector3(targets[5] + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// target[5]
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(targets[5],d_artifact_settings.sg_checkmaster_pnt_y,d_artifact_settings.sg_checkmaster_pnt_z),TVector3(-1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("P5"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(targets[5] + clear_xy, d_artifact_settings.sg_checkmaster_pnt_y, d_artifact_settings.sg_checkmaster_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		}
	}
	else if(d_parameters.stepgauge_type == static_cast<int>(MeasureTypes::STEPGAUGE_KOBA))
	{
		probe_axis = d_refsys.AxisFromWorld(d_active_tool.ijk);
		
		for(measurement_cntr = 0;measurement_cntr < measurement_count;++measurement_cntr)
		{
			// move points
			command_move = new TCommandMove;
			v1.Set(-clear_xy,d_artifact_settings.sg_koba_pnt_y, d_artifact_settings.sg_koba_pnt_z);
			v1 -= (probe_axis * d_artifact_settings.sg_koba_clear_probe);
			command_move->Add_Point(v1);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// point start
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(0,d_artifact_settings.sg_koba_pnt_y,d_artifact_settings.sg_koba_pnt_z),TVector3(1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("PZ"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(v1);
			v1.x = targets[0] + clear_xy;
			command_move->Add_Point(v1);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// target[0]
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(targets[0],d_artifact_settings.sg_koba_pnt_y,d_artifact_settings.sg_koba_pnt_z),TVector3(-1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("P0"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(v1);
			v1.x = targets[1] + clear_xy;
			command_move->Add_Point(v1);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// target[1]
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(targets[1],d_artifact_settings.sg_koba_pnt_y,d_artifact_settings.sg_koba_pnt_z),TVector3(-1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("P1"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(v1);
			v1.x = targets[2] + clear_xy;
			command_move->Add_Point(v1);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// target[2]
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(targets[2],d_artifact_settings.sg_koba_pnt_y,d_artifact_settings.sg_koba_pnt_z),TVector3(-1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("P2"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(v1);
			v1.x = targets[3] + clear_xy;
			command_move->Add_Point(v1);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// target[3]
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(targets[3],d_artifact_settings.sg_koba_pnt_y,d_artifact_settings.sg_koba_pnt_z),TVector3(-1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("P3"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(v1);
			v1.x = targets[4] + clear_xy;
			command_move->Add_Point(v1);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// target[4]
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(targets[4],d_artifact_settings.sg_koba_pnt_y,d_artifact_settings.sg_koba_pnt_z),TVector3(-1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("P4"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(v1);
			v1.x = targets[5] + clear_xy;
			command_move->Add_Point(v1);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// target[5]
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(targets[5],d_artifact_settings.sg_koba_pnt_y,d_artifact_settings.sg_koba_pnt_z),TVector3(-1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("P5"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(v1);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		}
	}
	else if(d_parameters.stepgauge_type == static_cast<int>(MeasureTypes::STEPGAUGE_STARRET))
	{
		starrett_clear_z = d_artifact_settings.sg_webber_clear_z + d_stepgauge_manual_points[0].z;
		
		for(measurement_cntr = 0;measurement_cntr < measurement_count;++measurement_cntr)
		{
			// move points
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(-clear_xy, d_stepgauge_manual_points[5].y, starrett_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// point start
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(0,d_stepgauge_manual_points[5].y,d_stepgauge_manual_points[5].z),TVector3(1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("PZ"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(-clear_xy, d_stepgauge_manual_points[5].y, starrett_clear_z));
			command_move->Add_Point(TVector3(targets[0] + clear_xy, 0, starrett_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// target[0]
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(targets[0],0,0),TVector3(-1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("P0"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(targets[0] + clear_xy, 0, starrett_clear_z));
			command_move->Add_Point(TVector3(targets[1] + clear_xy, 0, starrett_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// target[1]
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(targets[1],0,0),TVector3(-1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("P1"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(targets[1] + clear_xy, 0, starrett_clear_z));
			command_move->Add_Point(TVector3(targets[2] + clear_xy, 0, starrett_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// target[2]
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(targets[2],0,0),TVector3(-1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("P2"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(targets[2] + clear_xy, 0, starrett_clear_z));
			command_move->Add_Point(TVector3(targets[3] + clear_xy, 0, starrett_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// target[3]
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(targets[3],0,0),TVector3(-1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("P3"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(targets[3] + clear_xy, 0, starrett_clear_z));
			command_move->Add_Point(TVector3(targets[4] + clear_xy, 0, starrett_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// target[4]
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(targets[4],0,0),TVector3(-1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("P4"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(targets[4] + clear_xy, 0, starrett_clear_z));
			command_move->Add_Point(TVector3(targets[5] + clear_xy, 0, starrett_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// target[5]
			memory_p = memory_id++;
			command_point = new TCommandPoint(memory_p);
			command_point->Add_Path_Touch_Point(TVector3(targets[5],0,0),TVector3(-1,0,0));
			command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
			
			command_dimension = new TCommandDimension;
			command_dimension->Set_Memory_Id_Feature1(memory_p);
			command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("P5"),0.0);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
			
			// moves
			command_move = new TCommandMove;
			command_move->Add_Point(TVector3(targets[5] + clear_xy, 0, starrett_clear_z));
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		}
	}
	
	return true;
}

bool TMeasureRoutines::Measure_Gaugeblock(void)
{
	TCommandPoint						*command_point;
	TCommandMove						*command_move;
	TCommandDimension					*command_dimension;
	TCommandMode						*command_mode;
	QString								text;
	TVector3							v1;
	double								nominal_length;
	double								clear_xy;
	int									memory_id(MEMORY_GAUGEBLOCK_POINT_MEASURE);
	int									memory_p1;
	int									memory_p2;
	int									repeat_cntr;
	double								tip_radius;

	nominal_length = d_parameters.gaugeblock_nominal_length;
	
	if(nominal_length < MIN_GAUGEBLOCK_LENGTH)
	{
		emit Add_Log_Text(QString("ERR:  Gauge Block nominal length too small.  Min length: %1").arg(MIN_GAUGEBLOCK_LENGTH,0,'f',3));
		return false;
	}
	
	emit Add_Log_Text(TCommandDimension::Dimension_Item_Text_Title_Err());
	
	tip_radius = d_active_tool.tip_diameter / 2.0;
	
	if(d_output_file.isOpen())
	{
		d_output_file.write("Alignment_Begin:\n");
		
		v1 = d_refsys.Rotation_Mat().X();
		text = QString("X_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Y();
		text = QString("Y_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Z();
		text = QString("Z_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Origin();
		text = QString("Translation:%1,%2,%3\n").arg(v1.x,0,'f',5).arg(v1.y,0,'f',5).arg(v1.z,0,'f',5);
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Alignment_End:\n\n");
		
		d_output_file.write("Temperature_Begin:\n");
		
		text = QString("X_Scale:%1\n").arg(d_parameters.axis_x_temperature,0,'f',3);
		d_output_file.write(text.toLatin1());
		
		text = QString("Y_Scale:%1\n").arg(d_parameters.axis_y_temperature,0,'f',3);
		d_output_file.write(text.toLatin1());
		
		text = QString("Z_Scale:%1\n").arg(d_parameters.axis_z_temperature,0,'f',3);
		d_output_file.write(text.toLatin1());
		
		text = QString("Gauge:%1\n").arg(d_parameters.part_temperature,0,'f',3);
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Temperature_End:\n\n");
		
		text = QString("Probe_Offset:%1,%2,%3\n").arg(d_active_tool.xyz.x,0,'f',4).arg(d_active_tool.xyz.y,0,'f',4).arg(d_active_tool.xyz.z,0,'f',4);
		d_output_file.write(text.toLatin1());
		
		text = QString("Probe_Vector:%1,%2,%3\n\n").arg(d_active_tool.ijk.x,0,'f',12).arg(d_active_tool.ijk.y,0,'f',12).arg(d_active_tool.ijk.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Measurement_Begin:\n\n");
		
		text = QString("Description:%1\n").arg(d_parameters.gaugeblock_position_name);
		d_output_file.write(text.toLatin1());
	}
	
	clear_xy = d_approach_distance;
	clear_xy += tip_radius;
	
	if(d_manual_mode)
	{
		// dcc mode
		command_mode = new TCommandMode();
		command_mode->Set_DCC_Mode();
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	}
	
	command_dimension = new TCommandDimension;
	command_dimension->Set_Memory_Id_Feature1(MEMORY_GAUGEBLOCK_PLANE_DCC);
	command_dimension->Add_Dimension(TCommandDimension::FIELD_FORM,QStringLiteral("Flatness"),0.0,0.0);
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
	
	for(repeat_cntr = 0;repeat_cntr < 3;++repeat_cntr)
	{
		// move points
		command_move = new TCommandMove;
		command_move->Add_Point(TVector3(-clear_xy, d_artifact_settings.gaugeblock_pln_y, d_artifact_settings.gaugeblock_clear_z));
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		
		// point start
		memory_p1 = memory_id++;
		command_point = new TCommandPoint(memory_p1);
		command_point->Add_Path_Touch_Point(TVector3(0,d_artifact_settings.gaugeblock_pln_y,d_artifact_settings.gaugeblock_pln_z),TVector3(1,0,0));
		command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
		
		command_dimension = new TCommandDimension;
		command_dimension->Set_Memory_Id_Feature1(memory_p1);
		command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("PtA.X"),0.0);
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
		
		// move points
		command_move = new TCommandMove;
		command_move->Add_Point(TVector3(-clear_xy, d_artifact_settings.gaugeblock_pln_y, d_artifact_settings.gaugeblock_clear_z));
		command_move->Add_Point(TVector3(nominal_length + clear_xy, d_artifact_settings.gaugeblock_pln_y, d_artifact_settings.gaugeblock_clear_z));
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		
		// end start
		memory_p2 = memory_id++;
		command_point = new TCommandPoint(memory_p2);
		command_point->Add_Path_Touch_Point(TVector3(nominal_length,d_artifact_settings.gaugeblock_pln_y,d_artifact_settings.gaugeblock_pln_z),TVector3(-1,0,0));
		command_point->Set_Output_Format(TCommandFeature::FORMAT_NONE);
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
		
		command_dimension = new TCommandDimension;
		command_dimension->Set_Memory_Id_Feature1(memory_p2);
		command_dimension->Add_Dimension(TCommandDimension::FIELD_X,QStringLiteral("PtB.X"),0.0);
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
		
		command_dimension = new TCommandDimension;
		command_dimension->Set_Memory_Id_Feature1(memory_p1);
		command_dimension->Set_Memory_Id_Feature2(memory_p2);
		command_dimension->Add_Dimension(TCommandDimension::FIELD_DIST_X,QStringLiteral("PtB-PtA"),0.0,0.0);
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
		
		// move points
		command_move = new TCommandMove;
		command_move->Add_Point(TVector3(nominal_length + clear_xy, d_artifact_settings.gaugeblock_pln_y, d_artifact_settings.gaugeblock_clear_z));
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	}
	
	return true;
}

bool TMeasureRoutines::Measure_Point_Repeat(void)
{
	TCommandPoint						*command_point;
	TCommandMove						*command_move;
	TCommandMode						*command_mode;
	QString								text;
	TVector3							v1;
	double								move_distance;
	int									memory_id(MEMORY_POINT_REPEAT_MEASURE);
	int									memory_point;
	int									repeat_cntr;
	int									repeat_count;

	emit this->Add_Log_Text(TCommandFeature::Output_Title_XYZ());
	
	move_distance = d_parameters.point_repeat_move_distance;
	
	if(move_distance < MIN_MOVE_DISTANCE)
	{
		emit Add_Log_Text(QString("ERR:  Move distance is too small.  Minimum move distance: %1").arg(MIN_MOVE_DISTANCE,0,'f',3));
		return false;
	}
	
	if(d_artifact_settings.plane_sample_radius < MIN_PLANE_SPACER)
	{
		emit Add_Log_Text(QString("ERR:  Point sample plane spacer too small.  Min plane spacer: %1").arg(MIN_PLANE_SPACER,0,'f',3));
		return false;
	}
	
	if(d_output_file.isOpen())
	{
		d_output_file.write("Alignment_Begin:\n");
		
		v1 = d_refsys.Rotation_Mat().X();
		text = QString("X_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Y();
		text = QString("Y_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Z();
		text = QString("Z_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Origin();
		text = QString("Translation:%1,%2,%3\n").arg(v1.x,0,'f',5).arg(v1.y,0,'f',5).arg(v1.z,0,'f',5);
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Alignment_End:\n\n");
		
		text = QString("Probe_Offset:%1,%2,%3\n").arg(d_active_tool.xyz.x,0,'f',4).arg(d_active_tool.xyz.y,0,'f',4).arg(d_active_tool.xyz.z,0,'f',4);
		d_output_file.write(text.toLatin1());
		
		text = QString("Probe_Vector:%1,%2,%3\n\n").arg(d_active_tool.ijk.x,0,'f',12).arg(d_active_tool.ijk.y,0,'f',12).arg(d_active_tool.ijk.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Measurement_Begin:\n\n");
		
		text = QString("Description:%1\n").arg(d_parameters.point_repeat_description);
		d_output_file.write(text.toLatin1());

		text = QString("Move Distance:%1\n").arg(move_distance,0,'f',3);
		d_output_file.write(text.toLatin1());

	}
	
	repeat_count = d_parameters.point_repeat_measurement_count;
	
	if(repeat_count < 1)
	{
		emit Add_Log_Text(QStringLiteral("ERR:  Repeat count must be greater than zero."));
		return false;
	}
	
	if(d_manual_mode)
	{
		// dcc mode
		command_mode = new TCommandMode();
		command_mode->Set_DCC_Mode();
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	}

	for(repeat_cntr = 0;repeat_cntr < repeat_count;++repeat_cntr)
	{
		// move points
		command_move = new TCommandMove;
		command_move->Add_Point(TVector3(move_distance, 0, 0));
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		
		// point
		memory_point = memory_id++;
		command_point = new TCommandPoint(memory_point);
		
		command_point->Add_Path_Touch_Point(TVector3(0,0,0),TVector3(-1.0,0,0));
		command_point->Set_Output_Format(TCommandFeature::FORMAT_XYZ);
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_point));
	}

	
	return true;
}

bool TMeasureRoutines::Measure_Sphere_Rpt(void)
{
	TCommandSphere						*command_sphere;
	TCommandMove						*command_move;
	TCommandMode						*command_mode;
	QString								text;
	TVector3							vec_60;
	TVector3							v1;
	double								sphere_radius;
	double								clear_dist;
	double								sphere_diameter;
	int									memory_id(MEMORY_SPHERE_REPEAT_MEASURE);
	int									memory_sphere;
	int									repeat_cntr;
	int									repeat_count;
	double								tip_radius;

	emit this->Add_Log_Text(TCommandFeature::Output_Title_XYZD());
	
	tip_radius = d_active_tool.tip_diameter / 2.0;

	if(d_output_file.isOpen())
	{
		d_output_file.write("Alignment_Begin:\n");
		
		v1 = d_refsys.Rotation_Mat().X();
		text = QString("X_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Y();
		text = QString("Y_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Z();
		text = QString("Z_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Origin();
		text = QString("Translation:%1,%2,%3\n").arg(v1.x,0,'f',5).arg(v1.y,0,'f',5).arg(v1.z,0,'f',5);
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Alignment_End:\n\n");
		
		text = QString("Probe_Offset:%1,%2,%3\n").arg(d_active_tool.xyz.x,0,'f',4).arg(d_active_tool.xyz.y,0,'f',4).arg(d_active_tool.xyz.z,0,'f',4);
		d_output_file.write(text.toLatin1());
		
		text = QString("Probe_Vector:%1,%2,%3\n\n").arg(d_active_tool.ijk.x,0,'f',12).arg(d_active_tool.ijk.y,0,'f',12).arg(d_active_tool.ijk.z,0,'f',12);
		d_output_file.write(text.toLatin1());
	}
	
	sphere_diameter = d_parameters.sphere_nominal_diameter;
	
	if(sphere_diameter < MIN_SPHERE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Sphere diameter too small.  Minimum diameter: %1").arg(MIN_SPHERE_DIAMETER,0,'f',3));
		return false;
	}
	
	sphere_radius = sphere_diameter / 2.0;
	
	repeat_count = d_parameters.sphere_repeat_measurement_count;
	
	if(repeat_count < 1)
	{
		emit Add_Log_Text(QStringLiteral("ERR:  Repeat count must be greater than zero."));
		return false;
	}
	
	clear_dist = d_approach_distance;
	clear_dist += tip_radius;
	clear_dist += sphere_radius;
	
	vec_60.Set(0.5,0.866025403784439,0);
	
	if(d_manual_mode)
	{
		// dcc mode
		command_mode = new TCommandMode();
		command_mode->Set_DCC_Mode();
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	}
	
	for(repeat_cntr = 0;repeat_cntr < repeat_count;++repeat_cntr)
	{
		// move points
		command_move = new TCommandMove;
		command_move->Add_Point(TVector3(0, 0, clear_dist));
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		
		// sphere
		memory_sphere = memory_id++;
		command_sphere = new TCommandSphere(memory_sphere);
		command_sphere->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_OD);
		
		command_sphere->Add_Path_Touch_Point(TVector3(0,0,sphere_radius),TVector3(0,0,-1));
		command_sphere->Add_Path_Move_Point(TVector3(0,0,clear_dist));
		
		command_sphere->Add_Path_Move_Point(TVector3(-clear_dist,0,clear_dist));
		command_sphere->Add_Path_Touch_Point(TVector3((-1.0 * sphere_radius),0,0),TVector3(1,0,0));
		command_sphere->Add_Path_Move_Point(TVector3(-clear_dist,0,clear_dist));
		
		command_sphere->Add_Path_Move_Point(TVector3(clear_dist * vec_60.x,-1 * clear_dist * vec_60.y,clear_dist));
		command_sphere->Add_Path_Touch_Point(TVector3(sphere_radius * vec_60.x,-1 * sphere_radius * vec_60.y,0),TVector3(-vec_60.x,vec_60.y,0));
		command_sphere->Add_Path_Move_Point(TVector3(clear_dist * vec_60.x,-1 * clear_dist * vec_60.y,clear_dist));
		
		command_sphere->Add_Path_Move_Point(TVector3(clear_dist * vec_60.x,clear_dist * vec_60.y,clear_dist));
		command_sphere->Add_Path_Touch_Point(TVector3(sphere_radius * vec_60.x,sphere_radius * vec_60.y,0),TVector3(-vec_60.x,-vec_60.y,0));
		command_sphere->Set_Output_Format(TCommandFeature::FORMAT_XYZD);
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_sphere));
		
		// move points
		command_move = new TCommandMove;
		command_move->Add_Point(TVector3(clear_dist * vec_60.x,clear_dist * vec_60.y,clear_dist));
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	}
	
	return true;
}

bool TMeasureRoutines::Measure_Sphere_Pftu(void)
{
	TCommandSphere						*command_sphere;
	TCommandMove						*command_move;
	TCommandMode						*command_mode;
	TVector3							vec;
	TVector3							v1;
	QString								text;
	double								sphere_radius;
	double								clear_dist;
	double								sphere_diameter;
	TVector3							point_table[25];
	TVector3							move_table[12];
	double								tip_radius;

	emit this->Add_Log_Text(TCommandFeature::Output_Title_XYZDF());
	
	tip_radius = d_active_tool.tip_diameter / 2.0;

	if(d_output_file.isOpen())
	{
		d_output_file.write("Alignment_Begin:\n");
		
		v1 = d_refsys.Rotation_Mat().X();
		text = QString("X_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Y();
		text = QString("Y_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Z();
		text = QString("Z_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Origin();
		text = QString("Translation:%1,%2,%3\n").arg(v1.x,0,'f',5).arg(v1.y,0,'f',5).arg(v1.z,0,'f',5);
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Alignment_End:\n\n");
		
		text = QString("Probe_Offset:%1,%2,%3\n").arg(d_active_tool.xyz.x,0,'f',4).arg(d_active_tool.xyz.y,0,'f',4).arg(d_active_tool.xyz.z,0,'f',4);
		d_output_file.write(text.toLatin1());
		
		text = QString("Probe_Vector:%1,%2,%3\n\n").arg(d_active_tool.ijk.x,0,'f',12).arg(d_active_tool.ijk.y,0,'f',12).arg(d_active_tool.ijk.z,0,'f',12);
		d_output_file.write(text.toLatin1());
	}
	
	sphere_diameter = d_parameters.sphere_nominal_diameter;
	
	if(sphere_diameter < MIN_SPHERE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Sphere diameter too small.  Minimum diameter: %1").arg(MIN_SPHERE_DIAMETER,0,'f',3));
		return false;
	}
	
	sphere_radius = sphere_diameter / 2.0;
	
	clear_dist = d_approach_distance;
	clear_dist += tip_radius;
	clear_dist += sphere_radius;
	
	// approach vector
	point_table[0].Set(0.000000000,0.000000000,-1.000000000);
	point_table[1].Set(-0.382683432,0.000000000,-0.923879533);
	point_table[2].Set(0.000000000,-0.382683432,-0.923879533);
	point_table[3].Set(0.382683432,0.000000000,-0.923879533);
	point_table[4].Set(0.000000000,0.382683432,-0.923879533);
	point_table[5].Set(-0.653281482,-0.270598050,-0.707106781);
	point_table[6].Set(-0.270598050,-0.653281482,-0.707106781);
	point_table[7].Set(0.270598050,-0.653281482,-0.707106781);
	point_table[8].Set(0.653281482,-0.270598050,-0.707106781);
	point_table[9].Set(0.653281482,0.270598050,-0.707106781);
	point_table[10].Set(0.270598050,0.653281482,-0.707106781);
	point_table[11].Set(-0.270598050,0.653281482,-0.707106781);
	point_table[12].Set(-0.653281482,0.270598050,-0.707106781);
	point_table[13].Set(-0.653281482,-0.653281482,-0.382683432);
	point_table[14].Set(0.653281482,-0.653281482,-0.382683432);
	point_table[15].Set(0.653281482,0.653281482,-0.382683432);
	point_table[16].Set(-0.653281482,0.653281482,-0.382683432);
	point_table[17].Set(-0.923879533,-0.382683432,0.000000000);
	point_table[18].Set(-0.382683432,-0.923879533,0.000000000);
	point_table[19].Set(0.382683432,-0.923879533,0.000000000);
	point_table[20].Set(0.923879533,-0.382683432,0.000000000);
	point_table[21].Set(0.923879533,0.382683432,0.000000000);
	point_table[22].Set(0.382683432,0.923879533,0.000000000);
	point_table[23].Set(-0.382683432,0.923879533,0.000000000);
	point_table[24].Set(-0.923879533,0.382683432,0.000000000);
	
	// surface normal vector
	move_table[0].Set(0.768177757,0.318189645,0.555570233);
	move_table[1].Set(0.000000000,0.923879533,0.382683432);
	move_table[2].Set(-0.923879533,0.000000000,0.382683432);
	move_table[3].Set(0.000000000,-0.923879533,0.382683432);
	move_table[4].Set(0.961939766,-0.191341716,0.195090322);
	move_table[5].Set(0.707106781,0.707106781,0.000000000);
	move_table[6].Set(0.000000000,1.000000000,0.000000000);
	move_table[7].Set(-0.707106781,0.707106781,0.000000000);
	move_table[8].Set(-1.000000000,0.000000000,0.000000000);
	move_table[9].Set(-0.707106781,-0.707106781,0.000000000);
	move_table[10].Set(0.000000000,-1.000000000,0.000000000);
	move_table[11].Set(0.707106781,-0.707106781,0.000000000);
	
	if(d_manual_mode)
	{
		// dcc mode
		command_mode = new TCommandMode();
		command_mode->Set_DCC_Mode();
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	}
	
	// move points
	command_move = new TCommandMove;
	command_move->Add_Point(TVector3(0, 0, clear_dist));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	
	// sphere
	command_sphere = new TCommandSphere(MEMORY_SPHERE_PFTU_MEASURE);
	command_sphere->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_OD);
	
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[0],point_table[0]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[1],point_table[1]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[2],point_table[2]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[3],point_table[3]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[4],point_table[4]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[5],point_table[5]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[6],point_table[6]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[7],point_table[7]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[8],point_table[8]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[9],point_table[9]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[10],point_table[10]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[11],point_table[11]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[12],point_table[12]);
	command_sphere->Add_Path_Move_Point(clear_dist * move_table[0]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[13],point_table[13]);
	command_sphere->Add_Path_Move_Point(clear_dist * move_table[1]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[14],point_table[14]);
	command_sphere->Add_Path_Move_Point(clear_dist * move_table[2]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[15],point_table[15]);
	command_sphere->Add_Path_Move_Point(clear_dist * move_table[3]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[16],point_table[16]);
	command_sphere->Add_Path_Move_Point(clear_dist * move_table[4]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[17],point_table[17]);
	command_sphere->Add_Path_Move_Point(clear_dist * move_table[5]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[18],point_table[18]);
	command_sphere->Add_Path_Move_Point(clear_dist * move_table[6]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[19],point_table[19]);
	command_sphere->Add_Path_Move_Point(clear_dist * move_table[7]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[20],point_table[20]);
	command_sphere->Add_Path_Move_Point(clear_dist * move_table[8]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[21],point_table[21]);
	command_sphere->Add_Path_Move_Point(clear_dist * move_table[9]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[22],point_table[22]);
	command_sphere->Add_Path_Move_Point(clear_dist * move_table[10]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[23],point_table[23]);
	command_sphere->Add_Path_Move_Point(clear_dist * move_table[11]);
	command_sphere->Add_Path_Touch_Point(-sphere_radius * point_table[24],point_table[24]);
	
	
	command_sphere->Set_Output_Format(TCommandFeature::FORMAT_XYZDF);
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_sphere));
	
	// move points
	command_move = new TCommandMove;
	
	vec = -1.0 * point_table[24];
	vec *= clear_dist;
	command_move->Add_Point(vec);
	
	vec.z = clear_dist;
	command_move->Add_Point(vec);
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	
	return true;
}

bool TMeasureRoutines::Measure_Ballbar_B89(void)
{
	TMeasureRoutines::TFeatureMemory	feature_memory;
	TCommandMove						*command_move;
	TCommandSphere						*command_sphere;
	TCommandDimension					*command_dimension;
	TCommandMode						*command_mode;
	TVector3							sph1;
	TVector3							sph2;
	TVector3							vec;
	TVector3							bar_axis;
	TVector3							clearance_axis;
	double								sphere_diameter;
	double								sphere_radius;
	double								ballbar_length;
	double								angle_start;
	double								angle_range;
	double								angle_increment;
	double								angle;
	double								clear_dist;
	int									memory_s1;
	int									memory_s2;
	int									pnt_cntr;
	int									meas_cntr;
	int									meas_count;
	double								tip_radius;
	bool								valid;
	static const double					DEG2RAD(0.01745329251994);
	static const double					BALLBAR_LENGTH_MIN(10.0);
	
	emit this->Add_Log_Text(TCommandFeature::Output_Title_XYZDF());
	
	sphere_diameter = d_parameters.ballbar_sphere_diameter;
	
	if(sphere_diameter < MIN_SPHERE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Ballbar sphere diameter too small.  Minimum diameter: %1").arg(MIN_SPHERE_DIAMETER,0,'f',3));
		return false;
	}
		
	meas_count = d_parameters.ballbar_measurement_count;
	
	if(meas_count < 1)
	{
		emit Add_Log_Text(QString("ERR:  Measurement count must be greater than zero.  Measurement count: %1").arg(meas_count));
		return false;
	}
	
	sphere_radius = sphere_diameter / 2.0;
	tip_radius = d_active_tool.tip_diameter / 2.0;
	
	feature_memory = this->Feature_Memory(MEMORY_BALLBAR_SPHERE1_DCC,&valid);
	
	if(valid)
	{
		sph1 = feature_memory.xyz;
	}
	else
	{
		emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(MEMORY_BALLBAR_SPHERE1_DCC));
		return false;
	}
	
	feature_memory = this->Feature_Memory(MEMORY_BALLBAR_SPHERE2_DCC,&valid);
	
	if(valid)
	{
		sph2 = feature_memory.xyz;
	}
	else
	{
		emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(MEMORY_BALLBAR_SPHERE2_DCC));
		return false;
	}
	
	bar_axis = sph2 - sph1;
	ballbar_length = bar_axis.Length();
	
	if(ballbar_length < BALLBAR_LENGTH_MIN)
	{
		emit Add_Log_Text(QString("ERR:  Ballbar length too short.  Minimum length: %1").arg(BALLBAR_LENGTH_MIN,0,'f',3));
		return false;
	}
	
	bar_axis.Normal();
	
	sph1 = d_refsys.FromWorld(sph1);
	sph2 = d_refsys.FromWorld(sph2);
	
	clear_dist = d_approach_distance;
	clear_dist += tip_radius;
	clear_dist += sphere_radius;
	
	angle_start = d_artifact_settings.ballbar_avoidance_angle * DEG2RAD;
	angle_range = (360.0 - 2.0 * d_artifact_settings.ballbar_avoidance_angle) * DEG2RAD;
	angle_increment = angle_range / 6.0;
	
	vec = d_active_tool.ijk * bar_axis;
	clearance_axis = vec * bar_axis;
	
	if(clearance_axis.Length() > 0.001)
	{
		clearance_axis.Normal();
	}
	else
	{
		clearance_axis.Set(0,0,1);
	}
	
	if((clearance_axis ^ d_active_tool.ijk) > 0)
	{
		clearance_axis *= (-1.0);
	}
	
	clearance_axis = d_refsys.AxisFromWorld(clearance_axis);
	
	if(d_manual_mode)
	{
		// dcc mode
		command_mode = new TCommandMode();
		command_mode->Set_DCC_Mode();
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	}
	
	for(meas_cntr = 0;meas_cntr < meas_count;++meas_cntr)
	{
		if(!(meas_cntr % 2))
		{
			// sphere1
			memory_s1 = MEMORY_BALLBAR_SPHERES_MEASURE + meas_cntr * 2;
			command_sphere = new TCommandSphere(memory_s1);
			command_sphere->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_OD);
			
			command_sphere->Set_Output_Format(TCommandFeature::FORMAT_XYZDF);
			
			angle = angle_start;
			for(pnt_cntr = 0;pnt_cntr < 8;++pnt_cntr)
			{
				if(pnt_cntr == 0)
				{
					command_sphere->Add_Path_Move_Point(sph1 + clearance_axis * d_artifact_settings.ballbar_clear_distance);
					command_sphere->Add_Path_Touch_Point(sph1 + TVector3(0,0,sphere_radius),TVector3(0,0,-1));
					command_sphere->Add_Path_Move_Point(sph1 + TVector3(0,0,clear_dist));
				}
				else if(pnt_cntr % 2)
				{
					vec.Set(cos(angle) * 0.707106781186548,sin(angle) * 0.707106781186548, 0.707106781186548);
					
					command_sphere->Add_Path_Move_Point(sph1 + vec * clear_dist);
					command_sphere->Add_Path_Touch_Point(sph1 + vec * sphere_radius,-1.0 * vec);
					command_sphere->Add_Path_Move_Point(sph1 + vec * clear_dist);
					
					angle += angle_increment;
				}
				else
				{
					vec.Set(cos(angle),sin(angle),0.0);
					
					command_sphere->Add_Path_Move_Point(sph1 + vec * clear_dist);
					command_sphere->Add_Path_Touch_Point(sph1 + vec * sphere_radius,-1.0 * vec);
					command_sphere->Add_Path_Move_Point(sph1 + vec * clear_dist);
					
					angle += angle_increment;
				}
			}
			
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_sphere));
			
			// move points
			command_move = new TCommandMove;
			command_move->Add_Point(sph1 + vec * clear_dist);
			command_move->Add_Point(sph1 + clearance_axis * d_artifact_settings.ballbar_clear_distance);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// sphere2
			memory_s2 = MEMORY_BALLBAR_SPHERES_MEASURE + 1 + meas_cntr * 2;
			command_sphere = new TCommandSphere(memory_s2);
			command_sphere->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_OD);
			
			command_sphere->Set_Output_Format(TCommandFeature::FORMAT_XYZDF);
			
			angle = angle_start + 3.141592653589793;
			for(pnt_cntr = 0;pnt_cntr < 8;++pnt_cntr)
			{
				if(pnt_cntr == 0)
				{
					command_sphere->Add_Path_Move_Point(sph2 + clearance_axis * d_artifact_settings.ballbar_clear_distance);
					command_sphere->Add_Path_Touch_Point(sph2 + TVector3(0,0,sphere_radius),TVector3(0,0,-1));
					command_sphere->Add_Path_Move_Point(sph2 + TVector3(0,0,clear_dist));
				}
				else if(pnt_cntr % 2)
				{
					vec.Set(cos(angle) * 0.707106781186548,sin(angle) * 0.707106781186548, 0.707106781186548);
					
					command_sphere->Add_Path_Move_Point(sph2 + vec * clear_dist);
					command_sphere->Add_Path_Touch_Point(sph2 + vec * sphere_radius,-1.0 * vec);
					command_sphere->Add_Path_Move_Point(sph2 + vec * clear_dist);
					
					angle += angle_increment;
				}
				else
				{
					vec.Set(cos(angle),sin(angle),0.0);
					
					command_sphere->Add_Path_Move_Point(sph2 + vec * clear_dist);
					command_sphere->Add_Path_Touch_Point(sph2 + vec * sphere_radius,-1.0 * vec);
					command_sphere->Add_Path_Move_Point(sph2 + vec * clear_dist);
					
					angle += angle_increment;
				}
			}
			
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_sphere));
			
			// move points
			command_move = new TCommandMove;
			command_move->Add_Point(sph2 + vec * clear_dist);
			command_move->Add_Point(sph2 + clearance_axis * d_artifact_settings.ballbar_clear_distance);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		}
		else
		{
			// sphere2
			memory_s2 = MEMORY_BALLBAR_SPHERES_MEASURE + 1 + meas_cntr * 2;
			command_sphere = new TCommandSphere(memory_s2);
			command_sphere->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_OD);
			
			command_sphere->Set_Output_Format(TCommandFeature::FORMAT_XYZDF);
			
			angle = angle_start + 3.141592653589793;
			for(pnt_cntr = 0;pnt_cntr < 8;++pnt_cntr)
			{
				if(pnt_cntr == 0)
				{
					command_sphere->Add_Path_Move_Point(sph2 + clearance_axis * d_artifact_settings.ballbar_clear_distance);
					command_sphere->Add_Path_Touch_Point(sph2 + TVector3(0,0,sphere_radius),TVector3(0,0,-1));
					command_sphere->Add_Path_Move_Point(sph2 + TVector3(0,0,clear_dist));
				}
				else if(pnt_cntr % 2)
				{
					vec.Set(cos(angle) * 0.707106781186548,sin(angle) * 0.707106781186548, 0.707106781186548);
					
					command_sphere->Add_Path_Move_Point(sph2 + vec * clear_dist);
					command_sphere->Add_Path_Touch_Point(sph2 + vec * sphere_radius,-1.0 * vec);
					command_sphere->Add_Path_Move_Point(sph2 + vec * clear_dist);
					
					angle += angle_increment;
				}
				else
				{
					vec.Set(cos(angle),sin(angle),0.0);
					
					command_sphere->Add_Path_Move_Point(sph2 + vec * clear_dist);
					command_sphere->Add_Path_Touch_Point(sph2 + vec * sphere_radius,-1.0 * vec);
					command_sphere->Add_Path_Move_Point(sph2 + vec * clear_dist);
					
					angle += angle_increment;
				}
			}
			
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_sphere));
			
			// move points
			command_move = new TCommandMove;
			command_move->Add_Point(sph2 + vec * clear_dist);
			command_move->Add_Point(sph2 + clearance_axis * d_artifact_settings.ballbar_clear_distance);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
			// sphere1
			memory_s1 = MEMORY_BALLBAR_SPHERES_MEASURE + meas_cntr * 2;
			command_sphere = new TCommandSphere(memory_s1);
			command_sphere->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_OD);
			
			command_sphere->Set_Output_Format(TCommandFeature::FORMAT_XYZDF);
			
			angle = angle_start;
			for(pnt_cntr = 0;pnt_cntr < 8;++pnt_cntr)
			{
				if(pnt_cntr == 0)
				{
					command_sphere->Add_Path_Move_Point(sph1 + clearance_axis * d_artifact_settings.ballbar_clear_distance);
					command_sphere->Add_Path_Touch_Point(sph1 + TVector3(0,0,sphere_radius),TVector3(0,0,-1));
					command_sphere->Add_Path_Move_Point(sph1 + TVector3(0,0,clear_dist));
				}
				else if(pnt_cntr % 2)
				{
					vec.Set(cos(angle) * 0.707106781186548,sin(angle) * 0.707106781186548, 0.707106781186548);
					
					command_sphere->Add_Path_Move_Point(sph1 + vec * clear_dist);
					command_sphere->Add_Path_Touch_Point(sph1 + vec * sphere_radius,-1.0 * vec);
					command_sphere->Add_Path_Move_Point(sph1 + vec * clear_dist);
					
					angle += angle_increment;
				}
				else
				{
					vec.Set(cos(angle),sin(angle),0.0);
					
					command_sphere->Add_Path_Move_Point(sph1 + vec * clear_dist);
					command_sphere->Add_Path_Touch_Point(sph1 + vec * sphere_radius,-1.0 * vec);
					command_sphere->Add_Path_Move_Point(sph1 + vec * clear_dist);
					
					angle += angle_increment;
				}
			}
			
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_sphere));
			
			// move points
			command_move = new TCommandMove;
			command_move->Add_Point(sph1 + vec * clear_dist);
			command_move->Add_Point(sph1 + clearance_axis * d_artifact_settings.ballbar_clear_distance);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		}
		
		
		// dimension
		command_dimension = new TCommandDimension;
		command_dimension->Set_Memory_Id_Feature1(memory_s1);
		command_dimension->Set_Memory_Id_Feature2(memory_s2);
		command_dimension->Add_Dimension(TCommandDimension::FIELD_DIST_XYZ,QStringLiteral("Sph2-Sph1"),0.0);
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
	}
	
	return true;
}

bool TMeasureRoutines::Measure_Ballbar_10360(void)
{
	TMeasureRoutines::TFeatureMemory	feature_memory;
	TCommandMove						*command_move;
	TCommandSphere						*command_sphere;
	TCommandDimension					*command_dimension;
	TCommandMode						*command_mode;
	TVector3							sph0;
	TVector3							sph5;
	TVector3							sphere_position;
	TVector3							vec;
	TVector3							bar_axis;
	TVector3							clearance_axis;
	QString								text;
	double								sphere_diameter;
	double								sphere_radius;
	double								angle_increment;
	double								angle;
	double								length;
	double								length_min;
	double								length_max;
	double								clear_dist;
	double								tip_radius;
	int									memory_s1;
	int									memory_s2;
	int									pnt_count;
	int									level_count[2];
	int									level_cntr;
	int									meas_cntr;
	int									index;
	int									feature_index;
	bool								add_clearance_moves[2];
	bool								valid;
	static const double					LENGTH_TOLERANCE(4.0);	// actual length must be within this tolerance based on the nominal sphere spacing
	static const double					PI2(6.28318530718);
	
	emit this->Add_Log_Text(TCommandFeature::Output_Title_XYZDF());
	
	sphere_diameter = d_parameters.ballbar_10360_sphere_diameter;
	
	if(sphere_diameter < MIN_SPHERE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Ballbar 10360 sphere diameter too small.  Minimum diameter: %1").arg(MIN_SPHERE_DIAMETER,0,'f',3));
		return false;
	}
	
	feature_memory = this->Feature_Memory(MEMORY_BALLBAR_10360_SPHERE0_DCC,&valid);
	
	if(valid)
	{
		sph0 = feature_memory.xyz;
	}
	else
	{
		emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(MEMORY_BALLBAR_10360_SPHERE0_DCC));
		return false;
	}

	feature_memory = this->Feature_Memory(MEMORY_BALLBAR_10360_SPHERE5_DCC,&valid);
	
	if(valid)
	{
		sph5 = feature_memory.xyz;
	}
	else
	{
		emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(MEMORY_BALLBAR_10360_SPHERE5_DCC));
		return false;
	}

	sphere_radius = sphere_diameter / 2.0;
	tip_radius = d_active_tool.tip_diameter / 2.0;
	
	clear_dist = d_approach_distance;
	clear_dist += tip_radius;
	clear_dist += sphere_radius;
	
	bar_axis = sph5 - sph0;

	length_max = bar_axis.Length();

	bar_axis.Normal();

	index = static_cast<int>(0.5 + length_max / d_parameters.ballbar_10360_sphere_spacing);
	length = d_parameters.ballbar_10360_sphere_spacing * static_cast<double>(index);
	sphere_position = sph0 + (d_parameters.ballbar_10360_sphere_spacing * static_cast<double>(index)) * bar_axis;

	vec = sph5 - sphere_position;
	length = vec.Length();
		
	if(length > LENGTH_TOLERANCE)
	{
		emit Add_Log_Text(QStringLiteral("ERR:  Ballbar 10360 spacing entry appears incorrect.  Last sphere does not align to sphere spacing"));
		return false;
	}
	
	sph0 = d_refsys.FromWorld(sph0);
	sph5 = d_refsys.FromWorld(sph5);

	pnt_count = d_artifact_settings.ballbar_10360_sphere_points;
	
	if(pnt_count < 4)
	{
		emit Add_Log_Text(QStringLiteral("ERR:  Ballbar 10360 point count too small.  Minimum 4 points required"));
		return false;
	}
	
	if(pnt_count < 14)
	{
		level_count[0] = pnt_count - 1;
		level_count[1] = 0;
	}
	else
	{
		if(pnt_count % 2) // odd number
		{
			level_count[0] = pnt_count / 2;
			level_count[1] = level_count[0];
		}
		else
		{
			level_count[0] = pnt_count / 2;
			level_count[1] = level_count[0] - 1;
		}
	}
	
	add_clearance_moves[0] = false;
	add_clearance_moves[1] = false;
		
	if(level_count[0] < 8)
	{
		add_clearance_moves[0] = true;
	}
	
	if(level_count[1] && (level_count[1] < 4))
	{
		add_clearance_moves[1] = true;
	}
			
	vec = d_active_tool.ijk * bar_axis;
	clearance_axis = vec * bar_axis;
	
	if(clearance_axis.Length() > 0.001)
	{
		clearance_axis.Normal();
	}
	else
	{
		clearance_axis.Set(0,0,1);
	}
	
	if((clearance_axis ^ d_active_tool.ijk) > 0)
	{
		clearance_axis *= (-1.0);
	}
	
	clearance_axis = d_refsys.AxisFromWorld(clearance_axis);
	
	if(d_manual_mode)
	{
		// dcc mode
		command_mode = new TCommandMode();
		command_mode->Set_DCC_Mode();
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	}
	
	length_min = length_max / 5.0;
		
	for(meas_cntr = 0,feature_index = 0;meas_cntr != 15;++meas_cntr)
	{
		// sphere0
		memory_s1 = MEMORY_BALLBAR_10360_SPHERES_MEASURE + feature_index;
		++feature_index;
		
		command_sphere = new TCommandSphere(memory_s1);
		command_sphere->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_OD);
		
		command_sphere->Set_Output_Format(TCommandFeature::FORMAT_XYZDF);
		
		command_sphere->Add_Path_Move_Point(sph0 + clearance_axis * d_artifact_settings.ballbar_10360_clear_distance);
		command_sphere->Add_Path_Touch_Point(sph0 + TVector3(0,0,sphere_radius),TVector3(0,0,-1));
		command_sphere->Add_Path_Move_Point(sph0 + TVector3(0,0,clear_dist));

		if(level_count[1])
		{
			angle = 0.0;
			angle_increment = PI2 / static_cast<double>(level_count[1]);
			
			for(level_cntr = 0;level_cntr < level_count[1];++level_cntr)
			{
				if(level_cntr == 0 || add_clearance_moves[1])
				{
					vec.Set(cos(angle),sin(angle),1.0);
					command_sphere->Add_Path_Move_Point(sph0 + vec * clear_dist);
				}

				vec.Set(cos(angle) * 0.707106781186548,sin(angle) * 0.707106781186548, 0.707106781186548);
				
				command_sphere->Add_Path_Move_Point(sph0 + vec * clear_dist);
				command_sphere->Add_Path_Touch_Point(sph0 + vec * sphere_radius,-1.0 * vec);
				command_sphere->Add_Path_Move_Point(sph0 + vec * clear_dist);
				
				if((level_cntr + 1) == level_count[1] || add_clearance_moves[1])
				{
					vec.Set(cos(angle),sin(angle),1.0);
					command_sphere->Add_Path_Move_Point(sph0 + vec * clear_dist);
				}

				angle += angle_increment;
			}
		}
		
		if(level_count[0])
		{
			angle = 0.0;
			angle_increment = PI2 / static_cast<double>(level_count[0]);
			
			for(level_cntr = 0;level_cntr < level_count[0];++level_cntr)
			{
				if(level_cntr == 0 || add_clearance_moves[0])
				{
					vec.Set(cos(angle),sin(angle),1.0);
					command_sphere->Add_Path_Move_Point(sph0 + vec * clear_dist);
				}

				vec.Set(cos(angle),sin(angle),0.0);
							
				command_sphere->Add_Path_Move_Point(sph0 + vec * clear_dist);
				command_sphere->Add_Path_Touch_Point(sph0 + vec * sphere_radius,-1.0 * vec);
				command_sphere->Add_Path_Move_Point(sph0 + vec * clear_dist);
				
				if((level_cntr + 1) == level_count[0] || add_clearance_moves[0])
				{
					vec.Set(cos(angle),sin(angle),1.0);
					command_sphere->Add_Path_Move_Point(sph0 + vec * clear_dist);
				}

				angle += angle_increment;
			}
		}
		
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_sphere));
		
		// move points
		command_move = new TCommandMove;
		command_move->Add_Point(sph0 + vec * clear_dist);
		command_move->Add_Point(sph0 + clearance_axis * d_artifact_settings.ballbar_10360_clear_distance);
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));

		switch(meas_cntr)
		{
			case 0:
			case 1:
			case 2:
				length = length_min;
				break;
				
			case 3:
			case 4:
			case 5:
				length = length_min * 2.0;
				break;
				
			case 6:
			case 7:
			case 8:
				length = length_min * 3.0;
				break;
				
			case 9:
			case 10:
			case 11:
				length = length_min * 4.0;
				break;
				
			case 12:
			case 13:
			case 14:
			default:
				length = length_min * 5.0;
				break;
		}
		
		index = static_cast<int>(0.5 + length / d_parameters.ballbar_10360_sphere_spacing);
		length = d_parameters.ballbar_10360_sphere_spacing * static_cast<double>(index);
		sphere_position = sph0 + length * d_refsys.AxisFromWorld(bar_axis);;

		// sphere
		memory_s2 = MEMORY_BALLBAR_10360_SPHERES_MEASURE + feature_index;
		++feature_index;

		command_sphere = new TCommandSphere(memory_s2);
		command_sphere->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_OD);
		
		command_sphere->Set_Output_Format(TCommandFeature::FORMAT_XYZDF);
		
		command_sphere->Add_Path_Move_Point(sphere_position + clearance_axis * d_artifact_settings.ballbar_10360_clear_distance);
		command_sphere->Add_Path_Touch_Point(sphere_position + TVector3(0,0,sphere_radius),TVector3(0,0,-1));
		command_sphere->Add_Path_Move_Point(sphere_position + TVector3(0,0,clear_dist));

		if(level_count[1])
		{
			angle = 0.0;
			angle_increment = PI2 / static_cast<double>(level_count[1]);
			
			for(level_cntr = 0;level_cntr < level_count[1];++level_cntr)
			{
				if(level_cntr == 0 || add_clearance_moves[1])
				{
					vec.Set(cos(angle),sin(angle),1.0);
					command_sphere->Add_Path_Move_Point(sphere_position + vec * clear_dist);
				}

				vec.Set(cos(angle) * 0.707106781186548,sin(angle) * 0.707106781186548, 0.707106781186548);
				
				command_sphere->Add_Path_Move_Point(sphere_position + vec * clear_dist);
				command_sphere->Add_Path_Touch_Point(sphere_position + vec * sphere_radius,-1.0 * vec);
				command_sphere->Add_Path_Move_Point(sphere_position + vec * clear_dist);
						
				if((level_cntr + 1) == level_count[1] || add_clearance_moves[1])
				{
					vec.Set(cos(angle),sin(angle),1.0);
					command_sphere->Add_Path_Move_Point(sphere_position + vec * clear_dist);
				}

				angle += angle_increment;
			}
		}
		
		if(level_count[0])
		{
			angle = 0.0;
			angle_increment = PI2 / static_cast<double>(level_count[0]);
			
			for(level_cntr = 0;level_cntr < level_count[0];++level_cntr)
			{
				if(level_cntr == 0 || add_clearance_moves[0])
				{
					vec.Set(cos(angle),sin(angle),1.0);
					command_sphere->Add_Path_Move_Point(sphere_position + vec * clear_dist);
				}

				vec.Set(cos(angle),sin(angle),0.0);
				
				command_sphere->Add_Path_Move_Point(sphere_position + vec * clear_dist);
				command_sphere->Add_Path_Touch_Point(sphere_position + vec * sphere_radius,-1.0 * vec);
				command_sphere->Add_Path_Move_Point(sphere_position + vec * clear_dist);
				
				if((level_cntr + 1) == level_count[0] || add_clearance_moves[0])
				{
					vec.Set(cos(angle),sin(angle),1.0);
					command_sphere->Add_Path_Move_Point(sphere_position + vec * clear_dist);
				}
				
				angle += angle_increment;
			}
		}
		
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_sphere));
		
		// move points
		command_move = new TCommandMove;
		command_move->Add_Point(sphere_position + vec * clear_dist);
		command_move->Add_Point(sphere_position + clearance_axis * d_artifact_settings.ballbar_10360_clear_distance);
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));

		// dimension
		command_dimension = new TCommandDimension;
		command_dimension->Set_Memory_Id_Feature1(memory_s1);
		command_dimension->Set_Memory_Id_Feature2(memory_s2);
		command_dimension->Add_Dimension(TCommandDimension::FIELD_DIST_XYZ,QStringLiteral("Sphx-Sph0"),0.0);
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_dimension));
		
	}
	
	return true;
}

bool TMeasureRoutines::Measure_Ringgauge(void)
{
	TCommandMove						*command_move;
	TCommandCircle						*command_circle;
	TCommandMode						*command_mode;
	TVector3							vec;
	TVector3							v1;
	QString								text;
	double								ringgauge_nominal_diameter;
	double								ringgauge_radius;
	double								angle;
	double								angle_increment;
	int									point_count;
	int									cntr;
	double								tip_radius;

	emit this->Add_Log_Text(TCommandFeature::Output_Title_XYZDF());
	
	tip_radius = d_active_tool.tip_diameter / 2.0;

	if(d_output_file.isOpen())
	{
		d_output_file.write("Alignment_Begin:\n");
		
		v1 = d_refsys.Rotation_Mat().X();
		text = QString("X_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Y();
		text = QString("Y_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Z();
		text = QString("Z_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Origin();
		text = QString("Translation:%1,%2,%3\n").arg(v1.x,0,'f',5).arg(v1.y,0,'f',5).arg(v1.z,0,'f',5);
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Alignment_End:\n\n");
		
		text = QString("Probe_Offset:%1,%2,%3\n").arg(d_active_tool.xyz.x,0,'f',4).arg(d_active_tool.xyz.y,0,'f',4).arg(d_active_tool.xyz.z,0,'f',4);
		d_output_file.write(text.toLatin1());
		
		text = QString("Probe_Vector:%1,%2,%3\n\n").arg(d_active_tool.ijk.x,0,'f',12).arg(d_active_tool.ijk.y,0,'f',12).arg(d_active_tool.ijk.z,0,'f',12);
		d_output_file.write(text.toLatin1());
	}
	
	
	ringgauge_nominal_diameter = d_parameters.ringgauge_nominal_diameter;
	
	if(ringgauge_nominal_diameter < MIN_RINGGAUGE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Ring gauge diameter too small.  Minimum diameter: %1").arg(MIN_RINGGAUGE_DIAMETER,0,'f',3));
		return false;
	}
	
	point_count = d_parameters.ringgauge_point_count;
	
	if(point_count < 3)
	{
		emit Add_Log_Text(QStringLiteral("ERR:  Ring gauge point count too small."));
		return false;
	}
	
	ringgauge_radius = ringgauge_nominal_diameter / 2.0;
	angle_increment = 6.28318530717959 / static_cast<double>(point_count);
	
	if(d_manual_mode)
	{
		// dcc mode
		command_mode = new TCommandMode();
		command_mode->Set_DCC_Mode();
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	}
	
	// circle
	command_circle = new TCommandCircle(MEMORY_RINGGAUGE_CIRCLE_MEASURE);
	command_circle->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_ID);
	command_circle->Set_Nominal_Projection_Axis(TVector3(0,0,1));
	
	command_circle->Set_Output_Format(TCommandFeature::FORMAT_XYZDF);
	command_circle->Add_Path_Move_Point(TVector3(ringgauge_radius - d_approach_distance - tip_radius,0,d_artifact_settings.ringgauge_clear_z));
	
	angle = 0;
	
	for(cntr = 0;cntr < point_count;++cntr)
	{
		vec.Set(cos(angle),sin(angle),0);
		
		command_circle->Add_Path_Touch_Point(TVector3(vec.x * ringgauge_radius,vec.y * ringgauge_radius,d_artifact_settings.ringgauge_cir_z),vec);
		
		angle += angle_increment;
	}
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_circle));
	
	// move points
	command_move = new TCommandMove;
	command_move->Add_Point(TVector3(vec.x * (ringgauge_radius - d_approach_distance - tip_radius),vec.y * (ringgauge_radius - d_approach_distance - tip_radius), d_artifact_settings.ringgauge_clear_z));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	
	return true;
}

bool TMeasureRoutines::Measure_Pingauge(void)
{
	TCommandMove						*command_move;
	TCommandCircle						*command_circle;
	TCommandMode						*command_mode;
	TVector3							vec;
	TVector3							v1;
	QString								text;
	double								pingauge_nominal_diameter;
	double								pingauge_radius;
	double								angle;
	double								angle_increment;
	int									point_count;
	int									cntr;
	bool								add_clearance_moves(false);;
	double								tip_radius;

	emit Add_Log_Text(TCommandFeature::Output_Title_XYZDF());
	
	tip_radius = d_active_tool.tip_diameter / 2.0;

	if(d_output_file.isOpen())
	{
		d_output_file.write("Alignment_Begin:\n");
		
		v1 = d_refsys.Rotation_Mat().X();
		text = QString("X_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Y();
		text = QString("Y_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Z();
		text = QString("Z_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Origin();
		text = QString("Translation:%1,%2,%3\n").arg(v1.x,0,'f',5).arg(v1.y,0,'f',5).arg(v1.z,0,'f',5);
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Alignment_End:\n\n");
		
		text = QString("Probe_Offset:%1,%2,%3\n").arg(d_active_tool.xyz.x,0,'f',4).arg(d_active_tool.xyz.y,0,'f',4).arg(d_active_tool.xyz.z,0,'f',4);
		d_output_file.write(text.toLatin1());
		
		text = QString("Probe_Vector:%1,%2,%3\n\n").arg(d_active_tool.ijk.x,0,'f',12).arg(d_active_tool.ijk.y,0,'f',12).arg(d_active_tool.ijk.z,0,'f',12);
		d_output_file.write(text.toLatin1());
	}
	
	
	pingauge_nominal_diameter = d_parameters.pingauge_nominal_diameter;
	
	if(pingauge_nominal_diameter < MIN_PINGAUGE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Pin gauge diameter too small.  Minimum diameter: %1").arg(MIN_PINGAUGE_DIAMETER,0,'f',3));
		return false;
	}
	
	point_count = d_parameters.pingauge_point_count;
	
	if(point_count < 3)
	{
		emit Add_Log_Text(QStringLiteral("ERR:  Pin gauge point count too small."));
		return false;
	}
	
	if(point_count < 8)
	{
		add_clearance_moves = true;
	}
	
	pingauge_radius = pingauge_nominal_diameter / 2.0;
	angle_increment = 6.28318530717959 / static_cast<double>(point_count);
	
	if(d_manual_mode)
	{
		// dcc mode
		command_mode = new TCommandMode();
		command_mode->Set_DCC_Mode();
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	}
	
	// circle
	command_circle = new TCommandCircle(MEMORY_PINGAUGE_CIRCLE_MEASURE);
	command_circle->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_OD);
	command_circle->Set_Nominal_Projection_Axis(TVector3(0,0,1));
	
	command_circle->Set_Output_Format(TCommandFeature::FORMAT_XYZDF);
	command_circle->Add_Path_Move_Point(TVector3(pingauge_radius + d_approach_distance + tip_radius,0,d_artifact_settings.pingauge_clear_z));
	
	angle = 0;
	
	for(cntr = 0;cntr < point_count;++cntr)
	{
		vec.Set(cos(angle),sin(angle),0);
		
		if(add_clearance_moves)
		{
			command_circle->Add_Path_Move_Point(TVector3(vec.x * (pingauge_radius + d_approach_distance + tip_radius),vec.y * (pingauge_radius + d_approach_distance + tip_radius),d_artifact_settings.pingauge_clear_z));
		}
		
		command_circle->Add_Path_Touch_Point(TVector3(vec.x * pingauge_radius,vec.y * pingauge_radius,d_artifact_settings.ringgauge_cir_z),-1.0 * vec);
		
		if(add_clearance_moves)
		{
			command_circle->Add_Path_Move_Point(TVector3(vec.x * (pingauge_radius + d_approach_distance + tip_radius),vec.y * (pingauge_radius + d_approach_distance + tip_radius),d_artifact_settings.pingauge_clear_z));
		}
		
		angle += angle_increment;
	}
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_circle));
	
	// move points
	command_move = new TCommandMove;
	command_move->Add_Point(TVector3(vec.x * (pingauge_radius + d_approach_distance + tip_radius),vec.y * (pingauge_radius + d_approach_distance + tip_radius), d_artifact_settings.pingauge_clear_z));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	
	return true;
}

bool TMeasureRoutines::Measure_Roll_Sphere(void)
{
	TCommandMove						*command_move;
	TCommandSphere						*command_sphere;
	TCommandMode						*command_mode;
	TCommandMoveCube					*command_move_cube;
	TVector3							vec;
	TVector3							v1;
	QString								text;
	double								sphere_nominal_diameter;
	double								sphere_radius;
	double								clear_dist;
	double								clear_dist_probe_axis;
	double								angle;
	double								start_angle;
	double								end_angle;
	double								increment;
	double								clearance_angle;
	double								clearance_increment;
	int									point_count;
	int									cntr;
	bool								add_clearance_moves(false);;
	double								tip_radius;
	static const double					PI2(6.28318530717959);
	
	sphere_nominal_diameter = d_parameters.sphere_nominal_diameter;
	
	if(sphere_nominal_diameter < MIN_SPHERE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Sphere diameter too small.  Minimum diameter: %1").arg(MIN_SPHERE_DIAMETER,0,'f',3));
		return false;
	}
	
	point_count = d_artifact_settings.roll_sphere_points;
	
	if(point_count < 5)
	{
		emit Add_Log_Text(QStringLiteral("ERR:  Sphere point count too small."));
		return false;
	}
	
	if(point_count < 9)
	{
		add_clearance_moves = true;
	}
	
	sphere_radius = sphere_nominal_diameter / 2.0;
	tip_radius = d_active_tool.tip_diameter / 2.0;
	
	clear_dist = d_approach_distance;
	clear_dist += tip_radius;
	clear_dist += sphere_radius;
	
	clear_dist_probe_axis = clear_dist + 50.0;
	
	start_angle = 60.0/360.0 * PI2;
	end_angle = 300.0/360.0 * PI2;
	increment = (end_angle - start_angle) / static_cast<double>(point_count - 2);	// includes point on top.
		
	if(d_manual_mode)
	{
		// dcc mode
		command_mode = new TCommandMode();
		command_mode->Set_DCC_Mode();
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	}
	
	// initial clearance moves
	
	// move points
	command_move = new TCommandMove;
	command_move->Add_Point(TVector3(0, 0, clear_dist + 10.0));
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
	
	// sphere
	command_sphere = new TCommandSphere(d_active_roll_sphere);
	command_sphere->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_OD);
	
	command_sphere->Add_Path_Touch_Point(TVector3(0,0,sphere_radius),TVector3(0,0,-1));
	command_sphere->Add_Path_Move_Point(TVector3(0,0,clear_dist));
	
	clearance_increment = increment / 3.0;	// two additional moves if clearance needed
	angle = start_angle;
	for(cntr = 0;cntr < (point_count - 1);++cntr)
	{
		vec.x = cos(angle);
		vec.y = sin(angle);
		
		if(cntr == 0)
		{
			command_sphere->Add_Path_Move_Point(TVector3(vec.x * clear_dist,vec.y * clear_dist,clear_dist));
		}
		else if(add_clearance_moves)
		{
			clearance_angle = angle - increment + clearance_increment;
			
			v1.x = cos(clearance_angle);
			v1.y = sin(clearance_angle);
			
			command_sphere->Add_Path_Move_Point(TVector3(v1.x * clear_dist,v1.y * clear_dist,0.0));
			
			clearance_angle += clearance_increment;
			
			v1.x = cos(clearance_angle);
			v1.y = sin(clearance_angle);
			
			command_sphere->Add_Path_Move_Point(TVector3(v1.x * clear_dist,v1.y * clear_dist,0.0));
		}
		
		command_sphere->Add_Path_Move_Point(TVector3(vec.x * clear_dist,vec.y * clear_dist,0));
		command_sphere->Add_Path_Touch_Point(TVector3(vec.x * sphere_radius,vec.y * sphere_radius,0),TVector3(-vec.x,-vec.y,0));
		command_sphere->Add_Path_Move_Point(TVector3(vec.x * clear_dist,vec.y * clear_dist,0));
				
		angle += increment;
	}
	
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_sphere));

	// move points
	command_move = new TCommandMove;
	command_move->Add_Point(TVector3(vec.x * clear_dist,vec.y * clear_dist,clear_dist));
	
	// final clearance moves
	v1 = TVector3(0,0,clear_dist_probe_axis);
	command_move->Add_Point(v1);
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));

	command_move_cube = new TCommandMoveCube;
	command_move_cube->Set_Axis(TCommandMoveCube::ZPLUS,d_artifact_settings.roll_clear_z);
	command_move_cube->Add_Point(v1);
	d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move_cube));

	switch(d_active_roll_sphere)
	{
		case MEMORY_SPHERE1_ROLL_DCC:
			d_active_roll_sphere = MEMORY_SPHERE2_ROLL_DCC;
			break;
			
		case MEMORY_SPHERE2_ROLL_DCC:
			d_active_roll_sphere = MEMORY_SPHERE3_ROLL_DCC;
			
			// manual mode
			command_mode = new TCommandMode();
			command_mode->Set_Manual_Mode();
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
			break;
			
		case MEMORY_SPHERE3_ROLL_DCC:
			d_active_roll_sphere = MEMORY_SPHERE4_ROLL_DCC;
			break;
			
		case MEMORY_SPHERE4_ROLL_DCC:
			d_active_roll_sphere = MEMORY_SPHERE4_ROLL_DCC + 1;	// not measured but indicates completion.
			
			// manual mode
			command_mode = new TCommandMode();
			command_mode->Set_Manual_Mode();
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
			break;
	}

	return true;
}

bool TMeasureRoutines::Calibrate_Tools(void)
{
	std::vector<TToolWidget::TToolItem>::const_iterator iter;
	TCommandMove						*command_move;
	TCommandMoveCube					*command_move_cube;
	TCommandMode						*command_mode;
	TCommandSphere						*command_sphere;
	TCommandSetTool						*command_settool;
	TCommandAlign						*command_align;
	TCommandAlign::TAlignStep			align_step;
	QString								text;
	TVector3							vec;
	TVector3							v1;
	int									cntr;
	int									memory_cntr;
	double								sphere_diameter;
	double								sphere_radius;
	double								clear_dist;
	double								clear_dist_probe_axis;
	double								start_angle;
	double								end_angle;
	double								increment;
	double								angle;
	double								tip_radius;
	double								clearance_angle;
	double								clearance_increment;
	bool								init;
	bool								add_clearance_moves(false);
	static const double					PI2(6.28318530717959);
	
	if(d_output_file.isOpen())
	{
		d_output_file.write("Alignment_Begin:\n");
		
		v1 = d_refsys.Rotation_Mat().X();
		text = QString("X_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Y();
		text = QString("Y_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Rotation_Mat().Z();
		text = QString("Z_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
		d_output_file.write(text.toLatin1());
		
		v1 = d_refsys.Origin();
		text = QString("Translation:%1,%2,%3\n").arg(v1.x,0,'f',5).arg(v1.y,0,'f',5).arg(v1.z,0,'f',5);
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Alignment_End:\n\n");
	}
	
	sphere_diameter = d_parameters.sphere_nominal_diameter;
	
	if(sphere_diameter < MIN_SPHERE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Sphere diameter too small.  Minimum diameter: %1").arg(MIN_SPHERE_DIAMETER,0,'f',3));
		return false;
	}
	
	sphere_radius = sphere_diameter / 2.0;
	tip_radius = d_active_tool.tip_diameter / 2.0;
	
	clear_dist = d_approach_distance;
	clear_dist += tip_radius;
	clear_dist += sphere_radius;
	
	clear_dist_probe_axis = clear_dist + 50.0;
	
	if(d_artifact_settings.calibration_sphere_points < 9)
	{
		add_clearance_moves = true;
	}

	if(d_manual_mode)
	{
		// dcc mode
		command_mode = new TCommandMode();
		command_mode->Set_DCC_Mode();
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_mode));
	}
	
	init = false;
	for(iter = d_calibration_tool_items.begin(),memory_cntr = 0;iter != d_calibration_tool_items.end();++iter,++memory_cntr)
	{
		// this was completed by the previous step
		if(init)
		{
			// set tool
			command_settool = new TCommandSetTool;
			command_settool->Set_Name((*iter).name);
			command_settool->Set_Offset((*iter).xyz);
			command_settool->Set_Vector((*iter).ijk);
			command_settool->Set_Angle_A((*iter).angle_a);
			command_settool->Set_Angle_B((*iter).angle_b);
			command_settool->Set_Tip_Diameter((*iter).tip_diameter);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_settool));
			
			// initial alignment
			command_align = new TCommandAlign;
			align_step.command_function = TCommandAlign::TYPE_SKEW1_PROBE_AXIS;
			align_step.memory_id = -1;
			align_step.axis = TReferenceSystem::ZMINUS;
			command_align->Add_Align_Step(align_step);
			
			align_step.command_function = TCommandAlign::TYPE_SKEW2_STEM_AXIS;
			align_step.memory_id = -1;
			align_step.axis = TReferenceSystem::XMINUS;
			align_step.projection_axis = TReferenceSystem::ZPLUS;
			command_align->Add_Align_Step(align_step);
			
			align_step.command_function = TCommandAlign::TYPE_ORIGIN;
			align_step.memory_id = MEMORY_SPHERE_DCC;
			align_step.axis = TReferenceSystem::XPLUS;
			command_align->Add_Align_Step(align_step);
			
			align_step.command_function = TCommandAlign::TYPE_ORIGIN;
			align_step.memory_id = MEMORY_SPHERE_DCC;
			align_step.axis = TReferenceSystem::YPLUS;
			command_align->Add_Align_Step(align_step);
			
			align_step.command_function = TCommandAlign::TYPE_ORIGIN;
			align_step.memory_id = MEMORY_SPHERE_DCC;
			align_step.axis = TReferenceSystem::ZPLUS;
			command_align->Add_Align_Step(align_step);
			
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_align));
			
			// initial clearance moves
			v1 = TVector3(0,0,clear_dist_probe_axis);
			
			command_move_cube = new TCommandMoveCube;
			command_move_cube->Set_Axis(TCommandMoveCube::ZPLUS,d_artifact_settings.calibration_sphere_clear_z);
			command_move_cube->Add_Point(v1);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move_cube));
			
			command_move = new TCommandMove;
			command_move->Add_Point(v1);
			d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
			
		}
		
		init = true;
		
		// move points
		command_move = new TCommandMove;
		command_move->Add_Point(TVector3(0, 0, clear_dist + 10.0));	// extra 10 mm provides some protection if the probe configuration is wrong (maybe)
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		
		// sphere
		command_sphere = new TCommandSphere(MEMORY_SPHERE_CALIBRATE_TOOL + memory_cntr);
		command_sphere->Set_Nominal_Diameter_Type(TCommandFeature::DIAMETER_TYPE_OD);
		
		command_sphere->Add_Path_Touch_Point(TVector3(0,0,sphere_radius),TVector3(0,0,-1));
		command_sphere->Add_Path_Move_Point(TVector3(0,0,clear_dist));

		if(((*iter).ijk ^ d_stem_vector_sphere) < -0.866)
		{
			start_angle = 0;
			end_angle = PI2;
			increment = (end_angle - start_angle) / static_cast<double>(d_artifact_settings.calibration_sphere_points - 1);	// includes point on top.
		}
		else
		{
			start_angle = 60.0/360.0 * PI2;
			end_angle = 300.0/360.0 * PI2;
			increment = (end_angle - start_angle) / static_cast<double>(d_artifact_settings.calibration_sphere_points - 2);	// includes point on top.
		}
		
		clearance_increment = increment / 3.0;	// two additional moves if clearance needed
		angle = start_angle;
		for(cntr = 0;cntr < (d_artifact_settings.calibration_sphere_points - 1);++cntr)
		{
			vec.x = cos(angle);
			vec.y = sin(angle);
						
			if(cntr == 0)
			{
				command_sphere->Add_Path_Move_Point(TVector3(vec.x * clear_dist,vec.y * clear_dist,clear_dist));
			}
			else if(add_clearance_moves)
			{
				clearance_angle = angle - increment + clearance_increment;
				
				v1.x = cos(clearance_angle);
				v1.y = sin(clearance_angle);
				
				command_sphere->Add_Path_Move_Point(TVector3(v1.x * clear_dist,v1.y * clear_dist,0.0));
				
				clearance_angle += clearance_increment;
				
				v1.x = cos(clearance_angle);
				v1.y = sin(clearance_angle);
				
				command_sphere->Add_Path_Move_Point(TVector3(v1.x * clear_dist,v1.y * clear_dist,0.0));
			}
			
			command_sphere->Add_Path_Move_Point(TVector3(vec.x * clear_dist,vec.y * clear_dist,0));
			command_sphere->Add_Path_Touch_Point(TVector3(vec.x * sphere_radius,vec.y * sphere_radius,0),TVector3(-vec.x,-vec.y,0));
			command_sphere->Add_Path_Move_Point(TVector3(vec.x * clear_dist,vec.y * clear_dist,0));
			
			angle += increment;
		}
		
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_sphere));
		
		// move points
		command_move = new TCommandMove;
		command_move->Add_Point(TVector3(vec.x * clear_dist,vec.y * clear_dist,clear_dist));
		
		// final clearance moves
		v1 = TVector3(0,0,clear_dist_probe_axis);
		command_move->Add_Point(v1);
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move));
		
		command_move_cube = new TCommandMoveCube;
		command_move_cube->Set_Axis(TCommandMoveCube::ZPLUS,d_artifact_settings.calibration_sphere_clear_z);
		command_move_cube->Add_Point(v1);
		d_command_sequence.Add_Command(reinterpret_cast<TCommand*>(command_move_cube));
		
	}
	
	return true;
}

bool TMeasureRoutines::Report_Stepgauge_10360(void)
{
	TMeasureRoutines::TFeatureMemory	feature_memory;
	TVector3							pnt;
	QString								text;
	int									memory_id(MEMORY_STEPGAUGE_POINT_START_MEASURE);
	int									cntr;
	int									repeat_cntr;
	bool								valid(false);
	static const int					STEP_COUNT(6);
	
	if(!d_output_file.isOpen())
	{
		emit Add_Log_Text(QString("ERR:  Output file not open."));
		return false;
	}
	
	for(cntr = 0;cntr < STEP_COUNT;++cntr)
	{
		for(repeat_cntr = 0;repeat_cntr < 3;++repeat_cntr)
		{
			feature_memory = this->Feature_Memory(memory_id,&valid);
			
			if(!valid)
			{
				emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(memory_id));
			}
			else
			{
				pnt = d_refsys.FromWorld(feature_memory.xyz);
				
				text = QString("Point:%1,A,%2,%3,%4\n").arg(cntr).arg(pnt.x,0,'f',5).arg(pnt.y,0,'f',5).arg(pnt.z,0,'f',5);
				d_output_file.write(text.toLatin1());
			}

			++memory_id;
			
			feature_memory = this->Feature_Memory(memory_id,&valid);
			
			if(!valid)
			{
				emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(memory_id));
			}
			else
			{
				pnt = d_refsys.FromWorld(feature_memory.xyz);
				
				text = QString("Point:%1,B,%2,%3,%4\n").arg(cntr).arg(pnt.x,0,'f',5).arg(pnt.y,0,'f',5).arg(pnt.z,0,'f',5);
				d_output_file.write(text.toLatin1());
			}
			
			++memory_id;
		}
		
		d_output_file.write("\n");
	}
	
	d_output_file.write("Measurement_End:\n");
	
	return true;
}

bool TMeasureRoutines::Report_Stepgauge_B89(void)
{
	TMeasureRoutines::TFeatureMemory	feature_memory;
	TVector3							pnt;
	QString								text;
	double								measure_length(0.0);
	int									target_count;
	int									target_cntr;
	int									measurement_count;
	int									measurement_cntr;
	int									memory_id(MEMORY_STEPGAUGE_POINT_START_MEASURE);
	bool								valid(false);
	
	if(!d_output_file.isOpen())
	{
		emit Add_Log_Text(QString("ERR:  Output file not open."));
		return false;
	}
	
	feature_memory = this->Feature_Memory(MEMORY_STEPGAUGE_POINT_END_DCC,&valid);
	
	if(valid)
	{
		pnt = d_refsys.FromWorld(feature_memory.xyz);
		pnt.y = 0;
		pnt.z = 0;
		measure_length = pnt.Length();
	}

	target_count = 1 + static_cast<int>((measure_length + 2.0 - d_parameters.stepgauge_measurement_b89_start) / d_parameters.stepgauge_b89_nominal_block_spacing);
	
	if(target_count < 1)
	{
		emit Add_Log_Text(QString("ERR:  Stepgauge too short for required measurements.  Length: %1").arg(measure_length,0,'f',3));
		return false;
	}
	
	measurement_count = d_parameters.stepgauge_b89_measurement_count;
	
	if(measurement_count < 1)
	{
		emit Add_Log_Text(QStringLiteral("ERR:  Stepgauge B89 measurement count must be greater than zero."));
		return false;
	}
	
	for(measurement_cntr = 0;measurement_cntr < measurement_count;++measurement_cntr)
	{
		for(target_cntr = 0;target_cntr < target_count;++target_cntr)
		{
			feature_memory = this->Feature_Memory(memory_id,&valid);
			
			if(!valid)
			{
				emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(memory_id));
			}
			else
			{
				pnt = d_refsys.FromWorld(feature_memory.xyz);
				
				text = QString("Point:%1,%2,%3,%4\n").arg(measurement_cntr).arg(pnt.x,0,'f',5).arg(pnt.y,0,'f',5).arg(pnt.z,0,'f',5);
				d_output_file.write(text.toLatin1());
			}
			
			++memory_id;
		}
		
		d_output_file.write("\n");
	}
	
	d_output_file.write("Measurement_End:\n");
	
	return true;
}

bool TMeasureRoutines::Report_Stepgauge_Square(void)
{
	TMeasureRoutines::TFeatureMemory	feature_memory;
	TVector3							pnt;
	QString								text;
	int									memory_id(MEMORY_STEPGAUGE_POINT_START_MEASURE);
	int									cntr;
	int									repeat_cntr;
	int									repeat_count;
	bool								valid(false);

	if(!d_output_file.isOpen())
	{
		emit Add_Log_Text(QString("ERR:  Output file not open."));
		return false;
	}
	
	repeat_count = d_parameters.stepgauge_square_measurement_count;
	
	for(repeat_cntr = 0;repeat_cntr < repeat_count;++repeat_cntr)
	{
		for(cntr = 0;cntr < 7;++cntr)
		{
			feature_memory = this->Feature_Memory(memory_id,&valid);
			
			if(!valid)
			{
				emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(memory_id));
			}
			else
			{
				pnt = d_refsys.FromWorld(feature_memory.xyz);
				
				text = QString("Point:%1,%2,%3,%4,%5\n").arg(repeat_cntr).arg(cntr).arg(pnt.x,0,'f',5).arg(pnt.y,0,'f',5).arg(pnt.z,0,'f',5);
				d_output_file.write(text.toLatin1());
			}
			
			++memory_id;
		}
		
		d_output_file.write("\n");
	}
	
	d_output_file.write("Measurement_End:\n");
	
	return true;
}

bool TMeasureRoutines::Report_Gaugeblock(void)
{
	TMeasureRoutines::TFeatureMemory	feature_memory;
	TVector3							pnt;
	QString								text;
	int									memory_id(MEMORY_GAUGEBLOCK_POINT_MEASURE);
	int									repeat_cntr;
	bool								valid(false);
	
	if(!d_output_file.isOpen())
	{
		emit Add_Log_Text(QString("ERR:  Output file not open."));
		return false;
	}
	
	feature_memory = this->Feature_Memory(MEMORY_GAUGEBLOCK_PLANE_DCC,&valid);
	
	if(!valid)
	{
		emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(memory_id));
	}
	else
	{
		text = QString("Datum_Flatness: %1 mm\n\n").arg(feature_memory.form,0,'f',5);
		d_output_file.write(text.toLatin1());
	}
	
	for(repeat_cntr = 0;repeat_cntr < 3;++repeat_cntr)
	{
		feature_memory = this->Feature_Memory(memory_id,&valid);
		
		if(!valid)
		{
			emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(memory_id));
		}
		else
		{
			pnt = d_refsys.FromWorld(feature_memory.xyz);
			
			text = QString("Point:1,A,%1,%2,%3\n").arg(pnt.x,0,'f',5).arg(pnt.y,0,'f',5).arg(pnt.z,0,'f',5);
			d_output_file.write(text.toLatin1());
		}
		
		++memory_id;
		
		feature_memory = this->Feature_Memory(memory_id,&valid);
		
		if(!valid)
		{
			emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(memory_id));
		}
		else
		{
			pnt = d_refsys.FromWorld(feature_memory.xyz);
			
			text = QString("Point:1,B,%1,%2,%3\n").arg(pnt.x,0,'f',5).arg(pnt.y,0,'f',5).arg(pnt.z,0,'f',5);
			d_output_file.write(text.toLatin1());
		}
		
		++memory_id;
	}
	
	d_output_file.write("\nMeasurement_End:\n");
	
	return true;
}

bool TMeasureRoutines::Report_Point_Repeat(void)
{
	TMeasureRoutines::TFeatureMemory	feature_memory;
	TVector3							pnt;
	TVector3							initial_position;
	QString								text;
	int									memory_id(MEMORY_POINT_REPEAT_MEASURE);
	int									repeat_cntr;
	int									repeat_count;
	bool								init_initial_position(true);
	bool								valid(false);
	
	if(!d_output_file.isOpen())
	{
		emit Add_Log_Text(QString("ERR:  Output file not open."));
		return false;
	}
	
	repeat_count = d_parameters.point_repeat_measurement_count;
	
	for(repeat_cntr = 0;repeat_cntr < repeat_count;++repeat_cntr)
	{
		feature_memory = this->Feature_Memory(memory_id,&valid);
		
		if(!valid)
		{
			emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(memory_id));
		}
		else
		{
			pnt = d_refsys.FromWorld(feature_memory.xyz);
			
			if(init_initial_position)
			{
				initial_position = pnt;
				init_initial_position = false;
			}
			
			pnt -= initial_position;
			
			text = QString("Point:%1,%2,%3,%4\n").arg(repeat_cntr+1).arg(pnt.x,0,'f',5).arg(pnt.y,0,'f',5).arg(pnt.z,0,'f',5);
			d_output_file.write(text.toLatin1());
		}

		++memory_id;
	}
	
	d_output_file.write("\nMeasurement_End:\n");
	
	return true;
}

bool TMeasureRoutines::Report_Sphere_Rpt(void)
{
	TMeasureRoutines::TFeatureMemory	feature_memory;
	TVector3							pnt;
	TVector3							initial_position;
	QString								text;
	int									memory_id(MEMORY_SPHERE_REPEAT_MEASURE);
	int									repeat_cntr;
	int									repeat_count;
	bool								init_initial_position(true);
	bool								valid(false);
	
	if(!d_output_file.isOpen())
	{
		emit Add_Log_Text(QString("ERR:  Output file not open."));
		return false;
	}
	
	repeat_count = d_parameters.sphere_repeat_measurement_count;
	
	for(repeat_cntr = 0;repeat_cntr < repeat_count;++repeat_cntr)
	{
		feature_memory = this->Feature_Memory(memory_id,&valid);
		
		if(!valid)
		{
			emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(memory_id));
		}
		else
		{
			pnt = d_refsys.FromWorld(feature_memory.xyz);
			
			if(init_initial_position)
			{
				initial_position = pnt;
				init_initial_position = false;
				
				d_output_file.write("Measurement_Begin:\n\n");
			}
			
			pnt -= initial_position;
			
			text = QString("Point:%1,%2,%3,%4\n").arg(repeat_cntr+1).arg(pnt.x,0,'f',5).arg(pnt.y,0,'f',5).arg(pnt.z,0,'f',5);
			d_output_file.write(text.toLatin1());
		}

		++memory_id;
	}
	
	d_output_file.write("\nMeasurement_End:\n");
	
	return true;
}

bool TMeasureRoutines::Report_Sphere_Pftu(void)
{
	TCommand							*command;
	TCommandSphere						*command_sphere;
	std::vector<TVector3>				points;
	std::vector<TVector3>::const_iterator iter;
	TVector3							pnt;
	TVector3							vec;
	TVector3							center;
	TVector3							initial_position;
	QString								text;
	double								diameter;
	double								form;
	double								radius;
	double								nominal_diameter;
	double								nominal_radius;
	double								pnt_deviation;
	double								tip_radius;

	if(!d_output_file.isOpen())
	{
		emit Add_Log_Text(QString("ERR:  Output file not open."));
		return false;
	}
	
	nominal_diameter = d_parameters.sphere_nominal_diameter;
	
	if(nominal_diameter < MIN_SPHERE_DIAMETER)
	{
		emit Add_Log_Text(QString("ERR:  Sphere diameter too small.  Minimum diameter: %1").arg(MIN_SPHERE_DIAMETER,0,'f',3));
		return false;
	}
	
	nominal_radius = nominal_diameter / 2.0;
	tip_radius = d_active_tool.tip_diameter / 2.0;
	
	text = QString("Nominal_Sphere_Radius: %1\n").arg(nominal_radius,0,'f',5);
	d_output_file.write(text.toLatin1());
	
	
	command = d_command_sequence.Get_Feature_Command(MEMORY_SPHERE_PFTU_MEASURE);
	
	if(command && command->Type() == TCommand::COMMAND_MEASURE_SPHERE)
	{
		command_sphere = reinterpret_cast<TCommandSphere*>(command);
	}
	else
	{
		emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(MEMORY_SPHERE_PFTU_MEASURE));
		return false;
	}
	
	if(command_sphere)
	{
		center = d_refsys.FromWorld(command_sphere->Actual_XYZ());
		diameter = command_sphere->Actual_Diameter();
		form = command_sphere->Form();
		
		text = QString("Sphere  Memory[%1]  XYZ: %2,%3,%4  Diam: %5  Form: %6\n\n")
					.arg(MEMORY_SPHERE_PFTU_MEASURE)
					.arg(center.x,0,'f',5)
					.arg(center.y,0,'f',5)
					.arg(center.z,0,'f',5)
					.arg(diameter,0,'f',5)
					.arg(form,0,'f',5);
		
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Point_Data_Begin:\n\n");
		
		d_output_file.write("# Format:  X, Y, Z, I, J, K, Radius,Deviation\n\n");
		
		points = command_sphere->Touch_Points();
		
		emit Add_Log_Text(QString("INF:  Memory[%1] Point Count: %2").arg(MEMORY_SPHERE_PFTU_MEASURE).arg(points.size()));
		
		for(iter = points.begin();iter != points.end();++iter)
		{
			pnt = d_refsys.FromWorld(*iter);
			pnt -= center;
			
			vec = pnt;
			
			if(vec.Length() > ZERO_EPSILON)
			{
				vec.Normal();
				
				pnt -= (tip_radius * vec);
				radius = pnt.Length();
				
				pnt_deviation = radius - nominal_radius;
				
				
				text = QString(" %1, %2, %3, %4, %5, %6, %7, %8\n")
							.arg(pnt.x,12,'f',5)
							.arg(pnt.y,12,'f',5)
							.arg(pnt.z,12,'f',5)
							.arg(vec.i,12,'f',9)
							.arg(vec.j,12,'f',9)
							.arg(vec.k,12,'f',9)
							.arg(radius,12,'f',5)
							.arg(pnt_deviation,12,'f',5);
				
				d_output_file.write(text.toLatin1());
				
			}
		}
		
		d_output_file.write("\nPoint_Data_End:\n");
	}
	
	return true;
}

bool TMeasureRoutines::Report_Ballbar(void)
{
	TMeasureRoutines::TFeatureMemory	sphere1_memory;
	TMeasureRoutines::TFeatureMemory	sphere2_memory;
	TVector3							sph1;
	TVector3							sph2;
	TVector3							vec;
	double								length;
	double								sphere1_form;
	double								sphere2_form;
	QString								text;
	int									meas_count;
	int									cntr;
	int									actual_meas_count;
	bool								valid1(false);
	bool								valid2(false);

	if(!d_output_file.isOpen())
	{
		emit Add_Log_Text(QString("ERR:  Output file not open."));
		return false;
	}
	
	meas_count = d_parameters.ballbar_measurement_count;
	actual_meas_count = 0;
	
	for(cntr = 0;cntr < meas_count;++cntr)
	{
		if(cntr == 0)
		{
			sph1.Set(0,0,0);
			sph2.Set(0,0,0);
			length = 0.0;
			sphere1_form = 0.0;
			sphere2_form = 0.0;
		}
				
		sphere1_memory = this->Feature_Memory(MEMORY_BALLBAR_SPHERES_MEASURE + cntr * 2,&valid1);
		sphere2_memory = this->Feature_Memory(MEMORY_BALLBAR_SPHERES_MEASURE + 1 + cntr * 2,&valid2);

		if(!valid1)
		{
			emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(MEMORY_BALLBAR_SPHERES_MEASURE + cntr * 2));
		}
		
		if(!valid2)
		{
			emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(MEMORY_BALLBAR_SPHERES_MEASURE + 1 + cntr * 2));
		}
		
		if(valid1 && valid2)
		{
			sph1 += sphere1_memory.xyz;
			sph2 += sphere2_memory.xyz;
			
			vec = sphere2_memory.xyz - sphere1_memory.xyz;
			length += vec.Length();
			
			sphere1_form += sphere1_memory.form;
			sphere2_form += sphere2_memory.form;
			
			++actual_meas_count;
		}
	}
	
	if(actual_meas_count)
	{
		text = d_parameters.ballbar_position_name;
		text.append(',');
		
		sph1 /= static_cast<double>(actual_meas_count);
		sph2 /= static_cast<double>(actual_meas_count);
		length /= static_cast<double>(actual_meas_count);
		sphere1_form /= static_cast<double>(actual_meas_count);
		sphere2_form /= static_cast<double>(actual_meas_count);
		
		text = QString("%1 %2,%3,%4,%5,%6,%7,%8,%9,%10,%11,%12,%13\n")
					.arg(text,-20)
					.arg(length,11,'f',4)
					.arg(sph1.x,11,'f',4)
					.arg(sph1.y,11,'f',4)
					.arg(sph1.z,11,'f',4)
					.arg(sph2.x,11,'f',4)
					.arg(sph2.y,11,'f',4)
					.arg(sph2.z,11,'f',4)
					.arg(sphere1_form,11,'f',4)
					.arg(sphere2_form,11,'f',4)
					.arg(d_active_tool.xyz.x,11,'f',4)
					.arg(d_active_tool.xyz.y,11,'f',4)
					.arg(d_active_tool.xyz.z,11,'f',4);
		
		d_output_file.write(text.toLatin1());
	}
	
	return true;
}

bool TMeasureRoutines::Report_Ballbar_10360(void)
{
	TMeasureRoutines::TFeatureMemory	feature_memory;
	TReferenceSystem					ballbar_refsys;
	TVector3							sph;
	TVector3							sph0;
	TVector3							sph5;
	TVector3							vec;
	TVector3							v1;
	QString								text;
	int									cntr;
	int									index_cntr;
	int									length_cntr;
	bool								valid(false);

	if(!d_output_file.isOpen())
	{
		emit Add_Log_Text(QString("ERR:  Output file not open."));
		return false;
	}
	
	feature_memory = this->Feature_Memory(MEMORY_BALLBAR_10360_SPHERE0_DCC,&valid);
	
	if(valid)
	{
		sph0 = feature_memory.xyz;
	}
	else
	{
		emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(MEMORY_BALLBAR_10360_SPHERE0_DCC));
		return false;
	}
	
	feature_memory = this->Feature_Memory(MEMORY_BALLBAR_10360_SPHERE5_DCC,&valid);
	
	if(valid)
	{
		sph5 = feature_memory.xyz;
	}
	else
	{
		emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(MEMORY_BALLBAR_10360_SPHERE5_DCC));
		return false;
	}

	vec = sph5 - sph0;
	vec.Normal();

	ballbar_refsys.Skew1(vec,TReferenceSystem::XPLUS);
	ballbar_refsys.Skew2(d_active_tool.ijk,TReferenceSystem::ZMINUS,TReferenceSystem::XPLUS);
	ballbar_refsys.Set_Origin(sph0);
	
	d_output_file.write("Alignment_Begin:\n");
	
	v1 = ballbar_refsys.Rotation_Mat().X();
	text = QString("X_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
	d_output_file.write(text.toLatin1());
	
	v1 = ballbar_refsys.Rotation_Mat().Y();
	text = QString("Y_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
	d_output_file.write(text.toLatin1());
	
	v1 = ballbar_refsys.Rotation_Mat().Z();
	text = QString("Z_Axis:%1,%2,%3\n").arg(v1.x,0,'f',12).arg(v1.y,0,'f',12).arg(v1.z,0,'f',12);
	d_output_file.write(text.toLatin1());
	
	v1 = ballbar_refsys.Origin();
	text = QString("Translation:%1,%2,%3\n").arg(v1.x,0,'f',5).arg(v1.y,0,'f',5).arg(v1.z,0,'f',5);
	d_output_file.write(text.toLatin1());
	
	d_output_file.write("Alignment_End:\n\n");
	
	d_output_file.write("Temperature_Begin:\n");
	
	text = QString("X_Scale:%1\n").arg(d_parameters.axis_x_temperature,0,'f',3);
	d_output_file.write(text.toLatin1());
	
	text = QString("Y_Scale:%1\n").arg(d_parameters.axis_y_temperature,0,'f',3);
	d_output_file.write(text.toLatin1());
	
	text = QString("Z_Scale:%1\n").arg(d_parameters.axis_z_temperature,0,'f',3);
	d_output_file.write(text.toLatin1());
	
	text = QString("Gauge:%1\n").arg(d_parameters.part_temperature,0,'f',3);
	d_output_file.write(text.toLatin1());
	
	d_output_file.write("Temperature_End:\n\n");
	
	text = QString("Probe_Offset:%1,%2,%3\n").arg(d_active_tool.xyz.x,0,'f',4).arg(d_active_tool.xyz.y,0,'f',4).arg(d_active_tool.xyz.z,0,'f',4);
	d_output_file.write(text.toLatin1());
	
	text = QString("Probe_Vector:%1,%2,%3\n\n").arg(d_active_tool.ijk.x,0,'f',12).arg(d_active_tool.ijk.y,0,'f',12).arg(d_active_tool.ijk.z,0,'f',12);
	d_output_file.write(text.toLatin1());
	
	d_output_file.write("Scales_Begin:\n");
	
	text = QString("X_Expansion_Coefficient:%1\n").arg(d_expansion_coeff_x,0,'f',1);
	d_output_file.write(text.toLatin1());
	
	text = QString("Y_Expansion_Coefficient:%1\n").arg(d_expansion_coeff_y,0,'f',1);
	d_output_file.write(text.toLatin1());
	
	text = QString("Z_Expansion_Coefficient:%1\n").arg(d_expansion_coeff_z,0,'f',1);
	d_output_file.write(text.toLatin1());
	
	d_output_file.write("Scales_End:\n\n");
	
	text = QString("Equipment_Identification:%1\n").arg(d_parameters.ballbar_10360_serial_number);
	d_output_file.write(text.toLatin1());
	
	d_output_file.write("Measurement_Begin:\n\n");

	text = QString("Description:%1\n").arg(d_parameters.ballbar_10360_position_name);
	d_output_file.write(text.toLatin1());
	
	text = QString("Sphere_Zero_Offset:%1\n\n").arg(d_parameters.ballbar_10360_sphere0_offset);
	d_output_file.write(text.toLatin1());

	length_cntr = 0;
	index_cntr = 0;
	
	for(cntr = 0;cntr < 30;++cntr)
	{
		feature_memory = this->Feature_Memory(MEMORY_BALLBAR_10360_SPHERES_MEASURE + cntr,&valid);
		
		if(!valid)
		{
			emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(MEMORY_BALLBAR_10360_SPHERES_MEASURE + cntr));
		}
		else
		{
			sph = ballbar_refsys.FromWorld(feature_memory.xyz);

			if(cntr % 2)
			{
				text = QString("Sphere:B,%1,%2,%3,%4,%5,%6,%7\n")
						.arg(length_cntr)
						.arg(index_cntr)
						.arg(sph.x,0,'f',5)
						.arg(sph.y,0,'f',5)
						.arg(sph.z,0,'f',5)
						.arg(feature_memory.diameter,0,'f',5)
						.arg(feature_memory.form,0,'f',5);
								
				++index_cntr;
				
				if(index_cntr > 2)
				{
					index_cntr = 0;
					++length_cntr;
				}
			}
			else
			{
				text = QString("Sphere:A,%1,%2,%3,%4,%5,%6,%7\n")
						.arg(length_cntr)
						.arg(index_cntr)
						.arg(sph.x,0,'f',5)
						.arg(sph.y,0,'f',5)
						.arg(sph.z,0,'f',5)
						.arg(feature_memory.diameter,0,'f',5)
						.arg(feature_memory.form,0,'f',5);
			}
			
			d_output_file.write(text.toLatin1());
		}
	}
	
	d_output_file.write("Measurement_End:\n");

	return true;
}

bool TMeasureRoutines::Report_Ringgauge(void)
{
	std::vector<TVector3>				points;
	std::vector<TVector3>::const_iterator iter;
	TCommand							*command;
	TCommandCircle						*command_circle;
	TVector3							center;
	TVector3							pnt;
	TVector3							vec;
	double								diameter;
	double								form;
	double								nominal_radius;
	double								pnt_deviation;
	QString								text;
	double								tip_radius;

	if(!d_output_file.isOpen())
	{
		emit Add_Log_Text(QString("ERR:  Output file not open."));
		return false;
	}

	tip_radius = d_active_tool.tip_diameter / 2.0;

	command = d_command_sequence.Get_Feature_Command(MEMORY_RINGGAUGE_CIRCLE_MEASURE);
	
	if(command && command->Type() == TCommand::COMMAND_MEASURE_CIRCLE)
	{
		command_circle = reinterpret_cast<TCommandCircle*>(command);
	}
	else
	{
		emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(MEMORY_RINGGAUGE_CIRCLE_MEASURE));
		command_circle = 0;
	}
	
	if(command_circle)
	{
		center = d_refsys.FromWorld(command_circle->Actual_XYZ());
		diameter = command_circle->Actual_Diameter();
		form = command_circle->Form();
		
		
		text = QString("Circle  Memory[%1]  XYZ: %2,%3,%4  Diam: %5  Form: %6\n\n")
						.arg(MEMORY_RINGGAUGE_CIRCLE_MEASURE)
						.arg(center.x,0,'f',5)
						.arg(center.y,0,'f',5)
						.arg(center.z,0,'f',5)
						.arg(diameter,0,'f',5)
						.arg(form,0,'f',5);
		
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Point_Data_Begin:\n\n");
		
		d_output_file.write("# Format:  X, Y, Z, I, J, K, Deviation (relative to circle center)\n\n");
		
		points = command_circle->Touch_Points();
		nominal_radius = diameter/2.0;
		
		emit Add_Log_Text(QString("INF:  Memory[%1] Point Count: %2").arg(MEMORY_RINGGAUGE_CIRCLE_MEASURE).arg(points.size()));
		
		for(iter = points.begin();iter != points.end();++iter)
		{
			pnt = d_refsys.FromWorld(*iter);
			pnt -= center;
			
			vec = pnt;
			
			if(vec.Length() > ZERO_EPSILON)
			{
				vec.Normal();
				
				pnt += (tip_radius * vec);
				
				pnt_deviation = pnt.Length() - nominal_radius;
				
				
				text = QString(" %1, %2, %3, %4, %5, %6, %7\n")
						.arg(pnt.x,12,'f',5)
						.arg(pnt.y,12,'f',5)
						.arg(pnt.z,12,'f',5)
						.arg(vec.i,12,'f',9)
						.arg(vec.j,12,'f',9)
						.arg(vec.k,12,'f',9)
						.arg(pnt_deviation,12,'f',5);
				
				d_output_file.write(text.toLatin1());
				
			}
		}
		
		d_output_file.write("\nPoint_Data_End:\n");
	}
	
	return true;
}

bool TMeasureRoutines::Report_Pingauge(void)
{
	std::vector<TVector3>				points;
	std::vector<TVector3>::const_iterator iter;
	TCommand							*command;
	TCommandCircle						*command_circle;
	TVector3							center;
	TVector3							pnt;
	TVector3							vec;
	double								diameter;
	double								form;
	double								nominal_radius;
	double								pnt_deviation;
	QString								text;
	double								tip_radius;

	if(!d_output_file.isOpen())
	{
		emit Add_Log_Text(QString("ERR:  Output file not open."));
		return false;
	}

	tip_radius = d_active_tool.tip_diameter / 2.0;

	command = d_command_sequence.Get_Feature_Command(MEMORY_PINGAUGE_CIRCLE_MEASURE);
	
	if(command && command->Type() == TCommand::COMMAND_MEASURE_CIRCLE)
	{
		command_circle = reinterpret_cast<TCommandCircle*>(command);
	}
	else
	{
		emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(MEMORY_PINGAUGE_CIRCLE_MEASURE));
		command_circle = 0;
	}
	
	if(command_circle)
	{
		center = d_refsys.FromWorld(command_circle->Actual_XYZ());
		diameter = command_circle->Actual_Diameter();
		form = command_circle->Form();
		
		
		text = QString("Circle  Memory[%1]  XYZ: %2,%3,%4  Diam: %5  Form: %6\n\n")
				.arg(MEMORY_PINGAUGE_CIRCLE_MEASURE)
				.arg(center.x,0,'f',5)
				.arg(center.y,0,'f',5)
				.arg(center.z,0,'f',5)
				.arg(diameter,0,'f',5)
				.arg(form,0,'f',5);
		
		d_output_file.write(text.toLatin1());
		
		d_output_file.write("Point_Data_Begin:\n\n");
		
		d_output_file.write("# Format:  X, Y, Z, I, J, K, Deviation (relative to circle center)\n\n");
		
		points = command_circle->Touch_Points();
		nominal_radius = diameter/2.0;
		
		emit Add_Log_Text(QString("INF:  Memory[%1] Point Count: %2").arg(MEMORY_PINGAUGE_CIRCLE_MEASURE).arg(points.size()));
		
		for(iter = points.begin();iter != points.end();++iter)
		{
			pnt = d_refsys.FromWorld(*iter);
			pnt -= center;
			
			vec = pnt;
			
			if(vec.Length() > ZERO_EPSILON)
			{
				vec.Normal();
				
				pnt -= (tip_radius * vec);
				
				pnt_deviation = pnt.Length() - nominal_radius;
				
				
				text = QString(" %1, %2, %3, %4, %5, %6, %7\n")
						.arg(pnt.x,12,'f',5)
						.arg(pnt.y,12,'f',5)
						.arg(pnt.z,12,'f',5)
						.arg(vec.i,12,'f',9)
						.arg(vec.j,12,'f',9)
						.arg(vec.k,12,'f',9)
						.arg(pnt_deviation,12,'f',5);
				
				d_output_file.write(text.toLatin1());
				
			}
		}
		
		d_output_file.write("\nPoint_Data_End:\n");
	}
	
	return true;
}

bool TMeasureRoutines::Report_Roll(void)
{
	TMeasureRoutines::TFeatureMemory	feature_memory;
	std::vector<TVector3>				points;
	std::vector<TVector3>::const_iterator iter;
	TVector3							center;
	TVector3							pnt;
	TVector3							vec;
	double								diameter;
	double								form;
	QString								text;
	TToolWidget::TToolItem				tool_item1;
	TToolWidget::TToolItem				tool_item2;
	bool								valid(false);

	if(!d_output_file.isOpen())
	{
		emit Add_Log_Text(QString("ERR:  Output file not open."));
		return false;
	}
	
	tool_item1 = this->Get_Tool_Item(d_parameters.roll_tool1_name,&valid);

	if(!valid)
	{
		emit Add_Log_Text(QString("ERR:  Tool data for '%1' not available.").arg(d_parameters.roll_tool1_name));
		return false;
	}
	
	tool_item2 = this->Get_Tool_Item(d_parameters.roll_tool2_name,&valid);

	if(!valid)
	{
		emit Add_Log_Text(QString("ERR:  Tool data for '%1' not available.").arg(d_parameters.roll_tool2_name));
		return false;
	}
	
	text = QString("Tool_Offset_1:%1,%2,%3\n").arg(tool_item1.xyz.x,0,'f',4).arg(tool_item1.xyz.y,0,'f',4).arg(tool_item1.xyz.z,0,'f',4);
	d_output_file.write(text.toLatin1());
	
	text = QString("Tool_Vector_1:%1,%2,%3\n").arg(tool_item1.ijk.x,0,'f',12).arg(tool_item1.ijk.y,0,'f',12).arg(tool_item1.ijk.z,0,'f',12);
	d_output_file.write(text.toLatin1());
	
	text = QString("Tool_Offset_2:%1,%2,%3\n").arg(tool_item2.xyz.x,0,'f',4).arg(tool_item2.xyz.y,0,'f',4).arg(tool_item2.xyz.z,0,'f',4);
	d_output_file.write(text.toLatin1());
	
	text = QString("Tool_Vector_2:%1,%2,%3\n\n").arg(tool_item2.ijk.x,0,'f',12).arg(tool_item2.ijk.y,0,'f',12).arg(tool_item2.ijk.z,0,'f',12);
	d_output_file.write(text.toLatin1());
	
	feature_memory = this->Feature_Memory(MEMORY_SPHERE1_ROLL_DCC,&valid);
	
	if(valid)
	{
		center = feature_memory.xyz;
		diameter = feature_memory.diameter;
		form = feature_memory.form;
		
		text = QString("Sphere_1_Tool_1 XYZ: %2,%3,%4  Diam: %5  Form: %6\n")
				.arg(center.x,0,'f',5)
				.arg(center.y,0,'f',5)
				.arg(center.z,0,'f',5)
				.arg(diameter,0,'f',5)
				.arg(form,0,'f',5);
		
		d_output_file.write(text.toLatin1());
	}
	else
	{
		emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(MEMORY_SPHERE1_ROLL_DCC));
	}

	feature_memory = this->Feature_Memory(MEMORY_SPHERE2_ROLL_DCC,&valid);
	
	if(valid)
	{
		center = feature_memory.xyz;
		diameter = feature_memory.diameter;
		form = feature_memory.form;
		
		text = QString("Sphere_1_Tool_2 XYZ: %2,%3,%4  Diam: %5  Form: %6\n")
				.arg(center.x,0,'f',5)
				.arg(center.y,0,'f',5)
				.arg(center.z,0,'f',5)
				.arg(diameter,0,'f',5)
				.arg(form,0,'f',5);

		d_output_file.write(text.toLatin1());
	}
	else
	{
		emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(MEMORY_SPHERE2_ROLL_DCC));
	}

	feature_memory = this->Feature_Memory(MEMORY_SPHERE3_ROLL_DCC,&valid);
	
	if(valid)
	{
		center = feature_memory.xyz;
		diameter = feature_memory.diameter;
		form = feature_memory.form;
		
		text = QString("Sphere_2_Tool_2 XYZ: %2,%3,%4  Diam: %5  Form: %6\n")
				.arg(center.x,0,'f',5)
				.arg(center.y,0,'f',5)
				.arg(center.z,0,'f',5)
				.arg(diameter,0,'f',5)
				.arg(form,0,'f',5);

		d_output_file.write(text.toLatin1());
	}
	else
	{
		emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(MEMORY_SPHERE3_ROLL_DCC));
	}

	feature_memory = this->Feature_Memory(MEMORY_SPHERE4_ROLL_DCC,&valid);
	
	if(valid)
	{
		center = feature_memory.xyz;
		diameter = feature_memory.diameter;
		form = feature_memory.form;
		
		text = QString("Sphere_2_Tool_1 XYZ: %2,%3,%4  Diam: %5  Form: %6\n")
				.arg(center.x,0,'f',5)
				.arg(center.y,0,'f',5)
				.arg(center.z,0,'f',5)
				.arg(diameter,0,'f',5)
				.arg(form,0,'f',5);

		d_output_file.write(text.toLatin1());
	}
	else
	{
		emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(MEMORY_SPHERE4_ROLL_DCC));
	}

	return true;
}

bool TMeasureRoutines::Report_Tool(void)
{
	std::vector<TToolWidget::TToolItem>::iterator iter;
	TCommandFeature						*command_feature;
	double								measured_sphere_diameter;
	double								nominal_sphere_diameter;
	double								tip_diameter;
	int									memory_cntr;
	QString								text;
	QString								tool_name_tip;
	
	nominal_sphere_diameter = d_parameters.sphere_nominal_diameter;
	
	text = QString("Nominal_Sphere_Diameter: %1\n\n").arg(nominal_sphere_diameter,0,'f',5);
	
	d_output_file.write(text.toLatin1());
	
	 for(iter = d_calibration_tool_items.begin(),memory_cntr = 0;iter != d_calibration_tool_items.end();++iter,++memory_cntr)
	 {
		command_feature = d_command_sequence.Get_Feature_Command(MEMORY_SPHERE_CALIBRATE_TOOL + memory_cntr);
		
		if(command_feature && command_feature->Type() == TCommand::COMMAND_MEASURE_SPHERE)
		{
			tool_name_tip = QString("%1_%2_%3").arg(d_calibration_tool_name).arg((*iter).angle_a,0,'f',1).arg((*iter).angle_b,0,'f',1);
	 
			measured_sphere_diameter = command_feature->Actual_Diameter();
	 
			// return to uncompensated size
			measured_sphere_diameter += (*iter).tip_diameter;
	 
			tip_diameter = measured_sphere_diameter - nominal_sphere_diameter;
	 
			if(tip_diameter < 0.0)
			{
				emit Add_Log_Text(QString("ERR:  Tip Diameter for %1:  %2  invalid.  Using 4.0000").arg(tool_name_tip,-30).arg(tip_diameter,0,'f',4));
				tip_diameter = 4.0;
			}
			else
			{
				emit Add_Log_Text(QString("INF:  Tip Diameter for %1:  %2").arg(tool_name_tip,-30).arg(tip_diameter,0,'f',4));
			}
	 
			(*iter).tip_diameter = tip_diameter;
	 
			text = QString("Name:%1\n").arg(tool_name_tip);
			d_output_file.write(text.toLatin1());
	 
			text = QString(" Probe_Offset:%1,%2,%3\n").arg((*iter).xyz.x,0,'f',4).arg((*iter).xyz.y,0,'f',4).arg((*iter).xyz.z,0,'f',4);
			d_output_file.write(text.toLatin1());
	 
			text = QString(" Probe_Vector:%1,%2,%3\n").arg((*iter).ijk.x,0,'f',12).arg((*iter).ijk.y,0,'f',12).arg((*iter).ijk.z,0,'f',12);
			d_output_file.write(text.toLatin1());
	 
			text = QString(" Tip_Diameter:%1\n\n").arg(tip_diameter,0,'f',5);
			d_output_file.write(text.toLatin1());
		}
		else
		{
			emit Add_Log_Text(QString("ERR:  Feature not found: Memory[%1]").arg(MEMORY_SPHERE_CALIBRATE_TOOL));
		}
	 }

	return true;
}

bool TMeasureRoutines::Vector_Different(
	const TVector3						&v1,
	const TVector3						&v2)
{
	if((fabs(v1.x - v2.x)) > ZERO_EPSILON)
	{
		return true;
	}
	
	if((fabs(v1.y - v2.y)) > ZERO_EPSILON)
	{
		return true;
	}
	
	if((fabs(v1.z - v2.z)) > ZERO_EPSILON)
	{
		return true;
	}
	
	return false;
}

QString TMeasureRoutines::Generate_Output_File_Name(void) const
{
	QString								file_name;
	
	file_name = d_output_path;
	
	switch(d_configuration)
	{
		case MeasureTypes::MEASURE_NULL:
			file_name.append(QString("/NULL.dat"));
			break;
			
		case MeasureTypes::MEASURE_STEPGAUGE_10360:
			file_name.append(QString("/B89.4.10360 Stepgauge - "));
			file_name.append(d_parameters.stepgauge_10360_position_name);
			file_name.append(" - ");
			file_name.append(QDateTime::currentDateTime().toString(" [ddd MMMM d yyyy hh-mm-ss]"));
			file_name.append(QStringLiteral(".dat"));
			break;
			
		case MeasureTypes::MEASURE_STEPGAUGE_B89:
			file_name.append(QString("/B89.4.1 Stepgauge - "));
			file_name.append(d_parameters.stepgauge_b89_position_name);
			file_name.append(" - ");
			file_name.append(QDateTime::currentDateTime().toString(" [ddd MMMM d yyyy hh-mm-ss]"));
			file_name.append(QStringLiteral(".dat"));
			break;
	
		case MeasureTypes::MEASURE_STEPGAUGE_SQUARE:
			file_name.append(QString("/Stepgauge Squareness - "));
			file_name.append(d_parameters.stepgauge_square_position_name);
			file_name.append(" - ");
			file_name.append(QDateTime::currentDateTime().toString(" [ddd MMMM d yyyy hh-mm-ss]"));
			file_name.append(QStringLiteral(".dat"));
			break;
			
		case MeasureTypes::MEASURE_GAUGEBLOCK:
			file_name.append(QString("/B89.4.10360 Gauge Block - "));
			file_name.append(d_parameters.gaugeblock_position_name);
			file_name.append(" - ");
			file_name.append(QDateTime::currentDateTime().toString(" [ddd MMMM d yyyy hh-mm-ss]"));
			file_name.append(QStringLiteral(".dat"));
			break;
			
		case MeasureTypes::MEASURE_POINT_REPEAT:
			file_name.append(QString("/Point Repeat - "));
			file_name.append(d_parameters.point_repeat_description);
			file_name.append(" - ");
			file_name.append(QDateTime::currentDateTime().toString(" [ddd MMMM d yyyy hh-mm-ss]"));
			file_name.append(QStringLiteral(".dat"));
			break;
			
		case MeasureTypes::MEASURE_SPHERE_REPEAT:
			file_name.append(QString("/B89.4.10360 Rpt"));
			file_name.append(" - ");
			file_name.append(QDateTime::currentDateTime().toString(" [ddd MMMM d yyyy hh-mm-ss]"));
			file_name.append(QStringLiteral(".dat"));
			break;
			
		case MeasureTypes::MEASURE_SPHERE_PFTU:
			file_name.append(QString("/B89.4.10360 Pftu"));
			file_name.append(" - ");
			file_name.append(QDateTime::currentDateTime().toString(" [ddd MMMM d yyyy hh-mm-ss]"));
			file_name.append(QStringLiteral(".dat"));
			break;
			
		case MeasureTypes::MEASURE_BALLBAR_B89:
			file_name.append(QString("/B89.4.1 BallBar"));
			file_name.append(" - ");
			file_name.append(QDateTime::currentDateTime().toString(" [ddd MMMM d yyyy]"));
			file_name.append(QStringLiteral(".dat"));
			break;
			
		case MeasureTypes::MEASURE_BALLBAR_10360:
			file_name.append(QString("/BallBar 10360 - "));
			file_name.append(d_parameters.ballbar_10360_position_name);
			file_name.append(" - ");
			file_name.append(QDateTime::currentDateTime().toString(" [ddd MMMM d yyyy hh-mm-ss]"));
			file_name.append(QStringLiteral(".dat"));
			break;

		case MeasureTypes::MEASURE_RINGGAUGE:
			file_name.append(QString("/Ring Gauge"));
			file_name.append(" - ");
			file_name.append(QDateTime::currentDateTime().toString(" [ddd MMMM d yyyy hh-mm-ss]"));
			file_name.append(QStringLiteral(".dat"));
			break;
			
		case MeasureTypes::MEASURE_PINGAUGE:
			file_name.append(QString("/Pin Gauge"));
			file_name.append(" - ");
			file_name.append(QDateTime::currentDateTime().toString(" [ddd MMMM d yyyy hh-mm-ss]"));
			file_name.append(QStringLiteral(".dat"));
			break;
			
		case MeasureTypes::MEASURE_ROLL_OFFSET_TOOL:
			file_name.append(QString("/Offset Tool Roll"));
			file_name.append(" - ");
			file_name.append(QDateTime::currentDateTime().toString(" [ddd MMMM d yyyy hh-mm-ss]"));
			file_name.append(QStringLiteral(".dat"));
			break;

		case MeasureTypes::CALIBRATE_TOOLS:
			file_name.append(QString("/Probe Calibration"));
			file_name.append(" - ");
			file_name.append(QDateTime::currentDateTime().toString(" [ddd MMMM d yyyy hh-mm-ss]"));
			file_name.append(QStringLiteral(".dat"));
			break;
	}
	
	return file_name;
}

void TMeasureRoutines::Initialize_Output_File(void)
{
	QString								text;
	QString								file_name;
	
	file_name = this->Generate_Output_File_Name();
	d_output_file.setFileName(file_name);
	
	if(QFile::exists(file_name))
	{
		if(!d_output_file.open(QIODevice::Append | QIODevice::Text))
		{
			emit Add_Log_Text(QString("ERR:  Cannot open output file: '%1'").arg(file_name));
			
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
		}
		
		return;
	}
	else
	{
		if(!d_output_file.open(QIODevice::WriteOnly | QIODevice::Text))
		{
			emit Add_Log_Text(QString("ERR:  Cannot open output file: '%1'").arg(file_name));
			
			d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;
			return;
		}
	}
	
	if(!d_output_file.isOpen())
	{
		return;
	}
	
	switch(d_configuration)
	{
		case MeasureTypes::MEASURE_NULL:
			break;
			
		case MeasureTypes::MEASURE_STEPGAUGE_10360:
			
			d_output_file.write("B89.4.10360_Raw_Measurement:Version=2:Type=Five_Gauge\n\n");
			
			d_output_file.write("Scales_Begin:\n");
			
			text = QString("X_Expansion_Coefficient:%1\n").arg(d_expansion_coeff_x,0,'f',1);
			d_output_file.write(text.toLatin1());
			
			text = QString("Y_Expansion_Coefficient:%1\n").arg(d_expansion_coeff_y,0,'f',1);
			d_output_file.write(text.toLatin1());
			
			text = QString("Z_Expansion_Coefficient:%1\n").arg(d_expansion_coeff_z,0,'f',1);
			d_output_file.write(text.toLatin1());
			
			d_output_file.write("Scales_End:\n\n");
			
			text = QString("Equipment_Identification:%1\n\n").arg(d_parameters.stepgauge_serial_number);
			d_output_file.write(text.toLatin1());
			
			break;
			
		case MeasureTypes::MEASURE_STEPGAUGE_B89:
			
			d_output_file.write("B89.4.1_Raw_Measurement:Version=1:Type=Step_Gauge\n\n");
			
			d_output_file.write("Scales_Begin:\n");
			
			text = QString("X_Expansion_Coefficient:%1\n").arg(d_expansion_coeff_x,0,'f',1);
			d_output_file.write(text.toLatin1());
			
			text = QString("Y_Expansion_Coefficient:%1\n").arg(d_expansion_coeff_y,0,'f',1);
			d_output_file.write(text.toLatin1());
			
			text = QString("Z_Expansion_Coefficient:%1\n").arg(d_expansion_coeff_z,0,'f',1);
			d_output_file.write(text.toLatin1());
			
			d_output_file.write("Scales_End:\n\n");
			
			text = QString("Equipment_Identification:%1\n\n").arg(d_parameters.stepgauge_serial_number);
			d_output_file.write(text.toLatin1());
			
			break;
			
		case MeasureTypes::MEASURE_STEPGAUGE_SQUARE:
			
			d_output_file.write("Stepgauge_Raw_Measurement:Version=1:Type=Squareness\n\n");
			
			d_output_file.write("Scales_Begin:\n");
			
			text = QString("X_Expansion_Coefficient:%1\n").arg(d_expansion_coeff_x,0,'f',1);
			d_output_file.write(text.toLatin1());
			
			text = QString("Y_Expansion_Coefficient:%1\n").arg(d_expansion_coeff_y,0,'f',1);
			d_output_file.write(text.toLatin1());
			
			text = QString("Z_Expansion_Coefficient:%1\n").arg(d_expansion_coeff_z,0,'f',1);
			d_output_file.write(text.toLatin1());
			
			d_output_file.write("Scales_End:\n\n");
			
			text = QString("Equipment_Identification:%1\n\n").arg(d_parameters.stepgauge_serial_number);
			d_output_file.write(text.toLatin1());
			
			break;
			
		case MeasureTypes::MEASURE_GAUGEBLOCK:
			
			d_output_file.write("B89.4.10360_Raw_Measurement:Version=1:Type=Single_Gauge\n\n");
			
			d_output_file.write("Scales_Begin:\n");
			
			text = QString("X_Expansion_Coefficient:%1\n").arg(d_expansion_coeff_x,0,'f',1);
			d_output_file.write(text.toLatin1());
			
			text = QString("Y_Expansion_Coefficient:%1\n").arg(d_expansion_coeff_y,0,'f',1);
			d_output_file.write(text.toLatin1());
			
			text = QString("Z_Expansion_Coefficient:%1\n").arg(d_expansion_coeff_z,0,'f',1);
			d_output_file.write(text.toLatin1());
			
			d_output_file.write("Scales_End:\n\n");
			
			text = QString("Equipment_Identification:%1\n\n").arg(d_parameters.gaugeblock_serial_number);
			d_output_file.write(text.toLatin1());
			
			break;
			
		case MeasureTypes::MEASURE_POINT_REPEAT:
			
			d_output_file.write("Artifact_Measurement:Version=1:Type=Point_Repeat\n\n");

			break;
			
		case MeasureTypes::MEASURE_SPHERE_REPEAT:
			
			d_output_file.write("B89.4.10360_Raw_Measurement:Version=2:Type=Sphere_Repeat\n\n");

			break;
			
		case MeasureTypes::MEASURE_SPHERE_PFTU:
			
			d_output_file.write("B89.4.10360_Raw_Measurement:Version=1:Type=Sphere_Pftu\n\n");
			
			break;
			
		case MeasureTypes::MEASURE_BALLBAR_B89:
			
			d_output_file.write("B89.4.1_Data:Version=1:Type=BallBar\n\n");
			
			d_output_file.write("# Note:  Sphere location offset by probe.\n\n");
			
			d_output_file.write("#   Name                 Length      Sph1 X      Sph1 Y      Sph1 Z      Sph2 X      Sph2 Y      Sph2 Z     Sph1 Form   Sph2 Form    Probe X     Probe Y     Probe Z\n");
			d_output_file.write("# -----------------------------------------------------------------------------------------------------------------------------------------------------------------------\n");
			break;
			
		case MeasureTypes::MEASURE_BALLBAR_10360:
			
			d_output_file.write("B89.4.10360_Raw_Measurement:Version=1:Type=BallBar_10360\n\n");

			break;

		case MeasureTypes::MEASURE_RINGGAUGE:
			
			d_output_file.write("Artifact_Measurement:Version=1:Type=Ring_Gauge\n\n");
			
			break;
			
		case MeasureTypes::MEASURE_PINGAUGE:
			
			d_output_file.write("Artifact_Measurement:Version=1:Type=Pin_Gauge\n\n");
			
			break;
			
		case MeasureTypes::MEASURE_ROLL_OFFSET_TOOL:
			
			d_output_file.write("Artifact_Measurement:Version=1:Type=Roll_Offset_Tool\n\n");

			break;

		case MeasureTypes::CALIBRATE_TOOLS:
			
			d_output_file.write("Artifact_Measurement:Version=2:Type=Tip_Calibration\n\n");
			
			break;
	}
}


QString TMeasureRoutines::Axis_Name(
	const TReferenceSystem::TAxis		axis)
{
	switch(axis)
	{
		case TReferenceSystem::ZPLUS:
			return QStringLiteral("ZPlus");
		case TReferenceSystem::ZMINUS:
			return QStringLiteral("ZMinus");
		case TReferenceSystem::YPLUS:
			return QStringLiteral("YPlus");
		case TReferenceSystem::YMINUS:
			return QStringLiteral("YMinus");
		case TReferenceSystem::XPLUS:
			return QStringLiteral("XPlus");
		case TReferenceSystem::XMINUS:
			return QStringLiteral("XMinus");
	}
	
	return QString();
}


