/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QCheckBox>
#include <QFrame>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QComboBox>
#include <QPlainTextEdit>
#include <QSpacerItem>
#include <QToolButton>
#include <QPushButton>
#include <QVBoxLayout>
#include <QWidget>
#include <QCloseEvent>
#include <QSizePolicy>
#include <QIcon>
#include <QSize>
#include <QRect>
#include <QFont>
#include <cmath>
#include <QDir>
#include <cmath>
#include <assert.h>

#include "../../../core/messagebox.h"

#include "inputwidget.h"
#include "pointcountwidget.h"
#include "pathpoint.h"
#include "coordinatedisplaywidget.h"
#include "tempcompinformationdialog.h"
#include "measure_routines.h"

#include "measuredialog.h"

static const double						ZERO_EPSILON(0.000001);

TMeasureDialog::TMeasureDialog(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QDialog(const_cast<QWidget*>(parent),flags)
{
	QFrame								*separator1_hline;
	QFrame								*separator2_hline;
	QFrame								*separator3_hline;
	QGridLayout							*input_widget_layout;
	QGridLayout							*statusbar_layout;
	QHBoxLayout							*executionbar_hlayout;
	QLabel								*tool_selection_label;
	QSpacerItem							*executionbar_hspacer;
	QSpacerItem							*statusbar_hspacer;
	QVBoxLayout							*verticalLayout;
	QWidget								*executionbar_spacer_widget;
	QSizePolicy							sizepolicy_expanding_minimum_expanding(QSizePolicy::Expanding, QSizePolicy::MinimumExpanding);
	QFont								text_font(QStringLiteral("courier new"));
	
	this->resize(801,601);
	text_font.setFixedPitch(true);
	
	sizepolicy_expanding_minimum_expanding.setHorizontalStretch(0);
	sizepolicy_expanding_minimum_expanding.setVerticalStretch(1);

	verticalLayout = new QVBoxLayout(this);

	executionbar_hlayout = new QHBoxLayout();
	
	d_play_button = new QToolButton(this);
	d_play_button->setMinimumSize(32,32);
	d_play_button->setMaximumSize(32,32);
	d_play_button->setIconSize(QSize(32,32));
	executionbar_hlayout->addWidget(d_play_button);
	
	d_pause_button = new QToolButton(this);
	d_pause_button->setMinimumSize(32,32);
	d_pause_button->setMaximumSize(32,32);
	d_pause_button->setIconSize(QSize(32,32));
	d_pause_button->setCheckable(true);
	executionbar_hlayout->addWidget(d_pause_button);
	
	d_stop_button = new QToolButton(this);
	d_stop_button->setMinimumSize(32,32);
	d_stop_button->setMaximumSize(32,32);
	d_stop_button->setIconSize(QSize(32,32));
	executionbar_hlayout->addWidget(d_stop_button);
	
	executionbar_spacer_widget = new QWidget(this);
	executionbar_spacer_widget->setMinimumSize(32,32);
	executionbar_spacer_widget->setMaximumSize(32,32);
	executionbar_hlayout->addWidget(executionbar_spacer_widget);

	tool_selection_label = new QLabel(this);
	executionbar_hlayout->addWidget(tool_selection_label);
	
	d_active_tool_combo = new QComboBox(this);
	d_active_tool_combo->setSizeAdjustPolicy(QComboBox::AdjustToContents);
	executionbar_hlayout->addWidget(d_active_tool_combo);

	executionbar_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	executionbar_hlayout->addItem(executionbar_hspacer);
	
	d_close_button = new QToolButton(this);
	d_close_button->setMinimumSize(32,32);
	d_close_button->setMaximumSize(32,32);
	d_close_button->setIconSize(QSize(32,32));
	executionbar_hlayout->addWidget(d_close_button);
	
	verticalLayout->addLayout(executionbar_hlayout);

	separator1_hline = new QFrame(this);
	separator1_hline->setFrameShape(QFrame::HLine);
	separator1_hline->setFrameShadow(QFrame::Sunken);
	verticalLayout->addWidget(separator1_hline);

	statusbar_layout = new QGridLayout();
	
	d_icon_label = new QLabel(this);
	d_icon_label->setFixedSize(128,128);
	d_icon_label->setFrameShape(QFrame::StyledPanel);
	d_icon_label->setFrameShadow(QFrame::Plain);
	statusbar_layout->addWidget(d_icon_label,0,0,2,1);

	d_point_count_widget = new TPointCountWidget(this);
	statusbar_layout->addWidget(d_point_count_widget,0,1,2,1);

	d_erase_button = new QPushButton(this);
	d_erase_button->setFixedSize(64,64);
	d_erase_button->setIconSize(QSize(64,64));
	d_erase_button->setFlat(true);
	d_erase_button->setFocusPolicy(Qt::NoFocus);
	d_erase_button->setAutoDefault(false);
	statusbar_layout->addWidget(d_erase_button,0,2,1,1);

	d_next_button = new QPushButton(this);
	d_next_button->setFixedSize(64,64);
	d_next_button->setIconSize(QSize(64,64));
	d_next_button->setFlat(true);
	d_next_button->setAutoDefault(true);
	statusbar_layout->addWidget(d_next_button,1,2,1,1);
	
	statusbar_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	statusbar_layout->addItem(statusbar_hspacer,0,3,2,1);

	d_coordinate_display_widget = new TCoordinateDisplayWidget(this);
	statusbar_layout->addWidget(d_coordinate_display_widget,0,4,2,1);

	verticalLayout->addLayout(statusbar_layout);
	
	separator2_hline = new QFrame(this);
	separator2_hline->setFrameShape(QFrame::HLine);
	separator2_hline->setFrameShadow(QFrame::Sunken);
	verticalLayout->addWidget(separator2_hline);
	
	d_alignment_mode_label = new QLabel(this);
	d_alignment_mode_label->setAlignment(Qt::AlignRight | Qt::AlignTrailing | Qt::AlignVCenter);
	
	d_alignment_options_combo = new QComboBox(this);

	d_temperature_compensation_check = new QCheckBox(this);

	input_widget_layout = new QGridLayout();

	d_input_widget_left = new TInputWidget(TInputWidget::MODE_LEFT,this);
	input_widget_layout->addWidget(d_input_widget_left,0,0,1,1);
	
	d_input_widget_right = new TInputWidget(TInputWidget::MODE_RIGHT,this);
	input_widget_layout->addWidget(d_input_widget_right,0,1,1,1);
	
	d_input_widget_left->Set_Static_Widget(d_alignment_mode_label,0,1);
	d_input_widget_left->Set_Static_Widget(d_alignment_options_combo,1,1);
	d_input_widget_right->Set_Static_Widget(d_temperature_compensation_check,1,2);
	
	verticalLayout->addLayout(input_widget_layout);

	separator3_hline = new QFrame(this);
	separator3_hline->setFrameShape(QFrame::HLine);
	separator3_hline->setFrameShadow(QFrame::Sunken);
	verticalLayout->addWidget(separator3_hline);

	d_information_text = new QPlainTextEdit(this);
	d_information_text->setFont(text_font);
	d_information_text->setLineWrapMode(QPlainTextEdit::NoWrap);
	d_information_text->setReadOnly(true);
	sizepolicy_expanding_minimum_expanding.setHeightForWidth(d_information_text->sizePolicy().hasHeightForWidth());
	d_information_text->setSizePolicy(sizepolicy_expanding_minimum_expanding);
	verticalLayout->addWidget(d_information_text);
	
	d_message_box = new TMessageBox(this);
	d_tempcomp_information_dialog = new TTempcompInformationDialog(this,Qt::WindowFlags(Qt::Tool));
	
	this->setWindowTitle(QStringLiteral("Measure Artifact"));
	
	d_play_button->setIcon(QIcon(":/icon32/play_icon32.png"));
	d_pause_button->setIcon(QIcon(":/icon32/pause_icon32.png"));
	d_stop_button->setIcon(QIcon(":/icon32/stop_icon32.png"));
	d_close_button->setIcon(QIcon(":/icon32/close_icon32.png"));
	d_erase_button->setIcon(QIcon(":/icon64/erase_icon64.png"));
	d_next_button->setIcon(QIcon(":/icon64/next_icon64.png"));
	
	this->Set_Pixmap(QPixmap(":/icon128/idle_icon128.png"),false);
	
	d_alignment_options_combo->addItem(QStringLiteral("Full Alignment"),MeasureTypes::ALIGNMENT_FULL);
	d_alignment_options_combo->addItem(QStringLiteral("Skip Manual Alignment"),MeasureTypes::ALIGNMENT_SKIP_MANUAL);
	d_alignment_options_combo->addItem(QStringLiteral("Skip Alignment"),MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC);
	d_alignment_options_combo->addItem(QStringLiteral("Alignment Only"),MeasureTypes::ALIGNMENT_ONLY);
	
	this->Set_Alignment_Option(MeasureTypes::ALIGNMENT_FULL);
	
	d_temperature_compensation_check->setText(QStringLiteral("Enable Temperature Compensation"));
	tool_selection_label->setText(QStringLiteral("Active Tool:"));
	d_alignment_mode_label->setText(QStringLiteral("Alignment Mode:"));
	
	d_play_button->setToolTip("Play Sequence");
	d_pause_button->setToolTip("Pause Sequence");
	d_stop_button->setToolTip("Stop Sequence");
	d_erase_button->setToolTip("Erase Last Hit");
	d_next_button->setToolTip("Next Step");
	d_close_button->setToolTip("Close Dialog");
	
	this->Set_State(TMeasureDialog::STATE_STOPPED);
	d_pause_button->setChecked(false);	// not paused
	d_erase_button->setEnabled(false);
	d_next_button->setEnabled(false);
	
	d_active_feature = 0;
	
	d_controller_type = TController::CONTROLLER_VIRTUAL;
	d_last_stepgauge_type = MeasureTypes::STEPGAUGE_NULL;
	d_configuration = MeasureTypes::MEASURE_NULL;
	d_last_configuration = MeasureTypes::MEASURE_NULL;
	d_display_coordinate_system = MeasureTypes::COORDINATE_SYSTEM_MACHINE;
	d_thermal_compensation_mode = TMeasureDialog::SOFTWARE_COMPENSATION_AXIS_AND_PART;
	
	d_enable_temperature_compensation = false;
	d_waiting_driver_sync = false;
	
	connect(d_play_button,&QToolButton::clicked,this,&TMeasureDialog::Start);
	connect(d_pause_button,&QToolButton::toggled,this,&TMeasureDialog::Pause);
	connect(d_stop_button,&QToolButton::clicked,this,&TMeasureDialog::Stop);
	connect(d_erase_button,&QPushButton::clicked,this,&TMeasureDialog::Erase);
	connect(d_next_button,&QPushButton::clicked,this,&TMeasureDialog::Next);
	connect(d_close_button,&QToolButton::clicked,this,&TMeasureDialog::Close);
	connect(d_temperature_compensation_check,&QCheckBox::toggled,this,&TMeasureDialog::Temperature_Compensation_Toggled);
	connect(d_active_tool_combo,static_cast<void (QComboBox::*)(const int)>(&QComboBox::activated),this,&TMeasureDialog::Tool_Selection_Changed);
	
	connect(d_input_widget_left,&TInputWidget::Selection_Changed_Combo,this,&TMeasureDialog::Input_Left_Option_Changed);
	connect(d_input_widget_right,&TInputWidget::Selection_Changed_Combo,this,&TMeasureDialog::Input_Right_Option_Changed);

	connect(&d_measure_routines,&TMeasureRoutines::Add_Log_Text,this,&TMeasureDialog::Add_Log_Text);
	connect(&d_measure_routines,&TMeasureRoutines::Update_Last_Configuration,this,&TMeasureDialog::Update_Last_Configuration);
}

TMeasureDialog::~TMeasureDialog(void)
{
}

std::vector<TDriver::TCommand> TMeasureDialog::Get_Next_Driver_Commands(void)
{
	std::vector<TToolWidget::TToolItem>	calibration_tool_items;
	std::vector<TToolWidget::TToolItem>::const_iterator tool_item_iter;
	std::vector<TDriver::TCommand>		driver_commands;
	std::vector<TPathPoint>				path_points;
	std::vector<TPathPoint>::iterator	path_iter;
	QString								text;
	TDriver::TCommand					driver_command;
	TCommand							*command;
	TCommandMove						*command_move;
	TCommandMoveCube					*command_move_cube;
	TCommandMode						*command_mode;
	TCommandSetTool						*command_set_tool;
	TCommandQueryTool					*command_query_tool;
	TCommandPrompt						*command_prompt;
	TCommandSync						*command_sync;
	TCommandMoveCube::TAxis				cube_axis;
	TVector3							refsys_origin;
	TVector3							pnt;
	TVector3							v1,v2;
	double								cube_position;
	double								dval;
	int									ival;
	bool								command_state;
	bool								state;

	if(d_current_state != TMeasureDialog::STATE_RUNNING)
	{
		return driver_commands;
	}
	
	if(d_active_feature)
	{
		return driver_commands;
	}
	
	if(d_measure_routines.Is_Sequence_Complete())
	{
		if(d_waiting_driver_sync)
		{
			return driver_commands;
		}
				
		d_measure_routines.Increment_Current_Sequence_Step(static_cast<MeasureTypes::TAlignmentOptions>(d_alignment_options_combo->currentData(Qt::UserRole).toInt()));
		d_measure_routines.Generate_Sequence();
		
		if(d_measure_routines.Current_Sequence_Function() == MeasureTypes::SEQUENCE_NULL)
		{
			this->Set_State(TMeasureDialog::STATE_STOPPED);
			
			d_tempcomp_information_dialog->hide();

			d_active_feature = 0;
			d_erase_button->setEnabled(false);
			d_next_button->setEnabled(false);
			
			if(d_configuration != MeasureTypes::MEASURE_ROLL_OFFSET_TOOL &&
			   d_configuration != MeasureTypes::CALIBRATE_TOOLS)
			{
				d_active_tool_combo->setEnabled(true);
			}
	
			d_measure_routines.Delete_Sequence_Commands();
			
			if(d_configuration == MeasureTypes::CALIBRATE_TOOLS)
			{
				calibration_tool_items = d_measure_routines.Calibration_Tool_Items();
				text = d_measure_routines.Calibration_Tool_Name();

				for(tool_item_iter = calibration_tool_items.begin();tool_item_iter != calibration_tool_items.end();++tool_item_iter)
				{
					emit Update_Tip_Diameter(text,(*tool_item_iter).angle_a,(*tool_item_iter).angle_b,(*tool_item_iter).tip_diameter);
				}
			}
		
			this->Add_Log_Text(QStringLiteral("INF:  Sequence Complete."));
			emit Sequence_End();

			return driver_commands;
		}
	}

	do
	{
		command = d_measure_routines.Get_Next_Command(&command_state);
		
		if(command_state)
		{
			d_measure_routines.Process_Command(d_display_coordinate_system,command);
			
			switch(command->Type())
			{
				case TCommand::COMMAND_NULL:
					break;
					
				case TCommand::COMMAND_MODAL_PROMPT:
					command_prompt = reinterpret_cast<TCommandPrompt*>(command);
					
					this->Add_Log_Text(command_prompt->Prompt_Text());
					break;
					
				case TCommand::COMMAND_DIMENSION:
					break;
					
				case TCommand::COMMAND_ALIGN:
					if(!d_pause_button->isChecked())
					{
						this->Set_Pixmap(QPixmap(":/icon128/alignment_icon128.png"),true);
					}
					break;
					
				case TCommand::COMMAND_QUERY_TOOL:
					
					command_query_tool = reinterpret_cast<TCommandQueryTool*>(command);
					
					driver_command.command_type = TDriver::DRIVER_QUERY_TOOL_DATA;
					driver_command.text = command_query_tool->Name();

					driver_commands.push_back(driver_command);

					break;

				case TCommand::COMMAND_SET_TOOL:
					
					command_set_tool = reinterpret_cast<TCommandSetTool*>(command);

					this->Add_Log_Text(QString("INF:  Tool Change: %1").arg(command_set_tool->Name()));
				
					d_measure_routines.Set_Active_Tool_Name(command_set_tool->Name());
					
					driver_command.command_type = TDriver::DRIVER_SET_TOOL_NAME;
					driver_command.text = command_set_tool->Name();
					driver_commands.push_back(driver_command);
					
					if(d_controller_type != TController::CONTROLLER_IPPCLIENT)	
					{
						driver_command.command_type = TDriver::DRIVER_SET_TOOL_TYPE;
						driver_command.text = QStringLiteral("TTP");
						driver_commands.push_back(driver_command);

						driver_command.command_type = TDriver::DRIVER_SET_TOOL_AB_ANGLES;
						driver_command.dvalue1 = command_set_tool->Angle_A();
						driver_command.dvalue2 = command_set_tool->Angle_B();
						driver_commands.push_back(driver_command);

						driver_command.command_type = TDriver::DRIVER_SET_TOOL_DATA;
						driver_command.xyz = command_set_tool->Offset();
						driver_command.dvalue1 = command_set_tool->Tip_Diameter();
						driver_commands.push_back(driver_command);
					}
					
					break;
					
				case TCommand::COMMAND_MODAL_MODE:
					
					command_mode = reinterpret_cast<TCommandMode*>(command);
					
					if(!command_mode->Is_Manual_Mode())
					{
						if(!d_pause_button->isChecked())
						{
							this->Set_Pixmap(QPixmap(":/icon128/warning_icon128.png"),true);
						}

						d_message_box->setText("DCC Mode");
						d_message_box->setInformativeText("Clear Probe For DCC Measurements");
						d_message_box->setDetailedText(QString());
						d_message_box->setStandardButtons(QMessageBox::Ok | QMessageBox::Cancel);
						d_message_box->setDefaultButton(QMessageBox::Ok);
						d_message_box->setIcon(QMessageBox::Warning);
						
						if(QMessageBox::Ok == d_message_box->exec())
						{
							driver_command.command_type = TDriver::DRIVER_SET_MODE_DCC;
							driver_commands.push_back(driver_command);
						}
						else
						{
							this->Stop();
						}
					}
					else
					{						
						driver_command.command_type = TDriver::DRIVER_SET_MODE_MANUAL;
						driver_commands.push_back(driver_command);
					}
					
					break;
				case TCommand::COMMAND_MODAL_SYNC:
					
					command_sync = reinterpret_cast<TCommandSync*>(command);
					
					driver_command.command_type = TDriver::DRIVER_SYNC;
					driver_command.ivalue = command_sync->Value();
					driver_commands.push_back(driver_command);
					
					d_waiting_driver_sync = true;

					break;
					
				case TCommand::COMMAND_MOVE:
					
					command_move = reinterpret_cast<TCommandMove*>(command);
					
					if(command_move->Has_Move_Points())
					{
						driver_command.ijk = TVector3();

						do
						{
							pnt = command_move->Get_Next_Point(&state);
							
							if(state)
							{
								driver_command.command_type = TDriver::DRIVER_MOVE_TO;
								driver_command.xyz = d_measure_routines.Refsys_ToWorld(pnt);

								if(d_temperature_compensation_check->isChecked())
								{
									driver_command.xyz = this->Remove_Temperature_Compensation(driver_command.xyz);
								}
								
								driver_commands.push_back(driver_command);
							}
							
						}while(state);

						return driver_commands;
					}
					break;
					
				case TCommand::COMMAND_MOVE_CUBE:
					
					command_move_cube = reinterpret_cast<TCommandMoveCube*>(command);
					
					if(command_move_cube->Has_Move_Points())
					{
						driver_command.ijk = TVector3();
						
						cube_axis = command_move_cube->Axis();
						cube_position = command_move_cube->Position();
						refsys_origin = d_measure_routines.Refsys_Origin();
						
						do
						{
							pnt = command_move_cube->Get_Next_Point(&state);
							
							if(state)
							{
								pnt = d_measure_routines.Refsys_ToWorld(pnt);
								
								switch(cube_axis)
								{
									case TCommandMoveCube::XMINUS:
										v1.Set(-1,0,0);
										break;
									case TCommandMoveCube::XPLUS:
										v1.Set(1,0,0);
										break;
									case TCommandMoveCube::YMINUS:
										v1.Set(0,-1,0);
										break;
									case TCommandMoveCube::YPLUS:
										v1.Set(0,1,0);
										break;
									case TCommandMoveCube::ZMINUS:
										v1.Set(0,0,-1);
										break;
									case TCommandMoveCube::ZPLUS:
										v1.Set(0,0,1);
										break;
								}
								
								v2 = refsys_origin + v1 * cube_position;
								
								dval = (v2 - pnt) ^ v1;
								pnt += (v1 * dval);
								
								driver_command.command_type = TDriver::DRIVER_MOVE_TO;
								driver_command.xyz = pnt;
								
								if(d_temperature_compensation_check->isChecked())
								{
									driver_command.xyz = this->Remove_Temperature_Compensation(driver_command.xyz);
								}
								
								driver_commands.push_back(driver_command);
							}
							
						}while(state);
						
						return driver_commands;
					}
					break;
					
				
				case TCommand::COMMAND_MEASURE_PLANE:
				case TCommand::COMMAND_MEASURE_LINE:
				case TCommand::COMMAND_MEASURE_POINT:
				case TCommand::COMMAND_MEASURE_SPHERE:
				case TCommand::COMMAND_MEASURE_CIRCLE:
					
					if(!d_pause_button->isChecked())
					{
						if(command->Type() == TCommand::COMMAND_MEASURE_PLANE)
						{
							this->Set_Pixmap(QPixmap(":/icon128/plane_icon128.png"),true);
						}
						else if(command->Type() == TCommand::COMMAND_MEASURE_LINE)
						{
							this->Set_Pixmap(QPixmap(":/icon128/line_icon128.png"),true);
						}
						else if(command->Type() == TCommand::COMMAND_MEASURE_POINT)
						{
							this->Set_Pixmap(QPixmap(":/icon128/point_icon128.png"),true);
						}
						else if(command->Type() == TCommand::COMMAND_MEASURE_SPHERE)
						{
							this->Set_Pixmap(QPixmap(":/icon128/sphere_icon128.png"),true);
						}
						else if(command->Type() == TCommand::COMMAND_MEASURE_CIRCLE)
						{
							this->Set_Pixmap(QPixmap(":/icon128/circle_icon128.png"),true);
						}
					}
					
					d_active_feature = reinterpret_cast<TCommandFeature*>(command);
					
					d_point_count_widget->Set_Nominal_Point_Count(d_active_feature->Nominal_Point_Count());
					
					ival = d_active_feature->Actual_Point_Count();
					
					d_erase_button->setEnabled((ival > 0) && d_measure_routines.Manual_Mode());
					d_point_count_widget->Set_Actual_Point_Count(ival);
					
					path_points = d_active_feature->Path_Points();
					
					for(path_iter = path_points.begin();path_iter != path_points.end();++path_iter)
					{
						switch((*path_iter).Type())
						{
							case TPathPoint::MOVE_POINT:
								
								driver_command.command_type = TDriver::DRIVER_MOVE_TO;
								driver_command.xyz = d_measure_routines.Refsys_ToWorld((*path_iter).XYZ());
								driver_command.ijk = TVector3();
								
								if(d_temperature_compensation_check->isChecked())
								{
									driver_command.xyz = this->Remove_Temperature_Compensation(driver_command.xyz);
								}
								
								driver_commands.push_back(driver_command);

								break;
								
							case TPathPoint::TOUCH_POINT:
								
								driver_command.command_type = TDriver::DRIVER_TOUCH_POINT;
								driver_command.xyz = d_measure_routines.Refsys_ToWorld((*path_iter).XYZ());
								driver_command.ijk = d_measure_routines.Refsys_AxisToWorld((*path_iter).IJK());
								
								if(d_temperature_compensation_check->isChecked())
								{
									// same idea as Remove_Temperature_Compensation() but does not include tip radius.
									driver_command.xyz -= this->Temperature_Compensation_Correction(driver_command.xyz - driver_command.ijk * d_measure_routines.Tip_Radius());
								}

								driver_commands.push_back(driver_command);
								
							break;
						}
					}
					
					return driver_commands;		// only send one feature at a time.
			}
		}
		
	} while(command_state);
	
	return driver_commands;
}

void TMeasureDialog::Driver_Error(
	const QString						&error,
	const int							severity)
{
	this->Add_Log_Text(QString("ERR:  Severity %1: '%2'").arg(severity).arg(error));

	if(d_current_state != TMeasureDialog::STATE_PAUSED)		// errors expected from controller when switching to paused state
	{
		if(d_current_state == TMeasureDialog::STATE_RUNNING)
		{
			this->Add_Log_Text(QStringLiteral("INF:  Measurement Sequence Cancelled"));
		}
		
		// send stop and reset for sever errors
		if(severity > 0)
		{
			this->Stop();
		}
	}
}

void TMeasureDialog::Sequence_Canceled(void)
{
	this->Stop();
}

void TMeasureDialog::Done_Key_Pressed(void)
{
	if(d_message_box->isVisible())
	{
		d_message_box->Send_Key(Qt::Key_Return);
	}
	
	if(d_next_button->isEnabled())
	{
		this->Next();
	}
}

void TMeasureDialog::Erase_Key_Pressed(void)
{
	this->Erase();
}

void TMeasureDialog::Enable_Temperature_Compensation(
	const bool							state)
{
	d_enable_temperature_compensation = state;
}

void TMeasureDialog::Set_Calibration_Tool_Data(
	const QString						&name,
	const std::vector<TToolWidget::TToolItem> &data)
{
	d_measure_routines.Set_Calibration_Tool_Data(name,data);
}

void TMeasureDialog::Reset(
	const TController::TControllerType 	controller_type,
	const MeasureTypes::TConfiguration	configuration,
	const QString						&output_path,
	const QStringList					&tool_list,
	const QString						&active_tool)
{
	bool								prev_state;
	int									index;
	TMeasureRoutines::TParameters		parameters;
		
	parameters = d_measure_routines.Measurement_Parameters();

	d_configuration = configuration;
	d_controller_type = controller_type;
	
	d_measure_routines.Set_Output_Path(output_path);
	d_measure_routines.Set_Configuration(configuration);
	
	prev_state = d_active_tool_combo->blockSignals(true);
	
	d_active_tool_combo->clear();
	d_active_tool_combo->addItems(tool_list);
	
	// use active tool name fist index (no name)
	if(active_tool.length())
	{
		index = d_active_tool_combo->findText(active_tool);
		
		if(!(index < 0))
		{
			d_active_tool_combo->setCurrentIndex(index);
			d_measure_routines.Set_Active_Tool_Name(active_tool);
		}
		else
		{
			d_active_tool_combo->setCurrentIndex(0);	// default to '-- select tool --'
		}
	}
	else if(d_active_tool_combo->count() > 0)
	{
		d_active_tool_combo->setCurrentIndex(0);	// default to '-- select tool --'
	}

	d_active_tool_combo->blockSignals(prev_state);
	d_active_tool_combo->setEnabled(true);
	
	prev_state = d_temperature_compensation_check->blockSignals(true);
	
	switch(d_configuration)
	{
		case MeasureTypes::MEASURE_NULL:

			this->setWindowTitle(QStringLiteral("Measure Direct"));
			d_configuration_pixmap = QPixmap(":/icon128/idle_icon128.png");
			
			break;
			
		case MeasureTypes::MEASURE_STEPGAUGE_10360:
			
			this->setWindowTitle(QStringLiteral("Measure Stepgauge 10360-2"));

			switch(parameters.stepgauge_type)
			{
				case MeasureTypes::STEPGAUGE_CHECKMASTER:
				case MeasureTypes::STEPGAUGE_NULL:
					d_configuration_pixmap = QPixmap(":/icon128/stepgauge_checkmaster_icon128.png");
					break;
					
				case MeasureTypes::STEPGAUGE_STARRET:
					d_configuration_pixmap = QPixmap(":/icon128/stepgauge_webber_icon128.png");
					break;
					
				case MeasureTypes::STEPGAUGE_KOBA:
					d_configuration_pixmap = QPixmap(":/icon128/stepgauge_koba_icon128.png");
					break;
			}

			if(d_thermal_compensation_mode == TMeasureDialog::SOFTWARE_COMPENSATION_AXIS_AND_PART)
			{
				d_temperature_compensation_check->setChecked(d_enable_temperature_compensation);
				d_temperature_compensation_check->setEnabled(true);
			}
			else
			{
				d_temperature_compensation_check->setChecked(true);
				d_temperature_compensation_check->setEnabled(false);
			}
			
			d_input_widget_left->Set_Entry_Count(6);
			d_input_widget_right->Set_Entry_Count(5);
			
			d_input_widget_left->Configure_Entry_As_Combo(0);
			d_input_widget_left->Set_Entry_Label_Text(0,QStringLiteral("Stepgauge Type:"));
			d_input_widget_left->Clear_Entry_Item_Combo(0);
			d_input_widget_left->Add_Entry_Combo_Item(0,QStringLiteral("Checkmaster"),MeasureTypes::STEPGAUGE_CHECKMASTER);
			d_input_widget_left->Add_Entry_Combo_Item(0,QStringLiteral("Starrett Webber"),MeasureTypes::STEPGAUGE_STARRET);
			d_input_widget_left->Add_Entry_Combo_Item(0,QStringLiteral("Koba"),MeasureTypes::STEPGAUGE_KOBA);
			d_input_widget_left->Select_Entry_Item_Combo_Data(0,parameters.stepgauge_type);
			d_input_widget_left->Enable_Entry_Item(0,true);
			
			d_input_widget_left->Configure_Entry_As_Text(1);
			d_input_widget_left->Set_Entry_Label_Text(1,QStringLiteral("Position Name:"));
			d_input_widget_left->Set_Entry_Item_Text(1,parameters.stepgauge_10360_position_name);
			
			d_input_widget_left->Configure_Entry_As_Text(2);
			d_input_widget_left->Set_Entry_Label_Text(2,QStringLiteral("Gauge Serial Number:"));
			d_input_widget_left->Set_Entry_Item_Text(2,parameters.stepgauge_serial_number);
			
			d_input_widget_left->Configure_Entry_As_Text(3);
			d_input_widget_left->Set_Entry_Label_Text(3,QStringLiteral("Measurement Zero Position (mm):"));
			d_input_widget_left->Set_Entry_Item_Text(3,QString("%1").arg(parameters.stepgauge_measurement_zero_position,0,'f',4));
			
			d_input_widget_left->Configure_Entry_As_Text(4);
			d_input_widget_left->Set_Entry_Label_Text(4,QStringLiteral("Zero Step Distance (mm):"));
			d_input_widget_left->Set_Entry_Item_Text(4,QString("%1").arg(parameters.stepgauge_zero_step_spacing,0,'f',4));
			
			d_input_widget_left->Configure_Entry_As_Text(5);
			d_input_widget_left->Set_Entry_Label_Text(5,QStringLiteral("Nominal Block Spacing (mm):"));
			d_input_widget_left->Set_Entry_Item_Text(5,QString("%1").arg(parameters.stepgauge_nominal_block_spacing,0,'f',4));

			d_input_widget_right->Configure_Entry_As_Text(0);
			d_input_widget_right->Set_Entry_Label_Text(0,QStringLiteral("X Axis Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(0,QString("%1").arg(parameters.axis_x_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(0,true);

			d_input_widget_right->Configure_Entry_As_Text(1);
			d_input_widget_right->Set_Entry_Label_Text(1,QStringLiteral("Y Axis Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(1,QString("%1").arg(parameters.axis_y_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(1,true);
			
			d_input_widget_right->Configure_Entry_As_Text(2);
			d_input_widget_right->Set_Entry_Label_Text(2,QStringLiteral("Z Axis Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(2,QString("%1").arg(parameters.axis_z_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(2,true);

			d_input_widget_right->Configure_Entry_As_Text(3);
			d_input_widget_right->Set_Entry_Label_Text(3,QStringLiteral("Stepgauge Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(3,QString("%1").arg(parameters.part_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(3,true);

			d_input_widget_right->Configure_Entry_As_Text(4);
			d_input_widget_right->Set_Entry_Label_Text(4,QStringLiteral("Stepgauge Expansion Coeff (um/m/C):"));
			d_input_widget_right->Set_Entry_Item_Text(4,QString("%1").arg(parameters.stepgauge_expansion_coeff,0,'f',1));

			break;
			
		case MeasureTypes::MEASURE_STEPGAUGE_B89:
			
			this->setWindowTitle(QStringLiteral("Measure Stepgauge B89.4.1"));
			
			switch(parameters.stepgauge_type)
			{
				case MeasureTypes::STEPGAUGE_CHECKMASTER:
				case MeasureTypes::STEPGAUGE_NULL:
					d_configuration_pixmap = QPixmap(":/icon128/stepgauge_checkmaster_icon128.png");
					break;
					
				case MeasureTypes::STEPGAUGE_STARRET:
					d_configuration_pixmap = QPixmap(":/icon128/stepgauge_webber_icon128.png");
					break;
					
				case MeasureTypes::STEPGAUGE_KOBA:
					d_configuration_pixmap = QPixmap(":/icon128/stepgauge_koba_icon128.png");
					break;
			}
			
			if(d_thermal_compensation_mode == TMeasureDialog::SOFTWARE_COMPENSATION_AXIS_AND_PART)
			{
				d_temperature_compensation_check->setChecked(d_enable_temperature_compensation);
				d_temperature_compensation_check->setEnabled(true);
			}
			else
			{
				d_temperature_compensation_check->setChecked(true);
				d_temperature_compensation_check->setEnabled(false);
			}
			
			d_input_widget_left->Set_Entry_Count(6);
			d_input_widget_right->Set_Entry_Count(5);
			
			d_input_widget_left->Configure_Entry_As_Combo(0);
			d_input_widget_left->Set_Entry_Label_Text(0,QStringLiteral("Stepgauge Type:"));
			d_input_widget_left->Clear_Entry_Item_Combo(0);
			d_input_widget_left->Add_Entry_Combo_Item(0,QStringLiteral("Checkmaster"),MeasureTypes::STEPGAUGE_CHECKMASTER);
			d_input_widget_left->Add_Entry_Combo_Item(0,QStringLiteral("Starrett Webber"),MeasureTypes::STEPGAUGE_STARRET);
			d_input_widget_left->Add_Entry_Combo_Item(0,QStringLiteral("Koba"),MeasureTypes::STEPGAUGE_KOBA);
			d_input_widget_left->Select_Entry_Item_Combo_Data(0,parameters.stepgauge_type);
			d_input_widget_left->Enable_Entry_Item(0,true);
			
			d_input_widget_left->Configure_Entry_As_Text(1);
			d_input_widget_left->Set_Entry_Label_Text(1,QStringLiteral("Position Name:"));
			d_input_widget_left->Set_Entry_Item_Text(1,parameters.stepgauge_b89_position_name);
			
			d_input_widget_left->Configure_Entry_As_Text(2);
			d_input_widget_left->Set_Entry_Label_Text(2,QStringLiteral("Gauge Serial Number:"));
			d_input_widget_left->Set_Entry_Item_Text(2,parameters.stepgauge_serial_number);
			
			d_input_widget_left->Configure_Entry_As_Text(3);
			d_input_widget_left->Set_Entry_Label_Text(3,QStringLiteral("Measurement Start (mm):"));
			d_input_widget_left->Set_Entry_Item_Text(3,QString("%1").arg(parameters.stepgauge_measurement_b89_start,0,'f',4));
			
			d_input_widget_left->Configure_Entry_As_Text(4);
			d_input_widget_left->Set_Entry_Label_Text(4,QStringLiteral("Nominal Block Spacing (mm):"));
			d_input_widget_left->Set_Entry_Item_Text(4,QString("%1").arg(parameters.stepgauge_b89_nominal_block_spacing,0,'f',4));
			
			d_input_widget_left->Configure_Entry_As_Text(5);
			d_input_widget_left->Set_Entry_Label_Text(5,QStringLiteral("Number of Measurements:"));
			d_input_widget_left->Set_Entry_Item_Text(5,QString("%1").arg(parameters.stepgauge_b89_measurement_count));
			
			d_input_widget_right->Configure_Entry_As_Text(0);
			d_input_widget_right->Set_Entry_Label_Text(0,QStringLiteral("X Axis Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(0,QString("%1").arg(parameters.axis_x_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(0,true);
			
			d_input_widget_right->Configure_Entry_As_Text(1);
			d_input_widget_right->Set_Entry_Label_Text(1,QStringLiteral("Y Axis Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(1,QString("%1").arg(parameters.axis_y_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(1,true);
			
			d_input_widget_right->Configure_Entry_As_Text(2);
			d_input_widget_right->Set_Entry_Label_Text(2,QStringLiteral("Z Axis Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(2,QString("%1").arg(parameters.axis_z_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(2,true);
			
			d_input_widget_right->Configure_Entry_As_Text(3);
			d_input_widget_right->Set_Entry_Label_Text(3,QStringLiteral("Stepgauge Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(3,QString("%1").arg(parameters.part_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(3,true);
			
			d_input_widget_right->Configure_Entry_As_Text(4);
			d_input_widget_right->Set_Entry_Label_Text(4,QStringLiteral("Stepgauge Expansion Coeff (um/m/C):"));
			d_input_widget_right->Set_Entry_Item_Text(4,QString("%1").arg(parameters.stepgauge_expansion_coeff,0,'f',1));

			break;
			
		case MeasureTypes::MEASURE_STEPGAUGE_SQUARE:

			this->setWindowTitle(QStringLiteral("Measure Stepgauge Squareness"));
			
			switch(parameters.stepgauge_type)
			{
				case MeasureTypes::STEPGAUGE_CHECKMASTER:
				case MeasureTypes::STEPGAUGE_NULL:
					d_configuration_pixmap = QPixmap(":/icon128/stepgauge_checkmaster_square_icon128.png");
					break;
					
				case MeasureTypes::STEPGAUGE_STARRET:
					d_configuration_pixmap = QPixmap(":/icon128/stepgauge_webber_square_icon128.png");
					break;
					
				case MeasureTypes::STEPGAUGE_KOBA:
					d_configuration_pixmap = QPixmap(":/icon128/stepgauge_koba_square_icon128.png");
					break;
			}
			
			if(d_thermal_compensation_mode == TMeasureDialog::SOFTWARE_COMPENSATION_AXIS_AND_PART)
			{
				d_temperature_compensation_check->setChecked(d_enable_temperature_compensation);
				d_temperature_compensation_check->setEnabled(true);
			}
			else
			{
				d_temperature_compensation_check->setChecked(true);
				d_temperature_compensation_check->setEnabled(false);
			}

			d_input_widget_left->Set_Entry_Count(7);
			d_input_widget_right->Set_Entry_Count(5);
			
			d_input_widget_left->Configure_Entry_As_Combo(0);
			d_input_widget_left->Set_Entry_Label_Text(0,QStringLiteral("Stepgauge Type:"));
			d_input_widget_left->Clear_Entry_Item_Combo(0);
			d_input_widget_left->Add_Entry_Combo_Item(0,QStringLiteral("Checkmaster"),MeasureTypes::STEPGAUGE_CHECKMASTER);
			d_input_widget_left->Add_Entry_Combo_Item(0,QStringLiteral("Starrett Webber"),MeasureTypes::STEPGAUGE_STARRET);
			d_input_widget_left->Add_Entry_Combo_Item(0,QStringLiteral("Koba"),MeasureTypes::STEPGAUGE_KOBA);
			d_input_widget_left->Select_Entry_Item_Combo_Data(0,parameters.stepgauge_type);
			d_input_widget_left->Enable_Entry_Item(0,true);
			
			d_input_widget_left->Configure_Entry_As_Text(1);
			d_input_widget_left->Set_Entry_Label_Text(1,QStringLiteral("Position Name:"));
			d_input_widget_left->Set_Entry_Item_Text(1,parameters.stepgauge_square_position_name);
			
			d_input_widget_left->Configure_Entry_As_Text(2);
			d_input_widget_left->Set_Entry_Label_Text(2,QStringLiteral("Gauge Serial Number:"));
			d_input_widget_left->Set_Entry_Item_Text(2,parameters.stepgauge_serial_number);
			
			d_input_widget_left->Configure_Entry_As_Text(3);
			d_input_widget_left->Set_Entry_Label_Text(3,QStringLiteral("Measurement Zero Position (mm):"));
			d_input_widget_left->Set_Entry_Item_Text(3,QString("%1").arg(parameters.stepgauge_measurement_zero_position,0,'f',4));
			
			d_input_widget_left->Configure_Entry_As_Text(4);
			d_input_widget_left->Set_Entry_Label_Text(4,QStringLiteral("Zero Step Distance (mm):"));
			d_input_widget_left->Set_Entry_Item_Text(4,QString("%1").arg(parameters.stepgauge_zero_step_spacing,0,'f',4));
			
			d_input_widget_left->Configure_Entry_As_Text(5);
			d_input_widget_left->Set_Entry_Label_Text(5,QStringLiteral("Nominal Block Spacing (mm):"));
			d_input_widget_left->Set_Entry_Item_Text(5,QString("%1").arg(parameters.stepgauge_nominal_block_spacing,0,'f',4));
			
			d_input_widget_left->Configure_Entry_As_Text(6);
			d_input_widget_left->Set_Entry_Label_Text(6,QStringLiteral("Number of Measurements:"));
			d_input_widget_left->Set_Entry_Item_Text(6,QString("%1").arg(parameters.stepgauge_square_measurement_count));
			
			d_input_widget_right->Configure_Entry_As_Text(0);
			d_input_widget_right->Set_Entry_Label_Text(0,QStringLiteral("X Axis Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(0,QString("%1").arg(parameters.axis_x_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(0,true);
			
			d_input_widget_right->Configure_Entry_As_Text(1);
			d_input_widget_right->Set_Entry_Label_Text(1,QStringLiteral("Y Axis Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(1,QString("%1").arg(parameters.axis_y_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(1,true);
			
			d_input_widget_right->Configure_Entry_As_Text(2);
			d_input_widget_right->Set_Entry_Label_Text(2,QStringLiteral("Z Axis Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(2,QString("%1").arg(parameters.axis_z_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(2,true);
			
			d_input_widget_right->Configure_Entry_As_Text(3);
			d_input_widget_right->Set_Entry_Label_Text(3,QStringLiteral("Stepgauge Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(3,QString("%1").arg(parameters.part_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(3,true);
			
			d_input_widget_right->Configure_Entry_As_Text(4);
			d_input_widget_right->Set_Entry_Label_Text(4,QStringLiteral("Stepgauge Expansion Coeff (um/m/C):"));
			d_input_widget_right->Set_Entry_Item_Text(4,QString("%1").arg(parameters.stepgauge_expansion_coeff,0,'f',1));

			break;
			
		case MeasureTypes::MEASURE_GAUGEBLOCK:
			
			this->setWindowTitle(QStringLiteral("Measure Gaugeblock 10360-2"));
			d_configuration_pixmap = QPixmap(":/icon128/gaugeblock_icon128.png");
			
			if(d_thermal_compensation_mode == TMeasureDialog::SOFTWARE_COMPENSATION_AXIS_AND_PART)
			{
				d_temperature_compensation_check->setChecked(d_enable_temperature_compensation);
				d_temperature_compensation_check->setEnabled(true);
			}
			else
			{
				d_temperature_compensation_check->setChecked(true);
				d_temperature_compensation_check->setEnabled(false);
			}

			d_input_widget_left->Set_Entry_Count(3);
			d_input_widget_right->Set_Entry_Count(5);

			d_input_widget_left->Configure_Entry_As_Text(0);
			d_input_widget_left->Set_Entry_Label_Text(0,QStringLiteral("Position Name:"));
			d_input_widget_left->Set_Entry_Item_Text(0,parameters.gaugeblock_position_name);

			d_input_widget_left->Configure_Entry_As_Text(1);
			d_input_widget_left->Set_Entry_Label_Text(1,QStringLiteral("Gauge Serial Number:"));
			d_input_widget_left->Set_Entry_Item_Text(1,parameters.gaugeblock_serial_number);
			
			d_input_widget_left->Configure_Entry_As_Text(2);
			d_input_widget_left->Set_Entry_Label_Text(2,QStringLiteral("Nominal Length:"));
			d_input_widget_left->Set_Entry_Item_Text(2,QString("%1").arg(parameters.gaugeblock_nominal_length,0,'f',4));
			
			d_input_widget_right->Configure_Entry_As_Text(0);
			d_input_widget_right->Set_Entry_Label_Text(0,QStringLiteral("X Axis Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(0,QString("%1").arg(parameters.axis_x_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(0,true);
			
			d_input_widget_right->Configure_Entry_As_Text(1);
			d_input_widget_right->Set_Entry_Label_Text(1,QStringLiteral("Y Axis Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(1,QString("%1").arg(parameters.axis_y_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(1,true);
			
			d_input_widget_right->Configure_Entry_As_Text(2);
			d_input_widget_right->Set_Entry_Label_Text(2,QStringLiteral("Z Axis Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(2,QString("%1").arg(parameters.axis_z_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(2,true);
			
			d_input_widget_right->Configure_Entry_As_Text(3);
			d_input_widget_right->Set_Entry_Label_Text(3,QStringLiteral("Gauge Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(3,QString("%1").arg(parameters.part_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(3,true);

			d_input_widget_right->Configure_Entry_As_Text(4);
			d_input_widget_right->Set_Entry_Label_Text(4,QStringLiteral("Gauge Expansion Coeff (um/m/C):"));
			d_input_widget_right->Set_Entry_Item_Text(4,QString("%1").arg(parameters.gaugeblock_expansion_coeff,0,'f',1));

			break;
			
		case MeasureTypes::MEASURE_POINT_REPEAT:
			
			this->setWindowTitle(QStringLiteral("Measure Point Repeat"));
			d_configuration_pixmap = QPixmap(":/icon128/point_repeat_icon128.png");
			
			if(d_thermal_compensation_mode == TMeasureDialog::SOFTWARE_COMPENSATION_AXIS_AND_PART)
			{
				d_temperature_compensation_check->setChecked(false);
				d_temperature_compensation_check->setEnabled(false);
			}
			else
			{
				d_temperature_compensation_check->setChecked(true);
				d_temperature_compensation_check->setEnabled(false);
			}
			
			d_input_widget_left->Set_Entry_Count(1);
			d_input_widget_right->Set_Entry_Count(2);
			
			d_input_widget_left->Configure_Entry_As_Text(0);
			d_input_widget_left->Set_Entry_Label_Text(0,QStringLiteral("Description:"));
			d_input_widget_left->Set_Entry_Item_Text(0,parameters.point_repeat_description);
			
			d_input_widget_right->Configure_Entry_As_Text(0);
			d_input_widget_right->Set_Entry_Label_Text(0,QStringLiteral("Move Distance:"));
			d_input_widget_right->Set_Entry_Item_Text(0,QString("%1").arg(parameters.point_repeat_move_distance,0,'f',4));

			d_input_widget_right->Configure_Entry_As_Text(1);
			d_input_widget_right->Set_Entry_Label_Text(1,QStringLiteral("Number of Measurements:"));
			d_input_widget_right->Set_Entry_Item_Text(1,QString("%1").arg(parameters.point_repeat_measurement_count));
			
			break;
			
		case MeasureTypes::MEASURE_SPHERE_REPEAT:

			this->setWindowTitle(QStringLiteral("Measure Sphere Repeat Rpt"));
			d_configuration_pixmap = QPixmap(":/icon128/sphere_repeat_icon128.png");
			
			if(d_thermal_compensation_mode == TMeasureDialog::SOFTWARE_COMPENSATION_AXIS_AND_PART)
			{
				d_temperature_compensation_check->setChecked(false);
				d_temperature_compensation_check->setEnabled(false);
			}
			else
			{
				d_temperature_compensation_check->setChecked(true);
				d_temperature_compensation_check->setEnabled(false);
			}

			d_input_widget_left->Set_Entry_Count(1);
			d_input_widget_right->Set_Entry_Count(1);
			
			d_input_widget_left->Configure_Entry_As_Text(0);
			d_input_widget_left->Set_Entry_Label_Text(0,QStringLiteral("Nominal Sphere Diameter:"));
			d_input_widget_left->Set_Entry_Item_Text(0,QString("%1").arg(parameters.sphere_nominal_diameter,0,'f',4));
					
			d_input_widget_right->Configure_Entry_As_Text(0);
			d_input_widget_right->Set_Entry_Label_Text(0,QStringLiteral("Number of Measurements:"));
			d_input_widget_right->Set_Entry_Item_Text(0,QString("%1").arg(parameters.sphere_repeat_measurement_count));

			break;
			
		case MeasureTypes::MEASURE_SPHERE_PFTU:
			
			this->setWindowTitle(QStringLiteral("Measure Sphere Pftu 10360-5"));
			d_configuration_pixmap = QPixmap(":/icon128/pftu_icon128.png");

			if(d_thermal_compensation_mode == TMeasureDialog::SOFTWARE_COMPENSATION_AXIS_AND_PART)
			{
				d_temperature_compensation_check->setChecked(false);
				d_temperature_compensation_check->setEnabled(false);
			}
			else
			{
				d_temperature_compensation_check->setChecked(true);
				d_temperature_compensation_check->setEnabled(false);
			}

			d_input_widget_left->Set_Entry_Count(1);
			d_input_widget_right->Set_Entry_Count(0);
			
			d_input_widget_left->Configure_Entry_As_Text(0);
			d_input_widget_left->Set_Entry_Label_Text(0,QStringLiteral("Nominal Sphere Diameter:"));
			d_input_widget_left->Set_Entry_Item_Text(0,QString("%1").arg(parameters.sphere_nominal_diameter,0,'f',4));

			break;
			
		case MeasureTypes::MEASURE_BALLBAR_B89:

			this->setWindowTitle(QStringLiteral("Measure B89.4.1 Ballbar"));
			d_configuration_pixmap = QPixmap(":/icon128/ballbar_icon128.png");

			if(d_thermal_compensation_mode == TMeasureDialog::SOFTWARE_COMPENSATION_AXIS_AND_PART)
			{
				d_temperature_compensation_check->setChecked(d_enable_temperature_compensation);
				d_temperature_compensation_check->setEnabled(true);
			}
			else
			{
				d_temperature_compensation_check->setChecked(true);
				d_temperature_compensation_check->setEnabled(false);
			}

			d_input_widget_left->Set_Entry_Count(3);
			d_input_widget_right->Set_Entry_Count(5);

			d_input_widget_left->Configure_Entry_As_Text(0);
			d_input_widget_left->Set_Entry_Label_Text(0,QStringLiteral("Position Name:"));
			d_input_widget_left->Set_Entry_Item_Text(0,parameters.ballbar_position_name);
			
			d_input_widget_left->Configure_Entry_As_Text(1);
			d_input_widget_left->Set_Entry_Label_Text(1,QStringLiteral("Ballbar Sphere Diameter:"));
			d_input_widget_left->Set_Entry_Item_Text(1,QString("%1").arg(parameters.ballbar_sphere_diameter,0,'f',4));
			
			d_input_widget_left->Configure_Entry_As_Text(2);
			d_input_widget_left->Set_Entry_Label_Text(2,QStringLiteral("Ballbar Measurement Count:"));
			d_input_widget_left->Set_Entry_Item_Text(2,QString("%1").arg(parameters.ballbar_measurement_count));

			d_input_widget_right->Configure_Entry_As_Text(0);
			d_input_widget_right->Set_Entry_Label_Text(0,QStringLiteral("X Axis Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(0,QString("%1").arg(parameters.axis_x_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(0,true);

			d_input_widget_right->Configure_Entry_As_Text(1);
			d_input_widget_right->Set_Entry_Label_Text(1,QStringLiteral("Y Axis Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(1,QString("%1").arg(parameters.axis_y_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(1,true);

			d_input_widget_right->Configure_Entry_As_Text(2);
			d_input_widget_right->Set_Entry_Label_Text(2,QStringLiteral("Z Axis Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(2,QString("%1").arg(parameters.axis_z_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(2,true);

			d_input_widget_right->Configure_Entry_As_Text(3);
			d_input_widget_right->Set_Entry_Label_Text(3,QStringLiteral("Ballbar Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(3,QString("%1").arg(parameters.part_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(3,true);
			
			d_input_widget_right->Configure_Entry_As_Text(4);
			d_input_widget_right->Set_Entry_Label_Text(4,QStringLiteral("Ballbar Expansion Coeff (um/m/C):"));
			d_input_widget_right->Set_Entry_Item_Text(4,QString("%1").arg(parameters.ballbar_expansion_coeff,0,'f',1));

			break;
			
		case MeasureTypes::MEASURE_BALLBAR_10360:

			this->setWindowTitle(QStringLiteral("Measure 10360 Ballbar"));
			d_configuration_pixmap = QPixmap(":/icon128/ballbar10360_icon128.png");

			if(d_thermal_compensation_mode == TMeasureDialog::SOFTWARE_COMPENSATION_AXIS_AND_PART)
			{
				d_temperature_compensation_check->setChecked(d_enable_temperature_compensation);
				d_temperature_compensation_check->setEnabled(true);
			}
			else
			{
				d_temperature_compensation_check->setChecked(true);
				d_temperature_compensation_check->setEnabled(false);
			}

			d_input_widget_left->Set_Entry_Count(5);
			d_input_widget_right->Set_Entry_Count(5);

			d_input_widget_left->Configure_Entry_As_Text(0);
			d_input_widget_left->Set_Entry_Label_Text(0,QStringLiteral("Position Name:"));
			d_input_widget_left->Set_Entry_Item_Text(0,parameters.ballbar_10360_position_name);
			
			d_input_widget_left->Configure_Entry_As_Text(1);
			d_input_widget_left->Set_Entry_Label_Text(1,QStringLiteral("Ballbar Serial Number:"));
			d_input_widget_left->Set_Entry_Item_Text(1,parameters.ballbar_10360_serial_number);
			d_input_widget_left->Enable_Entry_Item(1,true);

			d_input_widget_left->Configure_Entry_As_Text(2);
			d_input_widget_left->Set_Entry_Label_Text(2,QStringLiteral("Ballbar Sphere Diameter:"));
			d_input_widget_left->Set_Entry_Item_Text(2,QString("%1").arg(parameters.ballbar_10360_sphere_diameter,0,'f',4));
			
			d_input_widget_left->Configure_Entry_As_Text(3);
			d_input_widget_left->Set_Entry_Label_Text(3,QStringLiteral("Sphere Zero Offset:"));
			d_input_widget_left->Set_Entry_Item_Text(3,QString("%1").arg(parameters.ballbar_10360_sphere0_offset));

			d_input_widget_left->Configure_Entry_As_Text(4);
			d_input_widget_left->Set_Entry_Label_Text(4,QStringLiteral("Ballbar Sphere Spacing:"));
			d_input_widget_left->Set_Entry_Item_Text(4,QString("%1").arg(parameters.ballbar_10360_sphere_spacing,0,'f',4));

			d_input_widget_right->Configure_Entry_As_Text(0);
			d_input_widget_right->Set_Entry_Label_Text(0,QStringLiteral("X Axis Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(0,QString("%1").arg(parameters.axis_x_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(0,true);

			d_input_widget_right->Configure_Entry_As_Text(1);
			d_input_widget_right->Set_Entry_Label_Text(1,QStringLiteral("Y Axis Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(1,QString("%1").arg(parameters.axis_y_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(1,true);

			d_input_widget_right->Configure_Entry_As_Text(2);
			d_input_widget_right->Set_Entry_Label_Text(2,QStringLiteral("Z Axis Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(2,QString("%1").arg(parameters.axis_z_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(2,true);

			d_input_widget_right->Configure_Entry_As_Text(3);
			d_input_widget_right->Set_Entry_Label_Text(3,QStringLiteral("Ballbar Temperature (C):"));
			d_input_widget_right->Set_Entry_Item_Text(3,QString("%1").arg(parameters.part_temperature,0,'f',2));
			d_input_widget_right->Enable_Entry_Item(3,true);
			
			d_input_widget_right->Configure_Entry_As_Text(4);
			d_input_widget_right->Set_Entry_Label_Text(4,QStringLiteral("Ballbar Expansion Coeff (um/m/C):"));
			d_input_widget_right->Set_Entry_Item_Text(4,QString("%1").arg(parameters.ballbar_10360_expansion_coeff,0,'f',1));

			break;
			
		case MeasureTypes::MEASURE_RINGGAUGE:
			
			this->setWindowTitle(QStringLiteral("Measure Ring Gauge"));
			d_configuration_pixmap = QPixmap(":/icon128/ring_icon128.png");
			
			if(d_thermal_compensation_mode == TMeasureDialog::SOFTWARE_COMPENSATION_AXIS_AND_PART)
			{
				d_temperature_compensation_check->setChecked(false);
				d_temperature_compensation_check->setEnabled(false);
			}
			else
			{
				d_temperature_compensation_check->setChecked(true);
				d_temperature_compensation_check->setEnabled(false);
			}

			d_input_widget_left->Set_Entry_Count(1);
			d_input_widget_right->Set_Entry_Count(1);
		
			d_input_widget_left->Configure_Entry_As_Text(0);
			d_input_widget_left->Set_Entry_Label_Text(0,QStringLiteral("Nominal Diameter:"));
			d_input_widget_left->Set_Entry_Item_Text(0,QString("%1").arg(parameters.ringgauge_nominal_diameter,0,'f',4));
			
			d_input_widget_right->Configure_Entry_As_Text(0);
			d_input_widget_right->Set_Entry_Label_Text(0,QStringLiteral("Number of Points:"));
			d_input_widget_right->Set_Entry_Item_Text(0,QString("%1").arg(parameters.ringgauge_point_count));
			
			break;
			
		case MeasureTypes::MEASURE_PINGAUGE:
			
			this->setWindowTitle(QStringLiteral("Measure Pin Gauge"));
			d_configuration_pixmap = QPixmap(":/icon128/pin_icon128.png");
			
			if(d_thermal_compensation_mode == TMeasureDialog::SOFTWARE_COMPENSATION_AXIS_AND_PART)
			{
				d_temperature_compensation_check->setChecked(false);
				d_temperature_compensation_check->setEnabled(false);
			}
			else
			{
				d_temperature_compensation_check->setChecked(true);
				d_temperature_compensation_check->setEnabled(false);
			}

			d_input_widget_left->Set_Entry_Count(1);
			d_input_widget_right->Set_Entry_Count(1);
			
			d_input_widget_left->Configure_Entry_As_Text(0);
			d_input_widget_left->Set_Entry_Label_Text(0,QStringLiteral("Nominal Diameter:"));
			d_input_widget_left->Set_Entry_Item_Text(0,QString("%1").arg(parameters.pingauge_nominal_diameter,0,'f',4));
			
			d_input_widget_right->Configure_Entry_As_Text(0);
			d_input_widget_right->Set_Entry_Label_Text(0,QStringLiteral("Number of Points:"));
			d_input_widget_right->Set_Entry_Item_Text(0,QString("%1").arg(parameters.pingauge_point_count));
			
			break;
			
		case MeasureTypes::MEASURE_ROLL_OFFSET_TOOL:

			this->setWindowTitle(QStringLiteral("Measure Roll Using Offset Tools"));
			d_configuration_pixmap = QPixmap(":/icon128/roll_icon128.png");
			
			d_active_tool_combo->setEnabled(false);

			if(d_thermal_compensation_mode == TMeasureDialog::SOFTWARE_COMPENSATION_AXIS_AND_PART)
			{
				d_temperature_compensation_check->setChecked(false);
				d_temperature_compensation_check->setEnabled(false);
			}
			else
			{
				d_temperature_compensation_check->setChecked(true);
				d_temperature_compensation_check->setEnabled(false);
			}

			d_input_widget_left->Set_Entry_Count(2);
			d_input_widget_right->Set_Entry_Count(4);

			d_input_widget_left->Configure_Entry_As_Combo(0);
			d_input_widget_left->Set_Entry_Label_Text(0,QStringLiteral("Tool 1:"));
			d_input_widget_left->Clear_Entry_Item_Combo(0);
			d_input_widget_left->Add_Entry_Combo_Items(0,tool_list);
			d_input_widget_left->Enable_Entry_Item(0,true);
			d_input_widget_left->Select_Entry_Item_Combo_Text(0,parameters.roll_tool1_name);

			d_input_widget_left->Configure_Entry_As_Combo(1);
			d_input_widget_left->Set_Entry_Label_Text(1,QStringLiteral("Tool 2:"));
			d_input_widget_left->Clear_Entry_Item_Combo(1);
			d_input_widget_left->Add_Entry_Combo_Items(1,tool_list);
			d_input_widget_left->Enable_Entry_Item(1,true);
			d_input_widget_left->Select_Entry_Item_Combo_Text(1,parameters.roll_tool2_name);
			
			d_input_widget_right->Configure_Entry_As_Text(0);
			d_input_widget_right->Set_Entry_Label_Text(0,QStringLiteral("Nominal Sphere Diameter:"));
			d_input_widget_right->Set_Entry_Item_Text(0,QString("%1").arg(parameters.sphere_nominal_diameter,0,'f',4));
						
			d_input_widget_right->Configure_Entry_As_Text(1);
			d_input_widget_right->Set_Entry_Label_Text(1,QStringLiteral("Stem Vector I:"));
			d_input_widget_right->Set_Entry_Item_Text(1,QString("%1").arg(parameters.sphere_stem_vector_i,0,'f',6));
			
			d_input_widget_right->Configure_Entry_As_Text(2);
			d_input_widget_right->Set_Entry_Label_Text(2,QStringLiteral("Stem Vector J:"));
			d_input_widget_right->Set_Entry_Item_Text(2,QString("%1").arg(parameters.sphere_stem_vector_j,0,'f',6));
			
			d_input_widget_right->Configure_Entry_As_Text(3);
			d_input_widget_right->Set_Entry_Label_Text(3,QStringLiteral("Stem Vector K:"));
			d_input_widget_right->Set_Entry_Item_Text(3,QString("%1").arg(parameters.sphere_stem_vector_k,0,'f',6));

			break;

		case MeasureTypes::CALIBRATE_TOOLS:
			
			this->setWindowTitle(QStringLiteral("Calibrate Tools"));
			d_configuration_pixmap = QPixmap(":/icon128/calibrate_icon128.png");

			d_active_tool_combo->setEnabled(false);
			
			if(d_thermal_compensation_mode == TMeasureDialog::SOFTWARE_COMPENSATION_AXIS_AND_PART)
			{
				d_temperature_compensation_check->setChecked(false);
				d_temperature_compensation_check->setEnabled(false);
			}
			else
			{
				d_temperature_compensation_check->setChecked(true);
				d_temperature_compensation_check->setEnabled(false);
			}

			d_input_widget_left->Set_Entry_Count(1);
			d_input_widget_right->Set_Entry_Count(3);
			
			d_input_widget_left->Configure_Entry_As_Text(0);
			d_input_widget_left->Set_Entry_Label_Text(0,QStringLiteral("Nominal Sphere Diameter:"));
			d_input_widget_left->Set_Entry_Item_Text(0,QString("%1").arg(parameters.sphere_nominal_diameter,0,'f',4));
			
			d_input_widget_right->Configure_Entry_As_Text(0);
			d_input_widget_right->Set_Entry_Label_Text(0,QStringLiteral("Stem Vector I:"));
			d_input_widget_right->Set_Entry_Item_Text(0,QString("%1").arg(parameters.sphere_stem_vector_i,0,'f',6));
			
			d_input_widget_right->Configure_Entry_As_Text(1);
			d_input_widget_right->Set_Entry_Label_Text(1,QStringLiteral("Stem Vector J:"));
			d_input_widget_right->Set_Entry_Item_Text(1,QString("%1").arg(parameters.sphere_stem_vector_j,0,'f',6));
			
			d_input_widget_right->Configure_Entry_As_Text(2);
			d_input_widget_right->Set_Entry_Label_Text(2,QStringLiteral("Stem Vector K:"));
			d_input_widget_right->Set_Entry_Item_Text(2,QString("%1").arg(parameters.sphere_stem_vector_k,0,'f',6));
			
			break;

	}
	
	this->Set_Pixmap(d_configuration_pixmap,false);

	d_temperature_compensation_check->blockSignals(prev_state);
	
	if(d_temperature_compensation_check->isChecked())
	{
		emit Refresh_Temperatures();
	}

	this->Set_State(TMeasureDialog::STATE_STOPPED);
}

void TMeasureDialog::Reset_Tool_Selection(void)
{
	bool								prev_state;
	
	if(d_active_tool_combo->count())
	{
		prev_state = d_active_tool_combo->blockSignals(true);
		d_active_tool_combo->setCurrentIndex(0);	// default to '-- select tool --'
		d_active_tool_combo->blockSignals(prev_state);
	}
}

void TMeasureDialog::Set_Artifact_Settings(
	const TSettingsWidget::TArtifactSettings &artifact_settings)
{
	d_measure_routines.Set_Artifact_Settings(artifact_settings);
}

void TMeasureDialog::Set_Machine_Settings(
	const TSettingsWidget::TMachineSettings &machine_settings)
{
	d_machine_settings = machine_settings;
	
	d_measure_routines.Set_Axis_Expansion_Coefficients(machine_settings.expansion_coeff_x,machine_settings.expansion_coeff_y,machine_settings.expansion_coeff_z);
}

void TMeasureDialog::Set_Thermal_Settings(
	const TMeasureDialog::TThermalCompensationMode mode)
{
	d_thermal_compensation_mode = mode;
}

bool TMeasureDialog::Add_Measured_Point(
	const TVector3						&xyz,
	const TVector3						&ijk)
{
	TVector3							xyz_comp;
	TCommandPoint						*point_feature(0);
	TVector3							vec;
	int									actual_count;
	int									nominal_count;
	
	if(d_active_feature)
	{
		actual_count = d_active_feature->Actual_Point_Count();
		nominal_count = d_point_count_widget->Nominal_Point_Count();
		
		if (actual_count == nominal_count)
		{
			   // use this hit as a trigger to jump to next feature
			this->Update_Active_Feature();
			return false;
		}

		if(d_temperature_compensation_check->isChecked())
		{
			xyz_comp = this->Add_Temperature_Compensation(xyz);
		}
		else
		{
			xyz_comp = xyz;
		}
		
		if(actual_count < nominal_count)
		{
			if(!d_measure_routines.Manual_Mode())
			{
				if(d_active_feature->Type() == TCommand::COMMAND_MEASURE_POINT)
				{					
					point_feature = reinterpret_cast<TCommandPoint*>(d_active_feature);

					vec = d_measure_routines.Refsys_AxisToWorld(point_feature->Nominal_Touch_Vector());
					
					d_active_feature->Add_Measured_Point(xyz_comp,vec); 	// controller IJK vector not always reliable as a compensation vector.
				}
				else
				{
					d_active_feature->Add_Measured_Point(xyz_comp,ijk);
				}
			}
			else
			{
				d_active_feature->Add_Measured_Point(xyz_comp,ijk);
			}
			
			++actual_count;
		}
		
		d_point_count_widget->Set_Actual_Point_Count(actual_count);
		
		if(actual_count == nominal_count)
		{
			d_next_button->setEnabled(d_measure_routines.Manual_Mode());
		}
		else
		{
			d_next_button->setEnabled(false);
		}
		
		d_erase_button->setEnabled(d_measure_routines.Manual_Mode());
		
		if(!d_measure_routines.Manual_Mode())
		{
			this->Update_Active_Feature();
		}
	}
	
	return true;
}

void TMeasureDialog::Update_Position(
	const TVector3						&xyz)
{
	TVector3							xyz_comp;
	
	if(d_temperature_compensation_check->isChecked())
	{
		xyz_comp = this->Add_Temperature_Compensation(xyz);
	}
	else
	{
		xyz_comp = xyz;
	}
	
	switch(d_display_coordinate_system)
	{
		case MeasureTypes::COORDINATE_SYSTEM_MACHINE:
			
			d_coordinate_display_widget->Set_XYZ(xyz_comp);
			
			break;
			
		case MeasureTypes::COORDINATE_SYSTEM_PART:
			
			d_coordinate_display_widget->Set_XYZ(d_measure_routines.Refsys_FromWorld(xyz_comp));
			
			break;
			
		case MeasureTypes::COORDINATE_SYSTEM_PART_ORIGIN_ONLY:
			
			d_coordinate_display_widget->Set_XYZ(xyz_comp - d_measure_routines.Refsys_Origin());
			
			break;
	}
}

void TMeasureDialog::Set_Temperature_X(
	const double						&value)
{
	TMeasureRoutines::TParameters		parameters;
	
	if(value < 10.0 || value > 40.0)
	{
		this->Add_Log_Text(QString("WAR:  X Axis temperature out of range and ignored.  Temp: %1 C").arg(value,0,'f',2));
		return;
	}
	
	parameters = d_measure_routines.Measurement_Parameters();

	parameters.axis_x_temperature = value;
	d_tempcomp_information_dialog->Set_Temperature_X(value);
	
	if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_10360)
	{
		d_input_widget_right->Set_Entry_Item_Text(0,QString("%1").arg(parameters.axis_x_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(0,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_B89)
	{
		d_input_widget_right->Set_Entry_Item_Text(0,QString("%1").arg(parameters.axis_x_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(0,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_SQUARE)
	{
		d_input_widget_right->Set_Entry_Item_Text(0,QString("%1").arg(parameters.axis_x_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(0,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_BALLBAR_B89)
	{
		d_input_widget_right->Set_Entry_Item_Text(0,QString("%1").arg(parameters.axis_x_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(0,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_BALLBAR_10360)
	{
		d_input_widget_right->Set_Entry_Item_Text(0,QString("%1").arg(parameters.axis_x_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(0,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_GAUGEBLOCK)
	{
		d_input_widget_right->Set_Entry_Item_Text(0,QString("%1").arg(parameters.axis_x_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(0,false);
	}
	
	d_measure_routines.Set_Measurement_Parameters(parameters);

}

void TMeasureDialog::Set_Temperature_Y(
	const double						&value)
{
	TMeasureRoutines::TParameters		parameters;

	if(value < 10.0 || value > 40.0)
	{
		this->Add_Log_Text(QString("WAR:  Y Axis temperature out of range and ignored.  Temp: %1 C").arg(value,0,'f',2));
		return;
	}
		
	parameters = d_measure_routines.Measurement_Parameters();

	parameters.axis_y_temperature = value;
	d_tempcomp_information_dialog->Set_Temperature_Y(value);

	if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_10360)
	{
		d_input_widget_right->Set_Entry_Item_Text(1,QString("%1").arg(parameters.axis_y_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(1,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_B89)
	{
		d_input_widget_right->Set_Entry_Item_Text(1,QString("%1").arg(parameters.axis_y_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(1,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_SQUARE)
	{
		d_input_widget_right->Set_Entry_Item_Text(1,QString("%1").arg(parameters.axis_y_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(1,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_BALLBAR_B89)
	{
		d_input_widget_right->Set_Entry_Item_Text(1,QString("%1").arg(parameters.axis_y_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(1,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_BALLBAR_10360)
	{
		d_input_widget_right->Set_Entry_Item_Text(1,QString("%1").arg(parameters.axis_y_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(1,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_GAUGEBLOCK)
	{
		d_input_widget_right->Set_Entry_Item_Text(1,QString("%1").arg(parameters.axis_y_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(1,false);
	}
	
	d_measure_routines.Set_Measurement_Parameters(parameters);

}

void TMeasureDialog::Set_Temperature_Z(
	const double						&value)
{
	TMeasureRoutines::TParameters		parameters;

	if(value < 10.0 || value > 40.0)
	{
		this->Add_Log_Text(QString("WAR:  Z Axis temperature out of range and ignored.  Temp: %1 C").arg(value,0,'f',2));
		return;
	}
		
	parameters = d_measure_routines.Measurement_Parameters();

	parameters.axis_z_temperature = value;
	d_tempcomp_information_dialog->Set_Temperature_Z(value);
	
	if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_10360)
	{
		d_input_widget_right->Set_Entry_Item_Text(2,QString("%1").arg(parameters.axis_z_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(2,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_B89)
	{
		d_input_widget_right->Set_Entry_Item_Text(2,QString("%1").arg(parameters.axis_z_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(2,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_SQUARE)
	{
		d_input_widget_right->Set_Entry_Item_Text(2,QString("%1").arg(parameters.axis_z_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(2,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_BALLBAR_B89)
	{
		d_input_widget_right->Set_Entry_Item_Text(2,QString("%1").arg(parameters.axis_z_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(2,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_BALLBAR_10360)
	{
		d_input_widget_right->Set_Entry_Item_Text(2,QString("%1").arg(parameters.axis_z_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(2,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_GAUGEBLOCK)
	{
		d_input_widget_right->Set_Entry_Item_Text(2,QString("%1").arg(parameters.axis_z_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(2,false);
	}
	
	d_measure_routines.Set_Measurement_Parameters(parameters);
}

void TMeasureDialog::Set_Temperature_Part(
	const double						&value)
{
	TMeasureRoutines::TParameters		parameters;
	
	if(value < 10.0 || value > 40.0)
	{
		this->Add_Log_Text(QString("WAR:  Part temperature out of range and ignored.  Temp: %1 C").arg(value,0,'f',2));
		return;
	}
		
	parameters = d_measure_routines.Measurement_Parameters();

	parameters.part_temperature = value;
	d_tempcomp_information_dialog->Set_Temperature_Part(value);
	
	if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_10360)
	{
		d_input_widget_right->Set_Entry_Item_Text(3,QString("%1").arg(parameters.part_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(3,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_B89)
	{
		d_input_widget_right->Set_Entry_Item_Text(3,QString("%1").arg(parameters.part_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(3,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_SQUARE)
	{
		d_input_widget_right->Set_Entry_Item_Text(3,QString("%1").arg(parameters.part_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(3,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_BALLBAR_B89)
	{
		d_input_widget_right->Set_Entry_Item_Text(3,QString("%1").arg(parameters.part_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(3,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_BALLBAR_10360)
	{
		d_input_widget_right->Set_Entry_Item_Text(3,QString("%1").arg(parameters.part_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(3,false);
	}
	else if(d_configuration == MeasureTypes::MEASURE_GAUGEBLOCK)
	{
		d_input_widget_right->Set_Entry_Item_Text(3,QString("%1").arg(parameters.part_temperature,0,'f',2));
		d_input_widget_right->Enable_Entry_Item(3,false);
	}
	
	d_measure_routines.Set_Measurement_Parameters(parameters);
}

void TMeasureDialog::Start(void)
{
	TVector3							vec;
	double								part_expansion_coefficient(11.5);
	int									index;
	QRect								rect;
	MeasureTypes::TAlignmentOptions		alignment_option;
	TMeasureRoutines::TParameters		parameters;
	
	index = d_active_tool_combo->currentIndex();
	alignment_option = static_cast<MeasureTypes::TAlignmentOptions>(d_alignment_options_combo->currentData(Qt::UserRole).toInt());
	
	if(index < 1 && 
		d_configuration != MeasureTypes::CALIBRATE_TOOLS &&
		d_configuration != MeasureTypes::MEASURE_ROLL_OFFSET_TOOL)
	{
		this->Add_Log_Text(QStringLiteral("ERR:  Tool must be set prior to measurement!"));
		d_measure_routines.Reset_Current_Sequence_Function();	// cannot measure this
		
		return;
	}
		
	this->Set_State(TMeasureDialog::STATE_RUNNING);
	
	d_measure_routines.Delete_Sequence_Commands();
	
	d_active_feature = 0;
	d_erase_button->setEnabled(false);
	d_next_button->setEnabled(false);
	d_measure_routines.Set_Manual_Mode(true);
	d_waiting_driver_sync = false;
	
	this->Clear_Log();
	
	parameters = d_measure_routines.Measurement_Parameters();

	this->Write_User_Inputs(&parameters);
	
	switch(d_configuration)
	{
		case MeasureTypes::MEASURE_NULL:
			d_measure_routines.Reset_Current_Sequence_Function();
			break;
			
		case MeasureTypes::MEASURE_STEPGAUGE_10360:
			
			d_display_coordinate_system = MeasureTypes::COORDINATE_SYSTEM_PART;
			
			if(parameters.stepgauge_zero_step_spacing > parameters.stepgauge_nominal_block_spacing)
			{
				this->Add_Log_Text(QStringLiteral("ERR:  Zero step distance cannot be greater than nominal block spacing!"));

				d_measure_routines.Reset_Current_Sequence_Function();	// cannot measure this
			}
			
			if(parameters.stepgauge_type != d_last_stepgauge_type &&
			   (alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL || alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC))
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);
				
				this->Add_Log_Text(QStringLiteral("WAR:  Manual alignment must be performed."));
			}
			else if(d_last_configuration != d_configuration &&
					d_last_configuration != MeasureTypes::MEASURE_STEPGAUGE_B89 &&
					d_last_configuration != MeasureTypes::MEASURE_STEPGAUGE_SQUARE &&
					(alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL || alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC))
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);
				
				this->Add_Log_Text(QStringLiteral("WAR:  Manual alignment must be performed."));
			}
			
			switch(alignment_option)
			{
				case MeasureTypes::ALIGNMENT_FULL:
					
					d_measure_routines.Refsys_Clear_Alignment();
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_STEPGAUGE_MANUAL);

					break;
					
				case MeasureTypes::ALIGNMENT_SKIP_MANUAL:
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_STEPGAUGE_DCC);
					break;
					
				case MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC:
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_MEASURE_STEPGAUGE_10360);
					break;
					
				case MeasureTypes::ALIGNMENT_ONLY:
					
					d_measure_routines.Refsys_Clear_Alignment();
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_STEPGAUGE_MANUAL);
					
					break;
			}

			
			if(d_temperature_compensation_check->isChecked())
			{
				if(parameters.axis_x_temperature > 40.0 || parameters.axis_x_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  X axis temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
				
				if(parameters.axis_y_temperature > 40.0 || parameters.axis_y_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  Y axis temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
				
				if(parameters.axis_z_temperature > 40.0 || parameters.axis_z_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  Z axis temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
				
				if(parameters.part_temperature > 40.0 || parameters.part_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  Part temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
			}
			
			part_expansion_coefficient = parameters.stepgauge_expansion_coeff;

			break;
			
		case MeasureTypes::MEASURE_STEPGAUGE_B89:
			
			d_display_coordinate_system = MeasureTypes::COORDINATE_SYSTEM_PART;
			
			if(parameters.stepgauge_type != d_last_stepgauge_type &&
			   (alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL || alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC))
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);
				
				this->Add_Log_Text(QStringLiteral("WAR:  Manual alignment must be performed."));
			}
			else if(d_last_configuration != d_configuration &&
					d_last_configuration != MeasureTypes::MEASURE_STEPGAUGE_10360 &&
					d_last_configuration != MeasureTypes::MEASURE_STEPGAUGE_SQUARE &&
					(alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL || alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC))
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);
				
				this->Add_Log_Text(QStringLiteral("WAR:  Manual alignment must be performed."));
			}
			
			switch(alignment_option)
			{
				case MeasureTypes::ALIGNMENT_FULL:
					
					d_measure_routines.Refsys_Clear_Alignment();
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_STEPGAUGE_MANUAL);
					
					break;
					
				case MeasureTypes::ALIGNMENT_SKIP_MANUAL:
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_STEPGAUGE_DCC);
					break;
					
				case MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC:
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_MEASURE_STEPGAUGE_B89);
					break;
					
				case MeasureTypes::ALIGNMENT_ONLY:
					
					d_measure_routines.Refsys_Clear_Alignment();
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_STEPGAUGE_MANUAL);
					
					break;
			}
			
			
			if(d_temperature_compensation_check->isChecked())
			{
				if(parameters.axis_x_temperature > 40.0 || parameters.axis_x_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  X axis temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
				
				if(parameters.axis_y_temperature > 40.0 || parameters.axis_y_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  Y axis temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
				
				if(parameters.axis_z_temperature > 40.0 || parameters.axis_z_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  Z axis temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
				
				if(parameters.part_temperature > 40.0 || parameters.part_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  Part temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
			}
			
			part_expansion_coefficient = parameters.stepgauge_expansion_coeff;
			
			break;
			
		case MeasureTypes::MEASURE_STEPGAUGE_SQUARE:
			
			d_display_coordinate_system = MeasureTypes::COORDINATE_SYSTEM_PART;
			
			if(parameters.stepgauge_type != d_last_stepgauge_type  &&
			   (alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL || alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC))
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);

				this->Add_Log_Text(QStringLiteral("WAR:  Manual alignment must be performed."));
			}
			else if(d_last_configuration != d_configuration &&
					d_last_configuration != MeasureTypes::MEASURE_STEPGAUGE_10360 &&
					d_last_configuration != MeasureTypes::MEASURE_STEPGAUGE_B89 &&
					(alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL || alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC))
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);

				this->Add_Log_Text(QStringLiteral("WAR:  Manual alignment must be performed."));
			}
			
			switch(alignment_option)
			{
				case MeasureTypes::ALIGNMENT_FULL:
					
					d_measure_routines.Refsys_Clear_Alignment();
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_STEPGAUGE_MANUAL);

					break;
					
				case MeasureTypes::ALIGNMENT_SKIP_MANUAL:
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_STEPGAUGE_DCC);
					break;
					
				case MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC:
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_MEASURE_STEPGAUGE_SQUARE);
					break;
					
				case MeasureTypes::ALIGNMENT_ONLY:
					
					d_measure_routines.Refsys_Clear_Alignment();
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_STEPGAUGE_MANUAL);
					
					break;
			}
			
			if(d_temperature_compensation_check->isChecked())
			{
				if(parameters.axis_x_temperature > 40.0 || parameters.axis_x_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  X axis temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
				
				if(parameters.axis_y_temperature > 40.0 || parameters.axis_y_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  Y axis temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
				
				if(parameters.axis_z_temperature > 40.0 || parameters.axis_z_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  Z axis temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
				
				if(parameters.part_temperature > 40.0 || parameters.part_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  Part temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
			}
			
			part_expansion_coefficient = parameters.stepgauge_expansion_coeff;

			break;
	
		case MeasureTypes::MEASURE_GAUGEBLOCK:
			
			d_display_coordinate_system = MeasureTypes::COORDINATE_SYSTEM_PART;
			
			if(d_last_configuration != d_configuration &&
			   alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL)
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);

				this->Add_Log_Text(QStringLiteral("WAR:  Manual alignment must be performed."));
			}
			else if(alignment_option == MeasureTypes::ALIGNMENT_ONLY || alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC)
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);
				
				this->Add_Log_Text(QStringLiteral("WAR:  Alignment option not available."));
			}

			switch(alignment_option)
			{
				case MeasureTypes::ALIGNMENT_FULL:
					
					d_measure_routines.Refsys_Clear_Alignment();
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_GAUGEBLOCK_MANUAL);
					
					break;
					
				case MeasureTypes::ALIGNMENT_SKIP_MANUAL:
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_GAUGEBLOCK_DCC);
					break;
					
				case MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC:
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_MEASURE_GAUGE_BLOCK);
					break;
					
				case MeasureTypes::ALIGNMENT_ONLY:
					
					d_measure_routines.Refsys_Clear_Alignment();
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_GAUGEBLOCK_MANUAL);
					
					break;
			}
			
			if(d_temperature_compensation_check->isChecked())
			{
				if(parameters.axis_x_temperature > 40.0 || parameters.axis_x_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  X axis temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
				
				if(parameters.axis_y_temperature > 40.0 || parameters.axis_y_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  Y axis temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
				
				if(parameters.axis_z_temperature > 40.0 || parameters.axis_z_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  Z axis temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
				
				if(parameters.part_temperature > 40.0 || parameters.part_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  Part temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
			}
			
			part_expansion_coefficient = parameters.gaugeblock_expansion_coeff;

			break;
		

		case MeasureTypes::MEASURE_POINT_REPEAT:
			
			d_display_coordinate_system = MeasureTypes::COORDINATE_SYSTEM_PART;
			
			if(d_last_configuration != d_configuration &&
			   alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL)
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);
				
				this->Add_Log_Text(QStringLiteral("WAR:  Manual alignment must be performed."));
			}
			else if(alignment_option == MeasureTypes::ALIGNMENT_ONLY || alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC)
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);
				
				this->Add_Log_Text(QStringLiteral("WAR:  Alignment option not available."));
			}
			
			switch(alignment_option)
			{
				case MeasureTypes::ALIGNMENT_FULL:
					
					d_measure_routines.Refsys_Clear_Alignment();
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_POINT_MANUAL);
					break;
					
				case MeasureTypes::ALIGNMENT_SKIP_MANUAL:
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_POINT_DCC);
					break;
					
				case MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC:
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_MEASURE_POINT_REPEAT);
					break;
					
				case MeasureTypes::ALIGNMENT_ONLY:
					
					d_measure_routines.Refsys_Clear_Alignment();
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_POINT_MANUAL);
					
					break;
			}
			
			break;
			
		case MeasureTypes::MEASURE_SPHERE_REPEAT:
			
			d_display_coordinate_system = MeasureTypes::COORDINATE_SYSTEM_PART_ORIGIN_ONLY;
			
			if(d_last_configuration != d_configuration &&
			   d_last_configuration != MeasureTypes::MEASURE_SPHERE_PFTU &&
			   alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL)
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);

				this->Add_Log_Text(QStringLiteral("WAR:  Manual alignment must be performed."));
			}
			else if(alignment_option == MeasureTypes::ALIGNMENT_ONLY || alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC)
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);
				
				this->Add_Log_Text(QStringLiteral("WAR:  Alignment option not available."));
			}

			if(alignment_option != MeasureTypes::ALIGNMENT_FULL)
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);
				
				this->Add_Log_Text(QStringLiteral("WAR:  Manual alignment must be performed."));
			}
			
			d_measure_routines.Refsys_Clear_Alignment();
			d_measure_routines.Refsys_Skew1(d_measure_routines.Tool_Vector(),TReferenceSystem::ZMINUS);
			d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_SPHERE_MANUAL);
						
			break;
			
		case MeasureTypes::MEASURE_SPHERE_PFTU:
			
			d_display_coordinate_system = MeasureTypes::COORDINATE_SYSTEM_PART_ORIGIN_ONLY;
			
			if(d_last_configuration != d_configuration &&
			   d_last_configuration != MeasureTypes::MEASURE_SPHERE_REPEAT &&
			   alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL)
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);

				this->Add_Log_Text(QStringLiteral("WAR:  Manual alignment must be performed."));
			}
			else if(alignment_option == MeasureTypes::ALIGNMENT_ONLY || alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC)
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);
				
				this->Add_Log_Text(QStringLiteral("WAR:  Alignment option not available."));
			}
			
			if(alignment_option != MeasureTypes::ALIGNMENT_FULL)
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);
				
				this->Add_Log_Text(QStringLiteral("WAR:  Manual alignment must be performed."));
			}
			
			d_measure_routines.Refsys_Clear_Alignment();
			d_measure_routines.Refsys_Skew1(d_measure_routines.Tool_Vector(),TReferenceSystem::ZMINUS);
			d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_SPHERE_MANUAL);

			break;
			
		case MeasureTypes::MEASURE_BALLBAR_B89:
			
			d_display_coordinate_system = MeasureTypes::COORDINATE_SYSTEM_MACHINE;

			if(d_last_configuration != d_configuration &&
			   alignment_option != MeasureTypes::ALIGNMENT_FULL)
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);

				this->Add_Log_Text(QStringLiteral("WAR:  Manual alignment must be performed."));
			}

			switch(alignment_option)
			{
				case MeasureTypes::ALIGNMENT_FULL:
				case MeasureTypes::ALIGNMENT_ONLY:
					
					d_measure_routines.Refsys_Clear_Alignment();
					d_measure_routines.Refsys_Skew1(d_measure_routines.Tool_Vector(),TReferenceSystem::ZMINUS);
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_BALLBAR_B89_MANUAL);
					
					break;
					
				case MeasureTypes::ALIGNMENT_SKIP_MANUAL:
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_BALLBAR_B89_DCC);
					break;
					
				case MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC:
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_MEASURE_BALLBAR_B89);
					break;
			}
			
			if(d_temperature_compensation_check->isChecked())
			{
				if(parameters.axis_x_temperature > 40.0 || parameters.axis_x_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  X axis temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
				
				if(parameters.axis_y_temperature > 40.0 || parameters.axis_y_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  Y axis temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
				
				if(parameters.axis_z_temperature > 40.0 || parameters.axis_z_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  Z axis temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
				
				if(parameters.part_temperature > 40.0 || parameters.part_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  Part temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
			}
			
			part_expansion_coefficient = parameters.ballbar_expansion_coeff;

			break;
			
		case MeasureTypes::MEASURE_BALLBAR_10360:
						
			d_display_coordinate_system = MeasureTypes::COORDINATE_SYSTEM_MACHINE;

			if(d_last_configuration != d_configuration &&
			   alignment_option != MeasureTypes::ALIGNMENT_FULL)
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);

				this->Add_Log_Text(QStringLiteral("WAR:  Manual alignment must be performed."));
			}

			switch(alignment_option)
			{
				case MeasureTypes::ALIGNMENT_FULL:
				case MeasureTypes::ALIGNMENT_ONLY:
				
					d_measure_routines.Refsys_Clear_Alignment();
					d_measure_routines.Refsys_Skew1(d_measure_routines.Tool_Vector(),TReferenceSystem::ZMINUS);
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_BALLBAR_10360_MANUAL);
					
					break;
					
				case MeasureTypes::ALIGNMENT_SKIP_MANUAL:
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_BALLBAR_10360_DCC);
					break;
					
				case MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC:
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_MEASURE_BALLBAR_10360);
					break;
			}
			
			if(d_temperature_compensation_check->isChecked())
			{
				if(parameters.axis_x_temperature > 40.0 || parameters.axis_x_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  X axis temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
				
				if(parameters.axis_y_temperature > 40.0 || parameters.axis_y_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  Y axis temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
				
				if(parameters.axis_z_temperature > 40.0 || parameters.axis_z_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  Z axis temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
				
				if(parameters.part_temperature > 40.0 || parameters.part_temperature < 10.0)
				{
					this->Add_Log_Text(QStringLiteral("ERR:  Part temperature out of range (10 - 40 C)."));
					d_measure_routines.Reset_Current_Sequence_Function();
				}
			}
			
			part_expansion_coefficient = parameters.ballbar_10360_expansion_coeff;

			break;
			
		case MeasureTypes::MEASURE_RINGGAUGE:
			
			d_display_coordinate_system = MeasureTypes::COORDINATE_SYSTEM_PART;

			if(d_last_configuration != d_configuration &&
			   (alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL || alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC))
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);

				this->Add_Log_Text(QStringLiteral("WAR:  Manual alignment must be performed."));
			}
			
			switch(alignment_option)
			{
				case MeasureTypes::ALIGNMENT_FULL:
					
					d_measure_routines.Refsys_Clear_Alignment();
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_RINGGAUGE_MANUAL);
				
					break;
					
				case MeasureTypes::ALIGNMENT_SKIP_MANUAL:
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_RINGGAUGE_DCC);
					break;
					
				case MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC:
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_MEASURE_RINGGAUGE);
					break;
					
				case MeasureTypes::ALIGNMENT_ONLY:
					
					d_measure_routines.Refsys_Clear_Alignment();
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_RINGGAUGE_MANUAL);
					
					break;
			}
			
			break;
			
		case MeasureTypes::MEASURE_PINGAUGE:
			
			d_display_coordinate_system = MeasureTypes::COORDINATE_SYSTEM_PART;
			
			if(d_last_configuration != d_configuration &&
			   (alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL || alignment_option == MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC))
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);

				this->Add_Log_Text(QStringLiteral("WAR:  Manual alignment must be performed."));
			}
			
			switch(alignment_option)
			{
				case MeasureTypes::ALIGNMENT_FULL:
					
					d_measure_routines.Refsys_Clear_Alignment();
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_PINGAUGE_MANUAL);
					
					break;
					
				case MeasureTypes::ALIGNMENT_SKIP_MANUAL:
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_PINGAUGE_DCC);
					break;
					
				case MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC:
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_MEASURE_PINGAUGE);
					break;
					
				case MeasureTypes::ALIGNMENT_ONLY:
					
					d_measure_routines.Refsys_Clear_Alignment();
					d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_PINGAUGE_MANUAL);
					
					break;
			}

			break;
			
		case MeasureTypes::MEASURE_ROLL_OFFSET_TOOL:
						
			d_display_coordinate_system = MeasureTypes::COORDINATE_SYSTEM_PART_ORIGIN_ONLY;
			
			vec.Set(parameters.sphere_stem_vector_i,parameters.sphere_stem_vector_j,parameters.sphere_stem_vector_k);
			d_measure_routines.Set_Sphere_Stem_Vector(vec);
			
			if(alignment_option != MeasureTypes::ALIGNMENT_FULL)
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);
				
				this->Add_Log_Text(QStringLiteral("WAR:  Manual alignment must be performed."));
			}
					
			d_measure_routines.Refsys_Clear_Alignment();
			d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_INIT_ROLL);

			break;
	
			
		case MeasureTypes::CALIBRATE_TOOLS:
			
			d_display_coordinate_system = MeasureTypes::COORDINATE_SYSTEM_MACHINE;

			// unit length checked in Write_User_Inputs
			vec.Set(parameters.sphere_stem_vector_i,parameters.sphere_stem_vector_j,parameters.sphere_stem_vector_k);
			d_measure_routines.Set_Sphere_Stem_Vector(vec);
			
			if(alignment_option != MeasureTypes::ALIGNMENT_FULL)
			{
				alignment_option = MeasureTypes::ALIGNMENT_FULL;
				this->Set_Alignment_Option(alignment_option);
				
				this->Add_Log_Text(QStringLiteral("WAR:  Manual alignment must be performed."));
			}

			d_measure_routines.Refsys_Clear_Alignment();
			d_measure_routines.Set_Current_Sequence_Function(MeasureTypes::SEQUENCE_ALIGN_SPHERE_MANUAL_CALIBRATION);

			break;
	}
	
	d_measure_routines.Set_Measurement_Parameters(parameters);
	
	if(d_thermal_compensation_mode == TMeasureDialog::CONTROLLER_AXIS_AND_PART)
	{
		emit Update_Part_Expansion_Coefficient(part_expansion_coefficient);
		
		this->Add_Log_Text(QStringLiteral("INF:  Temperature compensation active on controller."));
	}
	else if(d_temperature_compensation_check->isChecked())
	{		
		d_tempcomp_information_dialog->Set_Temperature_X(parameters.axis_x_temperature);
		d_tempcomp_information_dialog->Set_Temperature_Y(parameters.axis_y_temperature);
		d_tempcomp_information_dialog->Set_Temperature_Z(parameters.axis_z_temperature);
		d_tempcomp_information_dialog->Set_Temperature_Part(parameters.part_temperature);
		
		if(d_thermal_compensation_mode == TMeasureDialog::CONTROLLER_AXIS_SOFTWARE_PART)
		{
			d_tempcomp_information_dialog->Set_CTE_X(0.0);
			d_tempcomp_information_dialog->Set_CTE_Y(0.0);
			d_tempcomp_information_dialog->Set_CTE_Z(0.0);
		}
		else	// TMeasureDialog::SOFTWARE_COMPENSATION_AXIS_AND_PART
		{
			d_tempcomp_information_dialog->Set_CTE_X(d_machine_settings.expansion_coeff_x);
			d_tempcomp_information_dialog->Set_CTE_Y(d_machine_settings.expansion_coeff_y);
			d_tempcomp_information_dialog->Set_CTE_Z(d_machine_settings.expansion_coeff_z);
		}
		
		d_tempcomp_information_dialog->Set_CTE_Part(part_expansion_coefficient);
				
		if(d_thermal_compensation_mode != TMeasureDialog::CONTROLLER_AXIS_AND_PART)
		{
			rect = this->geometry();	// frameGeometry() not reliable
			
			d_tempcomp_information_dialog->move(rect.topRight());	
			d_tempcomp_information_dialog->show();
		}
	}
	
	d_measure_routines.Generate_Sequence();
	d_measure_routines.Start_Sequence();
	
	if(d_temperature_compensation_check->isChecked())
	{
		emit Refresh_Temperatures();		
	}
	
	d_active_tool_combo->setEnabled(false);
	
	emit Sequence_Start();
}

void TMeasureDialog::Pause(
	bool								state)
{
	if(state)
	{
		this->Set_State(TMeasureDialog::STATE_PAUSED);
		emit Pause_Driver();
	}
	else
	{
		this->Set_State(TMeasureDialog::STATE_RUNNING);
		emit Resume_Driver();
	}
}

void TMeasureDialog::Stop(void)
{
	emit Stop_Driver();
	emit Sequence_End();
	
	d_tempcomp_information_dialog->hide();

	if(d_configuration != MeasureTypes::MEASURE_ROLL_OFFSET_TOOL &&
	   d_configuration != MeasureTypes::CALIBRATE_TOOLS)
	{
		d_active_tool_combo->setEnabled(true);
	}
	
	this->Set_State(TMeasureDialog::STATE_STOPPED);
}

void TMeasureDialog::Erase(void)
{
	int									count;
	
	if(d_active_feature)
	{
		count = d_active_feature->Actual_Point_Count();
		
		if(count)
		{
			d_active_feature->Erase_Last_Measured_Point();
			d_point_count_widget->Set_Actual_Point_Count(count-1);
			
			if(count == 1)	// removed last point
			{
				d_erase_button->setEnabled(false);
			}
		}
		else
		{
			d_erase_button->setEnabled(false);
		}
		
		d_next_button->setEnabled(false);
	}
}

void TMeasureDialog::Next(void)
{
	this->Update_Active_Feature();
}

void TMeasureDialog::Close(void)
{
	TMeasureRoutines::TParameters		parameters;
	
	parameters = d_measure_routines.Measurement_Parameters();

	this->Write_User_Inputs(&parameters);	// entries may be updated
	
	d_measure_routines.Set_Measurement_Parameters(parameters);

	if(d_current_state != TMeasureDialog::STATE_STOPPED)
	{
		emit Stop_Driver();
	}

	this->Set_State(TMeasureDialog::STATE_STOPPED);
	
	this->close();
}

void TMeasureDialog::Temperature_Compensation_Toggled(
	bool								state)
{
	d_enable_temperature_compensation = state;
		
	if(state)
	{
		emit Refresh_Temperatures();
	}
	else
	{
		if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_10360)
		{
			d_input_widget_right->Enable_Entry_Item(0,true);
			d_input_widget_right->Enable_Entry_Item(1,true);
			d_input_widget_right->Enable_Entry_Item(2,true);
			d_input_widget_right->Enable_Entry_Item(3,true);
		}
		else if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_B89)
		{
			d_input_widget_right->Enable_Entry_Item(0,true);
			d_input_widget_right->Enable_Entry_Item(1,true);
			d_input_widget_right->Enable_Entry_Item(2,true);
			d_input_widget_right->Enable_Entry_Item(3,true);
		}
		else if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_SQUARE)
		{
			d_input_widget_right->Enable_Entry_Item(0,true);
			d_input_widget_right->Enable_Entry_Item(1,true);
			d_input_widget_right->Enable_Entry_Item(2,true);
			d_input_widget_right->Enable_Entry_Item(3,true);
		}
		else if(d_configuration == MeasureTypes::MEASURE_BALLBAR_B89)
		{
			d_input_widget_right->Enable_Entry_Item(0,true);
			d_input_widget_right->Enable_Entry_Item(1,true);
			d_input_widget_right->Enable_Entry_Item(2,true);
			d_input_widget_right->Enable_Entry_Item(3,true);
		}
		else if(d_configuration == MeasureTypes::MEASURE_BALLBAR_10360)
		{
			d_input_widget_right->Enable_Entry_Item(0,true);
			d_input_widget_right->Enable_Entry_Item(1,true);
			d_input_widget_right->Enable_Entry_Item(2,true);
			d_input_widget_right->Enable_Entry_Item(3,true);
		}
		else if(d_configuration == MeasureTypes::MEASURE_GAUGEBLOCK)
		{
			d_input_widget_right->Enable_Entry_Item(0,true);
			d_input_widget_right->Enable_Entry_Item(1,true);
			d_input_widget_right->Enable_Entry_Item(2,true);
			d_input_widget_right->Enable_Entry_Item(3,true);
		}
	}
}

void TMeasureDialog::Input_Left_Option_Changed(
	int									option_index,
	int									option_data)
{
	QString								tool_name;
		
	if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_10360 ||
	   d_configuration == MeasureTypes::MEASURE_STEPGAUGE_B89 ||
	   d_configuration == MeasureTypes::MEASURE_STEPGAUGE_SQUARE)
	{
		if(option_index == 0)
		{
			switch(option_data)
			{
				case MeasureTypes::STEPGAUGE_CHECKMASTER:
				case MeasureTypes::STEPGAUGE_NULL:
					d_configuration_pixmap = QPixmap(":/icon128/stepgauge_checkmaster_icon128.png");
					break;
					
				case MeasureTypes::STEPGAUGE_STARRET:
					d_configuration_pixmap = QPixmap(":/icon128/stepgauge_webber_icon128.png");
					break;
					
				case MeasureTypes::STEPGAUGE_KOBA:
					d_configuration_pixmap = QPixmap(":/icon128/stepgauge_koba_icon128.png");
					break;
			}
			
			this->Set_Pixmap(d_configuration_pixmap,false);
		}
	}
	else if(d_configuration == MeasureTypes::MEASURE_ROLL_OFFSET_TOOL)
	{
		if(option_index == 0 || option_index == 1)
		{
			tool_name = d_input_widget_left->Entry_Item_Text(option_index);
		}
	}
}

void TMeasureDialog::Input_Right_Option_Changed(
	int									option_index,
	int									option_data)
{
	Q_UNUSED(option_index);
	Q_UNUSED(option_data);
}

void TMeasureDialog::Tool_Selection_Changed(
	int									index)
{
	QString								name;
	
	name = d_active_tool_combo->itemText(index);
	d_measure_routines.Set_Active_Tool_Name(name);

	emit Change_Tool(name);
}

void TMeasureDialog::Add_Log_Text(
	const QString						&text)
{
	d_information_text->moveCursor(QTextCursor::End);
	d_information_text->appendPlainText(text);
}

void TMeasureDialog::Update_Last_Configuration(void)
{
	TMeasureRoutines::TParameters		parameters;
	
	parameters = d_measure_routines.Measurement_Parameters();

	d_last_configuration = d_configuration;
	d_last_stepgauge_type = static_cast<MeasureTypes::TStepgaugeType>(parameters.stepgauge_type);
}

void TMeasureDialog::closeEvent(
	QCloseEvent							*event)
{	
	TMeasureRoutines::TParameters		parameters;
	
	parameters = d_measure_routines.Measurement_Parameters();
	
	this->Write_User_Inputs(&parameters);
	
	d_measure_routines.Set_Measurement_Parameters(parameters);

	if(d_current_state != TMeasureDialog::STATE_STOPPED)
	{
		event->ignore();
		return;
	}
	
	event->accept();
}

void TMeasureDialog::Set_State(
	const TState						state)
{
	bool								prev_state;

	d_current_state = state;
	
	switch(state)
	{
		case TMeasureDialog::STATE_STOPPED:
		
			d_play_button->setEnabled(true);
			d_pause_button->setEnabled(false);
			d_stop_button->setEnabled(false);
			d_close_button->setEnabled(true);
			
			d_input_widget_left->Enable_Entry_Items(true);
			d_input_widget_right->Enable_Entry_Items(true);
			d_alignment_mode_label->setEnabled(true);
						
			this->Set_Pixmap(d_configuration_pixmap,false);
	
			d_point_count_widget->Set_Nominal_Point_Count(0);
			d_point_count_widget->Set_Actual_Point_Count(0);
			
			d_erase_button->setEnabled(false);
			d_next_button->setEnabled(false);

			d_alignment_options_combo->setEnabled(true);
			
			if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_10360 ||
			   d_configuration == MeasureTypes::MEASURE_STEPGAUGE_B89 ||
			   d_configuration == MeasureTypes::MEASURE_STEPGAUGE_SQUARE ||
			   d_configuration == MeasureTypes::MEASURE_BALLBAR_B89 ||
			   d_configuration == MeasureTypes::MEASURE_BALLBAR_10360 ||
			   d_configuration == MeasureTypes::MEASURE_GAUGEBLOCK)
			{
				if(d_thermal_compensation_mode == TMeasureDialog::SOFTWARE_COMPENSATION_AXIS_AND_PART)
				{
					d_temperature_compensation_check->setEnabled(true);
				}
				else
				{
					d_temperature_compensation_check->setEnabled(false);
				}
				
				if(d_temperature_compensation_check->isChecked())
				{
					emit Refresh_Temperatures();
				}
			}
			else
			{
				d_temperature_compensation_check->setEnabled(false);
			}

			d_measure_routines.Close_Output_File();

			prev_state = d_pause_button->blockSignals(true);
			d_pause_button->setChecked(false);
			d_pause_button->blockSignals(prev_state);
			
			break;
		
		case TMeasureDialog::STATE_PAUSED:
		
			d_play_button->setEnabled(false);
			d_pause_button->setEnabled(true);
			d_stop_button->setEnabled(true);
			d_close_button->setEnabled(false);
			
			d_input_widget_left->Enable_Entry_Items(false);
			d_input_widget_right->Enable_Entry_Items(false);
			d_alignment_mode_label->setEnabled(false);
			
			d_alignment_options_combo->setEnabled(false);
			d_temperature_compensation_check->setEnabled(false);
						
			this->Set_Pixmap(QPixmap(":/icon128/pause_icon128.png"),false);

			break;
		
		case TMeasureDialog::STATE_RUNNING:
		
			d_play_button->setEnabled(false);
			d_pause_button->setEnabled(true);
			d_stop_button->setEnabled(true);
			d_close_button->setEnabled(false);
			
			d_input_widget_left->Enable_Entry_Items(false);
			d_input_widget_right->Enable_Entry_Items(false);
			d_alignment_mode_label->setEnabled(false);
			
			this->Restore_Pixmap();
			
			d_alignment_options_combo->setEnabled(false);
			d_temperature_compensation_check->setEnabled(false);
			break;
	}
}

void TMeasureDialog::Set_Pixmap(
	const QPixmap 						&pixmap,
	const bool 							save_current)
{
	d_icon_label->setPixmap(pixmap);
	
	if(save_current)
	{
		d_saved_pixmap = pixmap;
	}
}

void TMeasureDialog::Restore_Pixmap(void)
{
	if(!d_saved_pixmap.isNull())
	{
		d_icon_label->setPixmap(d_saved_pixmap);
	}
}

TVector3 TMeasureDialog::Add_Temperature_Compensation(
	const TVector3						&pnt) const
{
	TVector3							pnt_out;
	
	pnt_out = pnt + this->Temperature_Compensation_Correction(pnt);

	return pnt_out;
}

TVector3 TMeasureDialog::Remove_Temperature_Compensation(
	const TVector3						&pnt) const
{
	TVector3							pnt_out;
	
	pnt_out = pnt - this->Temperature_Compensation_Correction(pnt);
	
	return pnt_out;
}

TVector3 TMeasureDialog::Temperature_Compensation_Correction(
	const TVector3						&pnt) const
{
	TVector3							pnt_correction;
	TVector3							compensation;
	double								part_expansion_coefficient;
	double								x_expansion_coefficient;
	double								y_expansion_coefficient;
	double								z_expansion_coefficient;
	TMeasureRoutines::TParameters		parameters;
	static const double					COEFF_CONVERSION(0.000001);
		
	if(d_thermal_compensation_mode == TMeasureDialog::CONTROLLER_AXIS_AND_PART)
	{
		return pnt_correction;
	}
	
	parameters = d_measure_routines.Measurement_Parameters();

	if(d_configuration == MeasureTypes::MEASURE_STEPGAUGE_10360 ||
	   d_configuration == MeasureTypes::MEASURE_STEPGAUGE_B89 ||
	   d_configuration == MeasureTypes::MEASURE_STEPGAUGE_SQUARE)
	{
		part_expansion_coefficient = parameters.stepgauge_expansion_coeff * COEFF_CONVERSION;
	}
	else if(d_configuration == MeasureTypes::MEASURE_BALLBAR_B89)
	{
		part_expansion_coefficient = parameters.ballbar_expansion_coeff * COEFF_CONVERSION;
	}
	else if(d_configuration == MeasureTypes::MEASURE_BALLBAR_10360)
	{
		part_expansion_coefficient = parameters.ballbar_10360_expansion_coeff * COEFF_CONVERSION;
	}
	else if(d_configuration == MeasureTypes::MEASURE_GAUGEBLOCK)
	{
		part_expansion_coefficient = parameters.gaugeblock_expansion_coeff * COEFF_CONVERSION;
	}
	else
	{
		return pnt_correction;
	}

	if(d_thermal_compensation_mode == TMeasureDialog::CONTROLLER_AXIS_SOFTWARE_PART)
	{
		x_expansion_coefficient = 0.0;
		y_expansion_coefficient = 0.0;
		z_expansion_coefficient = 0.0;
	}
	else	// TMeasureDialog::SOFTWARE_COMPENSATION_AXIS_AND_PART
	{
		x_expansion_coefficient = d_machine_settings.expansion_coeff_x * COEFF_CONVERSION;
		y_expansion_coefficient = d_machine_settings.expansion_coeff_y * COEFF_CONVERSION;
		z_expansion_coefficient = d_machine_settings.expansion_coeff_z * COEFF_CONVERSION;
	}
	
	compensation.x = (1.0 + x_expansion_coefficient * (parameters.axis_x_temperature - 20.0)) /
	(1.0 + part_expansion_coefficient * (parameters.part_temperature - 20.0));
	
	compensation.y = (1.0 + y_expansion_coefficient * (parameters.axis_y_temperature - 20.0)) /
	(1.0 + part_expansion_coefficient * (parameters.part_temperature - 20.0));
	
	compensation.z = (1.0 + z_expansion_coefficient * (parameters.axis_z_temperature - 20.0)) /
	(1.0 + part_expansion_coefficient * (parameters.part_temperature - 20.0));
	
	pnt_correction.Set(pnt.x * compensation.x,pnt.y * compensation.y,pnt.z * compensation.z);
	pnt_correction -= pnt;
	
	if(d_tempcomp_information_dialog->Update_Scale_Required())
	{
		d_tempcomp_information_dialog->Set_Scale_X(compensation.x);
		d_tempcomp_information_dialog->Set_Scale_Y(compensation.y);
		d_tempcomp_information_dialog->Set_Scale_Z(compensation.z);
	}

	return pnt_correction;
}

void TMeasureDialog::Set_Alignment_Option(
	const MeasureTypes::TAlignmentOptions		option)
{
	switch(option)
	{
		case MeasureTypes::ALIGNMENT_FULL:
			d_alignment_options_combo->setCurrentIndex(0);
			break;
			
		case MeasureTypes::ALIGNMENT_SKIP_MANUAL:
			d_alignment_options_combo->setCurrentIndex(1);
			break;
			
		case MeasureTypes::ALIGNMENT_SKIP_MANUAL_AND_DCC:
			d_alignment_options_combo->setCurrentIndex(2);
			break;
			
		case MeasureTypes::ALIGNMENT_ONLY:
			d_alignment_options_combo->setCurrentIndex(3);
			break;
	}
}

void TMeasureDialog::Update_Active_Feature(void)
{
	TCommandFeature::TDiameterType		diameter_type;
	TMeasureRoutines::TFeatureMemory 	feature_memory;
	TReferenceSystem					refsys;
	TMat4								mat4;
	TVector3							feat_xyz;
	TVector3							feat_ijk;

	if(d_active_feature)
	{
		if(!d_active_feature->Waiting_For_Measured_Points())
		{
			this->Set_Pixmap(QPixmap(":/icon128/working_icon128.png"),true);
			
			if(!d_active_feature->Calculate_Feature(d_measure_routines.Refsys_Axis_Z(),d_measure_routines.Tip_Radius()))
			{
				this->Add_Log_Text(QStringLiteral("ERR:  Error calculating best fit feature from points."));
				this->Stop();
				return;
			}
			
			diameter_type = d_active_feature->Nominal_Diameter_Type();
			if(diameter_type != TCommandFeature::DIAMETER_TYPE_NONE)
			{
				if(d_active_feature->Actual_Diameter_Type() != diameter_type)
				{
					if(diameter_type == TCommandFeature::DIAMETER_TYPE_ID)
					{
						this->Add_Log_Text(QStringLiteral("ERR:  Feature must be type ID.  Calculated as type OD."));
					}
					else
					{
						this->Add_Log_Text(QStringLiteral("ERR:  Feature must be type OD.  Calculated as type ID."));
					}
					
					this->Stop();
					return;
				}
			}
			
			feature_memory.memory_id = d_active_feature->Memory_Id();
			feature_memory.xyz = d_active_feature->Actual_XYZ();
			feature_memory.ijk = d_active_feature->Actual_IJK();
			feature_memory.form = d_active_feature->Form();
			feature_memory.diameter = d_active_feature->Actual_Diameter();
			
			d_measure_routines.Set_Feature_Memory(feature_memory);
			
			if(d_active_feature->Display_Format() == TCommandFeature::FORMAT_DEFAULT)
			{
				switch(d_display_coordinate_system)
				{
					case MeasureTypes::COORDINATE_SYSTEM_MACHINE:
						feat_xyz = feature_memory.xyz;
						feat_ijk = feature_memory.ijk;
						break;
						
					case MeasureTypes::COORDINATE_SYSTEM_PART:
						feat_xyz = d_measure_routines.Refsys_FromWorld(feature_memory.xyz);
						feat_ijk = d_measure_routines.Refsys_AxisFromWorld(feature_memory.ijk);
						break;
						
					case MeasureTypes::COORDINATE_SYSTEM_PART_ORIGIN_ONLY:
						feat_xyz = feature_memory.xyz - d_measure_routines.Refsys_Origin();
						feat_ijk = feature_memory.ijk;
						break;
				}
				
				switch(d_active_feature->Type())
				{
					case TCommand::COMMAND_MEASURE_PLANE:
						
						this->Add_Log_Text(QString("# Memory[%1]")
										   .arg(feature_memory.memory_id));
						
						this->Add_Log_Text(QStringLiteral("      Type: Plane"));
						this->Add_Log_Text(QString("       XYZ: %3, %4, %5")
										   .arg(feat_xyz.x,12,'f',4)
										   .arg(feat_xyz.y,12,'f',4)
										   .arg(feat_xyz.z,12,'f',4));
						this->Add_Log_Text(QString("       IJK:   %3, %4, %5")
										   .arg(feat_ijk.i,12,'f',6)
										   .arg(feat_ijk.j,12,'f',6)
										   .arg(feat_ijk.k,12,'f',6));
						this->Add_Log_Text(QString("      Form: %1")
										   .arg(feature_memory.form,12,'f',4));
						
						break;
						
					case TCommand::COMMAND_MEASURE_LINE:
						
						this->Add_Log_Text(QString("# Memory[%1]")
										   .arg(feature_memory.memory_id));
						
						this->Add_Log_Text(QStringLiteral("      Type: Line"));
						this->Add_Log_Text(QString("       XYZ: %3, %4, %5")
										   .arg(feat_xyz.x,12,'f',4)
										   .arg(feat_xyz.y,12,'f',4)
										   .arg(feat_xyz.z,12,'f',4));
						this->Add_Log_Text(QString("       IJK:   %3, %4, %5")
										   .arg(feat_ijk.i,12,'f',6)
										   .arg(feat_ijk.j,12,'f',6)
										   .arg(feat_ijk.k,12,'f',6));
						this->Add_Log_Text(QString("      Form: %1")
										   .arg(feature_memory.form,12,'f',4));
						
						break;
						
					case TCommand::COMMAND_MEASURE_POINT:
						
						
						this->Add_Log_Text(QString("# Memory[%1]")
										   .arg(feature_memory.memory_id));
						
						this->Add_Log_Text(QStringLiteral("      Type: Point"));
						this->Add_Log_Text(QString("       XYZ: %3, %4, %5")
										   .arg(feat_xyz.x,12,'f',4)
										   .arg(feat_xyz.y,12,'f',4)
										   .arg(feat_xyz.z,12,'f',4));
						
						
						break;
						
					case TCommand::COMMAND_MEASURE_SPHERE:
						
						
						this->Add_Log_Text(QString("# Memory[%1]")
										   .arg(feature_memory.memory_id));
						
						this->Add_Log_Text(QStringLiteral("      Type: Sphere"));
						this->Add_Log_Text(QString("       XYZ: %3, %4, %5")
										   .arg(feat_xyz.x,12,'f',4)
										   .arg(feat_xyz.y,12,'f',4)
										   .arg(feat_xyz.z,12,'f',4));
						this->Add_Log_Text(QString("      Diam: %1")
										   .arg(feature_memory.diameter,12,'f',4));
						
						this->Add_Log_Text(QString("      Form: %1")
										   .arg(feature_memory.form,12,'f',4));
						
						
						break;
						
					case TCommand::COMMAND_MEASURE_CIRCLE:
						
						
						this->Add_Log_Text(QString("# Memory[%1]")
										   .arg(feature_memory.memory_id));
						
						this->Add_Log_Text(QStringLiteral("      Type: Circle"));
						this->Add_Log_Text(QString("       XYZ: %3, %4, %5")
										   .arg(feat_xyz.x,12,'f',4)
										   .arg(feat_xyz.y,12,'f',4)
										   .arg(feat_xyz.z,12,'f',4));
						this->Add_Log_Text(QString("      Diam: %1")
										   .arg(feature_memory.diameter,12,'f',4));
						
						this->Add_Log_Text(QString("      Form: %1")
										   .arg(feature_memory.form,12,'f',4));
						
						
						break;
						
					default:
						break;
				}
			}
			else
			{
				switch(d_active_feature->Display_Format())
				{
					case TCommandFeature::FORMAT_NONE:
					case TCommandFeature::FORMAT_DEFAULT:
						break;
						
					case TCommandFeature::FORMAT_XYZ:
					case TCommandFeature::FORMAT_XYZD:
					case TCommandFeature::FORMAT_XYZIJK:
					case TCommandFeature::FORMAT_XYZIJKD:
					case TCommandFeature::FORMAT_XYZDF:
						
						refsys.Clear_Alignment();
						
						switch(d_display_coordinate_system)
						{
							case MeasureTypes::COORDINATE_SYSTEM_MACHINE:
								this->Add_Log_Text(d_active_feature->Feature_Item_Text(refsys));
								break;
								
							case MeasureTypes::COORDINATE_SYSTEM_PART:
								
								mat4.X(d_measure_routines.Refsys_Axis_X());
								mat4.Y(d_measure_routines.Refsys_Axis_Y());
								mat4.Z(d_measure_routines.Refsys_Axis_Z());
								
								refsys.Set_Rotation_Mat(mat4);
								refsys.Set_Origin(d_measure_routines.Refsys_Origin());

								this->Add_Log_Text(d_active_feature->Feature_Item_Text(refsys));

								break;
								
							case MeasureTypes::COORDINATE_SYSTEM_PART_ORIGIN_ONLY:
								
								refsys.Set_Origin(d_measure_routines.Refsys_Origin());
								this->Add_Log_Text(d_active_feature->Feature_Item_Text(refsys));
								
								break;
						}
						
						break;
				}
				
			}
			
			d_active_feature = 0;
			d_erase_button->setEnabled(false);
			d_next_button->setEnabled(false);
		}
	}
}

void TMeasureDialog::Write_User_Inputs(
	TMeasureRoutines::TParameters		* const data)
{
	TVector3							vec;
	
	assert(data);
		
	switch(d_configuration)
	{
		case MeasureTypes::MEASURE_NULL:
			break;
			
		case MeasureTypes::MEASURE_STEPGAUGE_10360:
			
			data->stepgauge_type = static_cast<MeasureTypes::TStepgaugeType>(d_input_widget_left->Entry_Item_Data(0));
			data->stepgauge_10360_position_name = d_input_widget_left->Entry_Item_Text(1);
			data->stepgauge_serial_number = d_input_widget_left->Entry_Item_Text(2);
			data->stepgauge_measurement_zero_position = d_input_widget_left->Entry_Item_Text(3).toDouble();
			data->stepgauge_zero_step_spacing = d_input_widget_left->Entry_Item_Text(4).toDouble();
			data->stepgauge_nominal_block_spacing = d_input_widget_left->Entry_Item_Text(5).toDouble();
			data->axis_x_temperature = d_input_widget_right->Entry_Item_Text(0).toDouble();
			data->axis_y_temperature = d_input_widget_right->Entry_Item_Text(1).toDouble();
			data->axis_z_temperature = d_input_widget_right->Entry_Item_Text(2).toDouble();
			data->part_temperature = d_input_widget_right->Entry_Item_Text(3).toDouble();
			data->stepgauge_expansion_coeff = d_input_widget_right->Entry_Item_Text(4).toDouble();

			break;
			
		case MeasureTypes::MEASURE_STEPGAUGE_B89:
			
			data->stepgauge_type = static_cast<MeasureTypes::TStepgaugeType>(d_input_widget_left->Entry_Item_Data(0));
			data->stepgauge_b89_position_name = d_input_widget_left->Entry_Item_Text(1);
			data->stepgauge_serial_number = d_input_widget_left->Entry_Item_Text(2);
			data->stepgauge_measurement_b89_start = d_input_widget_left->Entry_Item_Text(3).toDouble();
			data->stepgauge_b89_nominal_block_spacing = d_input_widget_left->Entry_Item_Text(4).toDouble();
			data->stepgauge_b89_measurement_count = d_input_widget_left->Entry_Item_Text(5).toInt();
			data->axis_x_temperature = d_input_widget_right->Entry_Item_Text(0).toDouble();
			data->axis_y_temperature = d_input_widget_right->Entry_Item_Text(1).toDouble();
			data->axis_z_temperature = d_input_widget_right->Entry_Item_Text(2).toDouble();
			data->part_temperature = d_input_widget_right->Entry_Item_Text(3).toDouble();
			data->stepgauge_expansion_coeff = d_input_widget_right->Entry_Item_Text(4).toDouble();

			break;
			
		case MeasureTypes::MEASURE_STEPGAUGE_SQUARE:
			
			data->stepgauge_type = static_cast<MeasureTypes::TStepgaugeType>(d_input_widget_left->Entry_Item_Data(0));
			data->stepgauge_square_position_name = d_input_widget_left->Entry_Item_Text(1);
			data->stepgauge_serial_number = d_input_widget_left->Entry_Item_Text(2);
			data->stepgauge_measurement_zero_position = d_input_widget_left->Entry_Item_Text(3).toDouble();
			data->stepgauge_zero_step_spacing = d_input_widget_left->Entry_Item_Text(4).toDouble();
			data->stepgauge_nominal_block_spacing = d_input_widget_left->Entry_Item_Text(5).toDouble();
			data->stepgauge_square_measurement_count = d_input_widget_left->Entry_Item_Text(6).toInt();
			data->axis_x_temperature = d_input_widget_right->Entry_Item_Text(0).toDouble();
			data->axis_y_temperature = d_input_widget_right->Entry_Item_Text(1).toDouble();
			data->axis_z_temperature = d_input_widget_right->Entry_Item_Text(2).toDouble();
			data->part_temperature = d_input_widget_right->Entry_Item_Text(3).toDouble();
			data->stepgauge_expansion_coeff = d_input_widget_right->Entry_Item_Text(4).toDouble();

			break;
			
		case MeasureTypes::MEASURE_GAUGEBLOCK:
			
			data->gaugeblock_position_name = d_input_widget_left->Entry_Item_Text(0);
			data->gaugeblock_serial_number = d_input_widget_left->Entry_Item_Text(1);
			data->gaugeblock_nominal_length = d_input_widget_left->Entry_Item_Text(2).toDouble();
			data->axis_x_temperature = d_input_widget_right->Entry_Item_Text(0).toDouble();
			data->axis_y_temperature = d_input_widget_right->Entry_Item_Text(1).toDouble();
			data->axis_z_temperature = d_input_widget_right->Entry_Item_Text(2).toDouble();
			data->part_temperature = d_input_widget_right->Entry_Item_Text(3).toDouble();
			data->gaugeblock_expansion_coeff = d_input_widget_right->Entry_Item_Text(4).toDouble();

			break;
			
		case MeasureTypes::MEASURE_POINT_REPEAT:
			
			data->point_repeat_description = d_input_widget_left->Entry_Item_Text(0);

			data->point_repeat_move_distance = d_input_widget_right->Entry_Item_Text(0).toDouble();
			data->point_repeat_measurement_count = d_input_widget_right->Entry_Item_Text(1).toInt();
			
			break;
			
		case MeasureTypes::MEASURE_SPHERE_REPEAT:
			
			data->sphere_nominal_diameter = d_input_widget_left->Entry_Item_Text(0).toDouble();
			data->sphere_repeat_measurement_count = d_input_widget_right->Entry_Item_Text(0).toInt();
			
			break;
			
		case MeasureTypes::MEASURE_SPHERE_PFTU:
			
			data->sphere_nominal_diameter = d_input_widget_left->Entry_Item_Text(0).toDouble();
			
			break;
			
		case MeasureTypes::MEASURE_BALLBAR_B89:
			
			data->ballbar_position_name = d_input_widget_left->Entry_Item_Text(0);
			data->ballbar_sphere_diameter = d_input_widget_left->Entry_Item_Text(1).toDouble();
			data->ballbar_measurement_count = d_input_widget_left->Entry_Item_Text(2).toInt();
			data->axis_x_temperature = d_input_widget_right->Entry_Item_Text(0).toDouble();
			data->axis_y_temperature = d_input_widget_right->Entry_Item_Text(1).toDouble();
			data->axis_z_temperature = d_input_widget_right->Entry_Item_Text(2).toDouble();
			data->part_temperature = d_input_widget_right->Entry_Item_Text(3).toDouble();
			data->ballbar_expansion_coeff = d_input_widget_right->Entry_Item_Text(4).toDouble();
			
			break;
			
		case MeasureTypes::MEASURE_BALLBAR_10360:
			
			data->ballbar_10360_position_name = d_input_widget_left->Entry_Item_Text(0);
			data->ballbar_10360_serial_number = d_input_widget_left->Entry_Item_Text(1);
			data->ballbar_10360_sphere_diameter = d_input_widget_left->Entry_Item_Text(2).toDouble();
			data->ballbar_10360_sphere0_offset = d_input_widget_left->Entry_Item_Text(3).toInt();
			data->ballbar_10360_sphere_spacing = d_input_widget_left->Entry_Item_Text(4).toDouble();
			data->axis_x_temperature = d_input_widget_right->Entry_Item_Text(0).toDouble();
			data->axis_y_temperature = d_input_widget_right->Entry_Item_Text(1).toDouble();
			data->axis_z_temperature = d_input_widget_right->Entry_Item_Text(2).toDouble();
			data->part_temperature = d_input_widget_right->Entry_Item_Text(3).toDouble();
			data->ballbar_10360_expansion_coeff = d_input_widget_right->Entry_Item_Text(4).toDouble();
			
			break;

		case MeasureTypes::MEASURE_RINGGAUGE:
			
			data->ringgauge_nominal_diameter = d_input_widget_left->Entry_Item_Text(0).toDouble();
			data->ringgauge_point_count = d_input_widget_right->Entry_Item_Text(0).toInt();
			
			break;
			
		case MeasureTypes::MEASURE_PINGAUGE:
			
			data->pingauge_nominal_diameter = d_input_widget_left->Entry_Item_Text(0).toDouble();
			data->pingauge_point_count = d_input_widget_right->Entry_Item_Text(0).toInt();
			
			break;
		
		case MeasureTypes::MEASURE_ROLL_OFFSET_TOOL:
			
			data->roll_tool1_name = d_input_widget_left->Entry_Item_Text(0);
			data->roll_tool2_name = d_input_widget_left->Entry_Item_Text(1);
			
			data->sphere_nominal_diameter = d_input_widget_right->Entry_Item_Text(0).toDouble();
					
			vec.i = d_input_widget_right->Entry_Item_Text(1).toDouble();
			vec.j = d_input_widget_right->Entry_Item_Text(2).toDouble();
			vec.k = d_input_widget_right->Entry_Item_Text(3).toDouble();
			
			if(vec.Length() > ZERO_EPSILON)
			{
				vec.Normal();
			}
			else
			{
				vec.Set(0,0,1);
			}
			
			data->sphere_stem_vector_i = vec.i;
			data->sphere_stem_vector_j = vec.j;
			data->sphere_stem_vector_k = vec.k;

			break;
		
		case MeasureTypes::CALIBRATE_TOOLS:
			
			data->sphere_nominal_diameter = d_input_widget_left->Entry_Item_Text(0).toDouble();
			
			vec.i = d_input_widget_right->Entry_Item_Text(0).toDouble();
			vec.j = d_input_widget_right->Entry_Item_Text(1).toDouble();
			vec.k = d_input_widget_right->Entry_Item_Text(2).toDouble();
			
			if(vec.Length() > ZERO_EPSILON)
			{
				vec.Normal();
			}
			else
			{
				vec.Set(0,0,1);
			}
			
			data->sphere_stem_vector_i = vec.i;
			data->sphere_stem_vector_j = vec.j;
			data->sphere_stem_vector_k = vec.k;

			break;
	}
}

void TMeasureDialog::Clear_Log(void)
{
	d_information_text->clear();
}

