/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QGridLayout>
#include <QLabel>
#include <QFrame>
#include <QSpacerItem>
#include <QDir>

#include "titlewidget.h"
#include "iconwidget.h"
#include "settings_machinedialog.h"
#include "settings_artifactdialog.h"
#include "informationwidget.h"

#include "settingswidget.h"

TSettingsWidget::TSettingsWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QGridLayout							*widget_layout;
	QSpacerItem							*control_hspacer;
	TTitleWidget						*title_widget;

	widget_layout = new QGridLayout(this);
	widget_layout->setContentsMargins(0,0,0,0);

	title_widget = new TTitleWidget(this);
	widget_layout->addWidget(title_widget,0,0,1,4);

	d_machine_settings_option = new TIconWidget(this);
	d_machine_settings_option->setSizePolicy(QSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred));
	widget_layout->addWidget(d_machine_settings_option,1,0,1,1);
	
	d_artifact_settings_option = new TIconWidget(this);
	d_artifact_settings_option->setSizePolicy(QSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred));
	widget_layout->addWidget(d_artifact_settings_option,1,1,1,1);

	control_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	widget_layout->addItem(control_hspacer,1,2,1,2);

	d_information_widget = new TInformationWidget(this);
	widget_layout->addWidget(d_information_widget,1,3,2,1);

	// defaults
	d_settings_machine_dialog = 0;
	d_settings_artifact_dialog = 0;

	d_machine_settings.move_speed = 100;
	d_machine_settings.touch_speed = 4;
	d_machine_settings.acceleration = 150;
	d_machine_settings.approach_distance = 4;
	
	assert(d_information_widget->Maximum_Row_Count() > 6);
	
	d_move_speed_index = 0;
	d_touch_speed_index = 1;
	d_acceleration_index = 2;
	d_approach_distance_index = 3;
	d_cte_x_index = 4;
	d_cte_y_index = 5;
	d_cte_z_index = 6;

	d_artifact_settings.sg_checkmaster_pln_x_start = 25.0;
	d_artifact_settings.sg_checkmaster_pln_x_end = -5.0;
	d_artifact_settings.sg_checkmaster_pln_y1 = 2.0;
	d_artifact_settings.sg_checkmaster_pln_y2 = 6.0;
	d_artifact_settings.sg_checkmaster_ln_x_start = 25.0;
	d_artifact_settings.sg_checkmaster_ln_x_end = -5.0;
	d_artifact_settings.sg_checkmaster_ln_z = -4.0;
	d_artifact_settings.sg_checkmaster_pnt_y = 4.0;
	d_artifact_settings.sg_checkmaster_pnt_z = -4.0;
	d_artifact_settings.sg_checkmaster_clear_z = 15.0;
	
	d_artifact_settings.sg_webber_clear_z = 25.0;
	
	d_artifact_settings.sg_koba_pln_x_start = 10.0;
	d_artifact_settings.sg_koba_pln_x_end = -10.0;
	d_artifact_settings.sg_koba_pln_y1 = 9.0;
	d_artifact_settings.sg_koba_pln_y2 = 46.0;
	d_artifact_settings.sg_koba_ln_x_start = 10.0;
	d_artifact_settings.sg_koba_ln_x_end = -10.0;
	d_artifact_settings.sg_koba_ln_z = -11.0;
	d_artifact_settings.sg_koba_pnt_y = 27.5;
	d_artifact_settings.sg_koba_pnt_z = -30.0;
	d_artifact_settings.sg_koba_clear_probe = 45.0;
	
	d_artifact_settings.gaugeblock_pln_y = 0.0;
	d_artifact_settings.gaugeblock_pln_z = 0.0;
	d_artifact_settings.gaugeblock_pln_sp = 2.0;
	d_artifact_settings.gaugeblock_clear_z = 25.0;
	
	d_artifact_settings.ballbar_clear_distance = 50.0;
	d_artifact_settings.ballbar_avoidance_angle = 60.0;
	
	d_artifact_settings.ballbar_10360_clear_distance = 50.0;
	d_artifact_settings.ballbar_10360_sphere_points = 13;

	d_artifact_settings.ringgauge_clear_z = 10.0;
	d_artifact_settings.ringgauge_pln_offset = 4.0;
	d_artifact_settings.ringgauge_cir_z = -4.0;
	
	d_artifact_settings.pingauge_clear_z = 10.0;
	d_artifact_settings.pingauge_pln_offset = -2.0;	// plane measured on top of pin
	d_artifact_settings.pingauge_cir_z = -4.0;		// circle measured below top of pin
	
	d_artifact_settings.plane_sample_radius = 3.0;
	
	d_artifact_settings.calibration_sphere_clear_z = 100.0;
	d_artifact_settings.calibration_sphere_points = 13;

	title_widget->Set_Text(QStringLiteral("Settings"));
	
	d_machine_settings_option->Set_Icon(QStringLiteral(":/icon64/machine_icon64.png"));
	d_machine_settings_option->Set_Text(QStringLiteral("Machine"));
	
	d_artifact_settings_option->Set_Icon(QStringLiteral(":/icon64/artifact_icon64.png"));
	d_artifact_settings_option->Set_Text(QStringLiteral("Artifacts"));
	
	d_information_widget->Set_Name_Text(d_touch_speed_index,QStringLiteral("Touch Speed:"));
	d_information_widget->Set_Name_Text(d_move_speed_index,QStringLiteral("Move Speed:"));
	d_information_widget->Set_Name_Text(d_acceleration_index,QStringLiteral("Acceleration:"));
	d_information_widget->Set_Name_Text(d_approach_distance_index,QStringLiteral("Approach Distance:"));
	d_information_widget->Set_Name_Text(d_cte_x_index,QStringLiteral("CTE X:"));
	d_information_widget->Set_Name_Text(d_cte_y_index,QStringLiteral("CTE Y:"));
	d_information_widget->Set_Name_Text(d_cte_z_index,QStringLiteral("CTE Z:"));

	connect(d_machine_settings_option,&TIconWidget::Clicked,this,&TSettingsWidget::Change_Machine_Settings);
	connect(d_artifact_settings_option,&TIconWidget::Clicked,this,&TSettingsWidget::Change_Artifact_Settings);
}

TSettingsWidget::~TSettingsWidget(void)
{
}

TSettingsWidget::TMachineSettings TSettingsWidget::Machine_Settings(void) const
{
	return d_machine_settings;
}

TSettingsWidget::TArtifactSettings TSettingsWidget::Artifact_Settings(void) const
{
	return d_artifact_settings;
}

void TSettingsWidget::Set_Machine_Settings(
	const TSettingsWidget::TMachineSettings &settings)
{
	d_machine_settings = settings;
	this->Update_Widget_Display();

}

void TSettingsWidget::Set_Artifact_Settings(
	const TSettingsWidget::TArtifactSettings &settings)
{
	d_artifact_settings = settings;
}

void TSettingsWidget::Reset(
	TController::TControllerType		controller_type)
{
	d_controller_type = controller_type;
}

void TSettingsWidget::Set_Temperature_Sensors_X(
	const QString						&sensors)
{
	if(d_settings_machine_dialog)
	{
		d_settings_machine_dialog->Set_Temperature_Sensors_X(sensors);
	}
	
	d_machine_settings.sensors_x = sensors;
}

void TSettingsWidget::Set_Temperature_Sensors_Y(
	const QString						&sensors)
{
	if(d_settings_machine_dialog)
	{
		d_settings_machine_dialog->Set_Temperature_Sensors_Y(sensors);
	}

	d_machine_settings.sensors_y = sensors;
}

void TSettingsWidget::Set_Temperature_Sensors_Z(
	const QString						&sensors)
{
	if(d_settings_machine_dialog)
	{
		d_settings_machine_dialog->Set_Temperature_Sensors_Z(sensors);
	}

	d_machine_settings.sensors_z = sensors;
}

void TSettingsWidget::Set_Temperature_Sensors_Part(
	const QString						&sensors)
{
	if(d_settings_machine_dialog)
	{
		d_settings_machine_dialog->Set_Temperature_Sensors_Part(sensors);
	}

	d_machine_settings.sensors_part = sensors;
}

void TSettingsWidget::Change_Machine_Settings(void)
{
	if(!d_settings_machine_dialog)
	{
		d_settings_machine_dialog = new TSettingsMachineDialog(this);
		
		connect(d_settings_machine_dialog,&TSettingsMachineDialog::Request_Sensor_IDs,this,&TSettingsWidget::Request_Sensor_IDs);
	}
	
	d_settings_machine_dialog->Reset(d_machine_settings,d_controller_type);
	
	if(QDialog::Accepted == d_settings_machine_dialog->exec())
	{
		d_machine_settings = d_settings_machine_dialog->Settings();
		this->Update_Widget_Display();

		emit Machine_Settings_Changed();
	}
}

void TSettingsWidget::Change_Artifact_Settings(void)
{
	if(!d_settings_artifact_dialog)
	{
		d_settings_artifact_dialog = new TSettingsArtifactDialog(this);
	}
	
	d_settings_artifact_dialog->Reset(d_artifact_settings);
	
	if(QDialog::Accepted == d_settings_artifact_dialog->exec())
	{
		d_artifact_settings = d_settings_artifact_dialog->Settings();
		
		emit Artifact_Settings_Changed();
	}
}

void TSettingsWidget::Update_Widget_Display(void)
{
	d_information_widget->Set_Value_Text(d_touch_speed_index,QString("%1 mm/sec").arg(d_machine_settings.touch_speed,0,'f',1));
	d_information_widget->Set_Value_Text(d_move_speed_index,QString("%1 mm/sec").arg(d_machine_settings.move_speed,0,'f',0));
	d_information_widget->Set_Value_Text(d_acceleration_index,QString("%1 mm/sec^2").arg(d_machine_settings.acceleration,0,'f',0));
	d_information_widget->Set_Value_Text(d_approach_distance_index,QString("%1 mm").arg(d_machine_settings.approach_distance,0,'f',1));
	d_information_widget->Set_Value_Text(d_cte_x_index,QString("%1 um/m/˚C").arg(d_machine_settings.expansion_coeff_x,0,'f',1));
	d_information_widget->Set_Value_Text(d_cte_y_index,QString("%1 um/m/˚C").arg(d_machine_settings.expansion_coeff_y,0,'f',1));
	d_information_widget->Set_Value_Text(d_cte_z_index,QString("%1 um/m/˚C").arg(d_machine_settings.expansion_coeff_z,0,'f',1));
}



