/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QDoubleSpinBox>
#include <QToolButton>
#include <QFrame>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QPushButton>
#include <QScrollArea>
#include <QSpacerItem>
#include <QWidget>
#include <QDateTime>
#include <QTimer>
#include <QFont>
#include <QSizePolicy>
#include <QList>
#include <assert.h>

#include "../../../core/gaugewidget.h"

#include "temperaturemonitordialog.h"

TTemperatureMonitorDialog::TTemperatureMonitorDialog(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QDialog(const_cast<QWidget*>(parent),flags)
{
	QScrollArea							*part_sensors_area;
	QScrollArea							*x_sensors_area;
	QScrollArea							*y_sensors_area;
	QScrollArea							*z_sensors_area;
	QFrame								*lower_hseparator_line;
	QFrame								*upper_hseparator_line;
	QGridLayout							*control_layout;
	QGridLayout							*dialog_layout;
	QHBoxLayout							*button_hlayout;
	QLabel								*lower_temperature_label;
	QLabel								*part_label;
	QLabel								*part_sensors_label;
	QLabel								*upper_temperature_label;
	QLabel								*x_axis_label;
	QLabel								*x_sensors_label;
	QLabel								*y_axis_label;
	QLabel								*y_sensors_label;
	QLabel								*z_axis_label;
	QLabel								*z_sensors_label;
	QPushButton							*close_button;
	QSpacerItem							*button_hspacer;
	QSpacerItem							*control_hspacer;
	QFont								gauge_font;
	QSizePolicy							sizepolicy_minimumexpanding_preferred(QSizePolicy::MinimumExpanding, QSizePolicy::Preferred);
	int									point_size;
	
	sizepolicy_minimumexpanding_preferred.setHorizontalStretch(0);
	sizepolicy_minimumexpanding_preferred.setVerticalStretch(0);

	this->resize(601,801);
	
	point_size = gauge_font.pointSize();
	
	if(!(point_size < 0))
	{
		point_size += 2;
		gauge_font.setPointSize(point_size);
	}

	dialog_layout = new QGridLayout(this);
	control_layout = new QGridLayout();

	upper_temperature_label = new QLabel(this);
	control_layout->addWidget(upper_temperature_label,0,0,1,1);

	d_upper_limit_spin = new QDoubleSpinBox(this);
	d_upper_limit_spin->setMinimum(20);
	d_upper_limit_spin->setMaximum(40);
	d_upper_limit_spin->setValue(25);
	control_layout->addWidget(d_upper_limit_spin,0,1,1,1);

	lower_temperature_label = new QLabel(this);
	control_layout->addWidget(lower_temperature_label,1,0,1,1);

	d_lower_limit_spin = new QDoubleSpinBox(this);
	d_lower_limit_spin->setMaximum(20);
	d_lower_limit_spin->setValue(15);
	control_layout->addWidget(d_lower_limit_spin,1,1,1,1);
	
	control_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	control_layout->addItem(control_hspacer,0,2,2,1);
	
	d_write_log_button = new QToolButton(this);
	d_write_log_button->setCheckable(true);
	control_layout->addWidget(d_write_log_button,0,3,1,1);

	dialog_layout->addLayout(control_layout,0,0,1,4);

	upper_hseparator_line = new QFrame(this);
	upper_hseparator_line->setFrameShape(QFrame::HLine);
	upper_hseparator_line->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(upper_hseparator_line,1,0,1,4);

	x_axis_label = new QLabel(this);
	x_axis_label->setFrameShape(QFrame::StyledPanel);
	dialog_layout->addWidget(x_axis_label,2,0,1,1);

	d_x_temperature_gauge = new TGaugeWidget(this);
	dialog_layout->addWidget(d_x_temperature_gauge,2,1,1,1);

	x_sensors_label = new QLabel(this);
	x_sensors_label->setFrameShape(QFrame::StyledPanel);
	x_sensors_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(x_sensors_label,2,2,1,1);

	x_sensors_area = new QScrollArea(this);
	x_sensors_area->setWidgetResizable(true);

	d_x_sensors_area_widget = new QWidget();
	x_sensors_area->setWidget(d_x_sensors_area_widget);

	d_x_sensors_hlayout = new QHBoxLayout(d_x_sensors_area_widget);
	d_x_sensors_hlayout->setContentsMargins(0,0,0,0);
	
	d_x_sensors_spacer_widget = new QWidget(d_x_sensors_area_widget);
	sizepolicy_minimumexpanding_preferred.setHeightForWidth(d_x_sensors_spacer_widget->sizePolicy().hasHeightForWidth());
	d_x_sensors_spacer_widget->setSizePolicy(sizepolicy_minimumexpanding_preferred);
	d_x_sensors_hlayout->addWidget(d_x_sensors_spacer_widget);
	
	dialog_layout->addWidget(x_sensors_area,2,3,1,1);

	y_axis_label = new QLabel(this);
	y_axis_label->setFrameShape(QFrame::StyledPanel);
	dialog_layout->addWidget(y_axis_label,3,0,1,1);

	d_y_temperature_gauge = new TGaugeWidget(this);
	dialog_layout->addWidget(d_y_temperature_gauge,3,1,1,1);

	y_sensors_label = new QLabel(this);
	y_sensors_label->setFrameShape(QFrame::StyledPanel);
	y_sensors_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(y_sensors_label,3,2,1,1);

	y_sensors_area = new QScrollArea(this);
	y_sensors_area->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);
	y_sensors_area->setSizeAdjustPolicy(QAbstractScrollArea::AdjustToContents);
	y_sensors_area->setWidgetResizable(true);

	d_y_sensors_area_widget = new QWidget();
	y_sensors_area->setWidget(d_y_sensors_area_widget);

	d_y_sensors_hlayout = new QHBoxLayout(d_y_sensors_area_widget);
	d_y_sensors_hlayout->setContentsMargins(0,0,0,0);

	d_y_sensors_spacer_widget = new QWidget(d_y_sensors_area_widget);
	sizepolicy_minimumexpanding_preferred.setHeightForWidth(d_y_sensors_spacer_widget->sizePolicy().hasHeightForWidth());
	d_y_sensors_spacer_widget->setSizePolicy(sizepolicy_minimumexpanding_preferred);
	d_y_sensors_hlayout->addWidget(d_y_sensors_spacer_widget);

	dialog_layout->addWidget(y_sensors_area,3,3,1,1);

	z_axis_label = new QLabel(this);
	z_axis_label->setFrameShape(QFrame::StyledPanel);
	dialog_layout->addWidget(z_axis_label,4,0,1,1);

	d_z_temperature_gauge = new TGaugeWidget(this);
	dialog_layout->addWidget(d_z_temperature_gauge,4,1,1,1);

	z_sensors_label = new QLabel(this);
	z_sensors_label->setFrameShape(QFrame::StyledPanel);
	z_sensors_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(z_sensors_label,4,2,1,1);

	z_sensors_area = new QScrollArea(this);
	z_sensors_area->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);
	z_sensors_area->setSizeAdjustPolicy(QAbstractScrollArea::AdjustToContents);
	z_sensors_area->setWidgetResizable(true);

	d_z_sensors_area_widget = new QWidget();
	z_sensors_area->setWidget(d_z_sensors_area_widget);

	d_z_sensors_hlayout = new QHBoxLayout(d_z_sensors_area_widget);
	d_z_sensors_hlayout->setContentsMargins(0,0,0,0);

	d_z_sensors_spacer_widget = new QWidget(d_z_sensors_area_widget);
	sizepolicy_minimumexpanding_preferred.setHeightForWidth(d_z_sensors_spacer_widget->sizePolicy().hasHeightForWidth());
	d_z_sensors_spacer_widget->setSizePolicy(sizepolicy_minimumexpanding_preferred);
	d_z_sensors_hlayout->addWidget(d_z_sensors_spacer_widget);

	dialog_layout->addWidget(z_sensors_area,4,3,1,1);

	part_label = new QLabel(this);
	part_label->setFrameShape(QFrame::StyledPanel);
	dialog_layout->addWidget(part_label,5,0,1,1);

	d_part_temperature_gauge = new TGaugeWidget(this);
	dialog_layout->addWidget(d_part_temperature_gauge,5,1,1,1);

	part_sensors_label = new QLabel(this);
	part_sensors_label->setFrameShape(QFrame::StyledPanel);
	part_sensors_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(part_sensors_label,5,2,1,1);

	part_sensors_area = new QScrollArea(this);
	part_sensors_area->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);
	part_sensors_area->setSizeAdjustPolicy(QAbstractScrollArea::AdjustToContents);
	part_sensors_area->setWidgetResizable(true);

	d_part_sensors_area_widget = new QWidget();
	part_sensors_area->setWidget(d_part_sensors_area_widget);

	d_part_sensors_hlayout = new QHBoxLayout(d_part_sensors_area_widget);
	d_part_sensors_hlayout->setContentsMargins(0,0,0,0);
	
	d_part_sensors_spacer_widget = new QWidget(d_part_sensors_area_widget);
	sizepolicy_minimumexpanding_preferred.setHeightForWidth(d_part_sensors_spacer_widget->sizePolicy().hasHeightForWidth());
	d_part_sensors_spacer_widget->setSizePolicy(sizepolicy_minimumexpanding_preferred);
	d_part_sensors_hlayout->addWidget(d_part_sensors_spacer_widget);

	dialog_layout->addWidget(part_sensors_area,5,3,1,1);

	lower_hseparator_line = new QFrame(this);
	lower_hseparator_line->setFrameShape(QFrame::HLine);
	lower_hseparator_line->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(lower_hseparator_line,6,0,1,4);

	button_hlayout = new QHBoxLayout();

	button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_hlayout->addItem(button_hspacer);

	close_button = new QPushButton(this);
	close_button->setAutoDefault(false);
	button_hlayout->addWidget(close_button);
	dialog_layout->addLayout(button_hlayout,7,0,1,4);
	
	d_update_timer = new QTimer(this);
	
	// defaults
	d_write_log_button->setChecked(false);

	this->setWindowTitle(QStringLiteral("Temperature Monitor"));
	
	upper_temperature_label->setText(QStringLiteral("Upper Temperature:"));
	lower_temperature_label->setText(QStringLiteral("Lower Temperature:"));
	d_write_log_button->setText(QStringLiteral("Log Temperatures"));
	x_axis_label->setText(QStringLiteral("X Axis:"));
	x_sensors_label->setText(QStringLiteral("X Sensors:"));
	y_axis_label->setText(QStringLiteral("Y Axis:"));
	y_sensors_label->setText(QStringLiteral("Y Sensors:"));
	z_axis_label->setText(QStringLiteral("Z Axis:"));
	z_sensors_label->setText(QStringLiteral("Z Sensors:"));
	part_label->setText(QStringLiteral("Part:"));
	part_sensors_label->setText(QStringLiteral("Part Sensors:"));
	close_button->setText(QStringLiteral("Close"));
	
	// defaults
	d_x_temperature_gauge->setFont(gauge_font);
	d_y_temperature_gauge->setFont(gauge_font);
	d_z_temperature_gauge->setFont(gauge_font);
	d_part_temperature_gauge->setFont(gauge_font);
	
	d_x_temperature_gauge->Set_Indicator_Type(TGaugeWidget::POINTER);
	d_y_temperature_gauge->Set_Indicator_Type(TGaugeWidget::POINTER);
	d_z_temperature_gauge->Set_Indicator_Type(TGaugeWidget::POINTER);
	d_part_temperature_gauge->Set_Indicator_Type(TGaugeWidget::POINTER);
	
	d_x_temperature_gauge->Enable_Value_Display(true);
	d_y_temperature_gauge->Enable_Value_Display(true);
	d_z_temperature_gauge->Enable_Value_Display(true);
	d_part_temperature_gauge->Enable_Value_Display(true);

	d_x_temperature_gauge->Set_Value_Display_Suffix(QStringLiteral("˚C"));
	d_y_temperature_gauge->Set_Value_Display_Suffix(QStringLiteral("˚C"));
	d_z_temperature_gauge->Set_Value_Display_Suffix(QStringLiteral("˚C"));
	d_part_temperature_gauge->Set_Value_Display_Suffix(QStringLiteral("˚C"));

	this->Min_Value_Changed(d_lower_limit_spin->value());
	this->Max_Value_Changed(d_upper_limit_spin->value());
	
	d_x_temperature_gauge->Set_Value(20.0);
	d_y_temperature_gauge->Set_Value(20.0);
	d_z_temperature_gauge->Set_Value(20.0);
	d_part_temperature_gauge->Set_Value(20.0);
	
	d_write_log_button->setEnabled(false);
	
	connect(close_button,&QPushButton::clicked,this,&TTemperatureMonitorDialog::Hide_Dialog);
	connect(d_lower_limit_spin,static_cast<void (QDoubleSpinBox::*)(double)>(&QDoubleSpinBox::valueChanged),this,&TTemperatureMonitorDialog::Min_Value_Changed);
	connect(d_upper_limit_spin,static_cast<void (QDoubleSpinBox::*)(double)>(&QDoubleSpinBox::valueChanged),this,&TTemperatureMonitorDialog::Max_Value_Changed);
	
	connect(d_update_timer,&QTimer::timeout,this,&TTemperatureMonitorDialog::Update_Sensors);
}

TTemperatureMonitorDialog::~TTemperatureMonitorDialog(void)
{
}

double TTemperatureMonitorDialog::Lower_Temperature_Limit(void) const
{
	return d_lower_limit_spin->value();
}

double TTemperatureMonitorDialog::Upper_Temperature_Limit(void) const
{
	return d_upper_limit_spin->value();
}

void TTemperatureMonitorDialog::Set_Lower_Temperature_Limit(
	const double						&value)
{
	d_lower_limit_spin->setValue(value);
}

void TTemperatureMonitorDialog::Set_Upper_Temperature_Limit(
	const double						&value)
{
	d_upper_limit_spin->setValue(value);
}

void TTemperatureMonitorDialog::Set_Temperature_Sensors_X(
	const QString						&sensors)
{
	QStringList							list;
	QList<QString>::const_iterator		iter;
	int									sensor_id;
	bool								state;
	
	this->Remove_X_Sensors();
	
	list = sensors.split(',');
	
	for(iter = list.begin();iter != list.end();++iter)
	{
		sensor_id = (*iter).toInt(&state);
		
		if(state && (sensor_id > 0))
		{
			this->Add_X_Sensor(sensor_id);
		}
	}
}

void TTemperatureMonitorDialog::Set_Temperature_Sensors_Y(
	const QString						&sensors)
{
	QStringList							list;
	QList<QString>::const_iterator		iter;
	int									sensor_id;
	bool								state;
	
	this->Remove_Y_Sensors();
	
	list = sensors.split(',');
	
	for(iter = list.begin();iter != list.end();++iter)
	{
		sensor_id = (*iter).toInt(&state);
		
		if(state && (sensor_id > 0))
		{
			this->Add_Y_Sensor(sensor_id);
		}
	}
}

void TTemperatureMonitorDialog::Set_Temperature_Sensors_Z(
	const QString						&sensors)
{
	QStringList							list;
	QList<QString>::const_iterator		iter;
	int									sensor_id;
	bool								state;
	
	this->Remove_Z_Sensors();
	
	list = sensors.split(',');
	
	for(iter = list.begin();iter != list.end();++iter)
	{
		sensor_id = (*iter).toInt(&state);
		
		if(state && (sensor_id > 0))
		{
			this->Add_Z_Sensor(sensor_id);
		}
	}
}

void TTemperatureMonitorDialog::Set_Temperature_Sensors_Part(
	const QString						&sensors)
{
	QStringList							list;
	QList<QString>::const_iterator		iter;
	int									sensor_id;
	bool								state;
	
	this->Remove_Part_Sensors();
	
	list = sensors.split(',');
	
	for(iter = list.begin();iter != list.end();++iter)
	{
		sensor_id = (*iter).toInt(&state);
		
		if(state && (sensor_id > 0))
		{
			this->Add_Part_Sensor(sensor_id);
		}
	}
}

void TTemperatureMonitorDialog::Enable_Log_Option(void)
{
	std::vector<TGaugeWidget*>::iterator iter;
	
	for(iter = d_x_sensors.begin();iter != d_x_sensors.end();++iter)
	{
		if((*iter)->User_Id() > 0)
		{
			d_write_log_button->setEnabled(true);
			return;
		}
	}
	
	for(iter = d_y_sensors.begin();iter != d_y_sensors.end();++iter)
	{
		if((*iter)->User_Id() > 0)
		{
			d_write_log_button->setEnabled(true);
			return;
		}
	}
	
	for(iter = d_z_sensors.begin();iter != d_z_sensors.end();++iter)
	{
		if((*iter)->User_Id() > 0)
		{
			d_write_log_button->setEnabled(true);
			return;
		}
	}
	
	for(iter = d_part_sensors.begin();iter != d_part_sensors.end();++iter)
	{
		if((*iter)->User_Id() > 0)
		{
			d_write_log_button->setEnabled(true);
			return;
		}
	}
		
	d_write_log_button->setEnabled(false);
	d_write_log_button->setChecked(false);
}

void TTemperatureMonitorDialog::Disable_Log_Option(void)
{
	d_write_log_button->setChecked(false);
}

void TTemperatureMonitorDialog::Remove_Part_Sensors(void)
{
	std::vector<TGaugeWidget*>::iterator iter;
	
	for(iter = d_part_sensors.begin();iter != d_part_sensors.end();++iter)
	{
		(*iter)->Set_User_Id(-1);
		(*iter)->hide();
	}
}

void TTemperatureMonitorDialog::Set_Sensor_Value(
	int 								sensor_id,
	const double 						&value,
	const QString						&log_path)
{
	std::vector<TGaugeWidget*>::iterator iter;
	double								dval;
	int									ival;
	QString								file_name;
	
	for(iter = d_x_sensors.begin();iter != d_x_sensors.end();++iter)
	{
		if((*iter)->User_Id() == sensor_id)
		{
			(*iter)->Set_Value(value);
			break;
		}
	}

	for(iter = d_y_sensors.begin();iter != d_y_sensors.end();++iter)
	{
		if((*iter)->User_Id() == sensor_id)
		{
			(*iter)->Set_Value(value);
			break;
		}
	}

	for(iter = d_z_sensors.begin();iter != d_z_sensors.end();++iter)
	{
		if((*iter)->User_Id() == sensor_id)
		{
			(*iter)->Set_Value(value);
			break;
		}
	}

	for(iter = d_part_sensors.begin();iter != d_part_sensors.end();++iter)
	{
		if((*iter)->User_Id() == sensor_id)
		{
			(*iter)->Set_Value(value);
			break;
		}
	}
	
	// update X average
	dval = 0.0;
	ival = 0;
	for(iter = d_x_sensors.begin();iter != d_x_sensors.end();++iter)
	{		
		if((*iter)->User_Id() > 0)
		{
			dval += (*iter)->Value();
			ival++;
		}
	}
	
	if(ival)
	{
		dval /= static_cast<double>(ival);
		d_x_temperature_gauge->Set_Value(dval);
	}
	else
	{
		d_x_temperature_gauge->Set_Value(20.0);
	}
	
		
	// update Y average
	dval = 0.0;
	ival = 0;
	for(iter = d_y_sensors.begin();iter != d_y_sensors.end();++iter)
	{		
		if((*iter)->User_Id() > 0)
		{
			dval += (*iter)->Value();
			ival++;
		}
	}
	
	if(ival)
	{
		dval /= static_cast<double>(ival);
		d_y_temperature_gauge->Set_Value(dval);
	}
	else
	{
		d_y_temperature_gauge->Set_Value(20.0);
	}
		
	// update Z average
	dval = 0.0;
	ival = 0;
	for(iter = d_z_sensors.begin();iter != d_z_sensors.end();++iter)
	{		
		if((*iter)->User_Id() > 0)
		{
			dval += (*iter)->Value();
			ival++;
		}
	}
	
	if(ival)
	{
		dval /= static_cast<double>(ival);
		d_z_temperature_gauge->Set_Value(dval);
	}
	else
	{
		d_z_temperature_gauge->Set_Value(20.0);
	}

	
	// update part average
	dval = 0.0;
	ival = 0;
	for(iter = d_part_sensors.begin();iter != d_part_sensors.end();++iter)
	{		
		if((*iter)->User_Id() > 0)
		{
			dval += (*iter)->Value();
			ival++;
		}
	}
	
	if(ival)
	{
		dval /= static_cast<double>(ival);
		d_part_temperature_gauge->Set_Value(dval);
	}
	else
	{
		d_part_temperature_gauge->Set_Value(20.0);
	}
	
	if(d_write_log_button->isChecked())
	{
		file_name = log_path;
		file_name.append(QString("/Temperature_Log"));
		file_name.append(QDateTime::currentDateTime().toString(" [ddd MMMM d yyyy]"));
		file_name.append(QStringLiteral(".dat"));
		
		this->Open_Temperature_Log(file_name);
		this->Write_Temperature_Log(sensor_id,value);
		
		d_temperature_log_file.close();
	}
}

void TTemperatureMonitorDialog::Hide_Dialog(void)
{
	this->hide();
}


void TTemperatureMonitorDialog::Min_Value_Changed(
	double								value)
{
	std::vector<TGaugeWidget*>::iterator iter;

	d_x_temperature_gauge->Set_Minimum_Value(value);
	d_y_temperature_gauge->Set_Minimum_Value(value);
	d_z_temperature_gauge->Set_Minimum_Value(value);
	d_part_temperature_gauge->Set_Minimum_Value(value);
	
	for(iter = d_x_sensors.begin();iter != d_x_sensors.end();++iter)
	{
		(*iter)->Set_Minimum_Value(value);
	}
	
	for(iter = d_y_sensors.begin();iter != d_y_sensors.end();++iter)
	{
		(*iter)->Set_Minimum_Value(value);
	}
	
	for(iter = d_z_sensors.begin();iter != d_z_sensors.end();++iter)
	{
		(*iter)->Set_Minimum_Value(value);
	}
	
	for(iter = d_part_sensors.begin();iter != d_part_sensors.end();++iter)
	{
		(*iter)->Set_Minimum_Value(value);
	}
	
	emit Temperature_Limits_Changed();
}

void TTemperatureMonitorDialog::Max_Value_Changed(
	double								value)
{
	std::vector<TGaugeWidget*>::iterator iter;

	d_x_temperature_gauge->Set_Maximum_Value(value);
	d_y_temperature_gauge->Set_Maximum_Value(value);
	d_z_temperature_gauge->Set_Maximum_Value(value);
	d_part_temperature_gauge->Set_Maximum_Value(value);
	
	for(iter = d_x_sensors.begin();iter != d_x_sensors.end();++iter)
	{
		(*iter)->Set_Maximum_Value(value);
	}
	
	for(iter = d_y_sensors.begin();iter != d_y_sensors.end();++iter)
	{
		(*iter)->Set_Maximum_Value(value);
	}
	
	for(iter = d_z_sensors.begin();iter != d_z_sensors.end();++iter)
	{
		(*iter)->Set_Maximum_Value(value);
	}
	
	for(iter = d_part_sensors.begin();iter != d_part_sensors.end();++iter)
	{
		(*iter)->Set_Maximum_Value(value);
	}
	
	emit Temperature_Limits_Changed();
}

void TTemperatureMonitorDialog::Add_X_Sensor(
	const int							id)
{
	std::vector<TGaugeWidget*>::iterator iter;
	TGaugeWidget						*sensor_gauge;
	QString								text;
	
	if(id < 0)
	{
		return;
	}
	
	this->Remove_X_Sensor(id);
	
	for(iter = d_x_sensors.begin();iter != d_x_sensors.end();++iter)
	{
		if((*iter)->User_Id() < 0)	// found free one
		{
			(*iter)->Set_User_Id(id);
			(*iter)->show();
			
			text = QString("Sensor %1").arg(id);
			(*iter)->Set_Display_Text(text);
			
			return;
		}
	}
	
	// must add a new sensor gauge
	d_x_sensors_hlayout->removeWidget(d_x_sensors_spacer_widget);
	
	sensor_gauge = new TGaugeWidget(d_x_sensors_area_widget);
	
	sensor_gauge->setFont(d_x_temperature_gauge->font());
	sensor_gauge->Set_Precision(d_x_temperature_gauge->Precision());
	sensor_gauge->Set_Minimum_Value(d_x_temperature_gauge->Minimum());
	sensor_gauge->Set_Maximum_Value(d_x_temperature_gauge->Maximum());
	sensor_gauge->Set_Value(d_x_temperature_gauge->Value());
	sensor_gauge->Set_User_Id(id);
	
	sensor_gauge->Enable_Value_Display(true);
	text = QString("Sensor %1").arg(id);
	sensor_gauge->Set_Display_Text(text);
	
	d_x_sensors.push_back(sensor_gauge);
	
	d_x_sensors_hlayout->addWidget(sensor_gauge);
	d_x_sensors_hlayout->addWidget(d_x_sensors_spacer_widget);
}

void TTemperatureMonitorDialog::Add_Y_Sensor(
	const int							id)
{
	std::vector<TGaugeWidget*>::iterator iter;
	TGaugeWidget						*sensor_gauge;
	QString								text;
	
	if(id < 0)
	{
		return;
	}
	
	this->Remove_Y_Sensor(id);
	
	for(iter = d_y_sensors.begin();iter != d_y_sensors.end();++iter)
	{
		if((*iter)->User_Id() < 0)	// found free one
		{
			(*iter)->Set_User_Id(id);
			(*iter)->show();
			
			text = QString("Sensor %1").arg(id);
			(*iter)->Set_Display_Text(text);
			
			return;
		}
	}
	
	// must add a new sensor gauge
	d_y_sensors_hlayout->removeWidget(d_y_sensors_spacer_widget);
	
	sensor_gauge = new TGaugeWidget(d_y_sensors_area_widget);
	
	sensor_gauge->setFont(d_y_temperature_gauge->font());
	sensor_gauge->Set_Precision(d_y_temperature_gauge->Precision());
	sensor_gauge->Set_Minimum_Value(d_y_temperature_gauge->Minimum());
	sensor_gauge->Set_Maximum_Value(d_y_temperature_gauge->Maximum());
	sensor_gauge->Set_Value(d_y_temperature_gauge->Value());
	sensor_gauge->Set_User_Id(id);
	
	sensor_gauge->Enable_Value_Display(true);
	text = QString("Sensor %1").arg(id);
	sensor_gauge->Set_Display_Text(text);
	
	d_y_sensors.push_back(sensor_gauge);
	
	d_y_sensors_hlayout->addWidget(sensor_gauge);
	d_y_sensors_hlayout->addWidget(d_y_sensors_spacer_widget);
}

void TTemperatureMonitorDialog::Add_Z_Sensor(
	const int							id)
{
	std::vector<TGaugeWidget*>::iterator iter;
	TGaugeWidget						*sensor_gauge;
	QString								text;
	
	if(id < 0)
	{
		return;
	}
	
	this->Remove_Z_Sensor(id);
	
	for(iter = d_z_sensors.begin();iter != d_z_sensors.end();++iter)
	{
		if((*iter)->User_Id() < 0)	// found free one
		{
			(*iter)->Set_User_Id(id);
			(*iter)->show();
			
			text = QString("Sensor %1").arg(id);
			(*iter)->Set_Display_Text(text);
			
			return;
		}
	}
	
	// must add a new sensor gauge
	d_z_sensors_hlayout->removeWidget(d_z_sensors_spacer_widget);
	
	sensor_gauge = new TGaugeWidget(d_z_sensors_area_widget);
	
	sensor_gauge->setFont(d_z_temperature_gauge->font());
	sensor_gauge->Set_Precision(d_z_temperature_gauge->Precision());
	sensor_gauge->Set_Minimum_Value(d_z_temperature_gauge->Minimum());
	sensor_gauge->Set_Maximum_Value(d_z_temperature_gauge->Maximum());
	sensor_gauge->Set_Value(d_z_temperature_gauge->Value());
	sensor_gauge->Set_User_Id(id);
	
	sensor_gauge->Enable_Value_Display(true);
	text = QString("Sensor %1").arg(id);
	sensor_gauge->Set_Display_Text(text);
	
	d_z_sensors.push_back(sensor_gauge);
	
	d_z_sensors_hlayout->addWidget(sensor_gauge);
	d_z_sensors_hlayout->addWidget(d_z_sensors_spacer_widget);
}

void TTemperatureMonitorDialog::Add_Part_Sensor(
	const int							id)
{
	std::vector<TGaugeWidget*>::iterator iter;
	TGaugeWidget						*sensor_gauge;
	QString								text;
	
	if(id < 0)
	{
		return;
	}
	
	this->Remove_Part_Sensor(id);
	
	for(iter = d_part_sensors.begin();iter != d_part_sensors.end();++iter)
	{
		if((*iter)->User_Id() < 0)	// found free one
		{
			(*iter)->Set_User_Id(id);
			(*iter)->show();
			
			text = QString("Sensor %1").arg(id);
			(*iter)->Set_Display_Text(text);
			
			return;
		}
	}
	
	// must add a new sensor gauge
	d_part_sensors_hlayout->removeWidget(d_part_sensors_spacer_widget);
	
	sensor_gauge = new TGaugeWidget(d_part_sensors_area_widget);
	
	sensor_gauge->setFont(d_part_temperature_gauge->font());
	sensor_gauge->Set_Precision(d_part_temperature_gauge->Precision());
	sensor_gauge->Set_Minimum_Value(d_part_temperature_gauge->Minimum());
	sensor_gauge->Set_Maximum_Value(d_part_temperature_gauge->Maximum());
	sensor_gauge->Set_Value(d_part_temperature_gauge->Value());
	sensor_gauge->Set_User_Id(id);
	
	sensor_gauge->Enable_Value_Display(true);
	text = QString("Sensor %1").arg(id);
	sensor_gauge->Set_Display_Text(text);
	
	d_part_sensors.push_back(sensor_gauge);
	
	d_part_sensors_hlayout->addWidget(sensor_gauge);
	d_part_sensors_hlayout->addWidget(d_part_sensors_spacer_widget);
}

void TTemperatureMonitorDialog::Remove_X_Sensor(
	const int							id)
{
	std::vector<TGaugeWidget*>::iterator iter;
	
	for(iter = d_x_sensors.begin();iter != d_x_sensors.end();++iter)
	{
		if((*iter)->User_Id() == id)
		{
			(*iter)->Set_User_Id(-1);
			(*iter)->hide();
			
			return;
		}
	}
}

void TTemperatureMonitorDialog::Remove_Y_Sensor(
	const int							id)
{
	std::vector<TGaugeWidget*>::iterator iter;
	
	for(iter = d_y_sensors.begin();iter != d_y_sensors.end();++iter)
	{
		if((*iter)->User_Id() == id)
		{
			(*iter)->Set_User_Id(-1);
			(*iter)->hide();
			
			return;
		}
	}
}

void TTemperatureMonitorDialog::Remove_Z_Sensor(
	const int							id)
{
	std::vector<TGaugeWidget*>::iterator iter;
	
	for(iter = d_z_sensors.begin();iter != d_z_sensors.end();++iter)
	{
		if((*iter)->User_Id() == id)
		{
			(*iter)->Set_User_Id(-1);
			(*iter)->hide();
			
			return;
		}
	}
}

void TTemperatureMonitorDialog::Remove_Part_Sensor(
	const int							id)
{
	std::vector<TGaugeWidget*>::iterator iter;
	
	for(iter = d_part_sensors.begin();iter != d_part_sensors.end();++iter)
	{
		if((*iter)->User_Id() == id)
		{
			(*iter)->Set_User_Id(-1);
			(*iter)->hide();
			
			return;
		}
	}
}

void TTemperatureMonitorDialog::Remove_X_Sensors(void)
{
	std::vector<TGaugeWidget*>::iterator iter;
	
	for(iter = d_x_sensors.begin();iter != d_x_sensors.end();++iter)
	{
		(*iter)->Set_User_Id(-1);
		(*iter)->hide();
	}
}

void TTemperatureMonitorDialog::Remove_Y_Sensors(void)
{
	std::vector<TGaugeWidget*>::iterator iter;
	
	for(iter = d_y_sensors.begin();iter != d_y_sensors.end();++iter)
	{
		(*iter)->Set_User_Id(-1);
		(*iter)->hide();
	}
}

void TTemperatureMonitorDialog::Remove_Z_Sensors(void)
{
	std::vector<TGaugeWidget*>::iterator iter;
	
	for(iter = d_z_sensors.begin();iter != d_z_sensors.end();++iter)
	{
		(*iter)->Set_User_Id(-1);
		(*iter)->hide();
	}
}

void TTemperatureMonitorDialog::Update_Sensors(void)
{
	std::vector<TGaugeWidget*>::iterator iter;
	
	switch(d_update_item_cntr)
	{
		case 0:
			
			for(iter = d_x_sensors.begin();iter != d_x_sensors.end();++iter)
			{
				if((*iter)->User_Id() > 0)
				{
					emit Request_Sensor_Value((*iter)->User_Id());
				}
			}
			
			++d_update_item_cntr;
			break;
			
		case 1:
			for(iter = d_y_sensors.begin();iter != d_y_sensors.end();++iter)
			{
				if((*iter)->User_Id() > 0)
				{
					emit Request_Sensor_Value((*iter)->User_Id());
				}
			}
			
			++d_update_item_cntr;
			break;
			
		case 2:
			for(iter = d_z_sensors.begin();iter != d_z_sensors.end();++iter)
			{
				if((*iter)->User_Id() > 0)
				{
					emit Request_Sensor_Value((*iter)->User_Id());
				}
			}
			
			++d_update_item_cntr;
			break;
			
		case 3:
			for(iter = d_part_sensors.begin();iter != d_part_sensors.end();++iter)
			{
				if((*iter)->User_Id() > 0)
				{
					emit Request_Sensor_Value((*iter)->User_Id());
				}
			}
			
			d_update_item_cntr = 0;
			break;
			
		default:
			d_update_item_cntr = 0;
			break;
			
	}
}

void TTemperatureMonitorDialog::showEvent(
	QShowEvent							*event)
{
	Q_UNUSED(event);
	
	static const int					UPDATE_INTERVAL(10000);	// 10 seconds
	
	if(d_update_timer->remainingTime() < 0)
	{
		this->Update_All_Sensors();				// update immediately
	}
	
	d_update_item_cntr = 0;
	d_update_timer->start(UPDATE_INTERVAL);
}

void TTemperatureMonitorDialog::hideEvent(
	QHideEvent							*event)
{
	Q_UNUSED(event);
	
	// only stop temperature events if there is no log
	if(!d_write_log_button->isChecked())
	{
		d_update_timer->stop();
	}
}


void TTemperatureMonitorDialog::Update_All_Sensors(void)
{
	d_update_item_cntr = 0;
	
	// force cycle sensors groups 0-3
	this->Update_Sensors();
	this->Update_Sensors();
	this->Update_Sensors();
	this->Update_Sensors();
}

bool TTemperatureMonitorDialog::Open_Temperature_Log(
	const QString						&file_name)
{
	if(d_temperature_log_file.isOpen())
	{
		d_temperature_log_file.close();
	}
	
	d_temperature_log_file.setFileName(file_name);
	
	if(d_temperature_log_file.exists())
	{
		d_temperature_log_file.open(QIODevice::Append | QIODevice::Text);
	}
	else
	{
		if(d_temperature_log_file.open(QIODevice::WriteOnly | QIODevice::Text))
		{
			d_temperature_log_file.write("Temperature_Log:Version=1:Type=Machine_Sensors\n\n");
			d_temperature_log_file.write("# [hhmmss],<sensor_id>,<temperature>\n\n");
		}
	}
	
	return d_temperature_log_file.isOpen();
}

bool TTemperatureMonitorDialog::Write_Temperature_Log(
	int									sensor_id,
	const double						&temperature)
{
	QString								log_text;
	
	if(d_temperature_log_file.isOpen())
	{
		log_text = QDateTime::currentDateTime().toString("[hhmmss]");
		log_text.append(QString(",%1").arg(sensor_id));
		log_text.append(QString(",%1\n").arg(temperature,0,'f',2));
		
		d_temperature_log_file.write(log_text.toLatin1());
	}
	
	return true;
}


