/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QComboBox>
#include <QFrame>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QListWidget>
#include <QListWidgetItem>
#include <QPushButton>
#include <QCheckBox>
#include <QSpacerItem>
#include <QStringList>
#include <QToolBox>
#include <QToolButton>
#include <QVBoxLayout>
#include <QWidget>
#include <QList>
#include <assert.h>

#include "nameeditdialog.h"
#include "toolbuilderdialog.h"
#include "tooladdanglesdialog.h"

#include "toolutilitiesdialog.h"

TToolUtilitiesDialog::TToolUtilitiesDialog(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QDialog(const_cast<QWidget*>(parent),flags)
{
	QFrame								*dialog_hline_separator;
	QFrame								*tool_configuration_upper_hline;
	QFrame								*tool_configuration_lower_hline;
	QGridLayout							*configuration_page_layout;
	QGridLayout							*orientation_page_layout;
	QGridLayout							*tooldata_page_layout;
	QHBoxLayout							*tooldata_angles_edit_hlayout;
	QHBoxLayout							*button_hlayout;
	QGridLayout							*offset_control_layout;
	QLabel								*orientation_a0b0_label;
	QLabel								*tool_name_label;
	QLabel								*tooldata_tip_diameter_label;
	QLabel								*tooldata_tip_average_diameter_label;
	QLabel								*tool_configuration_label;
	QLabel								*tool_options_label;
	QLabel								*orientation_a90b0_label;
	QLabel								*tooldata_angles_label;
	QLabel								*tool_probe_image_label;
	QLabel								*tool_offset_x_fixed_label;
	QLabel								*tool_offset_x_fixed_units_label;
	QLabel								*tool_offset_y_fixed_label;
	QLabel								*tool_offset_y_fixed_units_label;
	QLabel								*tool_offset_z_fixed_label;
	QLabel								*tool_offset_z_fixed_units_label;
	QLabel								*tool_offset_x_rot_label;
	QLabel								*tool_offset_x_rot_units_label;
	QLabel								*tool_offset_y_rot_label;
	QLabel								*tool_offset_y_rot_units_label;
	QLabel								*tool_offset_z_rot_label;
	QLabel								*tool_offset_z_rot_units_label;
	QPushButton							*close_button;
	QSpacerItem							*tooldata_angles_edit_hspacer;
	QSpacerItem							*button_hspacer;
	QSpacerItem							*configuration_page_vspacer;
	QSpacerItem							*orientation_hspacer;
	QSpacerItem							*orientation_page_vspacer;
	QSpacerItem							*tool_configuration_hspacer;
	QSpacerItem							*tool_data_hspacer;
	QSpacerItem							*tool_control_vspacer;
	QVBoxLayout							*dialog_vlayout;
	QWidget								*orientation_page;
	QSizePolicy							size_policy_preferred_fixed(QSizePolicy::Preferred, QSizePolicy::Fixed);
	
	size_policy_preferred_fixed.setHorizontalStretch(0);
	size_policy_preferred_fixed.setVerticalStretch(0);

	this->resize(763,620);

	dialog_vlayout = new QVBoxLayout(this);

	d_toolbox = new QToolBox(this);

	// orientation page
	orientation_page = new QWidget();
	d_toolbox->addItem(orientation_page,QStringLiteral("Probe Head Orientation"));

	orientation_page_layout = new QGridLayout(orientation_page);

	orientation_a0b0_label = new QLabel(orientation_page);
	orientation_page_layout->addWidget(orientation_a0b0_label,0,0,1,1);

	d_orientation_a0b0_combo = new QComboBox(orientation_page);
	orientation_page_layout->addWidget(d_orientation_a0b0_combo,0,1,1,1);

	orientation_a90b0_label = new QLabel(orientation_page);
	orientation_page_layout->addWidget(orientation_a90b0_label,1,0,1,1);

	d_orientation_a90b0_combo = new QComboBox(orientation_page);
	orientation_page_layout->addWidget(d_orientation_a90b0_combo,1,1,1,1);

	orientation_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	orientation_page_layout->addItem(orientation_hspacer,0,2,2,1);
	
	d_orientation_valid_label = new QLabel(orientation_page);
	d_orientation_valid_label->setAlignment(Qt::AlignLeft|Qt::AlignVCenter);
	orientation_page_layout->addWidget(d_orientation_valid_label,1,3,1,1);

	orientation_page_vspacer = new QSpacerItem(0,0,QSizePolicy::Minimum,QSizePolicy::Expanding);
	orientation_page_layout->addItem(orientation_page_vspacer,2,0,1,4);

	
	// tool configuration page
	d_tool_configuration_page = new QWidget();
	d_toolbox->addItem(d_tool_configuration_page,QStringLiteral("Tool Configuration"));

	configuration_page_layout = new QGridLayout(d_tool_configuration_page);

	tool_name_label = new QLabel(d_tool_configuration_page);
	configuration_page_layout->addWidget(tool_name_label,0,0,1,1);
	
	d_tool_name_combo = new QComboBox(d_tool_configuration_page);
	size_policy_preferred_fixed.setHeightForWidth(d_tool_name_combo->sizePolicy().hasHeightForWidth());
	d_tool_name_combo->setSizePolicy(size_policy_preferred_fixed);
	d_tool_name_combo->setInsertPolicy(QComboBox::InsertAlphabetically);
	configuration_page_layout->addWidget(d_tool_name_combo,0,1,1,1);
	
	d_tool_add_name_button = new QToolButton(d_tool_configuration_page);
	d_tool_add_name_button->setMinimumSize(16,16);
	d_tool_add_name_button->setMaximumSize(16,16);
	d_tool_add_name_button->setIconSize(QSize(16,16));
	configuration_page_layout->addWidget(d_tool_add_name_button,0,2,1,1);

	d_tool_remove_name_button = new QToolButton(d_tool_configuration_page);
	d_tool_remove_name_button->setMinimumSize(16,16);
	d_tool_remove_name_button->setMaximumSize(16,16);
	d_tool_remove_name_button->setIconSize(QSize(16,16));
	configuration_page_layout->addWidget(d_tool_remove_name_button,0,3,1,1);
	
	tool_configuration_upper_hline = new QFrame(d_tool_configuration_page);
	tool_configuration_upper_hline->setFrameShape(QFrame::HLine);
	tool_configuration_upper_hline->setFrameShadow(QFrame::Sunken);
	configuration_page_layout->addWidget(tool_configuration_upper_hline,1,0,1,4);

	tool_configuration_label = new QLabel(d_tool_configuration_page);
	tool_configuration_label->setAlignment(Qt::AlignLeading|Qt::AlignLeft|Qt::AlignTop);
	configuration_page_layout->addWidget(tool_configuration_label,2,0,2,1);
	
	d_tool_configuration_combo = new QComboBox(d_tool_configuration_page);
	size_policy_preferred_fixed.setHeightForWidth(d_tool_configuration_combo->sizePolicy().hasHeightForWidth());
	d_tool_configuration_combo->setSizePolicy(size_policy_preferred_fixed);
	d_tool_configuration_combo->setInsertPolicy(QComboBox::InsertAlphabetically);
	configuration_page_layout->addWidget(d_tool_configuration_combo,2,1,1,1);
	
	d_tool_add_configuration_button = new QToolButton(d_tool_configuration_page);
	d_tool_add_configuration_button->setMinimumSize(16,16);
	d_tool_add_configuration_button->setMaximumSize(16,16);
	d_tool_add_configuration_button->setIconSize(QSize(16,16));
	configuration_page_layout->addWidget(d_tool_add_configuration_button,2,2,1,1);
	
	d_tool_remove_configuration_button = new QToolButton(d_tool_configuration_page);
	d_tool_remove_configuration_button->setMinimumSize(16,16);
	d_tool_remove_configuration_button->setMaximumSize(16,16);
	d_tool_remove_configuration_button->setIconSize(QSize(16,16));
	configuration_page_layout->addWidget(d_tool_remove_configuration_button,2,3,1,1);

	offset_control_layout = new QGridLayout();
	
	tool_probe_image_label = new QLabel(this);
	tool_probe_image_label->setMinimumSize(200,200);
	tool_probe_image_label->setMaximumSize(200,200);
	offset_control_layout->addWidget(tool_probe_image_label,0,0,7,1);
	
	tool_offset_x_fixed_label = new QLabel(this);
	tool_offset_x_fixed_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	offset_control_layout->addWidget(tool_offset_x_fixed_label,0,1,1,1);
	
	d_tool_offset_x_fixed_edit = new QLineEdit(this);
	d_tool_offset_x_fixed_edit->setAlignment(Qt::AlignCenter);
	offset_control_layout->addWidget(d_tool_offset_x_fixed_edit,0,2,1,1);
	
	tool_offset_x_fixed_units_label = new QLabel(this);
	offset_control_layout->addWidget(tool_offset_x_fixed_units_label,0,3,1,1);
	
	tool_offset_y_fixed_label = new QLabel(this);
	tool_offset_y_fixed_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	offset_control_layout->addWidget(tool_offset_y_fixed_label,1,1,1,1);
	
	d_tool_offset_y_fixed_edit = new QLineEdit(this);
	d_tool_offset_y_fixed_edit->setAlignment(Qt::AlignCenter);
	offset_control_layout->addWidget(d_tool_offset_y_fixed_edit,1,2,1,1);
	
	tool_offset_y_fixed_units_label = new QLabel(this);
	offset_control_layout->addWidget(tool_offset_y_fixed_units_label,1,3,1,1);
	
	tool_offset_z_fixed_label = new QLabel(this);
	tool_offset_z_fixed_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	offset_control_layout->addWidget(tool_offset_z_fixed_label,2,1,1,1);
	
	d_tool_offset_z_fixed_edit = new QLineEdit(this);
	d_tool_offset_z_fixed_edit->setAlignment(Qt::AlignCenter);
	offset_control_layout->addWidget(d_tool_offset_z_fixed_edit,2,2,1,1);
	
	tool_offset_z_fixed_units_label = new QLabel(this);
	offset_control_layout->addWidget(tool_offset_z_fixed_units_label,2,3,1,1);
	
	tool_offset_x_rot_label = new QLabel(this);
	tool_offset_x_rot_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	offset_control_layout->addWidget(tool_offset_x_rot_label,3,1,1,1);
	
	d_tool_offset_x_rot_edit = new QLineEdit(this);
	d_tool_offset_x_rot_edit->setAlignment(Qt::AlignCenter);
	offset_control_layout->addWidget(d_tool_offset_x_rot_edit,3,2,1,1);
	
	tool_offset_x_rot_units_label = new QLabel(this);
	offset_control_layout->addWidget(tool_offset_x_rot_units_label,3,3,1,1);
	
	tool_offset_y_rot_label = new QLabel(this);
	tool_offset_y_rot_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	offset_control_layout->addWidget(tool_offset_y_rot_label,4,1,1,1);
	
	d_tool_offset_y_rot_edit = new QLineEdit(this);
	d_tool_offset_y_rot_edit->setAlignment(Qt::AlignCenter);
	offset_control_layout->addWidget(d_tool_offset_y_rot_edit,4,2,1,1);
	
	tool_offset_y_rot_units_label = new QLabel(this);
	offset_control_layout->addWidget(tool_offset_y_rot_units_label,4,3,1,1);
	
	tool_offset_z_rot_label = new QLabel(this);
	tool_offset_z_rot_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	offset_control_layout->addWidget(tool_offset_z_rot_label,5,1,1,1);
	
	d_tool_offset_z_rot_edit = new QLineEdit(this);
	d_tool_offset_z_rot_edit->setAlignment(Qt::AlignCenter);
	offset_control_layout->addWidget(d_tool_offset_z_rot_edit,5,2,1,1);
	
	tool_offset_z_rot_units_label = new QLabel(this);
	offset_control_layout->addWidget(tool_offset_z_rot_units_label,5,3,1,1);

	tool_control_vspacer = new QSpacerItem(0,0,QSizePolicy::Minimum,QSizePolicy::Expanding);
	offset_control_layout->addItem(tool_control_vspacer,6,1,1,3);

	configuration_page_layout->addLayout(offset_control_layout,3,1,1,1);
	
	tool_configuration_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	configuration_page_layout->addItem(tool_configuration_hspacer,0,4,6,1);

	tool_configuration_lower_hline = new QFrame(d_tool_configuration_page);
	tool_configuration_lower_hline->setFrameShape(QFrame::HLine);
	tool_configuration_lower_hline->setFrameShadow(QFrame::Sunken);
	configuration_page_layout->addWidget(tool_configuration_lower_hline,4,0,1,4);

	tool_options_label = new QLabel(d_tool_configuration_page);
	tool_options_label->setAlignment(Qt::AlignLeading|Qt::AlignLeft|Qt::AlignTop);
	configuration_page_layout->addWidget(tool_options_label,5,0,1,1);

	d_tool_motorized_head_check = new QCheckBox(d_tool_configuration_page);
	configuration_page_layout->addWidget(d_tool_motorized_head_check,5,1,1,1);
	
	configuration_page_vspacer = new QSpacerItem(0,0,QSizePolicy::Minimum,QSizePolicy::Expanding);
	configuration_page_layout->addItem(configuration_page_vspacer,6,0,1,4);
	
	// tool data page
	d_tooldata_page = new QWidget();
	d_toolbox->addItem(d_tooldata_page,QStringLiteral("Tool Data"));

	tooldata_page_layout = new QGridLayout(d_tooldata_page);
	
	d_tooldata_name_combo = new QComboBox(d_tool_configuration_page);
	d_tooldata_name_combo->setInsertPolicy(QComboBox::InsertAlphabetically);
	tooldata_page_layout->addWidget(d_tooldata_name_combo,0,0,1,1);
	
	tooldata_angles_label = new QLabel(d_tooldata_page);
	tooldata_page_layout->addWidget(tooldata_angles_label,1,0,1,1);

	d_tooldata_angles_list = new QListWidget(d_tooldata_page);
	d_tooldata_angles_list->setSortingEnabled(false);
	d_tooldata_angles_list->setSelectionMode(QAbstractItemView::ExtendedSelection);
	tooldata_page_layout->addWidget(d_tooldata_angles_list,2,0,10,1);

	d_tooldata_offset_x_label = new QLabel(d_tooldata_page);
	d_tooldata_offset_x_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	tooldata_page_layout->addWidget(d_tooldata_offset_x_label,2,1,1,1);

	d_tooldata_offset_x_edit = new QLineEdit(d_tooldata_page);
	d_tooldata_offset_x_edit->setAlignment(Qt::AlignCenter);
	d_tooldata_offset_x_edit->setReadOnly(true);
	tooldata_page_layout->addWidget(d_tooldata_offset_x_edit,2,2,1,1);

	d_tooldata_offset_y_label = new QLabel(d_tooldata_page);
	d_tooldata_offset_y_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	tooldata_page_layout->addWidget(d_tooldata_offset_y_label,3,1,1,1);

	d_tooldata_offset_y_edit = new QLineEdit(d_tooldata_page);
	d_tooldata_offset_y_edit->setAlignment(Qt::AlignCenter);
	d_tooldata_offset_y_edit->setReadOnly(true);
	tooldata_page_layout->addWidget(d_tooldata_offset_y_edit,3,2,1,1);

	d_tooldata_offset_z_label = new QLabel(d_tooldata_page);
	d_tooldata_offset_z_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	tooldata_page_layout->addWidget(d_tooldata_offset_z_label,4,1,1,1);

	d_tooldata_offset_z_edit = new QLineEdit(d_tooldata_page);
	d_tooldata_offset_z_edit->setAlignment(Qt::AlignCenter);
	d_tooldata_offset_z_edit->setReadOnly(true);
	tooldata_page_layout->addWidget(d_tooldata_offset_z_edit,4,2,1,1);

	d_tooldata_vector_i_label = new QLabel(d_tooldata_page);
	d_tooldata_vector_i_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	tooldata_page_layout->addWidget(d_tooldata_vector_i_label,5,1,1,1);

	d_tooldata_vector_i_edit = new QLineEdit(d_tooldata_page);
	d_tooldata_vector_i_edit->setAlignment(Qt::AlignCenter);
	d_tooldata_vector_i_edit->setReadOnly(true);
	tooldata_page_layout->addWidget(d_tooldata_vector_i_edit,5,2,1,1);

	d_tooldata_vector_j_label = new QLabel(d_tooldata_page);
	d_tooldata_vector_j_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	tooldata_page_layout->addWidget(d_tooldata_vector_j_label,6,1,1,1);

	d_tooldata_vector_j_edit = new QLineEdit(d_tooldata_page);
	d_tooldata_vector_j_edit->setAlignment(Qt::AlignCenter);
	d_tooldata_vector_j_edit->setReadOnly(true);
	tooldata_page_layout->addWidget(d_tooldata_vector_j_edit,6,2,1,1);

	d_tooldata_vector_k_label = new QLabel(d_tooldata_page);
	d_tooldata_vector_k_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	tooldata_page_layout->addWidget(d_tooldata_vector_k_label,7,1,1,1);

	d_tooldata_vector_k_edit = new QLineEdit(d_tooldata_page);
	d_tooldata_vector_k_edit->setAlignment(Qt::AlignCenter);
	d_tooldata_vector_k_edit->setReadOnly(true);
	tooldata_page_layout->addWidget(d_tooldata_vector_k_edit,7,2,1,1);

	tooldata_tip_diameter_label = new QLabel(d_tooldata_page);
	tooldata_tip_diameter_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	tooldata_page_layout->addWidget(tooldata_tip_diameter_label,8,1,1,1);

	d_tooldata_tip_diameter_edit = new QLineEdit(d_tooldata_page);
	d_tooldata_tip_diameter_edit->setAlignment(Qt::AlignCenter);
	tooldata_page_layout->addWidget(d_tooldata_tip_diameter_edit,8,2,1,1);

	tooldata_tip_average_diameter_label = new QLabel(d_tooldata_page);
	tooldata_tip_average_diameter_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	tooldata_page_layout->addWidget(tooldata_tip_average_diameter_label,9,1,1,1);
	
	d_tooldata_tip_average_diameter_edit = new QLineEdit(d_tooldata_page);
	d_tooldata_tip_average_diameter_edit->setAlignment(Qt::AlignCenter);
	d_tooldata_tip_average_diameter_edit->setReadOnly(true);
	tooldata_page_layout->addWidget(d_tooldata_tip_average_diameter_edit,9,2,1,1);

	d_tooldata_calibrate_button = new QPushButton(d_tooldata_page);
	d_tooldata_calibrate_button->setAutoDefault(false);
	tooldata_page_layout->addWidget(d_tooldata_calibrate_button,10,2,1,1);

	tooldata_angles_edit_hlayout = new QHBoxLayout();

	d_tooldata_add_angles_button = new QToolButton(d_tooldata_page);
	d_tooldata_add_angles_button->setMinimumSize(16,16);
	d_tooldata_add_angles_button->setMaximumSize(16,16);
	d_tooldata_add_angles_button->setIconSize(QSize(16,16));
	tooldata_angles_edit_hlayout->addWidget(d_tooldata_add_angles_button);

	d_tooldata_remove_angles_button = new QToolButton(d_tooldata_page);
	d_tooldata_remove_angles_button->setMinimumSize(16,16);
	d_tooldata_remove_angles_button->setMaximumSize(16,16);
	d_tooldata_remove_angles_button->setIconSize(QSize(16,16));
	tooldata_angles_edit_hlayout->addWidget(d_tooldata_remove_angles_button);

	tooldata_angles_edit_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	tooldata_angles_edit_hlayout->addItem(tooldata_angles_edit_hspacer);
	tooldata_page_layout->addLayout(tooldata_angles_edit_hlayout,12,0,1,1);

	tool_data_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	tooldata_page_layout->addItem(tool_data_hspacer,2,3,10,1);
	
	dialog_vlayout->addWidget(d_toolbox);

	dialog_hline_separator = new QFrame(this);
	dialog_hline_separator->setFrameShape(QFrame::HLine);
	dialog_hline_separator->setFrameShadow(QFrame::Sunken);
	dialog_vlayout->addWidget(dialog_hline_separator);

	button_hlayout = new QHBoxLayout();

	button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_hlayout->addItem(button_hspacer);

	close_button = new QPushButton(this);
	close_button->setAutoDefault(false);
	button_hlayout->addWidget(close_button);
	dialog_vlayout->addLayout(button_hlayout);
	
	tool_probe_image_label->setPixmap(QPixmap(QStringLiteral(":/images/probe_offsets.png")));
	d_tool_add_name_button->setIcon(QIcon(":/icon16/add_icon16.png"));
	d_tool_remove_name_button->setIcon(QIcon(":/icon16/remove_icon16.png"));
	d_tool_add_configuration_button->setIcon(QIcon(":/icon16/add_icon16.png"));
	d_tool_remove_configuration_button->setIcon(QIcon(":/icon16/remove_icon16.png"));
	d_tooldata_add_angles_button->setIcon(QIcon(":/icon16/add_icon16.png"));
	d_tooldata_remove_angles_button->setIcon(QIcon(":/icon16/remove_icon16.png"));

	this->setWindowTitle(QStringLiteral("Probe Utilities"));
	
	orientation_a0b0_label->setText(QStringLiteral("Orientation At Angle A0 B0:"));
	orientation_a90b0_label->setText(QStringLiteral("Orientation At Angle A90 B0:"));
	d_orientation_valid_label->setText("Orientation Valid");
	tool_name_label->setText(QStringLiteral("Name:"));
	tool_configuration_label->setText(QStringLiteral("Offset Configuration:"));
	tool_options_label->setText(QStringLiteral("Options:"));
	d_tool_motorized_head_check->setText(QStringLiteral("Motorized Probe Head"));
	tool_offset_x_fixed_label->setText(QStringLiteral("Fixed X:"));
	tool_offset_x_fixed_units_label->setText(QStringLiteral("mm"));
	tool_offset_y_fixed_label->setText(QStringLiteral("Fixed Y:"));
	tool_offset_y_fixed_units_label->setText(QStringLiteral("mm"));
	tool_offset_z_fixed_label->setText(QStringLiteral("Fixed Z:"));
	tool_offset_z_fixed_units_label->setText(QStringLiteral("mm"));
	tool_offset_x_rot_label->setText(QStringLiteral("Rotational X:"));
	tool_offset_x_rot_units_label->setText(QStringLiteral("mm"));
	tool_offset_y_rot_label->setText(QStringLiteral("Rotational Y:"));
	tool_offset_y_rot_units_label->setText(QStringLiteral("mm"));
	tool_offset_z_rot_label->setText(QStringLiteral("Rotational Z:"));
	tool_offset_z_rot_units_label->setText(QStringLiteral("mm"));
	tooldata_angles_label->setText(QStringLiteral("Probe Head Angles:"));
	d_tooldata_offset_x_label->setText(QStringLiteral("Offset X:"));
	d_tooldata_offset_y_label->setText(QStringLiteral("Offset Y:"));
	d_tooldata_offset_z_label->setText(QStringLiteral("Offset Z:"));
	d_tooldata_vector_i_label->setText(QStringLiteral("Vector I:"));
	d_tooldata_vector_j_label->setText(QStringLiteral("Vector J:"));
	d_tooldata_vector_k_label->setText(QStringLiteral("Vector K:"));
	tooldata_tip_diameter_label->setText(QStringLiteral("Tip Diameter:"));
	tooldata_tip_average_diameter_label->setText(QStringLiteral("Average Tip Diameter:"));
	d_tooldata_calibrate_button->setText(QStringLiteral("Calibrate Tools"));
	close_button->setText(QStringLiteral("Close"));
	
	// defaults
	d_toolbox->setCurrentIndex(0);

	d_name_edit_dialog = 0;
	d_tool_builder_dialog = 0;
	d_tool_add_angles_dialog = 0;

	d_orientation_a0b0_combo->addItem(QStringLiteral("X Minus"),TToolData::AXIS_XMINUS);
	d_orientation_a0b0_combo->addItem(QStringLiteral("X Plus"),TToolData::AXIS_XPLUS);
	d_orientation_a0b0_combo->addItem(QStringLiteral("Y Minus"),TToolData::AXIS_YMINUS);
	d_orientation_a0b0_combo->addItem(QStringLiteral("Y Plus"),TToolData::AXIS_YPLUS);
	d_orientation_a0b0_combo->addItem(QStringLiteral("Z Minus"),TToolData::AXIS_ZMINUS);
	d_orientation_a0b0_combo->addItem(QStringLiteral("Z Plus"),TToolData::AXIS_ZPLUS);
	
	d_orientation_a90b0_combo->addItem(QStringLiteral("X Minus"),TToolData::AXIS_XMINUS);
	d_orientation_a90b0_combo->addItem(QStringLiteral("X Plus"),TToolData::AXIS_XPLUS);
	d_orientation_a90b0_combo->addItem(QStringLiteral("Y Minus"),TToolData::AXIS_YMINUS);
	d_orientation_a90b0_combo->addItem(QStringLiteral("Y Plus"),TToolData::AXIS_YPLUS);
	d_orientation_a90b0_combo->addItem(QStringLiteral("Z Minus"),TToolData::AXIS_ZMINUS);
	d_orientation_a90b0_combo->addItem(QStringLiteral("Z Plus"),TToolData::AXIS_ZPLUS);
	
	d_orientation_a0b0_combo->setCurrentIndex(4);	// z_minus
	d_orientation_a90b0_combo->setCurrentIndex(2);	// y_minus
	d_tool_data.Set_Probe_Head_Orientation(TToolData::AXIS_ZMINUS,TToolData::AXIS_YMINUS);
	
	d_tool_motorized_head_check->setChecked(true);
	
	d_tool_configuration_page->setEnabled(true);
	d_tooldata_page->setEnabled(true);

	d_default_tip_diameter = 4.0;
	
	this->Orientation_Changed();

	connect(close_button,&QPushButton::clicked,this,&TToolUtilitiesDialog::close);
	connect(d_toolbox,&QToolBox::currentChanged,this,&TToolUtilitiesDialog::Page_Changed);
	
	connect(d_orientation_a0b0_combo,static_cast<void (QComboBox::*)(int)>(&QComboBox::currentIndexChanged),this,&TToolUtilitiesDialog::Orientation_Changed);
	connect(d_orientation_a90b0_combo,static_cast<void (QComboBox::*)(int)>(&QComboBox::currentIndexChanged),this,&TToolUtilitiesDialog::Orientation_Changed);
	
	connect(d_tool_name_combo,static_cast<void (QComboBox::*)(int)>(&QComboBox::currentIndexChanged),this,&TToolUtilitiesDialog::Tool_Name_Changed);
	connect(d_tool_add_name_button,&QToolButton::clicked,this,&TToolUtilitiesDialog::Tool_Add_Name);
	connect(d_tool_remove_name_button,&QToolButton::clicked,this,&TToolUtilitiesDialog::Tool_Remove_Name);
	
	connect(d_tool_configuration_combo,static_cast<void (QComboBox::*)(int)>(&QComboBox::currentIndexChanged),this,&TToolUtilitiesDialog::Configuration_Changed);
	connect(d_tool_add_configuration_button,&QToolButton::clicked,this,&TToolUtilitiesDialog::Configuration_Add);
	connect(d_tool_remove_configuration_button,&QToolButton::clicked,this,&TToolUtilitiesDialog::Configuration_Remove);
	connect(d_tool_motorized_head_check,&QCheckBox::toggled,this,&TToolUtilitiesDialog::Configuration_Motorized_Head_Changed);
	
	connect(d_tool_offset_x_fixed_edit,&QLineEdit::editingFinished,this,&TToolUtilitiesDialog::Configuration_Offset_Changed);
	connect(d_tool_offset_y_fixed_edit,&QLineEdit::editingFinished,this,&TToolUtilitiesDialog::Configuration_Offset_Changed);
	connect(d_tool_offset_z_fixed_edit,&QLineEdit::editingFinished,this,&TToolUtilitiesDialog::Configuration_Offset_Changed);
	connect(d_tool_offset_x_rot_edit,&QLineEdit::editingFinished,this,&TToolUtilitiesDialog::Configuration_Offset_Changed);
	connect(d_tool_offset_y_rot_edit,&QLineEdit::editingFinished,this,&TToolUtilitiesDialog::Configuration_Offset_Changed);
	connect(d_tool_offset_z_rot_edit,&QLineEdit::editingFinished,this,&TToolUtilitiesDialog::Configuration_Offset_Changed);
	
	connect(d_tooldata_name_combo,static_cast<void (QComboBox::*)(int)>(&QComboBox::currentIndexChanged),this,&TToolUtilitiesDialog::ToolData_Name_Changed);
	connect(d_tooldata_add_angles_button,&QToolButton::clicked,this,&TToolUtilitiesDialog::ToolData_Add_Angle);
	connect(d_tooldata_remove_angles_button,&QToolButton::clicked,this,&TToolUtilitiesDialog::ToolData_Remove_Angle);
	connect(d_tooldata_angles_list,&QListWidget::itemSelectionChanged,this,&TToolUtilitiesDialog::ToolData_Angle_Selection_Changed);
	connect(d_tooldata_tip_diameter_edit,&QLineEdit::editingFinished,this,&TToolUtilitiesDialog::ToolData_Tip_Diameter_Changed);
	
	connect(d_tooldata_calibrate_button,&QPushButton::clicked,this,&TToolUtilitiesDialog::ToolData_Calibrate_Probes);
}

TToolUtilitiesDialog::~TToolUtilitiesDialog(void)
{
}

bool TToolUtilitiesDialog::Save_Tool_Data(
	const QString						&path) const
{	
	return d_tool_data.Save_Tool_Data(path);
}

int TToolUtilitiesDialog::Tool_Count(void) const
{
	return d_tool_data.Tool_Count();
}

TToolData::TTool TToolUtilitiesDialog::Tool(
	const int							index) const
{
	return d_tool_data.Tool(index);
}

TToolData::TToolEntry TToolUtilitiesDialog::Tool_Entry(
	const QString						&tool_name,
	const double						&angle_a,
	const double						&angle_b,
	TVector3							* const offset,
	TVector3							* const vector,
	bool								* valid)
{
	return d_tool_data.Tool_Entry(tool_name,angle_a,angle_b,offset,vector,valid);
}

QString TToolUtilitiesDialog::Calibrate_Tool_Name(void) const
{
	return d_tooldata_name_combo->currentText();
}

std::vector<TToolData::TToolEntry> TToolUtilitiesDialog::Calibrate_Tool_Data(void) const
{
	QList<QListWidgetItem*>				sel_items;
	QList<QListWidgetItem*>::iterator	iter;
	std::vector<TToolData::TToolEntry>	tool_data;
	QString								tool_name;
	double								angle_a;
	double								angle_b;

	tool_name = d_tooldata_name_combo->currentText();
	sel_items = d_tooldata_angles_list->selectedItems();

	for(iter = sel_items.begin();iter != sel_items.end();++iter)
	{
		angle_a = (*iter)->data(Qt::UserRole).toDouble();
		angle_b = (*iter)->data(Qt::UserRole+1).toDouble();
		
		tool_data.push_back(d_tool_data.Tool_Entry(tool_name,angle_a,angle_b));
	}
	
	return tool_data;
}

void TToolUtilitiesDialog::Reset(
	const QString						&active_tool_name)
{
	QStringList							list;
	bool								prev_state;

	d_calibrate_on_close = false;

	d_active_tool_name = active_tool_name;
	
	// configuration must be setup before tool names
	prev_state = d_tool_configuration_combo->blockSignals(true);
	
	d_tool_configuration_combo->clear();
	list = d_tool_data.Offset_Configuration_Names();
	
	d_tool_configuration_combo->addItems(list);
	
	d_tool_configuration_combo->blockSignals(prev_state);
	
	d_tooldata_calibrate_button->setEnabled(false);
	
	this->Page_Changed(d_toolbox->currentIndex());
}

bool TToolUtilitiesDialog::Load_Tool_Data(
	const QString						&path)
{
	bool								state;
	
	state = d_tool_data.Load_Tool_Data(path);
	
	d_tool_motorized_head_check->setChecked(d_tool_data.Is_Motorized_Probe_Head());
	
	this->Update_Orientation_Display();

	return state;
}

void TToolUtilitiesDialog::Update_Tool_Tip_Diameter(
	const QString						&tool_name,
	const double						&angle_a,
	const double						&angle_b,
	const double						&tip_diameter)
{
	d_tool_data.Update_Tool_Tip_Diameter(tool_name,angle_a,angle_b,tip_diameter);
}

void TToolUtilitiesDialog::Convert_Angle_Signature(
	const int							angle_ab_signature,
	double								* const angle_a,
	double								* const angle_b)
{
	assert(angle_a);
	assert(angle_b);
	
	TToolData::Decode_AB_Angles(angle_a,angle_b,angle_ab_signature);
}

void TToolUtilitiesDialog::Page_Changed(
	int									index)
{
	switch(index)
	{
		case 0: // orientation
			break;
			
		case 1:	// offset configuration
			if(d_tool_data.Probe_Head_Orientation_Valid())
			{
				this->Initialize_Tool_Page();
			}

			break;
			
		case 2: // tool data
			d_tooldata_name_combo->setFocus();	// this makes sure any 'editFinished' signals from page2 are processed.
			
			if(d_tool_data.Probe_Head_Orientation_Valid())
			{
				this->Initialize_Tool_Data_Page();
			}
			break;
	}
}

void TToolUtilitiesDialog::Orientation_Changed(void)
{
	bool								valid;
	TToolData::TAxis					a0b0;
	TToolData::TAxis					a90b0;
	
	a0b0 = static_cast<TToolData::TAxis>(d_orientation_a0b0_combo->currentData(Qt::UserRole).toInt());
	a90b0 = static_cast<TToolData::TAxis>(d_orientation_a90b0_combo->currentData(Qt::UserRole).toInt());
	
	valid = d_tool_data.Set_Probe_Head_Orientation(a0b0,a90b0);
	
	if(valid)
	{
		d_orientation_valid_label->setText("Orientation Valid");

		d_tool_configuration_page->setEnabled(true);
		d_tooldata_page->setEnabled(true);
	}
	else
	{
		d_orientation_valid_label->setText("Orientation Invalid");

		d_tool_configuration_page->setEnabled(false);
		d_tooldata_page->setEnabled(false);
	}
	
	this->Configuration_Changed();	// forces an update of the active configruation.
}

void TToolUtilitiesDialog::Tool_Name_Changed(void)
{
	TToolData::TTool					tool;
	int									index;
	bool								valid;
		
	d_active_tool_name = d_tool_name_combo->currentText();
	
	if(d_active_tool_name.length())
	{
		tool = d_tool_data.Tool(d_active_tool_name,&valid);
		
		if(valid)
		{
			index = d_tool_configuration_combo->findText(tool.configuration_name);
			
			if((index < 0) == false && index == d_tool_configuration_combo->currentIndex())
			{
				this->Configuration_Changed();
			}
			else
			{
				d_tool_configuration_combo->setCurrentIndex(index);	// okay if set to -1
			}
		}
	}
}

void TToolUtilitiesDialog::Tool_Add_Name(void)
{
	QStringList							list;
	int									cntr;
	int									index;
	
	if(!d_name_edit_dialog)
	{
		d_name_edit_dialog = new TNameEditDialog(this);
	}
	
	for(cntr = 0;cntr < d_tool_name_combo->count();++cntr)
	{
		list.push_back(d_tool_name_combo->itemText(cntr));
	}
	
	d_name_edit_dialog->Reset(QStringLiteral("Add Tool"),QStringLiteral("Tool Name"),list);
	
	if(QDialog::Accepted == d_name_edit_dialog->exec())
	{
		d_active_tool_name = d_name_edit_dialog->Name();
		
		d_tool_data.Add_Tool(d_active_tool_name,d_tool_configuration_combo->currentText());
		d_tool_name_combo->addItem(d_active_tool_name);

		index = d_tool_name_combo->findText(d_active_tool_name);
		
		if(!(index < 0))
		{
			d_tool_name_combo->setCurrentIndex(index);
		}
	}
}

void TToolUtilitiesDialog::Tool_Remove_Name(void)
{
	int									index;
	QString								tool_name;
	
	index = d_tool_name_combo->currentIndex();
	tool_name = d_tool_name_combo->currentText();
	
	if(!(index < 0))
	{
		d_tool_name_combo->removeItem(index);
	}
	
	d_tool_data.Remove_Tool(tool_name);

	d_active_tool_name = d_tool_name_combo->currentText();
}

void TToolUtilitiesDialog::Configuration_Changed(void)
{
	QString								text;
	bool								valid;
	TToolData::TConfiguration			offset_configuration;
		
	text = d_tool_configuration_combo->currentText();
	
	offset_configuration = d_tool_data.Offset_Configuration(text,&valid);
	
	if(valid)
	{
		d_tool_offset_x_fixed_edit->setText(QString("%1").arg(offset_configuration.offset_fixed.x,0,'f',3));
		d_tool_offset_y_fixed_edit->setText(QString("%1").arg(offset_configuration.offset_fixed.y,0,'f',3));
		d_tool_offset_z_fixed_edit->setText(QString("%1").arg(offset_configuration.offset_fixed.z,0,'f',3));
		
		d_tool_offset_x_rot_edit->setText(QString("%1").arg(offset_configuration.offset_rotational.x,0,'f',3));
		d_tool_offset_y_rot_edit->setText(QString("%1").arg(offset_configuration.offset_rotational.y,0,'f',3));
		d_tool_offset_z_rot_edit->setText(QString("%1").arg(offset_configuration.offset_rotational.z,0,'f',3));

		// update configuration in active tool
		d_tool_data.Update_Tool_Configuration(d_tool_name_combo->currentText(),offset_configuration.configuration_name);
		
		emit Invalidate_Active_Tool();
	}
}

void TToolUtilitiesDialog::Configuration_Add(void)
{
	TVector3							fixed_offset;
	TVector3							rotational_offset;
	QString								text;
	QStringList							list;
	int									cntr;
	int									index;
	
	if(!d_tool_builder_dialog)
	{
		d_tool_builder_dialog = new TToolBuilderDialog(this);
	}

	for(cntr = 0;cntr < d_tool_configuration_combo->count();++cntr)
	{
		list.push_back(d_tool_configuration_combo->itemText(cntr));
	}
	
	d_tool_builder_dialog->Reset(list);
	
	if(QDialog::Accepted == d_tool_builder_dialog->exec())
	{
		fixed_offset = d_tool_data.Offset_In_Head_Orientation(d_tool_builder_dialog->Fixed_Offset());
		rotational_offset = d_tool_data.Offset_In_Head_Orientation(d_tool_builder_dialog->Rotational_Offset());
		
		text = d_tool_builder_dialog->Configuration_Name();
		d_tool_configuration_combo->addItem(text);

		index = d_tool_configuration_combo->findText(text);
		
		if(!(index < 0))
		{
			d_tool_configuration_combo->setCurrentIndex(index);
			
			d_tool_offset_x_fixed_edit->setText(QString("%1").arg(fixed_offset.x,0,'f',3));
			d_tool_offset_y_fixed_edit->setText(QString("%1").arg(fixed_offset.y,0,'f',3));
			d_tool_offset_z_fixed_edit->setText(QString("%1").arg(fixed_offset.z,0,'f',3));
			
			d_tool_offset_x_rot_edit->setText(QString("%1").arg(rotational_offset.x,0,'f',3));
			d_tool_offset_y_rot_edit->setText(QString("%1").arg(rotational_offset.y,0,'f',3));
			d_tool_offset_z_rot_edit->setText(QString("%1").arg(rotational_offset.z,0,'f',3));
			
			this->Configuration_Offset_Changed();	// force creation of a new configuration entry
			
			// update configuration in active tool
			d_tool_data.Update_Tool_Configuration(d_tool_name_combo->currentText(),text);
		}
	}
}

void TToolUtilitiesDialog::Configuration_Remove(void)
{
	int									index;
	QString								configuration_name;
	
	index = d_tool_configuration_combo->currentIndex();
	configuration_name = d_tool_configuration_combo->currentText();
	
	if(!(index < 0))
	{
		d_tool_configuration_combo->removeItem(index);
	}
	
	d_tool_data.Remove_Configuration(configuration_name);
}

void TToolUtilitiesDialog::Configuration_Offset_Changed(void)
{
	QString								configuration_name;
	TVector3							offset_fixed;
	TVector3							offset_rotational;
	
	configuration_name = d_tool_configuration_combo->currentText();
	
	if(configuration_name.length() == 0)
	{
		return;
	}
	
	offset_fixed.x = d_tool_offset_x_fixed_edit->text().toDouble();
	offset_fixed.y = d_tool_offset_y_fixed_edit->text().toDouble();
	offset_fixed.z = d_tool_offset_z_fixed_edit->text().toDouble();
	
	offset_rotational.x = d_tool_offset_x_rot_edit->text().toDouble();
	offset_rotational.y = d_tool_offset_y_rot_edit->text().toDouble();
	offset_rotational.z = d_tool_offset_z_rot_edit->text().toDouble();
	
	d_tool_data.Update_Configuration(configuration_name,offset_fixed,offset_rotational);
	
	emit Invalidate_Active_Tool();
}

void TToolUtilitiesDialog::Configuration_Motorized_Head_Changed(
	bool								state)
{
	d_tool_data.Set_Motorized_Probe_Head(state);
}

void TToolUtilitiesDialog::ToolData_Name_Changed(void)
{
	std::vector<TToolData::TToolEntry>::const_iterator iter;
	QListWidgetItem						*item;
	QString								tool_name;
	TToolData::TTool					tool;
	double								angle_a;
	double								angle_b;
	bool								valid;
	
	d_tooldata_angles_list->clear();
	
	tool_name = d_tooldata_name_combo->currentText();
	
	d_active_tool_name = tool_name;
	
	if(tool_name.length())
	{
		tool = d_tool_data.Tool(tool_name,&valid);
		
		if(valid)
		{
			for(iter = tool.tool_entries.begin();iter != tool.tool_entries.end();++iter)
			{
				item = new QListWidgetItem(d_tooldata_angles_list);
				
				d_tool_data.Decode_AB_Angles(&angle_a,&angle_b,(*iter).angle_ab_signature);
				
				item->setText(QString("A%1 B%2").arg(angle_a,0,'f',1).arg(angle_b,0,'f',1));
				
				item->setData(Qt::UserRole,angle_a);
				item->setData(Qt::UserRole+1,angle_b);
			}
		}
	}
}

void TToolUtilitiesDialog::ToolData_Add_Angle(void)
{
	std::vector<TToolAddAnglesDialog::TAngle> angles;
	std::vector<TToolAddAnglesDialog::TAngle>::iterator iter;
	QString								tool_name;

	if(!d_tool_add_angles_dialog)
	{
		d_tool_add_angles_dialog = new TToolAddAnglesDialog(this);
	}
	
	d_tool_add_angles_dialog->Reset();
	
	if(QDialog::Accepted == d_tool_add_angles_dialog->exec())
	{
		tool_name = d_tooldata_name_combo->currentText();
		
		angles = d_tool_add_angles_dialog->Angles();
		
		for(iter = angles.begin();iter != angles.end();++iter)
		{
			d_tool_data.Add_Tool_Angle(tool_name,(*iter).angle_a,(*iter).angle_b,d_default_tip_diameter);
		}
	}
	
	this->ToolData_Name_Changed();
}

void TToolUtilitiesDialog::ToolData_Remove_Angle(void)
{
	QList<QListWidgetItem*>				sel_items;
	QList<QListWidgetItem*>::iterator	iter;
	QString								tool_name;
	double								angle_a;
	double								angle_b;
	int									row;

	sel_items = d_tooldata_angles_list->selectedItems();
	tool_name = d_tooldata_name_combo->currentText();
	
	for(iter = sel_items.begin();iter != sel_items.end();++iter)
	{
		angle_a = (*iter)->data(Qt::UserRole).toDouble();
		angle_b = (*iter)->data(Qt::UserRole+1).toDouble();

		d_tool_data.Remove_Tool_Angle(tool_name,angle_a,angle_b);
		
		row = d_tooldata_angles_list->row(*iter);
	
		d_tooldata_angles_list->takeItem(row);
		
		delete (*iter);
	}
}

void TToolUtilitiesDialog::ToolData_Angle_Selection_Changed(void)
{
	QList<QListWidgetItem*>				sel_items;
	QList<QListWidgetItem*>::const_iterator	iter;
	TToolData::TToolEntry				tool_data;
	TVector3							offset;
	TVector3							vector;
	double								angle_a;
	double								angle_b;
	double								average;
	int									cntr;
	bool								valid;
	
	sel_items = d_tooldata_angles_list->selectedItems();
	
	d_tooldata_offset_x_edit->setText(QString());
	d_tooldata_offset_y_edit->setText(QString());
	d_tooldata_offset_z_edit->setText(QString());
	
	d_tooldata_vector_i_edit->setText(QString());
	d_tooldata_vector_j_edit->setText(QString());
	d_tooldata_vector_k_edit->setText(QString());
	
	d_tooldata_offset_x_edit->setEnabled(false);
	d_tooldata_offset_y_edit->setEnabled(false);
	d_tooldata_offset_z_edit->setEnabled(false);

	d_tooldata_vector_i_edit->setEnabled(false);
	d_tooldata_vector_j_edit->setEnabled(false);
	d_tooldata_vector_k_edit->setEnabled(false);
	
	d_tooldata_offset_x_label->setEnabled(false);
	d_tooldata_offset_y_label->setEnabled(false);
	d_tooldata_offset_z_label->setEnabled(false);
	
	d_tooldata_vector_i_label->setEnabled(false);
	d_tooldata_vector_j_label->setEnabled(false);
	d_tooldata_vector_k_label->setEnabled(false);
	
	if(sel_items.size() == 0)
	{
		d_tooldata_tip_diameter_edit->setText(QString());
		d_tooldata_tip_average_diameter_edit->setText(QString());
	}
	
	if(sel_items.size() == 1)
	{
		angle_a = sel_items[0]->data(Qt::UserRole).toDouble();
		angle_b = sel_items[0]->data(Qt::UserRole+1).toDouble();
		
		tool_data = d_tool_data.Tool_Entry(d_tooldata_name_combo->currentText(),angle_a,angle_b,&offset,&vector,&valid);
		
		if(valid)
		{
			d_tooldata_offset_x_edit->setText(QString("%1").arg(offset.x,0,'f',3));
			d_tooldata_offset_y_edit->setText(QString("%1").arg(offset.y,0,'f',3));
			d_tooldata_offset_z_edit->setText(QString("%1").arg(offset.z,0,'f',3));
			
			d_tooldata_vector_i_edit->setText(QString("%1").arg(vector.x,0,'f',6));
			d_tooldata_vector_j_edit->setText(QString("%1").arg(vector.y,0,'f',6));
			d_tooldata_vector_k_edit->setText(QString("%1").arg(vector.z,0,'f',6));

			d_tooldata_tip_diameter_edit->setText(QString("%1").arg(tool_data.tip_diameter,0,'f',4));
			d_tooldata_tip_average_diameter_edit->setText(QString("%1").arg(tool_data.tip_diameter,0,'f',4));
			
			d_tooldata_offset_x_edit->setEnabled(true);
			d_tooldata_offset_y_edit->setEnabled(true);
			d_tooldata_offset_z_edit->setEnabled(true);
			
			d_tooldata_vector_i_edit->setEnabled(true);
			d_tooldata_vector_j_edit->setEnabled(true);
			d_tooldata_vector_k_edit->setEnabled(true);
			
			d_tooldata_offset_x_label->setEnabled(true);
			d_tooldata_offset_y_label->setEnabled(true);
			d_tooldata_offset_z_label->setEnabled(true);
			
			d_tooldata_vector_i_label->setEnabled(true);
			d_tooldata_vector_j_label->setEnabled(true);
			d_tooldata_vector_k_label->setEnabled(true);
		}
	}
	else if(sel_items.size() > 1)
	{
		cntr = 0;
		average = 0.0;
		
		for(iter = sel_items.begin();iter != sel_items.end(); ++iter)
		{
			angle_a = (*iter)->data(Qt::UserRole).toDouble();
			angle_b = (*iter)->data(Qt::UserRole+1).toDouble();

			tool_data = d_tool_data.Tool_Entry(d_tooldata_name_combo->currentText(),angle_a,angle_b,&offset,&vector,&valid);

			if(valid)
			{
				average += tool_data.tip_diameter;
				cntr++;
			}
		}
		
		if(cntr > 0)
		{
			average /= static_cast<double>(cntr);
			
			d_tooldata_tip_average_diameter_edit->setText(QString("%1").arg(average,0,'f',4));
		}
	}
	
	d_tooldata_calibrate_button->setEnabled(sel_items.size() > 0);
	d_tooldata_remove_angles_button->setEnabled(sel_items.size() > 0);
}

void TToolUtilitiesDialog::ToolData_Tip_Diameter_Changed(void)
{
	QList<QListWidgetItem*>				sel_items;
	QList<QListWidgetItem*>::iterator	item_iter;
	double								tip_diameter;
	double								angle_a;
	double								angle_b;
	QString								tool_name;
	static const double					MIN_TIP_DIAMETER(0.5);
	static const double					MAX_TIP_DIAMETER(10.0);
	
	tip_diameter = d_tooldata_tip_diameter_edit->text().toDouble();
	
	if(tip_diameter > MIN_TIP_DIAMETER && tip_diameter < MAX_TIP_DIAMETER)
	{
		d_default_tip_diameter = tip_diameter;
		
		tool_name = d_tooldata_name_combo->currentText();
		
		sel_items = d_tooldata_angles_list->selectedItems();

		for(item_iter = sel_items.begin();item_iter != sel_items.end();++item_iter)
		{
			angle_a = (*item_iter)->data(Qt::UserRole).toDouble();
			angle_b = (*item_iter)->data(Qt::UserRole+1).toDouble();

			d_tool_data.Update_Tool_Tip_Diameter(tool_name,angle_a,angle_b,tip_diameter);
		}
	}
	
	d_tooldata_tip_average_diameter_edit->setText(QString("%1").arg(tip_diameter,0,'f',4));
}

void TToolUtilitiesDialog::ToolData_Calibrate_Probes(void)
{
	d_calibrate_on_close = true;
	
	this->close();
}

void TToolUtilitiesDialog::Update_Orientation_Display(void)
{
	TToolData::TAxis					a0b0;
	TToolData::TAxis					a90b0;
	
	a0b0 = d_tool_data.Probe_Head_Orientation_A0B0();
	a90b0 = d_tool_data.Probe_Head_Orientation_A90B0();
	
	switch(a0b0)
	{
		case TToolData::AXIS_XMINUS:
			d_orientation_a0b0_combo->setCurrentIndex(0);
			break;
			
		case TToolData::AXIS_XPLUS:
			d_orientation_a0b0_combo->setCurrentIndex(1);
			break;
			
		case TToolData::AXIS_YMINUS:
			d_orientation_a0b0_combo->setCurrentIndex(2);
			break;
			
		case TToolData::AXIS_YPLUS:
			d_orientation_a0b0_combo->setCurrentIndex(3);
			break;
			
		case TToolData::AXIS_ZMINUS:
			d_orientation_a0b0_combo->setCurrentIndex(4);
			break;
			
		case TToolData::AXIS_ZPLUS:
			d_orientation_a0b0_combo->setCurrentIndex(5);
			break;
	}
	
	switch(a90b0)
	{
		case TToolData::AXIS_XMINUS:
			d_orientation_a90b0_combo->setCurrentIndex(0);
			break;
			
		case TToolData::AXIS_XPLUS:
			d_orientation_a90b0_combo->setCurrentIndex(1);
			break;
			
		case TToolData::AXIS_YMINUS:
			d_orientation_a90b0_combo->setCurrentIndex(2);
			break;
			
		case TToolData::AXIS_YPLUS:
			d_orientation_a90b0_combo->setCurrentIndex(3);
			break;
			
		case TToolData::AXIS_ZMINUS:
			d_orientation_a90b0_combo->setCurrentIndex(4);
			break;
			
		case TToolData::AXIS_ZPLUS:
			d_orientation_a90b0_combo->setCurrentIndex(5);
			break;
	}
}

void TToolUtilitiesDialog::Initialize_Tool_Page(void)
{
	QStringList							list;
	int									index;
	bool								prev_state;
	
	prev_state = d_tool_name_combo->blockSignals(true);
	d_tool_name_combo->clear();
	
	list = d_tool_data.Tool_Base_Names();
	d_tool_name_combo->addItems(list);
	
	d_tool_name_combo->blockSignals(prev_state);

	index = d_tool_name_combo->findText(d_active_tool_name);
	
	if(!(index < 0))
	{
		d_tool_name_combo->setCurrentIndex(index);
	}
	else
	{
		if(d_tool_name_combo->count())
		{
			d_tool_name_combo->setCurrentIndex(0);
		}
	}
	
	this->Tool_Name_Changed();
}

void TToolUtilitiesDialog::Initialize_Tool_Data_Page(void)
{
	QStringList							list;
	int									index;
	bool								prev_state;

	prev_state = d_tooldata_name_combo->blockSignals(true);
	d_tooldata_name_combo->clear();

	list = d_tool_data.Tool_Base_Names();
	d_tooldata_name_combo->addItems(list);
	
	d_tooldata_name_combo->blockSignals(prev_state);
	
	if(list.size())
	{
		
		index = d_tooldata_name_combo->findText(d_active_tool_name);
		
		if(!(index < 0))
		{
			d_tooldata_name_combo->setCurrentIndex(index);
		}
		else
		{
			if(d_tooldata_name_combo->count())
			{
				d_tooldata_name_combo->setCurrentIndex(0);
			}
		}
		
		d_tooldata_add_angles_button->setEnabled(true);
	}
	else
	{
		d_tooldata_add_angles_button->setEnabled(false);
	}
	
	d_tooldata_remove_angles_button->setEnabled(false);
	
	d_tooldata_offset_x_edit->setEnabled(false);
	d_tooldata_offset_y_edit->setEnabled(false);
	d_tooldata_offset_z_edit->setEnabled(false);
	
	d_tooldata_vector_i_edit->setEnabled(false);
	d_tooldata_vector_j_edit->setEnabled(false);
	d_tooldata_vector_k_edit->setEnabled(false);
	
	d_tooldata_offset_x_label->setEnabled(false);
	d_tooldata_offset_y_label->setEnabled(false);
	d_tooldata_offset_z_label->setEnabled(false);
	
	d_tooldata_vector_i_label->setEnabled(false);
	d_tooldata_vector_j_label->setEnabled(false);
	d_tooldata_vector_k_label->setEnabled(false);

	this->ToolData_Name_Changed();
}

