/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QGridLayout>
#include <QLabel>
#include <QSpacerItem>
#include <QRegularExpression>
#include <QRegularExpressionMatch>
#include <assert.h>

#include "../../../core/xmlfile.h"

#include "iconwidget.h"
#include "titlewidget.h"
#include "toolutilitiesdialog.h"
#include "informationwidget.h"

#include "toolwidget.h"

TToolWidget::TToolWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QGridLayout							*widget_layout;
	QSpacerItem							*control_hspacer;
	TTitleWidget						*title_widget;
		
	widget_layout = new QGridLayout(this);
	widget_layout->setContentsMargins(0,0,0,0);

	title_widget = new TTitleWidget(this);
	widget_layout->addWidget(title_widget,0,0,1,3);
	
	d_tool_utilities_widget = new TIconWidget(this);
	d_tool_utilities_widget->setSizePolicy(QSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred));
	widget_layout->addWidget(d_tool_utilities_widget,1,0,1,1);
	
	d_refresh_tools_widget = new TIconWidget(this);
	d_refresh_tools_widget->setSizePolicy(QSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred));
	widget_layout->addWidget(d_refresh_tools_widget,1,1,1,1);

	control_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	widget_layout->addItem(control_hspacer,1,2,1,1);
		
	d_information_widget = new TInformationWidget(this);
	widget_layout->addWidget(d_information_widget,1,3,2,1);

	d_tool_utilities_dialog = new TToolUtilitiesDialog(this);
	
	// defaults
	assert(d_information_widget->Maximum_Row_Count() > 4);
	
	d_tool_name_index = 0;
	d_tool_offset_x_index = 1;
	d_tool_offset_y_index = 2;
	d_tool_offset_z_index = 3;
	d_tool_diam_index = 4;

	title_widget->Set_Text(QStringLiteral("Tools"));
	
	d_tool_utilities_widget->Set_Icon(QStringLiteral(":/icon64/tools_icon64.png"));
	d_tool_utilities_widget->Set_Text(QStringLiteral("Configure\nTools"));
	
	d_refresh_tools_widget->Set_Icon(QStringLiteral(":/icon64/tools_refresh_icon64.png"));
	d_refresh_tools_widget->Set_Text(QStringLiteral("Refresh\nTools"));

	d_information_widget->Set_Name_Text(d_tool_name_index,QStringLiteral("Active Tool:"));
	d_information_widget->Set_Name_Text(d_tool_offset_x_index,QStringLiteral("Offset X:"));
	d_information_widget->Set_Name_Text(d_tool_offset_y_index,QStringLiteral("Offset Y:"));
	d_information_widget->Set_Name_Text(d_tool_offset_z_index,QStringLiteral("Offset Z:"));
	d_information_widget->Set_Name_Text(d_tool_diam_index,QStringLiteral("Tip Diameter:"));

	connect(d_tool_utilities_widget,&TIconWidget::Clicked,this,&TToolWidget::Tool_Item_Clicked);
	connect(d_tool_utilities_dialog,&TToolUtilitiesDialog::Invalidate_Active_Tool,this,&TToolWidget::Invalidate_Active_Tool);
	connect(d_refresh_tools_widget,&TIconWidget::Clicked,this,&TToolWidget::Refresh_Tools);
}

TToolWidget::~TToolWidget(void)
{
}

QString TToolWidget::Active_Tool_Name(void) const
{
	if(d_active_tool_valid)
	{
		return d_active_tool_name;
	}
	
	return QString();
}

bool TToolWidget::Is_Motorized_Probe_Head(void) const
{
	return d_tool_utilities_dialog->Is_Motorized_Probe_Head();
}

bool TToolWidget::Save_Tool_Data(
	const QString						&path) const
{
	return d_tool_utilities_dialog->Save_Tool_Data(path);
}

void TToolWidget::Reset(
	TController::TControllerType		controller_type)
{
	d_controller_type = controller_type;

	switch(d_controller_type)
	{
		case TController::CONTROLLER_DC:
		case TController::CONTROLLER_LEITZ:
		case TController::CONTROLLER_VIRTUAL:
			d_tool_utilities_widget->setEnabled(true);
			d_refresh_tools_widget->setEnabled(false);
			break;
			
		case TController::CONTROLLER_IPPCLIENT:
			d_tool_utilities_widget->setEnabled(false);
			d_refresh_tools_widget->setEnabled(true);
			break;
	}
	
	d_active_tool_valid = false;
}

void TToolWidget::Set_Active_Tool_Name(
	const QString 						&name)
{
	TToolWidget::TToolItem				tool_item;
	bool								valid;

	d_active_tool_name = name;
	d_active_tool_valid = (name.length() > 0);
	
	if(d_active_tool_valid)
	{
		tool_item = this->Tool_Item(name,&valid);
		
		if(valid)
		{
			this->Update_Tool_Information(tool_item);
		}
	}
	else
	{
		d_information_widget->Set_Value_Text(d_tool_name_index,QString());
		d_information_widget->Set_Value_Text(d_tool_offset_x_index,QString());
		d_information_widget->Set_Value_Text(d_tool_offset_y_index,QString());
		d_information_widget->Set_Value_Text(d_tool_offset_z_index,QString());
		d_information_widget->Set_Value_Text(d_tool_diam_index,QString());
	}
}

QStringList TToolWidget::Get_Tool_List(void) const
{
	TToolData::TTool					tool;
	std::vector<TToolData::TToolEntry>::const_iterator entry_iter;
	std::vector<TToolWidget::TToolItem>::const_iterator item_iter;
	QStringList							list;
	double								angle_a;
	double								angle_b;
	int									tool_count;
	int									cntr;
	
	if(d_controller_type == TController::CONTROLLER_IPPCLIENT)
	{
		for(item_iter = d_ipp_tool_items.begin();item_iter != d_ipp_tool_items.end();++item_iter)
		{
			list.push_back((*item_iter).name);
		}
	}
	else
	{
		tool_count = d_tool_utilities_dialog->Tool_Count();
		
		for(cntr = 0;cntr < tool_count;++cntr)
		{
			tool = d_tool_utilities_dialog->Tool(cntr);
			
			for(entry_iter = tool.tool_entries.begin();entry_iter != tool.tool_entries.end();++entry_iter)
			{
				d_tool_utilities_dialog->Convert_Angle_Signature((*entry_iter).angle_ab_signature,&angle_a,&angle_b);
				list.push_back(TToolWidget::Format_Tool_Name(tool.tool_base_name,angle_a,angle_b));
			}
		}
	}
	
	return list;
}

TToolWidget::TToolItem TToolWidget::Tool_Item(
	const QString 						&name,
	bool								* const valid) const
{
	TToolData::TToolEntry				tool_entry;
	TToolWidget::TToolItem				tool_item;
	std::vector<TToolWidget::TToolItem>::const_iterator iter;
	QString								tool_base_name;
	TVector3							offset;
	TVector3							vector;
		
	*valid = false;
	
	tool_item.name = name;
	tool_item.angle_a = 0.0;
	tool_item.angle_b = 0.0;
	tool_item.tip_diameter = 0.0;

	if(d_controller_type == TController::CONTROLLER_IPPCLIENT)
	{
		for(iter = d_ipp_tool_items.begin();iter != d_ipp_tool_items.end();++iter)
		{
			if((*iter).name == name)
			{
				tool_item = (*iter);
				
				*valid = true;
				break;
			}
		}
	}
	else
	{
		if(TToolWidget::Extract_Tool_Name_Data(name,&tool_base_name,&tool_item.angle_a,&tool_item.angle_b))
		{
			tool_entry = d_tool_utilities_dialog->Tool_Entry(tool_base_name,tool_item.angle_a,tool_item.angle_b,&offset,&vector,valid);
			
			tool_item.xyz = offset;
			tool_item.ijk = vector;
			tool_item.tip_diameter = tool_entry.tip_diameter;
		}
	}
	
	return tool_item;
}

void TToolWidget::Invalidate_Active_Tool(void)
{
	d_active_tool_valid = false;
	d_active_tool_name.clear();
	
	d_information_widget->Set_Value_Text(d_tool_name_index,QString());
	d_information_widget->Set_Value_Text(d_tool_offset_x_index,QString());
	d_information_widget->Set_Value_Text(d_tool_offset_y_index,QString());
	d_information_widget->Set_Value_Text(d_tool_offset_z_index,QString());
	d_information_widget->Set_Value_Text(d_tool_diam_index,QString());
}

void TToolWidget::Tool_Item_Clicked(void)
{
	std::vector<TToolData::TToolEntry>	tool_entries;
	std::vector<TToolData::TToolEntry>::iterator iter_entries;
	TToolData::TToolEntry				tool_entry;
	TToolItem							tool_item;
	TVector3							offset;
	TVector3							vector;
	double								angle_a;
	double								angle_b;
	bool								valid;
	
	d_tool_utilities_dialog->Reset(QString());
	d_calibration_tool_items.clear();
	
	d_tool_utilities_dialog->exec();
	
	if(d_tool_utilities_dialog->Calibrate_On_Close())
	{
		d_calibration_tool_name = d_tool_utilities_dialog->Calibrate_Tool_Name();
		tool_entries = d_tool_utilities_dialog->Calibrate_Tool_Data();
		
		for(iter_entries = tool_entries.begin();iter_entries != tool_entries.end();++iter_entries)
		{
			d_tool_utilities_dialog->Convert_Angle_Signature((*iter_entries).angle_ab_signature,&angle_a,&angle_b);
			
			tool_entry = d_tool_utilities_dialog->Tool_Entry(d_calibration_tool_name,angle_a,angle_b,&offset,&vector,&valid);
			
			if(valid)
			{
				tool_item.name = TToolWidget::Format_Tool_Name(d_calibration_tool_name,angle_a,angle_b);
				
				tool_item.angle_a = angle_a;
				tool_item.angle_b = angle_b;
				tool_item.tip_diameter = tool_entry.tip_diameter;
				tool_item.xyz = offset;
				tool_item.ijk = vector;
				
				d_calibration_tool_items.push_back(tool_item);
			}
		}
		
		if(d_calibration_tool_items.size())
		{
			emit Calibrate_Tools();
		}
	}
}

void TToolWidget::Clear_Ipp_Tools(void)
{
	d_ipp_tool_items.clear();
}

void TToolWidget::Add_Ipp_Tool(
	const QString						&name)
{
	TToolWidget::TToolItem				tool_item;
	
	tool_item.name = name;
	tool_item.angle_a = 0.0;
	tool_item.angle_b = 0.0;
	tool_item.xyz.Set(0,0,0);
	tool_item.ijk.Set(0,0,-1);
	tool_item.tip_diameter = 0.0;
	
	d_ipp_tool_items.push_back(tool_item);
}

void TToolWidget::Update_Ipp_Tool(
	const TToolWidget::TToolItem 		&item)
{
	std::vector<TToolWidget::TToolItem>::iterator iter;
	
	for(iter = d_ipp_tool_items.begin();iter != d_ipp_tool_items.end();++iter)
	{
		if((*iter).name == item.name)
		{
			*iter = item;
			break;
		}
	}
}

bool TToolWidget::Load_Tool_Data(
	const QString						&path)
{	
	return 	d_tool_utilities_dialog->Load_Tool_Data(path);
}

void TToolWidget::Update_Tool_Tip_Diameter(
	const QString						&tool_base_name,
	const double						&angle_a,
	const double						&angle_b,
	const double						&tip_diameter)
{
	d_tool_utilities_dialog->Update_Tool_Tip_Diameter(tool_base_name,angle_a,angle_b,tip_diameter);

	if(d_active_tool_valid && d_active_tool_name == tool_base_name)
	{
		d_information_widget->Set_Value_Text(d_tool_diam_index,QString("%1").arg(tip_diameter,0,'f',4));
	}
}

void TToolWidget::Update_Tool_Information(
	const TToolWidget::TToolItem 		&tool_item)
{
	if(d_active_tool_valid && d_active_tool_name.compare(tool_item.name) == 0)
	{
		d_information_widget->Set_Value_Text(d_tool_name_index,tool_item.name);
		d_information_widget->Set_Value_Text(d_tool_offset_x_index,QString("%1").arg(tool_item.xyz.x,0,'f',4));
		d_information_widget->Set_Value_Text(d_tool_offset_y_index,QString("%1").arg(tool_item.xyz.y,0,'f',4));
		d_information_widget->Set_Value_Text(d_tool_offset_z_index,QString("%1").arg(tool_item.xyz.z,0,'f',4));
		d_information_widget->Set_Value_Text(d_tool_diam_index,QString("%1").arg(tool_item.tip_diameter,0,'f',4));
	}
}

void TToolWidget::Update_Tool_Information_Offsets(
	const double 						&x,
	const double 						&y,
	const double 						&z)
{
	d_information_widget->Set_Value_Text(d_tool_offset_x_index,QString("%1").arg(x,0,'f',4));
	d_information_widget->Set_Value_Text(d_tool_offset_y_index,QString("%1").arg(y,0,'f',4));
	d_information_widget->Set_Value_Text(d_tool_offset_z_index,QString("%1").arg(z,0,'f',4));
}

QString TToolWidget::Format_Tool_Name(
	const QString 						&tool_base_name,
	const double 						&angle_a,
	const double 						&angle_b)
{
	QString								tool_name;
	
	tool_name = QString("%1_A%2_B%3").arg(tool_base_name).arg(angle_a,0,'f',1).arg(angle_b,0,'f',1);

	return tool_name;
}

bool TToolWidget::Extract_Tool_Name_Data(
	const QString 						&tool_name,
	QString 							* const tool_base_name,
	double 								* const angle_a,
	double 								* const angle_b)
{
	QRegularExpression					regular_expression;
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	int									index;

	assert(tool_base_name);
	assert(angle_a);
	assert(angle_b);
	
	regular_expression.setPattern(QStringLiteral("[_A]{2}([-0-9.]{3,6})[_B]{2}([-0-9.]{3,6})$"));
	expression_match = regular_expression.match(tool_name);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
				
		if(expression_text_list.size() > 2)
		{
			*angle_a = expression_text_list[1].toDouble();
			*angle_b = expression_text_list[2].toDouble();
		
			index = expression_match.capturedStart();
			*tool_base_name = tool_name;
			tool_base_name->truncate(index);
			
			return true;
		}
	}

	return false;
}

