/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QFrame>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QPushButton>
#include <QSpacerItem>
#include <QSpinBox>
#include <QToolButton>
#include <QDateTime>
#include <QFont>
#include <cmath>
#include <assert.h>

#include "../../../core/mat4.h"
#include "../../../core/bestfitsphere.h"
#include "../../../core/bestfitplane.h"
#include "../../../core/bestfitcircle.h"
#include "../../../core/bestfitline.h"
#include "../../../core/messagebox.h"


#include "validationdialog.h"

TValidationDialog::TValidationDialog(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QDialog(const_cast<QWidget*>(parent),flags)
{
	QFrame								*seperator_hline1;
	QFrame								*seperator_hline2;
	QGridLayout							*dialog_layout;
	QHBoxLayout							*button_hlayout;
	QLabel								*circle_label;
	QLabel								*line_label;
	QLabel								*maximum_points_label;
	QLabel								*minimum_points_label;
	QLabel								*plane_label;
	QLabel								*sphere_label;
	QLabel								*total_sets_label;
	QLabel								*minimum_points_comment_label;
	QPushButton							*close_button;
	QSpacerItem							*button_hspacer;
	QSpacerItem							*dialog_vspacer;
	QSpacerItem							*option_hspacer;
	std::random_device					random_device;
	QFont								information_font;
	static const int					MAX_POINTS(100);
	static const int					MAX_SETS(50);
	
	information_font.setItalic(true);

	this->resize(447,332);

	dialog_layout = new QGridLayout(this);

	minimum_points_label = new QLabel(this);
	dialog_layout->addWidget(minimum_points_label,0,0,1,1);

	d_min_points_spin = new QSpinBox(this);
	d_min_points_spin->setMinimum(2);
	d_min_points_spin->setMaximum(4);
	dialog_layout->addWidget(d_min_points_spin,0,1,1,1);

	minimum_points_comment_label = new QLabel(this);
	minimum_points_comment_label->setFont(information_font);
	dialog_layout->addWidget(minimum_points_comment_label,0,2,1,1);
	
	maximum_points_label = new QLabel(this);
	dialog_layout->addWidget(maximum_points_label,1,0,1,1);

	d_max_points_spin = new QSpinBox(this);
	d_max_points_spin->setMinimum(5);
	d_max_points_spin->setMaximum(MAX_POINTS);
	dialog_layout->addWidget(d_max_points_spin,1,1,1,1);

	total_sets_label = new QLabel(this);
	dialog_layout->addWidget(total_sets_label,2,0,1,1);

	d_total_sets_spin = new QSpinBox(this);
	d_total_sets_spin->setMinimum(2);
	d_total_sets_spin->setMaximum(MAX_SETS);
	dialog_layout->addWidget(d_total_sets_spin,2,1,1,1);

	seperator_hline1 = new QFrame(this);
	seperator_hline1->setFrameShape(QFrame::HLine);
	seperator_hline1->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(seperator_hline1,3,0,1,3);

	line_label = new QLabel(this);
	line_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(line_label,4,0,1,1);

	d_generate_line_button = new QToolButton(this);
	dialog_layout->addWidget(d_generate_line_button,4,1,1,1);

	plane_label = new QLabel(this);
	plane_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(plane_label,5,0,1,1);

	d_generate_plane_button = new QToolButton(this);
	dialog_layout->addWidget(d_generate_plane_button,5,1,1,1);

	sphere_label = new QLabel(this);
	sphere_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(sphere_label,6,0,1,1);

	d_generate_sphere_button = new QToolButton(this);
	dialog_layout->addWidget(d_generate_sphere_button,6,1,1,1);

	circle_label = new QLabel(this);
	circle_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(circle_label,7,0,1,1);

	d_generate_circle_button = new QToolButton(this);
	dialog_layout->addWidget(d_generate_circle_button,7,1,1,1);
	
	option_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	dialog_layout->addItem(option_hspacer,4,2,4,1);

	seperator_hline2 = new QFrame(this);
	seperator_hline2->setFrameShape(QFrame::HLine);
	seperator_hline2->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(seperator_hline2,8,0,1,3);

	dialog_vspacer = new QSpacerItem(0,0,QSizePolicy::Minimum,QSizePolicy::Expanding);
	dialog_layout->addItem(dialog_vspacer,9,0,1,3);

	button_hlayout = new QHBoxLayout();

	button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_hlayout->addItem(button_hspacer);

	close_button = new QPushButton(this);
	close_button->setAutoDefault(false);
	button_hlayout->addWidget(close_button);
	dialog_layout->addLayout(button_hlayout,10,0,1,3);


	QWidget::setTabOrder(d_min_points_spin,d_max_points_spin);
	QWidget::setTabOrder(d_max_points_spin,d_total_sets_spin);
	QWidget::setTabOrder(d_total_sets_spin,d_generate_line_button);
	QWidget::setTabOrder(d_generate_line_button,d_generate_plane_button);
	QWidget::setTabOrder(d_generate_plane_button,d_generate_sphere_button);
	QWidget::setTabOrder(d_generate_sphere_button,d_generate_circle_button);
	QWidget::setTabOrder(d_generate_circle_button,close_button);
	
	d_random_engine.seed(random_device());
	
	d_message_box = new TMessageBox(this);
	
	// defaults
	d_min_points_spin->setValue(4);
	d_max_points_spin->setValue(MAX_POINTS);
	d_total_sets_spin->setValue(MAX_SETS);
	
	this->setWindowTitle(QStringLiteral("Validation Dialog"));
	
	minimum_points_label->setText(QStringLiteral("Minimum Points:"));
	minimum_points_comment_label->setText(QStringLiteral("Note: First result at feature minimum."));
	maximum_points_label->setText(QStringLiteral("Maximum Points:"));
	total_sets_label->setText(QStringLiteral("Total Sets:"));
	line_label->setText(QStringLiteral("Line"));
	plane_label->setText(QStringLiteral("Plane"));
	sphere_label->setText(QStringLiteral("Sphere"));
	circle_label->setText(QStringLiteral("Circle"));
	close_button->setText(QStringLiteral("Close"));
	
	d_generate_line_button->setText(QStringLiteral("Generate"));
	d_generate_plane_button->setText(QStringLiteral("Generate"));
	d_generate_circle_button->setText(QStringLiteral("Generate"));
	d_generate_sphere_button->setText(QStringLiteral("Generate"));
	
	connect(close_button,&QPushButton::clicked,this,&TValidationDialog::close);
	connect(d_min_points_spin,static_cast<void (QSpinBox::*)(int)>(&QSpinBox::valueChanged),this,&TValidationDialog::Min_Points_Changed);
	connect(d_generate_line_button,&QPushButton::clicked,this,&TValidationDialog::Generate_Line);
	connect(d_generate_plane_button,&QPushButton::clicked,this,&TValidationDialog::Generate_Plane);
	connect(d_generate_circle_button,&QPushButton::clicked,this,&TValidationDialog::Generate_Circle);
	connect(d_generate_sphere_button,&QPushButton::clicked,this,&TValidationDialog::Generate_Sphere);
}

TValidationDialog::~TValidationDialog(void)
{
}

void TValidationDialog::Reset(
	const QString						&output_path)
{
	d_output_path = output_path;
}

void TValidationDialog::Min_Points_Changed(
	int									point_count)
{
	d_generate_line_button->setEnabled(false);
	d_generate_plane_button->setEnabled(false);
	d_generate_sphere_button->setEnabled(false);
	d_generate_circle_button->setEnabled(false);

	if(point_count == 2)
	{
		d_generate_line_button->setEnabled(true);
	}
	else if(point_count == 3)
	{
		d_generate_line_button->setEnabled(true);
		d_generate_plane_button->setEnabled(true);
		d_generate_circle_button->setEnabled(true);
	}
	else if(point_count > 3)
	{
		d_generate_line_button->setEnabled(true);
		d_generate_plane_button->setEnabled(true);
		d_generate_sphere_button->setEnabled(true);
		d_generate_circle_button->setEnabled(true);
	}
}

void TValidationDialog::Generate_Line(void)
{
	QString								file_name;
	QDateTime							date_time;
	std::vector<TVector3>				pnts;
	TVector3							v1,v2,v3;
	TVector3							pt;
	TVector3							offset;
	TMat4								mat;
	int									point_count;
	int									point_count_max;
	int									point_count_min;
	int									set_count;
	int									cntr;
	double								length_increment;
	double								length;
	static const double					HALF_LENGTH(50.0);
	static const double					MINIMUM_DOT(0.10000);
	static const double					MAX_RADIAL_ERROR(0.050);
	
	if(d_output_file.isOpen())
	{
		d_output_file.close();
	}
	
	date_time = QDateTime::currentDateTime();
	
	file_name = d_output_path;
	file_name += QString("/Line Test Data %1.dat").arg(date_time.toString("[ddd MMMM d yyyy hh-mm-ss]"));
	
	d_output_file.setFileName(file_name);
	
	if(!d_output_file.open(QIODevice::WriteOnly))
	{
		return;
	}
	
	d_output_file.write("Best_Fit_Validation:Version=1:Type=Line\r\n");
	
	set_count = d_total_sets_spin->value();
	
	point_count_max = d_max_points_spin->value();
	point_count_min = d_min_points_spin->value();
	
	v1 = this->Random_Axis();

	do
	{
		v2 = this->Random_Axis();
		
	} while(fabs(v1^v2) < MINIMUM_DOT);
	
	v3 = v1 * v2;
	v2 = v3 * v1;
	
	v2.Normal();
	v3.Normal();
	
	mat.X(v1);
	mat.Y(v2);
	mat.Z(v3);
	
	mat.Transpose();
	
	offset.x = this->Random(1000.0);
	offset.y = this->Random(1000.0);
	offset.z = this->Random(1000.0);
	
	// two point line
	pt.Set(0.0,0.0,HALF_LENGTH);
	pt += offset;
	pnts.push_back(mat * pt);
	
	pt.Set(0.0,0.0,-1.0 * HALF_LENGTH);
	pt += offset;
	pnts.push_back(mat * pt);
	
	this->Output_Line_Results(set_count,pnts);
	pnts.clear();
	
	--set_count;
	
	while(set_count > 0)
	{
		v1 = this->Random_Axis();

		do
		{
			v2 = this->Random_Axis();
			
		} while(fabs(v1^v2) < MINIMUM_DOT);
		
		v3 = v1 * v2;
		v2 = v3 * v1;
		
		v2.Normal();
		v3.Normal();
		
		mat.X(v1);
		mat.Y(v2);
		mat.Z(v3);

		mat.Transpose();

		do
		{
			point_count = this->Random(point_count_max);
			
		}while(point_count < point_count_min);
		
		length = HALF_LENGTH;
		length_increment = (2.0 * HALF_LENGTH) / static_cast<double>(point_count - 1);
		
		offset.x = this->Random(1000.0);
		offset.y = this->Random(1000.0);
		offset.z = this->Random(1000.0);

		for(cntr = 0;cntr < point_count;++cntr)
		{
			pt.Set(this->Random(MAX_RADIAL_ERROR),this->Random(MAX_RADIAL_ERROR),length);
			pt += offset;
			pnts.push_back(mat * pt);

			length -= length_increment;
		}
		
		this->Output_Line_Results(set_count,pnts);
		pnts.clear();

		--set_count;
	};
	
	d_output_file.close();
	
	d_message_box->setText("Data Generated");
	d_message_box->setInformativeText("The test data for the feature has been created");
	d_message_box->setDetailedText(QString());
	d_message_box->setStandardButtons(QMessageBox::Ok);
	d_message_box->setDefaultButton(QMessageBox::Ok);
	d_message_box->setIcon(QMessageBox::Information);
	
	d_message_box->exec();
}

void TValidationDialog::Generate_Plane(void)
{
	QString								file_name;
	QDateTime							date_time;
	std::vector<TVector3>				pnts;
	TVector3							v1,v2,v3;
	TVector3							pt;
	TVector3							offset;
	TMat4								mat;
	int									point_count;
	int									point_count_max;
	int									point_count_min;
	int									set_count;
	int									cntr;
	static const double					HALF_LENGTH(50.0);
	static const double					HALF_WIDTH(25.0);
	static const double					MINIMUM_DOT(0.10000);
	static const double					MAX_RADIAL_ERROR(0.050);
	
	if(d_output_file.isOpen())
	{
		d_output_file.close();
	}
	
	date_time = QDateTime::currentDateTime();
	
	file_name = d_output_path;
	file_name += QString("/Plane Test Data %1.dat").arg(date_time.toString("[ddd MMMM d yyyy hh-mm-ss]"));
	
	d_output_file.setFileName(file_name);
	
	if(!d_output_file.open(QIODevice::WriteOnly))
	{
		return;
	}
	
	d_output_file.write("Best_Fit_Validation:Version=1:Type=Plane\r\n");
	
	set_count = d_total_sets_spin->value();
	
	point_count_max = d_max_points_spin->value();
	point_count_min = d_min_points_spin->value();
	
	v1 = this->Random_Axis();
	
	do
	{
		v2 = this->Random_Axis();
		
	} while(fabs(v1^v2) < MINIMUM_DOT);
	
	v3 = v1 * v2;
	v2 = v3 * v1;
	
	v2.Normal();
	v3.Normal();
	
	mat.X(v1);
	mat.Y(v2);
	mat.Z(v3);
	
	mat.Transpose();

	offset.x = this->Random(1000.0);
	offset.y = this->Random(1000.0);
	offset.z = this->Random(1000.0);
	
	// three point plane
	pt.Set(HALF_LENGTH,HALF_WIDTH,0.0);
	pt += offset;
	pnts.push_back(mat * pt);
	
	pt.Set(-1.0 * HALF_LENGTH,HALF_WIDTH,0.0);
	pt += offset;
	pnts.push_back(mat * pt);
	
	pt.Set(0.0,-1.0 * HALF_WIDTH,0.0);
	pt += offset;
	pnts.push_back(mat * pt);
	
	this->Output_Plane_Results(set_count,pnts);
	pnts.clear();
	
	--set_count;
	
	while(set_count > 0)
	{
		v1 = this->Random_Axis();
		
		do
		{
			v2 = this->Random_Axis();
			
		} while(fabs(v1^v2) < MINIMUM_DOT);
		
		v3 = v1 * v2;
		v2 = v3 * v1;
		
		v2.Normal();
		v3.Normal();
		
		mat.X(v1);
		mat.Y(v2);
		mat.Z(v3);
		
		mat.Transpose();

		do
		{
			point_count = this->Random(point_count_max);
			
		}while(point_count < point_count_min);
		
		offset.x = this->Random(1000.0);
		offset.y = this->Random(1000.0);
		offset.z = this->Random(1000.0);
		
		for(cntr = 0;cntr < point_count;++cntr)
		{
			pt.Set(this->Random(HALF_LENGTH),this->Random(HALF_WIDTH),this->Random(MAX_RADIAL_ERROR));
			pt += offset;
			pnts.push_back(mat * pt);
		}
		
		this->Output_Plane_Results(set_count,pnts);
		pnts.clear();
		
		--set_count;
	};
	
	d_output_file.close();
	
	d_message_box->setText("Data Generated");
	d_message_box->setInformativeText("The test data for the feature has been created");
	d_message_box->setDetailedText(QString());
	d_message_box->setStandardButtons(QMessageBox::Ok);
	d_message_box->setDefaultButton(QMessageBox::Ok);
	d_message_box->setIcon(QMessageBox::Information);
	
	d_message_box->exec();
}

void TValidationDialog::Generate_Circle(void)
{
	QString								file_name;
	QDateTime							date_time;
	std::vector<TVector3>				pnts;
	TVector3							v1,v2,v3;
	TVector3							pt;
	TVector3							offset;
	TMat4								mat;
	int									point_count;
	int									point_count_max;
	int									point_count_min;
	int									set_count;
	int									cntr;
	double								angle;
	static const double					RADIUS(50.0);
	static const double					MINIMUM_DOT(0.10000);
	static const double					MAX_RADIAL_ERROR(0.050);
	static const double					MAX_ANGLE(6.283185307179586);
	
	if(d_output_file.isOpen())
	{
		d_output_file.close();
	}
	
	date_time = QDateTime::currentDateTime();
	
	file_name = d_output_path;
	file_name += QString("/Circle Test Data %1.dat").arg(date_time.toString("[ddd MMMM d yyyy hh-mm-ss]"));
	
	d_output_file.setFileName(file_name);
	
	if(!d_output_file.open(QIODevice::WriteOnly))
	{
		return;
	}
	
	d_output_file.write("Best_Fit_Validation:Version=1:Type=Circle\r\n");
	
	set_count = d_total_sets_spin->value();
	
	point_count_max = d_max_points_spin->value();
	point_count_min = d_min_points_spin->value();
	
	v1 = this->Random_Axis();
	
	if(fabs(v1.z) > fabs(v1.x) && fabs(v1.z) > fabs(v1.y))
	{
		v1.Set(0,0,1);
	}
	else if(fabs(v1.y) > fabs(v1.x) && fabs(v1.y) > fabs(v1.z))
	{
		v1.Set(0,1,0);
	}
	else
	{
		v1.Set(1,0,0);
	}
	
	do
	{
		v2 = this->Random_Axis();
		
	} while(fabs(v1^v2) < MINIMUM_DOT);
	
	v3 = v1 * v2;
	v2 = v3 * v1;
	
	v2.Normal();
	v3.Normal();
	
	mat.X(v2);
	mat.Y(v3);
	mat.Z(v1);
	
	offset.x = this->Random(1000.0);
	offset.y = this->Random(1000.0);
	offset.z = this->Random(1000.0);
	
	mat.Transpose();
	
	// three point circle
	pt.Set(RADIUS,0.0,0.0);
	pt += offset;
	pnts.push_back(mat * pt);

	pt.Set(-1.0 * RADIUS,0.0,0.0);
	pt += offset;
	pnts.push_back(mat * pt);

	pt.Set(0.0,RADIUS,0.0);
	pt += offset;
	pnts.push_back(mat * pt);

	this->Output_Circle_Results(set_count,pnts,v1);
	pnts.clear();
	
	--set_count;
	
	while(set_count > 0)
	{
		v1 = this->Random_Axis();
		
		if(fabs(v1.z) > fabs(v1.x) && fabs(v1.z) > fabs(v1.y))
		{
			v1.Set(0,0,1);
		}
		else if(fabs(v1.y) > fabs(v1.x) && fabs(v1.y) > fabs(v1.z))
		{
			v1.Set(0,1,0);
		}
		else
		{
			v1.Set(1,0,0);
		}
		
		do
		{
			v2 = this->Random_Axis();
			
		} while(fabs(v1^v2) < MINIMUM_DOT);
		
		v3 = v1 * v2;
		v2 = v3 * v1;
		
		v2.Normal();
		v3.Normal();
		
		mat.X(v2);
		mat.Y(v3);
		mat.Z(v1);
		
		mat.Transpose();

		do
		{
			point_count = this->Random(point_count_max);
			
		}while(point_count < point_count_min);
		
		offset.x = this->Random(1000.0);
		offset.y = this->Random(1000.0);
		offset.z = this->Random(1000.0);
		
		for(cntr = 0;cntr < point_count;++cntr)
		{
			angle = this->Random(MAX_ANGLE);

			pt.Set(cos(angle) * RADIUS + this->Random(MAX_RADIAL_ERROR),sin(angle) * RADIUS + this->Random(MAX_RADIAL_ERROR),0.0);
			pt += offset;
			pnts.push_back(mat * pt);
		}
		
		this->Output_Circle_Results(set_count,pnts,v1);
		pnts.clear();
		
		--set_count;
	};
	
	d_output_file.close();
	
	d_message_box->setText("Data Generated");
	d_message_box->setInformativeText("The test data for the feature has been created");
	d_message_box->setDetailedText(QString());
	d_message_box->setStandardButtons(QMessageBox::Ok);
	d_message_box->setDefaultButton(QMessageBox::Ok);
	d_message_box->setIcon(QMessageBox::Information);
	
	d_message_box->exec();
}

void TValidationDialog::Generate_Sphere(void)
{
	QString								file_name;
	QDateTime							date_time;
	std::vector<TVector3>				pnts;
	TVector3							v1,v2,v3;
	TVector3							pt;
	TVector3							offset;
	TMat4								mat;
	int									point_count;
	int									point_count_max;
	int									point_count_min;
	int									set_count;
	int									cntr;
	double								azimuth,elevation;
	static const double					RADIUS(50.0);
	static const double					MINIMUM_DOT(0.10000);
	static const double					MAX_RADIAL_ERROR(0.050);
	static const double					MAX_AZIMUTH(6.283185307179586);
	static const double					MAX_ELEVATION(1.570796326794897);

	if(d_output_file.isOpen())
	{
		d_output_file.close();
	}
	
	date_time = QDateTime::currentDateTime();
	
	file_name = d_output_path;
	file_name += QString("/Sphere Test Data %1.dat").arg(date_time.toString("[ddd MMMM d yyyy hh-mm-ss]"));
	
	d_output_file.setFileName(file_name);
	
	if(!d_output_file.open(QIODevice::WriteOnly))
	{
		return;
	}
	
	d_output_file.write("Best_Fit_Validation:Version=1:Type=Sphere\r\n");
	
	set_count = d_total_sets_spin->value();
	
	point_count_max = d_max_points_spin->value();
	point_count_min = d_min_points_spin->value();
	
	v1 = this->Random_Axis();
	
	do
	{
		v2 = this->Random_Axis();
		
	} while(fabs(v1^v2) < MINIMUM_DOT);
	
	v3 = v1 * v2;
	v2 = v3 * v1;
	
	v2.Normal();
	v3.Normal();
	
	mat.X(v1);
	mat.Y(v2);
	mat.Z(v3);
	
	mat.Transpose();
	
	offset.x = this->Random(1000.0);
	offset.y = this->Random(1000.0);
	offset.z = this->Random(1000.0);
	
	// four point sphere
	pt.Set(RADIUS,0.0,0.0);
	pt += offset;
	pnts.push_back(mat * pt);
	
	pt.Set(0.0,RADIUS,0.0);
	pt += offset;
	pnts.push_back(mat * pt);
	
	pt.Set(-1.0 * RADIUS,0.0,0.0);
	pt += offset;
	pnts.push_back(mat * pt);

	pt.Set(0.0,0.0,RADIUS);
	pt += offset;
	pnts.push_back(mat * pt);

	this->Output_Sphere_Results(set_count,pnts);
	pnts.clear();
	
	--set_count;
	
	while(set_count > 0)
	{
		v1 = this->Random_Axis();
		
		do
		{
			v2 = this->Random_Axis();
			
		} while(fabs(v1^v2) < MINIMUM_DOT);
		
		v3 = v1 * v2;
		v2 = v3 * v1;
		
		v2.Normal();
		v3.Normal();
		
		mat.X(v1);
		mat.Y(v2);
		mat.Z(v3);
		
		mat.Transpose();
		
		do
		{
			point_count = this->Random(point_count_max);
			
		}while(point_count < point_count_min);
		
		offset.x = this->Random(1000.0);
		offset.y = this->Random(1000.0);
		offset.z = this->Random(1000.0);
		
		for(cntr = 0;cntr < point_count;++cntr)
		{
			azimuth = this->Random(MAX_AZIMUTH);
			elevation = this->Random(MAX_ELEVATION);

			pt.Set(cos(azimuth) * cos(elevation) * RADIUS + this->Random(MAX_RADIAL_ERROR),sin(azimuth) * cos(elevation) * RADIUS + this->Random(MAX_RADIAL_ERROR),sin(elevation) * RADIUS + this->Random(MAX_RADIAL_ERROR));
			pt += offset;
			pnts.push_back(mat * pt);
		}
		
		this->Output_Sphere_Results(set_count,pnts);
		pnts.clear();
		
		--set_count;
	};
	
	d_output_file.close();
	
	d_message_box->setText("Data Generated");
	d_message_box->setInformativeText("The test data for the feature has been created");
	d_message_box->setDetailedText(QString());
	d_message_box->setStandardButtons(QMessageBox::Ok);
	d_message_box->setDefaultButton(QMessageBox::Ok);
	d_message_box->setIcon(QMessageBox::Information);
	
	d_message_box->exec();
}

void TValidationDialog::Output_Line_Results(
	const int							index,
	std::vector<TVector3>				&pnts)
{
	QString								text;
	TBestFitLine						line;
	TVector3							pos;
	TVector3							vec;
	double								form;
	std::vector<TVector3>::const_iterator iter;
	
	if(line.Fit(static_cast<int>(pnts.size()),&pnts[0]))
	{
		text = QString("Line %1 %2 ").arg(index).arg(pnts.size());
		d_output_file.write(text.toLatin1());
		
		pos = line.Position();
		vec = line.Axis();
		form = line.FormError();

		text = QString("%1 %2 %3 %4 %5 %6 %7\r\n")
					.arg(pos.x,0,'f',6)
					.arg(pos.y,0,'f',6)
					.arg(pos.z,0,'f',6)
					.arg(vec.i,0,'f',9)
					.arg(vec.j,0,'f',9)
					.arg(vec.k,0,'f',9)
					.arg(form,0,'f',6);
		
		d_output_file.write(text.toLatin1());
		
		for(iter = pnts.begin();iter != pnts.end();++iter)
		{
			text = QString("  %1 %2 %3\r\n")
			.arg((*iter).x,0,'f',6)
			.arg((*iter).y,0,'f',6)
			.arg((*iter).z,0,'f',6);
			
			d_output_file.write(text.toLatin1());
		}
	}
}

void TValidationDialog::Output_Plane_Results(
	const int							index,
	std::vector<TVector3>				&pnts)
{
	QString								text;
	TBestFitPlane						plane;
	TVector3							pos;
	TVector3							vec;
	double								form;
	std::vector<TVector3>::const_iterator iter;
	
	if(plane.Fit(static_cast<int>(pnts.size()),&pnts[0]))
	{
		text = QString("Plane %1 %2 ").arg(index).arg(pnts.size());
		d_output_file.write(text.toLatin1());
		
		pos = plane.Position();
		vec = plane.Axis();
		form = plane.FormError();
		
		text = QString("%1 %2 %3 %4 %5 %6 %7\r\n")
						.arg(pos.x,0,'f',6)
						.arg(pos.y,0,'f',6)
						.arg(pos.z,0,'f',6)
						.arg(vec.i,0,'f',9)
						.arg(vec.j,0,'f',9)
						.arg(vec.k,0,'f',9)
						.arg(form,0,'f',6);
		
		d_output_file.write(text.toLatin1());
		
		for(iter = pnts.begin();iter != pnts.end();++iter)
		{
			text = QString("  %1 %2 %3\r\n")
						.arg((*iter).x,0,'f',6)
						.arg((*iter).y,0,'f',6)
						.arg((*iter).z,0,'f',6);
			
			d_output_file.write(text.toLatin1());
		}
	}
}

void TValidationDialog::Output_Circle_Results(
	const int							index,
	std::vector<TVector3>				&pnts,
	const TVector3						&axis)
{
	QString								text;
	TBestFitCircle						circle;
	TVector3							pos;
	TVector3							vec;
	double								form;
	double								diameter;
	std::vector<TVector3>::const_iterator iter;
	
	if(circle.Fit(axis,static_cast<int>(pnts.size()),&pnts[0]))
	{
		text = QString("Circle %1 %2 ").arg(index).arg(pnts.size());
		d_output_file.write(text.toLatin1());
		
		pos = circle.Position();
		vec = circle.Axis();
		diameter = circle.Diameter();
		form = circle.FormError();
		
		text = QString("%1 %2 %3 %4 %5 %6 %7 %8\r\n")
					.arg(pos.x,0,'f',6)
					.arg(pos.y,0,'f',6)
					.arg(pos.z,0,'f',6)
					.arg(vec.i,0,'f',9)
					.arg(vec.j,0,'f',9)
					.arg(vec.k,0,'f',9)
					.arg(diameter,0,'f',6)
					.arg(form,0,'f',6);
		
		d_output_file.write(text.toLatin1());
		
		for(iter = pnts.begin();iter != pnts.end();++iter)
		{
			text = QString("  %1 %2 %3\r\n")
			.arg((*iter).x,0,'f',6)
			.arg((*iter).y,0,'f',6)
			.arg((*iter).z,0,'f',6);
			
			d_output_file.write(text.toLatin1());
		}
	}
}

void TValidationDialog::Output_Sphere_Results(
	const int							index,
	std::vector<TVector3>				&pnts)
{
	QString								text;
	TBestFitSphere						sphere;
	TVector3							pos;
	double								form;
	double								diam;
	std::vector<TVector3>::const_iterator iter;
	
	if(sphere.Fit(static_cast<int>(pnts.size()),&pnts[0]))
	{
		text = QString("Sphere %1 %2 ").arg(index).arg(pnts.size());
		d_output_file.write(text.toLatin1());
		
		pos = sphere.Position();
		form = sphere.FormError();
		diam = sphere.Diameter();
		
		text = QString("%1 %2 %3 %4 %5\r\n")
				.arg(pos.x,0,'f',6)
				.arg(pos.y,0,'f',6)
				.arg(pos.z,0,'f',6)
				.arg(diam,0,'f',6)
				.arg(form,0,'f',6);
		
		d_output_file.write(text.toLatin1());
		
		for(iter = pnts.begin();iter != pnts.end();++iter)
		{
			text = QString("  %1 %2 %3\r\n")
					.arg((*iter).x,0,'f',6)
					.arg((*iter).y,0,'f',6)
					.arg((*iter).z,0,'f',6);
			
			d_output_file.write(text.toLatin1());
		}
	}
}

TVector3 TValidationDialog::Random_Axis(void)
{
	TVector3							axis;
	static const double					ZERO_EPSILON(0.000001);
	
	do
	{
		axis.x = this->Random();
		axis.y = this->Random();
		axis.z = this->Random();
		
	}while(axis.Length() < ZERO_EPSILON);
	
	axis.Normal();
	
	return axis;
}


double TValidationDialog::Random(void)
{
	std::uniform_real_distribution<double> random_range(-1.0, 1.0);
	
	return random_range(d_random_engine);
}

double TValidationDialog::Random(
	const double						&max_value)
{
	std::uniform_real_distribution<double> random_range(-1.0 * max_value, max_value);
	
	return random_range(d_random_engine);
}

int TValidationDialog::Random(
	const int							max_count)
{
	std::uniform_real_distribution<double> random_range(0.0, static_cast<double>(max_count) + 0.999);	// allow for truncation of upper limit
	
	return static_cast<int>(random_range(d_random_engine));
}


