/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QStringList>
#include <QByteArray>
#include <QTextStream>

#include "../../../core/vector2.h"

#include "measuredata_pftu.h"

TMeasurePFTU::TMeasurePFTU(void)
{
	d_file_type = TMeasureData::TYPE_SPHERE_PFTU;
}

TMeasurePFTU::~TMeasurePFTU(void)
{
}

QStringList TMeasurePFTU::Text_Report(void) const
{
	std::vector<TMeasurePFTU::TPointData>::const_iterator iter;
	QStringList							list;
	QString								text;
	TVector3							vec;
	
	list.push_back(QStringLiteral("ASME B89.4.10360 Pftu"));
	list.push_back(QStringLiteral("================================================================"));
	list.push_back(QString());

	list.push_back(QStringLiteral("Alignment"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	vec = d_alignment_data.X();
	text = QString("  X Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Y();
	text = QString("  Y Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Z();
	text = QString("  Z Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Origin();
	text = QString("  Origin: %1, %2, %3").arg(vec.x,12,'f',4).arg(vec.y,12,'f',4).arg(vec.z,12,'f',4);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Tool"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  Offset: %1, %2, %3").arg(tool_offset.x,12,'f',4).arg(tool_offset.y,12,'f',4).arg(tool_offset.z,12,'f',4);
	list.push_back(text);
	
	text = QString("  Vector: %1, %2, %3").arg(d_probe_data_vector.x,12,'f',9).arg(d_probe_data_vector.y,12,'f',9).arg(d_probe_data_vector.z,12,'f',9);
	list.push_back(text);

	list.push_back(QString());
	list.push_back(QStringLiteral("Feature"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  Nominal Rad.: %1 mm").arg(d_sphere_nominal_radius,0,'f',4);
	list.push_back(text);

	text = QString("   Description: %1").arg(d_sphere_id);
	list.push_back(text);

	text = QString("           XYZ: %1, %2, %3").arg(d_sphere_position.x,0,'f',4).arg(d_sphere_position.y,0,'f',4).arg(d_sphere_position.z,0,'f',4);
	list.push_back(text);

	text = QString("          Diam: %1 mm").arg(d_sphere_diameter,0,'f',4);
	list.push_back(text);

	text = QString("          Form: %1 mm").arg(d_sphere_form,0,'f',4);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Data Points"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	list.push_back(QStringLiteral("      X           Y           Z              I             J             K          Radius       Dev"));
	
	for(iter = d_point_data.begin();iter != d_point_data.end();++iter)
	{
		text = QString("%1, %2, %3,").arg((*iter).xyz.x,10,'f',4).arg((*iter).xyz.y,10,'f',4).arg((*iter).xyz.z,10,'f',4);
		text += QString("   %1, %2, %3,").arg((*iter).ijk.x,12,'f',9).arg((*iter).ijk.y,12,'f',9).arg((*iter).ijk.z,12,'f',9);
		text += QString(" %1, %2").arg((*iter).radius,10,'f',4).arg((*iter).deviation,10,'f',4);
		list.push_back(text);
	}
	
	return list;
}

std::vector<std::vector<TVector2> > TMeasurePFTU::Graph_Data(void) const
{
	std::vector<std::vector<TVector2> > graph_data;
	
	return graph_data;
}

bool TMeasurePFTU::Load_Data(
	const QByteArray					&file_data)
{
	QTextStream							stream(file_data,QIODevice::ReadOnly);
	TMeasurePFTU::TPointData			point_data;
	QString								line;
	QString								text;
	QString								text_section;
	QStringList							list;
	int									index;
	bool								found_alignment(false);
	bool								inside_alignment(false);
	bool								found_probe_offset(false);
	bool								found_probe_vector(false);
	bool								found_sphere_nominal(false);
	bool								found_results(false);
	bool								found_data(false);
	bool								inside_data(false);
	bool								init(true);
	
	d_point_data.clear();
	
	while(!stream.atEnd())
	{
		line = stream.readLine().trimmed();
		
		if(line.startsWith('#') ||
		   line.length() == 0)
		{
			continue;
		}
		
		if(init)
		{
			if(!line.startsWith(QStringLiteral("B89.4.10360_Raw_Measurement:Version=1:Type=Sphere_Pftu")))
			{
				d_last_error = QStringLiteral("Artifact identification line not recognized");
				return false;
			}
			
			init = false;
		}
		else
		{
			if(line.compare(QStringLiteral("Alignment_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_alignment = true;
				inside_alignment = true;
			}
			else if(line.compare(QStringLiteral("Alignment_End:"),Qt::CaseInsensitive) == 0)
			{
				inside_alignment = false;
			}
			else if(line.compare(QStringLiteral("Point_Data_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_data = true;
				inside_data = true;
			}
			else if(line.compare(QStringLiteral("Point_Data_End::"),Qt::CaseInsensitive) == 0)
			{
				inside_data = false;
			}
			else if(line.startsWith(QStringLiteral("Probe_Offset:"),Qt::CaseInsensitive))
			{
				found_probe_offset = true;
				
				text = line.mid(13);	// remove 'Probe_Offset:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					tool_offset.x = list[0].toDouble();
					tool_offset.y = list[1].toDouble();
					tool_offset.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Probe offset data not recognized.");
					return false;
				}
				
			}
			else if(line.startsWith(QStringLiteral("Probe_Vector:"),Qt::CaseInsensitive))
			{
				found_probe_vector = true;
				
				text = line.mid(13);	// remove 'Probe_Vector:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					d_probe_data_vector.x = list[0].toDouble();
					d_probe_data_vector.y = list[1].toDouble();
					d_probe_data_vector.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Probe vector data not recognized.");
					return false;
				}
			}
			else if(line.startsWith(QStringLiteral("Nominal_Sphere_Radius:"),Qt::CaseInsensitive))
			{
				found_sphere_nominal = true;
				
				text = line.mid(22);	// remove 'Nominal_Sphere_Radius:'
				d_sphere_nominal_radius = text.toDouble();
				
			}
			else
			{
				if(inside_alignment == false &&
				   inside_data == false)
				{
					if(line.startsWith(QStringLiteral("Sphere"),Qt::CaseInsensitive))
					{
						found_results = true;
						
						text = line;
						index = text.indexOf(QStringLiteral("Form:"),Qt::CaseInsensitive);
						
						if(index < 0)
						{
							d_last_error = QStringLiteral("Feature form error not found.");
							return false;
						}
						
						text_section = text.mid(index + 5);
						d_sphere_form = text_section.toDouble();
						
						text.truncate(index);
						
						index = text.indexOf(QStringLiteral("Diam:"),Qt::CaseInsensitive);
						
						if(index < 0)
						{
							d_last_error = QStringLiteral("Feature diameter not found.");
							return false;
						}
						
						text_section = text.mid(index + 5);
						d_sphere_diameter = text_section.toDouble();
						
						text.truncate(index);
						
						index = text.indexOf(QStringLiteral("XYZ:"),Qt::CaseInsensitive);
						
						if(index < 0)
						{
							d_last_error = QStringLiteral("Feature position not found.");
							return false;
						}
						
						text_section = text.mid(index + 4);
						list = text_section.split(',');
						
						if(list.size() == 3)
						{
							d_sphere_position.x = list[0].toDouble();
							d_sphere_position.y = list[1].toDouble();
							d_sphere_position.z = list[2].toDouble();
						}
						else
						{
							d_last_error = QStringLiteral("Feature position not recognized.");
							return false;
						}
						
						text.truncate(index);
						
						d_sphere_id = text.trimmed();
					}
				}
				else if(inside_alignment == true &&
						inside_data == false)
				{
					if(line.startsWith(QStringLiteral("X_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'X_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.X(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("X Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Y_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'Y_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Y(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Y Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Z_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'Z_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Z(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Z Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Translation:"),Qt::CaseInsensitive))
					{
						text = line.mid(12);	// remove 'Translation:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Origin(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Alignment translation data not recognized.");
							return false;
						}
					}
				}
				else if(inside_alignment == false &&
						inside_data == true)
				{
					list = line.split(',');
					
					if(list.size() == 8)
					{
						point_data.xyz.x = list[0].toDouble();
						point_data.xyz.y = list[1].toDouble();
						point_data.xyz.z = list[2].toDouble();
						
						point_data.ijk.x = list[3].toDouble();
						point_data.ijk.y = list[4].toDouble();
						point_data.ijk.z = list[5].toDouble();
						
						point_data.radius = list[6].toDouble();
						point_data.deviation = list[7].toDouble();
						
						d_point_data.push_back(point_data);
					}
				}
			}
		}
	}
	
	if(found_alignment == false ||
	   found_probe_offset == false ||
	   found_probe_vector == false ||
	   found_results == false ||
	   found_sphere_nominal == false ||
	   found_data == false)
	{
		d_last_error = QStringLiteral("Data missing from measurement file.");
		return false;
	}
	
	return true;
}
