/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QStringList>
#include <QByteArray>
#include <QTextStream>

#include "../../../core/vector2.h"

#include "measuredata_rpt.h"

TMeasureRPT::TMeasureRPT(void)
{
	d_file_type = TMeasureData::TYPE_SPHERE_RPT;
}

TMeasureRPT::~TMeasureRPT(void)
{
}

QStringList TMeasureRPT::Text_Report(void) const
{
	std::vector<TVector3>::const_iterator iter;
	QStringList							list;
	QString								text;
	TVector3							vec;
	TVector3							vmax,vmin;
	bool								init;
	
	list.push_back(QStringLiteral("ASME B89.4.10360 Rpt"));
	list.push_back(QStringLiteral("================================================================"));
	list.push_back(QString());
	
	list.push_back(QStringLiteral("Alignment"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	vec = d_alignment_data.X();
	text = QString("  X Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Y();
	text = QString("  Y Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Z();
	text = QString("  Z Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Origin();
	text = QString("  Origin: %1, %2, %3").arg(vec.x,12,'f',4).arg(vec.y,12,'f',4).arg(vec.z,12,'f',4);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Tool"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  Offset: %1, %2, %3").arg(tool_offset.x,12,'f',4).arg(tool_offset.y,12,'f',4).arg(tool_offset.z,12,'f',4);
	list.push_back(text);
	
	text = QString("  Vector: %1, %2, %3").arg(d_probe_data_vector.x,12,'f',9).arg(d_probe_data_vector.y,12,'f',9).arg(d_probe_data_vector.z,12,'f',9);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Sphere Center"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	list.push_back(QStringLiteral("        X             Y             Z"));
	list.push_back(QString());
	
	init = true;
	for(iter = d_points.begin();iter != d_points.end();++iter)
	{
		vec = (*iter);
		
		text = QString("%1  %2  %3").arg(vec.x,12,'f',4).arg(vec.y,12,'f',4).arg(vec.z,12,'f',4);
		list.push_back(text);
		
		if(init)
		{
			vmin = vec;
			vmax = vec;
			
			init = false;
		}
		else
		{
			if(vec.x < vmin.x) vmin.x = vec.x;
			if(vec.y < vmin.y) vmin.y = vec.y;
			if(vec.z < vmin.z) vmin.z = vec.z;
			
			if(vec.x > vmax.x) vmax.x = vec.x;
			if(vec.y > vmax.y) vmax.y = vec.y;
			if(vec.z > vmax.z) vmax.z = vec.z;
		}
	}

	list.push_back(QString());
	list.push_back(QStringLiteral("Range:"));
	
	text = QString("%1  %2  %3 mm").arg(vmax.x-vmin.x,12,'f',4).arg(vmax.y-vmin.y,12,'f',4).arg(vmax.z-vmin.z,12,'f',4);
	list.push_back(text);


	return list;
}

std::vector<std::vector<TVector2> > TMeasureRPT::Graph_Data(void) const
{
	std::vector<std::vector<TVector2> > graph_data;
	
	return graph_data;
}

bool TMeasureRPT::Load_Data(
	const QByteArray					&file_data)
{
	QTextStream							stream(file_data,QIODevice::ReadOnly);
	std::vector<TVector3>::iterator		iter;
	QString								line;
	QString								text;
	QString								text_section;
	QStringList							list;
	TVector3							pnt;
	bool								found_alignment(false);
	bool								inside_alignment(false);
	bool								found_probe_offset(false);
	bool								found_probe_vector(false);
	bool								found_data(false);
	bool								inside_data(false);
	bool								init(true);
	
	d_points.clear();
	
	while(!stream.atEnd())
	{
		line = stream.readLine().trimmed();
		
		if(line.startsWith('#') ||
		   line.length() == 0)
		{
			continue;
		}
		
		if(init)
		{
			if(!line.startsWith(QStringLiteral("B89.4.10360_Raw_Measurement:Version=2:Type=Sphere_Repeat")))
			{
				d_last_error = QStringLiteral("Artifact identification line not recognized");
				return false;
			}
			
			init = false;
		}
		else
		{
			if(line.compare(QStringLiteral("Alignment_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_alignment = true;
				inside_alignment = true;
			}
			else if(line.compare(QStringLiteral("Alignment_End:"),Qt::CaseInsensitive) == 0)
			{
				inside_alignment = false;
			}
			else if(line.startsWith(QStringLiteral("Probe_Offset:"),Qt::CaseInsensitive))
			{
				found_probe_offset = true;
				
				text = line.mid(13);	// remove 'Probe_Offset:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					tool_offset.x = list[0].toDouble();
					tool_offset.y = list[1].toDouble();
					tool_offset.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Probe offset data not recognized.");
					return false;
				}
				
			}
			else if(line.startsWith(QStringLiteral("Probe_Vector:"),Qt::CaseInsensitive))
			{
				found_probe_vector = true;
				
				text = line.mid(13);	// remove 'Probe_Vector:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					d_probe_data_vector.x = list[0].toDouble();
					d_probe_data_vector.y = list[1].toDouble();
					d_probe_data_vector.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Probe vector data not recognized.");
					return false;
				}
			}
			else if(line.compare(QStringLiteral("Measurement_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_data = true;
				inside_data = true;
			}
			else if(line.compare(QStringLiteral("Measurement_End::"),Qt::CaseInsensitive) == 0)
			{
				inside_data = false;
			}
			else
			{
				if(inside_alignment == true &&
				   inside_data == false)
				{
					if(line.startsWith(QStringLiteral("X_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'X_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.X(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("X Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Y_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'Y_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Y(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Y Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Z_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'Z_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Z(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Z Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Translation:"),Qt::CaseInsensitive))
					{
						text = line.mid(12);	// remove 'Translation:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Origin(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Alignment translation data not recognized.");
							return false;
						}
					}
				}
				else if(inside_alignment == false &&
						inside_data == true)
				{
					list = line.split(',');
					
					if(list.size() == 4)
					{
						pnt.Set(list[1].toDouble(),list[2].toDouble(),list[3].toDouble());
						d_points.push_back(pnt);
					}
				}
			}
		}
	}
	
	if(found_alignment == false ||
	   found_probe_offset == false ||
	   found_probe_vector == false ||
	   found_data == false)
	{
		d_last_error = QStringLiteral("Data missing from measurement file.");
		return false;
	}
	
	if(d_points.size())
	{
		pnt = d_points[0];
		
		for(iter = d_points.begin();iter != d_points.end();++iter)
		{
			(*iter) -= pnt;
		}
	}
	
	return true;
}
