/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QStringList>
#include <QByteArray>
#include <QTextStream>
#include <cmath>
#include <assert.h>

#include "../../../core/vector2.h"

#include "measuredata_stepgaugeb89.h"

TMeasureStepgaugeB89::TMeasureStepgaugeB89(void)
{
	d_file_type = TMeasureData::TYPE_STEPGAUGE_B89;
}

TMeasureStepgaugeB89::~TMeasureStepgaugeB89(void)
{
}

QStringList TMeasureStepgaugeB89::Text_Report(void) const
{
	std::vector<TMeasureStepgaugeB89::TAnalysisData>::const_iterator iter;
//	std::vector<TMeasureStepgaugeB89::TPointData>::const_iterator iter;
	QStringList							list;
	QString								text;
	double								deviation;
	double								range;
	TVector3							vec;
		
	list.push_back(QStringLiteral("ASME B89.4.1 Stepgauge Measurement"));
	list.push_back(QStringLiteral("================================================================"));
	list.push_back(QString());
	
	list.push_back(QStringLiteral("Description"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  Position:  %1").arg(d_measurement_description);
	list.push_back(text);
	
	text = QString("  Equipment: %1").arg(d_equipment_identification);
	list.push_back(text);
	
	text = QString("  Ref. Step: %1 mm").arg(d_reference_step_position,0,'f',3);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Alignment"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	vec = d_alignment_data.X();
	text = QString("  X Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Y();
	text = QString("  Y Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Z();
	text = QString("  Z Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Origin();
	text = QString("  Origin: %1, %2, %3").arg(vec.x,12,'f',4).arg(vec.y,12,'f',4).arg(vec.z,12,'f',4);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Tool"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  Offset: %1, %2, %3").arg(tool_offset.x,12,'f',4).arg(tool_offset.y,12,'f',4).arg(tool_offset.z,12,'f',4);
	list.push_back(text);
	
	text = QString("  Vector: %1, %2, %3").arg(d_probe_data_vector.x,12,'f',9).arg(d_probe_data_vector.y,12,'f',9).arg(d_probe_data_vector.z,12,'f',9);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Scale Expansion Coefficients"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  X Axis: %1 um/m/˚C").arg(d_scale_expansion_coefficient.x,0,'f',1);
	list.push_back(text);
	
	text = QString("  Y Axis: %1 um/m/˚C").arg(d_scale_expansion_coefficient.y,0,'f',1);
	list.push_back(text);
	
	text = QString("  Z Axis: %1 um/m/˚C").arg(d_scale_expansion_coefficient.z,0,'f',1);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Axis and Part Temperature"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  X Axis: %1 ˚C").arg(d_temperature_x,0,'f',2);
	list.push_back(text);
	
	text = QString("  Y Axis: %1 ˚C").arg(d_temperature_y,0,'f',2);
	list.push_back(text);
	
	text = QString("  Z Axis: %1 ˚C").arg(d_temperature_z,0,'f',2);
	list.push_back(text);
	
	text = QString("  Part:   %1 ˚C").arg(d_temperature_gauge,0,'f',2);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Measurement Points mm"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	//                             012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789
	list.push_back(QStringLiteral("     Nominal      Actual     Deviation     Min         Max        Range      Std. Dev."));
	list.push_back(QStringLiteral("-----------------------------------------------------------------------------------------"));
	list.push_back(QString());

	for(iter = d_analysis_data.begin();iter != d_analysis_data.end();++iter)
	{
		deviation = (*iter).mean - (*iter).nominal;
		range = (*iter).max - (*iter).min;
		
		text = QString("%1").arg((*iter).nominal,12,'f',4);
		text += QString("%1").arg((*iter).mean,12,'f',4);
		text += QString("%1").arg(deviation,12,'f',4);
		text += QString("%1").arg((*iter).min,12,'f',4);
		text += QString("%1").arg((*iter).max,12,'f',4);
		text += QString("%1").arg(range,12,'f',4);
		text += QString("%1").arg((*iter).std_dev,12,'f',4);
		list.push_back(text);
	}

	return list;
}

std::vector<std::vector<TVector2> > TMeasureStepgaugeB89::Graph_Data(void) const
{
	std::vector<std::vector<TVector2> > graph_data;
	std::vector<TVector2>				graph_item_points;
	std::vector<TPointData>::const_iterator iter;
	TVector2							pnt;
	double								deviation;
	bool								init;
	double								last_position;
	
	init = true;
	for(iter = d_point_data.begin();iter != d_point_data.end();++iter)
	{
		if(init)
		{
			last_position = (*iter).actual;
			init = false;
		}
		
		deviation = (*iter).actual - (*iter).nominal;
		pnt.Set((*iter).actual,deviation);

		if((*iter).actual < last_position)
		{
			// new graph dataset
			graph_data.push_back(graph_item_points);
			graph_item_points.clear();
		}
		
		graph_item_points.push_back(pnt);
		
		last_position = (*iter).actual;
	}
	
	if(graph_item_points.size())
	{
		graph_data.push_back(graph_item_points);
		graph_item_points.clear();
	}

	return graph_data;
}

bool TMeasureStepgaugeB89::Load_Data(
	const QByteArray					&file_data)
{
	QTextStream							stream(file_data,QIODevice::ReadOnly);
	TMeasureStepgaugeB89::TPointData	point_data;
	QString								line;
	QString								text;
	QString								text_section;
	QStringList							list;
	bool								found_id(false);
	bool								found_description(false);
	bool								found_scale(false);
	bool								inside_scale(false);
	bool								found_temperature(false);
	bool								inside_temperature(false);
	bool								found_alignment(false);
	bool								inside_alignment(false);
	bool								found_probe_offset(false);
	bool								found_probe_vector(false);
	bool								found_reference_position(false);
	bool								found_data(false);
	bool								inside_data(false);
	bool								init(true);
	
	d_point_data.clear();
	
	while(!stream.atEnd())
	{
		line = stream.readLine().trimmed();
		
		if(line.startsWith('#') ||
		   line.length() == 0)
		{
			continue;
		}
		
		if(init)
		{
			if(!line.startsWith(QStringLiteral("B89.4.1_Raw_Measurement:Version=1:Type=Step_Gauge")))
			{
				d_last_error = QStringLiteral("Stepgauge identification line not recognized");
				return false;
			}
			
			init = false;
		}
		else
		{
			if(line.compare(QStringLiteral("Scales_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_scale = true;
				inside_scale = true;
			}
			else if(line.compare(QStringLiteral("Scales_End:"),Qt::CaseInsensitive) == 0)
			{
				inside_scale = false;
			}
			else if(line.startsWith(QStringLiteral("Equipment_Identification:"),Qt::CaseInsensitive))
			{
				found_id = true;
				
				d_equipment_identification = line.mid(25);	// remove 'Equipment_Identification:'
			}
			else if(line.compare(QStringLiteral("Alignment_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_alignment = true;
				inside_alignment = true;
			}
			else if(line.compare(QStringLiteral("Alignment_End:"),Qt::CaseInsensitive) == 0)
			{
				inside_alignment = false;
			}
			else if(line.compare(QStringLiteral("Temperature_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_temperature = true;
				inside_temperature = true;
			}
			else if(line.compare(QStringLiteral("Temperature_End:"),Qt::CaseInsensitive) == 0)
			{
				inside_temperature = false;
			}
			else if(line.compare(QStringLiteral("Measurement_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_data = true;
				inside_data = true;
			}
			else if(line.compare(QStringLiteral("Measurement_End::"),Qt::CaseInsensitive) == 0)
			{
				inside_data = false;
			}
			else if(line.startsWith(QStringLiteral("Probe_Offset:"),Qt::CaseInsensitive))
			{
				found_probe_offset = true;
				
				text = line.mid(13);	// remove 'Probe_Offset:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					tool_offset.x = list[0].toDouble();
					tool_offset.y = list[1].toDouble();
					tool_offset.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Probe offset data not recognized.");
					return false;
				}
				
			}
			else if(line.startsWith(QStringLiteral("Probe_Vector:"),Qt::CaseInsensitive))
			{
				found_probe_vector = true;
				
				text = line.mid(13);	// remove 'Probe_Vector:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					d_probe_data_vector.x = list[0].toDouble();
					d_probe_data_vector.y = list[1].toDouble();
					d_probe_data_vector.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Probe vector data not recognized.");
					return false;
				}
			}
			else
			{
				if(inside_alignment == true &&
				   inside_data == false &&
				   inside_temperature == false &&
				   inside_scale == false)
				{
					if(line.startsWith(QStringLiteral("X_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'X_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.X(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("X Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Y_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'Y_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Y(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Y Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Z_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'Z_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Z(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Z Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Translation:"),Qt::CaseInsensitive))
					{
						text = line.mid(12);	// remove 'Translation:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Origin(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Alignment translation data not recognized.");
							return false;
						}
					}
				}
				else if(inside_alignment == false &&
						inside_data == false &&
						inside_temperature == true &&
						inside_scale == false)
				{
					if(line.startsWith(QStringLiteral("X_Scale:"),Qt::CaseInsensitive))
					{
						text = line.mid(8);	// remove 'X_Scale:'
						d_temperature_x = text.toDouble();
					}
					else if(line.startsWith(QStringLiteral("Y_Scale:"),Qt::CaseInsensitive))
					{
						text = line.mid(8);	// remove 'Y_Scale:'
						d_temperature_y = text.toDouble();
					}
					else if(line.startsWith(QStringLiteral("Z_Scale:"),Qt::CaseInsensitive))
					{
						text = line.mid(8);	// remove 'Z_Scale:'
						d_temperature_z = text.toDouble();
					}
					else if(line.startsWith(QStringLiteral("Gauge:"),Qt::CaseInsensitive))
					{
						text = line.mid(6);	// remove 'Gauge:'
						d_temperature_gauge = text.toDouble();
					}
				}
				else if(inside_alignment == false &&
						inside_data == false &&
						inside_temperature == false &&
						inside_scale == true)
				{
					if(line.startsWith(QStringLiteral("X_Expansion_Coefficient:"),Qt::CaseInsensitive))
					{
						text = line.mid(24);	// remove 'X_Expansion_Coefficient:'
						d_scale_expansion_coefficient.x = text.toDouble();
					}
					else if(line.startsWith(QStringLiteral("Y_Expansion_Coefficient:"),Qt::CaseInsensitive))
					{
						text = line.mid(24);	// remove 'Y_Expansion_Coefficient:'
						d_scale_expansion_coefficient.y = text.toDouble();
					}
					else if(line.startsWith(QStringLiteral("Z_Expansion_Coefficient:"),Qt::CaseInsensitive))
					{
						text = line.mid(24);	// remove 'Z_Expansion_Coefficient:'
						d_scale_expansion_coefficient.z = text.toDouble();
					}
				}
				else if(inside_alignment == false &&
						inside_data == true &&
						inside_temperature == false &&
						inside_scale == false)
				{
					if(line.startsWith(QStringLiteral("Description:"),Qt::CaseInsensitive))
					{
						found_description = true;
						d_measurement_description = line.mid(12);	// remove 'Description:'
					}
					else if(line.startsWith(QStringLiteral("Reference_Step_Position:"),Qt::CaseInsensitive))
					{
						found_reference_position = true;
						text = line.mid(24);	// remove 'Reference_Step_Position:'
						
						d_reference_step_position = text.toDouble();
					}
					else if(line.startsWith(QStringLiteral("Point:"),Qt::CaseInsensitive))
					{
						text = line.mid(6);	// remove 'Point:'
						
						list = text.split(',');
						
						if(list.size() == 4)
						{
							point_data.actual = list[1].toDouble();
							point_data.nominal = point_data.actual;
							
							d_point_data.push_back(point_data);
						}
					}
				}
			}
		}
	}
	
	if(found_alignment == false ||
	   found_id == false ||
	   found_description == false ||
	   found_scale == false ||
	   found_temperature == false ||
	   found_probe_offset == false ||
	   found_probe_vector == false ||
	   found_reference_position == false ||
	   found_data == false)
	{
		d_last_error = QStringLiteral("Data missing from measurement file.");
		return false;
	}
	
	d_analysis_data.clear();
	
	return true;
}

bool TMeasureStepgaugeB89::Set_Nominal_Data(
	const std::vector<TVector3>			&data)
{
	std::vector<TPointData>::iterator	iter;
	double								dval;
	double								nominal;
	bool								valid;
	
	for(iter = d_point_data.begin();iter != d_point_data.end();++iter)
	{
		dval = (*iter).actual;
		
		nominal = this->Get_Nominal(dval,data,&valid);
		
		if(valid)
		{
			(*iter).nominal = nominal - d_reference_step_position;
		}
		else
		{
			d_last_error = QStringLiteral("Nominal data not found.  Possible wrong reference step position or measurement error exceeds maximum allowed.");
			
			return false;
		}
	}
	
	this->Create_Analysis_Data();

	return true;
}

double TMeasureStepgaugeB89::Get_Nominal(
	const double						&position,
	const std::vector<TVector3>			&data,
	bool								* const valid) const
{
	std::vector<TVector3>::const_iterator iter;
	double								nominal(0.0);
	double								deviation;
	static const double					MAXIMUM_DEVIATION(2.0);	// nominal must be within +/- of this value.
	
	assert(valid);
	*valid = false;
	
	for(iter = data.begin();iter != data.end();++iter)
	{
		deviation = fabs((*iter).x - position);
		
		if(!(deviation > MAXIMUM_DEVIATION))
		{
			*valid = true;
			
			nominal = (*iter).x;
			break;
		}
	}
	
	return nominal;
}

void TMeasureStepgaugeB89::Create_Analysis_Data(void)
{
	std::vector<TMeasureStepgaugeB89::TPointData>::const_iterator point_iter;
	std::vector<TMeasureStepgaugeB89::TAnalysisData>::iterator analysis_iter;
	TMeasureStepgaugeB89::TAnalysisData analysis_data;
	double								min,max;
	double								dval;
	bool								found_point;
	static const double					SEARCH_RADIUS(1.0);
	
	for(point_iter = d_point_data.begin();point_iter != d_point_data.end();++point_iter)
	{
		found_point = false;
		for(analysis_iter = d_analysis_data.begin();analysis_iter != d_analysis_data.end();++analysis_iter)
		{
			min = (*analysis_iter).nominal - SEARCH_RADIUS;
			max = (*analysis_iter).nominal + SEARCH_RADIUS;
			dval = (*point_iter).nominal;
			
			if((dval > min) && (dval < max))
			{
				found_point = true;
				(*analysis_iter).actual.push_back((*point_iter).actual);
			}
		}
		
		if(!found_point)
		{
			analysis_data.nominal = (*point_iter).nominal;
			analysis_data.actual.clear();
			analysis_data.actual.push_back((*point_iter).actual);
			
			d_analysis_data.push_back(analysis_data);
		}
	}
	
	for(analysis_iter = d_analysis_data.begin();analysis_iter != d_analysis_data.end();++analysis_iter)
	{
		this->Calculate_Stats((*analysis_iter).actual,&(*analysis_iter).min,&(*analysis_iter).max,&(*analysis_iter).mean,&(*analysis_iter).std_dev);
	}
}

bool TMeasureStepgaugeB89::Calculate_Stats(
	const std::vector<double> 			&data,
	double 								* const min,
	double 								* const max, 
	double 								* const average,
	double 								* const std_dev) const
{
	std::vector<double>::const_iterator iter;
	double								dval;
	double								sum;
	bool								init(true);
	
	assert(min);
	assert(max);
	assert(average);
	assert(std_dev);
	
	*min = 0.0;
	*max = 0.0;
	*average = 0.0;
	*std_dev = 0.0;
	
	if(!data.size())
	{
		return false;
	}
	
	for(iter = data.begin();iter != data.end();++iter)
	{		
		if(init)
		{
			*min = (*iter);
			*max = (*iter);
			
			init = false;
		}
		else
		{
			if((*iter) > (*max)) (*max) = (*iter);
			if((*iter) < (*min)) (*min) = (*iter);
		}
				
		*average += (*iter);
	}
	
	*average /= static_cast<double>(data.size());
	
	sum = 0.0;
	for(iter = data.begin();iter != data.end();++iter)
	{
		dval = (*iter) - (*average);
		sum += (dval * dval);
	}
	
	if(data.size() > 1)
	{
		*std_dev = sqrt(sum / static_cast<double>(data.size() - 1));
	}
	
	return true;
}

