/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QStringList>
#include <QByteArray>
#include <QTextStream>

#include "../../../core/vector2.h"

#include "measuredata_tool.h"

TMeasureTool::TMeasureTool(void)
{
	d_file_type = TMeasureData::TYPE_PROBE_CALIBRATION;
	d_version = 0;
}

TMeasureTool::~TMeasureTool(void)
{
}

QStringList TMeasureTool::Text_Report(void) const
{
	std::vector<TMeasureTool::TToolData>::const_iterator iter;
	QStringList							list;
	QString								text;
	TVector3							vec;
	
	list.push_back(QStringLiteral("Tool Calibration Results"));
	list.push_back(QStringLiteral("================================================================"));
	list.push_back(QString());
	
	list.push_back(QStringLiteral("Alignment"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	vec = d_alignment_data.X();
	text = QString("  X Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Y();
	text = QString("  Y Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Z();
	text = QString("  Z Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Origin();
	text = QString("  Origin: %1, %2, %3").arg(vec.x,12,'f',4).arg(vec.y,12,'f',4).arg(vec.z,12,'f',4);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Sphere"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  Diameter:     %1 mm").arg(d_nominal_sphere_diameter,0,'f',4);
	list.push_back(text);
	list.push_back(QString());

	if(d_version == 1)
	{
		list.append(d_report_text);
	}
	else
	{
									// 0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789
		list.push_back(QStringLiteral("Name                            Offset X  Offset Y  Offset Z  Vector I  Vector J  Vector K  Tip Diameter"));
		list.push_back(QStringLiteral("-------------------------------------------------------------------------------------------------------------"));
		list.push_back(QString());

		for(iter = d_tools.begin();iter != d_tools.end();++iter)
		{
			text = QString("%1").arg((*iter).name,-30);
			text += QString("%1").arg((*iter).offset.x,10,'f',4);
			text += QString("%1").arg((*iter).offset.y,10,'f',4);
			text += QString("%1").arg((*iter).offset.z,10,'f',4);
			text += QString("%1").arg((*iter).vector.i,10,'f',6);
			text += QString("%1").arg((*iter).vector.j,10,'f',6);
			text += QString("%1").arg((*iter).vector.k,10,'f',6);
			text += QString("%1").arg((*iter).tip_diameter,10,'f',4);
			list.push_back(text);
		}
	}
	
	return list;
}

std::vector<std::vector<TVector2> > TMeasureTool::Graph_Data(void) const
{
	std::vector<std::vector<TVector2> > graph_data;
	
	return graph_data;
}

bool TMeasureTool::Load_Data(
	const QByteArray					&file_data)
{
	QTextStream							stream(file_data,QIODevice::ReadOnly);
	QString								line;
	QString								text;
	QString								text_section;
	QStringList							list;
	TMeasureTool::TToolData				tool_data;
	bool								found_alignment(false);
	bool								inside_alignment(false);
	bool								found_sphere_nominal(false);
	bool								init(true);
	
	d_report_text.clear();
	d_tools.clear();
	
	while(!stream.atEnd())
	{
		line = stream.readLine().trimmed();
		
		if(line.startsWith('#'))
		{
			continue;
		}
		
		if(init)
		{
			if(line.startsWith(QStringLiteral("Artifact_Measurement:Version=1:Type=Tip_Calibration")))
			{
				d_version = 1;
			}
			else if(line.startsWith(QStringLiteral("Artifact_Measurement:Version=2:Type=Tip_Calibration")))
			{
				d_version = 2;
			}
			else
			{
				d_last_error = QStringLiteral("Artifact identification line not recognized");
				return false;
			}
			
			init = false;
		}
		else
		{
			if(line.compare(QStringLiteral("Alignment_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_alignment = true;
				inside_alignment = true;
			}
			else if(line.compare(QStringLiteral("Alignment_End:"),Qt::CaseInsensitive) == 0)
			{
				inside_alignment = false;
			}
			else if(line.startsWith(QStringLiteral("Nominal_Sphere_Diameter:"),Qt::CaseInsensitive))
			{
				found_sphere_nominal = true;
				
				text = line.mid(24);	// remove 'Nominal_Sphere_Diameter:'
				d_nominal_sphere_diameter = text.toDouble();
				
			}
			else
			{
				if(inside_alignment)
				{
					if(line.startsWith(QStringLiteral("X_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'X_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.X(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("X Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Y_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'Y_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Y(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Y Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Z_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'Z_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Z(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Z Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Translation:"),Qt::CaseInsensitive))
					{
						text = line.mid(12);	// remove 'Translation:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Origin(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Alignment translation data not recognized.");
							return false;
						}
					}
				}
				else if(found_sphere_nominal)
				{
					if(d_version == 2)
					{
						if(line.startsWith(QStringLiteral("Name:"),Qt::CaseInsensitive))
						{
							text = line.mid(5);	// remove 'Name:'
							
							if(text.length())
							{
								tool_data.name = text;
							}
							else
							{
								d_last_error = QStringLiteral("Name data not recognized.");
								return false;
							}
						}
						else if(line.startsWith(QStringLiteral("Probe_Offset:"),Qt::CaseInsensitive))
						{
							text = line.mid(13);	// remove 'Probe_Offset:'
							list = text.split(',');
							
							if(list.size() == 3)
							{
								tool_data.offset.Set(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
							}
							else
							{
								d_last_error = QStringLiteral("Probe_Offset data not recognized.");
								return false;
							}
						}
						else if(line.startsWith(QStringLiteral("Probe_Vector:"),Qt::CaseInsensitive))
						{
							text = line.mid(13);	// remove 'Probe_Vector:'
							list = text.split(',');
							
							if(list.size() == 3)
							{
								tool_data.vector.Set(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
							}
							else
							{
								d_last_error = QStringLiteral("Probe_Vector data not recognized.");
								return false;
							}
						}
						else if(line.startsWith(QStringLiteral("Tip_Diameter:"),Qt::CaseInsensitive))
						{
							text = line.mid(13);	// remove 'Tip_Diameter:'
							
							if(text.length())
							{
								tool_data.tip_diameter = text.toDouble();
							}
							else
							{
								d_last_error = QStringLiteral("Tip_Diameter data not recognized.");
								return false;
							}
							
							d_tools.push_back(tool_data);
						}
					}
					else if(d_version == 1)
					{
						d_report_text.push_back(line);
					}
				}
			}
		}
	}
	
	if(found_alignment == false ||
	   found_sphere_nominal == false)
	{
		d_last_error = QStringLiteral("Data missing from measurement file.");
		return false;
	}
	
	return true;
}
