/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////



#include <QApplication>
#include <QHBoxLayout>
#include <QListWidget>
#include <QListWidgetItem>
#include <QSpacerItem>
#include <QToolButton>
#include <QVBoxLayout>
#include <QSize>
#include <QDir>
#include <QStringList>
#include <QList>

#include "measuredinputdialog.h"

#include "measuredwidget.h"

TMeasuredWidget::TMeasuredWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QHBoxLayout							*button_hlayout;
	QSpacerItem							*button_hspacer;
	QVBoxLayout							*verticalLayout;

	verticalLayout = new QVBoxLayout(this);

	d_measured_list = new QListWidget(this);
	d_measured_list->setSelectionMode(QAbstractItemView::ExtendedSelection);
	verticalLayout->addWidget(d_measured_list);

	button_hlayout = new QHBoxLayout();

	d_remove_button = new QToolButton(this);
	d_remove_button->setMinimumSize(16,16);
	d_remove_button->setMaximumSize(16,16);
	d_remove_button->setIconSize(QSize(16,16));
	button_hlayout->addWidget(d_remove_button);
	
	d_edit_button = new QToolButton(this);
	d_edit_button->setMinimumSize(16,16);
	d_edit_button->setMaximumSize(16,16);
	d_edit_button->setIconSize(QSize(16,16));
	button_hlayout->addWidget(d_edit_button);
	
	d_refresh_button = new QToolButton(this);
	d_refresh_button->setMinimumSize(16,16);
	d_refresh_button->setMaximumSize(16,16);
	d_refresh_button->setIconSize(QSize(16,16));
	button_hlayout->addWidget(d_refresh_button);

	button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_hlayout->addItem(button_hspacer);
	verticalLayout->addLayout(button_hlayout);
	
	d_application_path = qApp->applicationDirPath();
	
#ifdef Q_OS_MAC
	int index = d_application_path.lastIndexOf(".app/");
	d_application_path.truncate(index);
	
	index = d_application_path.lastIndexOf('/');
	d_application_path.truncate(index);
#endif
	
	d_input_settings.input_relative_path = QStringLiteral("/Measurement_Data");
	d_input_settings.input_absolute_path = d_data_path + QStringLiteral("/Measurement_Data");
	d_input_settings.use_relative_path = false;
	
	// defaults
	d_measured_input_dialog = 0;
	d_remove_button->setEnabled(false);

	this->setWindowTitle(QStringLiteral("Measured"));
	
	d_remove_button->setToolTip(QStringLiteral("Remove Measurement"));
	d_edit_button->setToolTip(QStringLiteral("Edit Search Path"));
	d_refresh_button->setToolTip(QStringLiteral("Refresh"));
	
	d_remove_button->setIcon(QIcon(":/icon16/remove_icon16.png"));
	d_edit_button->setIcon(QIcon(":/icon16/edit_icon16.png"));
	d_refresh_button->setIcon(QIcon(":/icon16/refresh_icon16.png"));
	
	connect(d_remove_button,&QToolButton::clicked,this,&TMeasuredWidget::Remove_Clicked);
	connect(d_edit_button,&QToolButton::clicked,this,&TMeasuredWidget::Edit_Clicked);
	connect(d_refresh_button,&QToolButton::clicked,this,&TMeasuredWidget::Refresh_Clicked);
	connect(d_measured_list,&QListWidget::itemSelectionChanged,this,&TMeasuredWidget::Selection_Changed);
}

TMeasuredWidget::~TMeasuredWidget(void)
{
}

TMeasuredWidget::TInputSettings TMeasuredWidget::Input_Settings(void) const
{
	return d_input_settings;
}

void TMeasuredWidget::Set_Input_Settings(
	const TMeasuredWidget::TInputSettings &settings)
{
	d_input_settings = settings;
	
	this->Update_Display();
}

void TMeasuredWidget::Refresh_File_List(void)
{
	this->Update_Display();
}

void TMeasuredWidget::Remove_Clicked(void)
{
	QList<QListWidgetItem*>				sel_items;
	QList<QListWidgetItem*>::iterator	iter;
	QStringList							file_list;

	sel_items = d_measured_list->selectedItems();

	for(iter = sel_items.begin();iter != sel_items.end();++iter)
	{
		file_list.push_back((*iter)->text());
	}
	
	if(file_list.size())
	{
		emit Remove_Data(d_data_path,file_list);
	}
}

void TMeasuredWidget::Edit_Clicked(void)
{
	if(!d_measured_input_dialog)
	{
		d_measured_input_dialog = new TMeasuredInputDialog(this);
	}
	
	d_measured_input_dialog->Reset(d_application_path,d_input_settings);
	
	if(QDialog::Accepted == d_measured_input_dialog->exec())
	{
		d_input_settings = d_measured_input_dialog->Settings();
		
		this->Update_Display();
		
		emit File_Path_Changed();
	}
}
void TMeasuredWidget::Refresh_Clicked(void)
{
	this->Update_Display();
}

void TMeasuredWidget::Selection_Changed(void)
{
	QList<QListWidgetItem*>				sel_items;
	QString								text;
	QString								file_name;
	
	sel_items = d_measured_list->selectedItems();
	
	if(sel_items.size() == 1)
	{
		text = sel_items[0]->text();
		
		file_name = d_data_path;
		file_name.append('/');
		file_name.append(text);
		
		emit Load_Data(file_name);
	}
	
	d_remove_button->setEnabled(sel_items.size() > 0);
}

void TMeasuredWidget::Update_Display(void)
{
	QDir								dir(d_application_path);
	QListWidgetItem						*item;
	QStringList							list;
	QList<QString>::iterator			iter;
	
	if(d_input_settings.use_relative_path)
	{
		dir.cd(d_input_settings.input_relative_path);
	}
	else
	{
		dir.setPath(d_input_settings.input_absolute_path);
	}
	
	d_data_path = dir.absolutePath();
	
	d_measured_list->clear();
	
	list = dir.entryList(QDir::Files | QDir::NoDot | QDir::NoDotDot);
	
	for(iter = list.begin();iter != list.end();++iter)
	{
		if((*iter).endsWith(QStringLiteral(".dat"),Qt::CaseInsensitive))
		{
			item = new QListWidgetItem(d_measured_list);
			item->setText((*iter));
		}
	}
}


