/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QHBoxLayout>
#include <QListWidget>
#include <QSpacerItem>
#include <QToolButton>
#include <QVBoxLayout>
#include <QSize>
#include <QDir>
#include <assert.h>

#include "../../core/xmlfile.h"
#include "../../core/messagebox.h"

#include "nominalwidget.h"

TNominalWidget::TNominalWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QHBoxLayout							*button_hlayout;
	QSpacerItem							*button_hspacer;
	QVBoxLayout							*verticalLayout;

	verticalLayout = new QVBoxLayout(this);

	d_nominal_list = new QListWidget(this);
	verticalLayout->addWidget(d_nominal_list);

	button_hlayout = new QHBoxLayout();

	d_add_button = new QToolButton(this);
	d_add_button->setMinimumSize(16,16);
	d_add_button->setMaximumSize(16,16);
	d_add_button->setIconSize(QSize(16,16));
	button_hlayout->addWidget(d_add_button);

	d_remove_button = new QToolButton(this);
	d_remove_button->setMinimumSize(16,16);
	d_remove_button->setMaximumSize(16,16);
	d_remove_button->setIconSize(QSize(16,16));
	button_hlayout->addWidget(d_remove_button);

	d_edit_button = new QToolButton(this);
	d_edit_button->setMinimumSize(16,16);
	d_edit_button->setMaximumSize(16,16);
	d_edit_button->setIconSize(QSize(16,16));
	button_hlayout->addWidget(d_edit_button);

	button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_hlayout->addItem(button_hspacer);
	verticalLayout->addLayout(button_hlayout);
	
	d_message_box = new TMessageBox(this);
	
	// defaults
	d_nominal_editor_dialog = 0;
	d_remove_button->setEnabled(false);
	d_edit_button->setEnabled(false);

	this->setWindowTitle(QStringLiteral("Nominal"));
	
	d_add_button->setToolTip(QStringLiteral("Add Nominal"));
	d_remove_button->setToolTip(QStringLiteral("Remove Nominal"));
	d_edit_button->setToolTip(QStringLiteral("Edit Nominal"));

	d_add_button->setIcon(QIcon(":/icon16/add_icon16.png"));
	d_remove_button->setIcon(QIcon(":/icon16/remove_icon16.png"));
	d_edit_button->setIcon(QIcon(":/icon16/edit_icon16.png"));

	
	connect(d_add_button,&QToolButton::clicked,this,&TNominalWidget::Add_Nominal);
	connect(d_remove_button,&QToolButton::clicked,this,&TNominalWidget::Remove_Nominal);
	connect(d_edit_button,&QToolButton::clicked,this,&TNominalWidget::Edit_Nominal);
	connect(d_nominal_list,&QListWidget::itemSelectionChanged,this,&TNominalWidget::Selection_Changed);
}

TNominalWidget::~TNominalWidget(void)
{
}

std::vector<TVector3> TNominalWidget::Nominal_Data(
	const QString						&serial,
	bool								* const valid)
{
	QListWidgetItem						*item;
	QString								text;
	int									count;
	int									cntr;
	
	assert(valid);
	*valid = false;
	
	if(serial.length())
	{
		count = d_nominal_list->count();
		
		for(cntr = 0;cntr < count;++cntr)
		{
			item = d_nominal_list->item(cntr);
			text = item->text();
			
			if(text.endsWith(serial,Qt::CaseInsensitive))
			{
				if(this->Load_Data(text))
				{
					*valid = true;
					return d_active_data;
				}
				
				break;
			}
		}
	}
	
	return d_active_data;
}

void TNominalWidget::Initialize(
	const QString						&data_path)
{
	QDir								dir;
	
	d_data_path = data_path;
	
	dir.setPath(d_data_path);
	
	if(!dir.exists(QStringLiteral("Nominal_Data")))
	{
		dir.mkdir(QStringLiteral("Nominal_Data"));
	}
	
	dir.cd(QStringLiteral("Nominal_Data"));
	
	this->Update_Display();
}

void TNominalWidget::Add_Nominal(void)
{
	std::vector<TVector3>				empty;
	QString								name;

	if(!d_nominal_editor_dialog)
	{
		d_nominal_editor_dialog = new TNominalEditorDialog(this);
	}
	
	d_nominal_editor_dialog->Reset(QString(),empty,TNominalEditorDialog::TYPE_1D);

	if(QDialog::Accepted == d_nominal_editor_dialog->exec())
	{
		d_active_data = d_nominal_editor_dialog->Nominal_Data();
		d_active_serial_number = d_nominal_editor_dialog->Serial_Number();
		d_nominal_type = d_nominal_editor_dialog->Nominal_Type();
		
		name = QString("SG_%1").arg(d_nominal_editor_dialog->Serial_Number());
		
		if(!this->Save_Data(name))
		{
//			qWarning("cannot save the data");
			return;
		}
		
		this->Update_Display();
	}
}

void TNominalWidget::Remove_Nominal(void)
{
	QString								name;
	QList<QListWidgetItem*>				sel_items;
	QList<QListWidgetItem*>::iterator	iter;

	sel_items = d_nominal_list->selectedItems();
	
	if(sel_items.size() == 0)
	{
		return;
	}
	
	d_message_box->setText("Remove Nominal Files");
	d_message_box->setInformativeText(QStringLiteral("Do you want to remove selected nominal files ?"));
	d_message_box->setDetailedText(QString());
	d_message_box->setStandardButtons(QMessageBox::Yes | QMessageBox::No);
	d_message_box->setDefaultButton(QMessageBox::Yes);
	d_message_box->setIcon(QMessageBox::Warning);
	
	if(QMessageBox::Yes != d_message_box->exec())
	{
		return;
	}

	for(iter = sel_items.begin();iter != sel_items.end();++iter)
	{		
		this->Remove_Data((*iter)->text());
	}
	
	this->Update_Display();
}

void TNominalWidget::Edit_Nominal(void)
{
	QString								name;
	QList<QListWidgetItem*>				sel_items;
	
	sel_items = d_nominal_list->selectedItems();
	
	if(sel_items.size() != 1)
	{
		return;
	}
	
	name = sel_items[0]->text();
	
	if(!this->Load_Data(name))
	{
//		qWarning("cannot load data");
		return;
	}
	
	if(!d_nominal_editor_dialog)
	{
		d_nominal_editor_dialog = new TNominalEditorDialog(this);
	}
	
	d_nominal_editor_dialog->Reset(d_active_serial_number,d_active_data,d_nominal_type);
	
	if(QDialog::Accepted == d_nominal_editor_dialog->exec())
	{
		d_active_data = d_nominal_editor_dialog->Nominal_Data();
		d_active_serial_number = d_nominal_editor_dialog->Serial_Number();
		d_nominal_type = d_nominal_editor_dialog->Nominal_Type();
		
		name = QString("SG_%1").arg(d_active_serial_number);
		
		if(!this->Save_Data(name))
		{
//			qWarning("cannot save the data");
			return;
		}
		
		this->Update_Display();
	}
}

void TNominalWidget::Selection_Changed(void)
{
	QList<QListWidgetItem*>				sel_items;
	
	sel_items = d_nominal_list->selectedItems();

	d_remove_button->setEnabled(sel_items.size() > 0);
	d_edit_button->setEnabled(sel_items.size() > 0);
}

void TNominalWidget::Update_Display(void)
{
	QDir								dir;
	QListWidgetItem						*item;
	QStringList							list;
	QList<QString>::iterator			iter;
	
	dir.setPath(QString("%1/Nominal_Data").arg(d_data_path));
	d_nominal_list->clear();
	
	list = dir.entryList(QDir::Files | QDir::NoDot | QDir::NoDotDot);
	
	for(iter = list.begin();iter != list.end();++iter)
	{
		if((*iter).startsWith(QStringLiteral("SG_"),Qt::CaseInsensitive))
		{
			item = new QListWidgetItem(d_nominal_list);
			item->setText((*iter));
		}
	}
}

bool TNominalWidget::Load_Data(
	const QString						&name)
{
	QString								file_name;
	TXmlFile							xml_file;
	int									version;
	
	file_name = d_data_path;
	file_name.append(QStringLiteral("/Nominal_Data/"));
	file_name.append(name);
	
	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Open("Step_Gauge_Nominal",&version);
		
		switch(version)
		{
			case 1:
				this->Load_Data_V1(&xml_file);
				break;
			case 2:
				this->Load_Data_V2(&xml_file);
				break;

			default:
				throw TXmlFileException(QStringLiteral("ERR:  File version not recognized"));
				break;
		}
	}
	catch(TXmlFileException exception)
	{
		return false;
	}
	
	return true;
}

bool TNominalWidget::Save_Data(
	const QString						&name)
{
	QString								file_name;
	TXmlFile							xml_file;
	QDomElement							e0;
	QDomElement							e1;
	std::vector<TVector3>::const_iterator iter;
	int									cntr;
	
	file_name = d_data_path;
	file_name.append(QStringLiteral("/Nominal_Data/"));
	file_name.append(name);
	
	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Reset("Step_Gauge_Nominal",2);
		
		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Serial_Number"));
		xml_file.Write_Text_Node(&e0,d_active_serial_number);
		
		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Nominal_Type"));
		
		switch(d_nominal_type)
		{
			case TNominalEditorDialog::TYPE_1D:
				xml_file.Write_Text_Node(&e0,QStringLiteral("Type_1D"));
				break;
				
			case TNominalEditorDialog::TYPE_3D:
				xml_file.Write_Text_Node(&e0,QStringLiteral("Type_3D"));
				break;
		}
		
		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Actual"));

		for(iter = d_active_data.begin(),cntr = 0;iter != d_active_data.end();++iter,++cntr)
		{
			e1 = xml_file.Create_Node(&e0,QString("Index_%1").arg(cntr));
			xml_file.Write_Text_Node(&e1,QString("%1,%2,%3").arg((*iter).x,0,'f',5).arg((*iter).y,0,'f',5).arg((*iter).z,0,'f',5));
		}
		
		xml_file.Write_File();
	}
	catch(TXmlFileException exception)
	{
		return false;
	}
	
	return true;
}

void TNominalWidget::Remove_Data(
	const QString						&name)
{
	QString								file_name;
	QFile								file;
	
	file_name = d_data_path;
	file_name.append(QStringLiteral("/Nominal_Data/"));
	file_name.append(name);
	
	file.setFileName(file_name);
	file.remove();
}

void TNominalWidget::Load_Data_V1(
	TXmlFile							* const xml_file)
{
	QDomElement							element;
	QDomElement							e0;
	QDomElement							e1;
	TVector3							vec;

	QString								text;
	bool								valid;
	int									cntr;
	
	assert(xml_file);
	
	d_active_data.clear();

	d_nominal_type = TNominalEditorDialog::TYPE_1D;
	
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Serial_Number"));
	d_active_serial_number = xml_file->Read_Text_Node(&e0);

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Actual"));
	e1 = xml_file->Get_Node(&e0,QStringLiteral("Index_0"),true);
	
	vec.Set(0,0,0);
	cntr = 0;
	while(!e1.isNull())
	{
		text = xml_file->Read_Text_Node(&e1);
		vec.x = text.toDouble(&valid);
		
		if(valid)
		{
			d_active_data.push_back(vec);
		}
		
		++cntr;
		e1 = xml_file->Get_Sibling_Node(&e1,QString("Index_%1").arg(cntr),true);
	}
}

void TNominalWidget::Load_Data_V2(
	TXmlFile							* const xml_file)
{
	QDomElement							element;
	QDomElement							e0;
	QDomElement							e1;
	TVector3							vec;
	QString								text;
	QStringList							list;
	bool								valid;
	int									cntr;
	
	assert(xml_file);
	
	d_active_data.clear();

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Serial_Number"));
	d_active_serial_number = xml_file->Read_Text_Node(&e0);
	
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Nominal_Type"));
	text = xml_file->Read_Text_Node(&e0);
	
	if(text.compare(QStringLiteral("Type_3D")) == 0)
	{
		d_nominal_type = TNominalEditorDialog::TYPE_3D;
	}
	else
	{
		d_nominal_type = TNominalEditorDialog::TYPE_1D;
	}

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Actual"));
	e1 = xml_file->Get_Node(&e0,QStringLiteral("Index_0"),true);
	
	cntr = 0;
	
	while(!e1.isNull())
	{
		text = xml_file->Read_Text_Node(&e1);
		list = text.split(',');
		
		if(list.size() == 3)
		{
			vec.x = list[0].toDouble(&valid);
			
			if(valid)
			{
				vec.y = list[1].toDouble(&valid);
				
				if(valid)
				{
					vec.z = list[2].toDouble(&valid);
					
					if(valid)
					{
						d_active_data.push_back(vec);
					}
				}
			}
		}
		
		++cntr;
		e1 = xml_file->Get_Sibling_Node(&e1,QString("Index_%1").arg(cntr),true);
	}
}


