/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2025  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QAction>
#include <QApplication>
#include <QGridLayout>
#include <QLabel>
#include <QPlainTextEdit>
#include <QVBoxLayout>
#include <QFont>

#include "../../core/xmlfile.h"

#include "changelogwidget.h"

TChangeLogWidget::TChangeLogWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QLabel								*change_label;
	QVBoxLayout							*widget_vlayout;
	QFont								title_font;
	QFont								text_font;

	title_font.setPointSize(15);
	title_font.setWeight(50);
	title_font.setItalic(true);
	title_font.setBold(false);
	title_font.setUnderline(true);
	
	text_font.setFamily(QStringLiteral("courier new"));
	text_font.setFixedPitch(true);

	widget_vlayout = new QVBoxLayout(this);

	change_label = new QLabel(this);
	change_label->setFont(title_font);
	widget_vlayout->addWidget(change_label);

	d_changelog_edit = new QPlainTextEdit(this);
	d_changelog_edit->setFont(text_font);
	d_changelog_edit->setLineWrapMode(QPlainTextEdit::NoWrap);
	widget_vlayout->addWidget(d_changelog_edit);

	change_label->setText(QStringLiteral("Change Log:"));
}

TChangeLogWidget::~TChangeLogWidget(void)
{
}

bool TChangeLogWidget::Save_Defaults(
	const QString 						&file_path)
{
	QString								file_name;
	TXmlFile							xml_file;
	QDomElement							e0;
	QString								text;
	
	file_name = file_path + QStringLiteral("/changes.xml");

	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Reset(QStringLiteral("Change_Data"),1);

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Text"));
		text = d_changelog_edit->toPlainText();
		
		text.replace(QStringLiteral("\n"),QStringLiteral("<br>"));
		
		xml_file.Write_Text_Node(&e0,text);

		xml_file.Write_File();
	}
	catch(TXmlFileException xml_file_exception)
	{
		d_last_error = xml_file_exception.ErrorText();
		return false;
	}

	return true;
}

QString TChangeLogWidget::Changes(void) const
{
	return d_changelog_edit->toPlainText();
}

bool TChangeLogWidget::Load_Defaults(
	const QString 						&file_path)
{
	TXmlFile							xml_file;
	QString								file_name;
	int									version;
	bool								result(true);
	
	file_name = file_path + QStringLiteral("/changes.xml");

	d_changelog_edit->clear();
	
	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Open(QStringLiteral("Change_Data"),&version);

		switch(version)
		{
			case 1:
				this->Load_V1(&xml_file);
				break;
				
			default:
				throw TXmlFileException(QStringLiteral("Change datafile version not recognized"));
		}
	}
	catch(TXmlFileException xml_file_exception)
	{
		d_last_error = xml_file_exception.ErrorText();
		
		this->Initialize();
		result = false;
	}
	return result;
}

void TChangeLogWidget::Clear_Changelog(void)
{
	d_changelog_edit->clear();
	
	d_changelog_edit->appendPlainText(QStringLiteral("# Newest revisions added to the top; oldest revisions at the bottom."));
}

void TChangeLogWidget::Add_Changelog_Line(
	const QString 						&line)
{
	d_changelog_edit->appendPlainText(line);
}

void TChangeLogWidget::Initialize(void)
{
	d_changelog_edit->clear();
	
	d_changelog_edit->appendPlainText(QStringLiteral("# Newest revisions added to the top; oldest revisions at the bottom."));
	d_changelog_edit->appendPlainText(QStringLiteral("* Sun Nov 6 2016 <author name>"));
	d_changelog_edit->appendPlainText(QStringLiteral("- Initial RPM release"));
}

void TChangeLogWidget::Load_V1(
	TXmlFile 							* const xml_file)
{
	QDomElement							e0;
	QString								text;
		
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Text"));
	text = xml_file->Read_Text_Node(&e0);
	
	text.replace(QStringLiteral("<br>"),QStringLiteral("\n"));

	d_changelog_edit->appendPlainText(text);
}
