/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2025  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QGridLayout>
#include <QLabel>
#include <QLineEdit>
#include <QSpacerItem>
#include <QFont>

#include "../../core/xmlfile.h"

#include "expressionwidget.h"

TExpressionsWidget::TExpressionsWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QGridLayout							*widget_layout;
	QLabel								*expression_label;
	QLabel								*build_path_label;
	QLabel								*cmake_bin_path_label;
	QLabel								*qmake_bin_path_label;
	QLabel								*source_library_path_label;
	QLabel								*source_desktop_menu_path_label;
	QLabel								*target_install_path_label;
	QLabel								*target_desktop_menu_path_label;
	QSpacerItem							*widget_vspacer;
	QFont								font;

	font.setPointSize(15);
	font.setWeight(50);
	font.setItalic(true);
	font.setBold(false);
	font.setUnderline(true);

	this->resize(745,468);

	widget_layout = new QGridLayout(this);

	expression_label = new QLabel(this);
	expression_label->setFont(font);
	widget_layout->addWidget(expression_label,0,0,1,2);

	build_path_label = new QLabel(this);
	widget_layout->addWidget(build_path_label,1,0,1,1);

	d_build_path_edit = new QLineEdit(this);
	widget_layout->addWidget(d_build_path_edit,1,1,1,1);

	cmake_bin_path_label = new QLabel(this);
	widget_layout->addWidget(cmake_bin_path_label,2,0,1,1);

	d_cmake_bin_path_edit = new QLineEdit(this);
	widget_layout->addWidget(d_cmake_bin_path_edit,2,1,1,1);

	qmake_bin_path_label = new QLabel(this);
	widget_layout->addWidget(qmake_bin_path_label,3,0,1,1);

	d_qmake_bin_path_edit = new QLineEdit(this);
	widget_layout->addWidget(d_qmake_bin_path_edit,3,1,1,1);

	source_library_path_label = new QLabel(this);
	widget_layout->addWidget(source_library_path_label,4,0,1,1);

	d_library_path_edit = new QLineEdit(this);
	widget_layout->addWidget(d_library_path_edit,4,1,1,1);

	source_desktop_menu_path_label = new QLabel(this);
	widget_layout->addWidget(source_desktop_menu_path_label,5,0,1,1);

	d_desktop_menu_path_edit = new QLineEdit(this);
	widget_layout->addWidget(d_desktop_menu_path_edit,5,1,1,1);

	target_install_path_label = new QLabel(this);
	widget_layout->addWidget(target_install_path_label,6,0,1,1);

	d_target_install_path_edit = new QLineEdit(this);
	widget_layout->addWidget(d_target_install_path_edit,6,1,1,1);

	target_desktop_menu_path_label = new QLabel(this);
	widget_layout->addWidget(target_desktop_menu_path_label,7,0,1,1);

	d_target_desktop_menu_path_edit = new QLineEdit(this);
	widget_layout->addWidget(d_target_desktop_menu_path_edit,7,1,1,1);

	widget_vspacer = new QSpacerItem(0,0,QSizePolicy::Minimum,QSizePolicy::Expanding);
	widget_layout->addItem(widget_vspacer,8,1,1,1);
	
	// defaults

	expression_label->setText(QStringLiteral("Expressions"));
	build_path_label->setText(QStringLiteral("Build Root Path:"));
	cmake_bin_path_label->setText(QStringLiteral("CMake Bin Path:"));
	qmake_bin_path_label->setText(QStringLiteral("QMake Bin Path:"));
	source_library_path_label->setText(QStringLiteral("Libraries Path:"));
	source_desktop_menu_path_label->setText(QStringLiteral("Desktop Menu Path:"));
	target_install_path_label->setText(QStringLiteral("Target Install Path:"));
	target_desktop_menu_path_label->setText(QStringLiteral("Target Desktop Menu Path:"));

}

TExpressionsWidget::~TExpressionsWidget(void)
{
}

bool TExpressionsWidget::Save_Defaults(
	const QString 						&file_path)
{
	QString								file_name;
	TXmlFile							xml_file;
	QDomElement							e0;
	
	file_name = file_path + QStringLiteral("/expressions.xml");

	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Reset(QStringLiteral("Expression_Data"),1);

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Build_Path"));
		xml_file.Write_Text_Node(&e0,d_build_path_edit->text());

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("CMake_Bin_Path"));
		xml_file.Write_Text_Node(&e0,d_cmake_bin_path_edit->text());

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("QMake_Bin_Path"));
		xml_file.Write_Text_Node(&e0,d_qmake_bin_path_edit->text());

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Library_Path"));
		xml_file.Write_Text_Node(&e0,d_library_path_edit->text());

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Desktop_Menu_Path"));
		xml_file.Write_Text_Node(&e0,d_desktop_menu_path_edit->text());

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Target_Install_Path"));
		xml_file.Write_Text_Node(&e0,d_target_install_path_edit->text());

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Target_Desktop_Menu_Path"));
		xml_file.Write_Text_Node(&e0,d_target_desktop_menu_path_edit->text());

		xml_file.Write_File();
	}
	catch(TXmlFileException xml_file_exception)
	{
		d_last_error = xml_file_exception.ErrorText();
		return false;
	}

	return true;
}

QString TExpressionsWidget::Build_Path(void) const
{
	return d_build_path_edit->text();
}

QString TExpressionsWidget::CMake_Bin_Path(void) const
{
	return d_cmake_bin_path_edit->text();
}

QString TExpressionsWidget::QMake_Bin_Path(void) const
{
	return d_qmake_bin_path_edit->text();
}

QString TExpressionsWidget::Library_Path(void) const
{
	return d_library_path_edit->text();
}

QString TExpressionsWidget::Desktop_Menu_Path(void) const
{
	return d_desktop_menu_path_edit->text();
}

QString TExpressionsWidget::Target_Install_Path(void) const 
{
	return d_target_install_path_edit->text();
}

QString TExpressionsWidget::Target_Desktop_Menu_Path(void) const
{
	return d_target_desktop_menu_path_edit->text();
}

bool TExpressionsWidget::Load_Defaults(
	const QString 						&file_path)
{
	TXmlFile							xml_file;
	QString								file_name;
	int									version;
	bool								result(true);
	
	file_name = file_path + QStringLiteral("/expressions.xml");
	
	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Open(QStringLiteral("Expression_Data"),&version);

		switch(version)
		{
			case 1:
				this->Load_V1(&xml_file);
				break;
				
			default:
				throw TXmlFileException(QStringLiteral("Expression datafile version not recognized"));
		}
	}
	catch(TXmlFileException xml_file_exception)
	{
		d_last_error = xml_file_exception.ErrorText();
		
		this->Initialize();
		result = false;
	}


	return result;
}

void TExpressionsWidget::Initialize(void)
{
	d_build_path_edit->setText(QStringLiteral("%{buildroot}%{_libdir}/%{name}"));
	d_cmake_bin_path_edit->setText(QStringLiteral("%{buildroot}%{_libdir}/%{name}/bin"));
	d_qmake_bin_path_edit->setText(QStringLiteral("%{buildroot}%{_libdir}/%{name}"));
	d_library_path_edit->setText(QStringLiteral("%{_topdir}/shared_libs"));
	d_desktop_menu_path_edit->setText(QStringLiteral("%{_topdir}/desktop_menu"));
	d_target_install_path_edit->setText(QStringLiteral("%{_libdir}"));
	d_target_desktop_menu_path_edit->setText(QStringLiteral("%{_datadir}"));
}

void TExpressionsWidget::Load_V1(
	TXmlFile 							* const xml_file)
{
	QDomElement							e0;
		
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Build_Path"));
	d_build_path_edit->setText(xml_file->Read_Text_Node(&e0));

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("CMake_Bin_Path"));
	d_cmake_bin_path_edit->setText(xml_file->Read_Text_Node(&e0));

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("QMake_Bin_Path"));
	d_qmake_bin_path_edit->setText(xml_file->Read_Text_Node(&e0));

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Library_Path"));
	d_library_path_edit->setText(xml_file->Read_Text_Node(&e0));

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Desktop_Menu_Path"));
	d_desktop_menu_path_edit->setText(xml_file->Read_Text_Node(&e0));

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Target_Install_Path"));
	d_target_install_path_edit->setText(xml_file->Read_Text_Node(&e0));
	
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Target_Desktop_Menu_Path"));
	d_target_desktop_menu_path_edit->setText(xml_file->Read_Text_Node(&e0));
}
