/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2025  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QHBoxLayout>
#include <QSpacerItem>
#include <QToolButton>
#include <QTreeWidget>
#include <QVBoxLayout>
#include <QStringList>
#include <QProcess>
#include <QDir>

#include "../../core/xmlfile.h"

#include "requireswidget.h"

TRequiresWidget::TRequiresWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QHBoxLayout							*button_hlayout;
	QSpacerItem							*button_hspacer;
	QToolButton							*clear_button;
	QToolButton							*refresh_button;
	QTreeWidgetItem						*package_tree_header_item;
	QVBoxLayout							*widget_vlayout;

	widget_vlayout = new QVBoxLayout(this);

	d_package_tree = new QTreeWidget(this);
	d_package_tree->setRootIsDecorated(false);
	d_package_tree->setColumnCount(2);

	widget_vlayout->addWidget(d_package_tree);

	button_hlayout = new QHBoxLayout();

	button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_hlayout->addItem(button_hspacer);

	clear_button = new QToolButton(this);
	clear_button->setFixedSize(QSize(16,16));
	button_hlayout->addWidget(clear_button);

	refresh_button = new QToolButton(this);
	refresh_button->setFixedSize(QSize(16,16));
	button_hlayout->addWidget(refresh_button);

	widget_vlayout->addLayout(button_hlayout);
	
	// defaults

	clear_button->setIcon(QIcon(QStringLiteral(":/gui/edit_clear_x16.png")));
	refresh_button->setIcon(QIcon(QStringLiteral(":/gui/edit_refresh_x16.png")));

	package_tree_header_item = d_package_tree->headerItem();
	package_tree_header_item->setText(0,QStringLiteral("Package Providing Library"));
	package_tree_header_item->setText(1,QStringLiteral("Library File Name"));
	
	connect(refresh_button,&QToolButton::clicked,this,&TRequiresWidget::Refresh);
	connect(clear_button,&QToolButton::clicked,this,&TRequiresWidget::Clear);
}

TRequiresWidget::~TRequiresWidget(void)
{
}
	
QStringList TRequiresWidget::Required_Packages(void) const
{
	std::vector<TRequiresWidget::TRequireEntry>::const_iterator iter;
	QStringList							list;
		
	for(iter = d_requires.begin();iter != d_requires.end();++iter)
	{
		list.push_back((*iter).name);
	}

	return list;
}

bool TRequiresWidget::Save_Defaults(
	const QString 						&file_path)
{
	std::vector<TRequiresWidget::TRequireEntry>::const_iterator iter;
	QStringList::const_iterator			str_iter;
	QString								file_name;
	TXmlFile							xml_file;
	QDomElement							e0;
	QDomElement							e1;
	QDomElement							e2;
	QString								text;
	
	file_name = file_path + QStringLiteral("/requires.xml");

	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Reset(QStringLiteral("Requires"),1);
		
		for(iter = d_requires.begin();iter != d_requires.end();++iter)
		{
			e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Package"));
			
			e1 = xml_file.Create_Node(&e0,QStringLiteral("Name"));
			xml_file.Write_Text_Node(&e1,(*iter).name);
			
			e1 = xml_file.Create_Node(&e0,QStringLiteral("Libraries"));
			
			for(str_iter = (*iter).libraries.begin();str_iter != (*iter).libraries.end();++str_iter)
			{
				e2 = xml_file.Create_Node(&e1,QStringLiteral("Name"));
				xml_file.Write_Text_Node(&e2,(*str_iter));
			}
		}
		
		xml_file.Write_File();
	}
	catch(TXmlFileException xml_file_exception)
	{
		d_last_error = xml_file_exception.ErrorText();
		return false;
	}

	return true;
}

bool TRequiresWidget::Load_Defaults(
	const QString 						&file_path)
{
	TXmlFile							xml_file;
	QString								file_name;
	int									version;
	bool								result(true);

	file_name = file_path + QStringLiteral("/requires.xml");

	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Open(QStringLiteral("Requires"),&version);

		switch(version)
		{
			case 1:
				this->Load_V1(&xml_file);
				break;
				
			default:
				throw TXmlFileException(QStringLiteral("Requires datafile version not recognized"));
		}
	}
	catch(TXmlFileException xml_file_exception)
	{
		d_last_error = xml_file_exception.ErrorText();
		
		this->Initialize();
		result = false;
	}
	
	this->Update_Tree();

	return result;
}

void TRequiresWidget::Refresh_Required_Packages(
	const QStringList					&targets)
{
	QString								dir_path;
	QDir								dir;
	QStringList							dir_list;
	QStringList::const_iterator			iter;
	
	if(d_shared_libs_path.length())
	{
		d_requires.clear();
		
		dir.setPath(d_shared_libs_path);
		
		dir_list = dir.entryList(QDir::AllDirs | QDir::NoDotAndDotDot);
		
		for(iter = dir_list.begin();iter != dir_list.end();++iter)
		{
			if(targets.contains(*iter))
			{
				dir_path = d_shared_libs_path;
				dir_path += QString("/%1").arg(*iter);
				
				this->Process_Directory(dir_path);
			}
		}
			
		this->Update_Tree();
	}
}

void TRequiresWidget::Clear(void)
{
	d_requires.clear();
	d_package_tree->clear();
}

void TRequiresWidget::Initialize(void)
{
	TRequiresWidget::TRequireEntry			require_entry;
	
	d_requires.clear();
	
	require_entry.name = QStringLiteral("libxcb-image0");
	require_entry.libraries.clear();
	require_entry.libraries.push_back(QStringLiteral("libxcb-image.so.0"));
	require_entry.libraries.push_back(QStringLiteral("libxcb-image.so.1"));
	require_entry.libraries.push_back(QStringLiteral("libxcb-image.so.2"));
	d_requires.push_back(require_entry);

	require_entry.name = QStringLiteral("libxcb-image1");
	require_entry.libraries.clear();
	require_entry.libraries.push_back(QStringLiteral("libxcb-image.so.0"));
	require_entry.libraries.push_back(QStringLiteral("libxcb-image.so.1"));
	require_entry.libraries.push_back(QStringLiteral("libxcb-image.so.2"));
	
	d_requires.push_back(require_entry);
}

void TRequiresWidget::Process_Directory(
	const QString 						&path)
{
	QDir								dir;
	QStringList							file_names;
	QStringList::const_iterator			iter;
	QString								file_name;
	
	dir.setPath(path);
	
	file_names = dir.entryList(QDir::Files);
	
	for(iter = file_names.begin();iter != file_names.end();++iter)
	{
		file_name = path;
		file_name += QString("/%1").arg(*iter);
		
		this->Process_File(file_name);
	}
}

bool TRequiresWidget::Process_File(
	const QString 						&file_name)
{
	QRegularExpression 					library_name_regexp(QStringLiteral("^.+ => (.+) \\("));
	QRegularExpression					package_name_regexp(QStringLiteral("^(.+)[-][0-9.]{1,20}[-][0-9.]{1,20}"));
	QRegularExpressionMatch 			match;
	QString								output;
	QString								library_name;
	QString								package_name;
	QStringList							output_lines;
	QStringList::const_iterator			iter;
	
#ifdef Q_OS_LINUX
	QProcess 							ldd;
	
	ldd.start(QStringLiteral("ldd"), QStringList() << file_name);
	ldd.waitForFinished();

	if(ldd.exitStatus() != QProcess::NormalExit ||
	   ldd.exitCode() != 0)
	{
		d_last_error = QString("ERR: %1").arg(QString(ldd.readAllStandardError()));
		return false;
	}
	
	output = ldd.readAllStandardOutput();
	
#else
	
	// test data
	output = QStringLiteral("linux-vdso.so.1 (0x00007ffc3a571000)\n");
	output += QStringLiteral("libQt5XcbQpa.so.5 => /usr/lib64/ballbarviewer/qtlibs/libQt5XcbQpa.so.5 (0x00007ffaca99e000)\n");
	output += QStringLiteral("libfontconfig.so.1 => /usr/lib/x86_64-linux-gnu/libfontconfig.so.1 (0x00007ffaca944000)\n");
	output += QStringLiteral("libfreetype.so.6 => /usr/lib/x86_64-linux-gnu/libfreetype.so.6 (0x00007ffaca885000)\n");
	output += QStringLiteral("libz.so.1 => /lib/x86_64-linux-gnu/libz.so.1 (0x00007ffaca869000)\n");
	output += QStringLiteral("libQt5Gui.so.5 => /usr/lib64/ballbarviewer/qtlibs/libQt5Gui.so.5 (0x00007ffac9f82000)\n");
	output += QStringLiteral("libQt5DBus.so.5 => /usr/lib64/ballbarviewer/qtlibs/libQt5DBus.so.5 (0x00007ffac9cf2000)\n");
	output += QStringLiteral("libQt5Core.so.5 => /usr/lib64/ballbarviewer/qtlibs/libQt5Core.so.5 (0x00007ffac9470000)\n");
	output += QStringLiteral("/lib64/ld-linux-x86-64.so.2 (0x00007ffacaeb8000)\n");
	output += QStringLiteral("libGL.so.1 => /usr/lib/x86_64-linux-gnu/libGL.so.1 (0x00007ffac93e8000)\n");
	output += QStringLiteral("libpthread.so.0 => /lib/x86_64-linux-gnu/libpthread.so.0 (0x00007ffac93c5000)\n");
	output += QStringLiteral("libX11-xcb.so.1 => /usr/lib/x86_64-linux-gnu/libX11-xcb.so.1 (0x00007ffac93c0000)\n");
	output += QStringLiteral("libxcb-icccm.so.4 => /usr/lib/x86_64-linux-gnu/libxcb-icccm.so.4 (0x00007ffac93b9000)\n");
	output += QStringLiteral("libxcb-image.so.0 => /usr/lib/x86_64-linux-gnu/libxcb-image.so.0 (0x00007ffac91b4000)\n");
	output += QStringLiteral("libxcb-shm.so.0 => /usr/lib/x86_64-linux-gnu/libxcb-shm.so.0 (0x00007ffac91ad000)\n");
	output += QStringLiteral("libxcb-util.so.1 => /usr/lib/x86_64-linux-gnu/libxcb-util.so.1 (0x00007ffac8fa7000)\n");
	output += QStringLiteral("libxcb-keysyms.so.1 => /usr/lib/x86_64-linux-gnu/libxcb-keysyms.so.1 (0x00007ffac8fa2000)\n");
	output += QStringLiteral("libxcb-randr.so.0 => /usr/lib/x86_64-linux-gnu/libxcb-randr.so.0 (0x00007ffac8f8f000)\n");
	
#endif
	
	output_lines = output.split(QStringLiteral("\n"));
	
	for(iter = output_lines.begin();iter != output_lines.end();++iter)
	{		
		match = library_name_regexp.match(*iter);
	
		if(match.hasMatch())
		{
			library_name = match.captured(1).trimmed();
			
			if(library_name.contains(QStringLiteral("libQt4")))
			{
				continue;
			}
			
			if(library_name.contains(QStringLiteral("libQt5")))
			{
				continue;
			}
			
			if(library_name.contains(QStringLiteral("libQt6")))
			{
				continue;
			}

			if(!this->Has_Library(library_name))
			{
#ifdef Q_OS_LINUX
				QProcess 							rpm;
				
				rpm.start(QStringLiteral("rpm"), QStringList() << QStringLiteral("-q") << QStringLiteral("--whatprovides") << library_name);
				rpm.waitForFinished();

				if(rpm.exitStatus() != QProcess::NormalExit || rpm.exitCode() != 0)
				{
					d_last_error = QString("ERR: %1").arg(QString(rpm.readAllStandardError()));
					return false;
				}
				
				output = rpm.readAllStandardOutput();
#else
				output = QStringLiteral("libprovider-standard0-0.1-1.23.x86_64");
#endif
				
				match = package_name_regexp.match(output);

				if(match.hasMatch())
				{
					package_name = match.captured(1).trimmed();
					this->Add_Library(package_name,library_name);
				}
			}
		}
	}
	
	return true;
}

bool TRequiresWidget::Has_Library(
	const QString 						&library_name)
{
	std::vector<TRequiresWidget::TRequireEntry>::iterator iter;
	
	for(iter = d_requires.begin();iter != d_requires.end();++iter)
	{
		if((*iter).libraries.contains(library_name))
		{
			return true;
		}
	}
	
	return false;
}

void TRequiresWidget::Add_Library(
	const QString 						&package_name,
	const QString 						&library_name)
{
	TRequiresWidget::TRequireEntry		require_entry;
	std::vector<TRequiresWidget::TRequireEntry>::iterator iter;
		
	if(package_name == QStringLiteral("glibc")) // no need to add this
	{
		return;
	}
	
	for(iter = d_requires.begin();iter != d_requires.end();++iter)
	{
		if((*iter).name == package_name)
		{
			if(!(*iter).libraries.contains(library_name))
			{
				(*iter).libraries.push_back(library_name);
			}
			
			return;
		}
	}
	
	require_entry.name = package_name;
	require_entry.libraries.push_back(library_name);
	
	d_requires.push_back(require_entry);
}

void TRequiresWidget::Update_Tree(void)
{
	QTreeWidgetItem						*package_item;
	QTreeWidgetItem						*library_item;
	std::vector<TRequiresWidget::TRequireEntry>::const_iterator	iter;
	QStringList::const_iterator			lib_iter;
	bool								root_item_entry;

	d_package_tree->clear();
	
	for(iter = d_requires.begin();iter != d_requires.end();++iter)
	{
		package_item = new QTreeWidgetItem(d_package_tree);
		package_item->setText(0,(*iter).name);
		package_item->setExpanded(true);
		root_item_entry = true;
		
		for(lib_iter = (*iter).libraries.begin();lib_iter != (*iter).libraries.end();++lib_iter)
		{
			if(root_item_entry)
			{
				package_item->setText(1,(*lib_iter));
				root_item_entry = false;
			}
			else
			{
				library_item = new QTreeWidgetItem(package_item);
				library_item->setText(1,(*lib_iter));
			}
		}
	}
	
	d_package_tree->sortByColumn(0,Qt::AscendingOrder);

	d_package_tree->resizeColumnToContents(0);
	d_package_tree->resizeColumnToContents(1);
}

void TRequiresWidget::Load_V1(
	TXmlFile 							* const xml_file)
{
	TRequiresWidget::TRequireEntry		require_entry;
	QDomElement							e0;
	QDomElement							e1;
	QDomElement							e2;
	
	d_requires.clear();

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Package"),true);

	while(!e0.isNull())
	{
		e1 = xml_file->Get_Node(&e0,QStringLiteral("Name"));

		require_entry.name = xml_file->Read_Text_Node(&e1);
		require_entry.libraries.clear();

		e1 = xml_file->Get_Node(&e0,QStringLiteral("Libraries"));
		
		e2 = xml_file->Get_Node(&e1,QStringLiteral("Name"),true);
		while(!e2.isNull())
		{
			require_entry.libraries.push_back(xml_file->Read_Text_Node(&e2));
			
			e2 = xml_file->Get_Sibling_Node(&e2,QStringLiteral("Name"),true);
		}
		
		d_requires.push_back(require_entry);

		e0 = xml_file->Get_Sibling_Node(&e0,QStringLiteral("Package"),true);
	}
}
