/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QGridLayout>
#include <QLabel>
#include <QLineEdit>
#include <QTreeWidget>
#include <QTreeWidgetItem>
#include <QToolButton>
#include <QSpacerItem>
#include <QHBoxLayout>
#include <QFont>

#include "../../core/xmlfile.h"
#include "../../core/messagebox.h"

#include "linkeditdialog.h"
#include "expressionwidget.h"

TExpressionsWidget::TExpressionsWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QGridLayout							*widget_layout;
	QLabel								*expression_label;
	QLabel								*build_path_label;
	QLabel								*cmake_bin_path_label;
	QLabel								*qmake_bin_path_label;
	QLabel								*source_library_path_label;
	QLabel								*source_desktop_menu_path_label;
	QLabel								*target_install_path_label;
	QLabel								*target_desktop_menu_path_label;
	QLabel								*target_additional_executables_label;
	QLabel								*links_label;
	QTreeWidgetItem						*links_tree_header_item;
	QSpacerItem							*link_button_hspacer;
	QHBoxLayout							*link_button_hlayout;
	QFont								font;

	font.setPointSize(15);
	font.setWeight(50);
	font.setItalic(true);
	font.setBold(false);
	font.setUnderline(true);

	widget_layout = new QGridLayout(this);

	expression_label = new QLabel(this);
	expression_label->setFont(font);
	widget_layout->addWidget(expression_label,0,0,1,2);

	build_path_label = new QLabel(this);
	widget_layout->addWidget(build_path_label,1,0,1,1);

	d_build_path_edit = new QLineEdit(this);
	widget_layout->addWidget(d_build_path_edit,1,1,1,1);

	cmake_bin_path_label = new QLabel(this);
	widget_layout->addWidget(cmake_bin_path_label,2,0,1,1);

	d_cmake_bin_path_edit = new QLineEdit(this);
	widget_layout->addWidget(d_cmake_bin_path_edit,2,1,1,1);

	qmake_bin_path_label = new QLabel(this);
	widget_layout->addWidget(qmake_bin_path_label,3,0,1,1);

	d_qmake_bin_path_edit = new QLineEdit(this);
	widget_layout->addWidget(d_qmake_bin_path_edit,3,1,1,1);

	source_library_path_label = new QLabel(this);
	widget_layout->addWidget(source_library_path_label,4,0,1,1);

	d_library_path_edit = new QLineEdit(this);
	widget_layout->addWidget(d_library_path_edit,4,1,1,1);

	source_desktop_menu_path_label = new QLabel(this);
	widget_layout->addWidget(source_desktop_menu_path_label,5,0,1,1);

	d_desktop_menu_path_edit = new QLineEdit(this);
	widget_layout->addWidget(d_desktop_menu_path_edit,5,1,1,1);

	target_install_path_label = new QLabel(this);
	widget_layout->addWidget(target_install_path_label,6,0,1,1);

	d_target_install_path_edit = new QLineEdit(this);
	widget_layout->addWidget(d_target_install_path_edit,6,1,1,1);

	target_desktop_menu_path_label = new QLabel(this);
	widget_layout->addWidget(target_desktop_menu_path_label,7,0,1,1);

	d_target_desktop_menu_path_edit = new QLineEdit(this);
	widget_layout->addWidget(d_target_desktop_menu_path_edit,7,1,1,1);

	target_additional_executables_label = new QLabel(this);
	widget_layout->addWidget(target_additional_executables_label,8,0,1,1);

	d_target_additional_executables_edit = new QLineEdit(this);
	widget_layout->addWidget(d_target_additional_executables_edit,8,1,1,1);
	
	links_label = new QLabel(this);
	links_label->setFont(font);
	widget_layout->addWidget(links_label,9,0,1,2);

	d_links_tree = new QTreeWidget(this);
	d_links_tree->setColumnCount(2);
	d_links_tree->setSelectionMode(QAbstractItemView::ExtendedSelection);
	links_tree_header_item = d_links_tree->headerItem();
	widget_layout->addWidget(d_links_tree,10,0,1,2);

	link_button_hlayout = new QHBoxLayout();

	link_button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	link_button_hlayout->addItem(link_button_hspacer);

	d_add_link_button = new QToolButton(this);
	d_add_link_button->setFixedSize(QSize(16,16));
	link_button_hlayout->addWidget(d_add_link_button);

	d_remove_link_button = new QToolButton(this);
	d_remove_link_button->setFixedSize(QSize(16,16));
	link_button_hlayout->addWidget(d_remove_link_button);

	d_edit_link_button = new QToolButton(this);
	d_edit_link_button->setFixedSize(QSize(16,16));
	link_button_hlayout->addWidget(d_edit_link_button);

	widget_layout->addLayout(link_button_hlayout,11,0,1,2);
	
	d_add_link_button->setIcon(QIcon(QStringLiteral(":/gui/edit_add_x16.png")));
	d_remove_link_button->setIcon(QIcon(QStringLiteral(":/gui/edit_remove_x16.png")));
	d_edit_link_button->setIcon(QIcon(QStringLiteral(":/gui/edit_edit_x16.png")));
	
	// defaults
	d_link_edit_dialog = 0;
	d_msg_box = 0;

	d_remove_link_button->setEnabled(false);
	d_edit_link_button->setEnabled(false);

	links_tree_header_item->setText(0,QStringLiteral("Name"));
	links_tree_header_item->setText(1,QStringLiteral("Path"));
	
	d_add_link_button->setToolTip(QStringLiteral("Add new link"));
	d_remove_link_button->setToolTip(QStringLiteral("Remove selected links"));
	d_edit_link_button->setToolTip(QStringLiteral("Edit selected link"));

	expression_label->setText(QStringLiteral("Expressions"));
	links_label->setText(QStringLiteral("Symbolic Links"));
	build_path_label->setText(QStringLiteral("Build Root Path:"));
	cmake_bin_path_label->setText(QStringLiteral("CMake Bin Path:"));
	qmake_bin_path_label->setText(QStringLiteral("QMake Bin Path:"));
	source_library_path_label->setText(QStringLiteral("Libraries Path:"));
	source_desktop_menu_path_label->setText(QStringLiteral("Desktop Menu Path:"));
	target_install_path_label->setText(QStringLiteral("Target Install Path:"));
	target_desktop_menu_path_label->setText(QStringLiteral("Target Desktop Menu Path:"));
	target_additional_executables_label->setText(QStringLiteral("Additional Executables:"));
	
	connect(d_links_tree,&QTreeWidget::itemClicked,this,&TExpressionsWidget::Link_Item_Clicked);
	connect(d_links_tree,&QTreeWidget::itemSelectionChanged,this,&TExpressionsWidget::Link_Selection_Changed);
	connect(d_add_link_button,&QToolButton::clicked,this,&TExpressionsWidget::Add_Link);
	connect(d_remove_link_button,&QToolButton::clicked,this,&TExpressionsWidget::Remove_Link);
	connect(d_edit_link_button,&QToolButton::clicked,this,&TExpressionsWidget::Edit_Link);

}

TExpressionsWidget::~TExpressionsWidget(void)
{
}

bool TExpressionsWidget::Save_Defaults(
	const QString 						&file_path)
{
	std::vector<TExpressionsWidget::TLink>::const_iterator iter;
	QString								file_name;
	TXmlFile							xml_file;
	QDomElement							e0;
	QDomElement							e1;
	QDomElement							e2;

	file_name = file_path + QStringLiteral("/expressions.xml");

	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Reset(QStringLiteral("Expression_Data"),2);

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Build_Path"));
		xml_file.Write_Text_Node(&e0,d_build_path_edit->text());

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("CMake_Bin_Path"));
		xml_file.Write_Text_Node(&e0,d_cmake_bin_path_edit->text());

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("QMake_Bin_Path"));
		xml_file.Write_Text_Node(&e0,d_qmake_bin_path_edit->text());

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Library_Path"));
		xml_file.Write_Text_Node(&e0,d_library_path_edit->text());

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Desktop_Menu_Path"));
		xml_file.Write_Text_Node(&e0,d_desktop_menu_path_edit->text());

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Target_Install_Path"));
		xml_file.Write_Text_Node(&e0,d_target_install_path_edit->text());

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Target_Desktop_Menu_Path"));
		xml_file.Write_Text_Node(&e0,d_target_desktop_menu_path_edit->text());

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Target_Additional_Executables"));
		xml_file.Write_Text_Node(&e0,d_target_additional_executables_edit->text());
		
		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Links"));

		for(iter = d_links.begin();iter != d_links.end();++iter)
		{
			e1 = xml_file.Create_Node(&e0,QStringLiteral("Link"));
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Name"));
			xml_file.Write_Text_Node(&e2,(*iter).name);
			
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Path"));
			xml_file.Write_Text_Node(&e2,(*iter).path);
			
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Checked"));
			(*iter).checked ? xml_file.Write_Text_Node(&e2,QStringLiteral("Yes")):xml_file.Write_Text_Node(&e2,QStringLiteral("No"));
		}

		xml_file.Write_File();
	}
	catch(TXmlFileException xml_file_exception)
	{
		d_last_error = xml_file_exception.ErrorText();
		return false;
	}

	return true;
}

QString TExpressionsWidget::Build_Path(void) const
{
	return d_build_path_edit->text();
}

QString TExpressionsWidget::CMake_Bin_Path(void) const
{
	return d_cmake_bin_path_edit->text();
}

QString TExpressionsWidget::QMake_Bin_Path(void) const
{
	return d_qmake_bin_path_edit->text();
}

QString TExpressionsWidget::Library_Path(void) const
{
	return d_library_path_edit->text();
}

QString TExpressionsWidget::Desktop_Menu_Path(void) const
{
	return d_desktop_menu_path_edit->text();
}

QString TExpressionsWidget::Target_Install_Path(void) const 
{
	return d_target_install_path_edit->text();
}

QString TExpressionsWidget::Target_Desktop_Menu_Path(void) const
{
	return d_target_desktop_menu_path_edit->text();
}

QString TExpressionsWidget::Target_Additional_Executables(void) const
{
	return d_target_additional_executables_edit->text();
}
	
int TExpressionsWidget::Symbolic_Links_Count(void) const
{
	return static_cast<int>(d_links.size());
}

TExpressionsWidget::TLink TExpressionsWidget::Symbolic_Link(
	const int 							index) const
{
	assert(static_cast<unsigned int>(index) < d_links.size());
	
	return d_links[index];
}

bool TExpressionsWidget::Load_Defaults(
	const QString 						&file_path)
{
	TXmlFile							xml_file;
	QString								file_name;
	int									version;
	bool								result(true);
	
	file_name = file_path + QStringLiteral("/expressions.xml");
	
	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Open(QStringLiteral("Expression_Data"),&version);

		switch(version)
		{
			case 1:
				this->Load_V1(&xml_file);
				break;
				
			case 2:
				this->Load_V2(&xml_file);
				break;

			default:
				throw TXmlFileException(QStringLiteral("Expression datafile version not recognized"));
		}
	}
	catch(TXmlFileException xml_file_exception)
	{
		d_last_error = xml_file_exception.ErrorText();

		this->Initialize();
		result = false;
	}

	this->Update_Link_Tree();

	return result;
}

void TExpressionsWidget::Set_Target_Additional_Executables(
	const QString &text)
{
	d_target_additional_executables_edit->setText(text);
}

void TExpressionsWidget::Uncheck_Symbolic_Links(void)
{
	std::vector<TExpressionsWidget::TLink>::iterator iter;
		
	for(iter = d_links.begin();iter != d_links.end();++iter)
	{
		(*iter).checked = false;
	}

	this->Update_Link_Tree();
}

void TExpressionsWidget::Add_Symbolic_Link(
	const QString						&name,
	const QString						&value)
{
	TExpressionsWidget::TLink			link;
	
	if(this->Link_Exists(name))
	{
		this->Enable_Link(name,true);
		return;
	}
	
	link.name = name;
	link.path = value;
	link.checked = true;
	
	d_links.push_back(link);
}

void TExpressionsWidget::Update_Symbolic_Link_Tree(void)
{
	this->Update_Link_Tree();
}

void TExpressionsWidget::Link_Item_Clicked(void)
{
	this->Update_Link_Data();
}

void TExpressionsWidget::Link_Selection_Changed(void)
{
	QList<QTreeWidgetItem*>				items;
	
	items = d_links_tree->selectedItems();
	
	d_remove_link_button->setEnabled(items.size() > 0);
	d_edit_link_button->setEnabled(items.size() == 1);
}

void TExpressionsWidget::Add_Link(void)
{
	TExpressionsWidget::TLink			link;

	if(!d_link_edit_dialog)
	{
		d_link_edit_dialog = new TLinkEditDialog(this);
	}
	
	d_link_edit_dialog->Reset(TLinkEditDialog::MODE_ADD);
	
	if(QDialog::Accepted == d_link_edit_dialog->exec())
	{
		link.name = d_link_edit_dialog->Name();
		link.path = d_link_edit_dialog->Path();
		link.checked = true;
				
		if(!this->Link_Exists(link.name))
		{
			d_links.push_back(link);
			this->Update_Link_Tree();
		}
	}
}

void TExpressionsWidget::Remove_Link(void)
{
	QList<QTreeWidgetItem*>				items;
	QList<QTreeWidgetItem*>::const_iterator iter;
	
	if(!d_msg_box)
	{
		d_msg_box = new TMessageBox(this);
	}
	
	d_msg_box->setText(QStringLiteral("Delete Links"));
	d_msg_box->setInformativeText(QStringLiteral("Do you want to delete the selected links?"));
	d_msg_box->setDetailedText(QString());
	d_msg_box->setStandardButtons(QMessageBox::Yes | QMessageBox::No);
	d_msg_box->setDefaultButton(QMessageBox::Yes);
	d_msg_box->setIcon(QMessageBox::Warning);
	
	if(QMessageBox::No == d_msg_box->exec())
	{
		return;
	}

	items = d_links_tree->selectedItems();

	for(iter = items.begin();iter != items.end();++iter)
	{
		this->Link_Remove((*iter)->text(0));
	}
	
	this->Update_Link_Tree();
}

void TExpressionsWidget::Edit_Link(void)
{
	QList<QTreeWidgetItem*>				items;
	TExpressionsWidget::TLink			link;
	
	items = d_links_tree->selectedItems();

	if(items.size() != 1)
	{
		return;
	}

	if(!d_link_edit_dialog)
	{
		d_link_edit_dialog = new TLinkEditDialog(this);
	}
		
	d_link_edit_dialog->Set_Name(items[0]->text(0));
	d_link_edit_dialog->Set_Path(items[0]->text(1));
	
	d_link_edit_dialog->Reset(TLinkEditDialog::MODE_EDIT);
	
	if(QDialog::Accepted == d_link_edit_dialog->exec())
	{
		items[0]->setText(0,d_link_edit_dialog->Name());
		items[0]->setText(1,d_link_edit_dialog->Path());

		this->Update_Link_Data();
	}
}

void TExpressionsWidget::Update_Link_Tree(void)
{
	std::vector<TExpressionsWidget::TLink>::const_iterator	iter;
	QTreeWidgetItem						*item;
	
	d_links_tree->clear();
		
	for(iter = d_links.begin();iter != d_links.end();++iter)
	{
		item = new QTreeWidgetItem(d_links_tree);
				
		(*iter).checked ? item->setCheckState(0,Qt::Checked) : item->setCheckState(0,Qt::Unchecked);

		item->setText(0,(*iter).name);
		item->setText(1,(*iter).path);
	}
	
	d_links_tree->resizeColumnToContents(0);
	d_links_tree->resizeColumnToContents(1);
}

void TExpressionsWidget::Update_Link_Data(void)
{
	QTreeWidgetItem						*item;
	TExpressionsWidget::TLink			link;
	int									count;
	int									cntr;
		
	d_links.clear();
	count = d_links_tree->topLevelItemCount();
	
	for(cntr = 0;cntr < count;++cntr)
	{
		item = d_links_tree->topLevelItem(cntr);
		
		link.name = item->text(0);
		link.path = item->text(1);

		link.checked = (item->checkState(0) == Qt::Checked);
		
		d_links.push_back(link);
	}
}
	
void TExpressionsWidget::Link_Remove(
	const QString 						&name)
{
	std::vector<TExpressionsWidget::TLink>::iterator iter;

	for(iter = d_links.begin();iter != d_links.end();++iter)
	{
		if((*iter).name == name)
		{
			d_links.erase(iter);
			return;
		}
	}
}

bool TExpressionsWidget::Link_Exists(
	const QString 						&name) const
{
	std::vector<TExpressionsWidget::TLink>::const_iterator iter;

	for(iter = d_links.begin();iter != d_links.end();++iter)
	{
		if((*iter).name == name)
		{
			return true;
		}
	}

	return false;
}

void TExpressionsWidget::Initialize(void)
{
	TExpressionsWidget::TLink			link;
	
	d_links.clear();
	
	link.name = QStringLiteral("core");
	link.path = QStringLiteral("/home/xxx/MyProjects/core");
	link.checked = false;
	
	d_links.push_back(link);

	d_build_path_edit->setText(QStringLiteral("%{buildroot}%{_libdir}/%{name}"));
	d_cmake_bin_path_edit->setText(QStringLiteral("%{buildroot}%{_libdir}/%{name}/bin"));
	d_qmake_bin_path_edit->setText(QStringLiteral("%{buildroot}%{_libdir}/%{name}"));
	d_library_path_edit->setText(QStringLiteral("%{_topdir}/shared_libs"));
	d_desktop_menu_path_edit->setText(QStringLiteral("%{_topdir}/desktop_menu"));
	d_target_install_path_edit->setText(QStringLiteral("%{_libdir}"));
	d_target_desktop_menu_path_edit->setText(QStringLiteral("%{_datadir}"));
	d_target_additional_executables_edit->setText(QString());
}

void TExpressionsWidget::Enable_Link(
	const QString 						&name,
	const bool 							state)
{
	std::vector<TExpressionsWidget::TLink>::iterator iter;

	for(iter = d_links.begin();iter != d_links.end();++iter)
	{
		if((*iter).name == name)
		{
			(*iter).checked = state;
			return;
		}
	}
}


void TExpressionsWidget::Load_V1(
	TXmlFile 							* const xml_file)
{
	QDomElement							e0;
		
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Build_Path"));
	d_build_path_edit->setText(xml_file->Read_Text_Node(&e0));

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("CMake_Bin_Path"));
	d_cmake_bin_path_edit->setText(xml_file->Read_Text_Node(&e0));

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("QMake_Bin_Path"));
	d_qmake_bin_path_edit->setText(xml_file->Read_Text_Node(&e0));

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Library_Path"));
	d_library_path_edit->setText(xml_file->Read_Text_Node(&e0));

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Desktop_Menu_Path"));
	d_desktop_menu_path_edit->setText(xml_file->Read_Text_Node(&e0));

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Target_Install_Path"));
	d_target_install_path_edit->setText(xml_file->Read_Text_Node(&e0));
	
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Target_Desktop_Menu_Path"));
	d_target_desktop_menu_path_edit->setText(xml_file->Read_Text_Node(&e0));
	
	d_target_additional_executables_edit->setText(QString());
	
	d_links.clear();
	d_links_tree->clear();
}

void TExpressionsWidget::Load_V2(
	TXmlFile 							* const xml_file)
{
	TExpressionsWidget::TLink			link;
	QDomElement							e0;
	QDomElement							e1;
	QDomElement							e2;
	QString								text;

	d_links.clear();

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Build_Path"));
	d_build_path_edit->setText(xml_file->Read_Text_Node(&e0));

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("CMake_Bin_Path"));
	d_cmake_bin_path_edit->setText(xml_file->Read_Text_Node(&e0));

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("QMake_Bin_Path"));
	d_qmake_bin_path_edit->setText(xml_file->Read_Text_Node(&e0));

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Library_Path"));
	d_library_path_edit->setText(xml_file->Read_Text_Node(&e0));

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Desktop_Menu_Path"));
	d_desktop_menu_path_edit->setText(xml_file->Read_Text_Node(&e0));

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Target_Install_Path"));
	d_target_install_path_edit->setText(xml_file->Read_Text_Node(&e0));
	
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Target_Desktop_Menu_Path"));
	d_target_desktop_menu_path_edit->setText(xml_file->Read_Text_Node(&e0));
	
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Target_Additional_Executables"));
	d_target_additional_executables_edit->setText(xml_file->Read_Text_Node(&e0,true));	// empty node is valid
	
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Links"));
	if(!e0.isNull())
	{
		e1 = xml_file->Get_Node(&e0,QStringLiteral("Link"));
		while(!e1.isNull())
		{
			e2 = xml_file->Get_Node(&e1,QStringLiteral("Name"));
			link.name = xml_file->Read_Text_Node(&e2);
			
			e2 = xml_file->Get_Node(&e1,QStringLiteral("Path"));
			link.path = xml_file->Read_Text_Node(&e2);
			
			e2 = xml_file->Get_Node(&e1,QStringLiteral("Checked"));
			text = xml_file->Read_Text_Node(&e2);
			
			(text == QStringLiteral("Yes")) ? link.checked = true:link.checked = false;

			d_links.push_back(link);

			e1 = xml_file->Get_Sibling_Node(&e1,QStringLiteral("Link"),true);
		}
	}
}
