/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <string.h>
#include <assert.h>

#include "mat4.h"

TMat4::TMat4(void)
{
    memset(d_mat_data,0,sizeof(d_mat_data));
    d_mat_data[0] = d_mat_data[5] = d_mat_data[10] = d_mat_data[15] = 1;
}

TMat4::TMat4(
     const TMat4                        &m)
{
     this->Set(m.d_mat_data);
};

TMat4::TMat4(
     const double						*Data)
{
     memcpy(d_mat_data,Data,sizeof(d_mat_data));
}

TMat4& TMat4::operator=(
	const TMat4							&m)
{
	if(&m != this)
	{
		this->Set(m.Get());
	}
	
	return *this;
}

TMat4::~TMat4(void)
{

}

#ifdef MAT4_DEBUG
QString TMat4::DebugData(void)
{
	QString								Text;

	text = QStringLiteral("       X            |       Y            |       Z            |\n");
	text += QStringLiteral("___________________________________________________________________________________\n");
	
	Text += QString("%1 %2 %3 %4")
			.arg(d_mat_data[0],20,'f',12)
			.arg(d_mat_data[4],20,'f',12)
			.arg(d_mat_data[8],20,'f',12)
			.arg(d_mat_data[12],20,'f',12);

	Text += QString("%1 %2 %3 %4")
			.arg(d_mat_data[1],20,'f',12)
			.arg(d_mat_data[5],20,'f',12)
			.arg(d_mat_data[9],20,'f',12)
			.arg(d_mat_data[13],20,'f',12);

	Text += QString("%1 %2 %3 %4")
			.arg(d_mat_data[2],20,'f',12)
			.arg(d_mat_data[6],20,'f',12)
			.arg(d_mat_data[10],20,'f',12)
			.arg(d_mat_data[14],20,'f',12);

	Text += QString("%1 %2 %3 %4")
			.arg(d_mat_data[3],20,'f',12)
			.arg(d_mat_data[7],20,'f',12)
			.arg(d_mat_data[11],20,'f',12)
			.arg(d_mat_data[15],20,'f',12);

	return Text;
}
#endif

void TMat4::Set(
    const double						*Data)
{
	memcpy(d_mat_data,Data,sizeof(d_mat_data));
}

void TMat4::Normal(void)
{
     TVector3							v;

     v = this->X();
     v.Normal();
     this->X(v);


     v = this->Y();
     v.Normal();
     this->Y(v);

     v = this->Z();
     v.Normal();
     this->Z(v);
}

void TMat4::Identity(void)
{
     memset(d_mat_data,0,sizeof(d_mat_data));
     d_mat_data[0] = d_mat_data[5] = d_mat_data[10] = d_mat_data[15] = 1;
}

void TMat4::Transpose(void)
{
     double								t;

     t = d_mat_data[1];  d_mat_data[1]  = d_mat_data[4];  d_mat_data[4]  = t;
     t = d_mat_data[2];  d_mat_data[2]  = d_mat_data[8];  d_mat_data[8]  = t;
     t = d_mat_data[3];  d_mat_data[3]  = d_mat_data[12]; d_mat_data[12] = t;
     t = d_mat_data[6];  d_mat_data[6]  = d_mat_data[9];  d_mat_data[9]  = t;
     t = d_mat_data[7];  d_mat_data[7]  = d_mat_data[13]; d_mat_data[13] = t;
     t = d_mat_data[11]; d_mat_data[11] = d_mat_data[14]; d_mat_data[14] = t;
}

void TMat4::Rotate(
     const TVector3						&axis,
     const double						&angle)
{
     double								rcos;
     double								rsin;
     TMat4								rotation_mat;

     rcos = cos(angle);
     rsin = sin(angle);

     rotation_mat[0] = rcos + axis.i * axis.i * (1.0 - rcos);
     rotation_mat[1] = axis.k * rsin + axis.j * axis.i * (1.0 - rcos);
     rotation_mat[2] = -axis.j * rsin + axis.k * axis.i * (1.0 - rcos);
     rotation_mat[4] = -axis.k * rsin + axis.i * axis.j * (1.0 - rcos);
     rotation_mat[5] = rcos + axis.j * axis.j * (1.0 - rcos);
     rotation_mat[6] = axis.i * rsin + axis.k * axis.j * (1.0 - rcos);
     rotation_mat[8] = axis.j * rsin + axis.i * axis.k * (1.0 - rcos);
     rotation_mat[9] = -axis.i * rsin + axis.j * axis.k * (1.0 - rcos);
     rotation_mat[10] = rcos + axis.k * axis.k * (1.0 - rcos);

     this->operator*=(rotation_mat);
}

void TMat4::Rotate(
	const double						&alpha,
	const double						&beta,
	const double						&gamma)
{
	double								alpha_cos;
	double								alpha_sin;
	double								beta_cos;
	double								beta_sin;
	double								gamma_cos;
	double								gamma_sin;
	TMat4								rotation_mat;
	
	alpha_cos = cos(alpha);
	alpha_sin = sin(alpha);
	beta_cos = cos(beta);
	beta_sin = sin(beta);
	gamma_cos = cos(gamma);
	gamma_sin = sin(gamma);
	
	rotation_mat[0] = (beta_cos * gamma_cos) - (alpha_cos * beta_sin * gamma_sin);
	rotation_mat[1] = (beta_sin * gamma_cos) + (alpha_cos * beta_cos * gamma_sin);
	rotation_mat[2] = alpha_sin * gamma_sin;
	rotation_mat[4] = -(beta_cos * gamma_sin) - (alpha_cos * beta_sin * gamma_cos);
	rotation_mat[5] = -(beta_sin * gamma_sin) + (alpha_cos * beta_cos * gamma_cos);
	rotation_mat[6] = alpha_sin * gamma_cos;
	rotation_mat[8] = alpha_sin * beta_sin;
	rotation_mat[9] = -(alpha_sin * beta_cos);
	rotation_mat[10] = alpha_cos;

	this->operator*=(rotation_mat);
}

TMat4 TMat4::operator*=(
     const TMat4						&m)
{
     double								Values[16];

     if(&m != this)
     {
          Values[0] = d_mat_data[0]*m[0]   + d_mat_data[4]*m[1]  + d_mat_data[8]*m[2]   + d_mat_data[12]*m[3];
          Values[1] = d_mat_data[1]*m[0]   + d_mat_data[5]*m[1]  + d_mat_data[9]*m[2]   + d_mat_data[13]*m[3];
          Values[2] = d_mat_data[2]*m[0]   + d_mat_data[6]*m[1]  + d_mat_data[10]*m[2]  + d_mat_data[14]*m[3];
          Values[3] = d_mat_data[3]*m[0]   + d_mat_data[7]*m[1]  + d_mat_data[11]*m[2]  + d_mat_data[15]*m[3];
          Values[4] = d_mat_data[0]*m[4]   + d_mat_data[4]*m[5]  + d_mat_data[8]*m[6]   + d_mat_data[12]*m[7];
          Values[5] = d_mat_data[1]*m[4]   + d_mat_data[5]*m[5]  + d_mat_data[9]*m[6]   + d_mat_data[13]*m[7];
          Values[6] = d_mat_data[2]*m[4]   + d_mat_data[6]*m[5]  + d_mat_data[10]*m[6]  + d_mat_data[14]*m[7];
          Values[7] = d_mat_data[3]*m[4]   + d_mat_data[7]*m[5]  + d_mat_data[11]*m[6]  + d_mat_data[15]*m[7];
          Values[8] = d_mat_data[0]*m[8]   + d_mat_data[4]*m[9]  + d_mat_data[8]*m[10]  + d_mat_data[12]*m[11];
          Values[9] = d_mat_data[1]*m[8]   + d_mat_data[5]*m[9]  + d_mat_data[9]*m[10]  + d_mat_data[13]*m[11];
          Values[10] = d_mat_data[2]*m[8]  + d_mat_data[6]*m[9]  + d_mat_data[10]*m[10] + d_mat_data[14]*m[11];
          Values[11] = d_mat_data[3]*m[8]  + d_mat_data[7]*m[9]  + d_mat_data[11]*m[10] + d_mat_data[15]*m[11];
          Values[12] = d_mat_data[0]*m[12] + d_mat_data[4]*m[13] + d_mat_data[8]*m[14]  + d_mat_data[12]*m[15];
          Values[13] = d_mat_data[1]*m[12] + d_mat_data[5]*m[13] + d_mat_data[9]*m[14]  + d_mat_data[13]*m[15];
          Values[14] = d_mat_data[2]*m[12] + d_mat_data[6]*m[13] + d_mat_data[10]*m[14] + d_mat_data[14]*m[15];
          Values[15] = d_mat_data[3]*m[12] + d_mat_data[7]*m[13] + d_mat_data[11]*m[14] + d_mat_data[15]*m[15];

          memcpy(d_mat_data,Values,sizeof(d_mat_data));
     }
     return *this;
}

TMat4 operator *(
     const TMat4						&m1,
     const TMat4						&m2)
{
     return TMat4(m1)*= m2;
}

TVector3 operator*(
     const TMat4						&m,
     const TVector3						&v)
{

     return TVector3(
               m[0] * v.x + m[1] * v.y + m[2]  * v.z + m[3],
               m[4] * v.x + m[5] * v.y + m[6]  * v.z + m[7],
               m[8] * v.x + m[9] * v.y + m[10] * v.z + m[11]);
}

