/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QIcon>
#include <QMouseEvent>
#include <QKeyEvent>
#include <QWheelEvent>
#include <QRubberBand>
#include <QSurfaceFormat>
#include <QPaintEvent>
#include <QOffscreenSurface>
#include <vector>
#include <assert.h>

#include "openglwidget.h"

static const double						Z_DEPTH_DIVISOR = 4;
static const double						ZERO_EPSILON = 0.000001;

TOpenGLWidget::TOpenGLWidget(
    const QWidget						*parent,
    Qt::WindowFlags 					flags)
#ifdef Q_OS_WIN
:	QOpenGLWidget(const_cast<QWidget*>(parent),flags | Qt::MSWindowsOwnDC),
#else
:	QOpenGLWidget(const_cast<QWidget*>(parent),flags),
#endif
	d_shift_key_down(false),
	d_ctrl_key_down(false)
{
	this->setFocusPolicy(Qt::StrongFocus);
	
    d_modelview_mat.Identity();
    d_projection_mat.Identity();

	d_rubberband = 0;
	
	d_processing_scale_to_fit = false;
	d_initialize_scale_to_fit = false;
	d_scale_to_fit_render_mode = true;
}

TOpenGLWidget::~TOpenGLWidget(void)
{
}

int TOpenGLWidget::Get_OpenGL_Version_Major(
	QString								* const version_text,
	QString								* const extensions_text)
{
	QOffscreenSurface					surface;
	QOpenGLContext						context;
	QString								text;
	int									major_version(0);
	int									index;
	
	assert(version_text);
	assert(extensions_text);
	
	surface.setFormat(QSurfaceFormat::defaultFormat());
	surface.create();
	
	if(context.create())
	{
		if(context.makeCurrent(&surface))
		{
			*version_text = QString::fromLatin1((const char*)context.functions()->glGetString(GL_VERSION));
			text = (*version_text).simplified();

			// get opengl version string.  could be 1 or 2.4 or 3.4.10 etc.
			index = text.indexOf(' ');
			if(!(index < 0))
			{
				text.truncate(index);
			}
			
			index = text.indexOf('.');
			if(!(index < 0))
			{
				text.truncate(index);
			}
			
			major_version = text.toInt();

			*extensions_text = QString::fromLatin1((const char*)context.functions()->glGetString(GL_EXTENSIONS));
			
			context.doneCurrent();
		}
	}
	
	surface.destroy();
	
	return major_version;
}

void TOpenGLWidget::mousePressEvent(
    QMouseEvent 						*event)
{
	QPoint								pos;
	short								x_gl;
	short                              	y_gl;
	Qt::MouseButtons					mouse_buttons;

	this->makeCurrent();

	this->setFocus();
	mouse_buttons = event->buttons();

	if(mouse_buttons & Qt::LeftButton)
	{
		d_rubberband_moving = false;
	}
	
//	d_mouse_down = QPoint(event->x(),event->y());
	d_mouse_down = event->pos();

	if(mouse_buttons & Qt::RightButton)
	{
		x_gl = d_mouse_down.x();
		y_gl = d_mouse_down.y();

		this->Convert_To_GLViewport(&x_gl,&y_gl);

		d_right_mouse_down.setX(static_cast<int>(x_gl));
		d_right_mouse_down.setY(static_cast<int>(y_gl));

		d_right_mouse_down_moving = false;

		glGetDoublev(GL_MODELVIEW_MATRIX,d_modelview_mat.Get());
		glGetDoublev(GL_PROJECTION_MATRIX,d_projection_mat.Get());

		d_modelview_mat.Normal();
		this->Set_Rotation_Center();
	}
}

void TOpenGLWidget::mouseMoveEvent(
    QMouseEvent 						*event)
{
	TMat4								rotate_mat;
	TMat4								inverse_modelview_mat;
	TVector3							rotate_axis;
	TVector3							v1,v2;
	QPoint								pos;
	QPoint								mouse_offset;
	double								rotate_angle;
	short								x_gl;
	short								y_gl;
	const double						ZERO_EPSILON(1e-8);
	Qt::MouseButtons					mouse_buttons;
	Qt::KeyboardModifiers				keyboard_modifiers;

	this->makeCurrent();
	
	mouse_offset = d_mouse_down - event->pos();
	
	if(mouse_offset.manhattanLength() < 2)
	{
		return;
	}
			
	mouse_buttons = event->buttons();
	keyboard_modifiers = event->modifiers();

	pos = event->pos();
	
	if(mouse_buttons == Qt::RightButton)
	{		
		x_gl = pos.x();
		y_gl = pos.y();

		this->Convert_To_GLViewport(&x_gl,&y_gl);

		mouse_offset = d_right_mouse_down;

		mouse_offset.setX(mouse_offset.x() - x_gl);
		mouse_offset.setY(mouse_offset.y() - y_gl);

		if(abs(mouse_offset.x()) < 2 && abs(mouse_offset.y()) < 2)
		{
			return;
		}

		// Rotate 3d
		if((keyboard_modifiers & Qt::ControlModifier) == Qt::ControlModifier || d_rotate_3d_mode == true)
		{
			rotate_axis.y = static_cast<double>(-mouse_offset.x());
			rotate_axis.x = static_cast<double>(mouse_offset.y());

			rotate_angle = 0.02;
			rotate_angle *= (1 + log(rotate_axis.Length()));

			rotate_axis = d_modelview_mat * rotate_axis;
			rotate_axis.Normal();

			rotate_mat.Rotate(rotate_axis,rotate_angle);
			d_modelview_mat*=rotate_mat;

			glMatrixMode(GL_MODELVIEW);
			glLoadMatrixd(d_modelview_mat.Get());

			inverse_modelview_mat = d_modelview_mat;
			inverse_modelview_mat.Transpose();

			v1 = d_rotation_center;
			v1 = inverse_modelview_mat * v1;
			v1.x *= d_projection_mat.X().x;
			v1.y *= d_projection_mat.Y().y;
			v1.z *= d_projection_mat.Z().z;

			d_projection_mat.Origin(v1);

			glMatrixMode(GL_PROJECTION);
			glLoadMatrixd(d_projection_mat.Get());
		}

		// Rotate 2d
		else if((keyboard_modifiers & Qt::ShiftModifier) == Qt::ShiftModifier || d_rotate_2d_mode == true)
		{
			v1.Set(static_cast<double>(d_right_mouse_down.x() - d_viewport_center.x()),
			       static_cast<double>(d_right_mouse_down.y() - d_viewport_center.y()),
			       0);

			v2.Set(static_cast<double>(x_gl - d_viewport_center.x()),
			       static_cast<double>(y_gl - d_viewport_center.y()),
			       0);

			if(v1.Length() > 5 && v2.Length() > 5)
			{
				v1.Normal();
				v2.Normal();
	
				rotate_angle = v1 ^ v2;

				if(rotate_angle > 0.0 && rotate_angle < 1.0)
				{
					rotate_axis = v1 * v2;
					rotate_angle = acos(rotate_angle);

					if(rotate_axis.Length() > ZERO_EPSILON)
					{
						rotate_axis = d_modelview_mat * rotate_axis;
						rotate_axis.Normal();

						rotate_mat.Rotate(rotate_axis,rotate_angle);
						d_modelview_mat*=rotate_mat;

						glMatrixMode(GL_MODELVIEW);
						glLoadMatrixd(d_modelview_mat.Get());

						inverse_modelview_mat = d_modelview_mat;
						inverse_modelview_mat.Transpose();

						v1 = d_rotation_center;
						v1 = inverse_modelview_mat * v1;
						v1.x *= d_projection_mat.X().x;
						v1.y *= d_projection_mat.Y().y;
						v1.z *= d_projection_mat.Z().z;
						d_projection_mat.Origin(v1);

						glMatrixMode(GL_PROJECTION);
						glLoadMatrixd(d_projection_mat.Get());
					}
				}
			}
		}
		// Set_Pan
		else
		{
			v1 = d_projection_mat.Origin();

			v1.x += static_cast<double>(mouse_offset.x())/-static_cast<double>(d_viewport_center.x());
			v1.y -= static_cast<double>(mouse_offset.y())/static_cast<double>(d_viewport_center.y());

			d_projection_mat.Origin(v1);

			glMatrixMode(GL_PROJECTION);
			glLoadMatrixd(d_projection_mat.Get());
		}

		this->update();

		d_right_mouse_down.setX(static_cast<int>(x_gl));
		d_right_mouse_down.setY(static_cast<int>(y_gl));

		d_right_mouse_down_moving = true;
	}

	if(mouse_buttons == Qt::LeftButton)
	{
		if(d_rubberband_moving)
		{
			d_rubberband_rect.setBottomRight(event->pos());
			d_rubberband->setGeometry(d_rubberband_rect.normalized());
		}
		else
		{			
			d_rubberband_rect.setTopLeft(d_mouse_down);
			d_rubberband_rect.setBottomRight(d_mouse_down);
	
			if (!d_rubberband)
			{
				d_rubberband = new QRubberBand(QRubberBand::Rectangle, this);
			}
	
			d_rubberband->setGeometry(d_rubberband_rect);
			d_rubberband->show();
			
			d_rubberband_moving = true;
		}
	}
}

void TOpenGLWidget::mouseReleaseEvent(
    QMouseEvent 						*event)
{
	QPoint								pos;
	short								x_gl;
	short								y_gl;

	this->makeCurrent();
	
	pos = event->pos();
	
	if(event->button() & Qt::LeftButton)
	{
		if(!d_rubberband_moving)
		{
			x_gl = pos.x();
			y_gl = pos.y();
	
			this->Convert_To_GLViewport(&x_gl,&y_gl);
			this->Process_Mouse_Click(x_gl,y_gl);
		}
		else
		{
			if(d_rubberband)
			{
				d_rubberband->hide();
				this->Zoom_To_Rect(d_rubberband_rect);
				this->update();
			}
		}
	}

	if(event->button() & Qt::RightButton)
	{
		if(!d_right_mouse_down_moving)
		{
			x_gl = pos.x();
			y_gl = pos.y();

			this->Convert_To_GLViewport(&x_gl,&y_gl);
			this->Click_Zoom(x_gl,y_gl);
			this->update();
		}
	}
}

void TOpenGLWidget::keyPressEvent(
    QKeyEvent 							*event)
{
	if(event->key() == Qt::Key_Shift)
	{
		d_shift_key_down = true;
		event->ignore();
	}

	if(event->key() == Qt::Key_Control)
	{
		d_ctrl_key_down = true;
		event->ignore();
	}
}

void TOpenGLWidget::keyReleaseEvent(
    QKeyEvent 							*event)
{
	if(event->key() == Qt::Key_Shift)
	{
		d_shift_key_down = false;
		event->ignore();
	}

	if(event->key() == Qt::Key_Control)
	{
		d_ctrl_key_down = false;
		event->ignore();
	}
}

void TOpenGLWidget::wheelEvent(
	QWheelEvent 						*event)
{
	this->makeCurrent();
	
	this->Mousewheel_Zoom(event->angleDelta().y());
	
	this->update();
}

void TOpenGLWidget::initializeGL(void)
{
	this->initializeOpenGLFunctions();
	this->GLInit();
	
	// This loads the last state of the views.
	// If the rendering context changes this maintains the current view.
	glMatrixMode(GL_MODELVIEW);
	glLoadMatrixd(d_modelview_mat.Get());
	
	glMatrixMode(GL_PROJECTION);
	glLoadMatrixd(d_projection_mat.Get());
	
	glDepthRange(0.0f,1.0f);
	glPixelTransferf(GL_DEPTH_BIAS,0.0f);
	glPixelTransferf(GL_DEPTH_SCALE,1.0f);
	
	d_pan_mode = true;
	d_rotate_2d_mode = false;
	d_rotate_3d_mode = false;
}

void TOpenGLWidget::resizeGL(
	int 								width,
	int 								height)
{
	int									viewport[4];
	TMat4								projection_mat;
	TVector3							projection_x,projection_y;
	double								viewport_scale_x(1.0);
	double								viewport_scale_y(1.0);
	
	//	this->makeCurrent();
	this->GLResize(width,height);
	
	if(width == 0 || height == 0)
	{
		return;
	}

	viewport[0] = 0;
	viewport[1] = 0;
	viewport[2] = width;
	viewport[3] = height;
	
	d_viewport_center.setX(viewport[2]/2);
	d_viewport_center.setY(viewport[3]/2);
	
	glViewport(viewport[0],viewport[1],viewport[2],viewport[3]);
	
	// load and adjust projection matrix
	glGetDoublev(GL_PROJECTION_MATRIX,projection_mat.Get());
	
	projection_x = projection_mat.X();
	projection_y = projection_mat.Y();
	
	if(viewport[2] != 0 && viewport[3] != 0)
	{
		viewport_scale_x = static_cast<double>(viewport[2]) / 2.0;
		viewport_scale_y = static_cast<double>(viewport[3]) / 2.0;
		
		// find ratio
		if(fabs(viewport_scale_x) > fabs(viewport_scale_y))
		{
			viewport_scale_x = viewport_scale_y / viewport_scale_x;
			
			// use Y axis projection size as reference
			projection_x.x = projection_y.y * viewport_scale_x;
		}
		else
		{
			viewport_scale_y = viewport_scale_x / viewport_scale_y;
			
			// use X axis projection size as reference
			projection_y.y = projection_x.x * viewport_scale_y;

		}
	}
	
	projection_mat.X(projection_x);
	projection_mat.Y(projection_y);

	glMatrixMode(GL_PROJECTION);
	glLoadMatrixd(projection_mat.Get());
}

void TOpenGLWidget::paintGL(void)
{
	if(d_scale_to_fit_render_mode)
	{
		this->Begin_Fit_To_Viewport();
		this->GLRender();
		this->End_Fit_To_Viewport();
		
		d_scale_to_fit_render_mode = false;
	}
	
	this->GLRender();
}

TOpenGLWidget::TSelectionItemType TOpenGLWidget::Selection_Item(
	const int 							offset) const
{
	assert(!(offset < 0));
	assert(offset < static_cast<int>(d_SelectionItems.size()));
	
	return d_SelectionItems[offset];
}

int TOpenGLWidget::Find_Selection(
	const TVector3 						&pick_point) const
{
	TVector3							MinPt;
	TVector3							MaxPt;
	int									SelectionItemCntr;
	int									selection_item;
	double								selection_item_distance(0);
	double								length;
	TVector3							center;
	TVector3							axis;
	TVector3							axis_perp;
	bool								init;
	std::vector<TOpenGLWidget::TSelectionItemType>::const_iterator iter;
	
	selection_item = -1;
	init = true;
	
	for(iter = d_SelectionItems.begin(), SelectionItemCntr = 0;iter != d_SelectionItems.end();++iter,++SelectionItemCntr)
	{
		if((*iter).selection_mode == TOpenGLWidget::SELECTION_LINE)
		{
			axis = (*iter).max_corner - (*iter).min_corner;
			
			if(this->Is_Point_Between(pick_point,(*iter).min_corner,(*iter).max_corner,axis))
			{				
				center = ((*iter).min_corner + (*iter).max_corner) / 2.0;

				if(axis.Length() > ZERO_EPSILON)
				{
					axis.Normal();
					
					axis_perp = pick_point - center;
					axis_perp = axis_perp * axis;
					
					length = axis_perp.Length();
					
					if(init)
					{
						selection_item_distance = length;
						selection_item = SelectionItemCntr;
						
						init = false;
					}
					else
					{
						if(length < selection_item_distance)
						{
							selection_item_distance = length;
							selection_item = SelectionItemCntr;
						}
					}
				}
			}
		}
		else if((*iter).selection_mode == TOpenGLWidget::SELECTION_BOX)
		{
			if(Is_Point_Between(pick_point,(*iter).min_corner,(*iter).max_corner,(*iter).axis_1))
			{
				if(Is_Point_Between(pick_point,(*iter).min_corner,(*iter).max_corner,(*iter).axis_2))
				{
					if(Is_Point_Between(pick_point,(*iter).min_corner,(*iter).max_corner,(*iter).axis_3))
					{
						selection_item = SelectionItemCntr;
						break;
					}
				}
			}
		}
	}
	
	return selection_item;
}

void TOpenGLWidget::Rotate_Model(
	const TVector3						&axis,
	const double						&angle)
{
	TMat4								modelview_mat;
	TVector3							axis_normal;
	
	axis_normal = axis;
	
	if(axis_normal.Length() < ZERO_EPSILON)
	{
		return;
	}
	
	axis_normal.Normal();
	
	this->makeCurrent();
	
	glGetDoublev(GL_MODELVIEW_MATRIX,modelview_mat.Get());
	
	modelview_mat.Rotate(axis_normal,angle);
	
	glMatrixMode(GL_MODELVIEW);
	glLoadMatrixd(modelview_mat.Get());
	d_modelview_mat = modelview_mat;
	
	this->update();
}

void TOpenGLWidget::Point_Raster_Position(
	int									* const x,
	int									* const y,
	const TVector3						&point)
{
	int                                	viewport[4];
	TMat4                               projection_mat;
	TMat4                               modelview_mat;
	double								offset_x;
	double								offset_y;
	TVector3                            origin;
	TVector3							pt;
	
	glGetDoublev(GL_PROJECTION_MATRIX,projection_mat.Get());
	glGetDoublev(GL_MODELVIEW_MATRIX,modelview_mat.Get());
	glGetIntegerv(GL_VIEWPORT,viewport);
	
	modelview_mat.Transpose();
	
	pt = modelview_mat * point;	// convert to screen coordinates

	origin = projection_mat.Origin();
	
	offset_x = pt.x * projection_mat.X().x;
	offset_y = pt.y * projection_mat.Y().y;
	
	*x = static_cast<int>((1 + origin.x + offset_x) * static_cast<double>(viewport[2] / 2));
	*y = static_cast<int>((1 + origin.y + offset_y) * static_cast<double>(viewport[3] / 2));

	// Convert XY opengl Window to XY Window
	*x = *x - viewport[0];
	*y = viewport[3] - *y + viewport[1];
}

void TOpenGLWidget::AddSelectionItem(
	const TOpenGLWidget::TSelectionItemType &t)
{
	TOpenGLWidget::TSelectionItemType	selected_item;
	TVector3							ref_axis;
	
	selected_item.selection_id = t.selection_id;
	selected_item.selection_mode = t.selection_mode;

	if(t.selection_mode == TOpenGLWidget::SELECTION_BOX)
	{
		selected_item.min_corner = t.min_corner;
		selected_item.max_corner = t.max_corner;
		
		ref_axis = selected_item.max_corner - selected_item.min_corner;
		
		if(ref_axis.Length() > ZERO_EPSILON)
		{
			ref_axis.Normal();
			
			// check test axis
			if(t.axis_1.Length() > ZERO_EPSILON && t.axis_2.Length() > ZERO_EPSILON && t.axis_3.Length() > ZERO_EPSILON)
			{
				if((t.axis_1 ^ ref_axis) > 0)
				{
					selected_item.axis_1 = t.axis_1;
				}
				else
				{
					selected_item.axis_1 = -1 * t.axis_1;
				}
				
				if((t.axis_2 ^ ref_axis) > 0)
				{
					selected_item.axis_2 = t.axis_2;
				}
				else
				{
					selected_item.axis_2 = -1 * t.axis_2;
				}
				
				if((t.axis_3 ^ ref_axis) > 0)
				{
					selected_item.axis_3 = t.axis_3;
				}
				else
				{
					selected_item.axis_3 = -1 * t.axis_3;
				}
				
				selected_item.axis_1.Normal();
				selected_item.axis_2.Normal();
				selected_item.axis_3.Normal();
			}
			else
			{
				selected_item.axis_1.Set(1,0,0);
				selected_item.axis_2.Set(0,1,0);
				selected_item.axis_3.Set(0,0,1);
			}
		}
		else
		{
			selected_item.axis_1.Set(1,0,0);
			selected_item.axis_2.Set(0,1,0);
			selected_item.axis_3.Set(0,0,1);
		}
	}
	else if(t.selection_mode == TOpenGLWidget::SELECTION_LINE)
	{
		selected_item.min_corner = t.min_corner;
		selected_item.max_corner = t.max_corner;
	}
	else
	{
		return;
	}
	
	d_SelectionItems.push_back(selected_item);
}

bool TOpenGLWidget::Is_Point_Between(
	const TVector3						&pnt,
	const TVector3						&start,
	const TVector3						&end,
	const TVector3						&axis) const
{
	double								d1,d2;
	TVector3							v1,v2;
	
	v1 = pnt - start;
	d1 = axis.i * v1.x + axis.j * v1.y + axis.k * v1.z;
	
	v2 = pnt - end;
	d2 = axis.i * v2.x + axis.j * v2.y + axis.k * v2.z;
	
	if(d1 > 0 && d2 < 0)
	{
		return true;
	}
	
	if(d2 > 0 && d1 < 0)
	{
		return true;
	}
	return false;
}

void TOpenGLWidget::Convert_To_GLViewport(
    short								*x,
    short 								*y)
{
	int									viewport[4];

	// GLViewport index [0],[1] define the upper left corner of the viewport relative to the host window
	// GLViewport index [2],[3] define the length and width of the viewport
	// The Y axis and starting point is lower left and goes positive to the top of the window
	glGetIntegerv(GL_VIEWPORT,viewport);

	// Convert XY windows to XY opengl
	*x = *x - viewport[0];
	*y = viewport[3] - *y + viewport[1];
}

void TOpenGLWidget::Convert_GLViewport_ToModel(
    const short                        x_gl,
    const short                        y_gl,
    double                             *x,
    double                             *y)
{
	int                                	viewport[4];
	TMat4                               projection_mat;
	TMat4                               modelview_mat;
	int                                	viewport_x_zero;
	int									viewport_y_zero;
	double								range_x;
	double								range_y;
	TVector3                            origin;

	glGetDoublev(GL_PROJECTION_MATRIX,projection_mat.Get());
	glGetDoublev(GL_MODELVIEW_MATRIX,modelview_mat.Get());
	glGetIntegerv(GL_VIEWPORT,viewport);

	// Find coordinates relative to model zero
	origin = projection_mat.Origin();

	viewport_x_zero = static_cast<int>((1 + origin.x) * static_cast<double>(viewport[2] / 2));
	viewport_y_zero = static_cast<int>((1 + origin.y) * static_cast<double>(viewport[3] / 2));

	// Returns range of viewport in model units
	range_x = 1 / projection_mat.X().x;
	range_y = 1 / projection_mat.Y().y;

	// Adjust Screen units to model units relative to zero
	*x = (static_cast<double>(x_gl - viewport_x_zero) * range_x) / static_cast<double>((viewport[2])/2);
	*y = (static_cast<double>(y_gl - viewport_y_zero) * range_y) / static_cast<double>((viewport[3])/2);
}

void TOpenGLWidget::Process_Mouse_Click(
	const short 						x_gl,
	const short 						y_gl)
{
	TMat4								modelview_mat;
	TMat4								projection_mat;
	TVector3							origin;
	TVector3							point;
	TVector3							model_point;
	int									viewport[4];
	int									viewport_x_zero;
	int									viewport_y_zero;
	int									selection_index;
	double								range_x;
	double								range_y;
	double								depth;
	static const double					BACKWALL_DEPTH(0.99);

	this->makeCurrent();
	glGetDoublev(GL_PROJECTION_MATRIX,projection_mat.Get());
	glGetDoublev(GL_MODELVIEW_MATRIX,modelview_mat.Get());

	// Convert XY screen coordinates to model coordinates relative to model zero

	glGetIntegerv(GL_VIEWPORT,viewport);

	depth = this->Get_GLPixel_Depth(x_gl,y_gl);

	if(depth > BACKWALL_DEPTH)
	{
		return;
	}
	
	depth -= (projection_mat.Origin().z)/2;
	depth = (depth - 0.5);
	depth /= (projection_mat.Z().z/2);
	
	// Find coordinates relative to model zero
	origin = projection_mat.Origin();

	viewport_x_zero = static_cast<int>((1 + origin.x) * static_cast<double>(viewport[2] / 2));
	viewport_y_zero = static_cast<int>((1 + origin.y) * static_cast<double>(viewport[3] / 2));

	// Returns range of viewport in model units
	range_x = 1 / projection_mat.X().x;
	range_y = 1 / projection_mat.Y().y;

	// Adjust Screen units to model units relative to zero
	point.x = (static_cast<double>(x_gl - viewport_x_zero) * range_x) / static_cast<double>((viewport[2])/2);
	point.y = (static_cast<double>(y_gl - viewport_y_zero) * range_y) / static_cast<double>((viewport[3])/2);
	point.z = depth;
	
	model_point = modelview_mat * point;
	
	selection_index = this->Find_Selection(model_point);
	
	if(selection_index < 0)
	{
		emit Model_Click_Point(model_point.x,model_point.y,model_point.z,static_cast<int>(-1));
	}
	else
	{
		emit Model_Click_Point(model_point.x,model_point.y,model_point.z,this->Selection_Item(selection_index).selection_id);
	}
}

void TOpenGLWidget::Set_Rotation_Center(void)
{
	TMat4								modelview_mat;
	TMat4								projection_mat;
	TVector3							vec;
	TVector3							pnt;
	int									viewport[4];
	double								depth;

	projection_mat = d_projection_mat;
	modelview_mat = d_modelview_mat;

	vec = projection_mat.X();
	pnt.x = projection_mat.Origin().x / vec.x;

	vec = projection_mat.Y();
	pnt.y = projection_mat.Origin().y / vec.y;

	vec = projection_mat.Z();
	pnt.z = projection_mat.Origin().z / vec.z;

	// Update Z value with image center of viewport
	glGetIntegerv(GL_VIEWPORT,viewport);

	depth = Get_GLPixel_Depth(viewport[2]/2,viewport[3]/2);

	if(!(depth > 0.99))
	{
		depth -= (projection_mat.Origin().z)/2;
		depth = (0.5 - depth);
		depth /= (projection_mat.Z().z/2);

		pnt.z = depth * (1 - projection_mat.Perspective());
	}

	pnt = modelview_mat * pnt;

	d_rotation_center = pnt;
}

double TOpenGLWidget::Get_GLPixel_Depth(
    const short							x_gl,
    const short							y_gl)
{
	int									viewport[4];
	GLfloat								depth;
	int									XPixel;
	int									YPixel;
	int									AreaSideLength;
	int									XAreaStart;
	int									YAreaStart;
	GLfloat								*AreaPoints;
	int									Cntr;
	int									AreaHitCntr;

	glGetIntegerv(GL_VIEWPORT,viewport);

	XPixel = x_gl + viewport[0];
	YPixel = y_gl + viewport[1];

	assert(XPixel < viewport[2]);
	assert(YPixel < viewport[3]);
	assert(XPixel >= 0);
	assert(YPixel >= 0);

	if(XPixel >= viewport[2] ||
	        YPixel >= viewport[3] ||
	        XPixel < 0 ||
	        YPixel < 0)
	{
		return 1;
	}

	glReadBuffer(GL_FRONT);
	glReadPixels(XPixel,YPixel,1,1,GL_DEPTH_COMPONENT,GL_FLOAT,&depth);

	if(depth > 1.0f)
	{
		depth = 1;
	}
	
	if(depth > 0.99)
	{
		// Checking for average depth in area of click
		// Using the smaller side.  Will always fit inside viewport
		if(viewport[2] < viewport[3])
		{
			AreaSideLength = viewport[2]/5;
		}
		else
		{
			AreaSideLength = viewport[3]/5;
		}

		if(AreaSideLength > 1)
		{
			XAreaStart = XPixel - AreaSideLength/2;
			YAreaStart = YPixel - AreaSideLength/2;

			// Make sure start and end points in inside viewport
			if(XAreaStart < 0)
			{
				XAreaStart -= XAreaStart;
			}
			else if(XAreaStart + AreaSideLength > viewport[2])
			{
				XAreaStart -= (XAreaStart + AreaSideLength - viewport[2]);
			}

			if(YAreaStart < 0)
			{
				YAreaStart-=YAreaStart;
			}
			else if(YAreaStart + AreaSideLength > viewport[3])
			{
				YAreaStart -= (YAreaStart + AreaSideLength - viewport[3]);
			}

			AreaPoints = new GLfloat[AreaSideLength * AreaSideLength];

			glReadBuffer(GL_FRONT);
			glReadPixels(XAreaStart,YAreaStart,AreaSideLength,AreaSideLength,GL_DEPTH_COMPONENT,GL_FLOAT,AreaPoints);

			AreaHitCntr = 0;
			depth = 0.0;
			for(Cntr = 0;Cntr < AreaSideLength * AreaSideLength;Cntr++)
			{
				if(!(*(AreaPoints+Cntr) > 0.99f))
				{
					depth += (*(AreaPoints+Cntr));
					AreaHitCntr++;
				}
			}

			if(AreaHitCntr)
			{
				depth /= static_cast<double>(AreaHitCntr);
			}
			else
			{
				depth = 1.0f;
			}

			delete[] AreaPoints;
		}
	}

	return static_cast<double>(depth);
}

void TOpenGLWidget::Zoom_To_Rect(
    const QRect							&rect)
{
	int									viewport[4];
	short								rect_frame[4];
	double								model_frame[4];
	double								range_x;
	double								range_y;
	double								max_range_xy;
	TVector3							origin;
	double								viewport_scale_x(1.0);
	double								viewport_scale_y(1.0);

	rect_frame[0] = static_cast<short>(rect.left());
	rect_frame[1] = static_cast<short>(rect.top());
	rect_frame[2] = static_cast<short>(rect.right());
	rect_frame[3] = static_cast<short>(rect.bottom());

	this->Convert_To_GLViewport(&rect_frame[0],&rect_frame[1]);
	this->Convert_To_GLViewport(&rect_frame[2],&rect_frame[3]);

	this->Convert_GLViewport_ToModel(rect_frame[0],rect_frame[1],&model_frame[0],&model_frame[1]);
	this->Convert_GLViewport_ToModel(rect_frame[2],rect_frame[3],&model_frame[2],&model_frame[3]);

	// swap values if inverted
	if(model_frame[0] > model_frame[2])
	{
		max_range_xy = model_frame[0];
		model_frame[0] = model_frame[2];
		model_frame[2] = max_range_xy;
	}

	if(model_frame[1] > model_frame[3])
	{
		max_range_xy = model_frame[1];
		model_frame[1] = model_frame[3];
		model_frame[3] = max_range_xy;
	}
	
	if((model_frame[2] - model_frame[0]) < ZERO_EPSILON)
	{
		return;
	}
	
	if((model_frame[3] - model_frame[1]) < ZERO_EPSILON)
	{
		return;
	}

	this->makeCurrent();

	// Rewrite projection matrix based on Selection box dimensions
	glGetDoublev(GL_PROJECTION_MATRIX,d_projection_mat.Get());
	glGetIntegerv(GL_VIEWPORT,viewport);

	origin.x = -1 * (model_frame[2] + model_frame[0]);
	origin.y = -1 * (model_frame[3] + model_frame[1]);
	origin.z = d_projection_mat.Origin().z;

	range_x = model_frame[2] - model_frame[0];
	range_y = model_frame[3] - model_frame[1];

	if(range_x > range_y)
	{
		max_range_xy = range_x;
	}
	else
	{
		max_range_xy = range_y;
	}

	if(max_range_xy < ZERO_EPSILON)
	{
		max_range_xy = ZERO_EPSILON;
	}

	origin.x = (origin.x / max_range_xy);
	origin.y = (origin.y / max_range_xy);
	
	// find scaling values for viewport dimensions
	// offset origin
	if(viewport[2] != 0 && viewport[3] != 0)
	{
		viewport_scale_x = static_cast<double>(viewport[2]) / 2.0;
		viewport_scale_y = static_cast<double>(viewport[3]) / 2.0;
		
		// find ratio
		if(fabs(viewport_scale_x) > fabs(viewport_scale_y))
		{
			viewport_scale_x = viewport_scale_y / viewport_scale_x;
			viewport_scale_y = 1.0;
			
			origin.x *= viewport_scale_x/viewport_scale_y;
		}
		else
		{
			viewport_scale_y = viewport_scale_x / viewport_scale_y;
			viewport_scale_x = 1.0;
			
			origin.y *= viewport_scale_y/viewport_scale_x;
		}
	}

	if(max_range_xy > ZERO_EPSILON)
	{
		max_range_xy = 2 / max_range_xy;
	}
	else
	{
		max_range_xy = ZERO_EPSILON;
	}

	d_projection_mat.X(TVector3(max_range_xy * viewport_scale_x,0,0));
	d_projection_mat.Y(TVector3(0,max_range_xy * viewport_scale_y,0));
	d_projection_mat.Origin(TVector3(origin.x,origin.y,origin.z));

	glMatrixMode(GL_PROJECTION);
	glLoadMatrixd(d_projection_mat.Get());
}

void TOpenGLWidget::Click_Zoom(
    const short							x_gl,
    const short							y_gl)
{
	Q_UNUSED(x_gl);

	TVector3							modelview_range;
	double								z_buffer_range;
	double								scale;
	int									viewport[4];
	int									viewport_y_center;

	modelview_range = d_max_modelview_corner - d_min_modelview_corner;
	
	// minimum range check
	if(modelview_range.z < ZERO_EPSILON)
	{
		modelview_range.z = ZERO_EPSILON;
	}
	
	z_buffer_range = modelview_range.z;

	z_buffer_range = 1.0 / z_buffer_range;	// range is 0..1
	z_buffer_range = -z_buffer_range / Z_DEPTH_DIVISOR;

	this->makeCurrent();
	glGetIntegerv(GL_VIEWPORT,viewport);

	viewport_y_center = viewport[3]/2;

	if(y_gl == viewport_y_center)
	{
		return;
	}

	scale =  viewport_y_center - y_gl;

	scale = (static_cast<double>(viewport_y_center) + scale/2) / static_cast<double>(viewport_y_center);

	glGetDoublev(GL_PROJECTION_MATRIX,d_projection_mat.Get());

	d_projection_mat.X(d_projection_mat.X() * scale);
	d_projection_mat.Y(d_projection_mat.Y() * scale);
//	d_projection_mat.Z(d_projection_mat.Z() * scale);	// causing scaling issue without resetting
	d_projection_mat.Z(TVector3(0,0,z_buffer_range) * scale);
	d_projection_mat.Origin(d_projection_mat.Origin() * scale);

	glMatrixMode(GL_PROJECTION);
	glLoadMatrixd(d_projection_mat.Get());
}

void TOpenGLWidget::Mousewheel_Zoom(
    const int							Delta)
{
	TVector3							modelview_range;
	double								z_buffer_range;
	double								scale;

	if(Delta == 0)
	{
		return;
	}
	
	if(Delta > 0)
	{
		scale = 1.05263157895;
	}
	else
	{
		scale = 0.95;
	}
	
	modelview_range = d_max_modelview_corner - d_min_modelview_corner;

	// minimum range check
	if(modelview_range.z < ZERO_EPSILON)
	{
		modelview_range.z = ZERO_EPSILON;
	}
	
	z_buffer_range = modelview_range.z;

	z_buffer_range = 1.0 / z_buffer_range;	// range is 0..1
	z_buffer_range = -z_buffer_range / Z_DEPTH_DIVISOR;

	this->makeCurrent();
	glGetDoublev(GL_PROJECTION_MATRIX,d_projection_mat.Get());

	d_projection_mat.X(d_projection_mat.X() * scale);
	d_projection_mat.Y(d_projection_mat.Y() * scale);
//	d_projection_mat.Z(d_projection_mat.Z() * scale);	// causing scaling issue without resetting
	d_projection_mat.Z(TVector3(0,0,z_buffer_range) * scale);
	d_projection_mat.Origin(d_projection_mat.Origin() * scale);

	glMatrixMode(GL_PROJECTION);
	glLoadMatrixd(d_projection_mat.Get());
}

void TOpenGLWidget::Begin_Fit_To_Viewport(void)
{
	d_processing_scale_to_fit = true;
	d_initialize_scale_to_fit = true;

	glGetDoublev(GL_MODELVIEW_MATRIX,d_modelview_mat_transpose.Get());
	d_modelview_mat_transpose.Transpose();
}

void TOpenGLWidget::Process_Vertex(
    const TVector3						&pnt)
{
	TVector3							modelview_point;
	
	modelview_point = d_modelview_mat_transpose * pnt;

	if(d_initialize_scale_to_fit)
	{
		d_min_modelview_corner = d_max_modelview_corner = modelview_point;

		d_initialize_scale_to_fit = false;
	}
	else
	{
		if(modelview_point.x > d_max_modelview_corner.x)
			d_max_modelview_corner.x = modelview_point.x;

		if(modelview_point.x < d_min_modelview_corner.x)
			d_min_modelview_corner.x = modelview_point.x;

		if(modelview_point.y > d_max_modelview_corner.y)
			d_max_modelview_corner.y = modelview_point.y;

		if(modelview_point.y < d_min_modelview_corner.y)
			d_min_modelview_corner.y = modelview_point.y;

		if(modelview_point.z > d_max_modelview_corner.z)
			d_max_modelview_corner.z = modelview_point.z;

		if(modelview_point.z < d_min_modelview_corner.z)
			d_min_modelview_corner.z = modelview_point.z;
	}
}

void TOpenGLWidget::End_Fit_To_Viewport(void)
{
	int									viewport[4];
	TVector3							modelview_range;
	TMat4								projection_mat;
	double								max_range_xy;
	double								z_buffer_range;
	TVector3							origin;
	double								viewport_scale_x(1.0);
	double								viewport_scale_y(1.0);

	d_processing_scale_to_fit = false;

	glGetDoublev(GL_PROJECTION_MATRIX,projection_mat.Get());
	glGetIntegerv(GL_VIEWPORT,viewport);
	
	modelview_range = d_max_modelview_corner - d_min_modelview_corner;
	
	// minimum range check
	if(modelview_range.x < ZERO_EPSILON)
	{
		modelview_range.x = ZERO_EPSILON;
	}
	
	if(modelview_range.y < ZERO_EPSILON)
	{
		modelview_range.y = ZERO_EPSILON;
	}
	
	if(modelview_range.z < ZERO_EPSILON)
	{
		modelview_range.z = ZERO_EPSILON;
	}

	origin.x = -1 * (d_max_modelview_corner.x + d_min_modelview_corner.x);
	origin.y = -1 * (d_max_modelview_corner.y + d_min_modelview_corner.y);
	origin.z = (d_max_modelview_corner.z + d_min_modelview_corner.z) / 2.0;	// use average for Z
	
	if(modelview_range.x > modelview_range.y)
	{
		max_range_xy = modelview_range.x;
	}
	else
	{
		max_range_xy = modelview_range.y;
	}
	
	z_buffer_range = modelview_range.z;
	
	origin.x = (origin.x / max_range_xy);
	origin.y = (origin.y / max_range_xy);
	origin.z = origin.z / (modelview_range.z * Z_DEPTH_DIVISOR);


	// find scaling values for viewport dimensions
	// offset origin
	if(viewport[2] != 0 && viewport[3] != 0)
	{
		viewport_scale_x = static_cast<double>(viewport[2]) / 2.0;
		viewport_scale_y = static_cast<double>(viewport[3]) / 2.0;
		
		// find ratio
		if(fabs(viewport_scale_x) > fabs(viewport_scale_y))
		{
			viewport_scale_x = viewport_scale_y / viewport_scale_x;
			viewport_scale_y = 1.0;
			
			origin.x *= viewport_scale_x/viewport_scale_y;
		}
		else
		{
			viewport_scale_y = viewport_scale_x / viewport_scale_y;
			viewport_scale_x = 1.0;
			
			origin.y *= viewport_scale_y/viewport_scale_x;
		}
	}

	max_range_xy = 2.0 / max_range_xy;		// range is -1..1
	z_buffer_range = 1.0 / z_buffer_range;	// range is 0..1
	
	z_buffer_range = -z_buffer_range / Z_DEPTH_DIVISOR;

	// Setting the depth to X times the max XY range
	// should still give reasonable selection values
	// and not have the image clip front or back.

	projection_mat.X(TVector3(max_range_xy * viewport_scale_x,0,0));
	projection_mat.Y(TVector3(0,max_range_xy * viewport_scale_y,0));
	projection_mat.Z(TVector3(0,0,z_buffer_range));
	projection_mat.Origin(TVector3(origin.x,origin.y,origin.z));

	glMatrixMode(GL_PROJECTION);
	glLoadMatrixd(projection_mat.Get());
}


