/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QListWidget>
#include <QListWidgetItem>
#include <QSpacerItem>
#include <QToolButton>

#include "../../core/xmlfile.h"

#include "artifactfreepointswidget.h"

Q_DECLARE_METATYPE(TArtifactFreePointsWidget::TPointData);

static const double						ZERO_EPSILON(0.001);

TArtifactFreePointsWidget::TArtifactFreePointsWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QGridLayout							*widget_layout;
	QHBoxLayout							*button_hlayout;
	QLabel								*approach_label;
	QLabel								*dir_i_label;
	QLabel								*dir_j_label;
	QLabel								*dir_k_label;
	QLabel								*name_label;
	QLabel								*pos_x_label;
	QLabel								*pos_y_label;
	QLabel								*pos_z_label;
	QLabel								*position_label;
	QSpacerItem							*button_hspacer;

	widget_layout = new QGridLayout(this);
	widget_layout->setSpacing(4);
	widget_layout->setContentsMargins(0,0,0,0);

	name_label = new QLabel(this);
	widget_layout->addWidget(name_label,0,0,1,2);

	d_name_edit = new QLineEdit(this);
	d_name_edit->setAlignment(Qt::AlignCenter);
	widget_layout->addWidget(d_name_edit,0,2,1,1);

	position_label = new QLabel(this);
	widget_layout->addWidget(position_label,1,0,1,1);

	pos_x_label = new QLabel(this);
	pos_x_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	widget_layout->addWidget(pos_x_label,1,1,1,1);

	d_pos_x_edit = new QLineEdit(this);
	d_pos_x_edit->setAlignment(Qt::AlignCenter);
	widget_layout->addWidget(d_pos_x_edit,1,2,1,1);

	pos_y_label = new QLabel(this);
	pos_y_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	widget_layout->addWidget(pos_y_label,2,1,1,1);

	d_pos_y_edit = new QLineEdit(this);
	d_pos_y_edit->setAlignment(Qt::AlignCenter);
	widget_layout->addWidget(d_pos_y_edit,2,2,1,1);

	pos_z_label = new QLabel(this);
	pos_z_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	widget_layout->addWidget(pos_z_label,3,1,1,1);

	d_pos_z_edit = new QLineEdit(this);
	d_pos_z_edit->setAlignment(Qt::AlignCenter);
	widget_layout->addWidget(d_pos_z_edit,3,2,1,1);

	approach_label = new QLabel(this);
	widget_layout->addWidget(approach_label,4,0,1,1);

	dir_i_label = new QLabel(this);
	dir_i_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	widget_layout->addWidget(dir_i_label,4,1,1,1);

	d_dir_i_edit = new QLineEdit(this);
	d_dir_i_edit->setAlignment(Qt::AlignCenter);
	widget_layout->addWidget(d_dir_i_edit,4,2,1,1);

	dir_j_label = new QLabel(this);
	dir_j_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	widget_layout->addWidget(dir_j_label,5,1,1,1);

	d_dir_j_edit = new QLineEdit(this);
	d_dir_j_edit->setAlignment(Qt::AlignCenter);
	widget_layout->addWidget(d_dir_j_edit,5,2,1,1);

	dir_k_label = new QLabel(this);
	dir_k_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	widget_layout->addWidget(dir_k_label,6,1,1,1);

	d_dir_k_edit = new QLineEdit(this);
	d_dir_k_edit->setAlignment(Qt::AlignCenter);
	widget_layout->addWidget(d_dir_k_edit,6,2,1,1);

	button_hlayout = new QHBoxLayout();

	button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_hlayout->addItem(button_hspacer);

	d_add_button = new QToolButton(this);
	d_add_button->setFixedSize(16,16);
	button_hlayout->addWidget(d_add_button);

	d_remove_button = new QToolButton(this);
	d_remove_button->setFixedSize(16,16);
	button_hlayout->addWidget(d_remove_button);

	d_send_button = new QToolButton(this);
	d_send_button->setFixedSize(16,16);
	button_hlayout->addWidget(d_send_button);
	widget_layout->addLayout(button_hlayout,7,0,1,3);

	d_points_list = new QListWidget(this);
	widget_layout->addWidget(d_points_list,8,0,1,3);

	QWidget::setTabOrder(d_name_edit,d_pos_x_edit);
	QWidget::setTabOrder(d_pos_x_edit,d_pos_y_edit);
	QWidget::setTabOrder(d_pos_y_edit,d_pos_z_edit);
	QWidget::setTabOrder(d_pos_z_edit,d_dir_i_edit);
	QWidget::setTabOrder(d_dir_i_edit,d_dir_j_edit);
	QWidget::setTabOrder(d_dir_j_edit,d_dir_k_edit);
	QWidget::setTabOrder(d_dir_k_edit,d_send_button);
	QWidget::setTabOrder(d_send_button,d_remove_button);
	QWidget::setTabOrder(d_remove_button,d_add_button);
	QWidget::setTabOrder(d_add_button,d_points_list);
	
	d_add_button->setIcon(QIcon(":/gui/add_button.png"));
	d_remove_button->setIcon(QIcon(":/gui/remove_button.png"));
	d_send_button->setIcon(QIcon(":/gui/send_button.png"));

	name_label->setText(QStringLiteral("Name:"));
	position_label->setText(QStringLiteral("Position"));
	pos_x_label->setText(QStringLiteral("X:"));
	pos_y_label->setText(QStringLiteral("Y:"));
	pos_z_label->setText(QStringLiteral("Z:"));
	approach_label->setText(QStringLiteral("Approach:"));
	dir_i_label->setText(QStringLiteral("I:"));
	dir_j_label->setText(QStringLiteral("J:"));
	dir_k_label->setText(QStringLiteral("K:"));
	
	connect(d_add_button,&QToolButton::clicked,this,&TArtifactFreePointsWidget::Add_Touch);
	connect(d_remove_button,&QToolButton::clicked,this,&TArtifactFreePointsWidget::Delete_Touch_Points);
	connect(d_send_button,&QToolButton::clicked,this,&TArtifactFreePointsWidget::Send_Touch);
	
	connect(d_points_list,&QListWidget::itemSelectionChanged,this,&TArtifactFreePointsWidget::Touch_Selection_Changed);
	connect(d_points_list,&QListWidget::itemDoubleClicked,this,&TArtifactFreePointsWidget::Touch_Double_Clicked);
}

TArtifactFreePointsWidget::~TArtifactFreePointsWidget(void)
{
}

bool TArtifactFreePointsWidget::Save_Touch_Points(
	const QString						&data_path) const
{
	QString								file_name;

	file_name = data_path;
	file_name.replace('\\','/');
	
	if(file_name.endsWith('/'))
	{
		file_name.append("manual_points.xml");
	}
	else
	{
		file_name.append("/manual_points.xml");
	}
	
	return this->Save_Data(file_name);
}

bool TArtifactFreePointsWidget::Load_Touch_Points(
	const QString						&data_path)
{
	QString								file_name;
	
	file_name = data_path;
	file_name.replace('\\','/');
	
	if(file_name.endsWith('/'))
	{
		file_name.append("manual_points.xml");
	}
	else
	{
		file_name.append("/manual_points.xml");
	}
	
	return this->Load_Data(file_name);
}

void TArtifactFreePointsWidget::Description_Changed(
	const QString						&text)
{
	d_add_button->setEnabled(false);
	
	if(text.length())
	{
		d_add_button->setEnabled(true);
	}
}

void TArtifactFreePointsWidget::Add_Touch(void)
{
	QVariant							variant;
	QListWidgetItem						*item;
	TArtifactFreePointsWidget::TPointData point_data;
	
	point_data.description = d_name_edit->text();
	
	point_data.xyz.x = d_pos_x_edit->text().toDouble();
	point_data.xyz.y = d_pos_y_edit->text().toDouble();
	point_data.xyz.z = d_pos_z_edit->text().toDouble();
	
	point_data.ijk.x = d_dir_i_edit->text().toDouble();
	point_data.ijk.y = d_dir_j_edit->text().toDouble();
	point_data.ijk.z = d_dir_k_edit->text().toDouble();
	
	if(point_data.ijk.Length() < ZERO_EPSILON)
	{
		point_data.ijk.Set(0,0,1);
	}
	
	point_data.ijk.Normal();
	
	item = new QListWidgetItem(d_points_list);
	
	variant.setValue(point_data);
	
	item->setData(Qt::UserRole,variant);
	item->setText(QString("%1  %2,%3,%4  %5,%6,%7")
				  .arg(point_data.description)
				  .arg(point_data.xyz.x,0,'f',1)
				  .arg(point_data.xyz.y,0,'f',1)
				  .arg(point_data.xyz.z,0,'f',1)
				  .arg(point_data.ijk.x,0,'f',2)
				  .arg(point_data.ijk.y,0,'f',2)
				  .arg(point_data.ijk.z,0,'f',2));
	
}

void TArtifactFreePointsWidget::Delete_Touch_Points(void)
{
	QList<QListWidgetItem*>				sel_items;
	QList<QListWidgetItem*>::iterator	iter;
	int									row;
	bool								prev_state;
	
	prev_state = d_points_list->blockSignals(true);

	sel_items = d_points_list->selectedItems();
	
	for(iter = sel_items.begin();iter != sel_items.end();++iter)
	{
		row = d_points_list->row(*iter);
		d_points_list->takeItem(row);
		
		delete (*iter);
	}
	
	d_points_list->blockSignals(prev_state);
}

void TArtifactFreePointsWidget::Send_Touch(void)
{
	TVector3							xyz;
	TVector3							ijk;
	
	xyz.x = d_pos_x_edit->text().toDouble();
	xyz.y = d_pos_y_edit->text().toDouble();
	xyz.z = d_pos_z_edit->text().toDouble();

	ijk.i = d_dir_i_edit->text().toDouble();
	ijk.j = d_dir_j_edit->text().toDouble();
	ijk.k = d_dir_k_edit->text().toDouble();
	
	emit Touch_Point(xyz.x,xyz.y,xyz.z,ijk.i,ijk.j,ijk.k);
}

void TArtifactFreePointsWidget::Touch_Selection_Changed(void)
{
	QList<QListWidgetItem*>				sel_items;
	QVariant							variant;
	TArtifactFreePointsWidget::TPointData point_data;
	bool								prev_state;
	int									selected_item_count;
	
	sel_items = d_points_list->selectedItems();
	d_remove_button->setEnabled(sel_items.size() > 0);
	
	d_send_button->setEnabled(false);
	
	selected_item_count = sel_items.size();
	
	if(selected_item_count == 0)
	{
		d_name_edit->setEnabled(true);
		
		d_pos_x_edit->setEnabled(true);
		d_pos_y_edit->setEnabled(true);
		d_pos_z_edit->setEnabled(true);
		
		d_dir_i_edit->setEnabled(true);
		d_dir_j_edit->setEnabled(true);
		d_dir_k_edit->setEnabled(true);
	}
	else if(selected_item_count == 1)
	{
		d_name_edit->setEnabled(true);
		
		d_pos_x_edit->setEnabled(true);
		d_pos_y_edit->setEnabled(true);
		d_pos_z_edit->setEnabled(true);
		
		d_dir_i_edit->setEnabled(true);
		d_dir_j_edit->setEnabled(true);
		d_dir_k_edit->setEnabled(true);

		variant = sel_items[0]->data(Qt::UserRole);
		point_data = qvariant_cast<TArtifactFreePointsWidget::TPointData>(variant);
		
		prev_state = d_name_edit->blockSignals(true);
		
		d_name_edit->setText(point_data.description);
		
		d_pos_x_edit->setText(QString("%1").arg(point_data.xyz.x,0,'f',4));
		d_pos_y_edit->setText(QString("%1").arg(point_data.xyz.y,0,'f',4));
		d_pos_z_edit->setText(QString("%1").arg(point_data.xyz.z,0,'f',4));
		
		d_dir_i_edit->setText(QString("%1").arg(point_data.ijk.x,0,'f',6));
		d_dir_j_edit->setText(QString("%1").arg(point_data.ijk.y,0,'f',6));
		d_dir_k_edit->setText(QString("%1").arg(point_data.ijk.z,0,'f',6));
		
		d_name_edit->blockSignals(prev_state);
		
		d_add_button->setEnabled(true);
		d_send_button->setEnabled(true);
	}
	else
	{
		d_add_button->setEnabled(false);
		d_send_button->setEnabled(false);
		
		prev_state = d_name_edit->blockSignals(true);
		d_name_edit->setText(QString());
		d_name_edit->blockSignals(prev_state);

		d_pos_x_edit->setText(QString());
		d_pos_y_edit->setText(QString());
		d_pos_z_edit->setText(QString());
		
		d_dir_i_edit->setText(QString());
		d_dir_j_edit->setText(QString());
		d_dir_k_edit->setText(QString());
		
		d_name_edit->setEnabled(false);
		
		d_pos_x_edit->setEnabled(false);
		d_pos_y_edit->setEnabled(false);
		d_pos_z_edit->setEnabled(false);

		d_dir_i_edit->setEnabled(false);
		d_dir_j_edit->setEnabled(false);
		d_dir_k_edit->setEnabled(false);
	}
}

void TArtifactFreePointsWidget::Touch_Double_Clicked(
	QListWidgetItem						*item)
{
	QVariant							variant;
	TArtifactFreePointsWidget::TPointData point_data;
	
	variant = item->data(Qt::UserRole);
	point_data = qvariant_cast<TArtifactFreePointsWidget::TPointData>(variant);
		
	emit Touch_Point(point_data.xyz.x,point_data.xyz.y,point_data.xyz.z,point_data.ijk.i,point_data.ijk.j,point_data.ijk.k);
}

bool TArtifactFreePointsWidget::Save_Data(
	const QString						&file_name) const
{
	TXmlFile							xml_file;
	QDomElement							e0;
	QDomElement							e1;
	QListWidgetItem						*item;
	TArtifactFreePointsWidget::TPointData point_data;
	int									cntr;
	
	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Reset("Manual_Points",1);
		
		for(cntr = 0;cntr < d_points_list->count();++cntr)
		{
			item = d_points_list->item(cntr);
			
			point_data = qvariant_cast<TArtifactFreePointsWidget::TPointData>(item->data(Qt::UserRole));
			
			e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Point"));
			
			e1 = xml_file.Create_Node(&e0,QStringLiteral("Description"));
			xml_file.Write_Text_Node(&e1,point_data.description);
			
			e1 = xml_file.Create_Node(&e0,QStringLiteral("XYZ"));
			xml_file.Write_Text_Node(&e1,QString("%1,%2,%3").arg(point_data.xyz.x,0,'f',5).arg(point_data.xyz.y,0,'f',5).arg(point_data.xyz.z,0,'f',5));
			
			e1 = xml_file.Create_Node(&e0,QStringLiteral("IJK"));
			xml_file.Write_Text_Node(&e1,QString("%1,%2,%3").arg(point_data.ijk.i,0,'f',9).arg(point_data.ijk.j,0,'f',9).arg(point_data.ijk.k,0,'f',9));
		}
		
		xml_file.Write_File();
	}
	catch(TXmlFileException exception)
	{
		return false;
	}
	
	return true;
}

bool TArtifactFreePointsWidget::Load_Data(
	const QString						&file_name)
{
	TXmlFile							xml_file;
	QDomElement							element;
	int									version;
	QString								text;
		
	d_points_list->clear();
	
	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Open("Manual_Points",&version);
		
		switch(version)
		{
			case 1:
				this->Load_Data_V1(&xml_file);
				break;
				
			default:
				throw TXmlFileException(QStringLiteral("ERR:  File version not recognized"));
				break;
		}
	}
	catch(TXmlFileException exception)
	{
	}
	
	return true;
}

void TArtifactFreePointsWidget::Load_Data_V1(
	TXmlFile							* const xml_file)
{
	QDomElement							element;
	QDomElement							e0;
	QDomElement							e1;
	QString								text;
	QStringList							list;
	TArtifactFreePointsWidget::TPointData point_data;
	QListWidgetItem						*item;
	QVariant							variant;
	
	assert(xml_file);
	
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Point"),true);
	
	while(!e0.isNull())
	{
		
		e1 = xml_file->Get_Node(&e0,QStringLiteral("Description"));
		point_data.description = xml_file->Read_Text_Node(&e1);
		
		e1 = xml_file->Get_Node(&e0,QStringLiteral("XYZ"));
		text = xml_file->Read_Text_Node(&e1);
		list = text.split(',');
		
		if(list.size() == 3)
		{
			point_data.xyz.x = list[0].toDouble();
			point_data.xyz.y = list[1].toDouble();
			point_data.xyz.z = list[2].toDouble();
		}
		else
		{
			throw TXmlFileException(QStringLiteral("ERR:  XYZ must have three values"));
		}
		
		e1 = xml_file->Get_Node(&e0,QStringLiteral("IJK"));
		text = xml_file->Read_Text_Node(&e1);
		list = text.split(',');
		
		if(list.size() == 3)
		{
			point_data.ijk.x = list[0].toDouble();
			point_data.ijk.y = list[1].toDouble();
			point_data.ijk.z = list[2].toDouble();
		}
		else
		{
			throw TXmlFileException(QStringLiteral("ERR:  IJK must have three values"));
		}
		
		item = new QListWidgetItem(d_points_list);
		
		variant.setValue(point_data);
		
		item->setData(Qt::UserRole,variant);
		item->setText(QString("%1  %2,%3,%4  %5,%6,%7")
					  .arg(point_data.description)
					  .arg(point_data.xyz.x,0,'f',1)
					  .arg(point_data.xyz.y,0,'f',1)
					  .arg(point_data.xyz.z,0,'f',1)
					  .arg(point_data.ijk.x,0,'f',2)
					  .arg(point_data.ijk.y,0,'f',2)
					  .arg(point_data.ijk.z,0,'f',2));
		
		e0 = xml_file->Get_Sibling_Node(&e0,QStringLiteral("Point"),true);
	}
}
