/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QGridLayout>
#include <QLabel>
#include <QLineEdit>
#include <QListWidget>
#include <QListWidgetItem>
#include <cmath>
#include <assert.h>

#include "../../core/mat4.h"
#include "artifactpointtype.h"

#include "artifactstepgaugewidget.h"

TArtifactStepgaugeWidget::TArtifactStepgaugeWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QGridLayout							*gridLayout;
	QLabel								*attributes_label;
	QLabel								*axis_i_label;
	QLabel								*axis_j_label;
	QLabel								*axis_k_label;
	QLabel								*axis_label;
	QLabel								*length_label;
	QLabel								*pos_x_label;
	QLabel								*pos_y_label;
	QLabel								*pos_z_label;
	QLabel								*position_label;
	QListWidgetItem						*item;

	gridLayout = new QGridLayout(this);
	gridLayout->setSpacing(4);
	gridLayout->setContentsMargins(0,0,0,0);

	position_label = new QLabel(this);
	gridLayout->addWidget(position_label,0,0,1,1);

	pos_x_label = new QLabel(this);
	pos_x_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	gridLayout->addWidget(pos_x_label,0,1,1,1);

	d_pos_x_edit = new QLineEdit(this);
	d_pos_x_edit->setAlignment(Qt::AlignCenter);
	gridLayout->addWidget(d_pos_x_edit,0,2,1,1);

	pos_y_label = new QLabel(this);
	pos_y_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	gridLayout->addWidget(pos_y_label,1,1,1,1);

	d_pos_y_edit = new QLineEdit(this);
	d_pos_y_edit->setAlignment(Qt::AlignCenter);
	gridLayout->addWidget(d_pos_y_edit,1,2,1,1);

	pos_z_label = new QLabel(this);
	pos_z_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	gridLayout->addWidget(pos_z_label,2,1,1,1);

	d_pos_z_edit = new QLineEdit(this);
	d_pos_z_edit->setAlignment(Qt::AlignCenter);
	gridLayout->addWidget(d_pos_z_edit,2,2,1,1);

	axis_label = new QLabel(this);
	gridLayout->addWidget(axis_label,3,0,1,1);

	axis_i_label = new QLabel(this);
	axis_i_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	gridLayout->addWidget(axis_i_label,3,1,1,1);

	d_axis_i_edit = new QLineEdit(this);
	d_axis_i_edit->setAlignment(Qt::AlignCenter);
	gridLayout->addWidget(d_axis_i_edit,3,2,1,1);

	axis_j_label = new QLabel(this);
	axis_j_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	gridLayout->addWidget(axis_j_label,4,1,1,1);

	d_axis_j_edit = new QLineEdit(this);
	d_axis_j_edit->setAlignment(Qt::AlignCenter);
	gridLayout->addWidget(d_axis_j_edit,4,2,1,1);

	axis_k_label = new QLabel(this);
	axis_k_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	gridLayout->addWidget(axis_k_label,5,1,1,1);

	d_axis_k_edit = new QLineEdit(this);
	d_axis_k_edit->setAlignment(Qt::AlignCenter);
	gridLayout->addWidget(d_axis_k_edit,5,2,1,1);

	attributes_label = new QLabel(this);
	gridLayout->addWidget(attributes_label,6,0,1,1);

	length_label = new QLabel(this);
	length_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	gridLayout->addWidget(length_label,6,1,1,1);

	d_length_edit = new QLineEdit(this);
	d_length_edit->setAlignment(Qt::AlignCenter);
	gridLayout->addWidget(d_length_edit,6,2,1,1);

	d_points_list = new QListWidget(this);
	gridLayout->addWidget(d_points_list,7,0,1,3);

	QWidget::setTabOrder(d_pos_x_edit,d_pos_y_edit);
	QWidget::setTabOrder(d_pos_y_edit,d_pos_z_edit);
	QWidget::setTabOrder(d_pos_z_edit,d_axis_i_edit);
	QWidget::setTabOrder(d_axis_i_edit,d_axis_j_edit);
	QWidget::setTabOrder(d_axis_j_edit,d_axis_k_edit);
	QWidget::setTabOrder(d_axis_k_edit,d_length_edit);
	QWidget::setTabOrder(d_length_edit,d_points_list);
	
	// defaults
	d_head_a = 0.0;
	d_head_b = 0.0;
	
	item = new QListWidgetItem;
	item->setText(QStringLiteral("PLN1 Plane Point 1"));
	item->setData(Qt::UserRole,TPointType::PLN1);
	d_points_list->addItem(item);

	item = new QListWidgetItem;
	item->setText(QStringLiteral("PLN2 Plane Point 2"));
	item->setData(Qt::UserRole,TPointType::PLN2);
	d_points_list->addItem(item);

	item = new QListWidgetItem;
	item->setText(QStringLiteral("PLN3 Plane Point 3"));
	item->setData(Qt::UserRole,TPointType::PLN3);
	d_points_list->addItem(item);

	item = new QListWidgetItem;
	item->setText(QStringLiteral("LN1 Line Point 1"));
	item->setData(Qt::UserRole,TPointType::LN1);
	d_points_list->addItem(item);

	item = new QListWidgetItem;
	item->setText(QStringLiteral("LN2 Line Point 2"));
	item->setData(Qt::UserRole,TPointType::LN2);
	d_points_list->addItem(item);

	item = new QListWidgetItem;
	item->setText(QStringLiteral("PT1 Start Point"));
	item->setData(Qt::UserRole,TPointType::PT1);
	d_points_list->addItem(item);

	item = new QListWidgetItem;
	item->setText(QStringLiteral("PT2 End Point"));
	item->setData(Qt::UserRole,TPointType::PT2);
	d_points_list->addItem(item);

	position_label->setText(QStringLiteral("Center:"));
	pos_x_label->setText(QStringLiteral("X:"));
	pos_y_label->setText(QStringLiteral("Y:"));
	pos_z_label->setText(QStringLiteral("Z:"));
	axis_label->setText(QStringLiteral("Axis:"));
	axis_i_label->setText(QStringLiteral("I:"));
	axis_j_label->setText(QStringLiteral("J:"));
	axis_k_label->setText(QStringLiteral("K:"));
	attributes_label->setText(QStringLiteral("Traits:"));
	length_label->setText(QStringLiteral("Length:"));

	connect(d_points_list,&QListWidget::itemDoubleClicked,this,&TArtifactStepgaugeWidget::Touch_Double_Clicked);
}

TArtifactStepgaugeWidget::~TArtifactStepgaugeWidget(void)
{
}

TVector3 TArtifactStepgaugeWidget::Attribute_Position(void) const
{
	TVector3							pos;
	
	pos.x = d_pos_x_edit->text().toDouble();
	pos.y = d_pos_y_edit->text().toDouble();
	pos.z = d_pos_z_edit->text().toDouble();

	return pos;
}

TVector3 TArtifactStepgaugeWidget::Attribute_Axis(void) const
{
	TVector3							axis;
	static const double					ZERO_EPSILON(0.001);
	
	axis.i = d_axis_i_edit->text().toDouble();
	axis.j = d_axis_j_edit->text().toDouble();
	axis.k = d_axis_k_edit->text().toDouble();
	
	if(axis.Length() < ZERO_EPSILON)
	{
		axis.Set(0,0,1);
	}
	else
	{
		axis.Normal();
	}

	return axis;
}

double TArtifactStepgaugeWidget::Attribute_Length(void) const
{
	double								len;
	
	len = d_length_edit->text().toDouble();
	
	return len;
}

void TArtifactStepgaugeWidget::Set_Attribute_Position(
	const TVector3 						&pos)
{
	d_pos_x_edit->setText(QString("%1").arg(pos.x,0,'f',4));
	d_pos_y_edit->setText(QString("%1").arg(pos.y,0,'f',4));
	d_pos_z_edit->setText(QString("%1").arg(pos.z,0,'f',4));
}

void TArtifactStepgaugeWidget::Set_Attribute_Axis(
	const TVector3 						&axis)
{
	d_axis_i_edit->setText(QString("%1").arg(axis.i,0,'f',6));
	d_axis_j_edit->setText(QString("%1").arg(axis.j,0,'f',6));
	d_axis_k_edit->setText(QString("%1").arg(axis.k,0,'f',6));
}

void TArtifactStepgaugeWidget::Set_Attribute_Length(
	const double 						&len)
{
	d_length_edit->setText(QString("%1").arg(len,0,'f',4));
}

void TArtifactStepgaugeWidget::Touch_Double_Clicked(
	QListWidgetItem						*item)
{
	TVector3							sensor_axis;
	TVector3							artifact_axis;
	TVector3							axis1;
	TVector3							axis2;
	TVector3							center_pt;
	TVector3							start_pt;
	TVector3							end_pt;
	TVector3							touch_pt;
	TVector3							touch_pt_dir;
	double								length;
	int									point_id;
	static const double					OFFSET_RADIUS(5.0);
	static const double					ZERO_EPSILON(0.0001);
	static const double					MIN_A_ANGLE(1.0);
	static const double					DEG2RAD(0.01745329251994);

	artifact_axis = this->Attribute_Axis();
	length = this->Attribute_Length();
	center_pt = this->Attribute_Position();
	
	start_pt = center_pt - (artifact_axis * length / 2.0);
	end_pt = center_pt + (artifact_axis * length / 2.0);

	if(d_head_a > MIN_A_ANGLE)
	{
		sensor_axis.x = sin(d_head_b * DEG2RAD) * sin(d_head_a * DEG2RAD);
		sensor_axis.y = -1.0 * cos(d_head_b * DEG2RAD) * sin(d_head_a * DEG2RAD);;
		sensor_axis.z = -1.0 * cos(d_head_a * DEG2RAD);
	}
	else
	{
		sensor_axis = d_tool_offset;
	}

	if(sensor_axis.Length() < ZERO_EPSILON)
	{
		sensor_axis.Set(0,0,-1);
	}
	else
	{
		sensor_axis.Normal();
	}
	
	axis2 = artifact_axis * sensor_axis;
	
	if(axis2.Length() < ZERO_EPSILON)
	{
		if(fabs(artifact_axis.x) < 0.707107)
		{
			axis2.Set(-1,0,0);
		}
		else
		{
			axis2.Set(0,-1,0);
		}
	}
	else
	{
		axis2.Normal();
	}
	
	axis1 = axis2 * artifact_axis;
	
	if(axis1.Length() < ZERO_EPSILON)
	{
		axis1.Set(0,0,-1);
	}
	else
	{
		axis1.Normal();
	}
	
	point_id = item->data(Qt::UserRole).toInt();
	
	switch(point_id)
	{
		case TPointType::PLN1:
			touch_pt = start_pt;
			touch_pt += (axis2 * OFFSET_RADIUS);
			touch_pt_dir = axis1;
			
			emit Touch_Point(touch_pt.x,touch_pt.y,touch_pt.z,touch_pt_dir.i,touch_pt_dir.j,touch_pt_dir.k);

			break;
			
		case TPointType::PLN2:
			touch_pt = center_pt;
			touch_pt -= (axis2 * OFFSET_RADIUS);
			touch_pt_dir = axis1;
			
			emit Touch_Point(touch_pt.x,touch_pt.y,touch_pt.z,touch_pt_dir.i,touch_pt_dir.j,touch_pt_dir.k);

			break;
			
		case TPointType::PLN3:
			touch_pt = end_pt;
			touch_pt += (axis2 * OFFSET_RADIUS);
			touch_pt_dir = axis1;
			
			emit Touch_Point(touch_pt.x,touch_pt.y,touch_pt.z,touch_pt_dir.i,touch_pt_dir.j,touch_pt_dir.k);

			break;
			
		case TPointType::LN1:
			touch_pt = start_pt;
			touch_pt -= (axis2 * OFFSET_RADIUS);
			touch_pt_dir = axis2;
			
			emit Touch_Point(touch_pt.x,touch_pt.y,touch_pt.z,touch_pt_dir.i,touch_pt_dir.j,touch_pt_dir.k);

			break;
			
		case TPointType::LN2:
			touch_pt = end_pt;
			touch_pt -= (axis2 * OFFSET_RADIUS);
			touch_pt_dir = axis2;
			
			emit Touch_Point(touch_pt.x,touch_pt.y,touch_pt.z,touch_pt_dir.i,touch_pt_dir.j,touch_pt_dir.k);

			break;
			
		case TPointType::PT1:
			touch_pt = start_pt;
			touch_pt_dir = artifact_axis;
			
			emit Touch_Point(touch_pt.x,touch_pt.y,touch_pt.z,touch_pt_dir.i,touch_pt_dir.j,touch_pt_dir.k);

			break;
			
		case TPointType::PT2:
			touch_pt = end_pt;
			touch_pt_dir = -1.0 * artifact_axis;
			
			emit Touch_Point(touch_pt.x,touch_pt.y,touch_pt.z,touch_pt_dir.i,touch_pt_dir.j,touch_pt_dir.k);

			break;
			
		default:
			break;
	}
}

