/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QFont>
#include <QGridLayout>
#include <QLabel>
#include <QListWidget>
#include <QSpacerItem>
#include <QSpinBox>
#include <QCheckBox>
#include <QIcon>
#include <QDir>

#include "bufferdisplaywidget.h"

TBufferDisplayWidget::TBufferDisplayWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QFont								font;
	QGridLayout							*widget_layout;
	QLabel								*buffer_size_label;
	QSpacerItem							*control_hspacer;

	font.setFamily(QStringLiteral("Courier New"));

	this->resize(495,373);

	widget_layout = new QGridLayout(this);
	widget_layout->setContentsMargins(0,0,0,-1);

	d_buffer_list = new QListWidget(this);
	d_buffer_list->setFont(font);
	widget_layout->addWidget(d_buffer_list,0,0,1,4);
	
	buffer_size_label = new QLabel(this);
	widget_layout->addWidget(buffer_size_label,1,0,1,1);
	
	d_buffer_spin = new QSpinBox(this);
	d_buffer_spin->setMinimum(6);
	d_buffer_spin->setMaximum(99);
	d_buffer_spin->setSingleStep(2);
	widget_layout->addWidget(d_buffer_spin,1,1,1,1);
	
	d_log_file_check = new QCheckBox(this);
	widget_layout->addWidget(d_log_file_check,1,2,1,1);
	
	control_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	widget_layout->addItem(control_hspacer,1,3,1,1);

	buffer_size_label->setText(QStringLiteral(" Buffer Size:"));
	d_log_file_check->setText(QStringLiteral("Log to file"));
	
	d_command_number = 0;
	
	connect(d_log_file_check,&QCheckBox::stateChanged,this,&TBufferDisplayWidget::Log_File);

}

TBufferDisplayWidget::~TBufferDisplayWidget(void)
{
}

int TBufferDisplayWidget::Buffer_Size(void) const
{
	return d_buffer_spin->value();
}

void TBufferDisplayWidget::Clear(void)
{
	d_buffer_list->clear();
	d_command_number = 0;
}

void TBufferDisplayWidget::Add_Entry(
	const TMode							mode,
	const QString						&text)
{
	QListWidgetItem						*item;
	QString								item_text;
	int									max_buffer_size;
	
	item = new QListWidgetItem(d_buffer_list);
	
	if(d_command_number == 2147483647 || d_command_number < 0)
	{
		d_command_number = 0;
	}
	
	++d_command_number;
	
	item_text = QString("[%1] %2").arg(d_command_number,10,10,QChar('.')).arg(text);
	
	// substitue new lines for symbols
	item_text.replace('\r',QStringLiteral("<CR>"));
	item_text.replace('\n',QStringLiteral("<LF>"));
	
	item->setText(item_text);
	
	switch(mode)
	{
		case TBufferDisplayWidget::BUFFER_SEND:
			item->setIcon(QIcon(QStringLiteral(":/communication/send.png")));
			break;
			
		case TBufferDisplayWidget::BUFFER_RECEIVE:
			item->setIcon(QIcon(QStringLiteral(":/communication/receive.png")));
			break;
	}
	
	max_buffer_size = d_buffer_spin->value();
	
	while(d_buffer_list->count() > max_buffer_size)
	{
		item = d_buffer_list->takeItem(0);
		delete item;
	}
	
	if(d_log_file.isOpen())
	{
		item_text = text + '\n';
		d_log_file.write(item_text.toLatin1());
	}
}

void TBufferDisplayWidget::Set_Buffer_Size(
	const int							size)
{
	d_buffer_spin->setValue(size);
}

void TBufferDisplayWidget::Log_File(
	int									state)
{
	QString								file_path;
	QString								file_name;
	
	if(d_log_file.isOpen())
	{
		d_log_file.close();
	}
	
	if(state != Qt::Unchecked)
	{
		
#ifdef Q_OS_LINUX
		file_path = QDir::homePath();
#else
		file_path = qApp->applicationDirPath();
		
#ifdef Q_OS_MAC
		int index = file_path.lastIndexOf(QStringLiteral(".app/"));
		file_path.truncate(index);
		
		index = file_path.lastIndexOf('/');
		file_path.truncate(index);
#endif

#endif
		
		file_name = QString("%1/virtual_cmm_communications.log").arg(file_path);
		
		d_log_file.setFileName(file_name);
		d_log_file.open(QIODevice::WriteOnly | QIODevice::Text);
	}
}

