/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <cmath>
#include <assert.h>

#include "../../core/mat4.h"

#include "command_executor.h"

static const double						ZERO_EPSILON(0.0001);

TCommandExecutor::TCommandExecutor(void)
{
	d_command_status = TCommandExecutor::COMMAND_IDLE;
	d_timer_interval = 50;
}

TCommandExecutor::~TCommandExecutor(void)
{
}

void TCommandExecutor::Set_Current_Position(
	const TVector3 						&current_position)
{
	d_current_position = current_position;
}

void TCommandExecutor::Add_Command(
	const TCommandExecutor::TCommand	&command)
{
	std::vector<TCommandExecutor::TCommand>	sub_commands;
	TCommandExecutor::TCommand			new_command;

	if(d_command_status == TCommandExecutor::COMMAND_IDLE)
	{		
		if(command.command_type == TCommandExecutor::COMMAND_DEFINE_MOVECIR_CENTER)
		{
			d_movecir_center = command.target;
			d_active_command = command;
		}
		else if(command.command_type == TCommandExecutor::COMMAND_MOVECIR)
		{
			sub_commands = this->Move_Arc(command.target,command.speed);
			
			new_command = command;
			new_command.command_type = TCommandExecutor::COMMAND_MOVE;
			sub_commands.push_back(new_command);
						
			d_command_queue.insert(d_command_queue.begin(),sub_commands.begin()+1,sub_commands.end());

			d_active_command = sub_commands[0];
			this->Set_Target(d_active_command.target);
			this->Set_Speed(d_active_command.speed);
		}
		else	// normal move or touch
		{
			d_active_command = command;
			this->Set_Target(command.target);
			this->Set_Speed(command.speed);
		}
		
		d_command_status = TCommandExecutor::COMMAND_EXECUTING;
	}
	else
	{
		d_command_queue.push_back(command);
	}
}

TVector3 TCommandExecutor::Execute(void)
{
	TVector3							new_position;
	TVector3							vec;
	
	if(d_command_status == TCommandExecutor::COMMAND_IDLE)
	{
		return d_current_position;
	}
	
	if(d_active_command.command_type == TCommandExecutor::COMMAND_DEFINE_MOVECIR_CENTER)
	{
		d_command_status = TCommandExecutor::COMMAND_COMPLETE;
		return d_current_position;
	}
	else if(d_active_command.command_type == TCommandExecutor::COMMAND_MOVECIR)
	{
		d_command_status = TCommandExecutor::COMMAND_COMPLETE;
		return d_current_position;
	}
	else if(d_active_command.command_type == TCommandExecutor::COMMAND_GETPOS)
	{
		d_command_status = TCommandExecutor::COMMAND_COMPLETE;
		return d_current_position;
	}

	new_position = d_current_position + d_speed * d_move_axis;
	
	vec = d_target_position - new_position;
		
	if(vec.Length() > ZERO_EPSILON)
	{		
		vec.Normal();
		
		if((vec ^ d_move_axis) < 0)
		{
			// went past the target
			d_command_status = TCommandExecutor::COMMAND_COMPLETE;
			d_current_position = d_target_position;
			return d_target_position;
		}
		else
		{
			d_current_position = new_position;
			d_command_status = TCommandExecutor::COMMAND_EXECUTING;			
			return d_current_position;
		}
	}
	
	// close enough
	d_command_status = TCommandExecutor::COMMAND_COMPLETE;
	d_current_position = d_target_position;
	
	return d_target_position;
}

bool TCommandExecutor::Execute_Next(void)
{
	std::vector<TCommandExecutor::TCommand>	sub_commands;
	TCommandExecutor::TCommand			new_command;

	if(d_command_status == TCommandExecutor::COMMAND_EXECUTING)
	{
		return true;
	}
	
	if(d_command_queue.size())
	{
		d_active_command = d_command_queue[0];
		d_command_queue.erase(d_command_queue.begin());
		
		if(d_active_command.command_type == TCommandExecutor::COMMAND_DEFINE_MOVECIR_CENTER)
		{
			d_movecir_center = d_active_command.target;
		}
		else if(d_active_command.command_type == TCommandExecutor::COMMAND_MOVECIR)
		{
			sub_commands = this->Move_Arc(d_active_command.target,d_active_command.speed);
			
			new_command = d_active_command;
			new_command.command_type = TCommandExecutor::COMMAND_MOVE;
			
			sub_commands.push_back(new_command);
			
			d_command_queue.insert(d_command_queue.begin(),sub_commands.begin()+1,sub_commands.end());
										
			d_active_command = sub_commands[0];

			this->Set_Target(d_active_command.target);
			this->Set_Speed(d_active_command.speed);
		}
		else if(d_active_command.command_type == TCommandExecutor::COMMAND_MOVE ||
				d_active_command.command_type == TCommandExecutor::COMMAND_TOUCH)
		{
			this->Set_Target(d_active_command.target);
			this->Set_Speed(d_active_command.speed);
		}
		
		d_command_status = TCommandExecutor::COMMAND_EXECUTING;
		return true;
	}
	
	d_command_status = TCommandExecutor::COMMAND_IDLE;
	
	return false;
}

void TCommandExecutor::Abort(void)
{
	d_command_status = TCommandExecutor::COMMAND_IDLE;
	d_command_queue.clear();
}

void TCommandExecutor::Set_Timer_Interval(
	const int							interval)
{
	d_timer_interval = interval;
}

void TCommandExecutor::Set_Target(
	const TVector3 						&target_position)
{
	d_target_position = target_position;
	
	d_move_axis = d_target_position - d_current_position;
	
	if(d_move_axis.Length() > ZERO_EPSILON)
	{
		d_move_axis.Normal();
		d_command_status = TCommandExecutor::COMMAND_EXECUTING;
	}
	else
	{
		d_move_axis.Set(1.0,0.0,0.0);
		d_command_status = TCommandExecutor::COMMAND_COMPLETE;
	}
}

void TCommandExecutor::Set_Speed(
	const double 						&dval)
{
	double								speed(dval);
	static const double					MINIMUM_SPEED(1.0);
	
	if(speed > MINIMUM_SPEED)
	{
		speed = dval;
	}
	else
	{
		speed = MINIMUM_SPEED;
	}
	
	d_speed = speed * static_cast<double>(d_timer_interval) / 1000.0;
}

std::vector<TCommandExecutor::TCommand> TCommandExecutor::Move_Arc(
	const TVector3 						&target,
	const double						&move_speed) const
{
	std::vector<TCommandExecutor::TCommand>	sub_moves;
	TCommandExecutor::TCommand			command;
	TVector3							v1,v2,v3;
	TVector3							target_axis;
	TVector3							pt;
	TMat4								mat;
	double								angle_range;
	double								start_radius;
	double								end_radius;
	double								angle_increment;
	double								radius_increment;
	double								angle;
	double								radius;
	double								dval;
	int									cntr;
	int									sub_move_count(3);
	static const int					MAX_SUB_MOVE_COUNT(30);
	
	v1 = d_current_position - d_movecir_center;
	v2 = target - d_movecir_center;
	
	start_radius = v1.Length();
	end_radius = v2.Length();
	
	dval = start_radius;
	if(end_radius > start_radius) dval = end_radius;
	
	dval *= 0.628318530718;	// 1 point / 10 mm
	
	sub_move_count += static_cast<int>(0.5 + dval);
	
	if(sub_move_count > MAX_SUB_MOVE_COUNT) sub_move_count = MAX_SUB_MOVE_COUNT;
	
	if(start_radius < ZERO_EPSILON || end_radius < ZERO_EPSILON)
	{
		return sub_moves;
	}
	
	v3 = v1 * v2;
	
	if(v3.Length() < ZERO_EPSILON)
	{
		return sub_moves;
	}
	
	v1.Normal();
	v3.Normal();
	
	target_axis = v2;
	target_axis.Normal();
	
	v2 = v3 * v1;
	
	v2.Normal();
	
	mat.X(v1);
	mat.Y(v2);
	mat.Z(v3);
	
	mat.Transpose();
	
	angle_range = acos(target_axis ^ v1);
	
	angle_increment = angle_range / (static_cast<double>(sub_move_count + 1));
	radius_increment = (end_radius - start_radius) / (static_cast<double>(sub_move_count + 1));
	
	command.command_type = TCommandExecutor::COMMAND_MOVE;
	command.speed = move_speed;
	command.completion_text = QString();

	angle = angle_increment;
	radius = start_radius + radius_increment;
	
	for(cntr = 0;cntr < sub_move_count;++cntr)
	{
		pt.Set(cos(angle) * radius,sin(angle) * radius,0.0);
		
		pt = mat * pt;
		pt += d_movecir_center;
		
		command.target = pt;

		sub_moves.push_back(command);
		
		angle += angle_increment;
		radius += radius_increment;
	}
		
	return sub_moves;
	
}
