/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QDockWidget>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QToolButton>
#include <QSpinBox>
#include <QDoubleSpinBox>
#include <QCheckBox>
#include <QSizePolicy>

#include "machineattributeswidget.h"

#include "machine_dock.h"

TMachineDock::TMachineDock(
	const QWidget						*parent)
:QObject(const_cast<QWidget*>(parent))
{
	QFrame								*upper_separator_hline;
	QFrame								*lower_separator_hline;
	QGridLayout							*grid_layout;
	QHBoxLayout							*button_protocol_layout;
	QSpacerItem							*button_protocol_hspacer;
	QHBoxLayout							*button_move_jogbox_layout;
	QSpacerItem							*button_move_jogbox_hspacer;
	QToolButton							*jogbox_toggle_button;
	QToolButton							*move_button;
	QToolButton							*machine_errors_button;
	QLabel								*move_x_label;
	QLabel								*move_y_label;
	QLabel								*move_z_label;
	QLabel								*path_buffer_label;
	QSpacerItem							*machine_vspacer;
	
	d_machine_dock = new QDockWidget(const_cast<QWidget*>(parent));
	
	d_machine_widget = new QWidget();
	d_machine_dock->setWidget(d_machine_widget);
	
	grid_layout = new QGridLayout(d_machine_widget);
	grid_layout->setSpacing(4);

	button_protocol_layout = new QHBoxLayout();
	
	d_protocol_button = new QToolButton(d_machine_widget);
	d_protocol_button->setSizePolicy(QSizePolicy(QSizePolicy::Minimum, QSizePolicy::Fixed));
	button_protocol_layout->addWidget(d_protocol_button);
	
	d_protocol_label = new QLabel(d_machine_widget);
	button_protocol_layout->addWidget(d_protocol_label);

	d_serial_connect_button = new QToolButton(d_machine_widget);
	d_serial_connect_button->setSizePolicy(QSizePolicy(QSizePolicy::Minimum, QSizePolicy::Fixed));
	button_protocol_layout->addWidget(d_serial_connect_button);
	
	button_protocol_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_protocol_layout->addItem(button_protocol_hspacer);
	
	grid_layout->addLayout(button_protocol_layout,0,0,1,2);

	upper_separator_hline = new QFrame(d_machine_widget);
	upper_separator_hline->setFrameShape(QFrame::HLine);
	upper_separator_hline->setFrameShadow(QFrame::Sunken);
	grid_layout->addWidget(upper_separator_hline,1,0,1,2);
	
	d_machine_attributes_widget = new TMachineAttributesWidget(d_machine_widget);
	grid_layout->addWidget(d_machine_attributes_widget,2,0,1,2);
	
	d_machine_errors_check = new QCheckBox(d_machine_widget);
	grid_layout->addWidget(d_machine_errors_check,3,0,1,1);
	
	machine_errors_button = new QToolButton(d_machine_widget);
	machine_errors_button->setSizePolicy(QSizePolicy(QSizePolicy::Minimum, QSizePolicy::Fixed));
	grid_layout->addWidget(machine_errors_button,3,1,1,1);

	d_override_tip_check = new QCheckBox(d_machine_widget);
	grid_layout->addWidget(d_override_tip_check,4,0,1,1);

	d_override_tip_spin = new QDoubleSpinBox(d_machine_widget);
	d_override_tip_spin->setAlignment(Qt::AlignCenter);
	d_override_tip_spin->setRange(0.1,8.0);
	d_override_tip_spin->setDecimals(4);
	grid_layout->addWidget(d_override_tip_spin,4,1,1,1);
	
	d_generate_noise_check = new QCheckBox(d_machine_widget);
	grid_layout->addWidget(d_generate_noise_check,5,0,1,1);
	
	d_generate_noise_spin = new QDoubleSpinBox(d_machine_widget);
	d_generate_noise_spin->setAlignment(Qt::AlignCenter);
	d_generate_noise_spin->setRange(0.001,0.25);
	d_generate_noise_spin->setDecimals(4);
	grid_layout->addWidget(d_generate_noise_spin,5,1,1,1);

	path_buffer_label = new QLabel(d_machine_widget);
	path_buffer_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	grid_layout->addWidget(path_buffer_label,6,0,1,1);
	
	d_touch_buffer_spin = new QSpinBox(d_machine_widget);
	d_touch_buffer_spin->setAlignment(Qt::AlignCenter);
	d_touch_buffer_spin->setRange(0,50);
	grid_layout->addWidget(d_touch_buffer_spin,6,1,1,1);
	
	lower_separator_hline = new QFrame(d_machine_widget);
	lower_separator_hline->setFrameShape(QFrame::HLine);
	lower_separator_hline->setFrameShadow(QFrame::Sunken);
	grid_layout->addWidget(lower_separator_hline,7,0,1,2);

	button_move_jogbox_layout = new QHBoxLayout();
	
	move_button = new QToolButton(d_machine_widget);
	move_button->setSizePolicy(QSizePolicy(QSizePolicy::Minimum, QSizePolicy::Fixed));
	button_move_jogbox_layout->addWidget(move_button);

	button_move_jogbox_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_move_jogbox_layout->addItem(button_move_jogbox_hspacer);
	
	jogbox_toggle_button = new QToolButton(d_machine_widget);
	jogbox_toggle_button->setSizePolicy(QSizePolicy(QSizePolicy::Minimum, QSizePolicy::Fixed));
	button_move_jogbox_layout->addWidget(jogbox_toggle_button);

	grid_layout->addLayout(button_move_jogbox_layout,8,0,1,2);
	
	move_x_label = new QLabel(d_machine_widget);
	move_x_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	grid_layout->addWidget(move_x_label,9,0,1,1);
	
	d_move_x_edit = new QLineEdit(d_machine_widget);
	d_move_x_edit->setAlignment(Qt::AlignCenter);
	grid_layout->addWidget(d_move_x_edit,9,1,1,1);
	
	move_y_label = new QLabel(d_machine_widget);
	move_y_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	grid_layout->addWidget(move_y_label,10,0,1,1);
	
	d_move_y_edit = new QLineEdit(d_machine_widget);
	d_move_y_edit->setAlignment(Qt::AlignCenter);
	grid_layout->addWidget(d_move_y_edit,10,1,1,1);
	
	move_z_label = new QLabel(d_machine_widget);
	move_z_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	grid_layout->addWidget(move_z_label,11,0,1,1);
	
	d_move_z_edit = new QLineEdit(d_machine_widget);
	d_move_z_edit->setAlignment(Qt::AlignCenter);
	grid_layout->addWidget(d_move_z_edit,11,1,1,1);

	machine_vspacer = new QSpacerItem(0,0,QSizePolicy::Minimum,QSizePolicy::Expanding);
	grid_layout->addItem(machine_vspacer,12,0,1,2);
	
	QWidget::setTabOrder(d_protocol_button,d_serial_connect_button);
	QWidget::setTabOrder(d_serial_connect_button,d_machine_attributes_widget);
	QWidget::setTabOrder(d_machine_attributes_widget,d_machine_errors_check);
	QWidget::setTabOrder(d_machine_errors_check,machine_errors_button);
	QWidget::setTabOrder(machine_errors_button,d_override_tip_check);
	QWidget::setTabOrder(d_override_tip_check,d_override_tip_spin);
	QWidget::setTabOrder(d_override_tip_spin,d_generate_noise_spin);
	QWidget::setTabOrder(d_generate_noise_spin,d_touch_buffer_spin);
	QWidget::setTabOrder(d_touch_buffer_spin,move_button);
	QWidget::setTabOrder(move_button,jogbox_toggle_button);
	QWidget::setTabOrder(jogbox_toggle_button,d_move_x_edit);
	QWidget::setTabOrder(d_move_x_edit,d_move_y_edit);
	QWidget::setTabOrder(d_move_y_edit,d_move_z_edit);
	
	// defaults
	d_touch_buffer_spin->setValue(0);
	d_override_tip_spin->setEnabled(false);
	d_generate_noise_spin->setEnabled(false);
	
	d_machine_dock->setWindowTitle(QStringLiteral("Machine"));
	
	move_button->setText(QStringLiteral("Move"));
	jogbox_toggle_button->setText(QStringLiteral("Jogbox"));
	path_buffer_label->setText(QStringLiteral("Path Buffer:"));
	move_x_label->setText(QStringLiteral("Move X:"));
	move_y_label->setText(QStringLiteral("Move Y:"));
	move_z_label->setText(QStringLiteral("Move Z:"));
	d_protocol_button->setText(QStringLiteral("Protocol"));
	d_protocol_label->setText(QStringLiteral("Serial"));
	d_serial_connect_button->setText(QStringLiteral("Open COMx"));
	d_machine_errors_check->setText(QStringLiteral("Enable Errors:"));
	machine_errors_button->setText(QStringLiteral("Edit Errors"));
	d_override_tip_check->setText(QStringLiteral("Override Tip:"));
	d_generate_noise_check->setText(QStringLiteral("Touch Noise:"));
	
	connect(d_machine_attributes_widget,&TMachineAttributesWidget::Value_Changed,this,&TMachineDock::Attribute_Changed);
		
	connect(jogbox_toggle_button,&QToolButton::clicked,this,&TMachineDock::Show_Hide_Jogbox);
	connect(move_button,&QToolButton::clicked,this,&TMachineDock::Move_Clicked);
	connect(machine_errors_button,&QToolButton::clicked,this,&TMachineDock::Edit_Machine_Errors);
	connect(d_protocol_button,&QToolButton::clicked,this,&TMachineDock::Change_Protocol);
	connect(d_serial_connect_button,&QToolButton::clicked,this,&TMachineDock::Connect_Serial);
	connect(d_override_tip_check,&QCheckBox::toggled,d_override_tip_spin,&QDoubleSpinBox::setEnabled);
	connect(d_generate_noise_check,&QCheckBox::toggled,d_generate_noise_spin,&QDoubleSpinBox::setEnabled);
	connect(d_machine_errors_check,&QCheckBox::toggled,this,&TMachineDock::Enable_Machine_Errors_Changed);
	connect(d_override_tip_check,&QCheckBox::toggled,this,&TMachineDock::Override_Tip_State_Changed);
	connect(d_generate_noise_check,&QCheckBox::toggled,this,&TMachineDock::Generate_Noise_State_Changed);
	connect(d_override_tip_spin,static_cast<void (QDoubleSpinBox::*)(double)>(&QDoubleSpinBox::valueChanged),this,&TMachineDock::Override_Tip_Diameter_Changed);
	connect(d_generate_noise_spin,static_cast<void (QDoubleSpinBox::*)(double)>(&QDoubleSpinBox::valueChanged),this,&TMachineDock::Generate_Noise_Value_Changed);
	
	connect(d_touch_buffer_spin,static_cast<void (QSpinBox::*)(int)>(&QSpinBox::valueChanged),this,&TMachineDock::Touch_Buffer_Changed);
}

TMachineDock::~TMachineDock(void)
{
}

TVector3 TMachineDock::Min_Machine(void) const
{
	TVector3							pt;
	
	pt.x = d_machine_attributes_widget->Min_X();
	pt.y = d_machine_attributes_widget->Min_Y();
	pt.z = d_machine_attributes_widget->Min_Z();

	return pt;
}

TVector3 TMachineDock::Max_Machine(void) const
{
	TVector3							pt;
		
	pt.x = d_machine_attributes_widget->Max_X();
	pt.y = d_machine_attributes_widget->Max_Y();
	pt.z = d_machine_attributes_widget->Max_Z();

	return pt;
}

double TMachineDock::Temperature_X(void) const
{
	return d_machine_attributes_widget->Temp_X();
}

double TMachineDock::Temperature_Y(void) const
{
	return d_machine_attributes_widget->Temp_Y();
}

double TMachineDock::Temperature_Z(void) const
{
	return d_machine_attributes_widget->Temp_Z();
}

double TMachineDock::Temperature_Part(void) const
{
	return d_machine_attributes_widget->Temp_Part();
}

bool TMachineDock::Machine_Errors_Enabled(void) const
{
	return d_machine_errors_check->isChecked();
}

bool TMachineDock::Override_Tip_Enabled(void) const
{
	return d_override_tip_check->isChecked();
}

double TMachineDock::Tip_Diameter(void) const
{
	return d_override_tip_spin->value();
}

bool TMachineDock::Generate_Noise_Enabled(void) const
{
	return d_generate_noise_check->isChecked();
}

double TMachineDock::Noise_Value(void) const
{
	return d_generate_noise_spin->value();
}

TVector3 TMachineDock::Move_Position(void) const
{
	TVector3							pt;
	
	pt.x = d_move_x_edit->text().toDouble();
	pt.y = d_move_y_edit->text().toDouble();
	pt.z = d_move_z_edit->text().toDouble();
	
	return pt;
}

int TMachineDock::Touch_Buffer_Size(void) const
{
	return d_touch_buffer_spin->value();
}

void TMachineDock::Set_Min_Machine(
	const TVector3 						&pnt)
{
	d_machine_attributes_widget->Set_Min_X(pnt.x);
	d_machine_attributes_widget->Set_Min_Y(pnt.y);
	d_machine_attributes_widget->Set_Min_Z(pnt.z);
}

void TMachineDock::Set_Max_Machine(
	const TVector3 						&pnt)
{
	d_machine_attributes_widget->Set_Max_X(pnt.x);
	d_machine_attributes_widget->Set_Max_Y(pnt.y);
	d_machine_attributes_widget->Set_Max_Z(pnt.z);
}

void TMachineDock::Set_Temperature_X(
	const double 						&value)
{
	d_machine_attributes_widget->Set_Temp_X(value);
}

void TMachineDock::Set_Temperature_Y(
	const double 						&value)
{
	d_machine_attributes_widget->Set_Temp_Y(value);
}

void TMachineDock::Set_Temperature_Z(
	const double 						&value)
{
	d_machine_attributes_widget->Set_Temp_Z(value);
}

void TMachineDock::Set_Temperature_Part(
	const double 						&value)
{
	d_machine_attributes_widget->Set_Temp_Part(value);
}

void TMachineDock::Set_Tip_Diameter(
	const double						&value)
{
	d_override_tip_spin->setValue(value);
}

void TMachineDock::Set_Noise(
	const double						&value)
{
	d_generate_noise_spin->setValue(value);
}

void TMachineDock::Enable_Machine_Errors(
	const bool 							state)
{
	d_machine_errors_check->setChecked(state);
}

void TMachineDock::Enable_Override_Tip(
	const bool 							state)
{
	d_override_tip_check->setChecked(state);
	d_override_tip_spin->setEnabled(state);
}

void TMachineDock::Enable_Noise(
	const bool 							state)
{
	d_generate_noise_check->setChecked(state);
	d_generate_noise_spin->setEnabled(state);
}

void TMachineDock::Set_Move_Position(
	const TVector3						&pnt)
{
	d_move_x_edit->setText(QString("%1").arg(pnt.x,0,'f',3));
	d_move_y_edit->setText(QString("%1").arg(pnt.y,0,'f',3));
	d_move_z_edit->setText(QString("%1").arg(pnt.z,0,'f',3));
}

void TMachineDock::Set_Touch_Buffer_Size(
	const int							buffer_size)
{
	d_touch_buffer_spin->setValue(buffer_size);
}

void TMachineDock::Enable_Protocol(
	const bool							state)
{
	d_protocol_button->setEnabled(state);
}

void TMachineDock::Set_Serial_Connect_Button(
	const bool							enabled,
	const QString						&text)
{
	if(enabled)
	{
		d_serial_connect_button->setEnabled(true);
		d_protocol_label->setText(QStringLiteral("Serial"));
	}
	else
	{
		d_serial_connect_button->setEnabled(false);
		d_protocol_label->setText(QStringLiteral("Ethernet"));
	}
	
	if(text.length())
	{
		d_serial_connect_button->setText(text);
	}
}

void TMachineDock::Move_Clicked(void)
{
	double								x,y,z;
	bool								state;
	
	x = d_move_x_edit->text().toDouble(&state);
	if(!state) return;
	
	y = d_move_y_edit->text().toDouble(&state);
	if(!state) return;
	
	z = d_move_z_edit->text().toDouble(&state);
	if(!state) return;
	
	emit Move(x,y,z);
}
